from __future__ import annotations

import json
from datetime import datetime
from typing import Annotated

from pretix_fattura_elettronica.models import InvoiceLog
from pretix_fattura_elettronica.utils import get_codice_fiscale

from django.db.models import Count

from pretix.base.models import Invoice, Order
from pydantic import BaseModel, Field, model_validator
from pydantic.functional_serializers import field_serializer

from .enums import DOC_TYPE as DT
from .enums import SETTINGS


class DettaglioLinea(BaseModel):
    numero_linea: Annotated[int, Field(serialization_alias="NumeroLinea")]
    descrizione: Annotated[str, Field(serialization_alias="Descrizione")]
    quantita: Annotated[str | None, Field(serialization_alias="Quantita")] = None
    unita_misura: Annotated[str | None, Field(serialization_alias="UnitaMisura")] = None
    prezzo_unitario: Annotated[str, Field(serialization_alias="PrezzoUnitario")]
    prezzo_totale: Annotated[str, Field(serialization_alias="PrezzoTotale")]
    aliquota_iva: Annotated[str, Field(serialization_alias="AliquotaIVA")]
    ritenuta: Annotated[str | None, Field(serialization_alias="Ritenuta")] = None
    natura: Annotated[str | None, Field(serialization_alias="Natura")] = None


class DatiRiepilogo(BaseModel):
    aliquota_iva: Annotated[str, Field(serialization_alias="AliquotaIVA")]
    imponibile_importo: Annotated[str, Field(serialization_alias="ImponibileImporto")]
    imposta: Annotated[str, Field(serialization_alias="Imposta")]


class DatiBeniServizi(BaseModel):
    dettaglio_linee: Annotated[
        list[DettaglioLinea], Field(serialization_alias="DettaglioLinee")
    ]
    dati_riepilogo: Annotated[
        list[DatiRiepilogo], Field(serialization_alias="DatiRiepilogo")
    ]


class IdFiscaleIVA(BaseModel):
    id_paese: Annotated[
        str, Field(serialization_alias="IdPaese", min_length=2, max_length=2)
    ]
    id_codice: Annotated[
        str, Field(serialization_alias="IdCodice", min_length=1, max_length=28)
    ]


class Anagrafica(BaseModel):
    """Denominazione or Nome and Cognome should be filled"""

    denominazione: Annotated[
        str | None, Field(serialization_alias="Denominazione")
    ] = None
    nome: Annotated[str | None, Field(serialization_alias="Nome")] = None
    cognome: Annotated[str | None, Field(serialization_alias="Cognome")] = None
    titolo: Annotated[str | None, Field(serialization_alias="Titolo")] = None
    cod_eori: Annotated[str | None, Field(serialization_alias="CodEORI")] = None

    @model_validator(mode="after")
    def check_valid_data(self) -> Anagrafica:
        if self.denominazione is None:
            if self.cognome is None and self.nome is None:
                raise ValueError(
                    "Necessaria denominazione oppure nome e cognome del destinatario."
                )
            if self.cognome is None or self.nome is None:
                raise ValueError(
                    "In mancanza di Ragione Sociale, nome e cognome non possono essere vuoti."
                )
        return self


class DatiAnagraficiCedente(BaseModel):
    id_fiscale_iva: Annotated[IdFiscaleIVA, Field(serialization_alias="IdFiscaleIVA")]
    codice_fiscale: Annotated[str | None, Field(serialization_alias="CodiceFiscale")]
    anagrafica: Annotated[Anagrafica, Field(serialization_alias="Anagrafica")]
    regime_fiscale: Annotated[str, Field(serialization_alias="RegimeFiscale")]


class DatiAnagraficiCessionario(BaseModel):
    id_fiscale_iva: Annotated[
        IdFiscaleIVA | None, Field(serialization_alias="IdFiscaleIVA")
    ]
    codice_fiscale: Annotated[str | None, Field(serialization_alias="CodiceFiscale")]
    anagrafica: Annotated[Anagrafica, Field(serialization_alias="Anagrafica")]


class Sede(BaseModel):
    indirizzo: Annotated[str, Field(serialization_alias="Indirizzo", min_length=2)]
    numero_civico: Annotated[
        str | None, Field(serialization_alias="NumeroCivico")
    ] = None
    cap: Annotated[str, Field(serialization_alias="CAP")]
    comune: Annotated[str, Field(serialization_alias="Comune", min_length=2)]
    provincia: Annotated[str | None, Field(serialization_alias="Provincia")] = None
    nazione: Annotated[
        str, Field(serialization_alias="Nazione", min_length=2, max_length=2)
    ]

    @field_serializer("cap")
    def serialize_cap(self, cap: str):
        if self.nazione == "IT":
            return cap

        return "00000"

    @field_serializer("indirizzo")
    def serialize_indirizzo(self, indirizzo: str):
        return indirizzo[:60]


class Contatti(BaseModel):
    telefono: Annotated[str | None, Field(serialization_alias="Telefono")] = None
    fax: Annotated[str | None, Field(serialization_alias="Fax")] = None
    email: Annotated[str | None, Field(serialization_alias="Email")] = None


class DatiTrasmissione(BaseModel):
    id_trasmittente: Annotated[
        IdFiscaleIVA | None, Field(serialization_alias="IdTrasmittente")
    ] = None
    progressivo_invio: Annotated[
        str | None, Field(serialization_alias="ProgressivoInvio")
    ] = None
    formato_trasmissione: Annotated[
        str, Field(serialization_alias="FormatoTrasmissione")
    ] = "FPR12"
    codice_destinatario: Annotated[str, Field(serialization_alias="CodiceDestinatario")]
    pec_destinatario: Annotated[
        str | None, Field(serialization_alias="PECDestinatario")
    ] = None

    @field_serializer("progressivo_invio")
    def serialize_progressivo_invio(self, progressivo_invio: str):
        return progressivo_invio.replace("TEST", "")


class CedentePrestatore(BaseModel):
    dati_anagrafici: Annotated[
        DatiAnagraficiCedente, Field(serialization_alias="DatiAnagrafici")
    ]
    sede: Annotated[Sede, Field(serialization_alias="Sede")]
    contatti: Annotated[Contatti | None, Field(serialization_alias="Contatti")] = None


class CessionarioCommittente(BaseModel):
    dati_anagrafici: Annotated[
        DatiAnagraficiCessionario, Field(serialization_alias="DatiAnagrafici")
    ]
    sede: Annotated[Sede, Field(serialization_alias="Sede")]


class DatiGeneraliDocumento(BaseModel):
    tipo_documento: Annotated[str, Field(serialization_alias="TipoDocumento")]
    divisa: Annotated[str, Field(serialization_alias="Divisa")]
    data: Annotated[datetime, Field(serialization_alias="Data")]
    numero: Annotated[str, Field(serialization_alias="Numero")]

    @field_serializer("data")
    def serialize_data(self, data: datetime):
        return data.date().strftime("%Y-%m-%d")


class DatiGenerali(BaseModel):
    dati_generali_documento: Annotated[
        DatiGeneraliDocumento, Field(serialization_alias="DatiGeneraliDocumento")
    ]


class FatturaElettronicaBody(BaseModel):
    dati_generali: Annotated[DatiGenerali, Field(serialization_alias="DatiGenerali")]
    dati_beni_servizi: Annotated[
        DatiBeniServizi, Field(serialization_alias="DatiBeniServizi")
    ]


class FatturaElettronicaHeader(BaseModel):
    dati_trasmissione: Annotated[
        DatiTrasmissione, Field(serialization_alias="DatiTrasmissione")
    ]
    cedente_prestatore: Annotated[
        CedentePrestatore, Field(serialization_alias="CedentePrestatore")
    ]
    cessionario_committente: Annotated[
        CessionarioCommittente, Field(serialization_alias="CessionarioCommittente")
    ]


class FatturaElettronica(BaseModel):
    fattura_elettronica_header: Annotated[
        FatturaElettronicaHeader, Field(serialization_alias="FatturaElettronicaHeader")
    ]
    fattura_elettronica_body: Annotated[
        list[FatturaElettronicaBody],
        Field(serialization_alias="FatturaElettronicaBody"),
    ]


class OrderSerializer:
    def __init__(self, order: Order) -> None:
        self._order = order

    @classmethod
    def serialize_invoices(cls, order: Order) -> list[FatturaElettronica]:
        return cls(order)._serialize_invoices()

    def _serialize_invoices(self) -> list[FatturaElettronica]:
        return [
            (invoice, InvoiceSerializer.serialize(invoice))
            for invoice in self._invoices
            if invoice not in self._invoice_already_sent
        ]

    @property
    def _invoices(self) -> list[Invoice]:
        return self._order.invoices.all()

    @property
    def _invoice_already_sent(self) -> set[Invoice]:
        already_sent = InvoiceLog.objects.filter(uuid__isnull=False)
        return set([inv.invoice for inv in already_sent])


class InvoiceSerializer:
    def __init__(self, invoice: Invoice) -> None:
        self._invoice = invoice

    @classmethod
    def serialize(cls, invoice: Invoice) -> FatturaElettronica:
        return cls(invoice)._serialize()

    def _serialize(self) -> FatturaElettronica:
        return FatturaElettronica(
            fattura_elettronica_header=self._invoice_header,
            fattura_elettronica_body=[self._invoice_body],
        )

    @property
    def _invoice_body(self) -> FatturaElettronicaBody:
        inv = self._invoice
        tipo_doc = DT.TD04 if inv.canceled and inv.is_cancellation else DT.TD01
        dati_generali = DatiGenerali(
            dati_generali_documento=DatiGeneraliDocumento(
                tipo_documento=tipo_doc,
                divisa=inv.event.currency,
                data=inv.date,
                numero=inv.number,
            )
        )
        lines = inv.lines.all()
        dettaglio_linee = [
            DettaglioLinea(
                numero_linea=i + 1,
                descrizione=line.description,
                prezzo_unitario=str(line.net_value),
                prezzo_totale=str(line.net_value),
                aliquota_iva=str(line.tax_rate),
                quantita="1.00",
            )
            for i, line in enumerate(lines)
        ]
        tax_summary = (
            inv.lines.all()
            .values("tax_rate", "tax_value", "gross_value")
            .annotate(count=Count("tax_rate"))
        )
        dati_riepilogo = [
            DatiRiepilogo(
                aliquota_iva=str(tax.get("tax_rate")),
                imponibile_importo=str(tax.get("gross_value") - tax.get("tax_value")),
                imposta=str(tax.get("tax_value")),
            )
            for tax in tax_summary
        ]
        dati_beni_servizi = DatiBeniServizi(
            dettaglio_linee=dettaglio_linee, dati_riepilogo=dati_riepilogo
        )
        return FatturaElettronicaBody(
            dati_generali=dati_generali, dati_beni_servizi=dati_beni_servizi
        )

    @property
    def _invoice_header(self) -> FatturaElettronicaHeader:
        inv = self._invoice
        recipient_vat_id, recipient_cf = self._recipient_fiscal_data
        codice_dest, pec_dest = self._recipient_fe_data
        dati_trasmissione = DatiTrasmissione(
            id_trasmittente=IdFiscaleIVA(
                id_paese=inv.invoice_from_country.code,
                id_codice=inv.invoice_from_vat_id,
            ),
            codice_destinatario=codice_dest,
            pec_destinatario=pec_dest,
            progressivo_invio=inv.number,
        )
        # Cedente Prestatore is who issue the invoice: e.g. Python Italia APS
        cedente_prestatore = CedentePrestatore(
            dati_anagrafici=DatiAnagraficiCedente(
                id_fiscale_iva=IdFiscaleIVA(
                    id_paese=inv.invoice_from_country.code,
                    id_codice=inv.invoice_from_vat_id,
                ),
                codice_fiscale=SETTINGS.CF,
                anagrafica=Anagrafica(denominazione=inv.invoice_from_name),
                regime_fiscale=SETTINGS.REGIME_FISCALE,
            ),
            sede=Sede(
                indirizzo=inv.invoice_from,
                numero_civico=None,
                cap=inv.invoice_from_zipcode,
                comune=inv.invoice_from_city,
                provincia=None,
                nazione=inv.invoice_from_country.code,
            ),
            contatti=Contatti(email=SETTINGS.EMAIL),
        )
        cessionario_committente = CessionarioCommittente(
            dati_anagrafici=DatiAnagraficiCessionario(
                id_fiscale_iva=IdFiscaleIVA(
                    id_paese=inv.invoice_to_country.code,
                    id_codice=recipient_vat_id,
                )
                if recipient_vat_id
                else None,
                codice_fiscale=recipient_cf,
                anagrafica=self._recipient_registry_data,
            ),
            sede=Sede(
                indirizzo=inv.invoice_to,
                numero_civico=None,
                cap=inv.invoice_to_zipcode,
                comune=inv.invoice_to_city,
                provincia=None,
                nazione=inv.invoice_to_country.code,
            ),
        )
        return FatturaElettronicaHeader(
            dati_trasmissione=dati_trasmissione,
            cedente_prestatore=cedente_prestatore,
            cessionario_committente=cessionario_committente,
        )

    @property
    def _recipient_registry_data(self) -> Anagrafica:
        inv = self._invoice
        complete_name = inv.order.invoice_address.name
        family_name = complete_name.rsplit(" ", 1)[-1] if complete_name else None
        name = (
            complete_name.rsplit(" ", 1)[0]
            if complete_name and " " in complete_name
            else None
        )
        return Anagrafica(
            denominazione=inv.invoice_to_company or None,
            nome=name,
            cognome=family_name,
        )

    @property
    def _recipient_fe_data(self) -> tuple[str, str]:
        inv = self._invoice
        is_business = inv.order.invoice_address.is_business
        meta_info = json.loads(inv.order.meta_info)
        if is_business:
            codice_destinatario = meta_info.get("sdi")
            pec_destinatario = meta_info.get("pec")
            if not codice_destinatario:
                raise ValueError("For a business invoice codice dest is required.")
        else:
            codice_destinatario = SETTINGS.CODICE_DESTINATARIO_DEFAULT
            pec_destinatario = meta_info.get("pec")
        return codice_destinatario, pec_destinatario

    @property
    def _recipient_fiscal_data(self) -> tuple[str | None, str]:
        inv = self._invoice

        is_business = inv.order.invoice_address.is_business
        codice_fiscale = get_codice_fiscale(inv.order)

        if is_business:
            vat_id = inv.invoice_to_vat_id
            if not vat_id:
                raise ValueError("For a business invoice VAT ID is required.")
        else:
            vat_id = None

            if not codice_fiscale:
                raise ValueError("Codice fiscale is required.")

        return vat_id, codice_fiscale
