import numpy as np
import time
from co2_potential.wrapper import (
    p1b, p1b_gradient, p1b_hessian_rev, p1b_hessian_fwd,
    p2b_4, p2b_gradient_4, p2b_hessian_4_rev, p2b_hessian_4_fwd,
    p2b_5, p2b_gradient_5, p2b_hessian_5_rev, p2b_hessian_5_fwd,
    sapt, sapt_gradient, sapt_hessian_rev, sapt_hessian_fwd
)

# Reference values for test geometries
REFERENCE = {
    "p1b_energy": 0.000243,         # kcal/mol
    "p2b_4_energy": 0.084233,       # kcal/mol    
    "p2b_5_energy": -0.022864,      # kcal/mol     
    "sapt_energy": -0.011835,       # kcal/mol    

    "p1b_gradient": [ # kcal/mol/angstrom
                     0.0000000000, 0.0000000000, -0.0000000000, 
                     0.0000000000, 0.0000000000,  0.5810561638, 
                     0.0000000000, 0.0000000000, -0.5810561638],

    "p2b_4_gradient": [ # kcal/mol/angstrom
                       1.2595053930, 0.0000000000, -0.0000000000, 
                      -0.5409125063, 0.0000000000,  0.2403028594, 
                      -0.5409125063, 0.0000000000, -0.2403028594, 
                      -1.2595053930, 0.0000000000,  0.0000000000, 
                       0.5409125063, 0.0000000000,  0.2403028594, 
                       0.5409125063, 0.0000000000, -0.2403028594],

    "p2b_5_gradient": [ # kcal/mol/angstrom
                       0.7108425295, 0.0000000000,  0.0000000000, 
                      -0.5660998637, 0.0000000000,  0.1570323202, 
                      -0.5660998637, 0.0000000000, -0.1570323202, 
                      -0.7108425295, 0.0000000000, -0.0000000000, 
                       0.5660998637, 0.0000000000,  0.1570323202, 
                       0.5660998637, 0.0000000000, -0.1570323202],  

    "sapt_gradient": [  # kcal/mol/angstrom
                      2.1878721003, 0.0000000000,  0.0000000000, 
                     -1.0334215608, 0.0000000000,  0.2315552428, 
                     -1.0334215608, 0.0000000000, -0.2315552428, 
                     -2.1878721003, 0.0000000000,  0.0000000000, 
                      1.0334215608, 0.0000000000,  0.2315552428, 
                      1.0334215608, 0.0000000000, -0.2315552428],  

    "p1b_hessian": [ # kcal/mol/angstrom^2
                    [334.80654,    0.00000,    0.00000, -167.40327,    0.00000,    0.00000, -167.40327,    0.00000,    0.00000],
                    [  0.00000,  334.80654,    0.00000,    0.00000, -167.40327,    0.00000,    0.00000, -167.40327,    0.00000],
                    [  0.00000,    0.00000, 4249.75635,    0.00000,    0.00000, -2124.87817,    0.00000,    0.00000, -2124.87817],
                    [-167.40327,    0.00000,    0.00000,   83.45161,    0.00000,    0.00000,   83.95166,    0.00000,    0.00000],
                    [  0.00000, -167.40327,    0.00000,    0.00000,   83.45161,    0.00000,    0.00000,   83.95166,    0.00000],
                    [  0.00000,    0.00000, -2124.87817,    0.00000,    0.00000, 2304.89183,    0.00000,    0.00000, -180.01366],
                    [-167.40327,    0.00000,    0.00000,   83.95166,    0.00000,    0.00000,   83.45161,    0.00000,    0.00000],
                    [  0.00000, -167.40327,    0.00000,    0.00000,   83.95166,    0.00000,    0.00000,   83.45161,    0.00000],
                    [  0.00000,    0.00000, -2124.87817,    0.00000,    0.00000, -180.01366,    0.00000,    0.00000, 2304.89183],
                   ],      

    "p2b_4_hessian": [ # kcal/mol/angstrom^2
                        [ 1.11920,  0.00000,  0.00000,  0.63730,  0.00000, -3.01519,  0.63730,  0.00000,  3.01519, -1.28336,  0.00000, -0.00000, -0.55522,  0.00000, -2.62010, -0.55522,  0.00000,  2.62010],
                        [ 0.00000, -6.22420,  0.00000,  0.00000,  2.94416,  0.00000,  0.00000,  2.94416,  0.00000,  0.00000,  5.01610,  0.00000,  0.00000, -2.34012,  0.00000,  0.00000, -2.34012,  0.00000],
                        [ 0.00000,  0.00000,-14.63625, -2.94211,  0.00000,  4.65449,  2.94211,  0.00000,  4.65449,  0.00000,  0.00000, 42.64569, -5.11199,  0.00000,-18.65921,  5.11199,  0.00000,-18.65921],
                        [ 0.63730,  0.00000, -2.94211, -1.37571,  0.00000,  2.59991,  0.12423,  0.00000, -0.54871, -0.55522,  0.00000,  5.11199, -0.20629,  0.00000, -0.73378,  1.37570,  0.00000, -3.48731],
                        [ 0.00000,  2.94416,  0.00000,  0.00000, -1.65732,  0.00000,  0.00000, -1.14260,  0.00000,  0.00000, -2.34012,  0.00000,  0.00000,  1.25190,  0.00000,  0.00000,  0.94398,  0.00000],
                        [-3.01519,  0.00000,  4.65449,  2.59991,  0.00000, -3.46292,  0.54871,  0.00000,  0.89625,  2.62010,  0.00000,-18.65921,  0.73378,  0.00000, 10.42482, -3.48731,  0.00000,  6.14657],
                        [ 0.63730,  0.00000,  2.94211,  0.12423,  0.00000,  0.54871, -1.37571,  0.00000, -2.59991, -0.55522,  0.00000, -5.11199,  1.37570,  0.00000,  3.48731, -0.20629,  0.00000,  0.73378],
                        [ 0.00000,  2.94416,  0.00000,  0.00000, -1.14260,  0.00000,  0.00000, -1.65732,  0.00000,  0.00000, -2.34012,  0.00000,  0.00000,  0.94398,  0.00000,  0.00000,  1.25190,  0.00000],
                        [ 3.01519,  0.00000,  4.65449, -0.54871,  0.00000,  0.89625, -2.59991,  0.00000, -3.46292, -2.62010,  0.00000,-18.65921,  3.48731,  0.00000,  6.14657, -0.73378,  0.00000, 10.42482],
                        [-1.28336,  0.00000,  0.00000, -0.55522,  0.00000,  2.62010, -0.55522,  0.00000, -2.62010,  1.11920,  0.00000,  0.00000,  0.63730,  0.00000,  3.01519,  0.63730,  0.00000, -3.01519],
                        [ 0.00000,  5.01610,  0.00000,  0.00000, -2.34012,  0.00000,  0.00000, -2.34012,  0.00000,  0.00000, -6.22420,  0.00000,  0.00000,  2.94416,  0.00000,  0.00000,  2.94416,  0.00000],
                        [-0.00000,  0.00000, 42.64569,  5.11199,  0.00000,-18.65921, -5.11199,  0.00000,-18.65921,  0.00000,  0.00000,-14.63625,  2.94211,  0.00000,  4.65449, -2.94211,  0.00000,  4.65449],
                        [-0.55522,  0.00000, -5.11199, -0.20629,  0.00000,  0.73378,  1.37570,  0.00000,  3.48731,  0.63730,  0.00000,  2.94211, -1.37571,  0.00000, -2.59991,  0.12423,  0.00000,  0.54871],
                        [ 0.00000, -2.34012,  0.00000,  0.00000,  1.25190,  0.00000,  0.00000,  0.94398,  0.00000,  0.00000,  2.94416,  0.00000,  0.00000, -1.65732,  0.00000,  0.00000, -1.14260,  0.00000],
                        [-2.62010,  0.00000,-18.65921, -0.73378,  0.00000, 10.42482,  3.48731,  0.00000,  6.14657,  3.01519,  0.00000,  4.65449, -2.59991,  0.00000, -3.46292, -0.54871,  0.00000,  0.89625],
                        [-0.55522,  0.00000,  5.11199,  1.37570,  0.00000, -3.48731, -0.20629,  0.00000, -0.73378,  0.63730,  0.00000, -2.94211,  0.12423,  0.00000, -0.54871, -1.37571,  0.00000,  2.59991],
                        [ 0.00000, -2.34012,  0.00000,  0.00000,  0.94398,  0.00000,  0.00000,  1.25190,  0.00000,  0.00000,  2.94416,  0.00000,  0.00000, -1.14260,  0.00000,  0.00000, -1.65732,  0.00000],
                        [ 2.62010,  0.00000,-18.65921, -3.48731,  0.00000,  6.14657,  0.73378,  0.00000, 10.42482, -3.01519,  0.00000,  4.65449,  0.54871,  0.00000,  0.89625,  2.59991,  0.00000, -3.46292],
                      ], 

    "p2b_5_hessian": [ # kcal/mol/angstrom^2
                        [-1.81462,  0.00000, -0.00000,  1.54872,  0.00000, -0.93270,  1.54872,  0.00000,  0.93270, -8.62241,  0.00000, -0.00000,  3.66980,  0.00000,  0.67969,  3.66980,  0.00000, -0.67969],
                        [ 0.00000, -2.25918,  0.00000,  0.00000,  1.03481,  0.00000,  0.00000,  1.03481,  0.00000,  0.00000,  5.20412,  0.00000,  0.00000, -2.50728,  0.00000,  0.00000, -2.50728,  0.00000],
                        [-0.00000,  0.00000, -7.79561, -2.59003,  0.00000,  1.50520,  2.59003,  0.00000,  1.50520,  0.00000,  0.00000, 50.50690, -4.82551,  0.00000, -22.86084,  4.82551,  0.00000, -22.86084],
                        [ 1.54872,  0.00000, -2.59003, -1.92325,  0.00000,  1.16054, -0.45822,  0.00000,  0.52866,  3.66980,  0.00000,  4.82551, -2.20717,  0.00000, -2.45259, -0.62989,  0.00000, -1.47209],
                        [ 0.00000,  1.03481,  0.00000,  0.00000, -0.67465,  0.00000,  0.00000, -0.20920,  0.00000,  0.00000, -2.50728,  0.00000,  0.00000,  1.34332,  0.00000,  0.00000,  1.01300,  0.00000],
                        [-0.93270,  0.00000,  1.50520,  1.16054,  0.00000, -0.72425, -0.52866,  0.00000,  1.10957, -0.67969,  0.00000, -22.86084,  2.45259,  0.00000,  8.74900, -1.47209,  0.00000, 12.22132],
                        [ 1.54872,  0.00000,  2.59003, -0.45822,  0.00000, -0.52866, -1.92325,  0.00000, -1.16054,  3.66980,  0.00000, -4.82551, -0.62989,  0.00000,  1.47209, -2.20717,  0.00000,  2.45259],
                        [ 0.00000,  1.03481,  0.00000,  0.00000, -0.20920,  0.00000,  0.00000, -0.67465,  0.00000,  0.00000, -2.50728,  0.00000,  0.00000,  1.01300,  0.00000,  0.00000,  1.34332,  0.00000],
                        [ 0.93270,  0.00000,  1.50520,  0.52866,  0.00000,  1.10957, -1.16054,  0.00000, -0.72425,  0.67969,  0.00000, -22.86084,  1.47209,  0.00000, 12.22132, -2.45259,  0.00000,  8.74900],
                        [-8.62241,  0.00000,  0.00000,  3.66980,  0.00000, -0.67969,  3.66980,  0.00000,  0.67969, -1.81462,  0.00000, -0.00000,  1.54872,  0.00000,  0.93270,  1.54872,  0.00000, -0.93270],
                        [ 0.00000,  5.20412,  0.00000,  0.00000, -2.50728,  0.00000,  0.00000, -2.50728,  0.00000,  0.00000, -2.25918,  0.00000,  0.00000,  1.03481,  0.00000,  0.00000,  1.03481,  0.00000],
                        [-0.00000,  0.00000, 50.50690,  4.82551,  0.00000, -22.86084, -4.82551,  0.00000, -22.86084, -0.00000,  0.00000, -7.79561,  2.59003,  0.00000,  1.50520, -2.59003,  0.00000,  1.50520],
                        [ 3.66980,  0.00000, -4.82551, -2.20717,  0.00000,  2.45259, -0.62989,  0.00000,  1.47209,  1.54872,  0.00000,  2.59003, -1.92325,  0.00000, -1.16054, -0.45822,  0.00000, -0.52866],
                        [ 0.00000, -2.50728,  0.00000,  0.00000,  1.34332,  0.00000,  0.00000,  1.01300,  0.00000,  0.00000,  1.03481,  0.00000,  0.00000, -0.67465,  0.00000,  0.00000, -0.20920,  0.00000],
                        [ 0.67969,  0.00000, -22.86084, -2.45259,  0.00000,  8.74900,  1.47209,  0.00000, 12.22132,  0.93270,  0.00000,  1.50520, -1.16054,  0.00000, -0.72425,  0.52866,  0.00000,  1.10957],
                        [ 3.66980,  0.00000,  4.82551, -0.62989,  0.00000, -1.47209, -2.20717,  0.00000, -2.45259,  1.54872,  0.00000, -2.59003, -0.45822,  0.00000,  0.52866, -1.92325,  0.00000,  1.16054],
                        [ 0.00000, -2.50728,  0.00000,  0.00000,  1.01300,  0.00000,  0.00000,  1.34332,  0.00000,  0.00000,  1.03481,  0.00000,  0.00000, -0.20920,  0.00000,  0.00000, -0.67465,  0.00000],
                        [-0.67969,  0.00000, -22.86084, -1.47209,  0.00000, 12.22132,  2.45259,  0.00000,  8.74900, -0.93270,  0.00000,  1.50520, -0.52866,  0.00000,  1.10957,  1.16054,  0.00000, -0.72425],
                      ],

    "sapt_hessian": [ # kcal/mol/angstrom^2
                        [ 0.91476,  0.00000, -0.00000,  0.67994,  0.00000,  0.64496,  0.67994,  0.00000, -0.64496, -15.40234,  0.00000,  0.00000,  6.56386,  0.00000,  1.26169,  6.56386,  0.00000, -1.26169],
                        [ 0.00000, -2.13879,  0.00000,  0.00000,  0.77768,  0.00000,  0.00000,  0.77768,  0.00000,   0.00000,  7.45035,  0.00000,  0.00000, -3.43346,  0.00000,  0.00000, -3.43346,  0.00000],
                        [-0.00000,  0.00000, -0.90445, -0.31329,  0.00000,  0.00000,  0.31329,  0.00000,  0.00000,   0.00000,  0.00000,  1.45216, -0.20470,  0.00000, -0.27386,  0.20470,  0.00000, -0.27386],
                        [ 0.67994,  0.00000, -0.31329, -1.25782,  0.00000, -0.43182,  0.00000,  0.00000,  0.00000,   6.56386,  0.00000,  0.20470, -3.66603,  0.00000, -0.25407, -2.31995,  0.00000,  0.79448],
                        [ 0.00000,  0.77768,  0.00000,  0.00000, -0.50210,  0.00000,  0.00000,  0.00000,  0.00000,   0.00000, -3.43346,  0.00000,  0.00000,  1.73036,  0.00000,  0.00000,  1.42753,  0.00000],
                        [ 0.64496,  0.00000,  0.00000, -0.43182,  0.00000, -0.02569,  0.00000,  0.00000,  0.00000,  -1.26169,  0.00000, -0.27386,  0.25407,  0.00000,  0.21257,  0.79448,  0.00000,  0.08697],
                        [ 0.67994,  0.00000,  0.31329,  0.00000,  0.00000,  0.00000, -1.25782,  0.00000,  0.43182,   6.56386,  0.00000, -0.20470, -2.31995,  0.00000, -0.79448, -3.66603,  0.00000,  0.25407],
                        [ 0.00000,  0.77768,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000, -0.50210,  0.00000,   0.00000, -3.43346,  0.00000,  0.00000,  1.42753,  0.00000,  0.00000,  1.73036,  0.00000],
                        [-0.64496,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.43182,  0.00000, -0.02569,   1.26169,  0.00000, -0.27386, -0.79448,  0.00000,  0.08697, -0.25407,  0.00000,  0.21257],
                        [-15.40234,  0.00000,  0.00000,  6.56386,  0.00000, -1.26169,  6.56386,  0.00000,  1.26169,   0.91476,  0.00000, -0.00000,  0.67994,  0.00000, -0.64496,  0.67994,  0.00000,  0.64496],
                        [ 0.00000,  7.45035,  0.00000,  0.00000, -3.43346,  0.00000,  0.00000, -3.43346,  0.00000,   0.00000, -2.13879,  0.00000,  0.00000,  0.77768,  0.00000,  0.00000,  0.77768,  0.00000],
                        [ 0.00000,  0.00000,  1.45216,  0.20470,  0.00000, -0.27386, -0.20470,  0.00000, -0.27386,  -0.00000,  0.00000, -0.90445,  0.31329,  0.00000,  0.00000, -0.31329,  0.00000,  0.00000],
                        [ 6.56386,  0.00000, -0.20470, -3.66603,  0.00000,  0.25407, -2.31995,  0.00000, -0.79448,   0.67994,  0.00000,  0.31329, -1.25782,  0.00000,  0.43182,  0.00000,  0.00000,  0.00000],
                        [ 0.00000, -3.43346,  0.00000,  0.00000,  1.73036,  0.00000,  0.00000,  1.42753,  0.00000,   0.00000,  0.77768,  0.00000,  0.00000, -0.50210,  0.00000,  0.00000,  0.00000,  0.00000],
                        [ 1.26169,  0.00000, -0.27386, -0.25407,  0.00000,  0.21257, -0.79448,  0.00000,  0.08697,  -0.64496,  0.00000,  0.00000,  0.43182,  0.00000, -0.02569,  0.00000,  0.00000,  0.00000],
                        [ 6.56386,  0.00000,  0.20470, -2.31995,  0.00000,  0.79448, -3.66603,  0.00000, -0.25407,   0.67994,  0.00000, -0.31329,  0.00000,  0.00000,  0.00000, -1.25782,  0.00000, -0.43182],
                        [ 0.00000, -3.43346,  0.00000,  0.00000,  1.42753,  0.00000,  0.00000,  1.73036,  0.00000,   0.00000,  0.77768,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000, -0.50210,  0.00000],
                        [-1.26169,  0.00000, -0.27386,  0.79448,  0.00000,  0.08697,  0.25407,  0.00000,  0.21257,   0.64496,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000, -0.43182,  0.00000, -0.02569],
                    ],  
}
TOL = 1e-5  

def timeit(func):
    def timed_func(*args, **kwargs):
        start = time.time()
        result = func(*args, **kwargs)
        elapsed = time.time() - start
        return result, elapsed
    return timed_func

def get_test_xyz(n_atoms):
    # CO2 monomer: 3 atoms, dimer: 6 atoms
    if n_atoms == 3:
        return np.array([
            0.0,  0.0,  0.000,
            0.0,  0.0, -1.162,
            0.0,  0.0,  1.162
        ], dtype=np.double)
    elif n_atoms == 6:
        return np.array([
            0.0,  0.0,  0.000,
            0.0,  0.0, -1.162,
            0.0,  0.0,  1.162,
            3.75, 0.0,  0.000,
            3.75, 0.0, -1.162,
            3.75, 0.0,  1.162
        ], dtype=np.double)
    else:
        raise ValueError("Only 3 (monomer) or 6 (dimer) atoms supported in test.")

def compare(val, ref, tol=TOL):
    diff = abs(val - ref)
    ok = diff < tol
    status = "PASS" if ok else f"FAIL (diff={diff:.2e})"
    return status

def compare_array(arr, ref, tol=TOL):
    ok = np.allclose(arr, ref, atol=tol)
    status = "PASS" if ok else f"FAIL (max diff={np.max(np.abs(arr-ref)):.2e})"
    return status

def test_energies():
    print("Testing energies...")
    xyz1 = get_test_xyz(3)
    xyz2 = get_test_xyz(6)
    e1, t1 = timeit(p1b)(xyz1)
    e2_4, t2_4 = timeit(p2b_4)(xyz2)
    e2_5, t2_5 = timeit(p2b_5)(xyz2)
    es, ts = timeit(sapt)(xyz2)
    print(f"p1b energy:   {e1:.6f} (in {t1:.6f} s) [{compare(e1, REFERENCE['p1b_energy'])}]")
    print(f"p2b_4 energy: {e2_4:.6f} (in {t2_4:.6f} s) [{compare(e2_4, REFERENCE['p2b_4_energy'])}]")
    print(f"p2b_5 energy: {e2_5:.6f} (in {t2_5:.6f} s) [{compare(e2_5, REFERENCE['p2b_5_energy'])}]")
    print(f"sapt energy:  {es:.6f} (in {ts:.6f} s) [{compare(es, REFERENCE['sapt_energy'])}]")

def test_gradients():
    print("Testing gradients...")
    xyz1 = get_test_xyz(3)
    xyz2 = get_test_xyz(6)
    grad1, t1 = timeit(p1b_gradient)(xyz1)
    grad2_4, t2_4 = timeit(p2b_gradient_4)(xyz2)
    grad2_5, t2_5 = timeit(p2b_gradient_5)(xyz2)
    grads, ts = timeit(sapt_gradient)(xyz2)
    print(f"p1b gradient:   computed in {t1:.6f} s [{compare_array(grad1, REFERENCE['p1b_gradient'])}]")
    print(f"p2b_4 gradient: computed in {t2_4:.6f} s [{compare_array(grad2_4, REFERENCE['p2b_4_gradient'])}]")
    print(f"p2b_5 gradient: computed in {t2_5:.6f} s [{compare_array(grad2_5, REFERENCE['p2b_5_gradient'])}]")
    print(f"sapt gradient:  computed in {ts:.6f} s [{compare_array(grads, REFERENCE['sapt_gradient'])}]")

def test_hessians():
    print("Testing hessians...")
    xyz1 = get_test_xyz(3)
    xyz2 = get_test_xyz(6)
    hess1r, t1r = timeit(p1b_hessian_rev)(xyz1)
    hess1f, t1f = timeit(p1b_hessian_fwd)(xyz1)
    hess2_4r, t2_4r = timeit(p2b_hessian_4_rev)(xyz2)
    hess2_4f, t2_4f = timeit(p2b_hessian_4_fwd)(xyz2)
    hess2_5r, t2_5r = timeit(p2b_hessian_5_rev)(xyz2)
    hess2_5f, t2_5f = timeit(p2b_hessian_5_fwd)(xyz2)
    hesssr, tsr = timeit(sapt_hessian_rev)(xyz2)
    hesssf, tsf = timeit(sapt_hessian_fwd)(xyz2)
    print(f"p1b hessian (rev):   computed in {t1r:.6f} s [{compare_array(hess1r, REFERENCE['p1b_hessian'])}]")
    print(f"p1b hessian (fwd):   computed in {t1f:.6f} s [{compare_array(hess1f, REFERENCE['p1b_hessian'])}]")
    print(f"p2b_4 hessian (rev): computed in {t2_4r:.6f} s [{compare_array(hess2_4r, REFERENCE['p2b_4_hessian'])}]")
    print(f"p2b_4 hessian (fwd): computed in {t2_4f:.6f} s [{compare_array(hess2_4f, REFERENCE['p2b_4_hessian'])}]")
    print(f"p2b_5 hessian (rev): computed in {t2_5r:.6f} s [{compare_array(hess2_5r, REFERENCE['p2b_5_hessian'])}]")
    print(f"p2b_5 hessian (fwd): computed in {t2_5f:.6f} s [{compare_array(hess2_5f, REFERENCE['p2b_5_hessian'])}]")
    print(f"sapt hessian (rev):  computed in {tsr:.6f} s [{compare_array(hesssr, REFERENCE['sapt_hessian'])}]")
    print(f"sapt hessian (fwd):  computed in {tsf:.6f} s [{compare_array(hesssf, REFERENCE['sapt_hessian'])}]")

def test_p1b():
    print("Testing p1b (monomer) functions...")
    xyz = get_test_xyz(3)
    grad = p1b_gradient(xyz)
    hessr = p1b_hessian_rev(xyz)
    hessf = p1b_hessian_fwd(xyz)
    e, te = timeit(p1b)(xyz)
    grad, tg = timeit(p1b_gradient)(xyz)
    hessr, thr = timeit(p1b_hessian_rev)(xyz)
    hessf, thf = timeit(p1b_hessian_fwd)(xyz)
    print(f"p1b energy:   {e:.6f} (in {te:.6f} s) [{compare(e, REFERENCE['p1b_energy'])}]")
    print(f"p1b gradient: computed in {tg:.6f} s [{compare_array(grad, REFERENCE['p1b_gradient'])}]")
    print(f"p1b hessian (rev): computed in {thr:.6f} s [{compare_array(hessr, REFERENCE['p1b_hessian'])}]")
    print(f"p1b hessian (fwd): computed in {thf:.6f} s [{compare_array(hessf, REFERENCE['p1b_hessian'])}]")

def test_p2b_4():
    print("Testing p2b_4 (dimer 4th order) functions...")
    xyz = get_test_xyz(6)
    grad, tg = timeit(p2b_gradient_4)(xyz)
    hessr, thr = timeit(p2b_hessian_4_rev)(xyz)
    hessf, thf = timeit(p2b_hessian_4_fwd)(xyz)
    e, te = timeit(p2b_4)(xyz)
    print(f"p2b_4 energy:   {e:.6f} (in {te:.6f} s) [{compare(e, REFERENCE['p2b_4_energy'])}]")
    print(f"p2b_4 gradient: computed in {tg:.6f} s [{compare_array(grad, REFERENCE['p2b_4_gradient'])}]")
    print(f"p2b_4 hessian (rev): computed in {thr:.6f} s [{compare_array(hessr, REFERENCE['p2b_4_hessian'])}]")
    print(f"p2b_4 hessian (fwd): computed in {thf:.6f} s [{compare_array(hessf, REFERENCE['p2b_4_hessian'])}]")

def test_p2b_5():
    print("Testing p2b_5 (dimer 5th order) functions...")
    xyz = get_test_xyz(6)
    grad, tg = timeit(p2b_gradient_5)(xyz)
    hessr, thr = timeit(p2b_hessian_5_rev)(xyz)
    hessf, thf = timeit(p2b_hessian_5_fwd)(xyz)
    e, te = timeit(p2b_5)(xyz)
    print(f"p2b_5 energy:   {e:.6f} (in {te:.6f} s) [{compare(e, REFERENCE['p2b_5_energy'])}]")
    print(f"p2b_5 gradient: computed in {tg:.6f} s [{compare_array(grad, REFERENCE['p2b_5_gradient'])}]")
    print(f"p2b_5 hessian (rev): computed in {thr:.6f} s [{compare_array(hessr, REFERENCE['p2b_5_hessian'])}]")
    print(f"p2b_5 hessian (fwd): computed in {thf:.6f} s [{compare_array(hessf, REFERENCE['p2b_5_hessian'])}]")

def test_sapt():
    print("Testing sapt functions...")
    xyz = get_test_xyz(6)
    grad, tg = timeit(sapt_gradient)(xyz)
    hessr, thr = timeit(sapt_hessian_rev)(xyz)
    hessf, thf = timeit(sapt_hessian_fwd)(xyz)
    e, te = timeit(sapt)(xyz)
    print(f"sapt energy:   {e:.6f} (in {te:.6f} s) [{compare(e, REFERENCE['sapt_energy'])}]")
    print(f"sapt gradient: computed in {tg:.6f} s [{compare_array(grad, REFERENCE['sapt_gradient'])}]")
    print(f"sapt hessian (rev): computed in {thr:.6f} s [{compare_array(hessr, REFERENCE['sapt_hessian'])}]")
    print(f"sapt hessian (fwd): computed in {thf:.6f} s [{compare_array(hessf, REFERENCE['sapt_hessian'])}]")

def test_all():
    test_energies()
    test_gradients()
    test_hessians()

if __name__ == "__main__":
    import argparse
    parser = argparse.ArgumentParser(description="Benchmark CO2 potential functions.")
    parser.add_argument("--energies", action="store_true", help="Test only energies")
    parser.add_argument("--gradients", action="store_true", help="Test only gradients")
    parser.add_argument("--hessians", action="store_true", help="Test only hessians")
    parser.add_argument("--p1b", action="store_true", help="Test only p1b functions")
    parser.add_argument("--p2b_4", action="store_true", help="Test only p2b_4 functions")
    parser.add_argument("--p2b_5", action="store_true", help="Test only p2b_5 functions")
    parser.add_argument("--sapt", action="store_true", help="Test only sapt functions")
    parser.add_argument("--all", action="store_true", help="Test all (default if no flag given)")
    args = parser.parse_args()

    if args.energies:
        test_energies()
    elif args.gradients:
        test_gradients()
    elif args.hessians:
        test_hessians()
    elif args.p1b:
        test_p1b()
    elif args.p2b_4:
        test_p2b_4()
    elif args.p2b_5:
        test_p2b_5()
    elif args.sapt:
        test_sapt()
    else:
        test_all()