"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("aws-cdk-lib/aws-lambda");
const events = require("aws-cdk-lib/aws-events");
const index_1 = require("../lib/index");
const assertions_1 = require("aws-cdk-lib/assertions");
const cdk = require("aws-cdk-lib");
function deployNewFunc(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler'
        },
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    return new index_1.EventbridgeToLambda(stack, 'test-eventbridge-lambda', props);
}
function deployNewEventBus(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler'
        },
        eventBusProps: { eventBusName: 'test' },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        }
    };
    return new index_1.EventbridgeToLambda(stack, 'test-new-eventbridge-lambda', props);
}
test('check lambda function properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testeventbridgelambdaLambdaFunctionServiceRole6D02CEEE",
                "Arn"
            ]
        },
        Runtime: "nodejs14.x",
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1"
            }
        }
    });
});
test('check lambda function permission for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Permission', {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            "Fn::GetAtt": [
                "testeventbridgelambdaLambdaFunction475423FD",
                "Arn"
            ]
        },
        Principal: "events.amazonaws.com",
        SourceArn: {
            "Fn::GetAtt": [
                "testeventbridgelambdaEventsRule7DB0954D",
                "Arn"
            ]
        }
    });
});
test('check lambda function role for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/lambda/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "LambdaFunctionServiceRolePolicy"
            }
        ]
    });
});
test('check events rule properties for deploy: true', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Events::Rule', {
        ScheduleExpression: "rate(5 minutes)",
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    "Fn::GetAtt": [
                        "testeventbridgelambdaLambdaFunction475423FD",
                        "Arn"
                    ]
                },
                Id: "Target0"
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.eventsRule !== null);
    expect(construct.lambdaFunction !== null);
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    try {
        new index_1.EventbridgeToLambda(stack, 'test-eventbridge-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check eventbus property, snapshot & eventbus exists', () => {
    const stack = new cdk.Stack();
    const construct = deployNewEventBus(stack);
    expect(construct.eventsRule !== null);
    expect(construct.lambdaFunction !== null);
    expect(construct.eventBus !== null);
    // Check whether eventbus exists
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Events::EventBus', 1);
});
test('check exception while passing existingEventBus & eventBusProps', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler'
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: { eventBusName: 'test' },
        existingEventBusInterface: new events.EventBus(stack, `test-existing-eventbus`, { eventBusName: 'test' })
    };
    const app = () => {
        new index_1.EventbridgeToLambda(stack, 'test-eventbridge-lambda', props);
    };
    expect(app).toThrowError();
});
test('check custom event bus resource with props when deploy:true', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler'
        },
        eventBusProps: {
            eventBusName: `testeventbus`
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        }
    };
    new index_1.EventbridgeToLambda(stack, 'test-new-eventbridge-with-props-lambda', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Events::EventBus', {
        Name: `testeventbus`
    });
});
//# sourceMappingURL=data:application/json;base64,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