"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const minimatch = require("minimatch");
const assets_1 = require("./assets");
const bundling_1 = require("./bundling");
const fs_1 = require("./fs");
const names_1 = require("./names");
const cache_1 = require("./private/cache");
const stack_1 = require("./stack");
const stage_1 = require("./stage");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * Stages a file or directory from a location on the file system into a staging directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 *
 * @stability stable
 */
class AssetStaging extends construct_compat_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this.sourcePath = path.resolve(props.sourcePath);
        this.fingerprintOptions = props;
        if (!fs.existsSync(this.sourcePath)) {
            throw new Error(`Cannot find asset at ${this.sourcePath}`);
        }
        this.sourceStats = fs.statSync(this.sourcePath);
        const outdir = (_b = stage_1.Stage.of(this)) === null || _b === void 0 ? void 0 : _b.assetOutdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly asset output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        this.assetOutdir = outdir;
        // Determine the hash type based on the props as props.assetHashType is
        // optional from a caller perspective.
        this.customSourceFingerprint = props.assetHash;
        this.hashType = determineHashType(props.assetHashType, this.customSourceFingerprint);
        // Decide what we're going to do, without actually doing it yet
        let stageThisAsset;
        let skip = false;
        if (props.bundling) {
            // Check if we actually have to bundle for this stack
            const bundlingStacks = (_c = this.node.tryGetContext(cxapi.BUNDLING_STACKS)) !== null && _c !== void 0 ? _c : ['*'];
            skip = !bundlingStacks.find(pattern => minimatch(stack_1.Stack.of(this).stackName, pattern));
            const bundling = props.bundling;
            stageThisAsset = () => this.stageByBundling(bundling, skip);
        }
        else {
            stageThisAsset = () => this.stageByCopying();
        }
        // Calculate a cache key from the props. This way we can check if we already
        // staged this asset and reuse the result (e.g. the same asset with the same
        // configuration is used in multiple stacks). In this case we can completely
        // skip file system and bundling operations.
        //
        // The output directory and whether this asset is skipped or not should also be
        // part of the cache key to make sure we don't accidentally return the wrong
        // staged asset from the cache.
        this.cacheKey = calculateCacheKey({
            outdir: this.assetOutdir,
            sourcePath: path.resolve(props.sourcePath),
            bundling: props.bundling,
            assetHashType: this.hashType,
            customFingerprint: this.customSourceFingerprint,
            extraHash: props.extraHash,
            exclude: props.exclude,
            ignoreMode: props.ignoreMode,
            skip,
        });
        const staged = AssetStaging.assetCache.obtain(this.cacheKey, stageThisAsset);
        this.stagedPath = staged.stagedPath;
        this.absoluteStagedPath = staged.stagedPath;
        this.assetHash = staged.assetHash;
        this.packaging = staged.packaging;
        this.isArchive = staged.isArchive;
    }
    /**
     * Clears the asset hash cache.
     *
     * @stability stable
     */
    static clearAssetHashCache() {
        this.assetCache.clear();
    }
    /**
     * (deprecated) A cryptographic hash of the asset.
     *
     * @deprecated see `assetHash`.
     */
    get sourceHash() {
        return this.assetHash;
    }
    /**
     * Return the path to the staged asset, relative to the Cloud Assembly (manifest) directory of the given stack.
     *
     * Only returns a relative path if the asset was staged, returns an absolute path if
     * it was not staged.
     *
     * A bundled asset might end up in the outDir and still not count as
     * "staged"; if asset staging is disabled we're technically expected to
     * reference source directories, but we don't have a source directory for the
     * bundled outputs (as the bundle output is written to a temporary
     * directory). Nevertheless, we will still return an absolute path.
     *
     * A non-obvious directory layout may look like this:
     *
     * ```
     *    CLOUD ASSEMBLY ROOT
     *      +-- asset.12345abcdef/
     *      +-- assembly-Stage
     *            +-- MyStack.template.json
     *            +-- MyStack.assets.json <- will contain { "path": "../asset.12345abcdef" }
     * ```
     *
     * @stability stable
     */
    relativeStagedPath(stack) {
        var _b;
        const asmManifestDir = (_b = stage_1.Stage.of(stack)) === null || _b === void 0 ? void 0 : _b.outdir;
        if (!asmManifestDir) {
            return this.stagedPath;
        }
        const isOutsideAssetDir = path.relative(this.assetOutdir, this.stagedPath).startsWith('..');
        if (isOutsideAssetDir || this.stagingDisabled) {
            return this.stagedPath;
        }
        return path.relative(asmManifestDir, this.stagedPath);
    }
    /**
     * Stage the source to the target by copying
     *
     * Optionally skip if staging is disabled, in which case we pretend we did something but we don't really.
     */
    stageByCopying() {
        const assetHash = this.calculateHash(this.hashType);
        const stagedPath = this.stagingDisabled
            ? this.sourcePath
            : path.resolve(this.assetOutdir, renderAssetFilename(assetHash, path.extname(this.sourcePath)));
        if (!this.sourceStats.isDirectory() && !this.sourceStats.isFile()) {
            throw new Error(`Asset ${this.sourcePath} is expected to be either a directory or a regular file`);
        }
        this.stageAsset(this.sourcePath, stagedPath, 'copy');
        return {
            assetHash,
            stagedPath,
            packaging: this.sourceStats.isDirectory() ? assets_1.FileAssetPackaging.ZIP_DIRECTORY : assets_1.FileAssetPackaging.FILE,
            isArchive: this.sourceStats.isDirectory() || ARCHIVE_EXTENSIONS.includes(path.extname(this.sourcePath).toLowerCase()),
        };
    }
    /**
     * Stage the source to the target by bundling
     *
     * Optionally skip, in which case we pretend we did something but we don't really.
     */
    stageByBundling(bundling, skip) {
        var _b;
        if (!this.sourceStats.isDirectory()) {
            throw new Error(`Asset ${this.sourcePath} is expected to be a directory when bundling`);
        }
        if (skip) {
            // We should have bundled, but didn't to save time. Still pretend to have a hash.
            // If the asset uses OUTPUT or BUNDLE, we use a CUSTOM hash to avoid fingerprinting
            // a potentially very large source directory. Other hash types are kept the same.
            let hashType = this.hashType;
            if (hashType === assets_1.AssetHashType.OUTPUT || hashType === assets_1.AssetHashType.BUNDLE) {
                this.customSourceFingerprint = names_1.Names.uniqueId(this);
                hashType = assets_1.AssetHashType.CUSTOM;
            }
            return {
                assetHash: this.calculateHash(hashType, bundling),
                stagedPath: this.sourcePath,
                packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
                isArchive: true,
            };
        }
        // Try to calculate assetHash beforehand (if we can)
        let assetHash = this.hashType === assets_1.AssetHashType.SOURCE || this.hashType === assets_1.AssetHashType.CUSTOM
            ? this.calculateHash(this.hashType, bundling)
            : undefined;
        const bundleDir = this.determineBundleDir(this.assetOutdir, assetHash);
        this.bundle(bundling, bundleDir);
        // Check bundling output content and determine if we will need to archive
        const bundlingOutputType = (_b = bundling.outputType) !== null && _b !== void 0 ? _b : bundling_1.BundlingOutput.AUTO_DISCOVER;
        const bundledAsset = determineBundledAsset(bundleDir, bundlingOutputType);
        // Calculate assetHash afterwards if we still must
        assetHash = assetHash !== null && assetHash !== void 0 ? assetHash : this.calculateHash(this.hashType, bundling, bundledAsset.path);
        const stagedPath = path.resolve(this.assetOutdir, renderAssetFilename(assetHash, bundledAsset.extension));
        this.stageAsset(bundledAsset.path, stagedPath, 'move');
        return {
            assetHash,
            stagedPath,
            packaging: bundledAsset.packaging,
            isArchive: true,
        };
    }
    /**
     * Whether staging has been disabled
     */
    get stagingDisabled() {
        return !!this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
    }
    /**
     * Copies or moves the files from sourcePath to targetPath.
     *
     * Moving implies the source directory is temporary and can be trashed.
     *
     * Will not do anything if source and target are the same.
     */
    stageAsset(sourcePath, targetPath, style) {
        // Is the work already done?
        const isAlreadyStaged = fs.existsSync(targetPath);
        if (isAlreadyStaged) {
            if (style === 'move' && sourcePath !== targetPath) {
                fs.removeSync(sourcePath);
            }
            return;
        }
        // Moving can be done quickly
        if (style == 'move') {
            fs.renameSync(sourcePath, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        if (this.sourceStats.isFile()) {
            fs.copyFileSync(sourcePath, targetPath);
        }
        else if (this.sourceStats.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${sourcePath}`);
        }
    }
    /**
     * Determine the directory where we're going to write the bundling output
     *
     * This is the target directory where we're going to write the staged output
     * files if we can (if the hash is fully known), or a temporary directory
     * otherwise.
     */
    determineBundleDir(outdir, sourceHash) {
        if (sourceHash) {
            return path.resolve(outdir, renderAssetFilename(sourceHash));
        }
        // When the asset hash isn't known in advance, bundler outputs to an
        // intermediate directory named after the asset's cache key
        return path.resolve(outdir, `bundling-temp-${this.cacheKey}`);
    }
    /**
     * Bundles an asset to the given directory
     *
     * If the given directory already exists, assume that everything's already
     * in order and don't do anything.
     *
     * @param options Bundling options
     * @param bundleDir Where to create the bundle directory
     * @returns The fully resolved bundle output directory.
     */
    bundle(options, bundleDir) {
        var _b, _c, _d;
        if (fs.existsSync(bundleDir)) {
            return;
        }
        fs.ensureDirSync(bundleDir);
        // Chmod the bundleDir to full access.
        fs.chmodSync(bundleDir, 0o777);
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_b = options.volumes) !== null && _b !== void 0 ? _b : [],
        ];
        let localBundling;
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            localBundling = (_c = options.local) === null || _c === void 0 ? void 0 : _c.tryBundle(bundleDir, options);
            if (!localBundling) {
                options.image.run({
                    command: options.command,
                    user,
                    volumes,
                    environment: options.environment,
                    workingDirectory: (_d = options.workingDirectory) !== null && _d !== void 0 ? _d : AssetStaging.BUNDLING_INPUT_DIR,
                });
            }
        }
        catch (err) {
            // When bundling fails, keep the bundle output for diagnosability, but
            // rename it out of the way so that the next run doesn't assume it has a
            // valid bundleDir.
            const bundleErrorDir = bundleDir + '-error';
            if (fs.existsSync(bundleErrorDir)) {
                // Remove the last bundleErrorDir.
                fs.removeSync(bundleErrorDir);
            }
            fs.renameSync(bundleDir, bundleErrorDir);
            throw new Error(`Failed to bundle asset ${this.node.path}, bundle output is located at ${bundleErrorDir}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            const outputDir = localBundling ? bundleDir : AssetStaging.BUNDLING_OUTPUT_DIR;
            throw new Error(`Bundling did not produce any output. Check that content is written to ${outputDir}.`);
        }
    }
    calculateHash(hashType, bundling, outputDir) {
        var _b;
        // When bundling a CUSTOM or SOURCE asset hash type, we want the hash to include
        // the bundling configuration. We handle CUSTOM and bundled SOURCE hash types
        // as a special case to preserve existing user asset hashes in all other cases.
        if (hashType == assets_1.AssetHashType.CUSTOM || (hashType == assets_1.AssetHashType.SOURCE && bundling)) {
            const hash = crypto.createHash('sha256');
            // if asset hash is provided by user, use it, otherwise fingerprint the source.
            hash.update((_b = this.customSourceFingerprint) !== null && _b !== void 0 ? _b : fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions));
            // If we're bundling an asset, include the bundling configuration in the hash
            if (bundling) {
                hash.update(JSON.stringify(bundling));
            }
            return hash.digest('hex');
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
            case assets_1.AssetHashType.OUTPUT:
                if (!outputDir) {
                    throw new Error(`Cannot use \`${hashType}\` hash type when \`bundling\` is not specified.`);
                }
                return fs_1.FileSystem.fingerprint(outputDir, this.fingerprintOptions);
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
_a = JSII_RTTI_SYMBOL_1;
AssetStaging[_a] = { fqn: "@aws-cdk/core.AssetStaging", version: "1.94.0" };
/**
 * (experimental) The directory inside the bundling container into which the asset sources will be mounted.
 *
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * (experimental) The directory inside the bundling container into which the bundled output should be written.
 *
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
/**
 * Cache of asset hashes based on asset configuration to avoid repeated file
 * system and bundling operations.
 */
AssetStaging.assetCache = new cache_1.Cache();
function renderAssetFilename(assetHash, extension = '') {
    return `asset.${assetHash}${extension}`;
}
/**
 * Determines the hash type from user-given prop values.
 *
 * @param assetHashType Asset hash type construct prop
 * @param customSourceFingerprint Asset hash seed given in the construct props
 */
function determineHashType(assetHashType, customSourceFingerprint) {
    const hashType = customSourceFingerprint
        ? (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.CUSTOM)
        : (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.SOURCE);
    if (customSourceFingerprint && hashType !== assets_1.AssetHashType.CUSTOM) {
        throw new Error(`Cannot specify \`${assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
    }
    if (hashType === assets_1.AssetHashType.CUSTOM && !customSourceFingerprint) {
        throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
    }
    return hashType;
}
/**
 * Calculates a cache key from the props. Normalize by sorting keys.
 */
function calculateCacheKey(props) {
    return crypto.createHash('sha256')
        .update(JSON.stringify(sortObject(props)))
        .digest('hex');
}
/**
 * Recursively sort object keys
 */
function sortObject(object) {
    if (typeof object !== 'object' || object instanceof Array) {
        return object;
    }
    const ret = {};
    for (const key of Object.keys(object).sort()) {
        ret[key] = sortObject(object[key]);
    }
    return ret;
}
/**
 * Returns the single archive file of a directory or undefined
 */
function singleArchiveFile(directory) {
    if (!fs.existsSync(directory)) {
        throw new Error(`Directory ${directory} does not exist.`);
    }
    if (!fs.statSync(directory).isDirectory()) {
        throw new Error(`${directory} is not a directory.`);
    }
    const content = fs.readdirSync(directory);
    if (content.length === 1) {
        const file = path.join(directory, content[0]);
        const extension = path.extname(content[0]).toLowerCase();
        if (fs.statSync(file).isFile() && ARCHIVE_EXTENSIONS.includes(extension)) {
            return file;
        }
    }
    return undefined;
}
/**
 * Returns the bundled asset to use based on the content of the bundle directory
 * and the type of output.
 */
function determineBundledAsset(bundleDir, outputType) {
    const archiveFile = singleArchiveFile(bundleDir);
    // auto-discover means that if there is an archive file, we take it as the
    // bundle, otherwise, we will archive here.
    if (outputType === bundling_1.BundlingOutput.AUTO_DISCOVER) {
        outputType = archiveFile ? bundling_1.BundlingOutput.ARCHIVED : bundling_1.BundlingOutput.NOT_ARCHIVED;
    }
    switch (outputType) {
        case bundling_1.BundlingOutput.NOT_ARCHIVED:
            return { path: bundleDir, packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY };
        case bundling_1.BundlingOutput.ARCHIVED:
            if (!archiveFile) {
                throw new Error('Bundling output directory is expected to include only a single .zip or .jar file when `output` is set to `ARCHIVED`');
            }
            return { path: archiveFile, packaging: assets_1.FileAssetPackaging.FILE, extension: path.extname(archiveFile) };
    }
}
//# sourceMappingURL=data:application/json;base64,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