import logging
import shutil
from os.path import commonpath
from pathlib import Path
from typing import List, Tuple, Union

from flywheel_bids.flywheel_bids_app_toolkit import BIDSAppContext

log = logging.getLogger(__name__)


def check_bids_dir(app_context: BIDSAppContext):
    """Check if the archived input unzipped with dataset description at the top level.

    Sets the BIDS directory to the top-level directory, where the
    dataset_description.json is located. Particularly useful for group-level zipped
    analyses.

    Note to FW devs: This method came out of a test case failure. Using an input archive from either the project-level or subject-level introduces another directory under /flywheel/v0/bids, resulting in gear failure. This method quickly adapts to a (accidental) nested structure.

    Args:
        app_context (BIDSAppContext): Details about the gear setup and BIDS options
    """
    dd_loc = list(Path(app_context.bids_dir).rglob("dataset_description.json"))[0]
    app_context.bids_dir = dd_loc.parent


def create_analysis_output_dir(app_context: BIDSAppContext):
    """Create analysis directory with the unique destination identifier.

    Args:
        app_context (BIDSAppContext): Details about the gear setup and BIDS options
    """
    # Create output directory
    log.info("Creating output directory %s", app_context.analysis_output_dir)
    Path(app_context.analysis_output_dir).mkdir(parents=True, exist_ok=True)


def determine_dir_structure(search_dir: Path):
    """Log directory structure (for debugging).

    The directory structure during any given BIDS App analysis is somewhat obscure.
    If the file path in the code is incorrect, this method can be called to
    help build the debugging message.

    Note: Example use in fw_gear_bids_mriqc.utils.store_iqms._find_output_files

    Args:
        search_dir (Path): directory to log for debugging purposes
    """
    search_path = Path(search_dir)
    for file_path in search_path.glob("**/*"):
        if file_path.is_file():
            log.debug(file_path)


def find_BIDS_algo_results_dir(
    destination_id: str, output_dir: Union[Path, str]
) -> Path:
    """Isolate BIDS results directory from Flywheel general output.

    Args:
        destination_id (str):
        output_dir (Union[Path, str]):

    Returns:
        The path to the BIDS results-specific directory
    """
    top_level = Path(output_dir).parent
    analysis_dirs = list(top_level.rglob(f"**/*{destination_id}*"))
    if len(analysis_dirs) > 1:
        common_path = commonpath(analysis_dirs)
        return Path(common_path)
    elif not analysis_dirs:
        raise ValueError(f"Did not find a folder matching {destination_id}")
    else:
        return analysis_dirs[0]


def make_dirs_and_files(files: Union[List[str], List[Path]]):
    """Create directories and touch files.

    Note: fw_gear_{BIDS_app}.utils.ry_run.py will use this method,
    but it is common enough to be part of a toolkit. One may also
    use it for testing setup.

    Args:
        files  Union[List[str], List[Path]]: paths to files to be created
    """
    for ff in files:
        if Path(ff).exists():
            log.debug("Exists: %s", str(ff))
        else:
            log.debug("Creating: %s", str(ff))
            dir_name = Path(ff).parents[0]
            Path(dir_name).mkdir(parents=True, exist_ok=True)
            Path(ff).touch(mode=0o777, exist_ok=True)


def reconcile_existing_and_unzipped_files(target_dir: Path, unzip_dir: Path):
    """Place unzipped files in the proper directory.

    Often archived/zipped inputs are unzipped into /flywheel/v0/work. Depending on the
    intended destination and whether extraneous enclosing folders exist (i.e., destination IDs), this method can be used to simply and consistently relocate files and/or directories to their intended, final resting places.

    Args:
        target_dir (Path): final, resting place for the dir or file
        unzip_dir (Path): initial location of the unzipped archive (often generated by GTK.utils.zip_tools.unzip_archive)
    """
    for file in unzip_dir.rglob("*"):
        if (target_dir / file.name).exists():
            log.info("Found %s but using existing", file.name)
        else:
            if not target_dir.exists():
                target_dir.mkdir(parents=True)
            log.debug("Moving %s to %s", file.name, target_dir / file.name)
            shutil.move(file, target_dir / file.name)


def split_extension(filename: str) -> Tuple[str, str]:
    """Remove flexible number of extensions.

    Imaging files and archives tend to have flexible numbers of extensions.
    This simple method deals with all extensions to return the most basic
    basename. e.g., /path/to/my/output/archive.tar.gz returns 'archive'

    Note: don't extend this method to all filenames, since 1.2.34.567.nii.gz
    or another UID-esque name would return problematically.

    Args:
        filename (str): any filename with or without multiple extensions

    Returns:
        filename (str) Non-UID basename without any extensions
        ext (str) The combined suffixes from the original filename
    """
    ext = "." + ".".join(Path(filename).name.split(".")[1:])
    filename = str(Path(filename).stem.split(".")[0])
    if len(filename) < 4:
        log.warning(f"Is {filename} the correct filename without an extension?")
    return filename, ext
