# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/02_helpers.ipynb.

# %% auto 0
__all__ = ['tool_is_present', 'tools_are_present', 'notification', 'write_list_to_file', 'execute_commands_from_file',
           'get_experiment_name', 'process_sample_sheet', 'find_fastqs', 'add_fastqs_to_metadata', 'get_stbit',
           'add_stbit_to_metadata', 'check_db', 'make_folder_if_not_exists', 'find_dirs', 'find_files']

# %% ../nbs/02_helpers.ipynb 3
# Standard libs to be used in the notebook
import os
import re
import sys
import shutil
from pathlib import Path
import requests
import pandas as pd
import subprocess

# Common to template
# add into settings.ini, requirements, package name is python-dotenv, for conda build ensure `conda config --add channels conda-forge`
import dotenv  # for loading config from .env files, https://pypi.org/project/python-dotenv/
import envyaml  # Allows to loads env vars into a yaml file, https://github.com/thesimj/envyaml
import fastcore  # To add functionality related to nbdev development, https://github.com/fastai/fastcore/
import fastcore.script

# %% ../nbs/02_helpers.ipynb 6
def tool_is_present(tool_name: str) -> bool:
    """This function check for programs reliant
    on other tools that the tool is present."""
    # Checks if a tool is present in current env
    return shutil.which(tool_name) is not None

# %% ../nbs/02_helpers.ipynb 7
def tools_are_present(tool_names: list) -> bool:
    """This function checks for a list of programs
    reliant on other tools that the tool is present."""
    # Checks if a list of tools are present in current env
    tools_present: bool = True
    for tool in tool_names:
        if not tool_is_present(tool):
            print(f"Tool {tool} is not present in current environment", file=sys.stderr)
            tools_present = False

    return tools_present

# %% ../nbs/02_helpers.ipynb 9
def notification(notification: dict) -> bool:
    """This function sends a notification to the user."""
    try:
        r = requests.post(
            url=notification["url"], params=notification["params"], timeout=10
        )
    except requests.exceptions.Timeout:
        print("Timeout occured", file=sys.stderr)
        return False
    except Exception as e:
        print(f"Exception occured: {e}", file=sys.stderr)
        return False
    finally:
        return True

# %% ../nbs/02_helpers.ipynb 10
def write_list_to_file(
    content: list[str], file_path: str, overwrite=True, to_stdout=False
) -> None:
    """This function takes a list of commands and put it to a file.
    This can be manipulated for wrapping in a resource manager."""
    # Create output dir for commands file
    if not os.path.exists(os.path.dirname(file_path)):
        os.makedirs(os.path.dirname(file_path))
    if os.path.isfile(file_path) and not overwrite:
        raise FileExistsError(f"File {file_path} already exists")
    if to_stdout or file_path == "-":
        for line in content:
            print(line)
    else:
        with open(file_path, "w") as f:
            for line in content:
                print(line, file=f)

# %% ../nbs/02_helpers.ipynb 11
def execute_commands_from_file(command_file_path: str) -> None:
    """This function executes commands from a file."""
    os.system(f"bash {command_file_path}")

# %% ../nbs/02_helpers.ipynb 12
def get_experiment_name(sample_sheet_path: Path) -> str:
    """This function parses a sample sheet
    to find the experiment name 'N_WGS_000'."""
    with open(sample_sheet_path, "r") as f:
        file_text = f.read()
    match = re.search("Experiment\ Name,(?P<experiment_name>.+)\n", file_text)
    if match is None or match.lastindex == 0:
        raise Exception(
            f"Could not find experiment name in sample sheet {sample_sheet_path}"
        )
    else:
        return match.group("experiment_name")

# %% ../nbs/02_helpers.ipynb 15
def process_sample_sheet(dir: Path, sample_sheet: str) -> list:
    """This function takes a dir as input, finds a sample sheet in that dir,
    and return a list list of dicts, one dict per sample
    from
    \nSampleID	Organism	SupplyingLab
    \n ecoli1  Ecoli FBI
    \nto
    \n[{'SampleID': 'ecoli1', 'Organism': 'Ecoli', 'SupplyingLab': 'FBI'}]
    """
    sample_sheet_path = os.path.join(dir, sample_sheet)
    if sample_sheet_path.endswith(".xlsx"):
        df = pd.read_excel(sample_sheet_path, engine="openpyxl").astype(str)
    else:  # assume it's a tsv
        df = pd.read_table(sample_sheet_path).astype(str)
    metadata = df.to_dict("records")
    return metadata

# %% ../nbs/02_helpers.ipynb 16
def find_fastqs(dir: Path) -> list:
    """This function finds files that end in .fastq.gz in a dir"""
    fastqs = [
        os.path.join(dir, file)
        for file in os.listdir(dir)
        if file.endswith(".fastq.gz")
    ]
    return fastqs

# %% ../nbs/02_helpers.ipynb 17
def add_fastqs_to_metadata(metadata: list, fastqs) -> list:
    """This function adds the fastq paths to the metadata."""
    for sample_dict in metadata:
        sampleid = sample_dict["SampleID"]
        for fastq in fastqs:
            if (sampleid in fastq) and ("R1" in fastq):
                sample_dict["R1"] = fastq
            elif (sampleid in fastq) and ("R2" in fastq):
                sample_dict["R2"] = fastq
    return metadata

# %% ../nbs/02_helpers.ipynb 18
def get_stbit(stbit: str) -> bool:
    """This function that checks if the stbit was provided,
    if not it sets up a default value for it."""
    if not stbit:
        stbit = "ST:NA,NA"
    return stbit

# %% ../nbs/02_helpers.ipynb 19
def add_stbit_to_metadata(metadata: list, stbit: str) -> list:
    """This function adds the stbit to the metadata."""
    for sample_dict in metadata:
        sample_dict["STBIT"] = stbit
    return metadata

# %% ../nbs/02_helpers.ipynb 20
def check_db(dir: Path) -> bool:
    """This function checks if the database dir exists."""
    if dir is not None:
        assert os.path.exists(dir), f"Database {dir} does not exist"
    return True

# %% ../nbs/02_helpers.ipynb 21
def make_folder_if_not_exists(folder_name):
    """This function creates output folders if they don't exists."""
    if not os.path.exists(folder_name):
        os.makedirs(folder_name)

# %% ../nbs/02_helpers.ipynb 22
def find_dirs(path):
    """This function finds dirs in a path,
    excluding files."""
    dirs = [
        os.path.join(path, dir)
        for dir in os.listdir(path)
        if os.path.isdir(os.path.join(path, dir))
    ]
    return dirs

# %% ../nbs/02_helpers.ipynb 23
def find_files(path):
    """This function finds files in a path,
    excluding dirs."""
    files = [
        os.path.join(path, file)
        for file in os.listdir(path)
        if os.path.isfile(os.path.join(path, file))
    ]
    return files

# %% ../nbs/02_helpers.ipynb 24
def make_folder_if_not_exists(folder_name):
    """This function creates output folders if they don't exists."""
    if not os.path.exists(folder_name):
        os.makedirs(folder_name)
