"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
const ESBUILD_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        this.props = props;
        Bundling.runsLocally = (_c = (_a = Bundling.runsLocally) !== null && _a !== void 0 ? _a : (_b = util_1.getEsBuildVersion()) === null || _b === void 0 ? void 0 : _b.startsWith(ESBUILD_VERSION)) !== null && _c !== void 0 ? _c : false;
        const projectRoot = path.dirname(props.depsLockFilePath);
        this.relativeEntryPath = path.relative(projectRoot, path.resolve(props.entry));
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(projectRoot, path.resolve(props.tsconfig));
        }
        this.externals = [
            ...(_d = props.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'],
            ...(_e = props.nodeModules) !== null && _e !== void 0 ? _e : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? (_f = props.dockerImage) !== null && _f !== void 0 ? _f : cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_g = props.buildArgs) !== null && _g !== void 0 ? _g : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    ESBUILD_VERSION: (_h = props.esbuildVersion) !== null && _h !== void 0 ? _h : ESBUILD_VERSION,
                },
            }) : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    var _a;
                    if (Bundling.runsLocally === false) {
                        process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...(_a = props.environment) !== null && _a !== void 0 ? _a : {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: path.dirname(props.entry),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(path.dirname(options.depsLockFilePath), {
            assetHashType: cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        const pathJoin = osPathJoin(osPlatform);
        const npx = osPlatform === 'win32' ? 'npx.cmd' : 'npx';
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const esbuildCommand = [
            npx, 'esbuild',
            '--bundle', pathJoin(inputDir, this.relativeEntryPath),
            `--target=${(_b = this.props.target) !== null && _b !== void 0 ? _b : toTarget(this.props.runtime)}`,
            '--platform=node',
            `--outfile=${pathJoin(outputDir, 'index.js')}`,
            ...this.props.minify ? ['--minify'] : [],
            ...this.props.sourceMap ? ['--sourcemap'] : [],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner='${this.props.banner}'`] : [],
            ...this.props.footer ? [`--footer='${this.props.footer}'`] : [],
        ].join(' ');
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            let installer = Installer.NPM;
            let lockFile = util_1.LockFile.NPM;
            if (this.props.depsLockFilePath.endsWith(util_1.LockFile.YARN)) {
                lockFile = util_1.LockFile.YARN;
                installer = Installer.YARN;
            }
            const osCommand = new OsCommand(osPlatform);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(outputDir, 'package.json'), { dependencies }),
                osCommand.copy(pathJoin(inputDir, lockFile), pathJoin(outputDir, lockFile)),
                osCommand.changeDirectory(outputDir),
                `${installer} install`,
            ]);
        }
        return chain([
            ...(_d = (_c = this.props.commandHooks) === null || _c === void 0 ? void 0 : _c.beforeBundling(inputDir, outputDir)) !== null && _d !== void 0 ? _d : [],
            esbuildCommand,
            ...(_f = (this.props.nodeModules && ((_e = this.props.commandHooks) === null || _e === void 0 ? void 0 : _e.beforeInstall(inputDir, outputDir)))) !== null && _f !== void 0 ? _f : [],
            depsCommand,
            ...(_h = (_g = this.props.commandHooks) === null || _g === void 0 ? void 0 : _g.afterBundling(inputDir, outputDir)) !== null && _h !== void 0 ? _h : [],
        ]);
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > ${filePath}`;
        }
        return `echo '${stringifiedData}' > ${filePath}`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy ${src} ${dest}`;
        }
        return `cp ${src} ${dest}`;
    }
    changeDirectory(dir) {
        return `cd ${dir}`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,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