"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApp = void 0;
const constructs_1 = require("constructs");
const cfn_resource_1 = require("../cfn-resource");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    if (root.node.scope && !stage_1.Stage.isStage(root)) {
        throw new Error('prepareApp can only be called on a stage or a root construct');
    }
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetCfnResources = findCfnResources(dependency.target);
        const sourceCfnResources = findCfnResources(dependency.source);
        for (const target of targetCfnResources) {
            for (const source of sourceCfnResources) {
                source.addDependsOn(target);
            }
        }
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare its template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    const includeStack = (stack) => {
        if (!stack_1.Stack.isStack(stack)) {
            return false;
        }
        if (!stack.nested) {
            return false;
        }
        // test: if we are not within a stage, then include it.
        if (!stage_1.Stage.of(stack)) {
            return true;
        }
        return stage_1.Stage.of(stack) === root;
    };
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (includeStack(stack)) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findCfnResources(root) {
    return root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource);
}
//# sourceMappingURL=data:application/json;base64,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