"use strict";
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const lib_1 = require("../lib");
const util_1 = require("./util");
function createTree(context) {
    const root = new lib_1.App();
    const highChild = new lib_1.Construct(root, 'HighChild');
    if (context) {
        Object.keys(context).forEach(key => highChild.node.setContext(key, context[key]));
    }
    const child1 = new lib_1.Construct(highChild, 'Child1');
    const child2 = new lib_1.Construct(highChild, 'Child2');
    const child1_1 = new lib_1.Construct(child1, 'Child11');
    const child1_2 = new lib_1.Construct(child1, 'Child12');
    const child1_1_1 = new lib_1.Construct(child1_1, 'Child111');
    const child2_1 = new lib_1.Construct(child2, 'Child21');
    return {
        root, child1, child2, child1_1, child1_2, child1_1_1, child2_1,
    };
}
class MyBeautifulConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
    }
}
/**
 * Returns a string with a tree representation of this construct and it's children.
 */
function toTreeString(node, depth = 0) {
    let out = '';
    for (let i = 0; i < depth; ++i) {
        out += '  ';
    }
    const name = node.node.id || '';
    out += `${node.constructor.name}${name.length > 0 ? ' [' + name + ']' : ''}\n`;
    for (const child of node.node.children) {
        out += toTreeString(child, depth + 1);
    }
    return out;
}
module.exports = {
    'the "Root" construct is a special construct which can be used as the root of the tree'(test) {
        const root = new lib_1.App();
        test.equal(root.node.id, '', 'if not specified, name of a root construct is an empty string');
        test.ok(!root.node.scope, 'no parent');
        test.equal(root.node.children.length, 1);
        test.done();
    },
    'constructs cannot be created with an empty name unless they are root'(test) {
        const root = new lib_1.App();
        test.throws(() => new lib_1.Construct(root, ''));
        test.done();
    },
    'construct.name returns the name of the construct'(test) {
        const t = createTree();
        test.equal(t.child1.node.id, 'Child1');
        test.equal(t.child2.node.id, 'Child2');
        test.equal(t.child1_1.node.id, 'Child11');
        test.equal(t.child1_2.node.id, 'Child12');
        test.equal(t.child1_1_1.node.id, 'Child111');
        test.equal(t.child2_1.node.id, 'Child21');
        test.done();
    },
    'construct id can use any character except the path separator'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'valid');
        new lib_1.Construct(root, 'ValiD');
        new lib_1.Construct(root, 'Va123lid');
        new lib_1.Construct(root, 'v');
        new lib_1.Construct(root, '  invalid');
        new lib_1.Construct(root, 'invalid   ');
        new lib_1.Construct(root, '123invalid');
        new lib_1.Construct(root, 'in valid');
        new lib_1.Construct(root, 'in_Valid');
        new lib_1.Construct(root, 'in-Valid');
        new lib_1.Construct(root, 'in\\Valid');
        new lib_1.Construct(root, 'in.Valid');
        test.done();
    },
    'if construct id contains path seperators, they will be replaced by double-dash'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Boom/Boom/Bam');
        test.deepEqual(c.node.id, 'Boom--Boom--Bam');
        test.done();
    },
    'if "undefined" is forcefully used as an "id", it will be treated as an empty string'(test) {
        const c = new lib_1.Construct(undefined, undefined);
        test.deepEqual(c.node.id, '');
        test.done();
    },
    'dont allow unresolved tokens to be used in construct IDs'(test) {
        // GIVEN
        const root = new lib_1.App();
        const token = lib_1.Lazy.stringValue({ produce: () => 'lazy' });
        // WHEN + THEN
        test.throws(() => new lib_1.Construct(root, `MyID: ${token}`), /Cannot use tokens in construct ID: MyID: \${Token/);
        test.done();
    },
    'construct.uniqueId returns a tree-unique alphanumeric id of this construct'(test) {
        const root = new lib_1.App();
        const child1 = new lib_1.Construct(root, 'This is the first child');
        const child2 = new lib_1.Construct(child1, 'Second level');
        const c1 = new lib_1.Construct(child2, 'My construct');
        const c2 = new lib_1.Construct(child1, 'My construct');
        test.deepEqual(c1.node.path, 'This is the first child/Second level/My construct');
        test.deepEqual(c2.node.path, 'This is the first child/My construct');
        test.deepEqual(c1.node.uniqueId, 'ThisisthefirstchildSecondlevelMyconstruct202131E0');
        test.deepEqual(c2.node.uniqueId, 'ThisisthefirstchildMyconstruct8C288DF9');
        test.done();
    },
    'cannot calculate uniqueId if the construct path is ["Default"]'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Default');
        test.throws(() => c.node.uniqueId, /Unable to calculate a unique id for an empty set of components/);
        test.done();
    },
    'construct.getChildren() returns an array of all children'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Child1');
        new lib_1.Construct(root, 'Child2');
        test.equal(child.node.children.length, 0, 'no children');
        test.equal(root.node.children.length, 3, 'three children are expected');
        test.done();
    },
    'construct.findChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.tryFindChild(child.node.id), child, 'findChild(name) can be used to retrieve the child from a parent');
        test.ok(!root.node.tryFindChild('NotFound'), 'findChild(name) returns undefined if the child is not found');
        test.done();
    },
    'construct.getChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.findChild(child.node.id), child, 'getChild(name) can be used to retrieve the child from a parent');
        test.throws(() => {
            root.node.findChild('NotFound');
        }, '', 'getChild(name) returns undefined if the child is not found');
        test.done();
    },
    'can remove children from the tree using tryRemoveChild()'(test) {
        const root = new lib_1.App();
        const childrenBeforeAdding = root.node.children.length; // Invariant to adding 'Metadata' resource or not
        // Add & remove
        const child = new lib_1.Construct(root, 'Construct');
        test.equals(true, root.node.tryRemoveChild(child.node.id));
        test.equals(false, root.node.tryRemoveChild(child.node.id)); // Second time does nothing
        test.equals(undefined, root.node.tryFindChild(child.node.id));
        test.equals(childrenBeforeAdding, root.node.children.length);
        test.done();
    },
    'construct.toString() and construct.toTreeString() can be used for diagnostics'(test) {
        const t = createTree();
        test.equal(t.root.toString(), '<root>');
        test.equal(t.child1_1_1.toString(), 'HighChild/Child1/Child11/Child111');
        test.equal(t.child2.toString(), 'HighChild/Child2');
        test.equal(toTreeString(t.root), 'App\n  TreeMetadata [Tree]\n  Construct [HighChild]\n    Construct [Child1]\n      Construct [Child11]\n        Construct [Child111]\n      Construct [Child12]\n    Construct [Child2]\n      Construct [Child21]\n');
        test.done();
    },
    'construct.getContext(key) can be used to read a value from context defined at the root level'(test) {
        const context = {
            ctx1: 12,
            ctx2: 'hello',
        };
        const t = createTree(context);
        test.equal(t.child1_2.node.tryGetContext('ctx1'), 12);
        test.equal(t.child1_1_1.node.tryGetContext('ctx2'), 'hello');
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'construct.setContext(k,v) sets context at some level and construct.getContext(key) will return the lowermost value defined in the stack'(test) {
        const root = new lib_1.App();
        const highChild = new lib_1.Construct(root, 'highChild');
        highChild.node.setContext('c1', 'root');
        highChild.node.setContext('c2', 'root');
        const child1 = new lib_1.Construct(highChild, 'child1');
        child1.node.setContext('c2', 'child1');
        child1.node.setContext('c3', 'child1');
        const child2 = new lib_1.Construct(highChild, 'child2');
        const child3 = new lib_1.Construct(child1, 'child1child1');
        child3.node.setContext('c1', 'child3');
        child3.node.setContext('c4', 'child3');
        test.equal(highChild.node.tryGetContext('c1'), 'root');
        test.equal(highChild.node.tryGetContext('c2'), 'root');
        test.equal(highChild.node.tryGetContext('c3'), undefined);
        test.equal(child1.node.tryGetContext('c1'), 'root');
        test.equal(child1.node.tryGetContext('c2'), 'child1');
        test.equal(child1.node.tryGetContext('c3'), 'child1');
        test.equal(child2.node.tryGetContext('c1'), 'root');
        test.equal(child2.node.tryGetContext('c2'), 'root');
        test.equal(child2.node.tryGetContext('c3'), undefined);
        test.equal(child3.node.tryGetContext('c1'), 'child3');
        test.equal(child3.node.tryGetContext('c2'), 'child1');
        test.equal(child3.node.tryGetContext('c3'), 'child1');
        test.equal(child3.node.tryGetContext('c4'), 'child3');
        test.done();
    },
    'construct.setContext(key, value) can only be called before adding any children'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'child1');
        test.throws(() => root.node.setContext('k', 'v'));
        test.done();
    },
    'fails if context key contains unresolved tokens'(test) {
        const root = new lib_1.App();
        test.throws(() => root.node.setContext(`my-${lib_1.Aws.REGION}`, 'foo'), /Invalid context key/);
        test.throws(() => root.node.tryGetContext(lib_1.Aws.REGION), /Invalid context key/);
        test.done();
    },
    'construct.pathParts returns an array of strings of all names from root to node'(test) {
        const tree = createTree();
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'HighChild/Child1/Child11/Child111');
        test.deepEqual(tree.child2.node.path, 'HighChild/Child2');
        test.done();
    },
    'if a root construct has a name, it should be included in the path'(test) {
        const tree = createTree({});
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'HighChild/Child1/Child11/Child111');
        test.done();
    },
    'construct can not be created with the name of a sibling'(test) {
        const root = new lib_1.App();
        // WHEN
        new lib_1.Construct(root, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(root, 'SameName');
        }, /There is already a Construct with name 'SameName' in App/);
        // WHEN
        const c0 = new lib_1.Construct(root, 'c0');
        new lib_1.Construct(c0, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(c0, 'SameName');
        }, /There is already a Construct with name 'SameName' in Construct \[c0\]/);
        test.done();
    },
    'addMetadata(type, data) can be used to attach metadata to constructs FIND_ME'(test) {
        const previousValue = util_1.reEnableStackTraceCollection();
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        test.deepEqual(con.node.metadata, [], 'starts empty');
        con.node.addMetadata('key', 'value');
        con.node.addMetadata('number', 103);
        con.node.addMetadata('array', [123, 456]);
        util_1.restoreStackTraceColection(previousValue);
        test.deepEqual(con.node.metadata[0].type, 'key');
        test.deepEqual(con.node.metadata[0].data, 'value');
        test.deepEqual(con.node.metadata[1].data, 103);
        test.deepEqual(con.node.metadata[2].data, [123, 456]);
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace[1].indexOf('FIND_ME') !== -1, 'First stack line should include this function\s name');
        test.done();
    },
    'addMetadata(type, undefined/null) is ignored'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'Foo');
        con.node.addMetadata('Null', null);
        con.node.addMetadata('Undefined', undefined);
        con.node.addMetadata('True', true);
        con.node.addMetadata('False', false);
        con.node.addMetadata('Empty', '');
        const exists = (key) => con.node.metadata.find(x => x.type === key);
        test.ok(!exists('Null'));
        test.ok(!exists('Undefined'));
        test.ok(exists('True'));
        test.ok(exists('False'));
        test.ok(exists('Empty'));
        test.done();
    },
    'addWarning(message) can be used to add a "WARNING" message entry to the construct'(test) {
        const previousValue = util_1.reEnableStackTraceCollection();
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addWarning('This construct is deprecated, use the other one instead');
        util_1.restoreStackTraceColection(previousValue);
        test.deepEqual(con.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.WARN);
        test.deepEqual(con.node.metadata[0].data, 'This construct is deprecated, use the other one instead');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addError(message) can be used to add a "ERROR" message entry to the construct'(test) {
        const previousValue = util_1.reEnableStackTraceCollection();
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addError('Stop!');
        util_1.restoreStackTraceColection(previousValue);
        test.deepEqual(con.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.ERROR);
        test.deepEqual(con.node.metadata[0].data, 'Stop!');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addInfo(message) can be used to add an "INFO" message entry to the construct'(test) {
        const previousValue = util_1.reEnableStackTraceCollection();
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addInfo('Hey there, how do you do?');
        util_1.restoreStackTraceColection(previousValue);
        test.deepEqual(con.node.metadata[0].type, cxschema.ArtifactMetadataEntryType.INFO);
        test.deepEqual(con.node.metadata[0].data, 'Hey there, how do you do?');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'multiple children of the same type, with explicit names are welcome'(test) {
        const root = new lib_1.App();
        new MyBeautifulConstruct(root, 'mbc1');
        new MyBeautifulConstruct(root, 'mbc2');
        new MyBeautifulConstruct(root, 'mbc3');
        new MyBeautifulConstruct(root, 'mbc4');
        test.ok(root.node.children.length >= 4);
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'construct.validate() can be implemented to perform validation, ConstructNode.validate(construct.node) will return all errors from the subtree (DFS)'(test) {
        class MyConstruct extends lib_1.Construct {
            validate() {
                return ['my-error1', 'my-error2'];
            }
        }
        class YourConstruct extends lib_1.Construct {
            validate() {
                return ['your-error1'];
            }
        }
        class TheirConstruct extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                new YourConstruct(this, 'YourConstruct');
            }
            validate() {
                return ['their-error'];
            }
        }
        class TestStack extends lib_1.App {
            constructor() {
                super();
                new MyConstruct(this, 'MyConstruct');
                new TheirConstruct(this, 'TheirConstruct');
            }
            validate() {
                return ['stack-error'];
            }
        }
        const stack = new TestStack();
        const errors = lib_1.ConstructNode.validate(stack.node).map((v) => ({ path: v.source.node.path, message: v.message }));
        // validate DFS
        test.deepEqual(errors, [
            { path: 'MyConstruct', message: 'my-error1' },
            { path: 'MyConstruct', message: 'my-error2' },
            { path: 'TheirConstruct/YourConstruct', message: 'your-error1' },
            { path: 'TheirConstruct', message: 'their-error' },
            { path: '', message: 'stack-error' },
        ]);
        test.done();
    },
    'construct.lock() protects against adding children anywhere under this construct (direct or indirect)'(test) {
        class LockableConstruct extends lib_1.Construct {
            lockMe() {
                this.node._actualNode._lock();
            }
            unlockMe() {
                this.node._actualNode._unlock();
            }
        }
        const stack = new lib_1.App();
        const c0a = new LockableConstruct(stack, 'c0a');
        const c0b = new lib_1.Construct(stack, 'c0b');
        const c1a = new lib_1.Construct(c0a, 'c1a');
        const c1b = new lib_1.Construct(c0a, 'c1b');
        c0a.lockMe();
        // now we should still be able to add children to c0b, but not to c0a or any its children
        new lib_1.Construct(c0b, 'c1a');
        test.throws(() => new lib_1.Construct(c0a, 'fail1'), /Cannot add children to "c0a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1a, 'fail2'), /Cannot add children to "c0a\/c1a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1b, 'fail3'), /Cannot add children to "c0a\/c1b" during synthesis/);
        c0a.unlockMe();
        new lib_1.Construct(c0a, 'c0aZ');
        new lib_1.Construct(c1a, 'c1aZ');
        new lib_1.Construct(c1b, 'c1bZ');
        test.done();
    },
    'findAll returns a list of all children in either DFS or BFS'(test) {
        // GIVEN
        const c1 = new lib_1.Construct(undefined, '1');
        const c2 = new lib_1.Construct(c1, '2');
        new lib_1.Construct(c1, '3');
        new lib_1.Construct(c2, '4');
        new lib_1.Construct(c2, '5');
        // THEN
        test.deepEqual(c1.node.findAll().map(x => x.node.id), c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id)); // default is PreOrder
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id), ['1', '2', '4', '5', '3']);
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.POSTORDER).map(x => x.node.id), ['4', '5', '2', '3', '1']);
        test.done();
    },
    'ancestors returns a list of parents up to root'(test) {
        const { child1_1_1 } = createTree();
        test.deepEqual(child1_1_1.node.scopes.map(x => x.node.id), ['', 'HighChild', 'Child1', 'Child11', 'Child111']);
        test.done();
    },
    '"root" returns the root construct'(test) {
        const { child1, child2, child1_1_1, root } = createTree();
        test.ok(child1.node.root === root);
        test.ok(child2.node.root === root);
        test.ok(child1_1_1.node.root === root);
        test.done();
    },
    'defaultChild': {
        'returns the child with id "Resource"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Resource');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns the child with id "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'can override defaultChild'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'Resource');
            const defaultChild = new lib_1.Construct(root, 'OtherResource');
            root.node.defaultChild = defaultChild;
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns "undefined" if there is no default'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'child2');
            test.equal(root.node.defaultChild, undefined);
            test.done();
        },
        'fails if there are both "Resource" and "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            new lib_1.Construct(root, 'Resource');
            test.throws(() => root.node.defaultChild, /Cannot determine default child for . There is both a child with id "Resource" and id "Default"/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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