"""Binary Sensor platform for La Marzocco espresso machines."""

from collections.abc import Callable
from dataclasses import dataclass
from typing import cast

from pylamarzocco import LaMarzoccoMachine
from pylamarzocco.const import BackFlushStatus, MachineState, ModelName, WidgetType
from pylamarzocco.models import BackFlush, MachineStatus

from homeassistant.components.binary_sensor import (
    BinarySensorDeviceClass,
    BinarySensorEntity,
    BinarySensorEntityDescription,
)
from homeassistant.const import EntityCategory
from homeassistant.core import HomeAssistant
from homeassistant.helpers.entity_platform import AddConfigEntryEntitiesCallback

from .coordinator import LaMarzoccoConfigEntry
from .entity import LaMarzoccoEntity, LaMarzoccoEntityDescription

# Coordinator is used to centralize the data updates
PARALLEL_UPDATES = 0


@dataclass(frozen=True, kw_only=True)
class LaMarzoccoBinarySensorEntityDescription(
    LaMarzoccoEntityDescription,
    BinarySensorEntityDescription,
):
    """Description of a La Marzocco binary sensor."""

    is_on_fn: Callable[[LaMarzoccoMachine], bool | None]


ENTITIES: tuple[LaMarzoccoBinarySensorEntityDescription, ...] = (
    LaMarzoccoBinarySensorEntityDescription(
        key="water_tank",
        translation_key="water_tank",
        device_class=BinarySensorDeviceClass.PROBLEM,
        is_on_fn=lambda machine: WidgetType.CM_NO_WATER in machine.dashboard.config,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    LaMarzoccoBinarySensorEntityDescription(
        key="brew_active",
        translation_key="brew_active",
        device_class=BinarySensorDeviceClass.RUNNING,
        is_on_fn=(
            lambda machine: cast(
                MachineStatus, machine.dashboard.config[WidgetType.CM_MACHINE_STATUS]
            ).status
            is MachineState.BREWING
        ),
        available_fn=lambda coordinator: not coordinator.websocket_terminated,
        entity_category=EntityCategory.DIAGNOSTIC,
    ),
    LaMarzoccoBinarySensorEntityDescription(
        key="backflush_enabled",
        translation_key="backflush_enabled",
        device_class=BinarySensorDeviceClass.RUNNING,
        is_on_fn=(
            lambda machine: cast(
                BackFlush,
                machine.dashboard.config.get(
                    WidgetType.CM_BACK_FLUSH, BackFlush(status=BackFlushStatus.OFF)
                ),
            ).status
            is BackFlushStatus.REQUESTED
        ),
        entity_category=EntityCategory.DIAGNOSTIC,
        supported_fn=lambda coordinator: (
            coordinator.device.dashboard.model_name is not ModelName.GS3_MP
        ),
    ),
    LaMarzoccoBinarySensorEntityDescription(
        key="websocket_connected",
        translation_key="websocket_connected",
        device_class=BinarySensorDeviceClass.CONNECTIVITY,
        is_on_fn=(lambda machine: machine.websocket.connected),
        entity_category=EntityCategory.DIAGNOSTIC,
        entity_registry_enabled_default=False,
    ),
)


async def async_setup_entry(
    hass: HomeAssistant,
    entry: LaMarzoccoConfigEntry,
    async_add_entities: AddConfigEntryEntitiesCallback,
) -> None:
    """Set up binary sensor entities."""
    coordinator = entry.runtime_data.config_coordinator

    async_add_entities(
        LaMarzoccoBinarySensorEntity(coordinator, description)
        for description in ENTITIES
        if description.supported_fn(coordinator)
    )


class LaMarzoccoBinarySensorEntity(LaMarzoccoEntity, BinarySensorEntity):
    """Binary Sensor representing espresso machine water reservoir status."""

    entity_description: LaMarzoccoBinarySensorEntityDescription

    @property
    def is_on(self) -> bool | None:
        """Return true if the binary sensor is on."""
        return self.entity_description.is_on_fn(self.coordinator.device)
