import iam = require('@aws-cdk/aws-iam');
import s3 = require('@aws-cdk/aws-s3');
import cdk = require('@aws-cdk/cdk');
import { CopyOptions } from './fs/copy-options';
/**
 * Defines the way an asset is packaged before it is uploaded to S3.
 */
export declare enum AssetPackaging {
    /**
     * Path refers to a directory on disk, the contents of the directory is
     * archived into a .zip.
     */
    ZipDirectory = "zip",
    /**
     * Path refers to a single file on disk. The file is uploaded as-is.
     */
    File = "file"
}
export interface AssetProps extends CopyOptions {
    /**
     * The disk location of the asset.
     */
    readonly path: string;
    /**
     * The packaging type for this asset.
     */
    readonly packaging: AssetPackaging;
    /**
     * A list of principals that should be able to read this asset from S3.
     * You can use `asset.grantRead(principal)` to grant read permissions later.
     *
     * @default - No principals that can read file asset.
     */
    readonly readers?: iam.IGrantable[];
}
export interface IAsset extends cdk.IConstruct {
    /**
     * A hash of the source of this asset, which is available at construction time. As this is a plain
     * string, it can be used in construct IDs in order to enforce creation of a new resource when
     * the content hash has changed.
     */
    readonly sourceHash: string;
    /**
     * A hash of the bundle for of this asset, which is only available at deployment time. As this is
     * a late-bound token, it may not be used in construct IDs, but can be passed as a resource
     * property in order to force a change on a resource when an asset is effectively updated. This is
     * more reliable than `sourceHash` in particular for assets which bundling phase involve external
     * resources that can change over time (such as Docker image builds).
     */
    readonly artifactHash: string;
}
/**
 * An asset represents a local file or directory, which is automatically uploaded to S3
 * and then can be referenced within a CDK application.
 */
export declare class Asset extends cdk.Construct implements IAsset {
    /**
     * Attribute that represents the name of the bucket this asset exists in.
     */
    readonly s3BucketName: string;
    /**
     * Attribute which represents the S3 object key of this asset.
     */
    readonly s3ObjectKey: string;
    /**
     * Attribute which represents the S3 URL of this asset.
     * @example https://s3.us-west-1.amazonaws.com/bucket/key
     */
    readonly s3Url: string;
    /**
     * The path to the asset (stringinfied token).
     *
     * If asset staging is disabled, this will just be the original path.
     * If asset staging is enabled it will be the staged path.
     */
    readonly assetPath: string;
    /**
     * The S3 bucket in which this asset resides.
     */
    readonly bucket: s3.IBucket;
    /**
     * Indicates if this asset is a zip archive. Allows constructs to ensure that the
     * correct file type was used.
     */
    readonly isZipArchive: boolean;
    readonly sourceHash: string;
    readonly artifactHash: string;
    /**
     * The S3 prefix where all different versions of this asset are stored
     */
    private readonly s3Prefix;
    constructor(scope: cdk.Construct, id: string, props: AssetProps);
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/awslabs/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     * (e.g. "Code" for AWS::Lambda::Function)
     */
    addResourceMetadata(resource: cdk.CfnResource, resourceProperty: string): void;
    /**
     * Grants read permissions to the principal on the asset's S3 object.
     */
    grantRead(grantee: iam.IGrantable): void;
}
export interface FileAssetProps {
    /**
     * File path.
     */
    readonly path: string;
    /**
     * A list of principals that should be able to read this file asset from S3.
     * You can use `asset.grantRead(principal)` to grant read permissions later.
     *
     * @default - No principals that can read file asset.
     */
    readonly readers?: iam.IGrantable[];
}
/**
 * An asset that represents a file on disk.
 */
export declare class FileAsset extends Asset {
    constructor(scope: cdk.Construct, id: string, props: FileAssetProps);
}
export interface ZipDirectoryAssetProps {
    /**
     * Path of the directory.
     */
    readonly path: string;
    /**
     * A list of principals that should be able to read this ZIP file from S3.
     * You can use `asset.grantRead(principal)` to grant read permissions later.
     *
     * @default - No principals that can read file asset.
     */
    readonly readers?: iam.IGrantable[];
}
/**
 * An asset that represents a ZIP archive of a directory on disk.
 */
export declare class ZipDirectoryAsset extends Asset {
    constructor(scope: cdk.Construct, id: string, props: ZipDirectoryAssetProps);
}
