"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const path = require("path");
const staging_1 = require("./staging");
/**
 * Defines the way an asset is packaged before it is uploaded to S3.
 */
var AssetPackaging;
(function (AssetPackaging) {
    /**
     * Path refers to a directory on disk, the contents of the directory is
     * archived into a .zip.
     */
    AssetPackaging["ZipDirectory"] = "zip";
    /**
     * Path refers to a single file on disk. The file is uploaded as-is.
     */
    AssetPackaging["File"] = "file";
})(AssetPackaging = exports.AssetPackaging || (exports.AssetPackaging = {}));
/**
 * An asset represents a local file or directory, which is automatically uploaded to S3
 * and then can be referenced within a CDK application.
 */
class Asset extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // stage the asset source (conditionally).
        const staging = new staging_1.Staging(this, 'Stage', {
            ...props,
            sourcePath: path.resolve(props.path),
        });
        this.sourceHash = staging.sourceHash;
        this.assetPath = staging.stagedPath;
        // sets isZipArchive based on the type of packaging and file extension
        const allowedExtensions = ['.jar', '.zip'];
        this.isZipArchive = props.packaging === AssetPackaging.ZipDirectory
            ? true
            : allowedExtensions.some(ext => staging.sourcePath.toLowerCase().endsWith(ext));
        validateAssetOnDisk(staging.sourcePath, props.packaging);
        // add parameters for s3 bucket and s3 key. those will be set by
        // the toolkit or by CI/CD when the stack is deployed and will include
        // the name of the bucket and the S3 key where the code lives.
        const bucketParam = new cdk.CfnParameter(this, 'S3Bucket', {
            type: 'String',
            description: `S3 bucket for asset "${this.node.path}"`,
        });
        const keyParam = new cdk.CfnParameter(this, 'S3VersionKey', {
            type: 'String',
            description: `S3 key for asset version "${this.node.path}"`
        });
        const hashParam = new cdk.CfnParameter(this, 'ArtifactHash', {
            description: `Artifact hash for asset "${this.node.path}"`,
            type: 'String',
        });
        this.s3BucketName = bucketParam.stringValue;
        this.s3Prefix = cdk.Fn.select(0, cdk.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, keyParam.stringValue)).toString();
        const s3Filename = cdk.Fn.select(1, cdk.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, keyParam.stringValue)).toString();
        this.s3ObjectKey = `${this.s3Prefix}${s3Filename}`;
        this.artifactHash = hashParam.stringValue;
        this.bucket = s3.Bucket.fromBucketName(this, 'AssetBucket', this.s3BucketName);
        // form the s3 URL of the object key
        this.s3Url = this.bucket.urlForObject(this.s3ObjectKey);
        // attach metadata to the lambda function which includes information
        // for tooling to be able to package and upload a directory to the
        // s3 bucket and plug in the bucket name and key in the correct
        // parameters.
        const asset = {
            path: this.assetPath,
            id: this.node.uniqueId,
            packaging: props.packaging,
            sourceHash: this.sourceHash,
            s3BucketParameter: bucketParam.logicalId,
            s3KeyParameter: keyParam.logicalId,
            artifactHashParameter: hashParam.logicalId,
        };
        this.node.addMetadata(cxapi.ASSET_METADATA, asset);
        for (const reader of (props.readers || [])) {
            this.grantRead(reader);
        }
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/awslabs/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     * (e.g. "Code" for AWS::Lambda::Function)
     */
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.getContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.options.metadata = resource.options.metadata || {};
        resource.options.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.options.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * Grants read permissions to the principal on the asset's S3 object.
     */
    grantRead(grantee) {
        // We give permissions on all files with the same prefix. Presumably
        // different versions of the same file will have the same prefix
        // and we don't want to accidentally revoke permission on old versions
        // when deploying a new version.
        this.bucket.grantRead(grantee, `${this.s3Prefix}*`);
    }
}
exports.Asset = Asset;
/**
 * An asset that represents a file on disk.
 */
class FileAsset extends Asset {
    constructor(scope, id, props) {
        super(scope, id, { packaging: AssetPackaging.File, ...props });
    }
}
exports.FileAsset = FileAsset;
/**
 * An asset that represents a ZIP archive of a directory on disk.
 */
class ZipDirectoryAsset extends Asset {
    constructor(scope, id, props) {
        super(scope, id, { packaging: AssetPackaging.ZipDirectory, ...props });
    }
}
exports.ZipDirectoryAsset = ZipDirectoryAsset;
function validateAssetOnDisk(assetPath, packaging) {
    if (!fs.existsSync(assetPath)) {
        throw new Error(`Cannot find asset at ${assetPath}`);
    }
    switch (packaging) {
        case AssetPackaging.ZipDirectory:
            if (!fs.statSync(assetPath).isDirectory()) {
                throw new Error(`${assetPath} is expected to be a directory when asset packaging is 'zip'`);
            }
            break;
        case AssetPackaging.File:
            if (!fs.statSync(assetPath).isFile()) {
                throw new Error(`${assetPath} is expected to be a regular file when asset packaging is 'file'`);
            }
            break;
        default:
            throw new Error(`Unsupported asset packaging format: ${packaging}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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