"use strict";
const fs = require("fs");
const os = require("os");
const path = require("path");
const copy_1 = require("../../lib/fs/copy");
const follow_mode_1 = require("../../lib/fs/follow-mode");
function tree(dir, depth = '') {
    const lines = [];
    for (const file of fs.readdirSync(dir).sort()) {
        const filePath = path.join(dir, file);
        const stat = fs.lstatSync(filePath);
        if (stat.isSymbolicLink()) {
            const linkDest = fs.readlinkSync(filePath);
            lines.push(depth + file + ' => ' + linkDest);
        }
        else if (stat.isDirectory()) {
            lines.push(depth + file + ' (D)');
            lines.push(...tree(filePath, depth + '    '));
        }
        else {
            lines.push(depth + file);
        }
    }
    return lines;
}
module.exports = {
    'Default: copies all files and subdirectories, with default follow mode is "External"'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'test1'), outdir);
        // THEN
        test.deepEqual(tree(outdir), [
            'external-link.txt',
            'file1.txt',
            'local-link.txt => file1.txt',
            'subdir (D)',
            '    file2.txt',
            'subdir2 (D)',
            '    empty-subdir (D)',
            '        .hidden',
            '    subdir3 (D)',
            '        file3.txt'
        ]);
        test.done();
    },
    'Always: follow all symlinks'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'symlinks'), outdir, {
            follow: follow_mode_1.FollowMode.Always
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'external-dir-link (D)',
            '    file2.txt',
            'external-link.txt',
            'indirect-external-link.txt',
            'local-dir-link (D)',
            '    file-in-subdir.txt',
            'local-link.txt',
            'normal-dir (D)',
            '    file-in-subdir.txt',
            'normal-file.txt'
        ]);
        test.done();
    },
    'Never: do not follow all symlinks'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'symlinks'), outdir, {
            follow: follow_mode_1.FollowMode.Never
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'external-dir-link => ../test1/subdir',
            'external-link.txt => ../test1/subdir2/subdir3/file3.txt',
            'indirect-external-link.txt => external-link.txt',
            'local-dir-link => normal-dir',
            'local-link.txt => normal-file.txt',
            'normal-dir (D)',
            '    file-in-subdir.txt',
            'normal-file.txt'
        ]);
        test.done();
    },
    'External: follow only external symlinks'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'symlinks'), outdir, {
            follow: follow_mode_1.FollowMode.External
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'external-dir-link (D)',
            '    file2.txt',
            'external-link.txt',
            'indirect-external-link.txt => external-link.txt',
            'local-dir-link => normal-dir',
            'local-link.txt => normal-file.txt',
            'normal-dir (D)',
            '    file-in-subdir.txt',
            'normal-file.txt'
        ]);
        test.done();
    },
    'exclude'(test) {
        // GIVEN
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
        // WHEN
        copy_1.copyDirectory(path.join(__dirname, 'fixtures', 'test1'), outdir, {
            exclude: [
                '*',
                '!subdir2',
                '!subdir2/**/*',
                '.*'
            ]
        });
        // THEN
        test.deepEqual(tree(outdir), [
            'subdir2 (D)',
            '    empty-subdir (D)',
            '    subdir3 (D)',
            '        file3.txt'
        ]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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