"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const os = require("os");
const path = require("path");
const asset_1 = require("../lib/asset");
const SAMPLE_ASSET_DIR = path.join(__dirname, 'sample-asset-directory');
function mkdtempSync() {
    return fs.mkdtempSync(path.join(os.tmpdir(), 'test.assets'));
}
module.exports = {
    'simple use case'(test) {
        const stack = new cdk.Stack();
        const asset = new asset_1.ZipDirectoryAsset(stack, 'MyAsset', {
            path: SAMPLE_ASSET_DIR
        });
        // verify that metadata contains an "aws:cdk:asset" entry with
        // the correct information
        const entry = asset.node.metadata.find(m => m.type === 'aws:cdk:asset');
        test.ok(entry, 'found metadata entry');
        // verify that now the template contains parameters for this asset
        const template = assert_1.SynthUtils.toCloudFormation(stack);
        test.deepEqual(stack.node.resolve(entry.data), {
            path: SAMPLE_ASSET_DIR,
            id: 'MyAsset',
            packaging: 'zip',
            sourceHash: '6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2',
            s3BucketParameter: 'MyAssetS3Bucket68C9B344',
            s3KeyParameter: 'MyAssetS3VersionKey68E1A45D',
            artifactHashParameter: 'MyAssetArtifactHashF518BDDE',
        });
        test.equal(template.Parameters.MyAssetS3Bucket68C9B344.Type, 'String');
        test.equal(template.Parameters.MyAssetS3VersionKey68E1A45D.Type, 'String');
        test.done();
    },
    'verify that the app resolves tokens in metadata'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'my-stack');
        const dirPath = path.resolve(__dirname, 'sample-asset-directory');
        new asset_1.ZipDirectoryAsset(stack, 'MyAsset', {
            path: dirPath
        });
        const synth = app.synthesizeStack(stack.name);
        test.deepEqual(synth.metadata['/my-stack/MyAsset'][0].data, {
            path: dirPath,
            id: "mystackMyAssetD6B1B593",
            packaging: "zip",
            sourceHash: '6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2',
            s3BucketParameter: "MyAssetS3Bucket68C9B344",
            s3KeyParameter: "MyAssetS3VersionKey68E1A45D",
            artifactHashParameter: 'MyAssetArtifactHashF518BDDE',
        });
        test.done();
    },
    '"file" assets'(test) {
        const stack = new cdk.Stack();
        const filePath = path.join(__dirname, 'file-asset.txt');
        const asset = new asset_1.FileAsset(stack, 'MyAsset', { path: filePath });
        const entry = asset.node.metadata.find(m => m.type === 'aws:cdk:asset');
        test.ok(entry, 'found metadata entry');
        // synthesize first so "prepare" is called
        const template = assert_1.SynthUtils.toCloudFormation(stack);
        test.deepEqual(stack.node.resolve(entry.data), {
            path: filePath,
            packaging: 'file',
            id: 'MyAsset',
            sourceHash: '78add9eaf468dfa2191da44a7da92a21baba4c686cf6053d772556768ef21197',
            s3BucketParameter: 'MyAssetS3Bucket68C9B344',
            s3KeyParameter: 'MyAssetS3VersionKey68E1A45D',
            artifactHashParameter: 'MyAssetArtifactHashF518BDDE',
        });
        // verify that now the template contains parameters for this asset
        test.equal(template.Parameters.MyAssetS3Bucket68C9B344.Type, 'String');
        test.equal(template.Parameters.MyAssetS3VersionKey68E1A45D.Type, 'String');
        test.done();
    },
    '"readers" or "grantRead" can be used to grant read permissions on the asset to a principal'(test) {
        const stack = new cdk.Stack();
        const user = new iam.User(stack, 'MyUser');
        const group = new iam.Group(stack, 'MyGroup');
        const asset = new asset_1.ZipDirectoryAsset(stack, 'MyAsset', {
            path: path.join(__dirname, 'sample-asset-directory'),
            readers: [user]
        });
        asset.grantRead(group);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: ["s3:GetObject*", "s3:GetBucket*", "s3:List*"],
                        Effect: 'Allow',
                        Resource: [
                            { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":s3:::", { Ref: "MyAssetS3Bucket68C9B344" }]] },
                            { "Fn::Join": ["",
                                    [
                                        "arn:", { Ref: "AWS::Partition" }, ":s3:::", { Ref: "MyAssetS3Bucket68C9B344" },
                                        "/",
                                        { "Fn::Select": [0, { "Fn::Split": ["||", { Ref: "MyAssetS3VersionKey68E1A45D" }] }] },
                                        "*"
                                    ]
                                ] }
                        ]
                    }
                ]
            }
        }));
        test.done();
    },
    'fails if directory not found'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new asset_1.ZipDirectoryAsset(stack, 'MyDirectory', {
            path: '/path/not/found/' + Math.random() * 999999
        }));
        test.done();
    },
    'multiple assets under the same parent'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new asset_1.ZipDirectoryAsset(stack, 'MyDirectory1', { path: path.join(__dirname, 'sample-asset-directory') });
        new asset_1.ZipDirectoryAsset(stack, 'MyDirectory2', { path: path.join(__dirname, 'sample-asset-directory') });
        // THEN: no error
        test.done();
    },
    'isZipArchive indicates if the asset represents a .zip file (either explicitly or via ZipDirectory packaging)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const nonZipAsset = new asset_1.FileAsset(stack, 'NonZipAsset', {
            path: path.join(__dirname, 'sample-asset-directory', 'sample-asset-file.txt')
        });
        const zipDirectoryAsset = new asset_1.ZipDirectoryAsset(stack, 'ZipDirectoryAsset', {
            path: path.join(__dirname, 'sample-asset-directory')
        });
        const zipFileAsset = new asset_1.FileAsset(stack, 'ZipFileAsset', {
            path: path.join(__dirname, 'sample-asset-directory', 'sample-zip-asset.zip')
        });
        const jarFileAsset = new asset_1.FileAsset(stack, 'JarFileAsset', {
            path: path.join(__dirname, 'sample-asset-directory', 'sample-jar-asset.jar')
        });
        // THEN
        test.equal(nonZipAsset.isZipArchive, false);
        test.equal(zipDirectoryAsset.isZipArchive, true);
        test.equal(zipFileAsset.isZipArchive, true);
        test.equal(jarFileAsset.isZipArchive, true);
        test.done();
    },
    'addResourceMetadata can be used to add CFN metadata to resources'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        stack.node.setContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT, true);
        const location = path.join(__dirname, 'sample-asset-directory');
        const resource = new cdk.CfnResource(stack, 'MyResource', { type: 'My::Resource::Type' });
        const asset = new asset_1.ZipDirectoryAsset(stack, 'MyAsset', { path: location });
        // WHEN
        asset.addResourceMetadata(resource, 'PropName');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('My::Resource::Type', {
            Metadata: {
                "aws:asset:path": location,
                "aws:asset:property": "PropName"
            }
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'asset metadata is only emitted if ASSET_RESOURCE_METADATA_ENABLED_CONTEXT is defined'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const resource = new cdk.CfnResource(stack, 'MyResource', { type: 'My::Resource::Type' });
        const asset = new asset_1.ZipDirectoryAsset(stack, 'MyAsset', { path: SAMPLE_ASSET_DIR });
        // WHEN
        asset.addResourceMetadata(resource, 'PropName');
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('My::Resource::Type', {
            Metadata: {
                "aws:asset:path": SAMPLE_ASSET_DIR,
                "aws:asset:property": "PropName"
            }
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'staging': {
        'copy file assets under .assets/${fingerprint}.ext'(test) {
            const tempdir = mkdtempSync();
            process.chdir(tempdir); // change current directory to somewhere in /tmp
            // GIVEN
            const app = new cdk_1.App({
                context: { [cxapi.ASSET_STAGING_DIR_CONTEXT]: '.assets' }
            });
            const stack = new cdk_1.Stack(app, 'stack');
            // WHEN
            new asset_1.FileAsset(stack, 'ZipFile', {
                path: path.join(SAMPLE_ASSET_DIR, 'sample-zip-asset.zip')
            });
            new asset_1.FileAsset(stack, 'TextFile', {
                path: path.join(SAMPLE_ASSET_DIR, 'sample-asset-file.txt')
            });
            // THEN
            app.run();
            test.ok(fs.existsSync(path.join(tempdir, '.assets')));
            test.ok(fs.existsSync(path.join(tempdir, '.assets', 'a7a79cdf84b802ea8b198059ff899cffc095a1b9606e919f98e05bf80779756b.zip')));
            fs.readdirSync(path.join(tempdir, '.assets'));
            test.done();
        },
        'copy directory under .assets/fingerprint/**'(test) {
            const tempdir = mkdtempSync();
            process.chdir(tempdir); // change current directory to somewhere in /tmp
            // GIVEN
            const app = new cdk_1.App({
                context: { [cxapi.ASSET_STAGING_DIR_CONTEXT]: '.assets' }
            });
            const stack = new cdk_1.Stack(app, 'stack');
            // WHEN
            new asset_1.ZipDirectoryAsset(stack, 'ZipDirectory', {
                path: SAMPLE_ASSET_DIR
            });
            // THEN
            app.run();
            test.ok(fs.existsSync(path.join(tempdir, '.assets')));
            const hash = '6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2';
            test.ok(fs.existsSync(path.join(tempdir, '.assets', hash, 'sample-asset-file.txt')));
            test.ok(fs.existsSync(path.join(tempdir, '.assets', hash, 'sample-jar-asset.jar')));
            fs.readdirSync(path.join(tempdir, '.assets'));
            test.done();
        },
        'staging path is relative if the dir is below the working directory'(test) {
            // GIVEN
            const tempdir = mkdtempSync();
            process.chdir(tempdir); // change current directory to somewhere in /tmp
            const staging = '.my-awesome-staging-directory';
            const app = new cdk_1.App({
                context: {
                    [cxapi.ASSET_STAGING_DIR_CONTEXT]: staging,
                    [cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT]: 'true',
                }
            });
            const stack = new cdk_1.Stack(app, 'stack');
            const resource = new cdk.CfnResource(stack, 'MyResource', { type: 'My::Resource::Type' });
            const asset = new asset_1.ZipDirectoryAsset(stack, 'MyAsset', { path: SAMPLE_ASSET_DIR });
            // WHEN
            asset.addResourceMetadata(resource, 'PropName');
            const session = app.run();
            const template = assert_1.SynthUtils.templateForStackName(session, stack.name);
            test.deepEqual(template.Resources.MyResource.Metadata, {
                "aws:asset:path": `.my-awesome-staging-directory/6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2`,
                "aws:asset:property": "PropName"
            });
            test.done();
        },
        'if staging directory is absolute, asset path is absolute'(test) {
            // GIVEN
            const staging = path.resolve(mkdtempSync());
            const app = new cdk_1.App({
                context: {
                    [cxapi.ASSET_STAGING_DIR_CONTEXT]: staging,
                    [cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT]: 'true',
                }
            });
            const stack = new cdk_1.Stack(app, 'stack');
            const resource = new cdk.CfnResource(stack, 'MyResource', { type: 'My::Resource::Type' });
            const asset = new asset_1.ZipDirectoryAsset(stack, 'MyAsset', { path: SAMPLE_ASSET_DIR });
            // WHEN
            asset.addResourceMetadata(resource, 'PropName');
            const session = app.run();
            const template = assert_1.SynthUtils.templateForStackName(session, stack.name);
            test.deepEqual(template.Resources.MyResource.Metadata, {
                "aws:asset:path": `${staging}/6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2`,
                "aws:asset:property": "PropName"
            });
            test.done();
        },
        'cdk metadata points to staged asset'(test) {
            // GIVEN
            const tempdir = mkdtempSync();
            process.chdir(tempdir); // change current directory to somewhere in /tmp
            const staging = '.stageme';
            const app = new cdk_1.App({
                context: {
                    [cxapi.ASSET_STAGING_DIR_CONTEXT]: staging,
                }
            });
            const stack = new cdk_1.Stack(app, 'stack');
            new asset_1.ZipDirectoryAsset(stack, 'MyAsset', { path: SAMPLE_ASSET_DIR });
            // WHEN
            const session = app.run();
            const artifact = session.getArtifact(stack.name);
            const md = Object.values(artifact.metadata || {})[0][0].data;
            test.deepEqual(md.path, '.stageme/6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2');
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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