from __future__ import annotations

import re

from dependency_injector.wiring import Provide, inject
from rich.style import Style
from rich.text import Text

from .. import __version__, styles
from ..containers import Container
from ..jenkins import Jenkins
from ..widgets import ExecutorStatusWidget, TextWidget
from .base import BaseView


class HomeView(BaseView):
    """A view that contains widgets that are displayed on the home screen."""

    @inject
    def __init__(self, client: Jenkins = Provide[Container.client]):
        """A view that contains widgets that are displayed on the home screen.

        # noqa: DAR101 client
        """

        super().__init__()
        self.client = client

    async def on_show(self) -> None:
        """Actions that are executed when the widget is shown."""

        server_version = self.client.version
        client_version = __version__

        HTML = re.compile(r"<[^>]+>")
        clean_description = HTML.sub("", self.client.description)

        self.app.nav.title = Text.assemble(
            *[
                Text.from_markup(f"[{styles.GREY}][bold]{clean_description}[/][/]\n"),
                Text.from_markup(
                    f"server: [{styles.GREEN}]{server_version}[/] [{styles.ORANGE}]⚡[/]client: [{styles.GREEN}]{client_version}[/]"
                ),
            ],
            justify="center",
        )

    async def on_mount(self) -> None:
        """Actions that are executed when the widget is mounted."""

        self.layout.add_column("col")
        self.layout.add_row("info", size=3)
        self.layout.add_row("executor", min_size=25)

        self.layout.add_areas(
            info="col,info",
            executor="col,executor",
        )

        server_address = Text(
            self.client.url, style=Style(link=self.client.url), overflow="ellipsis"
        )

        self.layout.place(
            info=TextWidget(
                text=f"Welcome to Jenkins TUI! 🚀\nYour instance url is: {server_address}",
            )
        )
        self.layout.place(executor=ExecutorStatusWidget())
