"""
Content Aggregator Module for Sabbath School Lessons

This module combines downloaded content into a single markdown file.
"""

import os
from datetime import datetime


class ContentAggregator:
    """Combines downloaded lesson content into a single markdown file."""
    
    @staticmethod
    def create_file_section(filename, content):
        """
        Format content as a file section with header
        
        Args:
            filename (str): Name of the file section
            content (str): Content to include in the section
            
        Returns:
            str: Formatted section with header
        """
        header = f"# File: {filename}\n#------------------------------------------------------------------------------\n\n"
        return header + content + "\n\n"

    @staticmethod
    def combine_lesson_content(lesson_data, output_path):
        """
        Combine all lesson content into a single markdown file
        
        Args:
            lesson_data (dict): Dictionary containing all downloaded content
            output_path (str): Path to save the combined markdown file
            
        Returns:
            str: Path to the combined markdown file
        """
        combined_content = []
        
        # Add a generation timestamp
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        combined_content.append(f"# Combined file generated by SabbathSchoolDownloader on {timestamp}")
        combined_content.append("# Source files from GitHub repository: SabbathSchool/lessons\n\n")
        
        # Add front matter if available
        if lesson_data['front_matter']:
            combined_content.append(ContentAggregator.create_file_section("front-matter.md", lesson_data['front_matter']))
        
        # Add each lesson in order
        # Sort week IDs to ensure correct order (week-01, week-02, etc.)
        sorted_week_ids = sorted(lesson_data['lessons'].keys())
        
        for week_id in sorted_week_ids:
            lesson = lesson_data['lessons'][week_id]
            combined_content.append(ContentAggregator.create_file_section(f"{week_id}.md", lesson['content']))
        
        # Add back matter if available
        if lesson_data['back_matter']:
            combined_content.append(ContentAggregator.create_file_section("back-matter.md", lesson_data['back_matter']))
        
        # Write to output file
        with open(output_path, 'w', encoding='utf-8') as f:
            f.write('\n'.join(combined_content))
        
        print(f"Combined lesson content written to: {output_path}")
        return output_path