import numpy as np
import mne
from copy import deepcopy
from ..util import find_corner
from .base import BaseSolver, InverseOperator

class SolverMVAB(BaseSolver):
    ''' Class for the Minimum Variance Adaptive Beamformer (MVAB) inverse
        solution [1].
    
    Attributes
    ----------

    References
    ----------
    [1] Vorobyov, S. A. (2013). Principles of minimum variance robust adaptive
    beamforming design. Signal Processing, 93(12), 3264-3277.

    '''
    def __init__(self, name="Minimum Variance Adaptive Beamformer", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, alpha='auto', **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        alpha : float
            The regularization parameter.
        
        Return
        ------
        self : object returns itself for convenience

        '''

        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)
        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        leadfield /= np.linalg.norm(leadfield, axis=0)
        n_chans, n_dipoles = self.leadfield.shape

        y = data
        y -= y.mean(axis=0)
        R_inv = np.linalg.inv(y@y.T)
        leadfield -= leadfield.mean(axis=0)
        self.get_alphas(reference=y@y.T)
  
        
        inverse_operators = []
        for alpha in self.alphas:
            inverse_operator = 1/(leadfield.T @ R_inv @ leadfield + alpha * np.identity(n_dipoles)) @ leadfield.T @ R_inv

            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self

class SolverLCMV(BaseSolver):
    ''' Class for the Linearly Constrained Minimum Variance Beamformer (LCMV)
        inverse solution [1].
    
    Attributes
    ----------

    References
    ----------
    [1] Van Veen, B. D., & Buckley, K. M. (1988). Beamforming: A versatile
    approach to spatial filtering. IEEE assp magazine, 5(2), 4-24.
    
    '''
    def __init__(self, name="LCMV Beamformer", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, alpha='auto', weight_norm=True, verbose=0, **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        weight_norm : bool
            Normalize the filter weight matrix W to unit length of the columns.
        alpha : float
            The regularization parameter.
        
        Return
        ------
        self : object returns itself for convenience

        '''
        self.weight_norm = weight_norm
        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)
        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        leadfield /= np.linalg.norm(leadfield, axis=0)
        n_chans, n_dipoles = self.leadfield.shape
        
        y = data
        y -= y.mean(axis=0)
        
        I = np.identity(n_chans)
        C = y@y.T
        
        # Recompute regularization based on the max eigenvalue of the Covariance
        # Matrix (opposed to that of the leadfield)
        self.alphas = np.logspace(-4, 1, self.n_reg_params) * np.diagonal(y@y.T).mean()

        inverse_operators = []
        for alpha in self.alphas:
            
            C_inv = np.linalg.inv(C + alpha*I)

            W = (C_inv @ leadfield) / np.diagonal(leadfield.T @ C_inv @ leadfield)

            if self.weight_norm:
                W /= np.linalg.norm(W, axis=0)
            inverse_operator = W.T
            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self



    
        

class SolverSMV(BaseSolver):
    ''' Class for the Standardized Minimum Variance (SMV) Beamformer inverse
        solution [1].
    
    Attributes
    ----------
    
    References
    ----------
    [1] Jonmohamadi, Y., Poudel, G., Innes, C., Weiss, D., Krueger, R., & Jones, R.
    (2014). Comparison of beamformers for EEG source signal reconstruction.
    Biomedical Signal Processing and Control, 14, 175-188.

    '''
    def __init__(self, name="SMV Beamformer", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, weight_norm=True, alpha='auto', **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        weight_norm : bool
            Normalize the filter weight matrix W to unit length of the columns.
        alpha : float
            The regularization parameter.
        
        Return
        ------
        self : object returns itself for convenience

        '''
        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)

        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        leadfield /= np.linalg.norm(leadfield, axis=0)
        n_chans, n_dipoles = self.leadfield.shape

        self.weight_norm = weight_norm

        y = data
        y -= y.mean(axis=0)
        I = np.identity(n_chans)
        
        # Recompute regularization based on the max eigenvalue of the Covariance
        # Matrix (opposed to that of the leadfield)
        C = y@y.T
        self.alphas = self.get_alphas(reference=C)

        inverse_operators = []
        for alpha in self.alphas:
            C_inv = np.linalg.inv(C + alpha * I)
            W = (C_inv @ leadfield) / np.sqrt(np.diagonal(leadfield.T @ C_inv @ leadfield))
            
            if self.weight_norm:
                W /= np.linalg.norm(W, axis=0)
            inverse_operator = W.T
            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self

class SolverWNMV(BaseSolver):
    ''' Class for the Weight-normalized Minimum Variance (WNMV) Beamformer
        inverse solution [1].
    
    Attributes
    ----------
    forward : mne.Forward
        The mne-python Forward model instance.
    
    References
    ----------
    [1] Jonmohamadi, Y., Poudel, G., Innes, C., Weiss, D., Krueger, R., & Jones,
    R. (2014). Comparison of beamformers for EEG source signal reconstruction.
    Biomedical Signal Processing and Control, 14, 175-188.

    '''
    def __init__(self, name="WNMV Beamformer", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, weight_norm=True, alpha='auto', **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        weight_norm : bool
            Normalize the filter weight matrix W to unit length of the columns.
        alpha : float
            The regularization parameter.
        
        Return
        ------
        self : object returns itself for convenience
        '''
        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)

        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        leadfield /= np.linalg.norm(leadfield, axis=0)
        n_chans, n_dipoles = self.leadfield.shape

        self.weight_norm = weight_norm
        y = data
        y -= y.mean(axis=0)
        I = np.identity(n_chans)

        # Recompute regularization based on the max eigenvalue of the Covariance
        # Matrix (opposed to that of the leadfield)
        C = y@y.T
        self.alphas = self.get_alphas(reference=C)
        
        inverse_operators = []
        for alpha in self.alphas:
            C_inv = np.linalg.inv(C + alpha * I)
            C_inv_2 = np.linalg.inv(C_inv)
            W = (C_inv @ leadfield) / np.sqrt(abs(np.diagonal(leadfield.T @ C_inv_2 @ leadfield)))

            if self.weight_norm:
                W /= np.linalg.norm(W, axis=0)

            inverse_operator = W.T
            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self

class SolverHOCMV(BaseSolver):
    ''' Class for the Higher-Order Covariance Minimum Variance (HOCMV)
        Beamformer inverse solution [1].
    
    Attributes
    ----------
    forward : mne.Forward
        The mne-python Forward model instance.
    
    References
    ----------
    [1] Jonmohamadi, Y., Poudel, G., Innes, C., Weiss, D., Krueger, R., & Jones,
    R. (2014). Comparison of beamformers for EEG source signal reconstruction.
    Biomedical Signal Processing and Control, 14, 175-188.

    '''
    def __init__(self, name="HOCMV Beamformer", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, weight_norm=True, alpha='auto', order=3, verbose=0, **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        weight_norm : bool
            Normalize the filter weight matrix W to unit length of the columns.
        alpha : float
            The regularization parameter.
        order : int
            The order of the covariance matrix. Should be a positive integer not
            evenly divisible by two {3, 5, 7, ...}
        
        Return
        ------
        self : object returns itself for convenience
        '''
        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)

        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        leadfield /= np.linalg.norm(leadfield, axis=0)
        n_chans, n_dipoles = self.leadfield.shape
        
        self.weight_norm = weight_norm
        
        y = data
        y -= y.mean(axis=0)
        I = np.identity(n_chans)

        # Recompute regularization based on the max eigenvalue of the Covariance
        # Matrix (opposed to that of the leadfield)
        C = y@y.T
        self.alphas = self.get_alphas(reference=C)
        
        inverse_operators = []
        for alpha in self.alphas:
            C_inv = np.linalg.inv(C + alpha * I)
            C_inv_n = deepcopy(C_inv)
            for _ in range(order-1):
                C_inv_n = np.linalg.inv(C_inv_n)

            W = (C_inv @ leadfield) / np.sqrt(abs(np.diagonal(leadfield.T @ C_inv_n @ leadfield)))
            if self.weight_norm:
                W /= np.linalg.norm(W, axis=0)

            inverse_operator = W.T
            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self

class SolverESMV(BaseSolver):
    ''' Class for the Eigenspace-based Minimum Variance (ESMV) Beamformer
        inverse solution [1].
    
    Attributes
    ----------
    forward : mne.Forward
        The mne-python Forward model instance.
    
    References
    ----------
    [1] Jonmohamadi, Y., Poudel, G., Innes, C., Weiss, D., Krueger, R., & Jones,
    R. (2014). Comparison of beamformers for EEG source signal reconstruction.
    Biomedical Signal Processing and Control, 14, 175-188.
    
    '''
    def __init__(self, name="ESMV Beamformer", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, alpha='auto', **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        alpha : float
            The regularization parameter.
        
        Return
        ------
        self : object returns itself for convenience
        
        '''
        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)

        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        leadfield /= np.linalg.norm(leadfield, axis=0)
        n_chans, n_dipoles = leadfield.shape
        

        y = data
        y -= y.mean(axis=0)
        leadfield -= leadfield.mean(axis=0)
        I = np.identity(n_chans)
        
        # Recompute regularization based on the max eigenvalue of the Covariance
        # Matrix (opposed to that of the leadfield)
        C = y@y.T
        self.alphas = self.get_alphas(reference=C)
        subspace = self.reduce_rank_matrix(C)

        
        inverse_operators = []
        for alpha in self.alphas:
            C_inv = np.linalg.inv(C + alpha * I)

            W_mv = (C_inv @ leadfield) / np.diagonal(leadfield.T @ C_inv @ leadfield)
            W = subspace @ W_mv

            inverse_operator = W.T
            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self

    def reduce_rank_matrix(self, C):
        # Calculte eigenvectors and eigenvalues
        U, s, _ = np.linalg.svd(C, full_matrices=False)
        
        # Find optimal rank using L-curve Corner approach:
        n_comp_l = find_corner(np.arange(len(s)), s)
        
        # Find optimal rank using drop-off approach:
        s_ = s / s.max()
        n_comp_d = np.where( abs(np.diff(s_)) < 0.001 )[0]
        if len(n_comp_d) > 0:
            n_comp_d = n_comp_d[0] + 2
        else:
            n_comp_d = n_comp_l
        
        # Kaiser Rule
        n_comp_k = np.where(s < np.mean(s))[0][0]
        print(n_comp_k, n_comp_l, n_comp_d)
        
        # Combine the approaches
        n_comp = np.ceil((n_comp_d + n_comp_l + n_comp_k)/3).astype(int)
        
        # Transform data
        subspace = U[:, :n_comp] @ U[:, :n_comp].T
        return subspace

class SolverMCMV(BaseSolver):
    ''' Class for the Multiple Constrained Minimum Variance (MCMV) Beamformer
    inverse solution [1].
    
    Attributes
    ----------

    
    References
    ----------
    [1] Nunes, A. S., Moiseev, A., Kozhemiako, N., Cheung, T., Ribary, U., &
    Doesburg, S. M. (2020). Multiple constrained minimum variance beamformer
    (MCMV) performance in connectivity analyses. NeuroImage, 208, 116386.

    '''
    def __init__(self, name="MCMV Beamformer", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, weight_norm=True, noise_cov=None, alpha='auto', verbose=0, **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        weight_norm : bool
            Normalize the filter weight matrix W to unit length of the columns.
        alpha : float
            The regularization parameter.
        
        Return
        ------
        self : object returns itself for convenience
        '''
        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)

        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        leadfield /= np.linalg.norm(leadfield, axis=0)
        n_chans, n_dipoles = leadfield.shape

        if noise_cov is None:
            noise_cov = np.identity(n_chans)
            
        self.weight_norm = weight_norm

        y = data
        y -= y.mean(axis=0)
        leadfield -= leadfield.mean(axis=0)
        I = np.identity(n_chans)

        # Recompute regularization based on the max eigenvalue of the Covariance
        # Matrix (opposed to that of the leadfield)
        C = y@y.T
        self.alphas = self.get_alphas(reference=C)

        inverse_operators = []
        for alpha in self.alphas:
            C_inv = np.linalg.inv(C + alpha * I)

            W = C_inv @ leadfield * np.diagonal(np.linalg.inv(leadfield.T @ C_inv @ leadfield))
            # W = C_inv @ leadfield @ np.linalg.inv(leadfield.T @ C_inv @ leadfield)
            
            # W = np.dot(leadfield.T, np.dot(C_inv, leadfield))
            # W = np.dot(np.linalg.inv(W + leadfield.T @ noise_cov @ leadfield), np.dot(leadfield.T, C_inv)).T
    

            if self.weight_norm:
                W /= np.linalg.norm(W, axis=0)

            inverse_operator = W.T
            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self

    

class SolverHOCMCMV(BaseSolver):
    ''' Class for the Higher-Order Covariance Multiple Constrained Minimum Variance (HOCMCMV)
        Beamformer inverse solution [1].
    
    Attributes
    ----------
    forward : mne.Forward
        The mne-python Forward model instance.
    
    References
    ----------
    [1] Jonmohamadi, Y., Poudel, G., Innes, C., Weiss, D., Krueger, R., & Jones,
    R. (2014). Comparison of beamformers for EEG source signal reconstruction.
    Biomedical Signal Processing and Control, 14, 175-188.

    '''
    def __init__(self, name="HOCMCMV Beamformer", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, weight_norm=True, alpha='auto', order=3, verbose=0, **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        weight_norm : bool
            Normalize the filter weight matrix W to unit length of the columns.
        alpha : float
            The regularization parameter.
        order : int
            The order of the covariance matrix. Should be a positive integer not
            evenly divisible by two {3, 5, 7, ...}
        
        Return
        ------
        self : object returns itself for convenience
        '''
        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)

        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        leadfield /= np.linalg.norm(leadfield, axis=0)
        n_chans, n_dipoles = self.leadfield.shape
        
        self.weight_norm = weight_norm
        
        y = data
        y -= y.mean(axis=0)
        I = np.identity(n_chans)

        # Recompute regularization based on the max eigenvalue of the Covariance
        # Matrix (opposed to that of the leadfield)
        C = y@y.T
        self.alphas = self.get_alphas(reference=C)
        
        inverse_operators = []
        for alpha in self.alphas:
            C_inv = np.linalg.inv(C + alpha * I)
            C_inv_n = deepcopy(C_inv)
            for _ in range(order-1):
                C_inv_n = np.linalg.inv(C_inv_n)
            W = C_inv @ leadfield * np.diagonal(np.linalg.inv(leadfield.T @ C_inv_n @ leadfield))
            
            if self.weight_norm:
                W /= np.linalg.norm(W, axis=0)

            inverse_operator = W.T
            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self

class SolverReciPSIICOS(BaseSolver):
    ''' Class for the Reciprocal Phase Shift Invariant Imaging of Coherent
    Sources (ReciPSIICOS) Beamformer inverse solution [1].
    
    Attributes
    ----------


    References
    ----------    
    [1] Kuznetsova, A., Nurislamova, Y., & Ossadtchi, A. (2021). Modified
    covariance beamformer for solving MEG inverse problem in the environment
    with correlated sources. Neuroimage, 228, 117677.

    '''
    def __init__(self, name="ReciPSIICOS", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, weight_norm=True, alpha='auto', verbose=0, **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        weight_norm : bool
            Normalize the filter weight matrix W to unit length of the columns.
        alpha : float
            The regularization parameter.
        
        Return
        ------
        self : object returns itself for convenience

        '''
        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)

        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        leadfield /= np.linalg.norm(leadfield, axis=0)
        n_chans, n_dipoles = leadfield.shape
        
        self.weight_norm = weight_norm
        
        y = data
        y -= y.mean(axis=0)

        
        # Step 1
        G_pwr = leadfield @ leadfield.T
        
        # Step 2
        U_pwr, S_pwr, _ = np.linalg.svd(G_pwr, full_matrices=False)
        k = find_corner(np.arange(len(S_pwr)), S_pwr)
        P = U_pwr[:, :k] @ U_pwr[:, :k].T
        
        
        I = np.identity(n_chans)
        # Recompute regularization based on the max eigenvalue of the Covariance
        # Matrix (opposed to that of the leadfield)
        C = y@y.T
        self.alphas = self.get_alphas(reference=G_pwr)
        # self.alphas = np.logspace(-4, 1, self.n_reg_params) * np.diagonal(y@y.T).mean()
        
        inverse_operators = []
        for alpha in self.alphas:
            C = y@y.T + alpha * I

            # Step 3
            C_x = (P@C).T

            # Step 4
            E, A, _ = np.linalg.svd(C_x, full_matrices=False)

            # Make new Covariance positive semidefinite
            C_x = E @ np.diag(np.abs(A)) @ E.T
            C_x_inv = np.linalg.inv(C_x)

            W = (C_x_inv @ leadfield) * np.diagonal(np.linalg.inv(leadfield.T @ C_x_inv @ leadfield))

            if self.weight_norm:
                W /= np.linalg.norm(W, axis=0)

            inverse_operator = W.T
            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self

class SolverSAM(BaseSolver):
    ''' Class for the Synthetic Aperture Magnetometry Beamformer (SAM) inverse
    solution [1].
    
    Attributes
    ----------
    
    References
    ----------
    [1] Robinson, S. E. V. J. (1999). Functional neuroimaging by synthetic
    aperture magnetometry (SAM). Recent advances in biomagnetism.

    '''
    def __init__(self, name="SAM Beamformer", reduce_rank=True, rank="auto", **kwargs):
        self.name = name
        return super().__init__(reduce_rank=reduce_rank, rank=rank, **kwargs)

    def make_inverse_operator(self, forward, mne_obj, *args, weight_norm=True, alpha='auto', verbose=0, **kwargs):
        ''' Calculate inverse operator.

        Parameters
        ----------
        forward : mne.Forward
            The mne-python Forward model instance.
        mne_obj : [mne.Evoked, mne.Epochs, mne.io.Raw]
            The MNE data object.
        weight_norm : bool
            Normalize the filter weight matrix W to unit length of the columns.
        alpha : float
            The regularization parameter.
        
        Return
        ------
        self : object returns itself for convenience
        '''
        super().make_inverse_operator(forward, *args, alpha=alpha, **kwargs)
        data = self.unpack_data_obj(mne_obj)

        self.weight_norm = weight_norm
        leadfield = self.leadfield
        leadfield -= leadfield.mean(axis=0)
        n_chans, n_dipoles = leadfield.shape
        

        y = data
        y -= y.mean(axis=0)
        leadfield -= leadfield.mean(axis=0)
        I = np.identity(n_chans)
  
        
        inverse_operators = []
        for alpha in self.alphas:
            C_inv = np.linalg.inv(y@y.T + alpha * I)
            W = []
            for i in range(n_dipoles):
                l = leadfield[:, i][:, np.newaxis]
                w = (C_inv@l) / (l.T@C_inv@l)
                W.append(w)
            W = np.stack(W, axis=1)[:, :, 0]
            if self.weight_norm:
                W = W / np.linalg.norm(W, axis=0)
            inverse_operator = W.T
            inverse_operators.append(inverse_operator)

        self.inverse_operators = [InverseOperator(inverse_operator, self.name) for inverse_operator in inverse_operators]
        return self
