# Licensed to the White Turing under one or more
# contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The SFC licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

'''List of Chromium Command Line Switches.'''


class Chromium(object):
    '''Frequently used commands mappings.
    
    There are lots of command lines which can be used with the Google Chrome browser.
    Some change behavior of features, others are for debugging or experimenting.
    This page lists the available switches including their conditions and descriptions.
    Last update occurred on 2020-04-16 from `https://peter.sh/experiments/chromium-command-line-switches/`.
    '''
    ACCEPT_RESOURCE_PROVIDER = 'accept-resource-provider'   # Flag indicating that a resource provider must be set up to provide cast receiver with resources. Apps cannot start until provided resources. This flag implies --alsa-check-close-timeout=0.
    ACCOUNT_CONSISTENCY = 'account-consistency'   # Command line flag for enabling account consistency. Default mode is disabled. Mirror is a legacy mode in which Google accounts are always addded to Chrome, and Chrome then adds them to the Google authentication cookies. Dice is a new experiment in which Chrome is aware of the accounts in the Google authentication cookies.
    ADABOOST = 'adaboost'   # 
    ADD_GPU_APPCONTAINER_CAPS = 'add-gpu-appcontainer-caps'   # Add additional capabilities to the AppContainer sandbox on the GPU process.
    ADD_XR_APPCONTAINER_CAPS = 'add-xr-appcontainer-caps'   # Add additional capabilities to the AppContainer sandbox used for XR compositing.
    AGC_STARTUP_MIN_VOLUME = 'agc-startup-min-volume'   # Override the default minimum starting volume of the Automatic Gain Control algorithm in WebRTC used with audio tracks from getUserMedia. The valid range is 12-255. Values outside that range will be clamped to the lowest or highest valid value inside WebRTC. TODO(tommi): Remove this switch when crbug.com/555577 is fixed.
    AGGRESSIVE = 'aggressive'   # 
    AGGRESSIVE_CACHE_DISCARD = 'aggressive-cache-discard'   # 
    AGGRESSIVE_TAB_DISCARD = 'aggressive-tab-discard'   # 
    ALL = 'all'   # 
    ALL_RENDERERS = 'all-renderers'   # 
    ALLARTICLES = 'allarticles'   # 
    ALLOW_CROSS_ORIGIN_AUTH_PROMPT = 'allow-cross-origin-auth-prompt'   # Allows third-party content included on a page to prompt for a HTTP basic auth username/password pair.
    ALLOW_EXTERNAL_PAGES = 'allow-external-pages'   # Allow access to external pages during web tests.
    ALLOW_FAILED_POLICY_FETCH_FOR_TEST = 'allow-failed-policy-fetch-for-test'   # If this flag is passed, failed policy fetches will not cause profile initialization to fail. This is useful for tests because it means that tests don't have to mock out the policy infrastructure.
    ALLOW_FILE_ACCESS_FROM_FILES = 'allow-file-access-from-files'   # By default, file:// URIs cannot read other file:// URIs. This is an override for developers who need the old behavior for testing.
    ALLOW_HTTP_BACKGROUND_PAGE = 'allow-http-background-page'   # Allows non-https URL for background_page for hosted apps.
    ALLOW_HTTP_SCREEN_CAPTURE = 'allow-http-screen-capture'   # Allow non-secure origins to use the screen capture API and the desktopCapture extension API.
    ALLOW_INSECURE_LOCALHOST = 'allow-insecure-localhost'   # Enables TLS/SSL errors on localhost to be ignored (no interstitial, no blocking of requests).
    ALLOW_LEGACY_EXTENSION_MANIFESTS = 'allow-legacy-extension-manifests'   # Allows the browser to load extensions that lack a modern manifest when that would otherwise be forbidden.
    ALLOW_LOOPBACK_IN_PEER_CONNECTION = 'allow-loopback-in-peer-connection'   # Allows loopback interface to be added in network list for peer connection.
    ALLOW_NACL_CRXFS_API = 'allow-nacl-crxfs-api'   # Specifies comma-separated list of extension ids or hosts to grant access to CRX file system APIs.
    ALLOW_NACL_FILE_HANDLE_API = 'allow-nacl-file-handle-api'   # Specifies comma-separated list of extension ids or hosts to grant access to file handle APIs.
    ALLOW_NACL_SOCKET_API = 'allow-nacl-socket-api'   # Specifies comma-separated list of extension ids or hosts to grant access to TCP/UDP socket APIs.
    ALLOW_NO_SANDBOX_JOB = 'allow-no-sandbox-job'   # Enables the sandboxed processes to run without a job object assigned to them. This flag is required to allow Chrome to run in RemoteApps or Citrix. This flag can reduce the security of the sandboxed processes and allow them to do certain API calls like shut down Windows or access the clipboard. Also we lose the chance to kill some processes until the outer job that owns them finishes.
    ALLOW_OUTDATED_PLUGINS = 'allow-outdated-plugins'   # Don't block outdated plugins.
    ALLOW_POPUPS_DURING_PAGE_UNLOAD = 'allow-popups-during-page-unload'   # Allow a page to show popups during its unloading. TODO(https://crbug.com/937569): Remove this in Chrome 82.
    ALLOW_RA_IN_DEV_MODE = 'allow-ra-in-dev-mode'   # Allows remote attestation (RA) in dev mode for testing purpose. Usually RA is disabled in dev mode because it will always fail. However, there are cases in testing where we do want to go through the permission flow even in dev mode. This can be enabled by this flag.
    ALLOW_RUNNING_INSECURE_CONTENT = 'allow-running-insecure-content'   # By default, an https page cannot run JavaScript, CSS or plugins from http URLs. This provides an override to get the old insecure behavior.
    ALLOW_SANDBOX_DEBUGGING = 'allow-sandbox-debugging'   # Allows debugging of sandboxed processes (see zygote_main_linux.cc).
    ALLOW_SILENT_PUSH = 'allow-silent-push'   # Allows Web Push notifications that do not show a notification.
    ALLOW_THIRD_PARTY_MODULES = 'allow-third-party-modules'   # Allows third party modules to inject by disabling the BINARY_SIGNATURE mitigation policy on Win10+. Also has other effects in ELF.
    ALLOW_UNSECURE_DLLS = 'allow-unsecure-dlls'   # Don't allow EnableSecureDllLoading to run when this is set. This is only to be used in tests.
    ALSA_AMP_DEVICE_NAME = 'alsa-amp-device-name'   # Name of the device the amp mixer should be opened on. If this flag is not specified it will default to the same device as kAlsaVolumeDeviceName.
    ALSA_AMP_ELEMENT_NAME = 'alsa-amp-element-name'   # Name of the simple mixer control element that the ALSA-based media library should use to toggle powersave mode on the system.
    ALSA_CHECK_CLOSE_TIMEOUT = 'alsa-check-close-timeout'   # Time in ms to wait before closing the PCM handle when no more mixer inputs remain. Assumed to be 0 if --accept-resource-provider is present.
    ALSA_ENABLE_UPSAMPLING = 'alsa-enable-upsampling'   # Flag that enables resampling audio with sample rate below 32kHz up to 48kHz. Should be set to true for internal audio products.
    ALSA_FIXED_OUTPUT_SAMPLE_RATE = 'alsa-fixed-output-sample-rate'   # Optional flag to set a fixed sample rate for the alsa device. Deprecated: Use --audio-output-sample-rate instead.
    ALSA_INPUT_DEVICE = 'alsa-input-device'   # The Alsa device to use when opening an audio input stream.
    ALSA_MUTE_DEVICE_NAME = 'alsa-mute-device-name'   # Name of the device the mute mixer should be opened on. If this flag is not specified it will default to the same device as kAlsaVolumeDeviceName.
    ALSA_MUTE_ELEMENT_NAME = 'alsa-mute-element-name'   # Name of the simple mixer control element that the ALSA-based media library should use to mute the system.
    ALSA_OUTPUT_AVAIL_MIN = 'alsa-output-avail-min'   # Minimum number of available frames for scheduling a transfer.
    ALSA_OUTPUT_BUFFER_SIZE = 'alsa-output-buffer-size'   # Size of the ALSA output buffer in frames. This directly sets the latency of the output device. Latency can be calculated by multiplying the sample rate by the output buffer size.
    ALSA_OUTPUT_DEVICE = 'alsa-output-device'   # The Alsa device to use when opening an audio stream.
    ALSA_OUTPUT_PERIOD_SIZE = 'alsa-output-period-size'   # Size of the ALSA output period in frames. The period of an ALSA output device determines how many frames elapse between hardware interrupts.
    ALSA_OUTPUT_START_THRESHOLD = 'alsa-output-start-threshold'   # How many frames need to be in the output buffer before output starts.
    ALSA_VOLUME_DEVICE_NAME = 'alsa-volume-device-name'   # Name of the device the volume control mixer should be opened on. Will use the same device as kAlsaOutputDevice and fall back to "default" if kAlsaOutputDevice is not supplied.
    ALSA_VOLUME_ELEMENT_NAME = 'alsa-volume-element-name'   # Name of the simple mixer control element that the ALSA-based media library should use to control the volume.
    ALSO_EMIT_SUCCESS_LOGS = 'also-emit-success-logs'   # Also emit full event trace logs for successful tests.
    ALT1 = 'alt1'   # The Chrome-Proxy "exp" directive value used by data reduction proxy to receive an alternative back end implementation.
    ALT10 = 'alt10'   # 
    ALT2 = 'alt2'   # 
    ALT3 = 'alt3'   # 
    ALT4 = 'alt4'   # 
    ALT5 = 'alt5'   # 
    ALT6 = 'alt6'   # 
    ALT7 = 'alt7'   # 
    ALT8 = 'alt8'   # 
    ALT9 = 'alt9'   # 
    ALTERNATE_ANDROID_MESSAGES_INSTALL_URL = 'alternate-android-messages-install-url'   # 
    ALTERNATE_ANDROID_MESSAGES_URL = 'alternate-android-messages-url'   # 
    ALWAYS_USE_COMPLEX_TEXT = 'always-use-complex-text'   # Always use the complex text path for web tests.
    ALWAYSTRUE = 'alwaystrue'   # 
    ANDROID_FONTS_PATH = 'android-fonts-path'   # Uses the android SkFontManager on linux. The specified directory should include the configuration xml file with the name "fonts.xml". This is used in blimp to emulate android fonts on linux.
    ANGLE = 'angle'   # 
    ANIMATION_DURATION_SCALE = 'animation-duration-scale'   # Scale factor to apply to every animation duration. Must be >= 0.0. This will only apply to LinearAnimation and its subclasses.
    APP = 'app'   # Specifies that the associated value should be launched in "application" mode.
    APP_AUTO_LAUNCHED = 'app-auto-launched'   # Specifies whether an app launched in kiosk mode was auto launched with zero delay. Used in order to properly restore auto-launched state during session restore flow.
    APP_ID = 'app-id'   # Specifies that the extension-app with the specified id should be launched according to its configuration.
    APP_MODE_AUTH_CODE = 'app-mode-auth-code'   # Value of GAIA auth code for --force-app-mode.
    APP_MODE_OAUTH_TOKEN = 'app-mode-oauth-token'   # Value of OAuth2 refresh token for --force-app-mode.
    APP_MODE_OEM_MANIFEST = 'app-mode-oem-manifest'   # Path for app's OEM manifest file.
    APP_SHELL_ALLOW_ROAMING = 'app-shell-allow-roaming'   # Allow roaming in the cellular network.
    APP_SHELL_HOST_WINDOW_SIZE = 'app-shell-host-window-size'   # Size for the host window to create (i.e. "800x600").
    APP_SHELL_PREFERRED_NETWORK = 'app-shell-preferred-network'   # SSID of the preferred WiFi network.
    APPLE = 'apple'   # 
    APPS_GALLERY_DOWNLOAD_URL = 'apps-gallery-download-url'   # The URL that the webstore APIs download extensions from. Note: the URL must contain one '%s' for the extension ID.
    APPS_GALLERY_UPDATE_URL = 'apps-gallery-update-url'   # The update url used by gallery/webstore extensions.
    APPS_GALLERY_URL = 'apps-gallery-url'   # The URL to use for the gallery link in the app launcher.
    APPS_KEEP_CHROME_ALIVE_IN_TESTS = 'apps-keep-chrome-alive-in-tests'   # Prevents Chrome from quitting when Chrome Apps are open.
    ARC_AVAILABILITY = 'arc-availability'   # Signals ARC support status on this device. This can take one of the following three values. - none: ARC is not installed on this device. (default) - installed: ARC is installed on this device, but not officially supported. Users can enable ARC only when Finch experiment is turned on. - officially-supported: ARC is installed and supported on this device. So users can enable ARC via settings etc.
    ARC_AVAILABLE = 'arc-available'   # DEPRECATED: Please use --arc-availability=installed. Signals the availability of the ARC instance on this device.
    ARC_DATA_CLEANUP_ON_START = 'arc-data-cleanup-on-start'   # Flag that forces ARC data be cleaned on each start.
    ARC_DISABLE_APP_SYNC = 'arc-disable-app-sync'   # Flag that disables ARC app sync flow that installs some apps silently. Used in autotests to resolve racy conditions.
    ARC_DISABLE_LOCALE_SYNC = 'arc-disable-locale-sync'   # Flag that disables ARC locale sync with Android container. Used in autotest to prevent conditions when certain apps, including Play Store may get restarted. Restarting Play Store may cause random test failures. Enabling this flag would also forces ARC container to use 'en-US' as a locale and 'en-US,en' as preferred languages.
    ARC_DISABLE_PLAY_AUTO_INSTALL = 'arc-disable-play-auto-install'   # Flag that disables ARC Play Auto Install flow that installs set of predefined apps silently. Used in autotests to resolve racy conditions.
    ARC_FORCE_CACHE_APP_ICONS = 'arc-force-cache-app-icons'   # Flag that forces ARC to cache icons for apps.
    ARC_FORCE_SHOW_OPTIN_UI = 'arc-force-show-optin-ui'   # Flag that forces the OptIn ui to be shown. Used in tests.
    ARC_PACKAGES_CACHE_MODE = 'arc-packages-cache-mode'   # Used in autotest to specifies how to handle packages cache. Can be copy - copy resulting packages.xml to the temporary directory. skip-copy - skip initial packages cache setup and copy resulting packages.xml to the temporary directory.
    ARC_PLAY_STORE_AUTO_UPDATE = 'arc-play-store-auto-update'   # Used in autotest to forces Play Store auto-update state. Can be on - auto-update is forced on. off - auto-update is forced off.
    ARC_START_MODE = 'arc-start-mode'   # Defines how to start ARC. This can take one of the following values: - always-start automatically start with Play Store UI support. - always-start-with-no-play-store automatically start without Play Store UI. If it is not set, then ARC is started in default mode.
    ARC_TRANSITION_MIGRATION_REQUIRED = 'arc-transition-migration-required'   # If this flag is present then the device had ARC M available and gets ARC N when updating. TODO(pmarko): Remove this when we assess that it's not necessary anymore: crbug.com/761348.
    ASH_COLOR_MODE = 'ash-color-mode'   # Indicates the current color mode of ash.
    ASH_CONSTRAIN_POINTER_TO_ROOT = 'ash-constrain-pointer-to-root'   # Force the pointer (cursor) position to be kept inside root windows.
    ASH_DEBUG_SHORTCUTS = 'ash-debug-shortcuts'   # Enable keyboard shortcuts useful for debugging.
    ASH_DEV_SHORTCUTS = 'ash-dev-shortcuts'   # Enable keyboard shortcuts used by developers only.
    ASH_DISABLE_TOUCH_EXPLORATION_MODE = 'ash-disable-touch-exploration-mode'   # Disable the Touch Exploration Mode. Touch Exploration Mode will no longer be turned on automatically when spoken feedback is enabled when this flag is set.
    ASH_ENABLE_CURSOR_MOTION_BLUR = 'ash-enable-cursor-motion-blur'   # Enable cursor motion blur.
    ASH_ENABLE_MAGNIFIER_KEY_SCROLLER = 'ash-enable-magnifier-key-scroller'   # Enables key bindings to scroll magnified screen.
    ASH_ENABLE_PALETTE_ON_ALL_DISPLAYS = 'ash-enable-palette-on-all-displays'   # Enables the palette on every display, instead of only the internal one.
    ASH_ENABLE_SOFTWARE_MIRRORING = 'ash-enable-software-mirroring'   # Enables software based mirroring.
    ASH_ENABLE_UNIFIED_DESKTOP = 'ash-enable-unified-desktop'   # Enables unified desktop mode.
    ASH_ENABLE_V1_APP_BACK_BUTTON = 'ash-enable-v1-app-back-button'   # Enables Backbutton on frame for v1 apps. TODO(oshima): Remove this once the feature is launched. crbug.com/749713.
    ASH_HIDE_NOTIFICATIONS_FOR_FACTORY = 'ash-hide-notifications-for-factory'   # Hides notifications that are irrelevant to Chrome OS device factory testing, such as battery level updates.
    ASH_HOST_WINDOW_BOUNDS = 'ash-host-window-bounds'   # Sets a window size, optional position, and optional scale factor. "1024x768" creates a window of size 1024x768. "100+200-1024x768" positions the window at 100,200. "1024x768*2" sets the scale factor to 2 for a high DPI display. "800,0+800-800x800" for two displays at 800x800 resolution. "800,0+800-800x800,0+1600-800x800" for three displays at 800x800 resolution.
    ASH_POWER_BUTTON_POSITION = 'ash-power-button-position'   # Power button position includes the power button's physical display side and the percentage for power button center position to the display's width/height in landscape_primary screen orientation. The value is a JSON object containing a "position" property with the value "left", "right", "top", or "bottom". For "left" and "right", a "y" property specifies the button's center position as a fraction of the display's height (in [0.0, 1.0]) relative to the top of the display. For "top" and "bottom", an "x" property gives the position as a fraction of the display's width relative to the left side of the display.
    ASH_SIDE_VOLUME_BUTTON_POSITION = 'ash-side-volume-button-position'   # The physical position info of the side volume button while in landscape primary screen orientation. The value is a JSON object containing a "region" property with the value "keyboard", "screen" and a "side" property with the value "left", "right", "top", "bottom".
    ASH_TOUCH_HUD = 'ash-touch-hud'   # Enables the heads-up display for tracking touch points.
    ATTESTATION_SERVER = 'attestation-server'   # Used in CryptohomeClient to determine which Google Privacy CA to use for attestation.
    AUDIO = 'audio'   # 
    AUDIO_BUFFER_SIZE = 'audio-buffer-size'   # Allow users to specify a custom buffer size for debugging purpose.
    AUDIO_OUTPUT_CHANNELS = 'audio-output-channels'   # Number of audio output channels. This will be used to send audio buffer with specific number of channels to ALSA and generate loopback audio. Default value is 2.
    AUDIO_OUTPUT_SAMPLE_RATE = 'audio-output-sample-rate'   # Specify fixed sample rate for audio output stream. If this flag is not specified the StreamMixer will choose sample rate based on the sample rate of the media stream.
    AUDIO_SERVICE_QUIT_TIMEOUT_MS = 'audio-service-quit-timeout-ms'   # Set a timeout (in milliseconds) for the audio service to quit if there are no client connections to it. If the value is negative the service never quits.
    AURA_LEGACY_POWER_BUTTON = 'aura-legacy-power-button'   # (Most) Chrome OS hardware reports ACPI power button releases correctly. Standard hardware reports releases immediately after presses.  If set, we lock the screen or shutdown the system immediately in response to a press instead of displaying an interactive animation.
    AUTH_SERVER_WHITELIST = 'auth-server-whitelist'   # Whitelist for Negotitate Auth servers.
    AUTH_SPNEGO_ACCOUNT_TYPE = 'auth-spnego-account-type'   # Android authentication account type for SPNEGO authentication
    AUTO = 'auto'   # 
    AUTO_OPEN_DEVTOOLS_FOR_TABS = 'auto-open-devtools-for-tabs'   # This flag makes Chrome auto-open DevTools window for each tab. It is intended to be used by developers and automation to not require user interaction for opening DevTools.
    AUTO_SELECT_DESKTOP_CAPTURE_SOURCE = 'auto-select-desktop-capture-source'   # This flag makes Chrome auto-select the provided choice when an extension asks permission to start desktop capture. Should only be used for tests. For instance, --auto-select-desktop-capture-source="Entire screen" will automatically select to share the entire screen in English locales.
    AUTOFILL_API_KEY = 'autofill-api-key'   # Sets the API key that will be used when calling Autofill API instead of using Chrome's baked key by default. You can use this to test new versions of the API that are not linked to the Chrome baked key yet.
    AUTOFILL_IOS_DELAY_BETWEEN_FIELDS = 'autofill-ios-delay-between-fields'   # The delay between filling two fields.
    AUTOFILL_METADATA_UPLOAD_ENCODING = 'autofill-metadata-upload-encoding'   # The randomized encoding type to use when sending metadata uploads. The value of the parameter must be one of the valid integer values of the AutofillRandomizedValue_EncodingType enum.
    AUTOFILL_SERVER_URL = 'autofill-server-url'   # Override the default autofill server URL with "scheme://host[:port]/prefix/".
    AUTOFILL_UPLOAD_THROTTLING_PERIOD_IN_DAYS = 'autofill-upload-throttling-period-in-days'   # The number of days after which to reset the registry of autofill events for which an upload has been sent.
    AUTOPLAY_POLICY = 'autoplay-policy'   # Command line flag name to set the autoplay policy.
    BACK_GESTURE_HORIZONTAL_THRESHOLD = 'back-gesture-horizontal-threshold'   # The number of pixels from the start of a left swipe gesture to consider as a 'back' gesture.
    BLINK_SETTINGS = 'blink-settings'   # Set blink settings. Format is <name>[=<value],<name>[=<value>],... The names are declared in Settings.json5. For boolean type, use "true", "false", or omit '=<value>' part to set to true. For enum type, use the int value of the enum value. Applied after other command line flags and prefs.
    BLOCK_NEW_WEB_CONTENTS = 'block-new-web-contents'   # If true, then all pop-ups and calls to window.open will fail.
    BOOTSTRAP = 'bootstrap'   # Values for the kExtensionContentVerification flag. See ContentVerifierDelegate::Mode for more explanation.
    BOTTOM_GESTURE_START_HEIGHT = 'bottom-gesture-start-height'   # The number of pixels up from the bottom of the screen to consider as a valid origin for a bottom swipe gesture. If set, overrides the value of both the above system-gesture-start-height flag and the default value in cast_system_gesture_handler.cc.
    BROWSER = 'browser'   # 
    BROWSER_STARTUP_DIALOG = 'browser-startup-dialog'   # Causes the browser process to display a dialog on launch.
    BROWSER_SUBPROCESS_PATH = 'browser-subprocess-path'   # Path to the exe to run for the renderer and plugin subprocesses.
    BROWSER_TEST = 'browser-test'   # Tells whether the code is running browser tests (this changes the startup URL used by the content shell and also disables features that can make tests flaky [like monitoring of memory pressure]).
    BWSI = 'bwsi'   # Indicates that the browser is in "browse without sign-in" (Guest session) mode. Should completely disable extensions, sync and bookmarks.
    BYPASS_APP_BANNER_ENGAGEMENT_CHECKS = 'bypass-app-banner-engagement-checks'   # This flag causes the user engagement checks for showing app banners to be bypassed. It is intended to be used by developers who wish to test that their sites otherwise meet the criteria needed to show app banners.
    CANVAS_MSAA_SAMPLE_COUNT = 'canvas-msaa-sample-count'   # The number of MSAA samples for canvas2D. Requires MSAA support by GPU to have an effect. 0 disables MSAA.
    CAST_APP_BACKGROUND_COLOR = 'cast-app-background-color'   # Background color used when Chromium hasn't rendered anything yet.
    CAST_INITIAL_SCREEN_HEIGHT = 'cast-initial-screen-height'   # 
    CAST_INITIAL_SCREEN_WIDTH = 'cast-initial-screen-width'   # Used to pass initial screen resolution to GPU process.  This allows us to set screen size correctly (so no need to resize when first window is created).
    CC_LAYER_TREE_TEST_LONG_TIMEOUT = 'cc-layer-tree-test-long-timeout'   # Increases timeout for memory checkers.
    CC_LAYER_TREE_TEST_NO_TIMEOUT = 'cc-layer-tree-test-no-timeout'   # Prevents the layer tree unit tests from timing out.
    CC_REBASELINE_PIXELTESTS = 'cc-rebaseline-pixeltests'   # Makes pixel tests write their output instead of read it.
    CC_SCROLL_ANIMATION_DURATION_IN_SECONDS = 'cc-scroll-animation-duration-in-seconds'   # Controls the duration of the scroll animation curve.
    CDM = 'cdm'   # 
    CELLULAR_FIRST = 'cellular-first'   # If this flag is set, it indicates that this device is a "Cellular First" device. Cellular First devices use cellular telephone data networks as their primary means of connecting to the internet. Setting this flag has two consequences: 1. Cellular data roaming will be enabled by default. 2. UpdateEngine will be instructed to allow auto-updating over cellular data connections.
    CHECK_DAMAGE_EARLY = 'check-damage-early'   # Checks damage early and aborts the frame if no damage, so that clients like Android WebView don't invalidate unnecessarily.
    CHECK_FOR_UPDATE_INTERVAL = 'check-for-update-interval'   # How often (in seconds) to check for updates. Should only be used for testing purposes.
    CHILD_WALLPAPER_LARGE = 'child-wallpaper-large'   # Default large wallpaper to use for kids accounts (as path to trusted, non-user-writable JPEG file).
    CHILD_WALLPAPER_SMALL = 'child-wallpaper-small'   # Default small wallpaper to use for kids accounts (as path to trusted, non-user-writable JPEG file).
    CHROMEOSMEMORYPRESSUREHANDLING = 'ChromeOSMemoryPressureHandling'   # The memory pressure thresholds selection which is used to decide whether and when a memory pressure event needs to get fired.
    CIPHER_SUITE_BLACKLIST = 'cipher-suite-blacklist'   # Comma-separated list of SSL cipher suites to disable.
    CLAMSHELL = 'clamshell'   # Values for the kAshUiMode flag.
    CLASS = 'class'   # The same as the --class argument in X applications.  Overrides the WM_CLASS window property with the given value.
    CLEANING_TIMEOUT = 'cleaning-timeout'   # Set the timeout for the cleaning phase, in minutes. 0 disables the timeout entirely. WARNING: this switch is used by internal test systems. Be careful when making changes.
    CLEANUP_ID = 'cleanup-id'   # Identifier used to group all reports generated during the same run of the cleaner, including runs before and after a reboot. The id is generated by the first cleaner process for a run, and propagated to spawned or scheduled cleaner processes. A new id will be generated in the first process for a subsequent cleanup, so cleanups from the same user cannot be tracked over time.
    CLEAR_KEY_CDM_PATH_FOR_TESTING = 'clear-key-cdm-path-for-testing'   # Specifies the path to the Clear Key CDM for testing, which is necessary to support External Clear Key key system when library CDM is enabled. Note that External Clear Key key system support is also controlled by feature kExternalClearKeyForTesting.
    CLEAR_LITEPAGE_REDIRECT_LOCAL_BLACKLIST_ON_STARTUP = 'clear-litepage-redirect-local-blacklist-on-startup'   # Clears the local Lite Page Redirect blacklist on startup.
    CLEAR_TOKEN_SERVICE = 'clear-token-service'   # Clears the token service before using it. This allows simulating the expiration of credentials during testing.
    CLOUD_PRINT_FILE = 'cloud-print-file'   # Tells chrome to display the cloud print dialog and upload the specified file for printing.
    CLOUD_PRINT_FILE_TYPE = 'cloud-print-file-type'   # Specifies the mime type to be used when uploading data from the file referenced by cloud-print-file. Defaults to "application/pdf" if unspecified.
    CLOUD_PRINT_JOB_TITLE = 'cloud-print-job-title'   # Used with kCloudPrintFile to specify a title for the resulting print job.
    CLOUD_PRINT_PRINT_TICKET = 'cloud-print-print-ticket'   # Used with kCloudPrintFile to specify a JSON print ticket for the resulting print job. Defaults to null if unspecified.
    CLOUD_PRINT_SETUP_PROXY = 'cloud-print-setup-proxy'   # Setup cloud print proxy for provided printers. This does not start service or register proxy for autostart.
    CLOUD_PRINT_URL = 'cloud-print-url'   # The URL of the cloud print service to use, overrides any value stored in preferences, and the default. Only used if the cloud print service has been enabled. Used for testing.
    CLOUD_PRINT_XMPP_ENDPOINT = 'cloud-print-xmpp-endpoint'   # The XMPP endpoint the cloud print service will use. Only used if the cloud print service has been enabled. Used for testing.
    COMPENSATE_FOR_UNSTABLE_PINCH_ZOOM = 'compensate-for-unstable-pinch-zoom'   # Enable compensation for unstable pinch zoom. Some touch screens display significant amount of wobble when moving a finger in a straight line. This makes two finger scroll trigger an oscillating pinch zoom. See crbug.com/394380 for details.
    COMPILE_SHADER_ALWAYS_SUCCEEDS = 'compile-shader-always-succeeds'   # Always return success when compiling a shader. Linking will still fail.
    COMPONENT_UPDATER = 'component-updater'   # Comma-separated options to troubleshoot the component updater. Only valid for the browser process.
    CONNECTIVITY_CHECK_URL = 'connectivity-check-url'   # Url for network connectivity checking. Default is "https://clients3.google.com/generate_204".
    CONSERVATIVE = 'conservative'   # 
    CONTENT_SHELL_HIDE_TOOLBAR = 'content-shell-hide-toolbar'   # Hides toolbar from content_shell's host window.
    CONTENT_SHELL_HOST_WINDOW_SIZE = 'content-shell-host-window-size'   # Size for the content_shell's host window (i.e. "800x600").
    CONTROLLER = 'controller'   # 
    CRASH = 'crash'   # Crash flag to force a crash right away. Mainly intended for ensuring crashes are properly recorded by crashpad.
    CRASH_DUMPS_DIR = 'crash-dumps-dir'   # The directory breakpad should store minidumps in.
    CRASH_HANDLER = 'crash-handler'   # Runs as the Crashpad handler.
    CRASH_LOOP_BEFORE = 'crash-loop-before'   # A time_t. Passed by session_manager into the Chrome user session, indicating that if Chrome crashes before the indicated time, session_manager will consider this to be a crash-loop situation and log the user out. Chrome mostly just passes this to crash_reporter if it crashes.
    CRASH_ON_FAILURE = 'crash-on-failure'   # When specified to "enable-leak-detection" command-line option, causes the leak detector to cause immediate crash when found leak.
    CRASH_ON_HANG_THREADS = 'crash-on-hang-threads'   # Comma-separated list of BrowserThreads that cause browser process to crash if the given browser thread is not responsive. UI/IO are the BrowserThreads that are supported.  For example: --crash-on-hang-threads=UI:18,IO:18 --> Crash the browser if UI or IO is not responsive for 18 seconds while the other browser thread is responsive.
    CRASH_SERVER_URL = 'crash-server-url'   # Server url to upload crash data to. Default is "https://clients2.google.com/cr/report" for prod devices. Default is "https://clients2.google.com/cr/staging_report" for non prod.
    CRASH_TEST = 'crash-test'   # Causes the browser process to crash on startup.
    CRASHPAD_HANDLER = 'crashpad-handler'   # A process type (switches::kProcessType) that indicates chrome.exe or setup.exe is being launched as crashpad_handler. This is only used on Windows. We bundle the handler into chrome.exe on Windows because there is high probability of a "new" .exe being blocked or interfered with by application firewalls, AV software, etc. On other platforms, crashpad_handler is a standalone executable.
    CREATE_BROWSER_ON_STARTUP_FOR_TESTS = 'create-browser-on-startup-for-tests'   # Some platforms like ChromeOS default to empty desktop. Browser tests may need to add this switch so that at least one browser instance is created on startup. TODO(nkostylev): Investigate if this switch could be removed. (http://crbug.com/148675)
    CROS_REGION = 'cros-region'   # Forces CrOS region value.
    CROS_REGIONS_MODE = 'cros-regions-mode'   # Control regions data load ("" is default).
    CROSH_COMMAND = 'crosh-command'   # Custom crosh command.
    CRYPTAUTH_HTTP_HOST = 'cryptauth-http-host'   # Overrides the default URL for Google APIs (https://www.googleapis.com) used by CryptAuth.
    CRYPTAUTH_V2_DEVICESYNC_HTTP_HOST = 'cryptauth-v2-devicesync-http-host'   # Overrides the default URL for CryptAuth v2 DeviceSync: https://cryptauthdevicesync.googleapis.com.
    CRYPTAUTH_V2_ENROLLMENT_HTTP_HOST = 'cryptauth-v2-enrollment-http-host'   # Overrides the default URL for CryptAuth v2 Enrollment: https://cryptauthenrollment.googleapis.com.
    CUSTOM_DEVTOOLS_FRONTEND = 'custom-devtools-frontend'   # Specify a custom path to devtools for devtools tests
    CUSTOM_LAUNCHER_PAGE = 'custom-launcher-page'   # Specifies the chrome-extension:// URL for the contents of an additional page added to the app launcher.
    CUSTOM_SUMMARY = 'custom_summary'   # Forces a custom summary to be displayed below the update menu item.
    D3D_SUPPORT = 'd3d-support'   # 
    D3D11 = 'd3d11'   # 
    D3D11_NULL = 'd3d11-null'   # Special switches for "NULL"/stub driver implementations.
    D3D9 = 'd3d9'   # 
    DAEMON = 'daemon'   # 
    DARK = 'dark'   # 
    DATA_PATH = 'data-path'   # Makes Content Shell use the given path for its data directory.
    DATA_REDUCTION_PROXY_CLIENT_CONFIG = 'data-reduction-proxy-client-config'   # Uses the encoded ClientConfig instead of fetching one from the config server. This value is always used, regardless of error or expiration. The value should be a base64 encoded binary protobuf.
    DATA_REDUCTION_PROXY_CONFIG_URL = 'data-reduction-proxy-config-url'   # The URL from which to retrieve the Data Reduction Proxy configuration.
    DATA_REDUCTION_PROXY_EXPERIMENT = 'data-reduction-proxy-experiment'   # The name of a Data Reduction Proxy experiment to run. These experiments are defined by the proxy server. Use --force-fieldtrials for Data Reduction Proxy field trials.
    DATA_REDUCTION_PROXY_HTTP_PROXIES = 'data-reduction-proxy-http-proxies'   # The semicolon-separated list of proxy server URIs to override the list of HTTP proxies returned by the Data Saver API. It is illegal to use |kDataReductionProxy| or |kDataReductionProxyFallback| switch in conjunction with |kDataReductionProxyHttpProxies|. If the URI omits a scheme, then the proxy server scheme defaults to HTTP, and if the port is omitted then the default port for that scheme is used. E.g. "http://foo.net:80", "http://foo.net", "foo.net:80", and "foo.net" are all equivalent.
    DATA_REDUCTION_PROXY_PINGBACK_URL = 'data-reduction-proxy-pingback-url'   # 
    DATA_REDUCTION_PROXY_SECURE_PROXY_CHECK_URL = 'data-reduction-proxy-secure-proxy-check-url'   # Sets a secure proxy check URL to test before committing to using the Data Reduction Proxy. Note this check does not go through the Data Reduction Proxy.
    DATA_REDUCTION_PROXY_SERVER_EXPERIMENTS_DISABLED = 'data-reduction-proxy-server-experiments-disabled'   # Disables server experiments that may be enabled through field trial.
    DBUS_STUB = 'dbus-stub'   # Forces the stub implementation of D-Bus clients.
    DEADLINE_TO_SYNCHRONIZE_SURFACES = 'deadline-to-synchronize-surfaces'   # The default number of the BeginFrames to wait to activate a surface with dependencies.
    DEBUG_DEVTOOLS = 'debug-devtools'   # Run devtools tests in debug mode (not bundled and minified)
    DEBUG_ENABLE_FRAME_TOGGLE = 'debug-enable-frame-toggle'   # Enables a frame context menu item that toggles the frame in and out of glass mode (Windows Vista and up only).
    DEBUG_PACKED_APPS = 'debug-packed-apps'   # Adds debugging entries such as Inspect Element to context menus of packed apps.
    DEBUG_PRINT = 'debug-print'   # Enables support to debug printing subsystem.
    DEFAULT = 'default'   # 
    DEFAULT_BACKGROUND_COLOR = 'default-background-color'   # The background color to be used if the page doesn't specify one. Provided as RGBA integer value in hex, e.g. 'ff0000ff' for red or '00000000' for transparent.
    DEFAULT_TILE_HEIGHT = 'default-tile-height'   # 
    DEFAULT_TILE_WIDTH = 'default-tile-width'   # Sets the tile size used by composited layers.
    DEFAULT_WALLPAPER_IS_OEM = 'default-wallpaper-is-oem'   # Indicates that the wallpaper images specified by kAshDefaultWallpaper{Large,Small} are OEM-specific (i.e. they are not downloadable from Google).
    DEFAULT_WALLPAPER_LARGE = 'default-wallpaper-large'   # Default large wallpaper to use (as path to trusted, non-user-writable JPEG file).
    DEFAULT_WALLPAPER_SMALL = 'default-wallpaper-small'   # Default small wallpaper to use (as path to trusted, non-user-writable JPEG file).
    DENY_PERMISSION_PROMPTS = 'deny-permission-prompts'   # Prevents permission prompts from appearing by denying instead of showing prompts.
    DERELICT_DETECTION_TIMEOUT = 'derelict-detection-timeout'   # Time in seconds before a machine at OOBE is considered derelict.
    DERELICT_IDLE_TIMEOUT = 'derelict-idle-timeout'   # Time in seconds before a derelict machines starts demo mode.
    DESKTOP = 'desktop'   # 
    DESKTOP_WINDOW_1080P = 'desktop-window-1080p'   # When present, desktop cast_shell will create 1080p window (provided display resolution is high enough).  Otherwise, cast_shell defaults to 720p.
    DETERMINISTIC_MODE = 'deterministic-mode'   # A meta flag. This sets a number of flags which put the browser into deterministic mode where begin frames should be issued over DevToolsProtocol (experimental).
    DEVICE_MANAGEMENT_URL = 'device-management-url'   # Specifies the URL at which to communicate with the device management backend to fetch configuration policies and perform other device tasks.
    DEVICE_SCALE_FACTOR = 'device-scale-factor'   # Device scale factor passed to certain processes like renderers, etc.
    DEVTOOLS_FLAGS = 'devtools-flags'   # Passes command line parameters to the DevTools front-end.
    DIAGNOSTICS = 'diagnostics'   # Triggers a plethora of diagnostic modes.
    DIAGNOSTICS_FORMAT = 'diagnostics-format'   # Sets the output format for diagnostic modes enabled by diagnostics flag.
    DIAGNOSTICS_RECOVERY = 'diagnostics-recovery'   # Tells the diagnostics mode to do the requested recovery step(s).
    DICE = 'dice'   # 
    DISABLE_2D_CANVAS_CLIP_AA = 'disable-2d-canvas-clip-aa'   # Disable antialiasing on 2d canvas clips
    DISABLE_2D_CANVAS_IMAGE_CHROMIUM = 'disable-2d-canvas-image-chromium'   # Disables Canvas2D rendering into a scanout buffer for overlay support.
    DISABLE_3D_APIS = 'disable-3d-apis'   # Disables client-visible 3D APIs, in particular WebGL and Pepper 3D. This is controlled by policy and is kept separate from the other enable/disable switches to avoid accidentally regressing the policy support for controlling access to these APIs.
    DISABLE_ACCELERATED_2D_CANVAS = 'disable-accelerated-2d-canvas'   # Disable gpu-accelerated 2d canvas.
    DISABLE_ACCELERATED_MJPEG_DECODE = 'disable-accelerated-mjpeg-decode'   # Disable hardware acceleration of mjpeg decode for captured frame, where available.
    DISABLE_ACCELERATED_VIDEO_DECODE = 'disable-accelerated-video-decode'   # Disables hardware acceleration of video decode, where available.
    DISABLE_ACCELERATED_VIDEO_ENCODE = 'disable-accelerated-video-encode'   # Disables hardware acceleration of video encode, where available.
    DISABLE_ANGLE_FEATURES = 'disable-angle-features'   # Disables specified comma separated ANGLE features if found.
    DISABLE_APP_LIST_DISMISS_ON_BLUR = 'disable-app-list-dismiss-on-blur'   # If set, the app list will not be dismissed when it loses focus. This is useful when testing the app list or a custom launcher page. It can still be dismissed via the other methods (like the Esc key).
    DISABLE_APPCONTAINER = 'disable-appcontainer'   # Disable appcontainer/lowbox for renderer on Win8+ platforms.
    DISABLE_ARC_CPU_RESTRICTION = 'disable-arc-cpu-restriction'   # Prevents any CPU restrictions being set on the ARC container. Only meant to be used by tests as some tests may time out if the ARC container is throttled.
    DISABLE_ARC_DATA_WIPE = 'disable-arc-data-wipe'   # Disables android user data wipe on opt out.
    DISABLE_ARC_OPT_IN_VERIFICATION = 'disable-arc-opt-in-verification'   # Disables ARC Opt-in verification process and ARC is enabled by default.
    DISABLE_AUDIO_OUTPUT = 'disable-audio-output'   # 
    DISABLE_AUDIO_SUPPORT_FOR_DESKTOP_SHARE = 'disable-audio-support-for-desktop-share'   # 
    DISABLE_AUTO_RELOAD = 'disable-auto-reload'   # Disable auto-reload of error pages.
    DISABLE_AVFOUNDATION_OVERLAYS = 'disable-avfoundation-overlays'   # Disable use of AVFoundation to draw video content.
    DISABLE_BACKGROUND_NETWORKING = 'disable-background-networking'   # Disable several subsystems which run network requests in the background. This is for use when doing network performance testing to avoid noise in the measurements.
    DISABLE_BACKGROUND_TIMER_THROTTLING = 'disable-background-timer-throttling'   # Disable task throttling of timer tasks from background pages.
    DISABLE_BACKGROUNDING_OCCLUDED_WINDOWS = 'disable-backgrounding-occluded-windows'   # Disable backgrounding renders for occluded windows. Done for tests to avoid nondeterministic behavior.
    DISABLE_BACKING_STORE_LIMIT = 'disable-backing-store-limit'   # Disable limits on the number of backing stores. Can prevent blinking for users with many windows/tabs and lots of memory.
    DISABLE_BEST_EFFORT_TASKS = 'disable-best-effort-tasks'   # Delays execution of TaskPriority::BEST_EFFORT tasks until shutdown.
    DISABLE_BLINK_FEATURES = 'disable-blink-features'   # Disable one or more Blink runtime-enabled features. Use names from runtime_enabled_features.json5, separated by commas. Applied after kEnableBlinkFeatures, and after other flags that change these features.
    DISABLE_BREAKPAD = 'disable-breakpad'   # Disables the crash reporting.
    DISABLE_BUNDLED_PPAPI_FLASH = 'disable-bundled-ppapi-flash'   # Disables the bundled PPAPI version of Flash.
    DISABLE_CANCEL_ALL_TOUCHES = 'disable-cancel-all-touches'   # Disable CancelAllTouches() function for the implementation on cancel single touches.
    DISABLE_CANVAS_AA = 'disable-canvas-aa'   # Disable antialiasing on 2d canvas.
    DISABLE_CHECKER_IMAGING = 'disable-checker-imaging'   # Disabled defering all image decodes to the image decode service, ignoring DecodingMode preferences specified on PaintImage.
    DISABLE_CLIENT_SIDE_PHISHING_DETECTION = 'disable-client-side-phishing-detection'   # Disables the client-side phishing detection feature. Note that even if client-side phishing detection is enabled, it will only be active if the user has opted in to UMA stats and SafeBrowsing is enabled in the preferences.
    DISABLE_COMPONENT_CLOUD_POLICY = 'disable-component-cloud-policy'   # Disables fetching and storing cloud policy for components.
    DISABLE_COMPONENT_EXTENSIONS_WITH_BACKGROUND_PAGES = 'disable-component-extensions-with-background-pages'   # Disable default component extensions with background pages - useful for performance tests where these pages may interfere with perf results.
    DISABLE_COMPONENT_UPDATE = 'disable-component-update'   # 
    DISABLE_COMPOSITED_ANTIALIASING = 'disable-composited-antialiasing'   # Disables layer-edge anti-aliasing in the compositor.
    DISABLE_COOKIE_ENCRYPTION = 'disable-cookie-encryption'   # Whether cookies stored as part of user profile are encrypted.
    DISABLE_CRASH_REPORTER = 'disable-crash-reporter'   # Disable crash reporter for headless. It is enabled by default in official builds.
    DISABLE_D3D11 = 'disable-d3d11'   # Disables use of D3D11.
    DISABLE_DATA_REDUCTION_PROXY_WARMUP_URL_FETCH = 'disable-data-reduction-proxy-warmup-url-fetch'   # Disables fetching of the warmup URL.
    DISABLE_DATA_REDUCTION_PROXY_WARMUP_URL_FETCH_CALLBACK = 'disable-data-reduction-proxy-warmup-url-fetch-callback'   # Disables the warmup URL fetcher to callback into DRP to report the result of the warmup fetch.
    DISABLE_DATABASES = 'disable-databases'   # Disables HTML5 DB support.
    DISABLE_DEFAULT_APPS = 'disable-default-apps'   # Disables installation of default apps on first run. This is used during automated testing.
    DISABLE_DEMO_MODE = 'disable-demo-mode'   # Disables the Chrome OS demo.
    DISABLE_DEV_SHM_USAGE = 'disable-dev-shm-usage'   # The /dev/shm partition is too small in certain VM environments, causing Chrome to fail or crash (see http://crbug.com/715363). Use this flag to work-around this issue (a temporary directory will always be used to create anonymous shared memory files).
    DISABLE_DEVICE_DISABLING = 'disable-device-disabling'   # If this switch is set, the device cannot be remotely disabled by its owner.
    DISABLE_DEVICE_DISCOVERY_NOTIFICATIONS = 'disable-device-discovery-notifications'   # Disables device discovery notifications.
    DISABLE_DINOSAUR_EASTER_EGG = 'disable-dinosaur-easter-egg'   # Disables the dinosaur easter egg on the offline interstitial.
    DISABLE_DIRECT_COMPOSITION = 'disable-direct-composition'   # Disables DirectComposition surface.
    DISABLE_DIRECT_COMPOSITION_LAYERS = 'disable-direct-composition-layers'   # Disables using DirectComposition layers.
    DISABLE_DOMAIN_BLOCKING_FOR_3D_APIS = 'disable-domain-blocking-for-3d-apis'   # Disable the per-domain blocking for 3D APIs after GPU reset. This switch is intended only for tests.
    DISABLE_DOMAIN_RELIABILITY = 'disable-domain-reliability'   # Disables Domain Reliability Monitoring.
    DISABLE_DWM_COMPOSITION = 'disable-dwm-composition'   # Disables use of DWM composition for top level windows.
    DISABLE_ENCRYPTION_MIGRATION = 'disable-encryption-migration'   # Disable encryption migration for user's cryptohome to run latest Arc.
    DISABLE_ES3_GL_CONTEXT = 'disable-es3-gl-context'   # Disables use of ES3 backend (use ES2 backend instead).
    DISABLE_ES3_GL_CONTEXT_FOR_TESTING = 'disable-es3-gl-context-for-testing'   # Disables use of ES3 backend at a lower level, for testing purposes. This isn't guaranteed to work everywhere, so it's test-only.
    DISABLE_EXPLICIT_DMA_FENCES = 'disable-explicit-dma-fences'   # Disable explicit DMA-fences
    DISABLE_EXTENSIONS = 'disable-extensions'   # Disable extensions.
    DISABLE_EXTENSIONS_EXCEPT = 'disable-extensions-except'   # Disable extensions except those specified in a comma-separated list.
    DISABLE_EXTENSIONS_FILE_ACCESS_CHECK = 'disable-extensions-file-access-check'   # Disable checking for user opt-in for extensions that want to inject script into file URLs (ie, always allow it). This is used during automated testing.
    DISABLE_EXTENSIONS_HTTP_THROTTLING = 'disable-extensions-http-throttling'   # Disable the net::URLRequestThrottlerManager functionality for requests originating from extensions.
    DISABLE_FEATURES = 'disable-features'   # Comma-separated list of feature names to disable. See also kEnableFeatures.
    DISABLE_FIELD_TRIAL_CONFIG = 'disable-field-trial-config'   # Disable field trial tests configured in fieldtrial_testing_config.json.
    DISABLE_FILE_SYSTEM = 'disable-file-system'   # Disable FileSystem API.
    DISABLE_FINE_GRAINED_TIME_ZONE_DETECTION = 'disable-fine-grained-time-zone-detection'   # Disables fine grained time zone detection.
    DISABLE_FLASH_3D = 'disable-flash-3d'   # Disable 3D inside of flapper.
    DISABLE_FLASH_STAGE3D = 'disable-flash-stage3d'   # Disable Stage3D inside of flapper.
    DISABLE_FONT_SUBPIXEL_POSITIONING = 'disable-font-subpixel-positioning'   # Force disables font subpixel positioning. This affects the character glyph sharpness, kerning, hinting and layout.
    DISABLE_FRAME_RATE_LIMIT = 'disable-frame-rate-limit'   # Disables begin frame limiting in both cc scheduler and display scheduler. Also implies --disable-gpu-vsync (see //ui/gl/gl_switches.h).
    DISABLE_GAIA_SERVICES = 'disable-gaia-services'   # Disables GAIA services such as enrollment and OAuth session restore. Used by 'fake' telemetry login.
    DISABLE_GESTURE_REQUIREMENT_FOR_PRESENTATION = 'disable-gesture-requirement-for-presentation'   # Disable user gesture requirement for presentation.
    DISABLE_GL_DRAWING_FOR_TESTS = 'disable-gl-drawing-for-tests'   # Disables GL drawing operations which produce pixel output. With this the GL output will not be correct but tests will run faster.
    DISABLE_GL_ERROR_LIMIT = 'disable-gl-error-limit'   # Disable the GL error log limit.
    DISABLE_GL_EXTENSIONS = 'disable-gl-extensions'   # Disables specified comma separated GL Extensions if found.
    DISABLE_GLSL_TRANSLATOR = 'disable-glsl-translator'   # Disable the GLSL translator.
    DISABLE_GPU = 'disable-gpu'   # Disables GPU hardware acceleration.  If software renderer is not in place, then the GPU process won't launch.
    DISABLE_GPU_APPCONTAINER = 'disable-gpu-appcontainer'   # Disables AppContainer sandbox on the GPU process.
    DISABLE_GPU_COMPOSITING = 'disable-gpu-compositing'   # Prevent the compositor from using its GPU implementation.
    DISABLE_GPU_DRIVER_BUG_WORKAROUNDS = 'disable-gpu-driver-bug-workarounds'   # Disable workarounds for various GPU driver bugs.
    DISABLE_GPU_EARLY_INIT = 'disable-gpu-early-init'   # Disable proactive early init of GPU process.
    DISABLE_GPU_LPAC = 'disable-gpu-lpac'   # Disables low-privilege AppContainer sandbox on the GPU process.
    DISABLE_GPU_MEMORY_BUFFER_COMPOSITOR_RESOURCES = 'disable-gpu-memory-buffer-compositor-resources'   # Do not force that all compositor resources be backed by GPU memory buffers.
    DISABLE_GPU_MEMORY_BUFFER_VIDEO_FRAMES = 'disable-gpu-memory-buffer-video-frames'   # Disable GpuMemoryBuffer backed VideoFrames.
    DISABLE_GPU_PROCESS_CRASH_LIMIT = 'disable-gpu-process-crash-limit'   # For tests, to disable the limit on the number of times the GPU process may be restarted.
    DISABLE_GPU_PROCESS_FOR_DX12_VULKAN_INFO_COLLECTION = 'disable-gpu-process-for-dx12-vulkan-info-collection'   # Disables the non-sandboxed GPU process for DX12 and Vulkan info collection
    DISABLE_GPU_PROGRAM_CACHE = 'disable-gpu-program-cache'   # Turn off gpu program caching
    DISABLE_GPU_RASTERIZATION = 'disable-gpu-rasterization'   # Disable GPU rasterization, i.e. rasterize on the CPU only. Overrides the kEnableGpuRasterization and kForceGpuRasterization flags.
    DISABLE_GPU_SANDBOX = 'disable-gpu-sandbox'   # Disables the GPU process sandbox.
    DISABLE_GPU_SHADER_DISK_CACHE = 'disable-gpu-shader-disk-cache'   # Disables the GPU shader on disk cache.
    DISABLE_GPU_VSYNC = 'disable-gpu-vsync'   # Stop the GPU from synchronizing presentation with vblank.
    DISABLE_GPU_WATCHDOG = 'disable-gpu-watchdog'   # Disable the thread that crashes the GPU process if it stops responding to messages.
    DISABLE_HANG_MONITOR = 'disable-hang-monitor'   # Suppresses hang monitor dialogs in renderer processes.  This may allow slow unload handlers on a page to prevent the tab from closing, but the Task Manager can be used to terminate the offending process in this case.
    DISABLE_HEADLESS_MODE = 'disable-headless-mode'   # Disables the shell from beginning in headless mode. Tests will then attempt to use the hardware GPU for rendering. This is only followed when kRunWebTests is set.
    DISABLE_HID_DETECTION_ON_OOBE = 'disable-hid-detection-on-oobe'   # Disables HID-detection OOBE screen.
    DISABLE_HIGHRES_TIMER = 'disable-highres-timer'   # Disable high-resolution timer on Windows.
    DISABLE_HISTOGRAM_CUSTOMIZER = 'disable-histogram-customizer'   # Disable the RenderThread's HistogramCustomizer.
    DISABLE_HOSTED_APP_SHIM_CREATION = 'disable-hosted-app-shim-creation'   # Disables app shim creation for hosted apps on Mac.
    DISABLE_IMAGE_ANIMATION_RESYNC = 'disable-image-animation-resync'   # Disallow image animations to be reset to the beginning to avoid skipping many frames. Only effective if compositor image animations are enabled.
    DISABLE_IN_PROCESS_STACK_TRACES = 'disable-in-process-stack-traces'   # Disables the in-process stack traces.
    DISABLE_IOS_PASSWORD_SUGGESTIONS = 'disable-ios-password-suggestions'   # Disable showing available password credentials in the keyboard accessory view when focused on form fields.
    DISABLE_IPC_FLOODING_PROTECTION = 'disable-ipc-flooding-protection'   # Disables the IPC flooding protection. It is activated by default. Some javascript functions can be used to flood the browser process with IPC. This protection limits the rate at which they can be used.
    DISABLE_JAVASCRIPT_HARMONY_SHIPPING = 'disable-javascript-harmony-shipping'   # Disable latest shipping ECMAScript 6 features.
    DISABLE_KILL_AFTER_BAD_IPC = 'disable-kill-after-bad-ipc'   # Don't kill a child process when it sends a bad IPC message.  Apart from testing, it is a bad idea from a security perspective to enable this switch.
    DISABLE_LCD_TEXT = 'disable-lcd-text'   # Disables LCD text.
    DISABLE_LEGACY_WINDOW = 'disable-legacy-window'   # Disable the Legacy Window which corresponds to the size of the WebContents.
    DISABLE_LOCAL_STORAGE = 'disable-local-storage'   # Disable LocalStorage.
    DISABLE_LOGGING = 'disable-logging'   # Force logging to be disabled.  Logging is enabled by default in debug builds.
    DISABLE_LOGGING_REDIRECT = 'disable-logging-redirect'   # Disables logging redirect for testing.
    DISABLE_LOGIN_ANIMATIONS = 'disable-login-animations'   # Avoid doing expensive animations upon login.
    DISABLE_LOGIN_SCREEN_APPS = 'disable-login-screen-apps'   # Disables apps on the login screen. By default, they are allowed and can be installed through policy.
    DISABLE_LOW_END_DEVICE_MODE = 'disable-low-end-device-mode'   # Force disabling of low-end device mode when set.
    DISABLE_LOW_LATENCY_DXVA = 'disable-low-latency-dxva'   # Disables using CODECAPI_AVLowLatencyMode when creating DXVA decoders.
    DISABLE_LOW_RES_TILING = 'disable-low-res-tiling'   # When using CPU rasterizing disable low resolution tiling. This uses less power, particularly during animations, but more white may be seen during fast scrolling especially on slower devices.
    DISABLE_MAC_OVERLAYS = 'disable-mac-overlays'   # Fall back to using CAOpenGLLayers display content, instead of the IOSurface based overlay display path.
    DISABLE_MACHINE_CERT_REQUEST = 'disable-machine-cert-request'   # Disables requests for an enterprise machine certificate during attestation.
    DISABLE_MAIN_FRAME_BEFORE_ACTIVATION = 'disable-main-frame-before-activation'   # Disables sending the next BeginMainFrame before the previous commit activates. Overrides the kEnableMainFrameBeforeActivation flag.
    DISABLE_MEDIA_SESSION_API = 'disable-media-session-api'   # Disable Media Session API
    DISABLE_MEDIA_SUSPEND = 'disable-media-suspend'   # 
    DISABLE_MODAL_ANIMATIONS = 'disable-modal-animations'   # Disable animations for showing and hiding modal dialogs.
    DISABLE_MOJO_RENDERER = 'disable-mojo-renderer'   # Rather than use the renderer hosted remotely in the media service, fall back to the default renderer within content_renderer. Does not change the behavior of the media service.
    DISABLE_MULTI_DISPLAY_LAYOUT = 'disable-multi-display-layout'   # Disables the multiple display layout UI.
    DISABLE_NAMESPACE_SANDBOX = 'disable-namespace-sandbox'   # Disables usage of the namespace sandbox.
    DISABLE_NEW_CONTENT_RENDERING_TIMEOUT = 'disable-new-content-rendering-timeout'   # Disables clearing the rendering output of a renderer when it didn't commit new output for a while after a top-frame navigation.
    DISABLE_NOTIFICATIONS = 'disable-notifications'   # Disables the Web Notification and the Push APIs.
    DISABLE_NV12_DXGI_VIDEO = 'disable-nv12-dxgi-video'   # Disables the video decoder from drawing to an NV12 textures instead of ARGB.
    DISABLE_OFFER_STORE_UNMASKED_WALLET_CARDS = 'disable-offer-store-unmasked-wallet-cards'   # Force hiding the local save checkbox in the autofill dialog box for getting the full credit card number for a wallet card. The card will never be stored locally.
    DISABLE_OOP_RASTERIZATION = 'disable-oop-rasterization'   # Disables OOP rasterization.  Takes precedence over the enable flag.
    DISABLE_ORIGIN_TRIAL_CONTROLLED_BLINK_FEATURES = 'disable-origin-trial-controlled-blink-features'   # Disables all RuntimeEnabledFeatures that can be enabled via OriginTrials.
    DISABLE_OVERSCROLL_EDGE_EFFECT = 'disable-overscroll-edge-effect'   # Disable overscroll edge effects like those found in Android views.
    DISABLE_PARTIAL_RASTER = 'disable-partial-raster'   # Disable partial raster in the renderer. Disabling this switch also disables the use of persistent gpu memory buffers.
    DISABLE_PEPPER_3D = 'disable-pepper-3d'   # Disable Pepper3D.
    DISABLE_PEPPER_3D_IMAGE_CHROMIUM = 'disable-pepper-3d-image-chromium'   # Disable Image Chromium for Pepper 3d.
    DISABLE_PER_USER_TIMEZONE = 'disable-per-user-timezone'   # Disables per-user timezone.
    DISABLE_PERFETTO = 'disable-perfetto'   # Disables the perfetto tracing backend. We need a separate command line argument from the kTracingPerfettoBackend feature, because feature flags are parsed too late during startup for early startup tracing support.
    DISABLE_PERMISSIONS_API = 'disable-permissions-api'   # Disables the Permissions API.
    DISABLE_PINCH = 'disable-pinch'   # Disables compositor-accelerated touch-screen pinch gestures.
    DISABLE_PNACL_CRASH_THROTTLING = 'disable-pnacl-crash-throttling'   # Disables crash throttling for Portable Native Client.
    DISABLE_POPUP_BLOCKING = 'disable-popup-blocking'   # Disable pop-up blocking.
    DISABLE_PREFER_COMPOSITING_TO_LCD_TEXT = 'disable-prefer-compositing-to-lcd-text'   # Disable the creation of compositing layers when it would prevent LCD text.
    DISABLE_PRESENTATION_API = 'disable-presentation-api'   # Disables the Presentation API.
    DISABLE_PRINT_PREVIEW = 'disable-print-preview'   # Disables print preview (For testing, and for users who don't like us. :[ )
    DISABLE_PROMPT_ON_REPOST = 'disable-prompt-on-repost'   # Normally when the user attempts to navigate to a page that was the result of a post we prompt to make sure they want to. This switch may be used to disable that check. This switch is used during automated testing.
    DISABLE_PULL_TO_REFRESH_EFFECT = 'disable-pull-to-refresh-effect'   # Disable the pull-to-refresh effect when vertically overscrolling content.
    DISABLE_PUSHSTATE_THROTTLE = 'disable-pushstate-throttle'   # Disables throttling of history.pushState/replaceState calls.
    DISABLE_READING_FROM_CANVAS = 'disable-reading-from-canvas'   # Taints all <canvas> elements, regardless of origin.
    DISABLE_REMOTE_CORE_ANIMATION = 'disable-remote-core-animation'   # Disable use of cross-process CALayers to display content directly from the GPU process on Mac.
    DISABLE_REMOTE_FONTS = 'disable-remote-fonts'   # Disables remote web font support. SVG font should always work whether this option is specified or not.
    DISABLE_REMOTE_PLAYBACK_API = 'disable-remote-playback-api'   # Disables the RemotePlayback API.
    DISABLE_RENDERER_ACCESSIBILITY = 'disable-renderer-accessibility'   # Turns off the accessibility in the renderer.
    DISABLE_RENDERER_BACKGROUNDING = 'disable-renderer-backgrounding'   # Prevent renderer process backgrounding when set.
    DISABLE_RESOURCE_SCHEDULER = 'disable-resource-scheduler'   # Whether the ResourceScheduler is disabled.  Note this is only useful for C++ Headless embedders who need to implement their own resource scheduling.
    DISABLE_RGBA_4444_TEXTURES = 'disable-rgba-4444-textures'   # Disables RGBA_4444 textures.
    DISABLE_ROLLBACK_OPTION = 'disable-rollback-option'   # Disables rollback option on reset screen.
    DISABLE_RTC_SMOOTHNESS_ALGORITHM = 'disable-rtc-smoothness-algorithm'   # Disables the new rendering algorithm for webrtc, which is designed to improve the rendering smoothness.
    DISABLE_SCREEN_ORIENTATION_LOCK = 'disable-screen-orientation-lock'   # Disable the locking feature of the screen orientation API.
    DISABLE_SEARCH_GEOLOCATION_DISCLOSURE = 'disable-search-geolocation-disclosure'   # Disables showing the search geolocation disclosure UI. Used for perf testing.
    DISABLE_SECCOMP_FILTER_SANDBOX = 'disable-seccomp-filter-sandbox'   # Disable the seccomp filter sandbox (seccomp-bpf) (Linux only).
    DISABLE_SETUID_SANDBOX = 'disable-setuid-sandbox'   # Disable the setuid sandbox (Linux only).
    DISABLE_SHADER_NAME_HASHING = 'disable-shader-name-hashing'   # Turn off user-defined name hashing in shaders.
    DISABLE_SHARED_WORKERS = 'disable-shared-workers'   # Disable shared workers.
    DISABLE_SIGNIN_FRAME_CLIENT_CERT_USER_SELECTION = 'disable-signin-frame-client-cert-user-selection'   # Disables user selection of client certificate on the sign-in frame on the Chrome OS sign-in profile. TODO(pmarko): Remove this flag in M-65 when the DeviceLoginScreenAutoSelectCertificateForUrls policy is enabled on the server side (https://crbug.com/723849) and completely disable user selection of certificates on the sign-in frame.
    DISABLE_SIGNIN_FRAME_CLIENT_CERTS = 'disable-signin-frame-client-certs'   # Disables client certificate authentication on the sign-in frame on the Chrome OS sign-in profile. TODO(pmarko): Remove this flag in M-66 if no issues are found (https://crbug.com/723849).
    DISABLE_SIGNIN_SCOPED_DEVICE_ID = 'disable-signin-scoped-device-id'   # Disables sending signin scoped device id to LSO with refresh token request.
    DISABLE_SITE_ISOLATION_FOR_POLICY = 'disable-site-isolation-for-policy'   # Just like kDisableSiteIsolation, but doesn't show the "stability and security will suffer" butter bar warning.
    DISABLE_SITE_ISOLATION_TRIALS = 'disable-site-isolation-trials'   # Disables site isolation.  Note that the opt-in (to site-per-process, isolate-origins, etc.) via enterprise policy and/or cmdline takes precedence over the kDisableSiteIsolation switch (i.e. the opt-in takes effect despite potential presence of kDisableSiteIsolation switch).  Note that for historic reasons the name of the switch misleadingly mentions "trials", but the switch also disables the default site isolation that ships on desktop since M67.  The name of the switch is preserved for backcompatibility of chrome://flags.
    DISABLE_SKIA_RUNTIME_OPTS = 'disable-skia-runtime-opts'   # Do not use runtime-detected high-end CPU optimizations in Skia.  This is useful for forcing a baseline code path for e.g. web tests.
    DISABLE_SMOOTH_SCROLLING = 'disable-smooth-scrolling'   # Disable smooth scrolling for testing.
    DISABLE_SOFTWARE_COMPOSITING_FALLBACK = 'disable-software-compositing-fallback'   # For tests, to disable falling back to software compositing if the GPU Process has crashed, and reached the GPU Process crash limit.
    DISABLE_SOFTWARE_RASTERIZER = 'disable-software-rasterizer'   # Disables the use of a 3D software rasterizer.
    DISABLE_SPEECH_API = 'disable-speech-api'   # Disables the Web Speech API (both speech recognition and synthesis).
    DISABLE_SPEECH_SYNTHESIS_API = 'disable-speech-synthesis-api'   # Disables the speech synthesis part of Web Speech API.
    DISABLE_SYNC = 'disable-sync'   # Disables syncing browser data to a Google Account.
    DISABLE_SYNC_TYPES = 'disable-sync-types'   # Disables syncing one or more sync data types that are on by default. See sync/base/model_type.h for possible types. Types should be comma separated, and follow the naming convention for string representation of model types, e.g.: --disable-synctypes='Typed URLs, Bookmarks, Autofill Profiles'
    DISABLE_TEST_ROOT_CERTS = 'disable-test-root-certs'   # Disables adding the test certs in the network process.
    DISABLE_THIRD_PARTY_KEYBOARD_WORKAROUND = 'disable-third-party-keyboard-workaround'   # Disables the 3rd party keyboard omnibox workaround.
    DISABLE_THREADED_ANIMATION = 'disable-threaded-animation'   # 
    DISABLE_THREADED_COMPOSITING = 'disable-threaded-compositing'   # Disable multithreaded GPU compositing of web content.
    DISABLE_THREADED_SCROLLING = 'disable-threaded-scrolling'   # Disable multithreaded, compositor scrolling of web content.
    DISABLE_TIMEOUTS_FOR_PROFILING = 'disable-timeouts-for-profiling'   # Disable timeouts that may cause the browser to die when running slowly. This is useful if running with profiling (such as debug malloc).
    DISABLE_TOUCH_ADJUSTMENT = 'disable-touch-adjustment'   # Disables touch adjustment.
    DISABLE_TOUCH_DRAG_DROP = 'disable-touch-drag-drop'   # Disables touch event based drag and drop.
    DISABLE_USB_KEYBOARD_DETECT = 'disable-usb-keyboard-detect'   # Disables the USB keyboard detection for blocking the OSK on Win8+.
    DISABLE_V8_IDLE_TASKS = 'disable-v8-idle-tasks'   # Disable V8 idle tasks.
    DISABLE_VOLUME_ADJUST_SOUND = 'disable-volume-adjust-sound'   # Disables volume adjust sound.
    DISABLE_VSYNC_FOR_TESTS = 'disable-vsync-for-tests'   # 
    DISABLE_VULKAN_FALLBACK_TO_GL_FOR_TESTING = 'disable-vulkan-fallback-to-gl-for-testing'   # Disables falling back to GL based hardware rendering if initializing Vulkan fails. This is to allow tests to catch regressions in Vulkan.
    DISABLE_VULKAN_SURFACE = 'disable-vulkan-surface'   # Disables VK_KHR_surface extension. Instead of using swapchain, bitblt will be used for present render result on screen.
    DISABLE_WAKE_ON_WIFI = 'disable-wake-on-wifi'   # Disables wake on wifi features.
    DISABLE_WEB_SECURITY = 'disable-web-security'   # Don't enforce the same-origin policy. (Used by people testing their sites.)
    DISABLE_WEBGL = 'disable-webgl'   # Disable all versions of WebGL.
    DISABLE_WEBGL_IMAGE_CHROMIUM = 'disable-webgl-image-chromium'   # Disables WebGL rendering into a scanout buffer for overlay support.
    DISABLE_WEBGL2 = 'disable-webgl2'   # Disable WebGL2.
    DISABLE_WEBRTC_ENCRYPTION = 'disable-webrtc-encryption'   # Disables encryption of RTP Media for WebRTC. When Chrome embeds Content, it ignores this switch on its stable and beta channels.
    DISABLE_WEBRTC_HW_DECODING = 'disable-webrtc-hw-decoding'   # Disables HW decode acceleration for WebRTC.
    DISABLE_WEBRTC_HW_ENCODING = 'disable-webrtc-hw-encoding'   # Disables HW encode acceleration for WebRTC.
    DISABLE_WIN32K_LOCKDOWN = 'disable-win32k-lockdown'   # Disables the Win32K process mitigation policy for child processes.
    DISABLE_WINDOWS10_CUSTOM_TITLEBAR = 'disable-windows10-custom-titlebar'   # Disables custom-drawing the window titlebar on Windows 10.
    DISABLE_XR_DEVICE_CONSENT_PROMPT_FOR_TESTING = 'disable-xr-device-consent-prompt-for-testing'   # Disables prompting the user for consent before fulfilling WebXR's requestSession(). Use only when absolutely necessary, such as for perf tests.
    DISABLE_YUV_IMAGE_DECODING = 'disable-yuv-image-decoding'   # Disable YUV image decoding for those formats and cases where it's supported. Has no effect unless GPU rasterization is enabled.
    DISABLE_ZERO_BROWSERS_OPEN_FOR_TESTS = 'disable-zero-browsers-open-for-tests'   # Some tests seem to require the application to close when the last browser window is closed. Thus, we need a switch to force this behavior for ChromeOS Aura, disable "zero window mode". TODO(pkotwicz): Investigate if this bug can be removed. (http://crbug.com/119175)
    DISABLE_ZERO_COPY = 'disable-zero-copy'   # Disable rasterizer that writes directly to GPU memory associated with tiles.
    DISABLE_ZERO_COPY_DXGI_VIDEO = 'disable-zero-copy-dxgi-video'   # Disable the video decoder from drawing directly to a texture.
    DISABLED = 'disabled'   # 
    DISALLOW_NON_EXACT_RESOURCE_REUSE = 'disallow-non-exact-resource-reuse'   # Disable re-use of non-exact resources to fulfill ResourcePool requests. Intended only for use in layout or pixel tests to reduce noise.
    DISK_CACHE_DIR = 'disk-cache-dir'   # Use a specific disk cache location, rather than one derived from the UserDatadir.
    DISK_CACHE_SIZE = 'disk-cache-size'   # Forces the maximum disk space to be used by the disk cache, in bytes.
    DISPLAY = 'display'   # Which X11 display to connect to. Emulates the GTK+ "--display=" command line argument.
    DMG_DEVICE = 'dmg-device'   # When switches::kProcessType is switches::kRelauncherProcess, if this switch is also present, the relauncher process will unmount and eject a mounted disk image and move its disk image file to the trash.  The argument's value must be a BSD device name of the form "diskN" or "diskNsM".
    DNS_LOG_DETAILS = 'dns-log-details'   # 
    DOCUMENT_USER_ACTIVATION_REQUIRED = 'document-user-activation-required'   # Autoplay policy that requires a document user activation.
    DOM_AUTOMATION = 'dom-automation'   # Specifies if the |DOMAutomationController| needs to be bound in the renderer. This binding happens on per-frame basis and hence can potentially be a performance bottleneck. One should only enable it when automating dom based tests.
    DONT_REQUIRE_LITEPAGE_REDIRECT_INFOBAR = 'dont-require-litepage-redirect-infobar'   # Do not require the user notification InfoBar to be shown before triggering a Lite Page Redirect preview.
    DRAW_VIEW_BOUNDS_RECTS = 'draw-view-bounds-rects'   # Draws a semitransparent rect to indicate the bounds of each view.
    DUMP_BLINK_RUNTIME_CALL_STATS = 'dump-blink-runtime-call-stats'   # Logs Runtime Call Stats for Blink. --single-process also needs to be used along with this for the stats to be logged.
    DUMP_BROWSER_HISTOGRAMS = 'dump-browser-histograms'   # Requests that a running browser process dump its collected histograms to a given file. The file is overwritten if it exists.
    DUMP_DOM = 'dump-dom'   # Instructs headless_shell to print document.body.innerHTML to stdout.
    DUMP_RAW_LOGS = 'dump-raw-logs'   # Dump the raw logs to a file with the same base name as the executable. The dumped file is a raw protobuf and has a "pb" extension. WARNING: this switch is used by internal test systems. Be careful when making changes.
    EASY_UNLOCK_APP_PATH = 'easy-unlock-app-path'   # Overrides the path of Easy Unlock component app.
    EDGE_TOUCH_FILTERING = 'edge-touch-filtering'   # Tells Chrome to do edge touch filtering. Useful for convertible tablet.
    EGL = 'egl'   # 
    ELEVATE = 'elevate'   # 
    ELEVATED = 'elevated'   # Identify that the process is already supposed to be elevated, so that we don't try again.
    EMBEDDED_EXTENSION_OPTIONS = 'embedded-extension-options'   # Enables extension options to be embedded in chrome://extensions rather than a new tab.
    EMULATE_SHADER_PRECISION = 'emulate-shader-precision'   # Emulate ESSL lowp and mediump float precisions by mutating the shaders to round intermediate values in ANGLE.
    ENABLE_ACCELERATED_2D_CANVAS = 'enable-accelerated-2d-canvas'   # Enable accelerated 2D canvas.
    ENABLE_ACCESSIBILITY_OBJECT_MODEL = 'enable-accessibility-object-model'   # Enable the experimental Accessibility Object Model APIs in development.
    ENABLE_ACCESSIBILITY_TAB_SWITCHER = 'enable-accessibility-tab-switcher'   # Enable the accessibility tab switcher.
    ENABLE_ADAPTIVE_SELECTION_HANDLE_ORIENTATION = 'enable-adaptive-selection-handle-orientation'   # Enable inverting of selection handles so that they are not clipped by the viewport boundaries.
    ENABLE_AGGRESSIVE_DOMSTORAGE_FLUSHING = 'enable-aggressive-domstorage-flushing'   # Enable the aggressive flushing of DOM Storage to minimize data loss.
    ENABLE_ANGLE_FEATURES = 'enable-angle-features'   # ANGLE features are defined per-backend in third_party/angle/include/platform Enables specified comma separated ANGLE features if found.
    ENABLE_APP_LIST = 'enable-app-list'   # If set, the app list will be enabled as if enabled from CWS.
    ENABLE_APPCONTAINER = 'enable-appcontainer'   # Ensable appcontainer/lowbox for renderer on Win8+ platforms.
    ENABLE_ARC = 'enable-arc'   # DEPRECATED. Please use --arc-availability=officially-supported. Enables starting the ARC instance upon session start.
    ENABLE_ARC_OOBE_OPTIN_NO_SKIP = 'enable-arc-oobe-optin-no-skip'   # Enables "hide Skip button" for ARC setup in the OOBE flow.
    ENABLE_ARCVM = 'enable-arcvm'   # Enables ARC VM.
    ENABLE_AUDIO_DEBUG_RECORDINGS_FROM_EXTENSION = 'enable-audio-debug-recordings-from-extension'   # If the WebRTC logging private API is active, enables audio debug recordings.
    ENABLE_AUTO_RELOAD = 'enable-auto-reload'   # Enable auto-reload of error pages.
    ENABLE_AUTOMATION = 'enable-automation'   # Enable indication that browser is controlled by automation.
    ENABLE_BACKGROUND_TRACING = 'enable-background-tracing'   # Enables background and upload trace to trace-upload-url. Trigger rules are pass as an argument.
    ENABLE_BEGIN_FRAME_CONTROL = 'enable-begin-frame-control'   # Whether or not begin frames should be issued over DevToolsProtocol (experimental).
    ENABLE_BENCHMARKING = 'enable-benchmarking'   # TODO(asvitkine): Consider removing or renaming this functionality. Enables the benchmarking extensions.
    ENABLE_BLE_ADVERTISING_IN_APPS = 'enable-ble-advertising-in-apps'   # Enable BLE Advertisiing in apps.
    ENABLE_BLINK_FEATURES = 'enable-blink-features'   # Enable one or more Blink runtime-enabled features. Use names from runtime_enabled_features.json5, separated by commas. Applied before kDisableBlinkFeatures, and after other flags that change these features.
    ENABLE_BOOKMARK_UNDO = 'enable-bookmark-undo'   # Enables the multi-level undo system for bookmarks.
    ENABLE_CARET_BROWSING = 'enable-caret-browsing'   # Enable native caret browsing, in which a moveable cursor is placed on a web page, allowing a user to select and navigate through non-editable text using just a keyboard. See https://crbug.com/977390 for links to i2i.
    ENABLE_CAST_RECEIVER = 'enable-cast-receiver'   # Enables the Cast Receiver.
    ENABLE_CHROME_TRACING_COMPUTATION = 'enable-chrome-tracing-computation'   # Enable the tracing service.
    ENABLE_CLOUD_PRINT_PROXY = 'enable-cloud-print-proxy'   # This applies only when the process type is "service". Enables the Cloud Print Proxy component within the service process.
    ENABLE_CLOUD_PRINT_XPS = 'enable-cloud-print-xps'   # Fallback to XPS. By default connector uses CDD.
    ENABLE_CONSUMER_KIOSK = 'enable-consumer-kiosk'   # Enables consumer kiosk mode for Chrome OS.
    ENABLE_CRASH_REPORTER = 'enable-crash-reporter'   # Enable crash reporter for headless.
    ENABLE_CRASH_REPORTER_FOR_TESTING = 'enable-crash-reporter-for-testing'   # Used for turning on Breakpad crash reporting in a debug environment where crash reporting is typically compiled but disabled.
    ENABLE_CRX_HASH_CHECK = 'enable-crx-hash-check'   # Enable package hash check: the .crx file sha256 hash sum should be equal to the one received from update manifest.
    ENABLE_DATA_REDUCTION_PROXY_BYPASS_WARNING = 'enable-data-reduction-proxy-bypass-warning'   # Enable the data reduction proxy bypass warning.
    ENABLE_DATA_REDUCTION_PROXY_FORCE_PINGBACK = 'enable-data-reduction-proxy-force-pingback'   # Enables sending a pageload metrics pingback after every page load.
    ENABLE_DATA_REDUCTION_PROXY_SAVINGS_PROMO = 'enable-data-reduction-proxy-savings-promo'   # Enables a 1 MB savings promo for the data reduction proxy.
    ENABLE_DEFER_ALL_SCRIPT_WITHOUT_OPTIMIZATION_HINTS = 'enable-defer-all-script-without-optimization-hints'   # Allows defer script preview on all https pages even if optimization hints are missing for that webpage.
    ENABLE_DEVICE_DISCOVERY_NOTIFICATIONS = 'enable-device-discovery-notifications'   # Enable device discovery notifications.
    ENABLE_DEVTOOLS_EXPERIMENTS = 'enable-devtools-experiments'   # If true devtools experimental settings are enabled.
    ENABLE_DIRECT_COMPOSITION_LAYERS = 'enable-direct-composition-layers'   # Enables using DirectComposition layers, even if hardware overlays aren't supported.
    ENABLE_DISTILLABILITY_SERVICE = 'enable-distillability-service'   # 
    ENABLE_DOM_DISTILLER = 'enable-dom-distiller'   # 
    ENABLE_DOMAIN_RELIABILITY = 'enable-domain-reliability'   # Enables Domain Reliability Monitoring.
    ENABLE_ENCRYPTION_MIGRATION = 'enable-encryption-migration'   # Enables encryption migration for user's cryptohome to run latest Arc.
    ENABLE_ENCRYPTION_SELECTION = 'enable-encryption-selection'   # Enables the feature of allowing the user to disable the backend via a setting.
    ENABLE_EXCLUSIVE_AUDIO = 'enable-exclusive-audio'   # Use exclusive mode audio streaming for Windows Vista and higher. Leads to lower latencies for audio streams which uses the AudioParameters::AUDIO_PCM_LOW_LATENCY audio path. See http://msdn.microsoft.com/en-us/library/windows/desktop/dd370844.aspx for details.
    ENABLE_EXPERIMENTAL_ACCESSIBILITY_AUTOCLICK = 'enable-experimental-accessibility-autoclick'   # Shows additional automatic click features that haven't launched yet.
    ENABLE_EXPERIMENTAL_ACCESSIBILITY_CHROMEVOX_LANGUAGE_SWITCHING = 'enable-experimental-accessibility-chromevox-language-switching'   # Enables language switching feature that hasn't launched yet.
    ENABLE_EXPERIMENTAL_ACCESSIBILITY_FEATURES = 'enable-experimental-accessibility-features'   # Shows additional checkboxes in Settings to enable Chrome OS accessibility features that haven't launched yet.
    ENABLE_EXPERIMENTAL_ACCESSIBILITY_LABELS_DEBUGGING = 'enable-experimental-accessibility-labels-debugging'   # Enables support for visually debugging the accessibility labels feature, which provides images descriptions for screen reader users.
    ENABLE_EXPERIMENTAL_ACCESSIBILITY_LANGUAGE_DETECTION = 'enable-experimental-accessibility-language-detection'   # Enables language detection on in-page text content which is then exposed to accessibility technology such as screen readers.
    ENABLE_EXPERIMENTAL_ACCESSIBILITY_SWITCH_ACCESS = 'enable-experimental-accessibility-switch-access'   # Shows setting to enable Switch Access before it has launched.
    ENABLE_EXPERIMENTAL_ACCESSIBILITY_SWITCH_ACCESS_TEXT = 'enable-experimental-accessibility-switch-access-text'   # Enables in progress Switch Access features for text input.
    ENABLE_EXPERIMENTAL_EXTENSION_APIS = 'enable-experimental-extension-apis'   # Enables extension APIs that are in development.
    ENABLE_EXPERIMENTAL_UI_AUTOMATION = 'enable-experimental-ui-automation'   # Toggles between IAccessible and UI Automation platform API.
    ENABLE_EXPERIMENTAL_WEB_PLATFORM_FEATURES = 'enable-experimental-web-platform-features'   # Enables Web Platform features that are in development.
    ENABLE_EXTENSION_ACTIVITY_LOG_TESTING = 'enable-extension-activity-log-testing'   # 
    ENABLE_EXTENSION_ACTIVITY_LOGGING = 'enable-extension-activity-logging'   # Enables logging for extension activity.
    ENABLE_EXTENSION_ASSETS_SHARING = 'enable-extension-assets-sharing'   # Enables sharing assets for installed default apps.
    ENABLE_FEATURES = 'enable-features'   # Comma-separated list of feature names to enable. See also kDisableFeatures.
    ENABLE_FIRST_RUN_UI_TRANSITIONS = 'enable-first-run-ui-transitions'   # Enables animated transitions during first-run tutorial. TODO(https://crbug.com/945966): Remove this.
    ENABLE_FONT_ANTIALIASING = 'enable-font-antialiasing'   # Enable font antialiasing for pixel tests.
    ENABLE_FULLSCREEN_TOOLBAR_REVEAL = 'enable-fullscreen-toolbar-reveal'   # Enables the fullscreen toolbar to reveal itself for tab strip changes.
    ENABLE_GAMEPAD_BUTTON_AXIS_EVENTS = 'enable-gamepad-button-axis-events'   # Enables gamepadbuttondown, gamepadbuttonup, gamepadbuttonchange, gamepadaxismove non-standard gamepad events.
    ENABLE_GPU_APPCONTAINER = 'enable-gpu-appcontainer'   # Enables AppContainer sandbox on the GPU process.
    ENABLE_GPU_BENCHMARKING = 'enable-gpu-benchmarking'   # Enables the GPU benchmarking extension
    ENABLE_GPU_CLIENT_LOGGING = 'enable-gpu-client-logging'   # Enable GPU client logging.
    ENABLE_GPU_CLIENT_TRACING = 'enable-gpu-client-tracing'   # Enables TRACE for GL calls in the renderer.
    ENABLE_GPU_COMMAND_LOGGING = 'enable-gpu-command-logging'   # Turn on Logging GPU commands.
    ENABLE_GPU_DEBUGGING = 'enable-gpu-debugging'   # Turn on Calling GL Error after every command.
    ENABLE_GPU_DRIVER_DEBUG_LOGGING = 'enable-gpu-driver-debug-logging'   # Enable logging of GPU driver debug messages.
    ENABLE_GPU_MEMORY_BUFFER_COMPOSITOR_RESOURCES = 'enable-gpu-memory-buffer-compositor-resources'   # Specify that all compositor resources should be backed by GPU memory buffers.
    ENABLE_GPU_MEMORY_BUFFER_VIDEO_FRAMES = 'enable-gpu-memory-buffer-video-frames'   # Enable GpuMemoryBuffer backed VideoFrames.
    ENABLE_GPU_RASTERIZATION = 'enable-gpu-rasterization'   # Allow heuristics to determine when a layer tile should be drawn with the Skia GPU backend. Only valid with GPU accelerated compositing + impl-side painting.
    ENABLE_GPU_SERVICE_LOGGING = 'enable-gpu-service-logging'   # Turns on GPU logging (debug build only).
    ENABLE_GPU_SERVICE_TRACING = 'enable-gpu-service-tracing'   # Turns on calling TRACE for every GL call.
    ENABLE_HARDWARE_OVERLAYS = 'enable-hardware-overlays'   # Enable compositing individual elements via hardware overlays when permitted by device. Setting the flag to "single-fullscreen" will try to promote a single fullscreen overlay and use it as main framebuffer where possible.
    ENABLE_HOSTED_APP_QUIT_NOTIFICATION = 'enable-hosted-app-quit-notification'   # Shows a notification when quitting Chrome with hosted apps running. Default behavior is to also quit all hosted apps.
    ENABLE_HUNG_RENDERER_INFOBAR = 'enable-hung-renderer-infobar'   # Enables a hung renderer InfoBar allowing the user to close or wait on unresponsive web content.
    ENABLE_INPUT = 'enable-input'   # Enables input event handling by the window manager.
    ENABLE_IOS_HANDOFF_TO_OTHER_DEVICES = 'enable-ios-handoff-to-other-devices'   # Enables support for Handoff from Chrome on iOS to the default browser of other Apple devices.
    ENABLE_LCD_TEXT = 'enable-lcd-text'   # Enables LCD text.
    ENABLE_LEAK_DETECTION = 'enable-leak-detection'   # Enables the leak detection of loading webpages. This allows us to check whether or not reloading a webpage releases web-related objects correctly.
    ENABLE_LOCAL_FILE_ACCESSES = 'enable-local-file-accesses'   # Enable file accesses. It should not be enabled for most Cast devices.
    ENABLE_LOCAL_SYNC_BACKEND = 'enable-local-sync-backend'   # Enabled the local sync backend implemented by the LoopbackServer.
    ENABLE_LOGGING = 'enable-logging'   # Controls whether console logging is enabled and optionally configures where it's routed.
    ENABLE_LONGPRESS_DRAG_SELECTION = 'enable-longpress-drag-selection'   # Enable drag manipulation of longpress-triggered text selections.
    ENABLE_LOW_END_DEVICE_MODE = 'enable-low-end-device-mode'   # Force low-end device mode when set.
    ENABLE_LOW_RES_TILING = 'enable-low-res-tiling'   # When using CPU rasterizing generate low resolution tiling. Low res tiles may be displayed during fast scrolls especially on slower devices.
    ENABLE_MACHINE_LEVEL_USER_CLOUD_POLICY = 'enable-machine-level-user-cloud-policy'   # Enables Machine level user cloud policy on Chromium build. This policy is always enabled on the branded builds.
    ENABLE_MAIN_FRAME_BEFORE_ACTIVATION = 'enable-main-frame-before-activation'   # Enables sending the next BeginMainFrame before the previous commit activates.
    ENABLE_MARKET_OPT_IN = 'enable-market-opt-in'   # Enables the marketing opt-in screen in OOBE.
    ENABLE_MEDIA_SUSPEND = 'enable-media-suspend'   # Suspend media pipeline on background tabs.
    ENABLE_NACL = 'enable-nacl'   # Runs the Native Client inside the renderer process and enables GPU plugin (internally adds lEnableGpuPlugin to the command line).
    ENABLE_NACL_DEBUG = 'enable-nacl-debug'   # Enables debugging via RSP over a socket.
    ENABLE_NACL_NONSFI_MODE = 'enable-nacl-nonsfi-mode'   # Enables Non-SFI mode, in which programs can be run without NaCl's SFI sandbox.
    ENABLE_NATIVE_GPU_MEMORY_BUFFERS = 'enable-native-gpu-memory-buffers'   # Enable native CPU-mappable GPU memory buffer support on Linux.
    ENABLE_NATURAL_SCROLL_DEFAULT = 'enable-natural-scroll-default'   # Enables natural scroll by default.
    ENABLE_NAVIGATION_TRACING = 'enable-navigation-tracing'   # Enables tracing for each navigation. It will attempt to trace each navigation for 10s, until the buffer is full, or until the next navigation. It only works if a URL was provided by --trace-upload-url.
    ENABLE_NET_BENCHMARKING = 'enable-net-benchmarking'   # Enables the network-related benchmarking extensions.
    ENABLE_NETWORK_INFORMATION_DOWNLINK_MAX = 'enable-network-information-downlink-max'   # Enables the type, downlinkMax attributes of the NetInfo API. Also, enables triggering of change attribute of the NetInfo API when there is a change in the connection type.
    ENABLE_NEW_APP_MENU_ICON = 'enable-new-app-menu-icon'   # 
    ENABLE_NTP_SEARCH_ENGINE_COUNTRY_DETECTION = 'enable-ntp-search-engine-country-detection'   # Enables using the default search engine country to show country specific popular sites on the NTP.
    ENABLE_OFFER_STORE_UNMASKED_WALLET_CARDS = 'enable-offer-store-unmasked-wallet-cards'   # Force showing the local save checkbox in the autofill dialog box for getting the full credit card number for a wallet card.
    ENABLE_OOP_RASTERIZATION = 'enable-oop-rasterization'   # Turns on out of process raster for the renderer whenever gpu raster would have been used.  Enables the chromium_raster_transport extension.
    ENABLE_OOP_RASTERIZATION_DDL = 'enable-oop-rasterization-ddl'   # Turns on skia deferred display list for out of process raster.
    ENABLE_OVERRIDE_BOOKMARKS_UI = 'enable-override-bookmarks-ui'   # Enables extensions to hide bookmarks UI elements.
    ENABLE_PARTIAL_RASTER = 'enable-partial-raster'   # Enable partial raster in the renderer.
    ENABLE_PEPPER_TESTING = 'enable-pepper-testing'   # Enables the testing interface for PPAPI.
    ENABLE_PERFETTO = 'enable-perfetto'   # Enables the perfetto tracing backend. We need a separate command line argument from the kTracingPerfettoBackend feature, because feature flags are parsed too late during startup for early startup tracing support.
    ENABLE_PIXEL_OUTPUT_IN_TESTS = 'enable-pixel-output-in-tests'   # Forces tests to produce pixel output when they normally wouldn't.
    ENABLE_PLUGIN_PLACEHOLDER_TESTING = 'enable-plugin-placeholder-testing'   # Enables testing features of the Plugin Placeholder. For internal use only.
    ENABLE_POTENTIALLY_ANNOYING_SECURITY_FEATURES = 'enable-potentially-annoying-security-features'   # Enables a number of potentially annoying security features (strict mixed content mode, powerful feature restrictions, etc.)
    ENABLE_PRECISE_MEMORY_INFO = 'enable-precise-memory-info'   # Make the values returned to window.performance.memory more granular and more up to date in shared worker. Without this flag, the memory information is still available, but it is bucketized and updated less frequently. This flag also applys to workers.
    ENABLE_PREFER_COMPOSITING_TO_LCD_TEXT = 'enable-prefer-compositing-to-lcd-text'   # Enable the creation of compositing layers when it would prevent LCD text.
    ENABLE_PRINT_BROWSER = 'enable-print-browser'   # Enables PrintBrowser mode, in which everything renders as though printed.
    ENABLE_PROFILE_SHORTCUT_MANAGER = 'enable-profile-shortcut-manager'   # Force-enables the profile shortcut manager. This is needed for tests since they use a custom-user-data-dir which disables this.
    ENABLE_REACHED_CODE_PROFILER = 'enable-reached-code-profiler'   # Enables the reached code profiler that samples all threads in all processes to determine which functions are almost never executed.
    ENABLE_REPORTING = 'enable-reporting'   # Enables metrics and crash reporting.
    ENABLE_REQUEST_TABLET_SITE = 'enable-request-tablet-site'   # Enables request of tablet site (via user agent override).
    ENABLE_RGBA_4444_TEXTURES = 'enable-rgba-4444-textures'   # Enables RGBA_4444 textures.
    ENABLE_SANDBOX_LOGGING = 'enable-sandbox-logging'   # Cause the OS X sandbox write to syslog every time an access to a resource is denied by the sandbox.
    ENABLE_SERVICE_BINARY_LAUNCHER = 'enable-service-binary-launcher'   # If true the ServiceProcessLauncher is used to launch services. This allows for service binaries to be loaded rather than using the utility process. This is only useful for tests.
    ENABLE_SERVICE_MANAGER_TRACING = 'enable-service-manager-tracing'   # Enable the tracing service.
    ENABLE_SGI_VIDEO_SYNC = 'enable-sgi-video-sync'   # Enable use of the SGI_video_sync extension, which can have driver/sandbox/window manager compatibility issues.
    ENABLE_SKIA_BENCHMARKING = 'enable-skia-benchmarking'   # Enables the Skia benchmarking extension
    ENABLE_SMOOTH_SCROLLING = 'enable-smooth-scrolling'   # On platforms that support it, enables smooth scroll animation.
    ENABLE_SPATIAL_NAVIGATION = 'enable-spatial-navigation'   # Enable spatial navigation
    ENABLE_SPDY_PROXY_AUTH = 'enable-spdy-proxy-auth'   # Enable the data reduction proxy.
    ENABLE_SPEECH_DISPATCHER = 'enable-speech-dispatcher'   # Allows sending text-to-speech requests to speech-dispatcher, a common Linux speech service. Because it's buggy, the user must explicitly enable it so that visiting a random webpage can't cause instability.
    ENABLE_SPOTLIGHT_ACTIONS = 'enable-spotlight-actions'   # Enables the Spotlight actions.
    ENABLE_STATS_COLLECTION_BINDINGS = 'enable-stats-collection-bindings'   # Specifies if the |StatsCollectionController| needs to be bound in the renderer. This binding happens on per-frame basis and hence can potentially be a performance bottleneck. One should only enable it when running a test that needs to access the provided statistics.
    ENABLE_STRICT_MIXED_CONTENT_CHECKING = 'enable-strict-mixed-content-checking'   # Blocks all insecure requests from secure contexts, and prevents the user from overriding that decision.
    ENABLE_STRICT_POWERFUL_FEATURE_RESTRICTIONS = 'enable-strict-powerful-feature-restrictions'   # Blocks insecure usage of a number of powerful features (device orientation, for example) that we haven't yet deprecated for the web at large.
    ENABLE_SUBRESOURCE_REDIRECT = 'enable-subresource-redirect'   # Feature flag to enable HTTPS subresource internal redirects to compressed versions.
    ENABLE_SWAP_BUFFERS_WITH_BOUNDS = 'enable-swap-buffers-with-bounds'   # Enables SwapBuffersWithBounds if it is supported.
    ENABLE_THIRD_PARTY_KEYBOARD_WORKAROUND = 'enable-third-party-keyboard-workaround'   # Enables the 3rd party keyboard omnibox workaround.
    ENABLE_THREAD_INSTRUCTION_COUNT = 'enable-thread-instruction-count'   # Controls whether or not retired instruction counts are surfaced for threads in trace events on Linux.  This flag requires the BPF sandbox to be disabled.
    ENABLE_THREADED_COMPOSITING = 'enable-threaded-compositing'   # Enabled threaded compositing for web tests.
    ENABLE_THREADED_TEXTURE_MAILBOXES = 'enable-threaded-texture-mailboxes'   # Simulates shared textures when share groups are not available. Not available everywhere.
    ENABLE_TOP_DRAG_GESTURE = 'enable-top-drag-gesture'   # Whether to enable detection and dispatch of a 'drag from the top' gesture.
    ENABLE_TOUCH_CALIBRATION_SETTING = 'enable-touch-calibration-setting'   # Enables the touch calibration option in MD settings UI for valid touch displays.
    ENABLE_TOUCH_DRAG_DROP = 'enable-touch-drag-drop'   # Enables touch event based drag and drop.
    ENABLE_TOUCHPAD_THREE_FINGER_CLICK = 'enable-touchpad-three-finger-click'   # Enables touchpad three-finger-click as middle button.
    ENABLE_TOUCHVIEW = 'enable-touchview'   # If the flag is present, it indicates 1) the device has accelerometer and 2) the device is a convertible device or a tablet device (thus is capable of entering tablet mode). If this flag is not set, then the device is not capable of entering tablet mode. For example, Samus has accelerometer, but is not a covertible or tablet, thus doesn't have this flag set, thus can't enter tablet mode.
    ENABLE_TRACE_APP_SOURCE = 'enable-trace-app-source'   # Pass launch source to platform apps.
    ENABLE_TRACING = 'enable-tracing'   # Enable tracing during the execution of browser tests.
    ENABLE_TRACING_OUTPUT = 'enable-tracing-output'   # The filename to write the output of the test tracing to.
    ENABLE_UI_DEVTOOLS = 'enable-ui-devtools'   # Enables DevTools server for UI (mus, ash, etc). Value should be the port the server is started on. Default port is 9223.
    ENABLE_UNSAFE_WEBGPU = 'enable-unsafe-webgpu'   # 
    ENABLE_USE_ZOOM_FOR_DSF = 'enable-use-zoom-for-dsf'   # Enable the mode that uses zooming to implment device scale factor behavior.
    ENABLE_USER_METRICS = 'enable-user-metrics'   # Enable user metrics from within the installer.
    ENABLE_USERMEDIA_SCREEN_CAPTURING = 'enable-usermedia-screen-capturing'   # Enable screen capturing support for MediaStream API.
    ENABLE_VIEWPORT = 'enable-viewport'   # Enables the use of the @viewport CSS rule, which allows pages to control aspects of their own layout. This also turns on touch-screen pinch gestures.
    ENABLE_VIRTUAL_KEYBOARD = 'enable-virtual-keyboard'   # 
    ENABLE_VIZ_DEVTOOLS = 'enable-viz-devtools'   # Enables inspecting Viz Display Compositor objects. Default port is 9229. For local inspection use chrome://inspect#other
    ENABLE_VIZ_HIT_TEST_DEBUG = 'enable-viz-hit-test-debug'   # Enables hit-test debug logging.
    ENABLE_VTUNE_SUPPORT = 'enable-vtune-support'   # Enable the Vtune profiler support.
    ENABLE_WAYLAND_IME = 'enable-wayland-ime'   # Try to enable wayland input method editor.
    ENABLE_WAYLAND_SERVER = 'enable-wayland-server'   # Enable the wayland server.
    ENABLE_WEB_AUTHENTICATION_TESTING_API = 'enable-web-authentication-testing-api'   # Enable the Web Authentication Testing API. https://w3c.github.io/webauthn
    ENABLE_WEB_BLUETOOTH_SCANNING = 'enable-web-bluetooth-scanning'   # Enable Web Bluetooth Scanning This switch enables Web Bluetooth Scanning without any permission prompt for testing.
    ENABLE_WEBGL_DRAFT_EXTENSIONS = 'enable-webgl-draft-extensions'   # Enables WebGL extensions not yet approved by the community.
    ENABLE_WEBGL_IMAGE_CHROMIUM = 'enable-webgl-image-chromium'   # Enables WebGL rendering into a scanout buffer for overlay support.
    ENABLE_WEBGL_SWAP_CHAIN = 'enable-webgl-swap-chain'   # Enables WebGL overlays for Windows.
    ENABLE_WEBGL2_COMPUTE_CONTEXT = 'enable-webgl2-compute-context'   # Enable WebGL2 Compute context.
    ENABLE_WEBRTC_SRTP_AES_GCM = 'enable-webrtc-srtp-aes-gcm'   # Enables negotiation of GCM cipher suites from RFC 7714 for SRTP in WebRTC. See https://tools.ietf.org/html/rfc7714 for further information.
    ENABLE_WEBRTC_SRTP_ENCRYPTED_HEADERS = 'enable-webrtc-srtp-encrypted-headers'   # Enables negotiation of encrypted header extensions from RFC 6904 for SRTP in WebRTC. See https://tools.ietf.org/html/rfc6904 for further information. TODO(https://crbug.com/954201): Remove this.
    ENABLE_WEBRTC_STUN_ORIGIN = 'enable-webrtc-stun-origin'   # Enables Origin header in Stun messages for WebRTC.
    ENABLE_WEBVR = 'enable-webvr'   # Enables interaction with virtual reality devices.
    ENABLE_WIN7_WEBRTC_HW_H264_DECODING = 'enable-win7-webrtc-hw-h264-decoding'   # Enables H264 HW decode acceleration for WebRtc on Win 7.
    ENABLE_ZERO_COPY = 'enable-zero-copy'   # Enable rasterizer that writes directly to GPU memory associated with tiles.
    ENABLED = 'enabled'   # 
    ENCODE_BINARY = 'encode-binary'   # Encode binary web test results (images, audio) using base64.
    ENFORCE = 'enforce'   # 
    ENFORCE_GL_MINIMUMS = 'enforce-gl-minimums'   # Enforce GL minimums.
    ENFORCE_VULKAN_PROTECTED_MEMORY = 'enforce-vulkan-protected-memory'   # Forces to use protected memory for vulkan compositing.
    ENFORCE_WEBRTC_IP_PERMISSION_CHECK = 'enforce-webrtc-ip-permission-check'   # Enforce IP Permission check. TODO(guoweis): Remove this once the feature is not under finch and becomes the default.
    ENFORCE_STRICT = 'enforce_strict'   # 
    ENSURE_FORCED_COLOR_PROFILE = 'ensure-forced-color-profile'   # Crash the browser at startup if the display's color profile does not match the forced color profile. This is necessary on Mac because Chrome's pixel output is always subject to the color conversion performed by the operating system. On all other platforms, this is a no-op.
    ENTERPRISE_DISABLE_ARC = 'enterprise-disable-arc'   # Disables ARC for managed accounts.
    ENTERPRISE_DISABLE_LICENSE_TYPE_SELECTION = 'enterprise-disable-license-type-selection'   # Disable license type selection by user during enrollment.
    ENTERPRISE_ENABLE_FORCED_RE_ENROLLMENT = 'enterprise-enable-forced-re-enrollment'   # Whether to enable forced enterprise re-enrollment.
    ENTERPRISE_ENABLE_INITIAL_ENROLLMENT = 'enterprise-enable-initial-enrollment'   # Whether to enable initial enterprise enrollment.
    ENTERPRISE_ENABLE_ZERO_TOUCH_ENROLLMENT = 'enterprise-enable-zero-touch-enrollment'   # Enables the zero-touch enterprise enrollment flow.
    ENTERPRISE_ENROLLMENT_INITIAL_MODULUS = 'enterprise-enrollment-initial-modulus'   # Power of the power-of-2 initial modulus that will be used by the auto-enrollment client. E.g. "4" means the modulus will be 2^4 = 16.
    ENTERPRISE_ENROLLMENT_MODULUS_LIMIT = 'enterprise-enrollment-modulus-limit'   # Power of the power-of-2 maximum modulus that will be used by the auto-enrollment client.
    ERROR_CONSOLE = 'error-console'   # Allows the ErrorConsole to collect runtime and manifest errors, and display them in the chrome:extensions page.
    EVALUATE_TYPE = 'evaluate-type'   # 
    EVALUATE_CAPABILITY = 'evaluate_capability'   # 
    EXPLICITLY_ALLOWED_PORTS = 'explicitly-allowed-ports'   # Allows overriding the list of restricted ports by passing a comma-separated list of port numbers.
    EXPOSE_INTERNALS_FOR_TESTING = 'expose-internals-for-testing'   # Exposes the window.internals object to JavaScript for interactive development and debugging of web tests that rely on it.
    EXTENSION_CONTENT_VERIFICATION = 'extension-content-verification'   # Name of the command line flag to force content verification to be on in one of various modes.
    EXTENSION_PROCESS = 'extension-process'   # Marks a renderer as extension process.
    EXTENSION_UPDATER_TEST_REQUEST = 'extension-updater-test-request'   # 
    EXTENSIONS_INSTALL_VERIFICATION = 'extensions-install-verification'   # Turns on extension install verification if it would not otherwise have been turned on.
    EXTENSIONS_NOT_WEBSTORE = 'extensions-not-webstore'   # Specifies a comma-separated list of extension ids that should be forced to be treated as not from the webstore when doing install verification.
    EXTENSIONS_ON_CHROME_URLS = 'extensions-on-chrome-urls'   # Enables extensions running scripts on chrome:// URLs. Extensions still need to explicitly request access to chrome:// URLs in the manifest.
    EXTERNAL_METRICS_COLLECTION_INTERVAL = 'external-metrics-collection-interval'   # Interval in seconds between Chrome reading external metrics from /var/lib/metrics/uma-events.
    EXTRA_SEARCH_QUERY_PARAMS = 'extra-search-query-params'   # Additional query params to insert in the search and instant URLs.  Useful for testing.
    FAIL_AUDIO_STREAM_CREATION = 'fail-audio-stream-creation'   # Causes the AudioManager to fail creating audio streams. Used when testing various failure cases.
    FAKE_DRIVEFS_LAUNCHER_CHROOT_PATH = 'fake-drivefs-launcher-chroot-path'   # An absolute path to the chroot hosting the DriveFS to use. This is only used when running on Linux, i.e. when IsRunningOnChromeOS() returns false.
    FAKE_DRIVEFS_LAUNCHER_SOCKET_PATH = 'fake-drivefs-launcher-socket-path'   # A relative path to socket to communicat with the fake DriveFS launcher within the chroot specified by kFakeDriveFsLauncherChrootPath. This is only used when running on Linux, i.e. when IsRunningOnChromeOS() returns false.
    FAKE_OOBE_CONFIGURATION_FILE = 'fake-oobe-configuration-file'   # Path to a OOBE configuration JSON file (used by FakeOobeConfigurationClient).
    FAKE_VARIATIONS_CHANNEL = 'fake-variations-channel'   # Fakes the channel of the browser for purposes of Variations filtering. This is to be used for testing only. Possible values are "stable", "beta", "dev" and "canary". This works for official builds as well.
    FALSE = 'false'   # Value indicating whether flag from command line switch is false.
    FAST_START = 'fast-start'   # If this flag is present then this command line is being delegated to an already running chrome process via the fast path, ie: before chrome.dll is loaded. It is useful to tell the difference for tracking purposes.
    FEEDBACK_SERVER = 'feedback-server'   # Alternative feedback server to use when submitting user feedback
    FIELD_TRIAL_HANDLE = 'field-trial-handle'   # Handle to the shared memory segment containing field trial state that is to be shared between processes. The argument to this switch is the handle id (pointer on Windows) as a string, followed by a comma, then the size of the shared memory segment as a string.
    FILE_URL_PATH_ALIAS = 'file-url-path-alias'   # Define an alias root directory which is replaced with the replacement string in file URLs. The format is "/alias=/replacement", which would turn file:///alias/some/path.html into file:///replacement/some/path.html.
    FILE_CHOOSER = 'file_chooser'   # 
    FIRST_EXEC_AFTER_BOOT = 'first-exec-after-boot'   # Passed to Chrome the first time that it's run after the system boots. Not passed on restart after sign out.
    FLAG_SWITCHES_BEGIN = 'flag-switches-begin'   # These two flags are added around the switches about:flags adds to the command line. This is useful to see which switches were added by about:flags on about:version. They don't have any effect.
    FLAG_SWITCHES_END = 'flag-switches-end'   # 
    FONT_CACHE_SHARED_HANDLE = 'font-cache-shared-handle'   # DirectWrite FontCache is shared by browser to renderers using shared memory. This switch allows us to pass the shared memory handle to the renderer.
    FONT_RENDER_HINTING = 'font-render-hinting'   # Sets font render hinting when running headless, affects Skia rendering and whether glyph subpixel positioning is enabled. Possible values: none|slight|medium|full|max. Default: full.
    FORCE_ANDROID_APP_MODE = 'force-android-app-mode'   # Forces Android application mode. This hides certain system UI elements and forces the app to be installed if it hasn't been already.
    FORCE_APP_MODE = 'force-app-mode'   # Forces application mode. This hides certain system UI elements and forces the app to be installed if it hasn't been already.
    FORCE_CAPTION_STYLE = 'force-caption-style'   # Forces the caption style for WebVTT captions.
    FORCE_CERT_VERIFIER_BUILTIN = 'force-cert-verifier-builtin'   # Forces Chrome to use CertVerifyProcBuiltin for verification of server certificates, ignoring the status of net::features::kCertVerifierBuiltinFeature.
    FORCE_COLOR_PROFILE = 'force-color-profile'   # Force all monitors to be treated as though they have the specified color profile. Accepted values are "srgb" and "generic-rgb" (currently used by Mac layout tests) and "color-spin-gamma24" (used by layout tests).
    FORCE_DARK_MODE = 'force-dark-mode'   # Forces dark mode in UI for platforms that support it.
    FORCE_DEV_MODE_HIGHLIGHTING = 'force-dev-mode-highlighting'   # Whether to force developer mode extensions highlighting.
    FORCE_DEVICE_SCALE_FACTOR = 'force-device-scale-factor'   # Overrides the device scale factor for the browser UI and the contents.
    FORCE_DEVTOOLS_AVAILABLE = 'force-devtools-available'   # Forces developer tools availability, no matter what values the enterprise policies DeveloperToolsDisabled and DeveloperToolsAvailability are set to.
    FORCE_EFFECTIVE_CONNECTION_TYPE = 'force-effective-connection-type'   # Forces Network Quality Estimator (NQE) to return a specific effective connection type.
    FORCE_EMPTY_CORB_ALLOWLIST = 'force-empty-corb-allowlist'   # Whether |extensions_features::kBypassCorbAllowlistParamName| should always be empty (i.e. ignoring hardcoded allowlist and the field trial param).  This switch is useful for manually verifying if an extension would continue to work fine after removing it from the allowlist.
    FORCE_ENABLE_LITE_PAGES = 'force-enable-lite-pages'   # Force enable all available previews on every page load.
    FORCE_ENABLE_METRICS_REPORTING = 'force-enable-metrics-reporting'   # Forces metrics reporting to be enabled.
    FORCE_ENABLE_NIGHT_MODE = 'force-enable-night-mode'   # Forces the night mode to be enabled.
    FORCE_ENABLE_STYLUS_TOOLS = 'force-enable-stylus-tools'   # Enables the stylus tools next to the status area.
    FORCE_FIELDTRIAL_PARAMS = 'force-fieldtrial-params'   # This option can be used to force parameters of field trials when testing changes locally. The argument is a param list of (key, value) pairs prefixed by an associated (trial, group) pair. You specify the param list for multiple (trial, group) pairs with a comma separator. Example: "Trial1.Group1:k1/v1/k2/v2,Trial2.Group2:k3/v3/k4/v4" Trial names, groups names, parameter names, and value should all be URL escaped for all non-alphanumeric characters.
    FORCE_FIELDTRIALS = 'force-fieldtrials'   # This option can be used to force field trials when testing changes locally. The argument is a list of name and value pairs, separated by slashes. If a trial name is prefixed with an asterisk, that trial will start activated. For example, the following argument defines two trials, with the second one activated: "GoogleNow/Enable/*MaterialDesignNTP/Default/" This option can also be used by the browser process to send the list of trials to a non-browser process, using the same format. See FieldTrialList::CreateTrialsFromString() in field_trial.h for details.
    FORCE_FIRST_RUN = 'force-first-run'   # Displays the First Run experience when the browser is started, regardless of whether or not it's actually the First Run (this overrides kNoFirstRun).
    FORCE_FIRST_RUN_UI = 'force-first-run-ui'   # Forces first-run UI to be shown for every login.
    FORCE_GPU_MEM_AVAILABLE_MB = 'force-gpu-mem-available-mb'   # Sets the total amount of memory that may be allocated for GPU resources
    FORCE_GPU_RASTERIZATION = 'force-gpu-rasterization'   # Always use the Skia GPU backend for drawing layer tiles. Only valid with GPU accelerated compositing + impl-side painting. Overrides the kEnableGpuRasterization flag.
    FORCE_HAPPINESS_TRACKING_SYSTEM = 'force-happiness-tracking-system'   # Force enables the Happiness Tracking System for the device. This ignores user profile check and time limits and shows the notification every time for any type of user. Should be used only for testing.
    FORCE_HIGH_CONTRAST = 'force-high-contrast'   # Forces high-contrast mode in native UI drawing, regardless of system settings. Note that this has limited effect on Windows: only Aura colors will be switched to high contrast, not other system colors.
    FORCE_LOAD_EASY_UNLOCK_APP_IN_TESTS = 'force-load-easy-unlock-app-in-tests'   # Force easy unlock app loading in test. TODO(xiyuan): Remove this when app could be bundled with Chrome.
    FORCE_LOGIN_MANAGER_IN_TESTS = 'force-login-manager-in-tests'   # Usually in browser tests the usual login manager bringup is skipped so that tests can change how it's brought up. This flag disables that.
    FORCE_LOGS_UPLOAD_FAILURE = 'force-logs-upload-failure'   # Force a logs upload failure to help test the logs upload retry.
    FORCE_MEDIA_RESOLUTION_HEIGHT = 'force-media-resolution-height'   # When present overrides the screen resolution used by CanDisplayType API, instead of using the values obtained from avsettings.
    FORCE_MEDIA_RESOLUTION_WIDTH = 'force-media-resolution-width'   # 
    FORCE_OVERLAY_FULLSCREEN_VIDEO = 'force-overlay-fullscreen-video'   # Forces use of hardware overlay for fullscreen video playback. Useful for testing the Android overlay fullscreen functionality on other platforms.
    FORCE_PNACL_SUBZERO = 'force-pnacl-subzero'   # Force use of the Subzero as the PNaCl translator instead of LLC.
    FORCE_PREFERS_REDUCED_MOTION = 'force-prefers-reduced-motion'   # Forces whether the user desires reduced motion, regardless of system settings.
    FORCE_PRESENTATION_RECEIVER_FOR_TESTING = 'force-presentation-receiver-for-testing'   # This forces pages to be loaded as presentation receivers.  Useful for testing behavior specific to presentation receivers. Spec: https://www.w3.org/TR/presentation-api/#interface-presentationreceiver
    FORCE_RASTER_COLOR_PROFILE = 'force-raster-color-profile'   # Force rastering to take place in the specified color profile. Accepted values are the same as for the kForceDisplayColorProfile case above.
    FORCE_RECOVERY_COMPONENT = 'force-recovery-component'   # Force the recovery component, when running an unofficial build. This only applies to the cleaner, not the reporter.
    FORCE_RENDERER_ACCESSIBILITY = 'force-renderer-accessibility'   # Force renderer accessibility to be on instead of enabling it on demand when a screen reader is detected. The disable-renderer-accessibility switch overrides this if present.
    FORCE_SELF_DELETE = 'force-self-delete'   # Force self-deletion even on non-official builds.
    FORCE_SHOW_UPDATE_MENU_BADGE = 'force-show-update-menu-badge'   # Forces the update menu badge to show.
    FORCE_STACKED_TAB_STRIP_LAYOUT = 'force-stacked-tab-strip-layout'   # Forces Chrome to use a stacked tab strip layout.
    FORCE_SYSTEM_COMPOSITOR_MODE = 'force-system-compositor-mode'   # Force system compositor mode when set.
    FORCE_TABLET_MODE = 'force-tablet-mode'   # Enables required things for the selected UI mode, regardless of whether the Chromebook is currently in the selected UI mode.
    FORCE_TABLET_POWER_BUTTON = 'force-tablet-power-button'   # If set, tablet-like power button behavior (i.e. tapping the button turns the screen off) is used even if the device is in laptop mode.
    FORCE_TEXT_DIRECTION = 'force-text-direction'   # Force the text rendering to a specific direction. Valid values are "ltr" (left-to-right) and "rtl" (right-to-left). Only tested meaningfully with RTL.
    FORCE_UI_DIRECTION = 'force-ui-direction'   # Force the UI to a specific direction. Valid values are "ltr" (left-to-right) and "rtl" (right-to-left).
    FORCE_UPDATE_MENU_TYPE = 'force-update-menu-type'   # Forces the update menu type to a specific type.
    FORCE_USE_CHROME_CAMERA = 'force-use-chrome-camera'   # Forces to use chrome camera app when camera icon is clicked.
    FORCE_VARIATION_IDS = 'force-variation-ids'   # Forces additional Chrome Variation Ids that will be sent in X-Client-Data header, specified as a 64-bit encoded list of numeric experiment ids. Ids prefixed with the character "t" will be treated as Trigger Variation Ids.
    FORCE_VIDEO_OVERLAYS = 'force-video-overlays'   # Force media player using SurfaceView instead of SurfaceTexture on Android.
    FORCE_WAVE_AUDIO = 'force-wave-audio'   # Use Windows WaveOut/In audio API even if Core Audio is supported.
    FORCE_WEBRTC_IP_HANDLING_POLICY = 'force-webrtc-ip-handling-policy'   # Override WebRTC IP handling policy to mimic the behavior when WebRTC IP handling policy is specified in Preferences.
    FULL_MEMORY_CRASH_REPORT = 'full-memory-crash-report'   # Generates full memory crash dump.
    GAIA_URL = 'gaia-url'   # 
    GAMEPAD_POLLING_INTERVAL = 'gamepad-polling-interval'   # Overrides the gamepad polling interval. Decreasing the interval improves input latency of buttons and axes but may negatively affect performance due to more CPU time spent in the input polling thread.
    GCM_CHECKIN_URL = 'gcm-checkin-url'   # Sets the checkin service endpoint that will be used for performing Google Cloud Messaging checkins.
    GCM_MCS_ENDPOINT = 'gcm-mcs-endpoint'   # Sets the Mobile Connection Server endpoint that will be used for Google Cloud Messaging.
    GCM_REGISTRATION_URL = 'gcm-registration-url'   # Sets the registration endpoint that will be used for creating new Google Cloud Messaging registrations.
    GENERATE_ACCESSIBILITY_TEST_EXPECTATIONS = 'generate-accessibility-test-expectations'   # For development / testing only. When running content_browsertests, saves output of failing accessibility tests to their expectations files in content/test/data/accessibility/, overwriting existing file content.
    GL = 'gl'   # 
    GL_NULL = 'gl-null'   # 
    GL_SHADER_INTERM_OUTPUT = 'gl-shader-interm-output'   # Include ANGLE's intermediate representation (AST) output in shader compilation info logs.
    GLES = 'gles'   # 
    GLES_NULL = 'gles-null'   # 
    GOOGLE_APIS_URL = 'google-apis-url'   # 
    GOOGLE_BASE_URL = 'google-base-url'   # Specifies an alternate URL to use for speaking to Google. Useful for testing.
    GOOGLE_DOODLE_URL = 'google-doodle-url'   # Overrides the URL used to fetch the current Google Doodle. Example: https://www.google.com/async/ddljson Testing? Try: https://www.gstatic.com/chrome/ntp/doodle_test/ddljson_android0.json https://www.gstatic.com/chrome/ntp/doodle_test/ddljson_android1.json https://www.gstatic.com/chrome/ntp/doodle_test/ddljson_android2.json https://www.gstatic.com/chrome/ntp/doodle_test/ddljson_android3.json https://www.gstatic.com/chrome/ntp/doodle_test/ddljson_android4.json
    GOOGLE_URL = 'google-url'   # 
    GPU = 'gpu'   # 
    GPU_BLACKLIST_TEST_GROUP = 'gpu-blacklist-test-group'   # Select a different set of GPU blacklist entries with the specificed test_group ID.
    GPU_DRIVER_BUG_LIST_TEST_GROUP = 'gpu-driver-bug-list-test-group'   # Enable an extra set of GPU driver bug list entries with the specified test_group ID. Note the default test group (group 0) is still active.
    GPU_LAUNCHER = 'gpu-launcher'   # Extra command line options for launching the GPU process (normally used for debugging). Use like renderer-cmd-prefix.
    GPU_NO_CONTEXT_LOST = 'gpu-no-context-lost'   # Inform Chrome that a GPU context will not be lost in power saving mode, screen saving mode, etc.  Note that this flag does not ensure that a GPU context will never be lost in any situations, say, a GPU reset.
    GPU_PREFERENCES = 'gpu-preferences'   # Passes encoded GpuPreferences to GPU process.
    GPU_PROCESS = 'gpu-process'   # Flags spied upon from other layers.
    GPU_PROGRAM_CACHE_SIZE_KB = 'gpu-program-cache-size-kb'   # Sets the maximum size of the in-memory gpu program cache, in kb
    GPU_RASTERIZATION_MSAA_SAMPLE_COUNT = 'gpu-rasterization-msaa-sample-count'   # The number of multisample antialiasing samples for GPU rasterization. Requires MSAA support on GPU to have an effect. 0 disables MSAA.
    GPU_SANDBOX_ALLOW_SYSV_SHM = 'gpu-sandbox-allow-sysv-shm'   # Allows shmat() system call in the GPU sandbox.
    GPU_SANDBOX_FAILURES_FATAL = 'gpu-sandbox-failures-fatal'   # Makes GPU sandbox failures fatal.
    GPU_SANDBOX_START_EARLY = 'gpu-sandbox-start-early'   # Starts the GPU sandbox before creating a GL context.
    GPU_STARTUP_DIALOG = 'gpu-startup-dialog'   # Causes the GPU process to display a dialog on launch.
    GPU2_STARTUP_DIALOG = 'gpu2-startup-dialog'   # Causes the second GPU process used for gpu info collection to display a dialog on launch.
    GRAPHICS_BUFFER_COUNT = 'graphics-buffer-count'   # 
    GUEST = 'guest'   # Causes the browser to launch directly in guest mode.
    GUEST_WALLPAPER_LARGE = 'guest-wallpaper-large'   # Large wallpaper to use in guest mode (as path to trusted, non-user-writable JPEG file).
    GUEST_WALLPAPER_SMALL = 'guest-wallpaper-small'   # Small wallpaper to use in guest mode (as path to trusted, non-user-writable JPEG file).
    H = 'h'   # 
    HAS_CHROMEOS_KEYBOARD = 'has-chromeos-keyboard'   # If set, the system is a Chromebook with a "standard Chrome OS keyboard", which generally means one with a Search key in the standard Caps Lock location above the Left Shift key. It should be unset for Chromebooks with both Search and Caps Lock keys (e.g. stout) and for devices like Chromeboxes that only use external keyboards.
    HAS_INTERNAL_STYLUS = 'has-internal-stylus'   # Whether this device has an internal stylus.
    HEADLESS = 'headless'   # Run in headless mode, i.e., without a UI or display server dependencies.
    HELP = 'help'   # 
    HIDE = 'hide'   # "Hide" value for kCrosRegionsMode (VPD values are hidden).
    HIDE_ANDROID_FILES_IN_FILES_APP = 'hide-android-files-in-files-app'   # If true, files in Android internal storage will be hidden in Files app.
    HIDE_ICONS = 'hide-icons'   # Makes Windows happy by allowing it to show "Enable access to this program" checkbox in Add/Remove Programs->Set Program Access and Defaults. This only shows an error box because the only way to hide Chrome is by uninstalling it.
    HIDE_SCROLLBARS = 'hide-scrollbars'   # Hide scrollbars from screenshots.
    HOMEDIR = 'homedir'   # Defines user homedir. This defaults to primary user homedir.
    HOMEPAGE = 'homepage'   # Specifies which page will be displayed in newly-opened tabs. We need this for testing purposes so that the UI tests don't depend on what comes up for http://google.com.
    HOST = 'host'   # 
    HOST_RESOLVER_RULES = 'host-resolver-rules'   # These mappings only apply to the host resolver.
    IGNORE_AUTOCOMPLETE_OFF_AUTOFILL = 'ignore-autocomplete-off-autofill'   # Ignores autocomplete="off" for Autofill data (profiles + credit cards).
    IGNORE_CERTIFICATE_ERRORS_SPKI_LIST = 'ignore-certificate-errors-spki-list'   # A set of public key hashes for which to ignore certificate-related errors.  If the certificate chain presented by the server does not validate, and one or more certificates have public key hashes that match a key from this list, the error is ignored.  The switch value must a be a comma-separated list of Base64-encoded SHA-256 SPKI Fingerprints (RFC 7469, Section 2.4).  This switch has no effect unless --user-data-dir (as defined by the content embedder) is also present.
    IGNORE_GPU_BLACKLIST = 'ignore-gpu-blacklist'   # Ignores GPU blacklist.
    IGNORE_LITEPAGE_REDIRECT_OPTIMIZATION_BLACKLIST = 'ignore-litepage-redirect-optimization-blacklist'   # Ignore the optimization hints blacklist for Lite Page Redirect previews.
    IGNORE_PREVIEWS_BLACKLIST = 'ignore-previews-blacklist'   # Ignore decisions made by PreviewsBlackList.
    IGNORE_URLFETCHER_CERT_REQUESTS = 'ignore-urlfetcher-cert-requests'   # Causes net::URLFetchers to ignore requests for SSL client certificates, causing them to attempt an unauthenticated SSL/TLS session. This is intended for use when testing various service URLs (eg: kPromoServerURL, kSbURLPrefix, kSyncServiceURL, etc).
    IGNORE_USER_PROFILE_MAPPING_FOR_TESTS = 'ignore-user-profile-mapping-for-tests'   # If true, profile selection in UserManager will always return active user's profile. TODO(nkostlyev): http://crbug.com/364604 - Get rid of this switch after we turn on multi-profile feature on ChromeOS.
    IME = 'ime'   # 
    IN_PROCESS_GPU = 'in-process-gpu'   # Run the GPU process as a thread in the browser process.
    INCOGNITO = 'incognito'   # Causes the browser to launch directly in incognito mode.
    INIT_DONE_NOTIFIER = 'init-done-notifier'   # The handle of an event to signal when the initialization of the main process is complete (including loading all DLL's). This is used by the integration test to check that forbidden modules aren't loaded outside the sandbox. If this is set, the main process will signal the event and then wait for the integration test to signal it as well before continuing. Ignored in official builds.
    INITIAL_VIRTUAL_TIME = 'initial-virtual-time'   # Start the renderer with an initial virtual time override specified in seconds since the epoch.
    INPUT = 'input'   # 
    INSTALL_AUTOGENERATED_THEME = 'install-autogenerated-theme'   # Installs an autogenerated theme based on the given RGB value. The format is "r,g,b", where r, g, b are a numeric values from 0 to 255.
    INSTALL_CHROME_APP = 'install-chrome-app'   # Causes Chrome to initiate an installation flow for the given app.
    INSTALL_SUPERVISED_USER_WHITELISTS = 'install-supervised-user-whitelists'   # A list of whitelists to install for a supervised user, for testing. The list is of the following form: <id>[:<name>],[<id>[:<name>],...]
    INSTANT_PROCESS = 'instant-process'   # Marks a renderer as an Instant process.
    INTEGRATION_TEST_TIMEOUT_MINUTES = 'integration-test-timeout-minutes'   # Set the timeout for integration tests in minutes. 0 disables the timeout entirely.
    INVALIDATION_USE_GCM_CHANNEL = 'invalidation-use-gcm-channel'   # Device invalidation service should use GCM network channel.
    IPC_CONNECTION_TIMEOUT = 'ipc-connection-timeout'   # Overrides the timeout, in seconds, that a child process waits for a connection from the browser before killing itself.
    IPC_DUMP_DIRECTORY = 'ipc-dump-directory'   # Dumps IPC messages sent from renderer processes to the browser process to the given directory. Used primarily to gather samples for IPC fuzzing.
    IPC_FUZZER_TESTCASE = 'ipc-fuzzer-testcase'   # Specifies the testcase used by the IPC fuzzer.
    ISOLATE_ORIGINS = 'isolate-origins'   # Require dedicated processes for a set of origins, specified as a comma-separated list. For example: --isolate-origins=https://www.foo.com,https://www.bar.com
    JAVASCRIPT_HARMONY = 'javascript-harmony'   # Enables experimental Harmony (ECMAScript 6) features.
    JS_FLAGS = 'js-flags'   # Specifies the flags passed to JS engine
    KEEP_ALIVE_FOR_TEST = 'keep-alive-for-test'   # Used for testing - keeps browser alive after last browser window closes.
    KERNELNEXT_RESTRICT_VMS = 'kernelnext-restrict-vms'   # If set, the Chrome settings will not expose the option to enable crostini unless the enable-experimental-kernel-vm-support flag is set in chrome://flags
    KIOSK = 'kiosk'   # Enable kiosk mode. Please note this is not Chrome OS kiosk mode.
    KIOSK_PRINTING = 'kiosk-printing'   # Enable automatically pressing the print button in print preview.
    LANG = 'lang'   # The language file that we want to try to open. Of the form language[-country] where language is the 2 letter code from ISO-639.
    LAST_LAUNCHED_APP = 'last-launched-app'   # Pass the app id information to the renderer process, to be used for logging. last-launched-app should be the app that just launched and is spawning the renderer.
    LAYER = 'layer'   # 
    LIGHT = 'light'   # 
    LITEPAGE_SERVER_PREVIEWS_HOST = 'litepage-server-previews-host'   # Override the Lite Page Preview Host.
    LITEPAGE_SERVER_SUBRESOURCE_HOST = 'litepage-server-subresource-host'   # Overrides the Lite Page Subresource host.
    LITEPAGE_REDIRECT_OVERRIDES_PAGE_HINTS = 'litepage_redirect_overrides_page_hints'   # Sets the trigger ordering of Lite Page Redirect to be higher than page hints.
    LOAD_AND_LAUNCH_APP = 'load-and-launch-app'   # Loads an app from the specified directory and launches it.
    LOAD_APPS = 'load-apps'   # Comma-separated list of paths to apps to load at startup. The first app in the list will be launched.
    LOAD_EMPTY_DLL = 'load-empty-dll'   # Attempt to load empty-dll.dll whenever this flag is set. For testing DLL loading.
    LOAD_EXTENSION = 'load-extension'   # Comma-separated list of paths to extensions to load at startup.
    LOAD_MEDIA_ROUTER_COMPONENT_EXTENSION = 'load-media-router-component-extension'   # Loads the Media Router component extension on startup.
    LOCAL_SYNC_BACKEND_DIR = 'local-sync-backend-dir'   # Specifies the local sync backend directory. The name is chosen to mimic user-data-dir etc. This flag only matters if the enable-local-sync-backend flag is present.
    LOG_BEST_EFFORT_TASKS = 'log-best-effort-tasks'   # Logs information about all tasks posted with TaskPriority::BEST_EFFORT. Use this to diagnose issues that are thought to be caused by TaskPriority::BEST_EFFORT execution fences. Note: Tasks posted to a non-BEST_EFFORT UpdateableSequencedTaskRunner whose priority is later lowered to BEST_EFFORT are not logged.
    LOG_FILE = 'log-file'   # Overrides the default file name to use for general-purpose logging (does not affect which events are logged).
    LOG_GPU_CONTROL_LIST_DECISIONS = 'log-gpu-control-list-decisions'   # Logs GPU control list decisions when enforcing blacklist rules.
    LOG_INTERFACE_CALLS_TO = 'log-interface-calls-to'   # Specifies a file to log all the interface calls of EngineRequestsImpl and CleanerEngineRequestsImpl.
    LOG_LEVEL = 'log-level'   # Sets the minimum log level. Valid values are from 0 to 3: INFO = 0, WARNING = 1, LOG_ERROR = 2, LOG_FATAL = 3.
    LOG_NET_LOG = 'log-net-log'   # Enables saving net log events to a file. If a value is given, it used as the path the the file, otherwise the file is named netlog.json and placed in the user data directory.
    LOGIN_MANAGER = 'login-manager'   # Enables Chrome-as-a-login-manager behavior.
    LOGIN_PROFILE = 'login-profile'   # Specifies the profile to use once a chromeos user is logged in. This parameter is ignored if user goes through login screen since user_id hash defines which profile directory to use. In case of browser restart within active session this parameter is used to pass user_id hash for primary user.
    LOGIN_USER = 'login-user'   # Specifies the user which is already logged in.
    LOGS_UPLOAD_RETRY_INTERVAL = 'logs-upload-retry-interval'   # Specify the time to wait between logs upload retries, in minutes.
    LOW_PRESSURE_TOUCH_FILTERING = 'low-pressure-touch-filtering'   # Tells Chrome to do filter out low pressure touches, as from a pencil. Should only be used if the driver level filtering is insufficient.
    LSO_URL = 'lso-url'   # 
    LTR = 'ltr'   # 
    MAIN_FRAME_RESIZES_ARE_ORIENTATION_CHANGES = 'main-frame-resizes-are-orientation-changes'   # Resizes of the main frame are caused by changing between landscape and portrait mode (i.e. Android) so the page should be rescaled to fit.
    MAKE_CHROME_DEFAULT = 'make-chrome-default'   # Indicates whether Chrome should be set as the default browser during installation.
    MAKE_DEFAULT_BROWSER = 'make-default-browser'   # Makes Chrome default browser
    MANAGED_USER_ID = 'managed-user-id'   # Sets the supervised user ID for any loaded or newly created profile to the given value. Pass an empty string to mark the profile as non-supervised. Used for testing.
    MANGLE_LOCALIZED_STRINGS = 'mangle-localized-strings'   # Transform localized strings to be longer, with beginning and end markers to make truncation visually apparent.
    MANUAL = 'manual'   # 
    MARKET_URL_FOR_TESTING = 'market-url-for-testing'   # Sets the market URL for Chrome for use in testing.
    MAX_ACTIVE_WEBGL_CONTEXTS = 'max-active-webgl-contexts'   # Allows user to override maximum number of active WebGL contexts per renderer process.
    MAX_DECODED_IMAGE_SIZE_MB = 'max-decoded-image-size-mb'   # Sets the maximium decoded image size limitation.
    MAX_FILE_SIZE = 'max-file-size'   # Limit the size of files the scanning engine is allowed to open.
    MAX_GUM_FPS = 'max-gum-fps'   # Override the maximum framerate as can be specified in calls to getUserMedia. This flag expects a value.  Example: --max-gum-fps=17.5
    MAX_OUTPUT_VOLUME_DBA1M = 'max-output-volume-dba1m'   # Calibrated max output volume dBa for voice content at 1 meter, if known.
    MAX_UNTILED_LAYER_HEIGHT = 'max-untiled-layer-height'   # Sets the width and height above which a composited layer will get tiled.
    MAX_UNTILED_LAYER_WIDTH = 'max-untiled-layer-width'   # 
    MEM_PRESSURE_SYSTEM_RESERVED_KB = 'mem-pressure-system-reserved-kb'   # Some platforms typically have very little 'free' memory, but plenty is available in buffers+cached.  For such platforms, configure this amount as the portion of buffers+cached memory that should be treated as unavailable.  If this switch is not used, a simple pressure heuristic based purely on free memory will be used.
    MEMLOG = 'memlog'   # 
    MEMLOG_SAMPLING_RATE = 'memlog-sampling-rate'   # 
    MEMLOG_STACK_MODE = 'memlog-stack-mode'   # 
    MEMORY_PRESSURE_OFF = 'memory-pressure-off'   # 
    MESSAGE_LOOP_TYPE_UI = 'message-loop-type-ui'   # Indicates the utility process should run with a message loop type of UI.
    METRICS_CLIENT_ID = 'metrics-client-id'   # This is how the metrics client ID is passed from the browser process to its children. With Crashpad, the metrics client ID is distinct from the crash client ID.
    METRICS_RECORDING_ONLY = 'metrics-recording-only'   # Enables the recording of metrics reports but disables reporting. In contrast to kDisableMetrics, this executes all the code that a normal client would use for reporting, except the report is dropped rather than sent to the server. This is useful for finding issues in the metrics code during UI and performance tests.
    METRICS_UPLOAD_INTERVAL = 'metrics-upload-interval'   # Override the standard time interval between each metrics report upload for UMA and UKM. It is useful to set to a short interval for debugging. Unit in seconds. (The default is 1800 seconds on desktop).
    MINIMAL = 'minimal'   # 
    MIRROR = 'mirror'   # Values for the kAccountConsistency flag.
    MIXED = 'mixed'   # 
    MIXER_SERVICE_ENDPOINT = 'mixer-service-endpoint'   # Endpoint that the mixer service listens on. On Linux/Android, this is a path for a UNIX domain socket (default is /tmp/mixer-service). On other platforms, this is a TCP port to listen on (on localhost) (default 12854).
    MIXER_SOURCE_AUDIO_READY_THRESHOLD_MS = 'mixer-source-audio-ready-threshold-ms'   # Specify the start threshold frames for audio output when using our mixer. This is mostly used to override the default value to a larger value, for platforms that can't handle the default start threshold without running into audio underruns.
    MIXER_SOURCE_INPUT_QUEUE_MS = 'mixer-source-input-queue-ms'   # Specify the buffer size for audio output when using our mixer. This is mostly used to override the default value to a larger value, for platforms that can't handle an audio buffer so small without running into audio underruns.
    MOCK = 'mock'   # 
    MOCK_CERT_VERIFIER_DEFAULT_RESULT_FOR_TESTING = 'mock-cert-verifier-default-result-for-testing'   # Set the default result for MockCertVerifier. This only works in test code.
    MOJO_LOCAL_STORAGE = 'mojo-local-storage'   # Use a Mojo-based LocalStorage implementation.
    MOJO_PIPE_TOKEN = 'mojo-pipe-token'   # 
    MONITORING_DESTINATION_ID = 'monitoring-destination-id'   # Allows setting a different destination ID for connection-monitoring GCM messages. Useful when running against a non-prod management server.
    MSE_AUDIO_BUFFER_SIZE_LIMIT_MB = 'mse-audio-buffer-size-limit-mb'   # Allows explicitly specifying MSE audio/video buffer sizes as megabytes. Default values are 150M for video and 12M for audio.
    MSE_VIDEO_BUFFER_SIZE_LIMIT_MB = 'mse-video-buffer-size-limit-mb'   # 
    MUTE_AUDIO = 'mute-audio'   # Mutes audio sent to the audio device so it is not audible during automated testing.
    NACL_BROKER = 'nacl-broker'   # Value for --type that causes the process to run as a NativeClient broker (used for launching NaCl loader processes on 64-bit Windows).
    NACL_DANGEROUS_NO_SANDBOX_NONSFI = 'nacl-dangerous-no-sandbox-nonsfi'   # Disable sandbox even for non SFI mode. This is particularly unsafe as non SFI NaCl heavily relies on the seccomp sandbox.
    NACL_DEBUG_MASK = 'nacl-debug-mask'   # Uses NaCl manifest URL to choose whether NaCl program will be debugged by debug stub. Switch value format: [!]pattern1,pattern2,...,patternN. Each pattern uses the same syntax as patterns in Chrome extension manifest. The only difference is that * scheme matches all schemes instead of matching only http and https. If the value doesn't start with !, a program will be debugged if manifest URL matches any pattern. If the value starts with !, a program will be debugged if manifest URL does not match any pattern.
    NACL_GDB = 'nacl-gdb'   # Native Client GDB debugger that will be launched automatically when needed.
    NACL_GDB_SCRIPT = 'nacl-gdb-script'   # GDB script to pass to the nacl-gdb debugger at startup.
    NACL_LOADER = 'nacl-loader'   # 
    NACL_LOADER_NONSFI = 'nacl-loader-nonsfi'   # Value for --type that causes the process to run as a NativeClient loader for non SFI mode.
    NATIVE = 'native'   # 
    NATIVE_MESSAGING_CONNECT_EXTENSION = 'native-messaging-connect-extension'   # Requests a native messaging connection be established between the extension with ID specified by this switch and the native messaging host named by the kNativeMessagingConnectHost switch.
    NATIVE_MESSAGING_CONNECT_HOST = 'native-messaging-connect-host'   # Requests a native messaging connection be established between the native messaging host named by this switch and the extension with ID specified by kNativeMessagingConnectExtension.
    NATIVE_WITH_THREAD_NAMES = 'native-with-thread-names'   # 
    NEED_ARC_MIGRATION_POLICY_CHECK = 'need-arc-migration-policy-check'   # If present, the device needs to check the policy to see if the migration to ext4 for ARC is allowed. It should be present only on devices that have been initially issued with ecrypfs encryption and have ARC (N+) available. For the devices in other categories this flag must be missing.
    NET_LOG_CAPTURE_MODE = 'net-log-capture-mode'   # Sets the granularity of events to capture in the network log. The mode can be set to one of the following values: "Default" "IncludeSensitive" "Everything"  See the enums of the corresponding name in net_log_capture_mode.h for a description of their meanings.
    NETIFS_TO_IGNORE = 'netifs-to-ignore'   # List of network interfaces to ignore. Ignored interfaces will not be used for network connectivity.
    NETWORK = 'network'   # 
    NETWORK_COUNTRY_ISO = 'network-country-iso'   # The telephony region (ISO country code) to use in phone number detection.
    NETWORK_QUIET_TIMEOUT = 'network-quiet-timeout'   # Sets the timeout seconds of the network-quiet timers in IdlenessDetector. Used by embedders who want to change the timeout time in order to run web contents on various embedded devices and changeable network bandwidths in different regions. For example, it's useful when using FirstMeaningfulPaint signal to dismiss a splash screen.
    NEW_NET_ERROR_PAGE_UI = 'new-net-error-page-ui'   # Enables new UI for net-error page.
    NEW_WINDOW = 'new-window'   # Launches URL in new browser window.
    NO_CRASH_UPLOAD = 'no-crash-upload'   # Prevent the crash client from uploading crash reports.
    NO_DEFAULT_BROWSER_CHECK = 'no-default-browser-check'   # Disables the default browser check. Useful for UI/browser tests where we want to avoid having the default browser info-bar displayed.
    NO_EXPERIMENTS = 'no-experiments'   # Disables all experiments set on about:flags. Does not disable about:flags itself. Useful if an experiment makes chrome crash at startup: One can start chrome with --no-experiments, disable the problematic lab at about:flags and then restart chrome without this switch again.
    NO_FIRST_RUN = 'no-first-run'   # Skip First Run tasks, whether or not it's actually the First Run. Overridden by kForceFirstRun. This does not drop the First Run sentinel and thus doesn't prevent first run from occuring the next time chrome is launched without this flag.
    NO_MANAGED_USER_ACKNOWLEDGMENT_CHECK = 'no-managed-user-acknowledgment-check'   # Disables checking whether we received an acknowledgment when registering a supervised user. Also disables the timeout during registration that waits for the ack. Useful when debugging against a server that does not support notifications.
    NO_NETWORK_PROFILE_WARNING = 'no-network-profile-warning'   # Whether or not the browser should warn if the profile is on a network share. This flag is only relevant for Windows currently.
    NO_PINGS = 'no-pings'   # Don't send hyperlink auditing pings
    NO_PROXY_SERVER = 'no-proxy-server'   # Don't use a proxy server, always make direct connections. Overrides any other proxy server flags that are passed.
    NO_RECOVERY_COMPONENT = 'no-recovery-component'   # Prevent downloading and running the recovery component.
    NO_REFERRERS = 'no-referrers'   # Don't send HTTP-Referer headers.
    NO_REPORT_UPLOAD = 'no-report-upload'   # Prevent the logging service from uploading logs and reports. WARNING: this switch is used by internal test systems. Be careful when making changes.
    NO_SANDBOX = 'no-sandbox'   # Disables the sandbox for all process types that are normally sandboxed.
    NO_SANDBOX_AND_ELEVATED = 'no-sandbox-and-elevated'   # Disables the sandbox and gives the process elevated privileges.
    NO_SELF_DELETE = 'no-self-delete'   # Prevent the executable from deleting itself after running.
    NO_SERVICE_AUTORUN = 'no-service-autorun'   # Disables the service process from adding itself as an autorun process. This does not delete existing autorun registrations, it just prevents the service from registering a new one.
    NO_STARTUP_WINDOW = 'no-startup-window'   # Does not automatically open a browser window on startup (used when launching Chrome for the purpose of hosting background apps).
    NO_USER_GESTURE_REQUIRED = 'no-user-gesture-required'   # Autoplay policy that does not require any user gesture.
    NO_V8_UNTRUSTED_CODE_MITIGATIONS = 'no-v8-untrusted-code-mitigations'   # Disables V8 mitigations for executing untrusted code.
    NO_WIFI = 'no-wifi'   # Disable features that require WiFi management.
    NO_ZYGOTE = 'no-zygote'   # Disables the use of a zygote process for forking child processes. Instead, child processes will be forked and exec'd directly. Note that --no-sandbox should also be used together with this flag because the sandbox needs the zygote to work.
    NOERRDIALOGS = 'noerrdialogs'   # Suppresses all error dialogs when present.
    NONE = 'none'   # Must be in sync with "sandbox_type" values as used in service manager's manifest.json catalog files.
    NONE_AND_ELEVATED = 'none_and_elevated'   # 
    NOTE_TAKING_APP_IDS = 'note-taking-app-ids'   # An optional comma-separated list of IDs of apps that can be used to take notes. If unset, a hardcoded list is used instead.
    NOTIFICATION_INLINE_REPLY = 'notification-inline-reply'   # Used in combination with kNotificationLaunchId to specify the inline reply entered in the toast in the Windows Action Center.
    NOTIFICATION_LAUNCH_ID = 'notification-launch-id'   # Used for launching Chrome when a toast displayed in the Windows Action Center has been activated. Should contain the launch ID encoded by Chrome.
    NTP_SNIPPETS_ADD_INCOMPLETE = 'ntp-snippets-add-incomplete'   # If this flag is set, we will add downloaded snippets that are missing some critical data to the list.
    NULL = 'null'   # 
    NUM_RASTER_THREADS = 'num-raster-threads'   # Number of worker threads used to rasterize content.
    OAUTH_ACCOUNT_MANAGER_URL = 'oauth-account-manager-url'   # 
    OAUTH2_CLIENT_ID = 'oauth2-client-id'   # 
    OAUTH2_CLIENT_SECRET = 'oauth2-client-secret'   # 
    OFF = 'off'   # 
    ON = 'on'   # 
    ON_THE_FLY_MHTML_HASH_COMPUTATION = 'on-the-fly-mhtml-hash-computation'   # Calculate the hash of an MHTML file as it is being saved. The browser process will write the serialized MHTML contents to a file and calculate its hash as it is streamed back from the renderer via a Mojo data pipe.
    OOBE_FORCE_SHOW_SCREEN = 'oobe-force-show-screen'   # Forces OOBE/login to force show a comma-separated list of screens from chromeos::kScreenNames in oobe_screen.cc. Supported screens are: user-image
    OOBE_GUEST_SESSION = 'oobe-guest-session'   # Indicates that a guest session has been started before OOBE completion.
    OOBE_SKIP_POSTLOGIN = 'oobe-skip-postlogin'   # Skips all other OOBE pages after user login.
    OOBE_SKIP_TO_LOGIN = 'oobe-skip-to-login'   # Skip to login screen.
    OOBE_TIMER_INTERVAL = 'oobe-timer-interval'   # Interval at which we check for total time on OOBE.
    OPENGRAPH = 'opengraph'   # 
    OPTIMIZATION_GUIDE_FETCH_HINTS_OVERRIDE = 'optimization-guide-fetch-hints-override'   # Overrides scheduling and time delays for fetching hints and causes a hints fetch immediately on start up using the provided comma separate lists of hosts.
    OPTIMIZATION_GUIDE_FETCH_HINTS_OVERRIDE_TIMER = 'optimization-guide-fetch-hints-override-timer'   # Overrides the hints fetch scheduling and delay, causing a hints fetch immediately on start up using the TopHostProvider. This is meant for testing.
    OPTIMIZATION_GUIDE_SERVICE_API_KEY = 'optimization-guide-service-api-key'   # Overrides the Optimization Guide Service API Key for remote requests to be made.
    OPTIMIZATION_GUIDE_SERVICE_URL = 'optimization-guide-service-url'   # Overrides the Optimization Guide Service URL that the HintsFetcher will request remote hints from.
    OPTIMIZATION_GUIDE_HINTS_OVERRIDE = 'optimization_guide_hints_override'   # Overrides the Hints Protobuf that would come from the component updater. If the value of this switch is invalid, regular hint processing is used. The value of this switch should be a base64 encoding of a binary Configuration message, found in optimization_guide's hints.proto. Providing a valid value to this switch causes Chrome startup to block on hints parsing.
    ORIGIN_TRIAL_DISABLED_FEATURES = 'origin-trial-disabled-features'   # Contains a list of feature names for which origin trial experiments should be disabled. Names should be separated by "|" characters.
    ORIGIN_TRIAL_DISABLED_TOKENS = 'origin-trial-disabled-tokens'   # Contains a list of token signatures for which origin trial experiments should be disabled. Tokens should be separated by "|" characters.
    ORIGIN_TRIAL_PUBLIC_KEY = 'origin-trial-public-key'   # Overrides the default public key for checking origin trial tokens.
    ORIGINAL_PROCESS_START_TIME = 'original-process-start-time'   # The time that a new chrome process which is delegating to an already running chrome process started. (See ProcessSingleton for more details.)
    OUTPUT = 'output'   # 
    OVERRIDE = 'override'   # "Override" value for kCrosRegionsMode (region's data is read first).
    OVERRIDE_ENABLED_CDM_INTERFACE_VERSION = 'override-enabled-cdm-interface-version'   # Overrides the default enabled library CDM interface version(s) with the one specified with this switch, which will be the only version enabled. For example, on a build where CDM 8, CDM 9 and CDM 10 are all supported (implemented), but only CDM 8 and CDM 9 are enabled by default: --override-enabled-cdm-interface-version=8 : Only CDM 8 is enabled --override-enabled-cdm-interface-version=9 : Only CDM 9 is enabled --override-enabled-cdm-interface-version=10 : Only CDM 10 is enabled --override-enabled-cdm-interface-version=11 : No CDM interface is enabled This can be used for local testing and debugging. It can also be used to enable an experimental CDM interface (which is always disabled by default) for testing while it's still in development.
    OVERRIDE_HARDWARE_SECURE_CODECS_FOR_TESTING = 'override-hardware-secure-codecs-for-testing'   # Overrides hardware secure codecs support for testing. If specified, real platform hardware secure codecs check will be skipped. Codecs are separated by comma. Valid codecs are "vp8", "vp9" and "avc1". For example: --override-hardware-secure-codecs-for-testing=vp8,vp9 --override-hardware-secure-codecs-for-testing=avc1 CENC encryption scheme is assumed to be supported for the specified codecs. If no valid codecs specified, no hardware secure codecs are supported. This can be used to disable hardware secure codecs support: --override-hardware-secure-codecs-for-testing
    OVERRIDE_METRICS_UPLOAD_URL = 'override-metrics-upload-url'   # Override the URL to which metrics logs are sent for debugging.
    OVERRIDE_PLUGIN_POWER_SAVER_FOR_TESTING = 'override-plugin-power-saver-for-testing'   # Override the behavior of plugin throttling for testing. By default the throttler is only enabled for a hard-coded list of plugins. Set the value to 'always' to always throttle every plugin instance. Set the value to 'never' to disable throttling.
    OVERRIDE_USE_SOFTWARE_GL_FOR_TESTS = 'override-use-software-gl-for-tests'   # Forces the use of software GL instead of hardware gpu.
    OZONE_DUMP_FILE = 'ozone-dump-file'   # Specify location for image dumps.
    OZONE_PLATFORM = 'ozone-platform'   # Specify ozone platform implementation to use.
    PACK_EXTENSION = 'pack-extension'   # Packages an extension to a .crx installable file from a given directory.
    PACK_EXTENSION_KEY = 'pack-extension-key'   # Optional PEM private key to use in signing packaged .crx.
    PARENT_WINDOW = 'parent-window'   # 
    PASSIVE_LISTENERS_DEFAULT = 'passive-listeners-default'   # Override the default value for the 'passive' field in javascript addEventListener calls. Values are defined as: 'documentonlytrue' to set the default be true only for document level nodes. 'true' to set the default to be true on all nodes (when not specified). 'forcealltrue' to force the value on all nodes.
    PASSTHROUGH = 'passthrough'   # 
    PASSWORD_STORE = 'password-store'   # Specifies which encryption storage backend to use. Possible values are kwallet, kwallet5, gnome, gnome-keyring, gnome-libsecret, basic. Any other value will lead to Chrome detecting the best backend automatically. TODO(crbug.com/571003): Once PasswordStore no longer uses the Keyring or KWallet for storing passwords, rename this flag to stop referencing passwords. Do not rename it sooner, though; developers and testers might rely on it keeping large amounts of testing passwords out of their Keyrings or KWallets.
    PDF_COMPOSITOR = 'pdf_compositor'   # 
    PEN_DEVICES = 'pen-devices'   # Tells chrome to interpret events from these devices as pen events. Only available with XInput 2 (i.e. X server 1.8 or above). The id's of the devices can be retrieved from 'xinput list'.
    PERF_TEST_PRINT_UMA_MEANS = 'perf-test-print-uma-means'   # Show the mean value of histograms that native performance tests are monitoring. Note that this is only applicable for PerformanceTest subclasses.
    PERFETTO_DISABLE_INTERNING = 'perfetto-disable-interning'   # Repeat internable data for each TraceEvent in the perfetto proto format.
    PERFETTO_OUTPUT_FILE = 'perfetto-output-file'   # If supplied, will enable Perfetto startup tracing and stream the output to the given file. TODO(oysteine): Remove once Perfetto starts early enough after process startup to be able to replace the legacy startup tracing.
    PERFORMANCE = 'performance'   # 
    PERMISSION_REQUEST_API_SCOPE = 'permission-request-api-scope'   # Development flag for permission request API. This flag is needed until the API is finalized. TODO(bauerb): Remove when this flag is not needed anymore.
    PERMISSION_REQUEST_API_URL = 'permission-request-api-url'   # Development flag for permission request API. This flag is needed until the API is finalized. TODO(bauerb): Remove when this flag is not needed anymore.
    POST_REBOOT = 'post-reboot'   # Indicates this run is post-reboot.
    POST_REBOOT_OVERRIDE_CMD = 'post-reboot-override-cmd'   # Indicates that this is a post-reboot run that should fetch its switches from an external registry key. Important note: When this switch is present in the command line, all other switches are ignored except for kCleanupIdSwitch, which is used to read post-reboot switches from the correct registry location.
    POST_REBOOT_TRIGGER = 'post-reboot-trigger'   # A freeform text string that records what triggered a post-reboot run.
    PPAPI = 'ppapi'   # 
    PPAPI_ANTIALIASED_TEXT_ENABLED = 'ppapi-antialiased-text-enabled'   # The boolean value (0/1) of FontRenderParams::antialiasing to be passed to Ppapi processes.
    PPAPI_BROKER = 'ppapi-broker'   # 
    PPAPI_FLASH_ARGS = 'ppapi-flash-args'   # "Command-line" arguments for the PPAPI Flash; used for debugging options.
    PPAPI_FLASH_PATH = 'ppapi-flash-path'   # Use the PPAPI (Pepper) Flash found at the given path.
    PPAPI_FLASH_VERSION = 'ppapi-flash-version'   # Report the given version for the PPAPI (Pepper) Flash. The version should be numbers separated by '.'s (e.g., "12.3.456.78"). If not specified, it defaults to "10.2.999.999".
    PPAPI_IN_PROCESS = 'ppapi-in-process'   # Runs PPAPI (Pepper) plugins in-process.
    PPAPI_PLUGIN_LAUNCHER = 'ppapi-plugin-launcher'   # Specifies a command that should be used to launch the ppapi plugin process. Useful for running the plugin process through purify or quantify.  Ex: --ppapi-plugin-launcher="path\to\purify /Run=yes"
    PPAPI_STARTUP_DIALOG = 'ppapi-startup-dialog'   # Causes the PPAPI sub process to display a dialog on launch. Be sure to use --no-sandbox as well or the sandbox won't allow the dialog to display.
    PPAPI_SUBPIXEL_RENDERING_SETTING = 'ppapi-subpixel-rendering-setting'   # The enum value of FontRenderParams::subpixel_rendering to be passed to Ppapi processes.
    PREVIOUS_APP = 'previous-app'   # previous-app should be the app that was running when last-launched-app started.
    PRINT_TO_PDF = 'print-to-pdf'   # Save a pdf file of the loaded page.
    PRIVET_IPV6_ONLY = 'privet-ipv6-only'   # Use IPv6 only for privet HTTP.
    PROCESS_PER_SITE = 'process-per-site'   # Enable the "Process Per Site" process model for all domains. This mode consolidates same-site pages so that they share a single process.  More details here: - https://www.chromium.org/developers/design-documents/process-models - The class comment in site_instance.h, listing the supported process models.  IMPORTANT: This isn't to be confused with --site-per-process (which is about isolation, not consolidation). You probably want the other one.
    PROCESS_PER_TAB = 'process-per-tab'   # Runs each set of script-connected tabs (i.e., a BrowsingInstance) in its own renderer process.  We default to using a renderer process for each site instance (i.e., group of pages from the same registered domain with script connections to each other). TODO(creis): This flag is currently a no-op.  We should refactor it to avoid "unnecessary" process swaps for cross-site navigations but still swap when needed for security (e.g., isolated origins).
    PRODUCT_VERSION = 'product-version'   # Outputs the product version information and quit. Used as an internal api to detect the installed version of Chrome on Linux.
    PROFILE_DIRECTORY = 'profile-directory'   # Selects directory of profile to associate with the first browser launched.
    PROFILE_REQUIRES_POLICY = 'profile-requires-policy'   # If set to "true", the profile requires policy during restart (policy load must succeed, otherwise session restart should fail).
    PROFILING = 'profiling'   # 
    PROFILING_AT_START = 'profiling-at-start'   # Starts the sampling based profiler for the browser process at startup. This will only work if chrome has been built with the gn arg enable_profiling = true. The output will go to the value of kProfilingFile.
    PROFILING_FILE = 'profiling-file'   # Specifies a location for profiling output. This will only work if chrome has been built with the gyp variable profiling=1 or gn arg enable_profiling=true.  {pid} if present will be replaced by the pid of the process. {count} if present will be incremented each time a profile is generated for this process. The default is chrome-profile-{pid} for the browser and test-profile-{pid} for tests.
    PROFILING_FLUSH = 'profiling-flush'   # Controls whether profile data is periodically flushed to a file. Normally the data gets written on exit but cases exist where chromium doesn't exit cleanly (especially when using single-process). A time in seconds can be specified.
    PROGRESS_BAR_ANIMATION = 'progress-bar-animation'   # Specifies Android phone page loading progress bar animation.
    PROXY_AUTO_DETECT = 'proxy-auto-detect'   # Forces proxy auto-detection.
    PROXY_BYPASS_LIST = 'proxy-bypass-list'   # Specifies a list of hosts for whom we bypass proxy settings and use direct connections. Ignored unless --proxy-server is also specified. This is a comma-separated list of bypass rules. See: "net/proxy_resolution/proxy_bypass_rules.h" for the format of these rules.
    PROXY_PAC_URL = 'proxy-pac-url'   # Uses the pac script at the given URL
    PROXY_SERVER = 'proxy-server'   # Uses a specified proxy server, overrides system settings. This switch only affects HTTP and HTTPS requests.
    PSEUDO = 'pseudo'   # 
    PUBLIC_ACCOUNTS_SAML_URL = 'public-accounts-saml-url'   # Url addrress of SAML provider for a SAML public session. TODO: Remove when https://crbug.com/984021 is fixed.
    PULL_TO_REFRESH = 'pull-to-refresh'   # Enables or disables pull-to-refresh gesture in response to vertical overscroll. Set the value to '0' to disable the feature, set to '1' to enable it for both touchpad and touchscreen, and set to '2' to enable it only for touchscreen. Defaults to disabled.
    PURGE_HINT_CACHE_STORE = 'purge_hint_cache_store'   # Purges the hint cache store on startup, so that it's guaranteed to be using fresh data.
    QUARANTINE_DIR = 'quarantine-dir'   # Specifies the quarantine folder instead of the default one.
    RDP_DESKTOP_SESSION = 'rdp_desktop_session'   # 
    READER_MODE_FEEDBACK = 'reader-mode-feedback'   # 
    READER_MODE_HEURISTICS = 'reader-mode-heuristics'   # 
    REALTIME_REPORTING_URL = 'realtime-reporting-url'   # Specifies the URL at which to upload real-time reports.
    REBASELINE_PIXEL_TESTS = 'rebaseline-pixel-tests'   # Makes browser pixel tests overwrite the reference if it does not match.
    REDIRECT_LIBASSISTANT_LOGGING = 'redirect-libassistant-logging'   # Redirects libassistant logging to /var/log/chrome/.
    REGISTER_FONT_FILES = 'register-font-files'   # Registers additional font files on Windows (for fonts outside the usual %WINDIR%\Fonts location). Multiple files can be used by separating them with a semicolon (;).
    REGISTER_PEPPER_PLUGINS = 'register-pepper-plugins'   # Register Pepper plugins (see pepper_plugin_list.cc for its format).
    REGULATORY_LABEL_DIR = 'regulatory-label-dir'   # The name of the per-model directory which contains per-region subdirectories with regulatory label files for this model. The per-model directories (if there are any) are located under "/usr/share/chromeos-assets/regulatory_labels/".
    RELAUNCHER = 'relauncher'   # A process type (switches::kProcessType) that relaunches the browser. See chrome/browser/mac/relauncher.h.
    REMOTE_DEBUGGING_ADDRESS = 'remote-debugging-address'   # Use the given address instead of the default loopback for accepting remote debugging connections. Should be used together with --remote-debugging-port. Note that the remote debugging protocol does not perform any authentication, so exposing it too widely can be a security risk.
    REMOTE_DEBUGGING_PIPE = 'remote-debugging-pipe'   # Enables remote debug over stdio pipes [in=3, out=4]. Optionally, specifies the format for the protocol messages, can be either "JSON" (the default) or "CBOR".
    REMOTE_DEBUGGING_PORT = 'remote-debugging-port'   # Enables remote debug over HTTP on the specified port.
    REMOTE_DEBUGGING_SOCKET_NAME = 'remote-debugging-socket-name'   # Enables remote debug over HTTP on the specified socket name.
    REMOTE_DEBUGGING_TARGETS = 'remote-debugging-targets'   # Porvides a list of addresses to discover DevTools remote debugging targets. The format is <host>:<port>,...,<host>:port.
    REMOVE_SCAN_ONLY_UWS = 'remove-scan-only-uws'   # Allow the engine to remove UwS that isn't marked cleanable. For testing only.
    RENDERER = 'renderer'   # 
    RENDERER_CLIENT_ID = 'renderer-client-id'   # 
    RENDERER_CMD_PREFIX = 'renderer-cmd-prefix'   # The contents of this flag are prepended to the renderer command line. Useful values might be "valgrind" or "xterm -e gdb --args".
    RENDERER_PROCESS_LIMIT = 'renderer-process-limit'   # Overrides the default/calculated limit to the number of renderer processes. Very high values for this setting can lead to high memory/resource usage or instability.
    RENDERER_SAMPLING = 'renderer-sampling'   # 
    RENDERER_STARTUP_DIALOG = 'renderer-startup-dialog'   # Causes the renderer process to display a dialog on launch. Passing this flag also adds service_manager::kNoSandbox on Windows non-official builds, since that's needed to show a dialog.
    RENDERER_WAIT_FOR_JAVA_DEBUGGER = 'renderer-wait-for-java-debugger'   # Block ChildProcessMain thread of the renderer's ChildProcessService until a Java debugger is attached.
    RENDERPASS = 'renderpass'   # 
    REPL = 'repl'   # Runs a read-eval-print loop that allows the user to evaluate Javascript expressions.
    REPORT_VP9_AS_AN_UNSUPPORTED_MIME_TYPE = 'report-vp9-as-an-unsupported-mime-type'   # Force to report VP9 as an unsupported MIME type.
    REQUIRE_AUDIO_HARDWARE_FOR_TESTING = 'require-audio-hardware-for-testing'   # When running tests on a system without the required hardware or libraries, this flag will cause the tests to fail. Otherwise, they silently succeed.
    REQUIRE_WLAN = 'require-wlan'   # Only connect to WLAN interfaces.
    RESET_VARIATION_STATE = 'reset-variation-state'   # Forces a reset of the one-time-randomized FieldTrials on this client, also known as the Chrome Variations state.
    RESTORE_LAST_SESSION = 'restore-last-session'   # Indicates the last session should be restored on startup. This overrides the preferences value. Note that this does not force automatic session restore following a crash, so as to prevent a crash loop. This switch is used to implement support for OS-specific "continue where you left off" functionality on OS X and Windows.
    RLZ_PING_DELAY = 'rlz-ping-delay'   # The rlz ping delay (in seconds) that overwrites the default value.
    RTL = 'rtl'   # 
    RUN_ALL_COMPOSITOR_STAGES_BEFORE_DRAW = 'run-all-compositor-stages-before-draw'   # Effectively disables pipelining of compositor frame production stages by waiting for each stage to finish before completing a frame.
    RUN_MANUAL = 'run-manual'   # Manual tests only run when --run-manual is specified. This allows writing tests that don't run automatically but are still in the same test binary. This is useful so that a team that wants to run a few tests doesn't have to add a new binary that must be compiled on all builds.
    RUN_WEB_TESTS = 'run-web-tests'   # Request the render trees of pages to be dumped as text once they have finished loading.
    RUN_WITHOUT_SANDBOX_FOR_TESTING = 'run-without-sandbox-for-testing'   # Load the engine outside the sandbox. This is only to be used for manual testing.
    SAFEBROWSING_MANUAL_DOWNLOAD_BLACKLIST = 'safebrowsing-manual-download-blacklist'   # List of comma-separated sha256 hashes of executable files which the download-protection service should treat as "dangerous."  For a file to show a warning, it also must be considered a dangerous filetype and not be whitelisted otherwise (by signature or URL) and must be on a supported OS. Hashes are in hex. This is used for manual testing when looking for ways to by-pass download protection.
    SAFESITES = 'SafeSites'   # 
    SAML_PASSWORD_CHANGE_URL = 'saml-password-change-url'   # Password change url for SAML users. Remove when https://crbug.com/941489 is fixed.
    SANDBOX_IPC = 'sandbox-ipc'   # Causes the process to run as a sandbox IPC subprocess.
    SANDBOX_MOJO_PIPE_TOKEN = 'sandbox-mojo-pipe-token'   # Mojo pipe token generated in the broker process and passed to the sandbox process to bind with the EngineCommands IPC interface.
    SANDBOXED_PROCESS_ID = 'sandboxed-process-id'   # Used to identify the id of the sandbox process that is intended to be spawned.
    SAVE_PAGE_AS_MHTML = 'save-page-as-mhtml'   # Disable saving pages as HTML-only, disable saving pages as HTML Complete (with a directory of sub-resources). Enable only saving pages as MHTML. See http://crbug.com/120416 for how to remove this switch.
    SCAN_LOCATIONS = 'scan-locations'   # Used to limit trace locations that will be scanned. Trace locations should be specified as integers, separated by commas. For example: --scan-locations=1,2,3,5
    SCANNING_TIMEOUT = 'scanning-timeout'   # Set the timeout for the scanning phase, in minutes. 0 disables the timeout entirely. WARNING: this switch is used by internal test systems. Be careful when making changes.
    SCHEDULER_CONFIGURATION = 'scheduler-configuration'   # Selects the scheduler configuration specified in the parameter.
    SCREEN_CONFIG = 'screen-config'   # Specifies the initial screen configuration, or state of all displays, for FakeDisplayDelegate, see class for format details.
    SCREENSHOT = 'screenshot'   # Save a screenshot of the loaded page.
    SEARCH_PROVIDER_LOGO_URL = 'search-provider-logo-url'   # Use a static URL for the logo of the default search engine. Example: https://www.google.com/branding/logo.png
    SECONDARY_DISPLAY_LAYOUT = 'secondary-display-layout'   # Specifies the layout mode and offsets for the secondary display for testing. The format is "<t|r|b|l>,<offset>" where t=TOP, r=RIGHT, b=BOTTOM and L=LEFT. For example, 'r,-100' means the secondary display is positioned on the right with -100 offset. (above than primary)
    SERVICE = 'service'   # The process type value which causes a process to run as a cloud print service process.  DO NOT CHANGE THIS VALUE. Cloud printing relies on an external binary launching Chrome with this process type.
    SERVICE_MANAGER = 'service-manager'   # The value of the |kProcessType| switch which tells the executable to assume the role of a standalone Service Manager instance.
    SERVICE_NAME = 'service-name'   # Indicates the name of the service to run. Useful for debugging, or if a service executable is built to support being run as a number of potential different services.
    SERVICE_REQUEST_ATTACHMENT_NAME = 'service-request-attachment-name'   # The name of the |service_manager::mojom::ServiceRequest| message pipe handle that is attached to the incoming Mojo invitation received by the service.
    SERVICE_REQUEST_CHANNEL_TOKEN = 'service-request-channel-token'   # The token to use to construct the message pipe for a service in a child process.
    SERVICE_RUNNER = 'service-runner'   # The value of the |kProcessType| switch which tells the executable to assume the role of a service instance.
    SERVICE_SANDBOX_TYPE = 'service-sandbox-type'   # Type of sandbox to apply to the process running the service, one of the values in the next block.
    SET_EXTENSION_THROTTLE_TEST_PARAMS = 'set-extension-throttle-test-params'   # Set the parameters for ExtensionURLLoaderThrottleBrowserTest.
    SHADER_DISK_CACHE_SIZE_KB = 'shader-disk-cache-size-kb'   # Allows explicitly specifying the shader disk cache size for embedded devices. Default value is 6MB. On Android, 2MB is default and 128KB for low-end devices.
    SHARED_FILES = 'shared-files'   # Describes the file descriptors passed to a child process in the following list format:  <file_id>:<descriptor_id>,<file_id>:<descriptor_id>,...  where <file_id> is an ID string from the manifest of the service being launched and <descriptor_id> is the numeric identifier of the descriptor for the child process can use to retrieve the file descriptor from the global descriptor table.
    SHELF_DENSE_CLAMSHELL = 'shelf-dense-clamshell'   # Smaller, denser shelf in clamshell mode.
    SHELF_HOTSEAT = 'shelf-hotseat'   # New modular design for the shelf with apps separated into a hotseat UI.
    SHELF_HOVER_PREVIEWS = 'shelf-hover-previews'   # App window previews when hovering over the shelf.
    SHELF_SCROLLABLE = 'shelf-scrollable'   # Scrollable list of apps on the shelf.
    SHILL_STUB = 'shill-stub'   # Overrides Shill stub behavior. By default, ethernet, wifi and vpn are enabled, and transitions occur instantaneously. Multiple options can be comma separated (no spaces). Note: all options are in the format 'foo=x'. Values are case sensitive and based on Shill names in service_constants.h. See FakeShillManagerClient::SetInitialNetworkState for implementation. Examples: 'clear=1' - Clears all default configurations 'wifi=on' - A wifi network is initially connected ('1' also works) 'wifi=off' - Wifi networks are all initially disconnected ('0' also works) 'wifi=disabled' - Wifi is initially disabled 'wifi=none' - Wifi is unavailable 'wifi=portal' - Wifi connection will be in Portal state 'cellular=1' - Cellular is initially connected 'cellular=LTE' - Cellular is initially connected, technology is LTE 'interactive=3' - Interactive mode, connect/scan/etc requests take 3 secs
    SHORT_MERGE_SESSION_TIMEOUT_FOR_TEST = 'short-merge-session-timeout-for-test'   # Use a short (1 second) timeout for merge session loader throttle testing.
    SHORT_REPORTING_DELAY = 'short-reporting-delay'   # Sets the Reporting API delay to under a second to allow much quicker reports.
    SHOW_AGGREGATED_DAMAGE = 'show-aggregated-damage'   # Adds a DebugBorderDrawQuad to the top of the root RenderPass showing the damage rect after surface aggregation. Note that when enabled this feature sets the entire output rect as damaged after adding the quad to highlight the real damage rect, which could hide damage rect problems.
    SHOW_ANDROID_FILES_IN_FILES_APP = 'show-android-files-in-files-app'   # If true, files in Android internal storage will be shown in Files app.
    SHOW_APP_LIST = 'show-app-list'   # If true the app list will be shown.
    SHOW_AUTOFILL_SIGNATURES = 'show-autofill-signatures'   # Annotates forms and fields with Autofill signatures.
    SHOW_AUTOFILL_TYPE_PREDICTIONS = 'show-autofill-type-predictions'   # Annotates forms with Autofill field type predictions.
    SHOW_COMPONENT_EXTENSION_OPTIONS = 'show-component-extension-options'   # Makes component extensions appear in chrome://settings/extensions.
    SHOW_COMPOSITED_LAYER_BORDERS = 'show-composited-layer-borders'   # Renders a border around compositor layers to help debug and study layer compositing.
    SHOW_DC_LAYER_DEBUG_BORDERS = 'show-dc-layer-debug-borders'   # Show debug borders for DC layers - red for overlays and blue for underlays. The debug borders are offset from the layer rect by a few pixels for clarity.
    SHOW_FPS_COUNTER = 'show-fps-counter'   # Draws a heads-up-display showing Frames Per Second as well as GPU memory usage. If you also use --enable-logging=stderr --vmodule="head*=1" then FPS will also be output to the console log.
    SHOW_ICONS = 'show-icons'   # See kHideIcons.
    SHOW_LAYER_ANIMATION_BOUNDS = 'show-layer-animation-bounds'   # Renders a border that represents the bounding box for the layer's animation.
    SHOW_LAYOUT_SHIFT_REGIONS = 'show-layout-shift-regions'   # Visibly render a border around layout shift rects in the web page to help debug and study layout shifts.
    SHOW_LOGIN_DEV_OVERLAY = 'show-login-dev-overlay'   # If true, the developer tool overlay will be shown for the login/lock screen. This makes it easier to test layout logic.
    SHOW_MAC_OVERLAY_BORDERS = 'show-mac-overlay-borders'   # Show borders around CALayers corresponding to overlays and partial damage.
    SHOW_OVERDRAW_FEEDBACK = 'show-overdraw-feedback'   # Visualize overdraw by color-coding elements based on if they have other elements drawn underneath. This is good for showing where the UI might be doing more rendering work than necessary. The colors are hinting at the amount of overdraw on your screen for each pixel, as follows:  True color: No overdraw. Blue: Overdrawn once. Green: Overdrawn twice. Pink: Overdrawn three times. Red: Overdrawn four or more times.
    SHOW_PAINT_RECTS = 'show-paint-rects'   # Visibly render a border around paint rects in the web page to help debug and study painting behavior.
    SHOW_PROPERTY_CHANGED_RECTS = 'show-property-changed-rects'   # Show rects in the HUD around layers whose properties have changed.
    SHOW_SCREENSPACE_RECTS = 'show-screenspace-rects'   # Show rects in the HUD around the screen-space transformed bounds of every layer.
    SHOW_SURFACE_DAMAGE_RECTS = 'show-surface-damage-rects'   # Show rects in the HUD around damage as it is recorded into each render surface.
    SHOW_TAPS = 'show-taps'   # Draws a circle at each touch point, similar to the Android OS developer option "Show taps".
    SHOW_WEBUI_LOCK = 'show-webui-lock'   # If true, the webui lock screen wil be shown. This is deprecated and will be removed in the future.
    SHOW_WEBUI_LOGIN = 'show-webui-login'   # Forces the webui login implementation.
    SILENT_DEBUGGER_EXTENSION_API = 'silent-debugger-extension-api'   # Does not show an infobar when an extension attaches to a page using chrome.debugger page. Required to attach to extension background pages.
    SILENT_LAUNCH = 'silent-launch'   # Causes Chrome to launch without opening any windows by default. Useful if one wishes to use Chrome as an ash server.
    SIMULATE_CRITICAL_UPDATE = 'simulate-critical-update'   # Simulates a critical update being available.
    SIMULATE_ELEVATED_RECOVERY = 'simulate-elevated-recovery'   # Simulates that elevation is needed to recover upgrade channel.
    SIMULATE_OUTDATED = 'simulate-outdated'   # Simulates that current version is outdated.
    SIMULATE_OUTDATED_NO_AU = 'simulate-outdated-no-au'   # Simulates that current version is outdated and auto-update is off.
    SIMULATE_UPGRADE = 'simulate-upgrade'   # Simulates an update being available.
    SINGLE_PROCESS = 'single-process'   # Runs the renderer and plugins in the same process as the browser
    SITE_PER_PROCESS = 'site-per-process'   # Enforces a one-site-per-process security policy: * Each renderer process, for its whole lifetime, is dedicated to rendering pages for just one site. * Thus, pages from different sites are never in the same process. * A renderer process's access rights are restricted based on its site. * All cross-site navigations force process swaps. * <iframe>s are rendered out-of-process whenever the src= is cross-site.  More details here: - https://www.chromium.org/developers/design-documents/site-isolation - https://www.chromium.org/developers/design-documents/process-models - The class comment in site_instance.h, listing the supported process models.  IMPORTANT: this isn't to be confused with --process-per-site (which is about process consolidation, not isolation). You probably want this one.
    SKIA_FONT_CACHE_LIMIT_MB = 'skia-font-cache-limit-mb'   # Specifies the max number of bytes that should be used by the skia font cache. If the cache needs to allocate more, skia will purge previous entries.
    SKIA_RESOURCE_CACHE_LIMIT_MB = 'skia-resource-cache-limit-mb'   # Specifies the max number of bytes that should be used by the skia resource cache. The previous entries are purged from the cache when the memory useage exceeds this limit.
    SLOW_DOWN_COMPOSITING_SCALE_FACTOR = 'slow-down-compositing-scale-factor'   # Re-draw everything multiple times to simulate a much slower machine. Give a slow down factor to cause renderer to take that many times longer to complete, such as --slow-down-compositing-scale-factor=2.
    SLOW_DOWN_RASTER_SCALE_FACTOR = 'slow-down-raster-scale-factor'   # Re-rasters everything multiple times to simulate a much slower machine. Give a scale factor to cause raster to take that many times longer to complete, such as --slow-down-raster-scale-factor=25.
    SMS_TEST_MESSAGES = 'sms-test-messages'   # Sends test messages on first call to RequestUpdate (stub only).
    SPDY_PROXY_AUTH_FALLBACK = 'spdy-proxy-auth-fallback'   # The origin of the data reduction proxy fallback.
    SPDY_PROXY_AUTH_ORIGIN = 'spdy-proxy-auth-origin'   # The origin of the data reduction proxy.
    SPDY_PROXY_AUTH_VALUE = 'spdy-proxy-auth-value'   # A test key for data reduction proxy authentication.
    SSL_KEY_LOG_FILE = 'ssl-key-log-file'   # Causes SSL key material to be logged to the specified file for debugging purposes. See https://developer.mozilla.org/en-US/docs/Mozilla/Projects/NSS/Key_Log_Format for the format.
    SSL_VERSION_MAX = 'ssl-version-max'   # Specifies the maximum SSL/TLS version ("tls1.2" or "tls1.3").
    SSL_VERSION_MIN = 'ssl-version-min'   # Specifies the minimum SSL/TLS version ("tls1", "tls1.1", "tls1.2", or "tls1.3").
    STABLE_RELEASE_MODE = 'stable-release-mode'   # This makes us disable some web-platform runtime features so that we test content_shell as if it was a stable release. It is only followed when kRunWebTest is set. For the features' level, see http://dev.chromium.org/blink/runtime-enabled-features.
    START_FULLSCREEN = 'start-fullscreen'   # Specifies if the browser should start in fullscreen mode, like if the user had pressed F11 right after startup.
    START_MAXIMIZED = 'start-maximized'   # Starts the browser maximized, regardless of any previous settings.
    START_STACK_PROFILER = 'start-stack-profiler'   # Starts the stack sampling profiler in the child process.
    STARTED = 'started'   # Value for kTestCrosGaiaIdMigration indicating that migration is started (i.e. all stored user keys will be converted to GaiaId)
    STUB = 'stub'   # 
    SUPPRESS_MESSAGE_CENTER_POPUPS = 'suppress-message-center-popups'   # Hides all Message Center notification popups (toasts). Used for testing.
    SURFACE = 'surface'   # 
    SWIFTSHADER = 'swiftshader'   # 
    SWIFTSHADER_WEBGL = 'swiftshader-webgl'   # 
    SYNC_ALLOW_INSECURE_XMPP_CONNECTION = 'sync-allow-insecure-xmpp-connection'   # Allows insecure XMPP connections for sync (for testing).
    SYNC_DEFERRED_STARTUP_TIMEOUT_SECONDS = 'sync-deferred-startup-timeout-seconds'   # Allows overriding the deferred init fallback timeout.
    SYNC_DISABLE_DEFERRED_STARTUP = 'sync-disable-deferred-startup'   # Enables deferring sync backend initialization until user initiated changes occur.
    SYNC_ENABLE_GET_UPDATE_BEFORE_COMMITS = 'sync-enable-get-update-before-commits'   # TODO(crbug.com/657130): Sync integration tests depend on the precommit get updates because invalidations aren't working for them. Therefore, they pass the command line switch to enable this feature. Once sync integrations test support invalidation, this should be removed. Enables feature to perform GetUpdate requests before every commit.
    SYNC_INCLUDE_SPECIFICS = 'sync-include-specifics'   # Controls whether the initial state of the "Capture Specifics" flag on chrome://sync-internals is enabled.
    SYNC_NOTIFICATION_HOST_PORT = 'sync-notification-host-port'   # Overrides the default host:port used for notifications.
    SYNC_SHORT_INITIAL_RETRY_OVERRIDE = 'sync-short-initial-retry-override'   # This flag causes sync to retry very quickly (see polling_constants.h) the when it encounters an error, as the first step towards exponential backoff.
    SYNC_SHORT_NUDGE_DELAY_FOR_TEST = 'sync-short-nudge-delay-for-test'   # This flag significantly shortens the delay between nudge cycles. Its primary purpose is to speed up integration tests. The normal delay allows coalescing and prevention of server overload, so don't use this unless you're really sure that it's what you want.
    SYNC_URL = 'sync-url'   # Overrides the default server used for profile sync.
    SYSTEM_DEVELOPER_MODE = 'system-developer-mode'   # Used by FakeDebugDaemonClient to specify that the system is running in dev mode when running in a Linux environment. The dev mode probing is done by session manager.
    SYSTEM_GESTURE_START_HEIGHT = 'system-gesture-start-height'   # The number of pixels from the very top or bottom of the screen to consider as a valid origin for the top or bottom swipe gesture. Overrides the default value in cast_system_gesture_handler.cc.
    SYSTEM_GESTURE_START_WIDTH = 'system-gesture-start-width'   # The number of pixels from the very left or right of the screen to consider as a valid origin for the left or right swipe gesture.  Overrides the default value in cast_system_gesture_handler.cc.
    SYSTEM_LOG_UPLOAD_FREQUENCY = 'system-log-upload-frequency'   # Frequency in Milliseconds for system log uploads. Should only be used for testing purposes.
    TASK_MANAGER_SHOW_EXTRA_RENDERERS = 'task-manager-show-extra-renderers'   # Sets the task manager to track extra renderer processes that might not normally be displayed in the task manager.
    TEST_CHILD_PROCESS = 'test-child-process'   # When running certain tests that spawn child processes, this switch indicates to the test framework that the current process is a child process.
    TEST_CROS_GAIA_ID_MIGRATION = 'test-cros-gaia-id-migration'   # Controls CrOS GaiaId migration for tests ("" is default).
    TEST_DO_NOT_INITIALIZE_ICU = 'test-do-not-initialize-icu'   # When running certain tests that spawn child processes, this switch indicates to the test framework that the current process should not initialize ICU to avoid creating any scoped handles too early in startup.
    TEST_ENCRYPTION_MIGRATION_UI = 'test-encryption-migration-ui'   # Enables testing for encryption migration UI.
    TEST_GL_LIB = 'test-gl-lib'   # Flag used for Linux tests: for desktop GL bindings, try to load this GL library first, but fall back to regular library if loading fails.
    TEST_LOGGING_PATH = 'test-logging-path'   # Set a path to save logs in while testing.
    TEST_LOGGING_URL = 'test-logging-url'   # Set a test logging URL, where logs will be uploaded.
    TEST_NAME = 'test-name'   # Passes the name of the current running automated test to Chrome.
    TEST_TYPE = 'test-type'   # Type of the current test harness ("browser" or "ui").
    TEST_WALLPAPER_SERVER = 'test-wallpaper-server'   # Enables the wallpaper picker to fetch images from the test server.
    TESTING = 'testing'   # Flag set during testing and stored as a crash key, to differentiate from crashes received from actual users. WARNING: this switch is used by internal test systems. Be careful when making changes.
    TETHER_HOST_SCANS_IGNORE_WIRED_CONNECTIONS = 'tether-host-scans-ignore-wired-connections'   # Tells the Chromebook to scan for a tethering host even if there is already a wired connection. This allows end-to-end tests to be deployed over ethernet without that connection preventing scans and thereby blocking the testing of cases with no preexisting connection. Should be used only for testing.
    TETHER_STUB = 'tether-stub'   # Overrides Tether with stub service. Provide integer arguments for the number of fake networks desired, e.g. 'tether-stub=2'.
    THIRD_PARTY_DOODLE_URL = 'third-party-doodle-url'   # Overrides the Doodle URL to use for third-party search engines. Testing? Try: https://www.gstatic.com/chrome/ntp/doodle_test/third_party_simple.json https://www.gstatic.com/chrome/ntp/doodle_test/third_party_animated.json
    TIMEOUT = 'timeout'   # Issues a stop after the specified number of milliseconds.  This cancels all navigation and causes the DOMContentLoaded event to fire.
    TINT_GL_COMPOSITED_CONTENT = 'tint-gl-composited-content'   # Tint GL-composited color.
    TLS1 = 'tls1'   # TLS 1.0 mode for the |kSSLVersionMin| switch.
    TLS1_1 = 'tls1.1'   # TLS 1.1 mode for the |kSSLVersionMin| switch.
    TLS1_2 = 'tls1.2'   # TLS 1.2 mode for |kSSLVersionMax| and |kSSLVersionMin| switches.
    TLS1_3 = 'tls1.3'   # TLS 1.2 mode for |kSSLVersionMax| and |kSSLVersionMin| switches.
    TOP_CHROME_TOUCH_UI = 'top-chrome-touch-ui'   # Controls touch-optimized UI layout for top chrome.
    TOP_CONTROLS_HIDE_THRESHOLD = 'top-controls-hide-threshold'   # Percentage of the browser controls need to be hidden before they will auto hide.
    TOP_CONTROLS_SHOW_THRESHOLD = 'top-controls-show-threshold'   # Percentage of the browser controls need to be shown before they will auto show.
    TOUCH_CALIBRATION = 'touch-calibration'   # The calibration factors given as "<left>,<right>,<top>,<bottom>".
    TOUCH_DEVICES = 'touch-devices'   # Tells chrome to interpret events from these devices as touch events. Only available with XInput 2 (i.e. X server 1.8 or above). The id's of the devices can be retrieved from 'xinput list'.
    TOUCH_EVENTS = 'touch-events'   # Enable support for touch event feature detection.
    TOUCH_NOISE_FILTERING = 'touch-noise-filtering'   # Tells Chrome to do additional touch noise filtering. Should only be used if the driver level filtering is insufficient.
    TOUCH_SELECTION_STRATEGY = 'touch-selection-strategy'   # Controls how text selection granularity changes when touch text selection handles are dragged. Should be "character" or "direction". If not specified, the platform default is used.
    TOUCHSCREEN_USABLE_WHILE_SCREEN_OFF = 'touchscreen-usable-while-screen-off'   # Chromebases' touchscreens can be used to wake from suspend, unlike the touchscreens on other Chrome OS devices. If set, the touchscreen is kept enabled while the screen is off so that it can be used to turn the screen back on after it has been turned off for inactivity but before the system has suspended.
    TOUCH_VIEW = 'touch_view'   # 
    TRACE_CONFIG_FILE = 'trace-config-file'   # Causes TRACE_EVENT flags to be recorded from startup. This flag will be ignored if --trace-startup or --trace-shutdown is provided.
    TRACE_STARTUP = 'trace-startup'   # Causes TRACE_EVENT flags to be recorded from startup. Optionally, can specify the specific trace categories to include (e.g. --trace-startup=base,net) otherwise, all events are recorded. Setting this flag results in the first call to BeginTracing() to receive all trace events since startup. In Chrome, you may find --trace-startup-file and --trace-startup-duration to control the auto-saving of the trace (not supported in the base-only TraceLog component).
    TRACE_STARTUP_DURATION = 'trace-startup-duration'   # Sets the time in seconds until startup tracing ends. If omitted a default of 5 seconds is used. Has no effect without --trace-startup, or if --startup-trace-file=none was supplied.
    TRACE_STARTUP_FILE = 'trace-startup-file'   # If supplied, sets the file which startup tracing will be stored into, if omitted the default will be used "chrometrace.log" in the current directory. Has no effect unless --trace-startup is also supplied. Example: --trace-startup --trace-startup-file=/tmp/trace_event.log As a special case, can be set to 'none' - this disables automatically saving the result to a file and the first manually recorded trace will then receive all events since startup.
    TRACE_STARTUP_OWNER = 'trace-startup-owner'   # Specifies the coordinator of the startup tracing session. If the legacy tracing backend is used instead of perfetto, providing this flag is not necessary. Valid values: 'controller', 'devtools', or 'system'. Defaults to 'controller'.  If 'controller' is specified, the session is controlled and stopped via the TracingController (e.g. to implement the timeout).  If 'devtools' is specified, the startup tracing session will be owned by DevTools and thus can be controlled (i.e. stopped) via the DevTools Tracing domain on the first session connected to the browser endpoint.  If 'system' is specified, the system Perfetto service should already be tracing on a supported platform (currently only Android). Session is stopped through the normal methods for stopping system traces.
    TRACE_STARTUP_RECORD_MODE = 'trace-startup-record-mode'   # If supplied, sets the tracing record mode; otherwise, the default "record-until-full" mode will be used.
    TRACE_TO_CONSOLE = 'trace-to-console'   # Sends a pretty-printed version of tracing info to the console.
    TRACE_TO_FILE = 'trace-to-file'   # Sends trace events from these categories to a file. --trace-to-file on its own sends to default categories.
    TRACE_TO_FILE_NAME = 'trace-to-file-name'   # Specifies the file name for --trace-to-file. If unspecified, it will go to a default file name.
    TRACE_UPLOAD_URL = 'trace-upload-url'   # Sets the target URL for uploading tracing data.
    TRANSLATE_RANKER_MODEL_URL = 'translate-ranker-model-url'   # Overrides the URL from which the translate ranker model is downloaded.
    TRANSLATE_SCRIPT_URL = 'translate-script-url'   # Overrides the default server used for Google Translate.
    TRANSLATE_SECURITY_ORIGIN = 'translate-security-origin'   # Overrides security-origin with which Translate runs in an isolated world.
    TRUE = 'true'   # Value indicating whether flag from command line switch is true.
    TRUSTABLE_BUNDLED_EXCHANGES_FILE = 'trustable-bundled-exchanges-file'   # Accepts specified file as a trustable BundledExchanges file. This flag should be used only for testing purpose.
    TRUSTED_CDN_BASE_URL_FOR_TESTS = 'trusted-cdn-base-url-for-tests'   # Specifies a base URL for the trusted CDN for tests.
    TRUSTED_DOWNLOAD_SOURCES = 'trusted-download-sources'   # Identifies a list of download sources as trusted, but only if proper group policy is set.
    TRY_CHROME_AGAIN = 'try-chrome-again'   # Experimental. Shows a dialog asking the user to try chrome. This flag is to be used only by the upgrade process.
    TRY_SUPPORTED_CHANNEL_LAYOUTS = 'try-supported-channel-layouts'   # Instead of always using the hardware channel layout, check if a driver supports the source channel layout.  Avoids outputting empty channels and permits drivers to enable stereo to multichannel expansion.  Kept behind a flag since some drivers lie about supported layouts and hang when used.  See http://crbug.com/259165 for more details.
    TYPE = 'type'   # Indicates the type of process to run. This may be "service-manager", "service-runner", or any other arbitrary value supported by the embedder.
    UI_COMPOSITOR_MEMORY_LIMIT_WHEN_VISIBLE_MB = 'ui-compositor-memory-limit-when-visible-mb'   # 
    UI_DISABLE_PARTIAL_SWAP = 'ui-disable-partial-swap'   # Disable partial swap which is needed for some OpenGL drivers / emulators.
    UI_DISABLE_ZERO_COPY = 'ui-disable-zero-copy'   # 
    UI_ENABLE_LAYER_LISTS = 'ui-enable-layer-lists'   # Switches the ui compositor to use layer lists instead of layer trees.
    UI_ENABLE_RGBA_4444_TEXTURES = 'ui-enable-rgba-4444-textures'   # 
    UI_ENABLE_ZERO_COPY = 'ui-enable-zero-copy'   # 
    UI_SHOW_COMPOSITED_LAYER_BORDERS = 'ui-show-composited-layer-borders'   # 
    UI_SHOW_FPS_COUNTER = 'ui-show-fps-counter'   # 
    UI_SHOW_LAYER_ANIMATION_BOUNDS = 'ui-show-layer-animation-bounds'   # 
    UI_SHOW_PAINT_RECTS = 'ui-show-paint-rects'   # 
    UI_SHOW_PROPERTY_CHANGED_RECTS = 'ui-show-property-changed-rects'   # 
    UI_SHOW_SCREENSPACE_RECTS = 'ui-show-screenspace-rects'   # 
    UI_SHOW_SURFACE_DAMAGE_RECTS = 'ui-show-surface-damage-rects'   # 
    UI_SLOW_ANIMATIONS = 'ui-slow-animations'   # 
    UNINSTALL = 'uninstall'   # Runs un-installation steps that were done by chrome first-run.
    UNLIMITED_STORAGE = 'unlimited-storage'   # Overrides per-origin quota settings to unlimited storage for any apps/origins.  This should be used only for testing purpose.
    UNSAFELY_ALLOW_PROTECTED_MEDIA_IDENTIFIER_FOR_DOMAIN = 'unsafely-allow-protected-media-identifier-for-domain'   # For automated testing of protected content, this switch allows specific domains (e.g. example.com) to skip asking the user for permission to share the protected media identifier. In this context, domain does not include the port number. User's content settings will not be affected by enabling this switch. Reference: http://crbug.com/718608 Example: --unsafely-allow-protected-media-identifier-for-domain=a.com,b.ca
    UNSAFELY_TREAT_INSECURE_ORIGIN_AS_SECURE = 'unsafely-treat-insecure-origin-as-secure'   # Treat given (insecure) origins as secure origins. Multiple origins can be supplied as a comma-separated list. For the definition of secure contexts, see https://w3c.github.io/webappsec-secure-contexts/ and https://www.w3.org/TR/powerful-features/#is-origin-trustworthy  Example: --unsafely-treat-insecure-origin-as-secure=http://a.test,http://b.test
    UPGRADE_TOKEN = 'upgrade-token'   # 
    UPLOAD_LOG_FILE = 'upload-log-file'   # Specifies the full path to a protocol buffer log file to be uploaded.
    USE_ANGLE = 'use-angle'   # Select which ANGLE backend to use. Options are: default: Attempts several ANGLE renderers until one successfully initializes, varying ES support by platform. d3d9: Legacy D3D9 renderer, ES2 only. d3d11: D3D11 renderer, ES2 and ES3. warp: D3D11 renderer using software rasterization, ES2 and ES3. gl: Desktop GL renderer, ES2 and ES3. gles: GLES renderer, ES2 and ES3.
    USE_CMD_DECODER = 'use-cmd-decoder'   # Use the Pass-through command decoder, skipping all validation and state tracking. Switch lives in ui/gl because it affects the GL binding initialization on platforms that would otherwise not default to using EGL bindings.
    USE_CRAS = 'use-cras'   # Use CRAS, the ChromeOS audio server.
    USE_CRASH_HANDLER_IN_TESTS = 'use-crash-handler-in-tests'   # If present, the test harness will use the crash reporter.
    USE_CRASH_HANDLER_WITH_ID = 'use-crash-handler-with-id'   # Specifies the IPC pipe name of the crash handler to use (instead of starting a new crash handler process).
    USE_FAKE_CODEC_FOR_PEER_CONNECTION = 'use-fake-codec-for-peer-connection'   # Replaces the existing codecs supported in peer connection with a single fake codec entry that create a fake video encoder and decoder.
    USE_FAKE_DEVICE_FOR_MEDIA_STREAM = 'use-fake-device-for-media-stream'   # Use fake device for Media Stream to replace actual camera and microphone.
    USE_FAKE_MJPEG_DECODE_ACCELERATOR = 'use-fake-mjpeg-decode-accelerator'   # Use a fake device for accelerated decoding of MJPEG. This allows, for example, testing of the communication to the GPU service without requiring actual accelerator hardware to be present.
    USE_FAKE_UI_FOR_MEDIA_STREAM = 'use-fake-ui-for-media-stream'   # Bypass the media stream infobar by selecting the default device for media streams (e.g. WebRTC). Works with --use-fake-device-for-media-stream.
    USE_FILE_FOR_FAKE_AUDIO_CAPTURE = 'use-file-for-fake-audio-capture'   # Play a .wav file as the microphone. Note that for WebRTC calls we'll treat the bits as if they came from the microphone, which means you should disable audio processing (lest your audio file will play back distorted). The input file is converted to suit Chrome's audio buses if necessary, so most sane .wav files should work. You can pass either <path> to play the file looping or <path>%noloop to stop after playing the file to completion.
    USE_FILE_FOR_FAKE_VIDEO_CAPTURE = 'use-file-for-fake-video-capture'   # Use an .y4m file to play as the webcam. See the comments in media/capture/video/file_video_capture_device.h for more details.
    USE_FIRST_DISPLAY_AS_INTERNAL = 'use-first-display-as-internal'   # Uses the 1st display in --ash-host-window-bounds as internal display. This is for debugging on linux desktop.
    USE_GL = 'use-gl'   # Select which implementation of GL the GPU process should use. Options are: desktop: whatever desktop OpenGL the user has installed (Linux and Mac default). egl: whatever EGL / GLES2 the user has installed (Windows default - actually ANGLE). swiftshader: The SwiftShader software renderer.
    USE_GPU_HIGH_THREAD_PRIORITY_FOR_PERF_TESTS = 'use-gpu-high-thread-priority-for-perf-tests'   # Increases the priority (to REALTIME_AUDIO) of gpu process and compositor thread. This is only to be used for perf tests on macOS for more reliable values.
    USE_GPU_IN_TESTS = 'use-gpu-in-tests'   # Use hardware gpu, if available, for tests.
    USE_MOBILE_USER_AGENT = 'use-mobile-user-agent'   # Set when Chromium should use a mobile user agent.
    USE_MOCK_CERT_VERIFIER_FOR_TESTING = 'use-mock-cert-verifier-for-testing'   # Use the MockCertVerifier. This only works in test code.
    USE_MOCK_KEYCHAIN = 'use-mock-keychain'   # 
    USE_SYSTEM_DEFAULT_PRINTER = 'use-system-default-printer'   # Uses the system default printer as the initially selected destination in print preview, instead of the most recently used destination.
    USE_TEMP_REGISTRY_PATH = 'use-temp-registry-path'   # Override the registry with the specified temporary registry. Used for tests.
    USE_VIZ_HIT_TEST_SURFACE_LAYER = 'use-viz-hit-test-surface-layer'   # Enables the viz hit-test logic (HitTestAggregator and HitTestQuery), with hit-test data coming from surface layer.
    USE_VULKAN = 'use-vulkan'   # Enable Vulkan support and select Vulkan implementation, must also have ENABLE_VULKAN defined.
    USER_AGENT = 'user-agent'   # A string used to override the default user agent with a custom one.
    USER_ALWAYS_AFFILIATED = 'user-always-affiliated'   # Always treat user as affiliated. TODO(antrim): Remove once test servers correctly produce affiliation ids.
    USER_DATA_DIR = 'user-data-dir'   # Directory where the browser stores the user profile.
    USER_GESTURE_REQUIRED = 'user-gesture-required'   # Autoplay policy to require a user gesture in order to play.
    USER_RESPONSE_TIMEOUT = 'user-response-timeout'   # Set the timeout for how long to wait for user response from Chrome, in minutes. 0 disables the timeout entirely.
    UTILITY = 'utility'   # 
    UTILITY_AND_BROWSER = 'utility-and-browser'   # 
    UTILITY_CMD_PREFIX = 'utility-cmd-prefix'   # The contents of this flag are prepended to the utility process command line. Useful values might be "valgrind" or "xterm -e gdb --args".
    UTILITY_SAMPLING = 'utility-sampling'   # 
    UTILITY_STARTUP_DIALOG = 'utility-startup-dialog'   # Causes the utility process to display a dialog on launch.
    V = 'v'   # Gives the default maximal active V-logging level; 0 is the default. Normally positive values are used for V-logging levels.
    V8_CACHE_OPTIONS = 'v8-cache-options'   # Set options to cache V8 data. (off, preparse data, or code)
    VALIDATE_CRX = 'validate-crx'   # Examines a .crx for validity and prints the result.
    VALIDATE_INPUT_EVENT_STREAM = 'validate-input-event-stream'   # In debug builds, asserts that the stream of input events is valid.
    VALIDATING = 'validating'   # The command decoder names that can be passed to --use-cmd-decoder.
    VARIATIONS_INSECURE_SERVER_URL = 'variations-insecure-server-url'   # Specifies a custom URL for the server to use as an insecure fallback when requests to |kVariationsServerURL| fail. Requests to this URL will be encrypted.
    VARIATIONS_OVERRIDE_COUNTRY = 'variations-override-country'   # Allows overriding the country used for evaluating variations. This is similar to the "Override Variations Country" entry on chrome://translate-internals, but is exposed as a command-line flag to allow testing First Run scenarios. Additionally, unlike chrome://translate-internals, the value isn't persisted across sessions.
    VARIATIONS_SERVER_URL = 'variations-server-url'   # Specifies a custom URL for the server which reports variation data to the client. Specifying this switch enables the Variations service on unofficial builds. See variations_service.cc.
    VERSION = 'version'   # 
    VIDEO_IMAGE_TEXTURE_TARGET = 'video-image-texture-target'   # Texture target for CHROMIUM_image backed video frame textures.
    VIDEO_THREADS = 'video-threads'   # Set number of threads to use for video decoding.
    VIDEO_UNDERFLOW_THRESHOLD_MS = 'video-underflow-threshold-ms'   # Allows clients to override the threshold for when the media renderer will declare the underflow state for the video stream when audio is present. TODO(dalecurtis): Remove once experiments for http://crbug.com/470940 finish.
    VIRTUAL_TIME_BUDGET = 'virtual-time-budget'   # If set the system waits the specified number of virtual milliseconds before deeming the page to be ready.  For determinism virtual time does not advance while there are pending network fetches (i.e no timers will fire). Once all network fetches have completed, timers fire and if the system runs out of virtual time is fastforwarded so the next timer fires immediatley, until the specified virtual time budget is exhausted.
    VMODULE = 'vmodule'   # Gives the per-module maximal V-logging levels to override the value given by --v.  E.g. "my_module=2,foo*=3" would change the logging level for all code in source files "my_module.*" and "foo*.*" ("-inl" suffixes are also disregarded for this matching).  Any pattern containing a forward or backward slash will be tested against the whole pathname and not just the module.  E.g., "*/foo/bar/*=2" would change the logging level for all code in source files under a "foo/bar" directory.
    VSYNC_INTERVAL = 'vsync-interval'   # Overrides the vsync interval used by the GPU process to refresh the display.
    VULKAN = 'vulkan'   # 
    VULKAN_NULL = 'vulkan-null'   # 
    WAIT_FOR_DEBUGGER = 'wait-for-debugger'   # Will wait for 60 seconds for a debugger to come to attach to the process.
    WAIT_FOR_DEBUGGER_CHILDREN = 'wait-for-debugger-children'   # Will add kWaitForDebugger to every child processes. If a value is passed, it will be used as a filter to determine if the child process should have the kWaitForDebugger flag passed on or not.
    WAIT_FOR_INITIAL_POLICY_FETCH_FOR_TEST = 'wait-for-initial-policy-fetch-for-test'   # Used to tell the policy infrastructure to not let profile initialization complete until policy is manually set by a test. This is used to provide backward compatibility with a few tests that incorrectly use the synchronously-initialized login profile to run their tests - do not add new uses of this flag.
    WAKE_ON_WIFI_PACKET = 'wake-on-wifi-packet'   # Enables wake on wifi packet feature, which wakes the device on the receipt of network packets from whitelisted sources.
    WALLET_SERVICE_USE_SANDBOX = 'wallet-service-use-sandbox'   # Use the sandbox Online Wallet service URL (for developer testing).
    WATCHER = 'watcher'   # Causes the process to run as a watcher process.
    WAVEOUT_BUFFERS = 'waveout-buffers'   # Number of buffers to use for WaveOut.
    WEBAPK_SERVER_URL = 'webapk-server-url'   # Custom WebAPK server URL for the sake of testing.
    WEBGL_ANTIALIASING_MODE = 'webgl-antialiasing-mode'   # Set the antialiasing method used for webgl. (none, explicit, implicit, or screenspace)
    WEBGL_MSAA_SAMPLE_COUNT = 'webgl-msaa-sample-count'   # Set a default sample count for webgl if msaa is enabled.
    WEBRTC_EVENT_LOG_PROACTIVE_PRUNING_DELTA = 'webrtc-event-log-proactive-pruning-delta'   # Sets the delay (in seconds) between proactive prunings of remote-bound WebRTC event logs which are pending upload. All positive values are legal. All negative values are illegal, and ignored. If set to 0, the meaning is "no proactive pruning".
    WEBRTC_EVENT_LOG_UPLOAD_DELAY_MS = 'webrtc-event-log-upload-delay-ms'   # WebRTC event logs will only be uploaded if the conditions hold for this many milliseconds.
    WEBRTC_EVENT_LOG_UPLOAD_NO_SUPPRESSION = 'webrtc-event-log-upload-no-suppression'   # Normally, remote-bound WebRTC event logs are uploaded only when no peer connections are active. With this flag, the upload is never suppressed.
    WEBRTC_EVENT_LOGGING = 'webrtc-event-logging'   # Enable capture and local storage of WebRTC event logs without visiting chrome://webrtc-internals. This is useful for automated testing. It accepts the path to which the local logs would be stored. Disabling is not possible without restarting the browser and relaunching without this flag.
    WEBRTC_MAX_CPU_CONSUMPTION_PERCENTAGE = 'webrtc-max-cpu-consumption-percentage'   # Configure the maximum CPU time percentage of a single core that can be consumed for desktop capturing. Default is 50. Set 100 to disable the throttling of the capture.
    WEBRTC_STUN_PROBE_TRIAL = 'webrtc-stun-probe-trial'   # Renderer process parameter for WebRTC Stun probe trial to determine the interval. Please see SetupStunProbeTrial in chrome_browser_field_trials_desktop.cc for more detail.
    WEBVIEW_DISABLE_SAFEBROWSING_SUPPORT = 'webview-disable-safebrowsing-support'   # used to disable safebrowsing functionality in webview
    WEBVIEW_ENABLE_SHARED_IMAGE = 'webview-enable-shared-image'   # Used to enable shared image API for webview.
    WEBVIEW_ENABLE_VULKAN = 'webview-enable-vulkan'   # Used to enable vulkan draw mode instead of interop draw mode for webview.
    WEBVIEW_LOG_JS_CONSOLE_MESSAGES = 'webview-log-js-console-messages'   # 
    WEBVIEW_SANDBOXED_RENDERER = 'webview-sandboxed-renderer'   # 
    WHITELISTED_EXTENSION_ID = 'whitelisted-extension-id'   # Adds the given extension ID to all the permission whitelists.
    WIN_JUMPLIST_ACTION = 'win-jumplist-action'   # Specifies which category option was clicked in the Windows Jumplist that resulted in a browser startup.
    WINDOW_POSITION = 'window-position'   # Specify the initial window position: --window-position=x,y
    WINDOW_SIZE = 'window-size'   # Sets the initial window size. Provided as string in the format "800,600".
    WINDOW_WORKSPACE = 'window-workspace'   # Specify the initial window workspace: --window-workspace=id
    WINHTTP_PROXY_RESOLVER = 'winhttp-proxy-resolver'   # Uses WinHTTP to fetch and evaluate PAC scripts. Otherwise the default is to use Chromium's network stack to fetch, and V8 to evaluate.
    WITH_CLEANUP_MODE_LOGS = 'with-cleanup-mode-logs'   # Identify that the elevated cleaner process is allowed to collect logs. This shouldn't be set if |kExecutionModeSwitch| is not ExecutionMode::kCleaner.
    WM_WINDOW_ANIMATIONS_DISABLED = 'wm-window-animations-disabled'   # If present animations are disabled.
    XR_COMPOSITING = 'xr_compositing'   # 
    ZYGOTE = 'zygote'   # Causes the process to run as a zygote.
