# -*- coding: utf-8 -*-
"""
# ---------------------------------------------------------------------------------------------------------
# ProjectName:  python-qdairlines-helper
# FileName:     book_payment.py
# Description:  预订支付控制器
# Author:       ASUS
# CreateDate:   2026/01/04
# Copyright ©2011-2026. Hunan xxxxxxx Company limited. All rights reserved.
# ---------------------------------------------------------------------------------------------------------
"""
from playwright.async_api import Page
from qdairlines_helper.utils.log_utils import logger
from typing import Dict, Any, List, Optional, Callable
from qdairlines_helper.controller.book_search import open_book_search_page, book_search
from qdairlines_helper.controller.order_verify import load_order_verify_po, order_verify
from qdairlines_helper.controller.add_passenger import add_passenger, load_add_passenger_po
from qdairlines_helper.controller.cash_pax_info import load_cash_pax_info_po, select_payment_type
from qdairlines_helper.controller.nhlms_cashdesk import load_nhlms_cash_desk_po, pay_account_payment


async def book_payment_callback(
        *, page: Page, protocol: str, domain: str, dep_city: str, arr_city: str, dep_date: str, flight_no: str,
        cabin: str, login_user: str, login_password: str, passengers: List[Dict[str, Any]], service_mobile: str,
        price_std: float, order_id: int, is_pay_completed_callback: Callable, product_type: str = "经济舱",
        payment_type: str = "汇付天下", sub_payment_type: str = "付款账户支付", operator_account: Optional[str] = None,
        timeout: float = 60.0, pay_password: Optional[str] = None, refresh_attempt: int = 3,
        price_increase_threshold: float = 20.0, price_reduction_threshold: float = 10.0, **kwargs: Any
) -> Dict[str, Any]:
    # 1. 打开预订搜索页面
    book_search_page = await open_book_search_page(page=page, protocol=protocol, domain=domain, timeout=timeout)

    # 2. 预订搜索
    await book_search(
        book_search_page=book_search_page, dep_city=dep_city, arr_city=arr_city, dep_date=dep_date, flight_no=flight_no,
        cabin=cabin, passengers=len(passengers), product_type=product_type, price_std=price_std, timeout=timeout,
        price_increase_threshold=price_increase_threshold, price_reduction_threshold=price_reduction_threshold,
    )
    logger.info(f"订单<{order_id}>，预订搜索结束")

    # 3. 加载添加乘客页面对象
    add_passenger_po = await load_add_passenger_po(page=page, protocol=protocol, domain=domain, timeout=timeout)

    # 4. 添加乘客
    await add_passenger(page=add_passenger_po, passengers=passengers, service_mobile=service_mobile, timeout=timeout)
    logger.info(f"订单<{order_id}>，添加乘客结束")

    # 5. 加载订单校验页面对象
    order_verify_po = await load_order_verify_po(page=page, protocol=protocol, domain=domain, timeout=timeout)

    # 6. 校验订单
    await order_verify(page=order_verify_po, refresh_attempt=refresh_attempt, timeout=timeout)
    logger.info(f"订单<{order_id}>，校验订单结束")

    # 7. 加载支付类型页面对象
    cash_pax_info_po = await load_cash_pax_info_po(page=page, protocol=protocol, domain=domain, timeout=timeout)

    # 8. 选择支付方式
    pre_order_no: str = await select_payment_type(page=cash_pax_info_po, payment_type=payment_type, timeout=timeout)
    logger.info(f"订单<{order_id}>，选择支付方式结束")

    # 9. 根据支付方式，加载不同的收银台页面对象
    if payment_type == "汇付天下":
        nhlms_cash_desk_po = await load_nhlms_cash_desk_po(context=kwargs.get("context"), timeout=timeout)

        # 10. 汇付天下操作支付
        if sub_payment_type == "付款账户支付":
            # 10. 汇付天下操作支付
            pay_transaction, actual_payment_amount = await pay_account_payment(
                page=nhlms_cash_desk_po, operator_account=operator_account, pay_password=pay_password, timeout=timeout,
                payment_type=sub_payment_type, is_pay_completed_callback=is_pay_completed_callback, order_id=order_id
            )
            logger.info(f"订单<{order_id}>，汇付天下操作支付结束")

            return dict(
                actual_payment_amount=actual_payment_amount, pre_order_no=pre_order_no, passengers=passengers,
                channel_user_id=login_user, channel_user_password=login_password, order_id=order_id,
                pay_transaction=pay_transaction
            )
        else:
            raise EnvironmentError(f"汇付天下暂不支持<{sub_payment_type}>操作支付方式")
    else:
        raise EnvironmentError(f"暂不支持<{payment_type}>支付方式")
