# -*- coding: utf-8 -*-
"""
# ---------------------------------------------------------------------------------------------------------
# ProjectName:  python-qdairlines-helper
# FileName:     book_search.py
# Description:  航班搜索控制器
# Author:       ASUS
# CreateDate:   2026/01/04
# Copyright ©2011-2026. Hunan xxxxxxx Company limited. All rights reserved.
# ---------------------------------------------------------------------------------------------------------
"""
import asyncio
from typing import Dict, Any
from playwright.async_api import Page, Locator
from qdairlines_helper.utils.log_utils import logger
import qdairlines_helper.config.url_const as url_const
from qdairlines_helper.po.book_search_page import BookSearchPage
from qdairlines_helper.utils.po_utils import get_ip_access_blocked_msg


async def open_book_search_page(*, page: Page, protocol: str, domain: str, timeout: float = 60.0) -> BookSearchPage:
    url_prefix = f"{protocol}://{domain}"
    book_search_url = url_prefix + url_const.login_url
    await page.goto(book_search_url)

    book_search_po = BookSearchPage(page=page, url=book_search_url)
    await book_search_po.url_wait_for(url=book_search_url, timeout=timeout)
    logger.info(f"即将进入青岛航空官网航班预订查询页面，页面URL<{book_search_url}>")
    ip_access_blocked_msg = await get_ip_access_blocked_msg(page=page, timeout=3)
    if ip_access_blocked_msg:
        raise EnvironmentError(ip_access_blocked_msg)
    return book_search_po


async def book_search(
        *, book_search_page: BookSearchPage, dep_city: str, arr_city: str, dep_date: str, flight_no: str,
        cabin: str, passengers: int, product_type: str = "经济舱", price_std: float, timeout: float = 60.0,
        price_increase_threshold: float = 20.0, price_reduction_threshold: float = 10.0
) -> None:
    try:
        # 1. 判断是否存在温馨提醒弹框
        continue_book_btn = await book_search_page.get_reminder_dialog_continue_book_btn(timeout=timeout)
        await continue_book_btn.click(button="left")
        logger.info("航班预订查询页面，出现温馨提醒弹框，【继续购票】按钮点击完成")
    except (Exception,):
        pass

    # 2.搜索栏输入起飞城市
    depart_city_input = await book_search_page.get_depart_city_input(timeout=timeout)
    await depart_city_input.fill(value=dep_city)
    await asyncio.sleep(delay=1)
    await depart_city_input.press("Enter")
    logger.info(f"航班预订查询页面，搜索栏-起飞城市<{dep_city}>输入完成")

    # 3.搜索栏输入抵达城市
    arrive_city_input = await book_search_page.get_arrive_city_input(timeout=timeout)
    await arrive_city_input.fill(value=arr_city)
    await asyncio.sleep(delay=1)
    await arrive_city_input.press("Enter")
    logger.info(f"航班预订查询页面，搜索栏-抵达城市<{arr_city}>输入完成")

    # 4.搜索栏输入起飞时间
    depart_date_input = await book_search_page.get_depart_date_input(timeout=timeout)
    await depart_date_input.fill(value=dep_date)
    await asyncio.sleep(delay=1)
    await depart_date_input.press("Enter")
    logger.info(f"航班预订查询页面，搜索栏-起飞日期<{dep_date}>输入完成")

    # 5.点击【查询机票】按钮
    flight_query_btn = await book_search_page.get_flight_query_btn(timeout=timeout)
    await flight_query_btn.click(button="left")
    logger.info(f"航班预订查询页面，【查询机票】按钮点击完成")

    # 6. 获取航班基本信息plane locator
    flight_info_plane: Locator = await book_search_page.get_flight_info_plane(timeout=timeout)
    page_flight_no: str = await book_search_page.get_flight_no(locator=flight_info_plane, timeout=timeout)
    if flight_no not in page_flight_no:
        raise RuntimeError(f"航班预订查询页面，没有搜索到航班<{flight_no}>数据")

    # 7. 获取产品类型
    flight_product_nav: Locator = await book_search_page.get_flight_product_nav(
        locator=flight_info_plane, product_type=product_type, timeout=timeout
    )
    await flight_product_nav.click(button="left")
    logger.info(f"航班预订查询页面，产品类型【{product_type}】选择完成")

    # 8. 获取所有的产品
    products: Dict[str, Any] = await book_search_page.get_flight_products(timeout=timeout)
    cabin_product: Dict[str, Any] = products.get(cabin)
    # 8.1 判断是否存在该舱位
    if not cabin_product:
        raise RuntimeError(f"航班预订查询页面，没有搜索到航班<{flight_no}>的{cabin}舱数据")
    # 8.2 判断余座
    seats_status = cabin_product.get("seats_status")
    logger.info(f"航班预订查询页面，航班<{flight_no}>舱位<{cabin}>的座位情况：{seats_status}")
    if seats_status < passengers:
        raise EnvironmentError(f"航班<{flight_no}>的余票: {seats_status}，数量不足")
    # 8.3. 判断货币符号，是否为 ¥(人民币结算)
    # 目前都为国内航班，暂不考虑货币种类

    # 8.4 判断销售价格是否满足预订需要
    amounts: Dict[str, Any] = cabin_product.get("amounts")
    amount: float = amounts.get("amount")
    if amount > price_std + price_increase_threshold:
        raise EnvironmentError(
            f"航班预订查询页面，航班<{flight_no}>官网价：{amount} 高于：订单销售价[{price_std}] + 上浮阈值[{price_increase_threshold}]，损失过高")
    if amount < price_std - price_reduction_threshold:
        raise EnvironmentError(
            f"航班预订查询页面，航班<{flight_no}>官网价：{amount} 低于：订单销售价[{price_std}] - 下降阈值[{price_reduction_threshold}]，收益过高")

    # 9. 点击【购票】按钮
    booking_btn: Locator = cabin_product.get("booking_btn")
    await booking_btn.click(button="left")
    logger.info(f"航班预订查询页面，【购票】按钮点击完成")
