# -*- coding: utf-8 -*-
"""
# ---------------------------------------------------------------------------------------------------------
# ProjectName:  python-qdairlines-helper
# FileName:     nhlms_cashdesk.py
# Description:  汇付天下收银台控制器
# Author:       ASUS
# CreateDate:   2026/01/05
# Copyright ©2011-2026. Hunan xxxxxxx Company limited. All rights reserved.
# ---------------------------------------------------------------------------------------------------------
"""
import inspect
from typing import Tuple, Callable
from playwright.async_api import BrowserContext
from qdairlines_helper.utils.log_utils import logger
import qdairlines_helper.config.url_const as url_const
from qdairlines_helper.po.nhlms_cash_desk_page import NhlmsCashDeskPage
from playwright_helper.utils.browser_utils import switch_for_table_window
from qdairlines_helper.utils.exception_utils import DuplicatePaymentError


async def load_nhlms_cash_desk_po(*, context: BrowserContext, timeout: float = 60.0) -> NhlmsCashDeskPage:
    nhlms_cashdesk_url = url_const.nhlms_cashdesk_domain + url_const.nhlms_cashdesk_url

    current_page = await switch_for_table_window(
        browser=context, url_keyword=url_const.nhlms_cashdesk_url, wait_time=int(timeout)
    )

    nhlms_cashdesk_po = NhlmsCashDeskPage(page=current_page, url=nhlms_cashdesk_url)
    await nhlms_cashdesk_po.url_wait_for(url=nhlms_cashdesk_url, timeout=timeout)
    logger.info(f"即将进入汇付天下收银台页面，页面URL<{nhlms_cashdesk_url}>")
    return nhlms_cashdesk_po


async def pay_account_payment(
        *, page: NhlmsCashDeskPage, operator_account: str, pay_password: str, order_id: int,
        is_pay_completed_callback: Callable, payment_type: str = "付款账户支付", timeout: float = 60.0
) -> Tuple[str, float]:
    # 1. 获取收银台支付流水
    pay_transaction = await page.get_order_transaction(timeout=timeout)
    logger.info(f"汇付天下收银台页面，支付流水<{pay_transaction}>获取完成")

    # 2. 获取订单支付金额
    actual_payment_amount = await page.get_order_amount(timeout=timeout)
    logger.info(f"汇付天下收银台页面，支付金额<{actual_payment_amount}>获取完成")

    # 3. 获取付款方式tab
    payment_type_tab = await page.get_payment_type_tab(payment_type=payment_type, timeout=timeout)
    await payment_type_tab.click(button="left")
    logger.info(f"汇付天下收银台页面，【{payment_type}】Tab点击完成")

    # 4. 输入操作员号
    username_input = await page.get_username_input(timeout=timeout)
    await username_input.fill(value=operator_account)
    logger.info(f"汇付天下收银台页面，操作员号<{operator_account}>输入完成")

    # 5. 输入交易密码
    password_input = await page.get_password_input(timeout=timeout)
    await password_input.fill(value=pay_password)
    logger.info(f"汇付天下收银台页面，交易密码<{pay_password}>输入完成")

    # 6. 校验订单是否已经被支付
    if inspect.iscoroutinefunction(is_pay_completed_callback):
        is_pay: bool = await is_pay_completed_callback(order_id=order_id)
    else:
        is_pay: bool = is_pay_completed_callback(order_id=order_id)
    if is_pay is True:
        raise DuplicatePaymentError(order_id=order_id)

    # 6. 点击【确认支付】
    confirm_payment_btn = await page.get_confirm_payment_btn(timeout=timeout)
    await confirm_payment_btn.click(button="left")
    logger.info("汇付天下收银台页面，【确认支付】按钮点击完成")
    return pay_transaction, actual_payment_amount
