# -*- coding: utf-8 -*-
"""
# ---------------------------------------------------------------------------------------------------------
# ProjectName:  python-qdairlines-helper
# FileName:     flight_order.py
# Description:  航班订单模块
# Author:       ASUS
# CreateDate:   2026/01/06
# Copyright ©2011-2026. Hunan xxxxxxx Company limited. All rights reserved.
# ---------------------------------------------------------------------------------------------------------
"""
from aiohttp import CookieJar
from typing import Optional, Dict, Any
import qdairlines_helper.config.url_const as url_const
from http_helper.client.async_proxy import HttpClientFactory as _HttpClientFactory


class FlightOrder:

    def __init__(
            self, *, playwright_state: Dict[str, Any], token: Optional[str], proxy: Optional[Dict[str, Any]] = None,
            domain: Optional[str] = None, protocol: Optional[str] = None, timeout: Optional[int] = None,
            retry: Optional[int] = None, enable_log: Optional[bool] = None, cookie_jar: Optional[CookieJar] = None
    ):
        self._domain = domain or "127.0.0.1:18070"
        self._protocol = protocol or "http"
        self._timeout = timeout or 60
        self._retry = retry or 0
        self._token = token
        self._origin = f"{self._protocol}://{self._domain}"
        self._enable_log = enable_log or True
        self._cookie_jar = cookie_jar or CookieJar()
        self._proxy = proxy
        self._playwright_state: Dict[str, Any] = playwright_state
        self.http_client: Optional[_HttpClientFactory] = None

    def _get_http_client(self) -> _HttpClientFactory:
        """延迟获取 HTTP 客户端"""
        if self.http_client is None:
            self.http_client = _HttpClientFactory(
                protocol=self._protocol,
                domain=self._domain,
                timeout=self._timeout,
                retry=self._retry,
                enable_log=self._enable_log,
                cookie_jar=self._cookie_jar,
                playwright_state=self._playwright_state
            )
        return self.http_client

    def _get_headers(self) -> Dict[str, str]:
        headers = {
            "content-type": "application/json;charset=utf-8",
            "origin": self._domain,
            "user-agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/143.0.0.0 Safari/537.36",
            "authorization": f"Bearer {self._token}"
        }
        return headers

    async def get_order_details(
            self, *, pre_order_no: str, user_id: str, proxy: Optional[Dict[str, Any]] = None,
            headers: Dict[str, str] = None, is_end: Optional[bool] = None
    ) -> Dict[str, Any]:
        json_data = {
            "orderNo": pre_order_no,
            "plat": "NB2C",
            "userId": user_id
        }
        _headers = self._get_headers()
        if headers is not None:
            _headers.update(headers)
        _headers["referer"] = f"{self._origin}{url_const.order_detail_url}?orderNo={pre_order_no}&orderType=2"
        if is_end is None:
            is_end = True
        if proxy:
            self._proxy = proxy
        exception_keywords = [r'<h3[^>]*class="font-bold"[^>]*>([^<]+)</h3>']
        return await self._get_http_client().request(
            method="POST",
            url=url_const.order_info_api_url,
            headers=_headers,
            is_end=is_end,
            json_data=json_data,
            proxy_config=self._proxy or None,
            exception_keywords=exception_keywords
        )
