# -*- coding: utf-8 -*-
"""
# ---------------------------------------------------------------------------------------------------------
# ProjectName:  python-qdairlines-helper
# FileName:     book_search_page.py
# Description:  预订查询页面对象
# Author:       ASUS
# CreateDate:   2026/01/04
# Copyright ©2011-2026. Hunan xxxxxxx Company limited. All rights reserved.
# ---------------------------------------------------------------------------------------------------------
"""
import re
from typing import Optional, Dict, Any, List
from playwright.async_api import Page, Locator
from playwright_helper.libs.base_po import BasePo
from qdairlines_helper.utils.log_utils import logger
import qdairlines_helper.config.url_const as url_const
from playwright_helper.utils.type_utils import convert_order_amount_text, safe_convert_advanced
from playwright.async_api import Error as PlaywrightError, TimeoutError as PlaywrightTimeoutError


class BookSearchPage(BasePo):
    url: str = url_const.book_search_url
    __page: Page

    def __init__(self, page: Page, url: Optional[str] = None) -> None:
        super().__init__(page, url or url_const.book_search_url)
        self.__page = page

    async def get_reminder_dialog_continue_book_btn(self, timeout: float = 5.0) -> Locator:
        """
        获取预订搜索页的温馨提醒弹框中的【继续购票】按钮
        :param timeout: 超时时间（秒）
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = '//div[@class="el-dialog__body"]//button[contains(@class, "search_btn")]'
        return await self.get_locator(selector=selector, timeout=timeout)

    async def get_depart_city_input(self, timeout: float = 5.0) -> Locator:
        """
        获取预订搜索页搜索栏的起飞城市输入框
        :param timeout: 超时时间（秒）
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = '//div[contains(@class, "flight_search_form")]//input[@id="orig"]'
        return await self.get_locator(selector=selector, timeout=timeout)

    async def get_arrive_city_input(self, timeout: float = 5.0) -> Locator:
        """
        获取预订搜索页搜索栏的抵达城市输入框
        :param timeout: 超时时间（秒）
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = '//div[contains(@class, "flight_search_form")]//input[@id="dest"]'
        return await self.get_locator(selector=selector, timeout=timeout)

    async def get_depart_date_input(self, timeout: float = 5.0) -> Locator:
        """
        获取预订搜索页搜索栏的起飞日期输入框
        :param timeout: 超时时间（秒）
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = '//div[contains(@class, "flight_search_form")]//input[contains(@placeholder, "去程日期")]'
        return await self.get_locator(selector=selector, timeout=timeout)

    async def get_flight_query_btn(self, timeout: float = 5.0) -> Locator:
        """
        获取预订搜索页搜索栏的【查询机票】按钮
        :param timeout: 超时时间（秒）
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = '//div[contains(@class, "flight_search_form")]//button[@class="search_btn"]'
        return await self.get_locator(selector=selector, timeout=timeout)

    async def get_flight_info_plane(self, timeout: float = 5.0) -> Locator:
        """
        获取预订搜索页航班内容栏航班基本信息
        :param timeout: 超时时间（秒）
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = '//div[@class="airlines_cont"]//div[@class="text-center el-row"]'
        return await self.get_locator(selector=selector, timeout=timeout)

    async def get_flight_no(self, locator: Locator, timeout: float = 5.0) -> str:
        """
        获取预订搜索页航班编号
        :param timeout: 超时时间（秒）
        :param locator: flight_info_plane Locator 对象
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = 'xpath=(.//div[contains(@class, "flight_qda")]/span)[2]'
        sub_locator: Locator = await self.get_sub_locator(locator=locator, selector=selector, timeout=timeout)
        return (await sub_locator.inner_text(timeout=timeout)).strip()

    async def get_flight_product_nav(self, locator: Locator, product_type: str, timeout: float = 5.0) -> Locator:
        """
        获取预订搜索页航班产品类型nav
        :param timeout: 超时时间（秒）
        :param product_type: 产品类型
        :param locator: flight_info_plane Locator 对象
        :return: (是否存在, 错误信息|元素对象)
        """
        if product_type == "经济舱":
            index = 1
        elif product_type == "超级经济舱":
            index = 2
        elif product_type == "公务舱":
            index = 3
        else:
            raise EnvironmentError(f"产品类型参数值<{product_type}>无效，目前仅支持<经济舱，超级经济舱，公务舱>")
        selector: str = f'xpath=(.//div[contains(@class, "nav_item el-col el-col-24")])[{index}]'
        return await self.get_sub_locator(locator=locator, selector=selector, timeout=timeout)

    async def get_flight_products(self, timeout: float = 5.0) -> Dict[str, Any]:
        """
        获取预订搜索页航班内容栏所有航班产品
        :param timeout: 超时时间（秒）
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = '//div[@class="airlines_cont"]//div[@class="text-center cabinClasses_info el-row" and not(@style="display: none;")]/table/tbody/tr'
        products_locator: Locator = await self.get_locator(selector=selector, timeout=timeout)
        locators: List[Locator] = await products_locator.all()
        flight_products = dict()
        for locator in locators:
            try:
                booking_btn: Locator = await self._get_flight_product_booking_btn(locator=locator, timeout=timeout)
                amounts: Dict[str, Any] = await self._get_flight_product_price(locator=locator, timeout=timeout)
                seats_status: int = await self._get_flight_product_seats_status(locator=locator, timeout=timeout)
                cabin = await self._get_flight_product_cabin(locator=locator, timeout=timeout)
                flight_products[cabin] = dict(
                    amounts=amounts, cabin=cabin, seats_status=seats_status, booking_btn=booking_btn
                )
            except (PlaywrightError, PlaywrightTimeoutError, EnvironmentError, RuntimeError, Exception) as e:
                logger.warning(e)
                continue
        return flight_products

    async def _get_flight_product_cabin(self, locator: Locator, timeout: float = 5.0) -> str:
        """
        获取预订搜索页航班内容栏航班产品下的舱位
        :param timeout: 超时时间（秒）
        :param locator: flight_product Locator 对象
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = 'xpath=.//div[@class="ticket_clazz"]/div/span'
        locator: Locator = await self.get_sub_locator(locator=locator, selector=selector, timeout=timeout)
        text: str = (await locator.inner_text(timeout=timeout)).strip()
        match = re.search(r'\((.*?)舱\)', text)
        if match:
            return match.group(1).strip()
        raise RuntimeError(f"获取到的航班舱位信息：{text}，提取异常")

    async def _get_flight_product_price(self, locator: Locator, timeout: float = 5.0) -> Dict[str, Any]:
        """
        获取预订搜索页航班内容栏航班产品下的销售价格
        :param timeout: 超时时间（秒）
        :param locator: flight_product Locator 对象
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = 'xpath=.//div[@class="price_flex_top"]/span'
        locator: Locator = await self.get_sub_locator(locator=locator, selector=selector, timeout=timeout)
        amount_text: str = (await locator.inner_text(timeout=timeout)).strip()
        return convert_order_amount_text(amount_text=amount_text)

    async def _get_flight_product_booking_btn(self, locator: Locator, timeout: float = 5.0) -> Locator:
        """
        获取预订搜索页航班内容栏航班产品下的【购票】按钮
        :param timeout: 超时时间（秒）
        :param locator: flight_product Locator 对象
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = 'xpath=（.//td[@class="ticket_num"]/div/span)[1]'
        return await self.get_sub_locator(locator=locator, selector=selector, timeout=timeout)

    async def _get_flight_product_seats_status(self, locator: Locator, timeout: float = 5.0) -> int:
        """
        获取预订搜索页航班内容栏航班产品下的余票信息
        :param timeout: 超时时间（秒）
        :param locator: flight_product Locator 对象
        :return: (是否存在, 错误信息|元素对象)
        """
        selector: str = 'xpath=（.//td[@class="ticket_num"]/div/span)[2]'
        locator: Locator = await self.get_sub_locator(locator=locator, selector=selector, timeout=timeout)
        more_seats_text: str = (await locator.inner_text(timeout=timeout)).strip()
        match = re.search(r'\d+', more_seats_text)
        if match:
            return safe_convert_advanced(value=match.group(1).strip())
        else:
            return 999999
