"""

--------------------

PropertyGroups are the base class for dynamically defined sets of properties.

They can be used to extend existing blender data with your own types which can
be animated, accessed from the user interface and from python.

[NOTE]
The values assigned to blender data are saved to disk but the class
definitions are not, this means whenever you load blender the class needs
to be registered too.
This is best done by creating an add-on which loads on startup and registers
your properties.

[NOTE]
PropertyGroups must be registered before assigning them to blender data.

Property types used in class declarations are all in bpy.props



```../examples/bpy.types.PropertyGroup.py```

```../examples/bpy.types.AddonPreferences.1.py```


--------------------

This script is the UIList subclass used to show material slots, with a bunch of additional commentaries.

Notice the name of the class, this naming convention is similar as the one for panels or menus.

[NOTE]
UIList subclasses must be registered for blender to use them.

```../examples/bpy.types.UIList.1.py```


--------------------

This script is an extended version of the UIList

 subclass used to show vertex groups. It is not used 'as is',
because iterating over all vertices in a 'draw' function is a very bad idea for UI performances! However, it's a good
example of how to create/use filtering/reordering callbacks.

```../examples/bpy.types.UIList.2.py```


--------------------

This example shows an implementation of USDHook

 to extend USD
export functionalty.

One may optionally define one or both of the following callback functions
in the USDHook

 subclass.

Hook function on_export()

 is called before the USD export finalizes,
allowing modifications to the USD stage immediately before it is
saved.  This function takes as an argument an instance of an
internally defined class USDSceneExportContext

 which provides the
following accessors to the scene data:

* get_stage()

 returns the USD stage to be saved.
* get_depsgraph()

 returns the Blender scene dependency graph.

Hook function on_material_export()

 is called for each material that is exported,
allowing modifications to the USD material, such as shader generation.
It is called with three arguments:

-export_context

: An instance of the internally defined type USDMaterialExportContext

.
-bl_material

: The source Blender material.
-usd_material

: The target USD material to be exported.

USDMaterialExportContext

 implements a get_stage()

 function which returns the
USD stage to be saved.

Note that the target USD material might already have connected shaders created by the USD exporter or
by other material export hooks.

The hook functions should return True

 on success or False

 if the operation was bypasssed or
otherwise failed to complete.  Exceptions raised by these functions will be reported in Blender, with
the exception details printed to the console.

The USDHookExample

 class in this example impements an on_export()

 function to add custom data to
the stage's root layer and an on_material_export()

 function to create a simple MaterialX

 shader
on the USD material.

```../examples/bpy.types.USDHook.py```


--------------------

The NodeTree.poll function determines if a node tree is visible
in the given context (similar to how Panel.poll
and Menu.poll define visibility). If it returns False,
the node tree type will not be selectable in the node editor.

A typical condition for shader nodes would be to check the active render engine
of the scene and only show nodes of the renderer they are designed for.

```../examples/bpy.types.NodeTree.py```

bpy.types.*

:maxdepth: 1
:glob:

Shared Enum Types <bpy_types_enum_items/index>

:hidden:
:maxdepth: 1


--------------------

```../examples/bpy.types.RenderEngine.py```

Base class for integrating USD Hydra based renderers.


--------------------

```../examples/bpy.types.HydraRenderEngine.py```


--------------------

This script is a simple panel which will draw into the object properties
section.

Notice the 'CATEGORY_PT_name' Panel.bl_idname, this is a naming
convention for panels.

[NOTE]
Panel subclasses must be registered for blender to use them.

```../examples/bpy.types.Panel.py```


--------------------

This panel has a Panel.poll and Panel.draw_header function,
even though the contents is basic this closely resembles blenders panels.

```../examples/bpy.types.Panel.1.py```


--------------------

A mix-in parent class can be used to share common properties and
Menu.poll function.

```../examples/bpy.types.Panel.2.py```


--------------------

This script demonstrates basic operations on object like creating new
object, placing it into a view layer, selecting it and making it active.

```../examples/bpy.types.Object.py```


--------------------

This example demonstrates access to the evaluated ID (such as object, material, etc.) state from
an original ID.
This is needed every time one needs to access state with animation, constraints, and modifiers
taken into account.

```../examples/bpy.types.Depsgraph.1.py```


--------------------

This example demonstrates access to the original ID.
Such access is needed to check whether object is selected, or to compare pointers.

```../examples/bpy.types.Depsgraph.2.py```


--------------------

Sometimes it is needed to know all the instances with their matrices (for example, when writing an
exporter or a custom render engine).
This example shows how to access all objects and instances in the scene.

```../examples/bpy.types.Depsgraph.3.py```


--------------------

Function to get a mesh from any object with geometry. It is typically used by exporters, render
engines and tools that need to access the evaluated mesh as displayed in the viewport.

Object.to_mesh() is closely interacting with dependency graph: its behavior depends on whether it
is used on original or evaluated object.

When is used on original object, the result mesh is calculated from the object without taking
animation or modifiers into account:

* For meshes this is similar to duplicating the source mesh.
* For curves this disables own modifiers, and modifiers of objects used as bevel and taper.
* For metaballs this produces an empty mesh since polygonization is done as a modifier evaluation.

When is used on evaluated object all modifiers are taken into account.

[NOTE]
The result mesh is owned by the object. It can be freed by calling object.to_mesh_clear().

[NOTE]
The result mesh must be treated as temporary, and can not be referenced from objects in the main
database. If the mesh intended to be used in a persistent manner use bpy.data.meshes.new_from_object()
instead.

[NOTE]
If object does not have geometry (i.e. camera) the functions returns None.

```../examples/bpy.types.Depsgraph.4.py```


--------------------

Function to copy a new mesh from any object with geometry. The mesh is added to the main
database and can be referenced by objects. Typically used by tools that create new objects
or apply modifiers.

When is used on original object, the result mesh is calculated from the object without taking
animation or modifiers into account:

* For meshes this is similar to duplicating the source mesh.
* For curves this disables own modifiers, and modifiers of objects used as bevel and taper.
* For metaballs this produces an empty mesh since polygonization is done as a modifier evaluation.

When is used on evaluated object all modifiers are taken into account.

All the references (such as materials) are re-mapped to original. This ensures validity and
consistency of the main database.

[NOTE]
If object does not have geometry (i.e. camera) the functions returns None.

```../examples/bpy.types.Depsgraph.5.py```


--------------------

This example is a combination of all previous ones, and shows how to write a simple exporter
script.

```../examples/bpy.types.Depsgraph.6.py```


--------------------

Function to get a curve from text and curve objects. It is typically used by exporters, render
engines, and tools that need to access the curve representing the object.

The function takes the evaluated dependency graph as a required parameter and optionally a boolean
apply_modifiers which defaults to false. If apply_modifiers is true and the object is a curve object,
the spline deform modifiers are applied on the control points. Note that constructive modifiers and
modifiers that are not spline-enabled will not be applied. So modifiers like Array will not be applied
and deform modifiers that have Apply On Spline disabled will not be applied.

If the object is a text object. The text will be converted into a 3D curve and returned. Modifiers are
never applied on text objects and apply_modifiers will be ignored. If the object is neither a curve nor
a text object, an error will be reported.

[NOTE]
The resulting curve is owned by the object. It can be freed by calling object.to_curve_clear().

[NOTE]
The resulting curve must be treated as temporary, and can not be referenced from objects in the main
database.

```../examples/bpy.types.Depsgraph.7.py```


--------------------

The mesh data is accessed in object mode and intended for compact storage,
for more flexible mesh editing from python see bmesh.

Blender stores 4 main arrays to define mesh geometry.

* Mesh.vertices (3 points in space)
* Mesh.edges (reference 2 vertices)
* Mesh.loops (reference a single vertex and edge)
* Mesh.polygons: (reference a range of loops)

Each polygon references a slice in the loop array, this way, polygons do not store vertices or corner data such as UVs directly,
only a reference to loops that the polygon uses.

Mesh.loops, Mesh.uv_layers Mesh.vertex_colors are all aligned so the same polygon loop
indices can be used to find the UVs and vertex colors as with as the vertices.

To compare mesh API options see: NGons and Tessellation Faces <info_gotcha_mesh_faces>

This example script prints the vertices and UVs for each polygon, assumes the active object is a mesh with UVs.

```../examples/bpy.types.Mesh.py```


--------------------

Here is an example of a simple menu. Menus differ from panels in that they must
reference from a header, panel or another menu.

Notice the 'CATEGORY_MT_name' in  Menu.bl_idname, this is a naming
convention for menus.

[NOTE]
Menu subclasses must be registered before referencing them from blender.

[NOTE]
Menus have their Layout.operator_context initialized as
'EXEC_REGION_WIN' rather than 'INVOKE_DEFAULT' (see Execution Context <operator-execution_context>).
If the operator context needs to initialize inputs from the
Operator.invoke function, then this needs to be explicitly set.

```../examples/bpy.types.Menu.py```


--------------------

This menu demonstrates some different functions.

```../examples/bpy.types.Menu.1.py```


--------------------

When creating menus for add-ons you can't reference menus
in Blender's default scripts.
Instead, the add-on can add menu items to existing menus.

The function menu_draw acts like Menu.draw.

```../examples/bpy.types.Menu.2.py```


--------------------

Preset menus are simply a convention that uses a menu sub-class
to perform the common task of managing presets.

This example shows how you can add a preset menu.

This example uses the object display options,
however you can use properties defined by your own scripts too.

```../examples/bpy.types.Menu.3.py```


--------------------

This example enables you to insert your own menu entry into the common
right click menu that you get while hovering over a UI button (e.g. operator,
value field, color, string, etc.)

To make the example work, you have to first select an object
then right click on an user interface element (maybe a color in the
material properties) and choose *Execute Custom Action*.

Executing the operator will then print all values.

```../examples/bpy.types.Menu.4.py```


--------------------

The Image data-block is a shallow wrapper around image or video file(s)
(on disk, as packed data, or generated).

All actual data like the pixel buffer, size, resolution etc. is
cached in an imbuf.types.ImBuf image buffer (or several buffers
in some cases, like UDIM textures, multi-views, animations...).

Several properties and functions of the Image data-block are then actually
using/modifying its image buffer, and not the Image data-block itself.

[WARNING]
One key limitation is that image buffers are not shared between different
Image data-blocks, and they are not duplicated when copying an image.
So until a modified image buffer is saved on disk, duplicating its Image
data-block will not propagate the underlying buffer changes to the new Image.

This example script generates an Image data-block with a given size,
change its first pixel, rescale it, and duplicates the image.

The duplicated image still has the same size and colors as the original image
at its creation, all editing in the original image's buffer is 'lost' in its copy.

```../examples/bpy.types.Image.py```


--------------------

This script shows simple operator which prints a message.

Since the operator only has an Operator.execute function it takes no
user input.

The function should return {'FINISHED'}

 or {'CANCELLED'}

, the latter
meaning that operator execution was aborted without making any changes, and
saving an undo entry isn't neccesary. If an error is detected after some changes
have already been made, use the {'FINISHED'}

 return code, or the behavior
of undo will be confusing for the user.

[NOTE]
Operator subclasses must be registered before accessing them from blender.

```../examples/bpy.types.Operator.py```


--------------------

Operator.invoke is used to initialize the operator from the context
at the moment the operator is called.
invoke() is typically used to assign properties which are then used by
execute().
Some operators don't have an execute() function, removing the ability to be
repeated from a script or macro.

This example shows how to define an operator which gets mouse input to
execute a function and that this operator can be invoked or executed from
the python api.

Also notice this operator defines its own properties, these are different
to typical class properties because blender registers them with the
operator, to use as arguments when called, saved for operator undo/redo and
automatically added into the user interface.

```../examples/bpy.types.Operator.1.py```


--------------------

This example shows how an operator can use the file selector.

Notice the invoke function calls a window manager method and returns
{'RUNNING_MODAL'}

, this means the file selector stays open and the operator does not
exit immediately after invoke finishes.

The file selector runs the operator, calling Operator.execute when the
user confirms.

The Operator.poll function is optional, used to check if the operator
can run.

```../examples/bpy.types.Operator.2.py```


--------------------

This operator uses its Operator.invoke function to call a popup.

```../examples/bpy.types.Operator.3.py```


--------------------

By default operator properties use an automatic user interface layout.
If you need more control you can create your own layout with a
Operator.draw function.

This works like the Panel and Menu draw functions, its used
for dialogs and file selectors.

```../examples/bpy.types.Operator.4.py```


--------------------

This operator defines a Operator.modal function that will keep being
run to handle events until it returns {'FINISHED'}

 or {'CANCELLED'}

.

Modal operators run every time a new event is detected, such as a mouse click
or key press. Conversely, when no new events are detected, the modal operator
will not run. Modal operators are especially useful for interactive tools, an
operator can have its own state where keys toggle options as the operator runs.
Grab, Rotate, Scale, and Fly-Mode are examples of modal operators.

Operator.invoke is used to initialize the operator as being active
by returning {'RUNNING_MODAL'}

, initializing the modal loop.

Notice __init__()

 and __del__()

 are declared.
For other operator types they are not useful but for modal operators they will
be called before the Operator.invoke and after the operator finishes.

```../examples/bpy.types.Operator.5.py```


--------------------

You may want to have an operator prompt the user to select an item
from a search field, this can be done using bpy.types.Operator.invoke_search_popup.

```../examples/bpy.types.Operator.6.py```

"""

import typing
import collections.abc
import typing_extensions
import bl_operators.anim
import bl_operators.assets
import bl_operators.clip
import bl_operators.constraint
import bl_operators.file
import bl_operators.freestyle
import bl_operators.node
import bl_operators.object
import bl_operators.presets
import bl_operators.spreadsheet
import bl_operators.userpref
import bl_operators.view3d
import bl_operators.wm
import bl_ui
import bl_ui.asset_shelf
import bl_ui.generic_ui_list
import bl_ui.node_add_menu
import bl_ui.node_add_menu_compositor
import bl_ui.node_add_menu_geometry
import bl_ui.node_add_menu_shader
import bl_ui.node_add_menu_texture
import bl_ui.properties_collection
import bl_ui.properties_constraint
import bl_ui.properties_data_armature
import bl_ui.properties_data_bone
import bl_ui.properties_data_camera
import bl_ui.properties_data_curve
import bl_ui.properties_data_curves
import bl_ui.properties_data_empty
import bl_ui.properties_data_gpencil
import bl_ui.properties_data_grease_pencil
import bl_ui.properties_data_lattice
import bl_ui.properties_data_light
import bl_ui.properties_data_lightprobe
import bl_ui.properties_data_mesh
import bl_ui.properties_data_metaball
import bl_ui.properties_data_modifier
import bl_ui.properties_data_pointcloud
import bl_ui.properties_data_shaderfx
import bl_ui.properties_data_speaker
import bl_ui.properties_data_volume
import bl_ui.properties_freestyle
import bl_ui.properties_grease_pencil_common
import bl_ui.properties_mask_common
import bl_ui.properties_material
import bl_ui.properties_material_gpencil
import bl_ui.properties_object
import bl_ui.properties_output
import bl_ui.properties_paint_common
import bl_ui.properties_particle
import bl_ui.properties_physics_cloth
import bl_ui.properties_physics_common
import bl_ui.properties_physics_dynamicpaint
import bl_ui.properties_physics_field
import bl_ui.properties_physics_fluid
import bl_ui.properties_physics_geometry_nodes
import bl_ui.properties_physics_rigidbody
import bl_ui.properties_physics_rigidbody_constraint
import bl_ui.properties_physics_softbody
import bl_ui.properties_render
import bl_ui.properties_scene
import bl_ui.properties_texture
import bl_ui.properties_view_layer
import bl_ui.properties_workspace
import bl_ui.properties_world
import bl_ui.space_clip
import bl_ui.space_console
import bl_ui.space_dopesheet
import bl_ui.space_filebrowser
import bl_ui.space_graph
import bl_ui.space_image
import bl_ui.space_info
import bl_ui.space_nla
import bl_ui.space_node
import bl_ui.space_outliner
import bl_ui.space_properties
import bl_ui.space_sequencer
import bl_ui.space_spreadsheet
import bl_ui.space_statusbar
import bl_ui.space_text
import bl_ui.space_time
import bl_ui.space_toolsystem_common
import bl_ui.space_toolsystem_toolbar
import bl_ui.space_topbar
import bl_ui.space_userpref
import bl_ui.space_view3d
import bl_ui.space_view3d_toolbar
import bpy._typing.rna_enums
import mathutils

class bpy_struct[_GenericType1]:
    """built-in base class for all classes in bpy.types."""

    id_data: typing.Any
    """ The `bpy.types.ID` object this datablock is from or None, (not available for all data types)"""

    bl_rna: BlenderRNA
    """ 

    :type: BlenderRNA
    """

    def as_pointer(self) -> int:
        """Returns the memory address which holds a pointer to Blender's internal data

        :return: int (memory address).
        :rtype: int
        """

    def driver_add(self, path: str | None, index: int | None = -1) -> FCurve:
        """Adds driver(s) to the given property

        :param path: path to the property to drive, analogous to the fcurve's data path.
        :type path: str | None
        :param index: array index of the property drive. Defaults to -1 for all indices or a single channel if the property is not an array.
        :type index: int | None
        :return: The driver(s) added.
        :rtype: FCurve
        """

    def driver_remove(self, path: str | None, index: int | None = -1) -> bool:
        """Remove driver(s) from the given property

        :param path: path to the property to drive, analogous to the fcurve's data path.
        :type path: str | None
        :param index: array index of the property drive. Defaults to -1 for all indices or a single channel if the property is not an array.
        :type index: int | None
        :return: Success of driver removal.
        :rtype: bool
        """

    def get(self, key: str | None, default=None):
        """Returns the value of the custom property assigned to key or default
        when not found (matches Python's dictionary function of the same name).

                :param key: The key associated with the custom property.
                :type key: str | None
                :param default: Optional argument for the value to return if
        key is not found.
        """

    def id_properties_clear(self):
        """

        :return: Remove the parent group for an RNA struct's custom IDProperties.
        """

    def id_properties_ensure(self):
        """

        :return: the parent group for an RNA struct's custom IDProperties.
        """

    def id_properties_ui(self, key):
        """

        :param key: String name of the property.
        :return: Return an object used to manage an IDProperty's UI data.
        """

    def is_property_hidden(self, property) -> bool:
        """Check if a property is hidden.

        :param property:
        :return: True when the property is hidden.
        :rtype: bool
        """

    def is_property_overridable_library(self, property) -> bool:
        """Check if a property is overridable.

        :param property:
        :return: True when the property is overridable.
        :rtype: bool
        """

    def is_property_readonly(self, property) -> bool:
        """Check if a property is readonly.

        :param property:
        :return: True when the property is readonly (not writable).
        :rtype: bool
        """

    def is_property_set(self, property, ghost: bool | None = True) -> bool:
        """Check if a property is set, use for testing operator properties.

                :param property:
                :param ghost: Used for operators that re-run with previous settings.
        In this case the property is not marked as set,
        yet the value from the previous execution is used.

        In rare cases you may want to set this option to false.
                :type ghost: bool | None
                :return: True when the property has been set.
                :rtype: bool
        """

    def items(self):
        """Returns the items of this objects custom properties (matches Python's
        dictionary function of the same name).

                :return: custom property key, value pairs.
        """

    def keyframe_delete(
        self,
        data_path: str | None,
        index: int | None = -1,
        frame: float | None = bpy.context.scene.frame_current,
        group: str | None = "",
    ) -> bool:
        """Remove a keyframe from this properties fcurve.

        :param data_path: path to the property to remove a key, analogous to the fcurve's data path.
        :type data_path: str | None
        :param index: array index of the property to remove a key. Defaults to -1 removing all indices or a single channel if the property is not an array.
        :type index: int | None
        :param frame: The frame on which the keyframe is deleted, defaulting to the current frame.
        :type frame: float | None
        :param group: The name of the group the F-Curve should be added to if it doesn't exist yet.
        :type group: str | None
        :return: Success of keyframe deletion.
        :rtype: bool
        """

    def keyframe_insert(
        self,
        data_path: str | None,
        index: int | None = -1,
        frame: float | None = bpy.context.scene.frame_current,
        group: str | None = "",
        options=set(),
    ) -> bool:
        """Insert a keyframe on the property given, adding fcurves and animation data when necessary.This is the most simple example of inserting a keyframe from python.Note that when keying data paths which contain nested properties this must be
        done from the `ID` subclass, in this case the `Armature` rather
        than the bone.

                :param data_path: path to the property to key, analogous to the fcurve's data path.
                :type data_path: str | None
                :param index: array index of the property to key.
        Defaults to -1 which will key all indices or a single channel if the property is not an array.
                :type index: int | None
                :param frame: The frame on which the keyframe is inserted, defaulting to the current frame.
                :type frame: float | None
                :param group: The name of the group the F-Curve should be added to if it doesn't exist yet.
                :type group: str | None
                :param options: Optional set of flags:

        INSERTKEY_NEEDED Only insert keyframes where they're needed in the relevant F-Curves.

        INSERTKEY_VISUAL Insert keyframes based on 'visual transforms'.

        INSERTKEY_XYZ_TO_RGB Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis.

        INSERTKEY_REPLACE Only replace already existing keyframes.

        INSERTKEY_AVAILABLE Only insert into already existing F-Curves.

        INSERTKEY_CYCLE_AWARE Take cyclic extrapolation into account (Cycle-Aware Keying option).
                :return: Success of keyframe insertion.
                :rtype: bool
        """

    def keys(self):
        """Returns the keys of this objects custom properties (matches Python's
        dictionary function of the same name).

                :return: custom property keys.
        """

    def path_from_id(self, property: str | None = "") -> str:
        """Returns the data path from the ID to this object (string).

                :param property: Optional property name which can be used if the path is
        to a property of this object.
                :type property: str | None
                :return: The path from `bpy.types.bpy_struct.id_data`
        to this struct and property (when given).
                :rtype: str
        """

    def path_resolve(self, path: str | None, coerce: bool | None = True):
        """Returns the property from the path, raise an exception when not found.

                :param path: path which this property resolves.
                :type path: str | None
                :param coerce: optional argument, when True, the property will be converted
        into its Python representation.
                :type coerce: bool | None
        """

    def pop(self, key: str | None, default=None):
        """Remove and return the value of the custom property assigned to key or default
        when not found (matches Python's dictionary function of the same name).

                :param key: The key associated with the custom property.
                :type key: str | None
                :param default: Optional argument for the value to return if
        key is not found.
        """

    def property_overridable_library_set(self, property, overridable) -> bool:
        """Define a property as overridable or not (only for custom properties!).

        :param property:
        :param overridable:
        :return: True when the overridable status of the property was successfully set.
        :rtype: bool
        """

    def property_unset(self, property):
        """Unset a property, will use default value afterward.

        :param property:
        """

    def type_recast(self):
        """Return a new instance, this is needed because types
        such as textures can be changed at runtime.

                :return: a new instance of this object with the type initialized again.
        """

    def values(self):
        """Returns the values of this objects custom properties (matches Python's
        dictionary function of the same name).

                :return: custom property values.
        """

    def __getitem__(self, key: int | str) -> typing.Any:
        """

        :param key:
        :type key: int | str
        :return:
        :rtype: typing.Any
        """

    def __setitem__(self, key: int | str, value: typing.Any):
        """

        :param key:
        :type key: int | str
        :param value:
        :type value: typing.Any
        """

    def __delitem__(self, key: int | str):
        """

        :param key:
        :type key: int | str
        """

class bpy_prop_collection[_GenericType1]:
    """built-in class used for all collections."""

    def find(self, key: str) -> int:
        """Returns the index of a key in a collection or -1 when not found
        (matches Python's string find function of the same name).

                :param key: The identifier for the collection member.
                :type key: str
                :return: index of the key.
                :rtype: int
        """

    def foreach_get(
        self,
        attr: str,
        seq: collections.abc.MutableSequence[bool]
        | collections.abc.MutableSequence[int]
        | collections.abc.MutableSequence[float]
        | typing_extensions.Buffer,
    ):
        """This is a function to give fast access to attributes within a collection.Only works for 'basic type' properties (bool, int and float)!
        Multi-dimensional arrays (like array of vectors) will be flattened into seq.

                :param attr:
                :type attr: str
                :param seq:
                :type seq: collections.abc.MutableSequence[bool] | collections.abc.MutableSequence[int] | collections.abc.MutableSequence[float] | typing_extensions.Buffer
        """

    def foreach_set(
        self,
        attr: str,
        seq: collections.abc.Sequence[bool]
        | collections.abc.Sequence[int]
        | collections.abc.Sequence[float]
        | typing_extensions.Buffer,
    ):
        """This is a function to give fast access to attributes within a collection.Only works for 'basic type' properties (bool, int and float)!
        seq must be uni-dimensional, multi-dimensional arrays (like array of vectors) will be re-created from it.

                :param attr:
                :type attr: str
                :param seq:
                :type seq: collections.abc.Sequence[bool] | collections.abc.Sequence[int] | collections.abc.Sequence[float] | typing_extensions.Buffer
        """

    def get[_GenericType2](
        self, key: str, default: _GenericType2 = None
    ) -> _GenericType1 | _GenericType2:
        """Returns the value of the item assigned to key or default when not found
        (matches Python's dictionary function of the same name).

                :param key: The identifier for the collection member.
                :type key: str
                :param default: Optional argument for the value to return if
        key is not found.
                :type default: _GenericType2
                :return:
                :rtype: _GenericType1 | _GenericType2
        """

    def items(self) -> list[tuple[str, _GenericType1]]:
        """Return the identifiers of collection members
        (matching Python's dict.items() functionality).

                :return: (key, value) pairs for each member of this collection.
                :rtype: list[tuple[str, _GenericType1]]
        """

    def keys(self) -> list[str]:
        """Return the identifiers of collection members
        (matching Python's dict.keys() functionality).

                :return: the identifiers for each member of this collection.
                :rtype: list[str]
        """

    def values(self) -> list[_GenericType1 | None]:
        """Return the values of collection
        (matching Python's dict.values() functionality).

                :return: the members of this collection.
                :rtype: list[_GenericType1 | None]
        """

    @typing.overload
    def __getitem__(self, key: int | str) -> _GenericType1:
        """

        :param key:
        :type key: int | str
        :return:
        :rtype: _GenericType1
        """

    @typing.overload
    def __getitem__(self, key: slice) -> list[_GenericType1, ...]:
        """

        :param key:
        :type key: slice
        :return:
        :rtype: list[_GenericType1, ...]
        """

    def __setitem__(self, key: int | str, value: _GenericType1 | None):
        """

        :param key:
        :type key: int | str
        :param value:
        :type value: _GenericType1 | None
        """

    @typing.overload
    def __setitem__(self, key: int, value: _GenericType1 | None):
        """

        :param key:
        :type key: int
        :param value:
        :type value: _GenericType1 | None
        """

    @typing.overload
    def __setitem__(self, key: str, value: _GenericType1):
        """

        :param key:
        :type key: str
        :param value:
        :type value: _GenericType1
        """

    def __iter__(self) -> collections.abc.Iterator[_GenericType1]:
        """

        :return:
        :rtype: collections.abc.Iterator[_GenericType1]
        """

    def __next__(self) -> _GenericType1:
        """

        :return:
        :rtype: _GenericType1
        """

    def __len__(self) -> int:
        """

        :return:
        :rtype: int
        """

    def __contains__(self, key: str | tuple[str, ...] | _GenericType1) -> bool:
        """

        :param key:
        :type key: str | tuple[str, ...] | _GenericType1
        :return:
        :rtype: bool
        """

class bpy_prop_array[_GenericType1]:
    def __get__(self, instance, owner) -> bpy_prop_array[_GenericType1]:
        """

        :param instance:
        :param owner:
        :return:
        :rtype: bpy_prop_array[_GenericType1]
        """

    def __set__(self, instance, value: collections.abc.Iterable[_GenericType1]):
        """

        :param instance:
        :param value:
        :type value: collections.abc.Iterable[_GenericType1]
        """

    def foreach_get(
        self,
        seq: collections.abc.MutableSequence[_GenericType1] | typing_extensions.Buffer,
    ):
        """

        :param seq:
        :type seq: collections.abc.MutableSequence[_GenericType1] | typing_extensions.Buffer
        """

    def foreach_set(
        self, seq: collections.abc.Sequence[_GenericType1] | typing_extensions.Buffer
    ):
        """

        :param seq:
        :type seq: collections.abc.Sequence[_GenericType1] | typing_extensions.Buffer
        """

    @typing.overload
    def __getitem__(self, key: int) -> _GenericType1:
        """

        :param key:
        :type key: int
        :return:
        :rtype: _GenericType1
        """

    @typing.overload
    def __getitem__(self, key: slice) -> list[_GenericType1, ...]:
        """

        :param key:
        :type key: slice
        :return:
        :rtype: list[_GenericType1, ...]
        """

    @typing.overload
    def __setitem__(self, key: int, value: _GenericType1):
        """

        :param key:
        :type key: int
        :param value:
        :type value: _GenericType1
        """

    @typing.overload
    def __setitem__(self, key: slice, value: collections.abc.Iterable[_GenericType1]):
        """

        :param key:
        :type key: slice
        :param value:
        :type value: collections.abc.Iterable[_GenericType1]
        """

    def __delitem__(self, key: int):
        """

        :param key:
        :type key: int
        """

    def __iter__(self) -> collections.abc.Iterator[_GenericType1]:
        """

        :return:
        :rtype: collections.abc.Iterator[_GenericType1]
        """

    def __next__(self) -> _GenericType1:
        """

        :return:
        :rtype: _GenericType1
        """

    def __len__(self) -> int:
        """

        :return:
        :rtype: int
        """

class AOVs(bpy_prop_collection[AOV], bpy_struct):
    """Collection of AOVs"""

    def add(self) -> AOV:
        """add

        :return: Newly created AOV
        :rtype: AOV
        """

    def remove(self, aov: AOV):
        """Remove an AOV

        :param aov: AOV to remove
        :type aov: AOV
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ActionFCurves(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of action F-Curves"""

    def new(
        self, data_path: str, index: int | None = 0, action_group: str = ""
    ) -> FCurve:
        """Add an F-Curve to the action

        :param data_path: Data Path, F-Curve data path to use
        :type data_path: str
        :param index: Index, Array index
        :type index: int | None
        :param action_group: Action Group, Acton group to add this F-Curve into
        :type action_group: str
        :return: Newly created F-Curve
        :rtype: FCurve
        """

    def find(self, data_path: str, index: int | None = 0) -> FCurve:
        """Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the action.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str
        :param index: Index, Array index
        :type index: int | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """

    def remove(self, fcurve: FCurve):
        """Remove F-Curve

        :param fcurve: F-Curve to remove
        :type fcurve: FCurve
        """

    def clear(self):
        """Remove all F-Curves"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ActionGroups(bpy_prop_collection[ActionGroup], bpy_struct):
    """Collection of action groups"""

    def new(self, name: str) -> ActionGroup:
        """Create a new action group and add it to the action

        :param name: New name for the action group
        :type name: str
        :return: Newly created action group
        :rtype: ActionGroup
        """

    def remove(self, action_group: ActionGroup):
        """Remove action group

        :param action_group: Action group to remove
        :type action_group: ActionGroup
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ActionPoseMarkers(bpy_prop_collection[TimelineMarker], bpy_struct):
    """Collection of timeline markers"""

    active: TimelineMarker | None
    """ Active pose marker for this action

    :type: TimelineMarker | None
    """

    active_index: int | None
    """ Index of active pose marker

    :type: int | None
    """

    def new(self, name: str) -> TimelineMarker:
        """Add a pose marker to the action

        :param name: New name for the marker (not unique)
        :type name: str
        :return: Newly created marker
        :rtype: TimelineMarker
        """

    def remove(self, marker: TimelineMarker):
        """Remove a timeline marker

        :param marker: Timeline marker to remove
        :type marker: TimelineMarker
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Addons(bpy_prop_collection[Addon], bpy_struct):
    """Collection of add-ons"""

    @classmethod
    def new(cls) -> Addon:
        """Add a new add-on

        :return: Add-on data
        :rtype: Addon
        """

    @classmethod
    def remove(cls, addon: Addon):
        """Remove add-on

        :param addon: Add-on to remove
        :type addon: Addon
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AnimDataDrivers(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of Driver F-Curves"""

    def new(self, data_path: str, index: int | None = 0) -> FCurve:
        """new

        :param data_path: Data Path, F-Curve data path to use
        :type data_path: str
        :param index: Index, Array index
        :type index: int | None
        :return: Newly Driver F-Curve
        :rtype: FCurve
        """

    def remove(self, driver: FCurve):
        """remove

        :param driver:
        :type driver: FCurve
        """

    def from_existing(self, src_driver: FCurve | None = None) -> FCurve:
        """Add a new driver given an existing one

        :param src_driver: Existing Driver F-Curve to use as template for a new one
        :type src_driver: FCurve | None
        :return: New Driver F-Curve
        :rtype: FCurve
        """

    def find(self, data_path: str, index: int | None = 0) -> FCurve:
        """Find a driver F-Curve. Note that this function performs a linear scan of all driver F-Curves.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str
        :param index: Index, Array index
        :type index: int | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AreaSpaces(bpy_prop_collection[Space], bpy_struct):
    """Collection of spaces"""

    active: Space | None
    """ Space currently being displayed in this area

    :type: Space | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ArmatureBones(bpy_prop_collection[Bone], bpy_struct):
    """Collection of armature bones"""

    active: Bone | None
    """ Armature's active bone

    :type: Bone | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ArmatureConstraintTargets(bpy_prop_collection[ConstraintTargetBone], bpy_struct):
    """Collection of target bones and weights"""

    def new(self) -> ConstraintTargetBone:
        """Add a new target to the constraint

        :return: New target bone
        :rtype: ConstraintTargetBone
        """

    def remove(self, target: ConstraintTargetBone):
        """Delete target from the constraint

        :param target: Target to remove
        :type target: ConstraintTargetBone
        """

    def clear(self):
        """Delete all targets from object"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ArmatureEditBones(bpy_prop_collection[EditBone], bpy_struct):
    """Collection of armature edit bones"""

    active: EditBone | None
    """ Armatures active edit bone

    :type: EditBone | None
    """

    def new(self, name: str) -> EditBone:
        """Add a new bone

        :param name: New name for the bone
        :type name: str
        :return: Newly created edit bone
        :rtype: EditBone
        """

    def remove(self, bone: EditBone):
        """Remove an existing bone from the armature

        :param bone: EditBone to remove
        :type bone: EditBone
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AssetTags(bpy_prop_collection[AssetTag], bpy_struct):
    """Collection of custom asset tags"""

    def new(self, name: str, skip_if_exists: bool | None = False) -> AssetTag:
        """Add a new tag to this asset

        :param name: Name
        :type name: str
        :param skip_if_exists: Skip if Exists, Do not add a new tag if one of the same type already exists
        :type skip_if_exists: bool | None
        :return: New tag
        :rtype: AssetTag
        """

    def remove(self, tag: AssetTag):
        """Remove an existing tag from this asset

        :param tag: Removed tag
        :type tag: AssetTag
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AttributeGroup(bpy_prop_collection[Attribute], bpy_struct):
    """Group of geometry attributes"""

    active: Attribute | None
    """ Active attribute

    :type: Attribute | None
    """

    active_color: Attribute | None
    """ Active color attribute for display and editing

    :type: Attribute | None
    """

    active_color_index: int | None
    """ Active color attribute index

    :type: int | None
    """

    active_color_name: str | None
    """ The name of the active color attribute for display and editing

    :type: str | None
    """

    active_index: int | None
    """ Active attribute index

    :type: int | None
    """

    default_color_name: str
    """ The name of the default color attribute used as a fallback for rendering

    :type: str
    """

    render_color_index: int
    """ The index of the color attribute used as a fallback for rendering

    :type: int
    """

    def new(
        self,
        name: str,
        type: bpy._typing.rna_enums.AttributeTypeItems | None,
        domain: bpy._typing.rna_enums.AttributeDomainItems | None,
    ) -> Attribute:
        """Add attribute to geometry

        :param name: Name, Name of geometry attribute
        :type name: str
        :param type: Type, Attribute type
        :type type: bpy._typing.rna_enums.AttributeTypeItems | None
        :param domain: Domain, Type of element that attribute is stored on
        :type domain: bpy._typing.rna_enums.AttributeDomainItems | None
        :return: New geometry attribute
        :rtype: Attribute
        """

    def remove(self, attribute: Attribute):
        """Remove attribute from geometry

        :param attribute: Geometry Attribute
        :type attribute: Attribute
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataActions(bpy_prop_collection[Action], bpy_struct):
    """Collection of actions"""

    def new(self, name: str) -> Action:
        """Add a new action to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New action data-block
        :rtype: Action
        """

    def remove(
        self,
        action: Action,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove an action from the current blendfile

        :param action: Action to remove
        :type action: Action
        :param do_unlink: Unlink all usages of this action before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this action
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this action
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataArmatures(bpy_prop_collection[Armature], bpy_struct):
    """Collection of armatures"""

    def new(self, name: str) -> Armature:
        """Add a new armature to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New armature data-block
        :rtype: Armature
        """

    def remove(
        self,
        armature: Armature,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove an armature from the current blendfile

        :param armature: Armature to remove
        :type armature: Armature
        :param do_unlink: Unlink all usages of this armature before deleting it (WARNING: will also delete objects instancing that armature data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this armature data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this armature data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataBrushes(bpy_prop_collection[Brush], bpy_struct):
    """Collection of brushes"""

    def new(
        self,
        name: str,
        mode: bpy._typing.rna_enums.ObjectModeItems | None = "TEXTURE_PAINT",
    ) -> Brush:
        """Add a new brush to the main database

        :param name: New name for the data-block
        :type name: str
        :param mode: Paint Mode for the new brush
        :type mode: bpy._typing.rna_enums.ObjectModeItems | None
        :return: New brush data-block
        :rtype: Brush
        """

    def remove(
        self,
        brush: Brush,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a brush from the current blendfile

        :param brush: Brush to remove
        :type brush: Brush
        :param do_unlink: Unlink all usages of this brush before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this brush
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this brush
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def create_gpencil_data(self, brush: Brush):
        """Add grease pencil brush settings

        :param brush: Brush
        :type brush: Brush
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataCacheFiles(bpy_prop_collection[CacheFile], bpy_struct):
    """Collection of cache files"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataCameras(bpy_prop_collection[Camera], bpy_struct):
    """Collection of cameras"""

    def new(self, name: str) -> Camera:
        """Add a new camera to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New camera data-block
        :rtype: Camera
        """

    def remove(
        self,
        camera: Camera,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a camera from the current blendfile

        :param camera: Camera to remove
        :type camera: Camera
        :param do_unlink: Unlink all usages of this camera before deleting it (WARNING: will also delete objects instancing that camera data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this camera
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this camera
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataCollections(bpy_prop_collection[Collection], bpy_struct):
    """Collection of collections"""

    def new(self, name: str) -> Collection:
        """Add a new collection to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New collection data-block
        :rtype: Collection
        """

    def remove(
        self,
        collection: Collection,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a collection from the current blendfile

        :param collection: Collection to remove
        :type collection: Collection
        :param do_unlink: Unlink all usages of this collection before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this collection
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this collection
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataCurves(bpy_prop_collection[Curve], bpy_struct):
    """Collection of curves"""

    def new(
        self, name: str, type: bpy._typing.rna_enums.ObjectTypeCurveItems | None
    ) -> Curve:
        """Add a new curve to the main database

        :param name: New name for the data-block
        :type name: str
        :param type: Type, The type of curve to add
        :type type: bpy._typing.rna_enums.ObjectTypeCurveItems | None
        :return: New curve data-block
        :rtype: Curve
        """

    def remove(
        self,
        curve: Curve,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a curve from the current blendfile

        :param curve: Curve to remove
        :type curve: Curve
        :param do_unlink: Unlink all usages of this curve before deleting it (WARNING: will also delete objects instancing that curve data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this curve data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this curve data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataFonts(bpy_prop_collection[VectorFont], bpy_struct):
    """Collection of fonts"""

    def load(self, filepath: str, check_existing: bool | None = False) -> VectorFont:
        """Load a new font into the main database

        :param filepath: path of the font to load
        :type filepath: str
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | None
        :return: New font data-block
        :rtype: VectorFont
        """

    def remove(
        self,
        vfont: VectorFont,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a font from the current blendfile

        :param vfont: Font to remove
        :type vfont: VectorFont
        :param do_unlink: Unlink all usages of this font before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this font
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this font
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataGreasePencils(bpy_prop_collection[GreasePencil], bpy_struct):
    """Collection of grease pencils"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def new(self, name: str) -> GreasePencil:
        """Add a new grease pencil datablock to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New grease pencil data-block
        :rtype: GreasePencil
        """

    def remove(
        self,
        grease_pencil: GreasePencil,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a grease pencil instance from the current blendfile

        :param grease_pencil: Grease Pencil to remove
        :type grease_pencil: GreasePencil
        :param do_unlink: Unlink all usages of this grease pencil before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this grease pencil
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this grease pencil
        :type do_ui_user: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataHairCurves(bpy_prop_collection[Curves], bpy_struct):
    """Collection of hair curves"""

    def new(self, name: str) -> Curves:
        """Add a new hair to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New curves data-block
        :rtype: Curves
        """

    def remove(
        self,
        curves: Curves,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a curves data-block from the current blendfile

        :param curves: Curves data-block to remove
        :type curves: Curves
        :param do_unlink: Unlink all usages of this curves before deleting it (WARNING: will also delete objects instancing that curves data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this curves data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this curves data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataImages(bpy_prop_collection[Image], bpy_struct):
    """Collection of images"""

    def new(
        self,
        name: str,
        width: int | None,
        height: int | None,
        alpha: bool | None = False,
        float_buffer: bool | None = False,
        stereo3d: bool | None = False,
        is_data: bool | None = False,
        tiled: bool | None = False,
    ) -> Image:
        """Add a new image to the main database

        :param name: New name for the data-block
        :type name: str
        :param width: Width of the image
        :type width: int | None
        :param height: Height of the image
        :type height: int | None
        :param alpha: Alpha, Use alpha channel
        :type alpha: bool | None
        :param float_buffer: Float Buffer, Create an image with floating-point color
        :type float_buffer: bool | None
        :param stereo3d: Stereo 3D, Create left and right views
        :type stereo3d: bool | None
        :param is_data: Is Data, Create image with non-color data color space
        :type is_data: bool | None
        :param tiled: Tiled, Create a tiled image
        :type tiled: bool | None
        :return: New image data-block
        :rtype: Image
        """

    def load(self, filepath: str, check_existing: bool | None = False) -> Image:
        """Load a new image into the main database

        :param filepath: Path of the file to load
        :type filepath: str
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | None
        :return: New image data-block
        :rtype: Image
        """

    def remove(
        self,
        image: Image,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove an image from the current blendfile

        :param image: Image to remove
        :type image: Image
        :param do_unlink: Unlink all usages of this image before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this image
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this image
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataLattices(bpy_prop_collection[Lattice], bpy_struct):
    """Collection of lattices"""

    def new(self, name: str) -> Lattice:
        """Add a new lattice to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New lattice data-block
        :rtype: Lattice
        """

    def remove(
        self,
        lattice: Lattice,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a lattice from the current blendfile

        :param lattice: Lattice to remove
        :type lattice: Lattice
        :param do_unlink: Unlink all usages of this lattice before deleting it (WARNING: will also delete objects instancing that lattice data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this lattice data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this lattice data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataLibraries(bpy_prop_collection[Library], bpy_struct):
    """Collection of libraries"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def remove(
        self,
        library: Library,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a library from the current blendfile

        :param library: Library to remove
        :type library: Library
        :param do_unlink: Unlink all usages of this library before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this library
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this library
        :type do_ui_user: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

    def load(
        self,
        filepath: bytes | str | None,
        link: bool | None = False,
        relative: bool | None = False,
        assets_only: bool | None = False,
        create_liboverrides: bool | None = False,
        reuse_liboverrides: bool | None = False,
        create_liboverrides_runtime: bool | None = False,
    ):
        """Returns a context manager which exposes 2 library objects on entering.
        Each object has attributes matching bpy.data which are lists of strings to be linked.

                :param filepath: The path to a blend file.
                :type filepath: bytes | str | None
                :param link: When False reference to the original file is lost.
                :type link: bool | None
                :param relative: When True the path is stored relative to the open blend file.
                :type relative: bool | None
                :param assets_only: If True, only list data-blocks marked as assets.
                :type assets_only: bool | None
                :param create_liboverrides: If True and link is True, liboverrides will
        be created for linked data.
                :type create_liboverrides: bool | None
                :param reuse_liboverrides: If True and create_liboverride is True,
        search for existing liboverride first.
                :type reuse_liboverrides: bool | None
                :param create_liboverrides_runtime: If True and create_liboverride is True,
        create (or search for existing) runtime liboverride.
                :type create_liboverrides_runtime: bool | None
        """

    def write(
        self,
        filepath: bytes | str | None,
        datablocks: set | None,
        path_remap: str | None = False,
        fake_user: bool | None = False,
        compress: bool | None = False,
    ):
        """Write data-blocks into a blend file.

                :param filepath: The path to write the blend-file.
                :type filepath: bytes | str | None
                :param datablocks: set of data-blocks (`bpy.types.ID` instances).
                :type datablocks: set | None
                :param path_remap: Optionally remap paths when writing the file:

        NONE No path manipulation (default).

        RELATIVE Remap paths that are already relative to the new location.

        RELATIVE_ALL Remap all paths to be relative to the new location.

        ABSOLUTE Make all paths absolute on writing.
                :type path_remap: str | None
                :param fake_user: When True, data-blocks will be written with fake-user flag enabled.
                :type fake_user: bool | None
                :param compress: When True, write a compressed blend file.
                :type compress: bool | None
        """

class BlendDataLights(bpy_prop_collection[Light], bpy_struct):
    """Collection of lights"""

    def new(
        self, name: str, type: bpy._typing.rna_enums.LightTypeItems | None
    ) -> Light:
        """Add a new light to the main database

        :param name: New name for the data-block
        :type name: str
        :param type: Type, The type of light to add
        :type type: bpy._typing.rna_enums.LightTypeItems | None
        :return: New light data-block
        :rtype: Light
        """

    def remove(
        self,
        light: Light,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a light from the current blendfile

        :param light: Light to remove
        :type light: Light
        :param do_unlink: Unlink all usages of this light before deleting it (WARNING: will also delete objects instancing that light data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this light data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this light data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataLineStyles(bpy_prop_collection[FreestyleLineStyle], bpy_struct):
    """Collection of line styles"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def new(self, name: str) -> FreestyleLineStyle:
        """Add a new line style instance to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New line style data-block
        :rtype: FreestyleLineStyle
        """

    def remove(
        self,
        linestyle: FreestyleLineStyle,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a line style instance from the current blendfile

        :param linestyle: Line style to remove
        :type linestyle: FreestyleLineStyle
        :param do_unlink: Unlink all usages of this line style before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this line style
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this line style
        :type do_ui_user: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataMasks(bpy_prop_collection[Mask], bpy_struct):
    """Collection of masks"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def new(self, name: str) -> Mask:
        """Add a new mask with a given name to the main database

        :param name: Mask, Name of new mask data-block
        :type name: str
        :return: New mask data-block
        :rtype: Mask
        """

    def remove(
        self,
        mask: Mask,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a mask from the current blendfile

        :param mask: Mask to remove
        :type mask: Mask
        :param do_unlink: Unlink all usages of this mask before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this mask
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this mask
        :type do_ui_user: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataMaterials(bpy_prop_collection[Material], bpy_struct):
    """Collection of materials"""

    def new(self, name: str) -> Material:
        """Add a new material to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New material data-block
        :rtype: Material
        """

    def create_gpencil_data(self, material: Material):
        """Add grease pencil material settings

        :param material: Material
        :type material: Material
        """

    def remove_gpencil_data(self, material: Material):
        """Remove grease pencil material settings

        :param material: Material
        :type material: Material
        """

    def remove(
        self,
        material: Material,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a material from the current blendfile

        :param material: Material to remove
        :type material: Material
        :param do_unlink: Unlink all usages of this material before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this material
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this material
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataMeshes(bpy_prop_collection[Mesh], bpy_struct):
    """Collection of meshes"""

    def new(self, name: str) -> Mesh:
        """Add a new mesh to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New mesh data-block
        :rtype: Mesh
        """

    def new_from_object(
        self,
        object: Object,
        preserve_all_data_layers: bool | None = False,
        depsgraph: Depsgraph | None = None,
    ) -> Mesh:
        """Add a new mesh created from given object (undeformed geometry if object is original, and final evaluated geometry, with all modifiers etc., if object is evaluated)

        :param object: Object to create mesh from
        :type object: Object
        :param preserve_all_data_layers: Preserve all data layers in the mesh, like UV maps and vertex groups. By default Blender only computes the subset of data layers needed for viewport display and rendering, for better performance
        :type preserve_all_data_layers: bool | None
        :param depsgraph: Dependency Graph, Evaluated dependency graph which is required when preserve_all_data_layers is true
        :type depsgraph: Depsgraph | None
        :return: Mesh created from object, remove it if it is only used for export
        :rtype: Mesh
        """

    def remove(
        self,
        mesh: Mesh,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a mesh from the current blendfile

        :param mesh: Mesh to remove
        :type mesh: Mesh
        :param do_unlink: Unlink all usages of this mesh before deleting it (WARNING: will also delete objects instancing that mesh data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this mesh data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this mesh data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataMetaBalls(bpy_prop_collection[MetaBall], bpy_struct):
    """Collection of metaballs"""

    def new(self, name: str) -> MetaBall:
        """Add a new metaball to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New metaball data-block
        :rtype: MetaBall
        """

    def remove(
        self,
        metaball: MetaBall,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a metaball from the current blendfile

        :param metaball: Metaball to remove
        :type metaball: MetaBall
        :param do_unlink: Unlink all usages of this metaball before deleting it (WARNING: will also delete objects instancing that metaball data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this metaball data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this metaball data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataMovieClips(bpy_prop_collection[MovieClip], bpy_struct):
    """Collection of movie clips"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def remove(
        self,
        clip: MovieClip,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a movie clip from the current blendfile.

        :param clip: Movie clip to remove
        :type clip: MovieClip
        :param do_unlink: Unlink all usages of this movie clip before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this movie clip
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this movie clip
        :type do_ui_user: bool | None
        """

    def load(self, filepath: str, check_existing: bool | None = False) -> MovieClip:
        """Add a new movie clip to the main database from a file (while check_existing is disabled for consistency with other load functions, behavior with multiple movie-clips using the same file may incorrectly generate proxies)

        :param filepath: path for the data-block
        :type filepath: str
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | None
        :return: New movie clip data-block
        :rtype: MovieClip
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataNodeTrees(bpy_prop_collection[NodeTree], bpy_struct):
    """Collection of node trees"""

    def new(self, name: str, type: typing.Literal["DUMMY"] | None) -> NodeTree:
        """Add a new node tree to the main database

        :param name: New name for the data-block
        :type name: str
        :param type: Type, The type of node_group to add
        :type type: typing.Literal['DUMMY'] | None
        :return: New node tree data-block
        :rtype: NodeTree
        """

    def remove(
        self,
        tree: NodeTree,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a node tree from the current blendfile

        :param tree: Node tree to remove
        :type tree: NodeTree
        :param do_unlink: Unlink all usages of this node tree before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this node tree
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this node tree
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of objects"""

    def new(self, name: str, object_data: ID | None) -> Object:
        """Add a new object to the main database

        :param name: New name for the data-block
        :type name: str
        :param object_data: Object data or None for an empty object
        :type object_data: ID | None
        :return: New object data-block
        :rtype: Object
        """

    def remove(
        self,
        object: Object,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove an object from the current blendfile

        :param object: Object to remove
        :type object: Object
        :param do_unlink: Unlink all usages of this object before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this object
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this object
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataPaintCurves(bpy_prop_collection[PaintCurve], bpy_struct):
    """Collection of paint curves"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataPalettes(bpy_prop_collection[Palette], bpy_struct):
    """Collection of palettes"""

    def new(self, name: str) -> Palette:
        """Add a new palette to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New palette data-block
        :rtype: Palette
        """

    def remove(
        self,
        palette: Palette,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a palette from the current blendfile

        :param palette: Palette to remove
        :type palette: Palette
        :param do_unlink: Unlink all usages of this palette before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this palette
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this palette
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataParticles(bpy_prop_collection[ParticleSettings], bpy_struct):
    """Collection of particle settings"""

    def new(self, name: str) -> ParticleSettings:
        """Add a new particle settings instance to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New particle settings data-block
        :rtype: ParticleSettings
        """

    def remove(
        self,
        particle: ParticleSettings,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a particle settings instance from the current blendfile

        :param particle: Particle Settings to remove
        :type particle: ParticleSettings
        :param do_unlink: Unlink all usages of those particle settings before deleting them
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this particle settings
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this particle settings
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataPointClouds(bpy_prop_collection[PointCloud], bpy_struct):
    """Collection of point clouds"""

    def new(self, name: str) -> PointCloud:
        """Add a new point cloud to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New point cloud data-block
        :rtype: PointCloud
        """

    def remove(
        self,
        pointcloud: PointCloud,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a point cloud from the current blendfile

        :param pointcloud: Point cloud to remove
        :type pointcloud: PointCloud
        :param do_unlink: Unlink all usages of this point cloud before deleting it (WARNING: will also delete objects instancing that point cloud data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this point cloud data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this point cloud data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataProbes(bpy_prop_collection[LightProbe], bpy_struct):
    """Collection of light probes"""

    def new(
        self, name: str, type: bpy._typing.rna_enums.LightprobesTypeItems | None
    ) -> LightProbe:
        """Add a new light probe to the main database

        :param name: New name for the data-block
        :type name: str
        :param type: Type, The type of light probe to add
        :type type: bpy._typing.rna_enums.LightprobesTypeItems | None
        :return: New light probe data-block
        :rtype: LightProbe
        """

    def remove(
        self,
        lightprobe: LightProbe,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a light probe from the current blendfile

        :param lightprobe: Light probe to remove
        :type lightprobe: LightProbe
        :param do_unlink: Unlink all usages of this light probe before deleting it (WARNING: will also delete objects instancing that light probe data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this light probe
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this light probe
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataScenes(bpy_prop_collection[Scene], bpy_struct):
    """Collection of scenes"""

    def new(self, name: str) -> Scene:
        """Add a new scene to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New scene data-block
        :rtype: Scene
        """

    def remove(self, scene: Scene, do_unlink: bool | None = True):
        """Remove a scene from the current blendfile

        :param scene: Scene to remove
        :type scene: Scene
        :param do_unlink: Unlink all usages of this scene before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataScreens(bpy_prop_collection[Screen], bpy_struct):
    """Collection of screens"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataSounds(bpy_prop_collection[Sound], bpy_struct):
    """Collection of sounds"""

    def load(self, filepath: str, check_existing: bool | None = False) -> Sound:
        """Add a new sound to the main database from a file

        :param filepath: path for the data-block
        :type filepath: str
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | None
        :return: New text data-block
        :rtype: Sound
        """

    def remove(
        self,
        sound: Sound,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a sound from the current blendfile

        :param sound: Sound to remove
        :type sound: Sound
        :param do_unlink: Unlink all usages of this sound before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this sound
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this sound
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataSpeakers(bpy_prop_collection[Speaker], bpy_struct):
    """Collection of speakers"""

    def new(self, name: str) -> Speaker:
        """Add a new speaker to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New speaker data-block
        :rtype: Speaker
        """

    def remove(
        self,
        speaker: Speaker,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a speaker from the current blendfile

        :param speaker: Speaker to remove
        :type speaker: Speaker
        :param do_unlink: Unlink all usages of this speaker before deleting it (WARNING: will also delete objects instancing that speaker data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this speaker data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this speaker data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataTexts(bpy_prop_collection[Text], bpy_struct):
    """Collection of texts"""

    def new(self, name: str) -> Text:
        """Add a new text to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New text data-block
        :rtype: Text
        """

    def remove(
        self,
        text: Text,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a text from the current blendfile

        :param text: Text to remove
        :type text: Text
        :param do_unlink: Unlink all usages of this text before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this text
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this text
        :type do_ui_user: bool | None
        """

    def load(self, filepath: str, internal: bool | None = False) -> Text:
        """Add a new text to the main database from a file

        :param filepath: path for the data-block
        :type filepath: str
        :param internal: Make internal, Make text file internal after loading
        :type internal: bool | None
        :return: New text data-block
        :rtype: Text
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataTextures(bpy_prop_collection[Texture], bpy_struct):
    """Collection of textures"""

    def new(
        self, name: str, type: bpy._typing.rna_enums.TextureTypeItems | None
    ) -> Texture:
        """Add a new texture to the main database

        :param name: New name for the data-block
        :type name: str
        :param type: Type, The type of texture to add
        :type type: bpy._typing.rna_enums.TextureTypeItems | None
        :return: New texture data-block
        :rtype: Texture
        """

    def remove(
        self,
        texture: Texture,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a texture from the current blendfile

        :param texture: Texture to remove
        :type texture: Texture
        :param do_unlink: Unlink all usages of this texture before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this texture
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this texture
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataVolumes(bpy_prop_collection[Volume], bpy_struct):
    """Collection of volumes"""

    def new(self, name: str) -> Volume:
        """Add a new volume to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New volume data-block
        :rtype: Volume
        """

    def remove(
        self,
        volume: Volume,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a volume from the current blendfile

        :param volume: Volume to remove
        :type volume: Volume
        :param do_unlink: Unlink all usages of this volume before deleting it (WARNING: will also delete objects instancing that volume data)
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this volume data
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this volume data
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataWindowManagers(bpy_prop_collection[WindowManager], bpy_struct):
    """Collection of window managers"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataWorkSpaces(bpy_prop_collection[WorkSpace], bpy_struct):
    """Collection of workspaces"""

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendDataWorlds(bpy_prop_collection[World], bpy_struct):
    """Collection of worlds"""

    def new(self, name: str) -> World:
        """Add a new world to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New world data-block
        :rtype: World
        """

    def remove(
        self,
        world: World,
        do_unlink: bool | None = True,
        do_id_user: bool | None = True,
        do_ui_user: bool | None = True,
    ):
        """Remove a world from the current blendfile

        :param world: World to remove
        :type world: World
        :param do_unlink: Unlink all usages of this world before deleting it
        :type do_unlink: bool | None
        :param do_id_user: Decrement user counter of all datablocks used by this world
        :type do_id_user: bool | None
        :param do_ui_user: Make sure interface does not reference this world
        :type do_ui_user: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoneCollectionMemberships(bpy_prop_collection[BoneCollection], bpy_struct):
    """The Bone Collections that contain this Bone"""

    def clear(self):
        """Remove this bone from all bone collections"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoneCollections(bpy_prop_collection[BoneCollection], bpy_struct):
    """The Bone Collections of this Armature"""

    active: BoneCollection | None
    """ Armature's active bone collection

    :type: BoneCollection | None
    """

    active_index: int | None
    """ The index of the Armature's active bone collection; -1 when there is no active collection

    :type: int | None
    """

    active_name: str | None
    """ The name of the Armature's active bone collection; empty when there is no active collection

    :type: str | None
    """

    def new(self, name: str) -> BoneCollection:
        """Add a new empty bone collection to the armature

        :param name: Name, Name of the new collection. Blender will ensure it is unique within the collections of the Armature
        :type name: str
        :return: Newly created bone collection
        :rtype: BoneCollection
        """

    def remove(self, bone_collection: BoneCollection | None):
        """Remove the bone collection from the armature

        :param bone_collection: Bone Collection, The bone collection to remove
        :type bone_collection: BoneCollection | None
        """

    def move(self, from_index: int | None, to_index: int | None):
        """Move a bone collection to a different position in the collection list

        :param from_index: From Index, Index to move
        :type from_index: int | None
        :param to_index: To Index, Target index
        :type to_index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CacheFileLayers(bpy_prop_collection[CacheFileLayer], bpy_struct):
    """Collection of cache layers"""

    active: CacheFileLayer | None
    """ Active layer of the CacheFile

    :type: CacheFileLayer | None
    """

    def new(self, filepath: str) -> CacheFileLayer:
        """Add a new layer

        :param filepath: File path to the archive used as a layer
        :type filepath: str
        :return: Newly created layer
        :rtype: CacheFileLayer
        """

    def remove(self, layer: CacheFileLayer):
        """Remove an existing layer from the cache file

        :param layer: Layer to remove
        :type layer: CacheFileLayer
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CacheObjectPaths(bpy_prop_collection[CacheObjectPath], bpy_struct):
    """Collection of object paths"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CameraBackgroundImages(bpy_prop_collection[CameraBackgroundImage], bpy_struct):
    """Collection of background images"""

    def new(self) -> CameraBackgroundImage:
        """Add new background image

        :return: Image displayed as viewport background
        :rtype: CameraBackgroundImage
        """

    def remove(self, image: CameraBackgroundImage):
        """Remove background image

        :param image: Image displayed as viewport background
        :type image: CameraBackgroundImage
        """

    def clear(self):
        """Remove all background images"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ChannelDriverVariables(bpy_prop_collection[DriverVariable], bpy_struct):
    """Collection of channel driver Variables"""

    def new(self) -> DriverVariable:
        """Add a new variable for the driver

        :return: Newly created Driver Variable
        :rtype: DriverVariable
        """

    def remove(self, variable: DriverVariable):
        """Remove an existing variable from the driver

        :param variable: Variable to remove from the driver
        :type variable: DriverVariable
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CollectionChildren(bpy_prop_collection[Collection], bpy_struct):
    """Collection of child collections"""

    def link(self, child: Collection):
        """Add this collection as child of this collection

        :param child: Collection to add
        :type child: Collection
        """

    def unlink(self, child: Collection | None):
        """Remove this child collection from a collection

        :param child: Collection to remove
        :type child: Collection | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CollectionObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of collection objects"""

    def link(self, object: Object):
        """Add this object to a collection

        :param object: Object to add
        :type object: Object
        """

    def unlink(self, object: Object | None):
        """Remove this object from a collection

        :param object: Object to remove
        :type object: Object | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorRampElements(bpy_prop_collection[ColorRampElement], bpy_struct):
    """Collection of Color Ramp Elements"""

    def new(self, position: float | None) -> ColorRampElement:
        """Add element to Color Ramp

        :param position: Position, Position to add element
        :type position: float | None
        :return: New element
        :rtype: ColorRampElement
        """

    def remove(self, element: ColorRampElement):
        """Delete element from Color Ramp

        :param element: Element to remove
        :type element: ColorRampElement
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeOutputFileFileSlots(
    bpy_prop_collection[NodeOutputFileSlotFile], bpy_struct
):
    """Collection of File Output node slots"""

    def new(self, name: str) -> NodeSocket:
        """Add a file slot to this node

        :param name: Name
        :type name: str
        :return: New socket
        :rtype: NodeSocket
        """

    def remove(self, socket: NodeSocket | None):
        """Remove a file slot from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """

    def clear(self):
        """Remove all file slots from this node"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a file slot to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeOutputFileLayerSlots(
    bpy_prop_collection[NodeOutputFileSlotLayer], bpy_struct
):
    """Collection of File Output node slots"""

    def new(self, name: str) -> NodeSocket:
        """Add a file slot to this node

        :param name: Name
        :type name: str
        :return: New socket
        :rtype: NodeSocket
        """

    def remove(self, socket: NodeSocket | None):
        """Remove a file slot from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """

    def clear(self):
        """Remove all file slots from this node"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a file slot to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveMapPoints(bpy_prop_collection[CurveMapPoint], bpy_struct):
    """Collection of Curve Map Points"""

    def new(self, position: float | None, value: float | None) -> CurveMapPoint:
        """Add point to CurveMap

        :param position: Position, Position to add point
        :type position: float | None
        :param value: Value, Value of point
        :type value: float | None
        :return: New point
        :rtype: CurveMapPoint
        """

    def remove(self, point: CurveMapPoint):
        """Delete point from CurveMap

        :param point: PointElement to remove
        :type point: CurveMapPoint
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveProfilePoints(bpy_prop_collection[CurveProfilePoint], bpy_struct):
    """Collection of Profile Points"""

    def add(self, x: float | None, y: float | None) -> CurveProfilePoint:
        """Add point to the profile

        :param x: X Position, X Position for new point
        :type x: float | None
        :param y: Y Position, Y Position for new point
        :type y: float | None
        :return: New point
        :rtype: CurveProfilePoint
        """

    def remove(self, point: CurveProfilePoint):
        """Delete point from the profile

        :param point: Point to remove
        :type point: CurveProfilePoint
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveSplines(bpy_prop_collection[Spline], bpy_struct):
    """Collection of curve splines"""

    active: Spline | None
    """ Active curve spline

    :type: Spline | None
    """

    def new(self, type: typing.Literal["POLY", "BEZIER", "NURBS"] | None) -> Spline:
        """Add a new spline to the curve

        :param type: type for the new spline
        :type type: typing.Literal['POLY','BEZIER','NURBS'] | None
        :return: The newly created spline
        :rtype: Spline
        """

    def remove(self, spline: Spline):
        """Remove a spline from a curve

        :param spline: The spline to remove
        :type spline: Spline
        """

    def clear(self):
        """Remove all splines from a curve"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DynamicPaintSurfaces(bpy_prop_collection[DynamicPaintSurface], bpy_struct):
    """Collection of Dynamic Paint Canvas surfaces"""

    active: DynamicPaintSurface | None
    """ Active Dynamic Paint surface being displayed

    :type: DynamicPaintSurface | None
    """

    active_index: int | None
    """ 

    :type: int | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FCurveKeyframePoints(bpy_prop_collection[Keyframe], bpy_struct):
    """Collection of keyframe points"""

    def insert(
        self,
        frame: float | None,
        value: float | None,
        options: set[typing.Literal["REPLACE", "NEEDED", "FAST"]] | None = {},
        keyframe_type: bpy._typing.rna_enums.BeztripleKeyframeTypeItems
        | None = "KEYFRAME",
    ) -> Keyframe:
        """Add a keyframe point to a F-Curve

                :param frame: X Value of this keyframe point
                :type frame: float | None
                :param value: Y Value of this keyframe point
                :type value: float | None
                :param options: Keyframe options

        REPLACE
        Replace -- Don't add any new keyframes, but just replace existing ones.

        NEEDED
        Needed -- Only adds keyframes that are needed.

        FAST
        Fast -- Fast keyframe insertion to avoid recalculating the curve each time.
                :type options: set[typing.Literal['REPLACE','NEEDED','FAST']] | None
                :param keyframe_type: Type of keyframe to insert
                :type keyframe_type: bpy._typing.rna_enums.BeztripleKeyframeTypeItems | None
                :return: Newly created keyframe
                :rtype: Keyframe
        """

    def add(self, count: int | None):
        """Add a keyframe point to a F-Curve

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """

    def remove(self, keyframe: Keyframe, fast: bool | None = False):
        """Remove keyframe from an F-Curve

        :param keyframe: Keyframe to remove
        :type keyframe: Keyframe
        :param fast: Fast, Fast keyframe removal to avoid recalculating the curve each time
        :type fast: bool | None
        """

    def clear(self):
        """Remove all keyframes from an F-Curve"""

    def sort(self):
        """Ensure all keyframe points are chronologically sorted"""

    def deduplicate(self):
        """Ensure there are no duplicate keys. Assumes that the points have already been sorted"""

    def handles_recalc(self):
        """Update handles after modifications to the keyframe points, to update things like auto-clamping"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FCurveModifiers(bpy_prop_collection[FModifier], bpy_struct):
    """Collection of F-Curve Modifiers"""

    active: FModifier | None
    """ Active F-Curve Modifier

    :type: FModifier | None
    """

    def new(self, type: bpy._typing.rna_enums.FmodifierTypeItems | None) -> FModifier:
        """Add a constraint to this object

        :param type: Constraint type to add
        :type type: bpy._typing.rna_enums.FmodifierTypeItems | None
        :return: New fmodifier
        :rtype: FModifier
        """

    def remove(self, modifier: FModifier):
        """Remove a modifier from this F-Curve

        :param modifier: Removed modifier
        :type modifier: FModifier
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierEnvelopeControlPoints(
    bpy_prop_collection[FModifierEnvelopeControlPoint], bpy_struct
):
    """Control points defining the shape of the envelope"""

    def add(self, frame: float | None) -> FModifierEnvelopeControlPoint:
        """Add a control point to a FModifierEnvelope

        :param frame: Frame to add this control-point
        :type frame: float | None
        :return: Newly created control-point
        :rtype: FModifierEnvelopeControlPoint
        """

    def remove(self, point: FModifierEnvelopeControlPoint):
        """Remove a control-point from an FModifierEnvelope

        :param point: Control-point to remove
        :type point: FModifierEnvelopeControlPoint
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FreestyleModules(bpy_prop_collection[FreestyleModuleSettings], bpy_struct):
    """A list of style modules (to be applied from top to bottom)"""

    def new(self) -> FreestyleModuleSettings:
        """Add a style module to scene render layer Freestyle settings

        :return: Newly created style module
        :rtype: FreestyleModuleSettings
        """

    def remove(self, module: FreestyleModuleSettings):
        """Remove a style module from scene render layer Freestyle settings

        :param module: Style module to remove
        :type module: FreestyleModuleSettings
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilFrames(bpy_prop_collection[GPencilFrame], bpy_struct):
    """Collection of grease pencil frames"""

    def new(
        self, frame_number: int | None, active: bool | None = False
    ) -> GPencilFrame:
        """Add a new grease pencil frame

        :param frame_number: Frame Number, The frame on which this sketch appears
        :type frame_number: int | None
        :param active: Active
        :type active: bool | None
        :return: The newly created frame
        :rtype: GPencilFrame
        """

    def remove(self, frame: GPencilFrame):
        """Remove a grease pencil frame

        :param frame: Frame, The frame to remove
        :type frame: GPencilFrame
        """

    def copy(self, source: GPencilFrame) -> GPencilFrame:
        """Copy a grease pencil frame

        :param source: Source, The source frame
        :type source: GPencilFrame
        :return: The newly copied frame
        :rtype: GPencilFrame
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilStrokePoints(bpy_prop_collection[GPencilStrokePoint], bpy_struct):
    """Collection of grease pencil stroke points"""

    def add(
        self,
        count: int | None,
        pressure: float | None = 1.0,
        strength: float | None = 1.0,
    ):
        """Add a new grease pencil stroke point

        :param count: Number, Number of points to add to the stroke
        :type count: int | None
        :param pressure: Pressure, Pressure for newly created points
        :type pressure: float | None
        :param strength: Strength, Color intensity (alpha factor) for newly created points
        :type strength: float | None
        """

    def pop(self, index: int | None = -1):
        """Remove a grease pencil stroke point

        :param index: Index, point index
        :type index: int | None
        """

    def update(self):
        """Recalculate internal triangulation data"""

    def weight_get(
        self, vertex_group_index: int | None = 0, point_index: int | None = 0
    ) -> float:
        """Get vertex group point weight

        :param vertex_group_index: Vertex Group Index, Index of Vertex Group in the array of groups
        :type vertex_group_index: int | None
        :param point_index: Point Index, Index of the Point in the array
        :type point_index: int | None
        :return: Weight, Point Weight
        :rtype: float
        """

    def weight_set(
        self,
        vertex_group_index: int | None = 0,
        point_index: int | None = 0,
        weight: float | None = 0.0,
    ):
        """Set vertex group point weight

        :param vertex_group_index: Vertex Group Index, Index of Vertex Group in the array of groups
        :type vertex_group_index: int | None
        :param point_index: Point Index, Index of the Point in the array
        :type point_index: int | None
        :param weight: Weight, Point Weight
        :type weight: float | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilStrokes(bpy_prop_collection[GPencilStroke], bpy_struct):
    """Collection of grease pencil stroke"""

    def new(self) -> GPencilStroke:
        """Add a new grease pencil stroke

        :return: The newly created stroke
        :rtype: GPencilStroke
        """

    def remove(self, stroke: GPencilStroke):
        """Remove a grease pencil stroke

        :param stroke: Stroke, The stroke to remove
        :type stroke: GPencilStroke
        """

    def close(self, stroke: GPencilStroke):
        """Close a grease pencil stroke adding geometry

        :param stroke: Stroke, The stroke to close
        :type stroke: GPencilStroke
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Gizmos(bpy_prop_collection[Gizmo], bpy_struct):
    """Collection of gizmos"""

    def new(self, type: str) -> Gizmo:
        """Add gizmo

        :param type: Gizmo identifier
        :type type: str
        :return: New gizmo
        :rtype: Gizmo
        """

    def remove(self, gizmo: Gizmo):
        """Delete gizmo

        :param gizmo: New gizmo
        :type gizmo: Gizmo
        """

    def clear(self):
        """Delete all gizmos"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GreasePencilLayers(bpy_prop_collection[GPencilLayer], bpy_struct):
    """Collection of grease pencil layers"""

    active: GPencilLayer | None
    """ Active grease pencil layer

    :type: GPencilLayer | None
    """

    active_index: int | None
    """ Index of active grease pencil layer

    :type: int | None
    """

    active_note: str | None
    """ Note/Layer to add annotation strokes to

    :type: str | None
    """

    def new(self, name: str, set_active: bool | None = True) -> GPencilLayer:
        """Add a new grease pencil layer

        :param name: Name, Name of the layer
        :type name: str
        :param set_active: Set Active, Set the newly created layer to the active layer
        :type set_active: bool | None
        :return: The newly created layer
        :rtype: GPencilLayer
        """

    def remove(self, layer: GPencilLayer):
        """Remove a grease pencil layer

        :param layer: The layer to remove
        :type layer: GPencilLayer
        """

    def move(self, layer: GPencilLayer, type: typing.Literal["UP", "DOWN"] | None):
        """Move a grease pencil layer in the layer stack

        :param layer: The layer to move
        :type layer: GPencilLayer
        :param type: Direction of movement
        :type type: typing.Literal['UP','DOWN'] | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GreasePencilMaskLayers(bpy_prop_collection[GPencilLayerMask], bpy_struct):
    """Collection of grease pencil masking layers"""

    active_mask_index: int | None
    """ Active index in layer mask array

    :type: int | None
    """

    def add(self, layer: GPencilLayer):
        """Add a layer to mask list

        :param layer: Layer to add as mask
        :type layer: GPencilLayer
        """

    def remove(self, mask: GPencilLayerMask):
        """Remove a layer from mask list

        :param mask: Mask to remove
        :type mask: GPencilLayerMask
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class IDMaterials(bpy_prop_collection[Material], bpy_struct):
    """Collection of materials"""

    def append(self, material: Material | None):
        """Add a new material to the data-block

        :param material: Material to add
        :type material: Material | None
        """

    def pop(self, *, index: int = -1) -> Material:
        """Remove a material from the data-block

        :param index: Index of material to remove
        :type index: int
        :return: Material to remove
        :rtype: Material
        """

    def clear(self):
        """Remove all materials from the data-block"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

    @typing.overload
    def __getitem__(self, key: int | str) -> Material | None:
        """

        :param key:
        :type key: int | str
        :return:
        :rtype: Material | None
        """

    @typing.overload
    def __getitem__(self, key: slice) -> list[Material | None, ...]:
        """

        :param key:
        :type key: slice
        :return:
        :rtype: list[Material | None, ...]
        """

class IDOverrideLibraryProperties(
    bpy_prop_collection[IDOverrideLibraryProperty], bpy_struct
):
    """Collection of override properties"""

    def add(self, rna_path: str) -> IDOverrideLibraryProperty:
        """Add a property to the override library when it doesn't exist yet

        :param rna_path: RNA Path, RNA-Path of the property to add
        :type rna_path: str
        :return: New Property, Newly created override property or existing one
        :rtype: IDOverrideLibraryProperty
        """

    def remove(self, property: IDOverrideLibraryProperty | None):
        """Remove and delete a property

        :param property: Property, Override property to be deleted
        :type property: IDOverrideLibraryProperty | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class IDOverrideLibraryPropertyOperations(
    bpy_prop_collection[IDOverrideLibraryPropertyOperation], bpy_struct
):
    """Collection of override operations"""

    def add(
        self,
        operation: typing.Literal[
            "NOOP",
            "REPLACE",
            "DIFF_ADD",
            "DIFF_SUB",
            "FACT_MULTIPLY",
            "INSERT_AFTER",
            "INSERT_BEFORE",
        ]
        | None,
        use_id: bool | None = False,
        subitem_reference_name: str = "",
        subitem_local_name: str = "",
        subitem_reference_id: ID | None = None,
        subitem_local_id: ID | None = None,
        subitem_reference_index: int | None = -1,
        subitem_local_index: int | None = -1,
    ) -> IDOverrideLibraryPropertyOperation:
        """Add a new operation

                :param operation: Operation, What override operation is performed

        NOOP
        No-Op -- Does nothing, prevents adding actual overrides (NOT USED).

        REPLACE
        Replace -- Replace value of reference by overriding one.

        DIFF_ADD
        Differential -- Stores and apply difference between reference and local value (NOT USED).

        DIFF_SUB
        Differential -- Stores and apply difference between reference and local value (NOT USED).

        FACT_MULTIPLY
        Factor -- Stores and apply multiplication factor between reference and local value (NOT USED).

        INSERT_AFTER
        Insert After -- Insert a new item into collection after the one referenced in subitem_reference_name/_id or _index.

        INSERT_BEFORE
        Insert Before -- Insert a new item into collection before the one referenced in subitem_reference_name/_id or _index (NOT USED).
                :type operation: typing.Literal['NOOP','REPLACE','DIFF_ADD','DIFF_SUB','FACT_MULTIPLY','INSERT_AFTER','INSERT_BEFORE'] | None
                :param use_id: Use ID Pointer Subitem, Whether the found or created liboverride operation should use ID pointers or not
                :type use_id: bool | None
                :param subitem_reference_name: Subitem Reference Name, Used to handle insertions or ID replacements into collection
                :type subitem_reference_name: str
                :param subitem_local_name: Subitem Local Name, Used to handle insertions or ID replacements into collection
                :type subitem_local_name: str
                :param subitem_reference_id: Subitem Reference ID, Used to handle ID replacements into collection
                :type subitem_reference_id: ID | None
                :param subitem_local_id: Subitem Local ID, Used to handle ID replacements into collection
                :type subitem_local_id: ID | None
                :param subitem_reference_index: Subitem Reference Index, Used to handle insertions or ID replacements into collection
                :type subitem_reference_index: int | None
                :param subitem_local_index: Subitem Local Index, Used to handle insertions or ID replacements into collection
                :type subitem_local_index: int | None
                :return: New Operation, Created operation
                :rtype: IDOverrideLibraryPropertyOperation
        """

    def remove(self, operation: IDOverrideLibraryPropertyOperation | None):
        """Remove and delete an operation

        :param operation: Operation, Override operation to be deleted
        :type operation: IDOverrideLibraryPropertyOperation | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class KeyConfigurations(bpy_prop_collection[KeyConfig], bpy_struct):
    """Collection of KeyConfigs"""

    active: KeyConfig | None
    """ Active key configuration (preset)

    :type: KeyConfig | None
    """

    addon: KeyConfig
    """ Key configuration that can be extended by add-ons, and is added to the active configuration when handling events

    :type: KeyConfig
    """

    default: KeyConfig
    """ Default builtin key configuration

    :type: KeyConfig
    """

    user: KeyConfig
    """ Final key configuration that combines keymaps from the active and add-on configurations, and can be edited by the user

    :type: KeyConfig
    """

    def new(self, name: str) -> KeyConfig:
        """new

        :param name: Name
        :type name: str
        :return: Key Configuration, Added key configuration
        :rtype: KeyConfig
        """

    def remove(self, keyconfig: KeyConfig):
        """remove

        :param keyconfig: Key Configuration, Removed key configuration
        :type keyconfig: KeyConfig
        """

    def find_item_from_operator(
        self,
        idname: str,
        context: bpy._typing.rna_enums.OperatorContextItems | None = "INVOKE_DEFAULT",
        properties: OperatorProperties | None = None,
        include: set[bpy._typing.rna_enums.EventTypeMaskItems] | None = {
            "ACTIONZONE",
            "KEYBOARD",
            "MOUSE",
            "NDOF",
        },
        exclude: set[bpy._typing.rna_enums.EventTypeMaskItems] | None = {},
    ):
        """find_item_from_operator

                :param idname: Operator Identifier
                :type idname: str
                :param context: context
                :type context: bpy._typing.rna_enums.OperatorContextItems | None
                :param properties:
                :type properties: OperatorProperties | None
                :param include: Include
                :type include: set[bpy._typing.rna_enums.EventTypeMaskItems] | None
                :param exclude: Exclude
                :type exclude: set[bpy._typing.rna_enums.EventTypeMaskItems] | None
                :return: keymap, `KeyMap`

        item, `KeyMapItem`
        """

    def update(self, keep_properties: bool | None = False):
        """update

        :param keep_properties: Keep Properties, Operator properties are kept to allow the operators to be registered again in the future
        :type keep_properties: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class KeyMapItems(bpy_prop_collection[KeyMapItem], bpy_struct):
    """Collection of keymap items"""

    def new(
        self,
        idname: str,
        type: bpy._typing.rna_enums.EventTypeItems | None,
        value: bpy._typing.rna_enums.EventValueItems | None,
        any: bool | None = False,
        shift: int | None = 0,
        ctrl: int | None = 0,
        alt: int | None = 0,
        oskey: int | None = 0,
        key_modifier: bpy._typing.rna_enums.EventTypeItems | None = "NONE",
        direction: bpy._typing.rna_enums.EventDirectionItems | None = "ANY",
        repeat: bool | None = False,
        head: bool | None = False,
    ) -> KeyMapItem:
        """new

        :param idname: Operator Identifier
        :type idname: str
        :param type: Type
        :type type: bpy._typing.rna_enums.EventTypeItems | None
        :param value: Value
        :type value: bpy._typing.rna_enums.EventValueItems | None
        :param any: Any
        :type any: bool | None
        :param shift: Shift
        :type shift: int | None
        :param ctrl: Ctrl
        :type ctrl: int | None
        :param alt: Alt
        :type alt: int | None
        :param oskey: OS Key
        :type oskey: int | None
        :param key_modifier: Key Modifier
        :type key_modifier: bpy._typing.rna_enums.EventTypeItems | None
        :param direction: Direction
        :type direction: bpy._typing.rna_enums.EventDirectionItems | None
        :param repeat: Repeat, When set, accept key-repeat events
        :type repeat: bool | None
        :param head: At Head, Force item to be added at start (not end) of key map so that it doesn't get blocked by an existing key map item
        :type head: bool | None
        :return: Item, Added key map item
        :rtype: KeyMapItem
        """

    def new_modal(
        self,
        propvalue: str,
        type: bpy._typing.rna_enums.EventTypeItems | None,
        value: bpy._typing.rna_enums.EventValueItems | None,
        any: bool | None = False,
        shift: int | None = 0,
        ctrl: int | None = 0,
        alt: int | None = 0,
        oskey: int | None = 0,
        key_modifier: bpy._typing.rna_enums.EventTypeItems | None = "NONE",
        direction: bpy._typing.rna_enums.EventDirectionItems | None = "ANY",
        repeat: bool | None = False,
    ) -> KeyMapItem:
        """new_modal

        :param propvalue: Property Value
        :type propvalue: str
        :param type: Type
        :type type: bpy._typing.rna_enums.EventTypeItems | None
        :param value: Value
        :type value: bpy._typing.rna_enums.EventValueItems | None
        :param any: Any
        :type any: bool | None
        :param shift: Shift
        :type shift: int | None
        :param ctrl: Ctrl
        :type ctrl: int | None
        :param alt: Alt
        :type alt: int | None
        :param oskey: OS Key
        :type oskey: int | None
        :param key_modifier: Key Modifier
        :type key_modifier: bpy._typing.rna_enums.EventTypeItems | None
        :param direction: Direction
        :type direction: bpy._typing.rna_enums.EventDirectionItems | None
        :param repeat: Repeat, When set, accept key-repeat events
        :type repeat: bool | None
        :return: Item, Added key map item
        :rtype: KeyMapItem
        """

    def new_from_item(self, item: KeyMapItem, head: bool | None = False) -> KeyMapItem:
        """new_from_item

        :param item: Item, Item to use as a reference
        :type item: KeyMapItem
        :param head: At Head
        :type head: bool | None
        :return: Item, Added key map item
        :rtype: KeyMapItem
        """

    def remove(self, item: KeyMapItem):
        """remove

        :param item: Item
        :type item: KeyMapItem
        """

    def from_id(self, id: int | None) -> KeyMapItem:
        """from_id

        :param id: id, ID of the item
        :type id: int | None
        :return: Item
        :rtype: KeyMapItem
        """

    def find_from_operator(
        self,
        idname: str,
        properties: OperatorProperties | None = None,
        include: set[bpy._typing.rna_enums.EventTypeMaskItems] | None = {
            "ACTIONZONE",
            "KEYBOARD",
            "MOUSE",
            "NDOF",
        },
        exclude: set[bpy._typing.rna_enums.EventTypeMaskItems] | None = {},
    ) -> KeyMapItem:
        """find_from_operator

        :param idname: Operator Identifier
        :type idname: str
        :param properties:
        :type properties: OperatorProperties | None
        :param include: Include
        :type include: set[bpy._typing.rna_enums.EventTypeMaskItems] | None
        :param exclude: Exclude
        :type exclude: set[bpy._typing.rna_enums.EventTypeMaskItems] | None
        :return:
        :rtype: KeyMapItem
        """

    def match_event(self, event: Event | None) -> KeyMapItem:
        """match_event

        :param event:
        :type event: Event | None
        :return:
        :rtype: KeyMapItem
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class KeyMaps(bpy_prop_collection[KeyMap], bpy_struct):
    """Collection of keymaps"""

    def new(
        self,
        name: str,
        space_type: bpy._typing.rna_enums.SpaceTypeItems | None = "EMPTY",
        region_type: bpy._typing.rna_enums.RegionTypeItems | None = "WINDOW",
        modal: bool | None = False,
        tool: bool | None = False,
    ) -> KeyMap:
        """Ensure the keymap exists. This will return the one with the given name/space type/region type, or create a new one if it does not exist yet.

        :param name: Name
        :type name: str
        :param space_type: Space Type
        :type space_type: bpy._typing.rna_enums.SpaceTypeItems | None
        :param region_type: Region Type
        :type region_type: bpy._typing.rna_enums.RegionTypeItems | None
        :param modal: Modal, Keymap for modal operators
        :type modal: bool | None
        :param tool: Tool, Keymap for active tools
        :type tool: bool | None
        :return: Key Map, Added key map
        :rtype: KeyMap
        """

    def remove(self, keymap: KeyMap):
        """remove

        :param keymap: Key Map, Removed key map
        :type keymap: KeyMap
        """

    def clear(self):
        """Remove all keymaps."""

    def find(
        self,
        name: str,
        space_type: bpy._typing.rna_enums.SpaceTypeItems | None = "EMPTY",
        region_type: bpy._typing.rna_enums.RegionTypeItems | None = "WINDOW",
    ) -> KeyMap:
        """find

        :param name: Name
        :type name: str
        :param space_type: Space Type
        :type space_type: bpy._typing.rna_enums.SpaceTypeItems | None
        :param region_type: Region Type
        :type region_type: bpy._typing.rna_enums.RegionTypeItems | None
        :return: Key Map, Corresponding key map
        :rtype: KeyMap
        """

    def find_modal(self, name: str) -> KeyMap:
        """find_modal

        :param name: Operator Name
        :type name: str
        :return: Key Map, Corresponding key map
        :rtype: KeyMap
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class KeyingSetPaths(bpy_prop_collection[KeyingSetPath], bpy_struct):
    """Collection of keying set paths"""

    active: KeyingSetPath | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSetPath | None
    """

    active_index: int | None
    """ Current Keying Set index

    :type: int | None
    """

    def add(
        self,
        target_id: ID | None,
        data_path: str,
        index: int | None = -1,
        group_method: bpy._typing.rna_enums.KeyingsetPathGroupingItems
        | None = "KEYINGSET",
        group_name: str = "",
    ) -> KeyingSetPath:
        """Add a new path for the Keying Set

        :param target_id: Target ID, ID data-block for the destination
        :type target_id: ID | None
        :param data_path: Data-Path, RNA-Path to destination property
        :type data_path: str
        :param index: Index, The index of the destination property (i.e. axis of Location/Rotation/etc.), or -1 for the entire array
        :type index: int | None
        :param group_method: Grouping Method, Method used to define which Group-name to use
        :type group_method: bpy._typing.rna_enums.KeyingsetPathGroupingItems | None
        :param group_name: Group Name, Name of Action Group to assign destination to (only if grouping mode is to use this name)
        :type group_name: str
        :return: New Path, Path created and added to the Keying Set
        :rtype: KeyingSetPath
        """

    def remove(self, path: KeyingSetPath):
        """Remove the given path from the Keying Set

        :param path: Path
        :type path: KeyingSetPath
        """

    def clear(self):
        """Remove all the paths from the Keying Set"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class KeyingSets(bpy_prop_collection[KeyingSet], bpy_struct):
    """Scene keying sets"""

    active: KeyingSet | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSet | None
    """

    active_index: int | None
    """ Current Keying Set index (negative for 'builtin' and positive for 'absolute')

    :type: int | None
    """

    def new(self, idname: str = "KeyingSet", name: str = "KeyingSet") -> KeyingSet:
        """Add a new Keying Set to Scene

        :param idname: IDName, Internal identifier of Keying Set
        :type idname: str
        :param name: Name, User visible name of Keying Set
        :type name: str
        :return: Newly created Keying Set
        :rtype: KeyingSet
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class KeyingSetsAll(bpy_prop_collection[KeyingSet], bpy_struct):
    """All available keying sets"""

    active: KeyingSet | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSet | None
    """

    active_index: int | None
    """ Current Keying Set index (negative for 'builtin' and positive for 'absolute')

    :type: int | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class LayerObjects(bpy_prop_collection[Object], bpy_struct):
    """Collections of objects"""

    active: Object | None
    """ Active object for this layer

    :type: Object | None
    """

    selected: bpy_prop_collection[Object]
    """ All the selected objects of this layer

    :type: bpy_prop_collection[Object]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Lightgroups(bpy_prop_collection[Lightgroup], bpy_struct):
    """Collection of Lightgroups"""

    def add(self, name: str = "") -> Lightgroup:
        """add

        :param name: Name, Name of newly created lightgroup
        :type name: str
        :return: Newly created Lightgroup
        :rtype: Lightgroup
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class LineStyleAlphaModifiers(bpy_prop_collection[LineStyleAlphaModifier], bpy_struct):
    """Alpha modifiers for changing line alphas"""

    def new(
        self,
        name: str,
        type: bpy._typing.rna_enums.LinestyleAlphaModifierTypeItems | None,
    ) -> LineStyleAlphaModifier:
        """Add a alpha modifier to line style

        :param name: New name for the alpha modifier (not unique)
        :type name: str
        :param type: Alpha modifier type to add
        :type type: bpy._typing.rna_enums.LinestyleAlphaModifierTypeItems | None
        :return: Newly added alpha modifier
        :rtype: LineStyleAlphaModifier
        """

    def remove(self, modifier: LineStyleAlphaModifier):
        """Remove a alpha modifier from line style

        :param modifier: Alpha modifier to remove
        :type modifier: LineStyleAlphaModifier
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class LineStyleColorModifiers(bpy_prop_collection[LineStyleColorModifier], bpy_struct):
    """Color modifiers for changing line colors"""

    def new(
        self,
        name: str,
        type: bpy._typing.rna_enums.LinestyleColorModifierTypeItems | None,
    ) -> LineStyleColorModifier:
        """Add a color modifier to line style

        :param name: New name for the color modifier (not unique)
        :type name: str
        :param type: Color modifier type to add
        :type type: bpy._typing.rna_enums.LinestyleColorModifierTypeItems | None
        :return: Newly added color modifier
        :rtype: LineStyleColorModifier
        """

    def remove(self, modifier: LineStyleColorModifier):
        """Remove a color modifier from line style

        :param modifier: Color modifier to remove
        :type modifier: LineStyleColorModifier
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class LineStyleGeometryModifiers(
    bpy_prop_collection[LineStyleGeometryModifier], bpy_struct
):
    """Geometry modifiers for changing line geometries"""

    def new(
        self,
        name: str,
        type: bpy._typing.rna_enums.LinestyleGeometryModifierTypeItems | None,
    ) -> LineStyleGeometryModifier:
        """Add a geometry modifier to line style

        :param name: New name for the geometry modifier (not unique)
        :type name: str
        :param type: Geometry modifier type to add
        :type type: bpy._typing.rna_enums.LinestyleGeometryModifierTypeItems | None
        :return: Newly added geometry modifier
        :rtype: LineStyleGeometryModifier
        """

    def remove(self, modifier: LineStyleGeometryModifier):
        """Remove a geometry modifier from line style

        :param modifier: Geometry modifier to remove
        :type modifier: LineStyleGeometryModifier
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class LineStyleTextureSlots(bpy_prop_collection[LineStyleTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> LineStyleTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: LineStyleTextureSlot
        """

    @classmethod
    def create(cls, index: int | None) -> LineStyleTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: LineStyleTextureSlot
        """

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class LineStyleThicknessModifiers(
    bpy_prop_collection[LineStyleThicknessModifier], bpy_struct
):
    """Thickness modifiers for changing line thickness"""

    def new(
        self,
        name: str,
        type: bpy._typing.rna_enums.LinestyleThicknessModifierTypeItems | None,
    ) -> LineStyleThicknessModifier:
        """Add a thickness modifier to line style

        :param name: New name for the thickness modifier (not unique)
        :type name: str
        :param type: Thickness modifier type to add
        :type type: bpy._typing.rna_enums.LinestyleThicknessModifierTypeItems | None
        :return: Newly added thickness modifier
        :rtype: LineStyleThicknessModifier
        """

    def remove(self, modifier: LineStyleThicknessModifier):
        """Remove a thickness modifier from line style

        :param modifier: Thickness modifier to remove
        :type modifier: LineStyleThicknessModifier
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Linesets(bpy_prop_collection[FreestyleLineSet], bpy_struct):
    """Line sets for associating lines and style parameters"""

    active: FreestyleLineSet | None
    """ Active line set being displayed

    :type: FreestyleLineSet | None
    """

    active_index: int | None
    """ Index of active line set slot

    :type: int | None
    """

    def new(self, name: str) -> FreestyleLineSet:
        """Add a line set to scene render layer Freestyle settings

        :param name: New name for the line set (not unique)
        :type name: str
        :return: Newly created line set
        :rtype: FreestyleLineSet
        """

    def remove(self, lineset: FreestyleLineSet):
        """Remove a line set from scene render layer Freestyle settings

        :param lineset: Line set to remove
        :type lineset: FreestyleLineSet
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class LoopColors(bpy_prop_collection[MeshLoopColorLayer], bpy_struct):
    """Collection of vertex colors"""

    active: MeshLoopColorLayer | None
    """ Active vertex color layer

    :type: MeshLoopColorLayer | None
    """

    active_index: int | None
    """ Active vertex color index

    :type: int | None
    """

    def new(self, name: str = "Col", do_init: bool | None = True) -> MeshLoopColorLayer:
        """Add a vertex color layer to Mesh

        :param name: Vertex color name
        :type name: str
        :param do_init: Whether new layer's data should be initialized by copying current active one
        :type do_init: bool | None
        :return: The newly created layer
        :rtype: MeshLoopColorLayer
        """

    def remove(self, layer: MeshLoopColorLayer):
        """Remove a vertex color layer

        :param layer: The layer to remove
        :type layer: MeshLoopColorLayer
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MaskLayers(bpy_prop_collection[MaskLayer], bpy_struct):
    """Collection of layers used by mask"""

    active: MaskLayer | None
    """ Active layer in this mask

    :type: MaskLayer | None
    """

    def new(self, name: str = "") -> MaskLayer:
        """Add layer to this mask

        :param name: Name, Name of new layer
        :type name: str
        :return: New mask layer
        :rtype: MaskLayer
        """

    def remove(self, layer: MaskLayer):
        """Remove layer from this mask

        :param layer: Shape to be removed
        :type layer: MaskLayer
        """

    def clear(self):
        """Remove all mask layers"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MaskSplinePoints(bpy_prop_collection[MaskSplinePoint], bpy_struct):
    """Collection of masking spline points"""

    def add(self, count: int | None):
        """Add a number of point to this spline

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """

    def remove(self, point: MaskSplinePoint):
        """Remove a point from a spline

        :param point: The point to remove
        :type point: MaskSplinePoint
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MaskSplines(bpy_prop_collection[MaskSpline], bpy_struct):
    """Collection of masking splines"""

    active: MaskSpline | None
    """ Active spline of masking layer

    :type: MaskSpline | None
    """

    active_point: MaskSplinePoint | None
    """ Active point of masking layer

    :type: MaskSplinePoint | None
    """

    def new(self) -> MaskSpline:
        """Add a new spline to the layer

        :return: The newly created spline
        :rtype: MaskSpline
        """

    def remove(self, spline: MaskSpline):
        """Remove a spline from a layer

        :param spline: The spline to remove
        :type spline: MaskSpline
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MeshEdges(bpy_prop_collection[MeshEdge], bpy_struct):
    """Collection of mesh edges"""

    def add(self, count: int | None):
        """add

        :param count: Count, Number of edges to add
        :type count: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MeshLoopTriangles(bpy_prop_collection[MeshLoopTriangle], bpy_struct):
    """Tessellation of mesh polygons into triangles"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MeshLoops(bpy_prop_collection[MeshLoop], bpy_struct):
    """Collection of mesh loops"""

    def add(self, count: int | None):
        """add

        :param count: Count, Number of loops to add
        :type count: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MeshPolygons(bpy_prop_collection[MeshPolygon], bpy_struct):
    """Collection of mesh polygons"""

    active: int | None
    """ The active face for this mesh

    :type: int | None
    """

    def add(self, count: int | None):
        """add

        :param count: Count, Number of polygons to add
        :type count: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MeshVertices(bpy_prop_collection[MeshVertex], bpy_struct):
    """Collection of mesh vertices"""

    def add(self, count: int | None):
        """add

        :param count: Count, Number of vertices to add
        :type count: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MetaBallElements(bpy_prop_collection[MetaElement], bpy_struct):
    """Collection of metaball elements"""

    active: MetaElement | None
    """ Last selected element

    :type: MetaElement | None
    """

    def new(
        self, type: bpy._typing.rna_enums.MetaelemTypeItems | None = "BALL"
    ) -> MetaElement:
        """Add a new element to the metaball

        :param type: Type for the new metaball element
        :type type: bpy._typing.rna_enums.MetaelemTypeItems | None
        :return: The newly created metaball element
        :rtype: MetaElement
        """

    def remove(self, element: MetaElement):
        """Remove an element from the metaball

        :param element: The element to remove
        :type element: MetaElement
        """

    def clear(self):
        """Remove all elements from the metaball"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MovieTrackingMarkers(bpy_prop_collection[MovieTrackingMarker], bpy_struct):
    """Collection of markers for movie tracking track"""

    def find_frame(
        self, frame: int | None, exact: bool | None = True
    ) -> MovieTrackingMarker:
        """Get marker for specified frame

        :param frame: Frame, Frame number to find marker for
        :type frame: int | None
        :param exact: Exact, Get marker at exact frame number rather than get estimated marker
        :type exact: bool | None
        :return: Marker for specified frame
        :rtype: MovieTrackingMarker
        """

    def insert_frame(
        self,
        frame: int | None,
        co: collections.abc.Sequence[float] | mathutils.Vector | None = (0.0, 0.0),
    ) -> MovieTrackingMarker:
        """Insert a new marker at the specified frame

        :param frame: Frame, Frame number to insert marker to
        :type frame: int | None
        :param co: Coordinate, Place new marker at the given frame using specified in normalized space coordinates
        :type co: collections.abc.Sequence[float] | mathutils.Vector | None
        :return: Newly created marker
        :rtype: MovieTrackingMarker
        """

    def delete_frame(self, frame: int | None):
        """Delete marker at specified frame

        :param frame: Frame, Frame number to delete marker from
        :type frame: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MovieTrackingObjectPlaneTracks(
    bpy_prop_collection[MovieTrackingPlaneTrack], bpy_struct
):
    """Collection of tracking plane tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MovieTrackingObjectTracks(bpy_prop_collection[MovieTrackingTrack], bpy_struct):
    """Collection of movie tracking tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

    def new(self, name: str = "", frame: int | None = 1) -> MovieTrackingTrack:
        """create new motion track in this movie clip

        :param name: Name of new track
        :type name: str
        :param frame: Frame, Frame number to add tracks on
        :type frame: int | None
        :return: Newly created track
        :rtype: MovieTrackingTrack
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MovieTrackingObjects(bpy_prop_collection[MovieTrackingObject], bpy_struct):
    """Collection of movie tracking objects"""

    active: MovieTrackingObject | None
    """ Active object in this tracking data object

    :type: MovieTrackingObject | None
    """

    def new(self, name: str) -> MovieTrackingObject:
        """Add tracking object to this movie clip

        :param name: Name of new object
        :type name: str
        :return: New motion tracking object
        :rtype: MovieTrackingObject
        """

    def remove(self, object: MovieTrackingObject):
        """Remove tracking object from this movie clip

        :param object: Motion tracking object to be removed
        :type object: MovieTrackingObject
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MovieTrackingPlaneMarkers(
    bpy_prop_collection[MovieTrackingPlaneMarker], bpy_struct
):
    """Collection of markers for movie tracking plane track"""

    def find_frame(
        self, frame: int | None, exact: bool | None = True
    ) -> MovieTrackingPlaneMarker:
        """Get plane marker for specified frame

        :param frame: Frame, Frame number to find marker for
        :type frame: int | None
        :param exact: Exact, Get plane marker at exact frame number rather than get estimated marker
        :type exact: bool | None
        :return: Plane marker for specified frame
        :rtype: MovieTrackingPlaneMarker
        """

    def insert_frame(self, frame: int | None) -> MovieTrackingPlaneMarker:
        """Insert a new plane marker at the specified frame

        :param frame: Frame, Frame number to insert marker to
        :type frame: int | None
        :return: Newly created plane marker
        :rtype: MovieTrackingPlaneMarker
        """

    def delete_frame(self, frame: int | None):
        """Delete plane marker at specified frame

        :param frame: Frame, Frame number to delete plane marker from
        :type frame: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MovieTrackingPlaneTracks(
    bpy_prop_collection[MovieTrackingPlaneTrack], bpy_struct
):
    """Collection of movie tracking plane tracks"""

    active: MovieTrackingPlaneTrack | None
    """ Active plane track in this tracking data object. Deprecated, use objects[name].plane_tracks.active

    :type: MovieTrackingPlaneTrack | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MovieTrackingReconstructedCameras(
    bpy_prop_collection[MovieReconstructedCamera], bpy_struct
):
    """Collection of solved cameras"""

    def find_frame(self, frame: int | None = 1) -> MovieReconstructedCamera:
        """Find a reconstructed camera for a give frame number

        :param frame: Frame, Frame number to find camera for
        :type frame: int | None
        :return: Camera for a given frame
        :rtype: MovieReconstructedCamera
        """

    def matrix_from_frame(self, frame: int | None = 1) -> mathutils.Matrix:
        """Return interpolated camera matrix for a given frame

        :param frame: Frame, Frame number to find camera for
        :type frame: int | None
        :return: Matrix, Interpolated camera matrix for a given frame
        :rtype: mathutils.Matrix
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class MovieTrackingTracks(bpy_prop_collection[MovieTrackingTrack], bpy_struct):
    """Collection of movie tracking tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object. Deprecated, use objects[name].tracks.active

    :type: MovieTrackingTrack | None
    """

    def new(self, name: str = "", frame: int | None = 1) -> MovieTrackingTrack:
        """Create new motion track in this movie clip

        :param name: Name of new track
        :type name: str
        :param frame: Frame, Frame number to add track on
        :type frame: int | None
        :return: Newly created track
        :rtype: MovieTrackingTrack
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class NlaStripFCurves(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of NLA strip F-Curves"""

    def find(self, data_path: str, index: int | None = 0) -> FCurve:
        """Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the NLA strip.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str
        :param index: Index, Array index
        :type index: int | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class NlaStrips(bpy_prop_collection[NlaStrip], bpy_struct):
    """Collection of NLA Strips"""

    def new(self, name: str, start: int | None, action: Action) -> NlaStrip:
        """Add a new Action-Clip strip to the track

        :param name: Name for the NLA Strips
        :type name: str
        :param start: Start Frame, Start frame for this strip
        :type start: int | None
        :param action: Action to assign to this strip
        :type action: Action
        :return: New NLA Strip
        :rtype: NlaStrip
        """

    def remove(self, strip: NlaStrip):
        """Remove a NLA Strip

        :param strip: NLA Strip to remove
        :type strip: NlaStrip
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class NlaTracks(bpy_prop_collection[NlaTrack], bpy_struct):
    """Collection of NLA Tracks"""

    active: NlaTrack | None
    """ Active NLA Track

    :type: NlaTrack | None
    """

    def new(self, prev: NlaTrack | None = None) -> NlaTrack:
        """Add a new NLA Track

        :param prev: NLA Track to add the new one after
        :type prev: NlaTrack | None
        :return: New NLA Track
        :rtype: NlaTrack
        """

    def remove(self, track: NlaTrack):
        """Remove a NLA Track

        :param track: NLA Track to remove
        :type track: NlaTrack
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class NodeGeometryRepeatOutputItems(bpy_prop_collection[RepeatItem], bpy_struct):
    """Collection of repeat items"""

    def new(
        self,
        socket_type: bpy._typing.rna_enums.NodeSocketDataTypeItems | None,
        name: str,
    ) -> RepeatItem:
        """Add a item to this repeat zone

        :param socket_type: Socket Type, Socket type of the item
        :type socket_type: bpy._typing.rna_enums.NodeSocketDataTypeItems | None
        :param name: Name
        :type name: str
        :return: Item, New item
        :rtype: RepeatItem
        """

    def remove(self, item: RepeatItem):
        """Remove an item from this repeat zone

        :param item: Item, The item to remove
        :type item: RepeatItem
        """

    def clear(self):
        """Remove all items from this repeat zone"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move an item to another position

        :param from_index: From Index, Index of the item to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the item
        :type to_index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class NodeGeometrySimulationOutputItems(
    bpy_prop_collection[SimulationStateItem], bpy_struct
):
    """Collection of simulation items"""

    def new(
        self,
        socket_type: bpy._typing.rna_enums.NodeSocketDataTypeItems | None,
        name: str,
    ) -> SimulationStateItem:
        """Add a item to this simulation zone

        :param socket_type: Socket Type, Socket type of the item
        :type socket_type: bpy._typing.rna_enums.NodeSocketDataTypeItems | None
        :param name: Name
        :type name: str
        :return: Item, New item
        :rtype: SimulationStateItem
        """

    def remove(self, item: SimulationStateItem):
        """Remove an item from this simulation zone

        :param item: Item, The item to remove
        :type item: SimulationStateItem
        """

    def clear(self):
        """Remove all items from this simulation zone"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move an item to another position

        :param from_index: From Index, Index of the item to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the item
        :type to_index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class NodeInputs(bpy_prop_collection[NodeSocket], bpy_struct):
    """Collection of Node Sockets"""

    def new(self, type: str, name: str, identifier: str = "") -> NodeSocket:
        """Add a socket to this node

        :param type: Type, Data type
        :type type: str
        :param name: Name
        :type name: str
        :param identifier: Identifier, Unique socket identifier
        :type identifier: str
        :return: New socket
        :rtype: NodeSocket
        """

    def remove(self, socket: NodeSocket | None):
        """Remove a socket from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """

    def clear(self):
        """Remove all sockets from this node"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class NodeLinks(bpy_prop_collection[NodeLink], bpy_struct):
    """Collection of Node Links"""

    def new(
        self, input: NodeSocket, output: NodeSocket, verify_limits: bool | None = True
    ) -> NodeLink:
        """Add a node link to this node tree

        :param input: The input socket
        :type input: NodeSocket
        :param output: The output socket
        :type output: NodeSocket
        :param verify_limits: Verify Limits, Remove existing links if connection limit is exceeded
        :type verify_limits: bool | None
        :return: New node link
        :rtype: NodeLink
        """

    def remove(self, link: NodeLink):
        """remove a node link from the node tree

        :param link: The node link to remove
        :type link: NodeLink
        """

    def clear(self):
        """remove all node links from the node tree"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class NodeOutputs(bpy_prop_collection[NodeSocket], bpy_struct):
    """Collection of Node Sockets"""

    def new(self, type: str, name: str, identifier: str = "") -> NodeSocket:
        """Add a socket to this node

        :param type: Type, Data type
        :type type: str
        :param name: Name
        :type name: str
        :param identifier: Identifier, Unique socket identifier
        :type identifier: str
        :return: New socket
        :rtype: NodeSocket
        """

    def remove(self, socket: NodeSocket | None):
        """Remove a socket from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """

    def clear(self):
        """Remove all sockets from this node"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Nodes(bpy_prop_collection[Node], bpy_struct):
    """Collection of Nodes"""

    active: Node | None
    """ Active node in this tree

    :type: Node | None
    """

    def new(self, type: str) -> Node:
        """Add a node to this node tree

        :param type: Type, Type of node to add (Warning: should be same as node.bl_idname, not node.type!)
        :type type: str
        :return: New node
        :rtype: Node
        """

    def remove(self, node: Node):
        """Remove a node from this node tree

        :param node: The node to remove
        :type node: Node
        """

    def clear(self):
        """Remove all nodes from this node tree"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class NodesModifierBakes(bpy_prop_collection[NodesModifierBake], bpy_struct):
    """Bake data for every bake node"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ObjectConstraints(bpy_prop_collection[Constraint], bpy_struct):
    """Collection of object constraints"""

    active: Constraint | None
    """ Active Object constraint

    :type: Constraint | None
    """

    def new(self, type: bpy._typing.rna_enums.ConstraintTypeItems | None) -> Constraint:
        """Add a new constraint to this object

        :param type: Constraint type to add
        :type type: bpy._typing.rna_enums.ConstraintTypeItems | None
        :return: New constraint
        :rtype: Constraint
        """

    def remove(self, constraint: Constraint):
        """Remove a constraint from this object

        :param constraint: Removed constraint
        :type constraint: Constraint
        """

    def clear(self):
        """Remove all constraint from this object"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a constraint to a different position

        :param from_index: From Index, Index to move
        :type from_index: int | None
        :param to_index: To Index, Target index
        :type to_index: int | None
        """

    def copy(self, constraint: Constraint) -> Constraint:
        """Add a new constraint that is a copy of the given one

        :param constraint: Constraint to copy - may belong to a different object
        :type constraint: Constraint
        :return: New constraint
        :rtype: Constraint
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ObjectGpencilModifiers(bpy_prop_collection[GpencilModifier], bpy_struct):
    """Collection of object grease pencil modifiers"""

    def new(
        self,
        name: str,
        type: bpy._typing.rna_enums.ObjectGreasepencilModifierTypeItems | None,
    ) -> GpencilModifier:
        """Add a new greasepencil_modifier

        :param name: New name for the greasepencil_modifier
        :type name: str
        :param type: Modifier type to add
        :type type: bpy._typing.rna_enums.ObjectGreasepencilModifierTypeItems | None
        :return: Newly created modifier
        :rtype: GpencilModifier
        """

    def remove(self, greasepencil_modifier: GpencilModifier):
        """Remove an existing greasepencil_modifier from the object

        :param greasepencil_modifier: Modifier to remove
        :type greasepencil_modifier: GpencilModifier
        """

    def clear(self):
        """Remove all grease pencil modifiers from the object"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ObjectModifiers(bpy_prop_collection[Modifier], bpy_struct):
    """Collection of object modifiers"""

    active: Modifier | None
    """ The active modifier in the list

    :type: Modifier | None
    """

    def new(
        self, name: str, type: bpy._typing.rna_enums.ObjectModifierTypeItems | None
    ) -> Modifier:
        """Add a new modifier

        :param name: New name for the modifier
        :type name: str
        :param type: Modifier type to add
        :type type: bpy._typing.rna_enums.ObjectModifierTypeItems | None
        :return: Newly created modifier
        :rtype: Modifier
        """

    def remove(self, modifier: Modifier):
        """Remove an existing modifier from the object

        :param modifier: Modifier to remove
        :type modifier: Modifier
        """

    def clear(self):
        """Remove all modifiers from the object"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a modifier to a different position

        :param from_index: From Index, Index to move
        :type from_index: int | None
        :param to_index: To Index, Target index
        :type to_index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ObjectShaderFx(bpy_prop_collection[ShaderFx], bpy_struct):
    """Collection of object effects"""

    def new(
        self, name: str, type: bpy._typing.rna_enums.ObjectShaderfxTypeItems | None
    ) -> ShaderFx:
        """Add a new shader fx

        :param name: New name for the effect
        :type name: str
        :param type: Effect type to add
        :type type: bpy._typing.rna_enums.ObjectShaderfxTypeItems | None
        :return: Newly created effect
        :rtype: ShaderFx
        """

    def remove(self, shader_fx: ShaderFx):
        """Remove an existing effect from the object

        :param shader_fx: Effect to remove
        :type shader_fx: ShaderFx
        """

    def clear(self):
        """Remove all effects from the object"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class PaletteColors(bpy_prop_collection[PaletteColor], bpy_struct):
    """Collection of palette colors"""

    active: PaletteColor | None
    """ 

    :type: PaletteColor | None
    """

    def new(self) -> PaletteColor:
        """Add a new color to the palette

        :return: The newly created color
        :rtype: PaletteColor
        """

    def remove(self, color: PaletteColor):
        """Remove a color from the palette

        :param color: The color to remove
        :type color: PaletteColor
        """

    def clear(self):
        """Remove all colors from the palette"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ParticleSettingsTextureSlots(
    bpy_prop_collection[ParticleSettingsTextureSlot], bpy_struct
):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> ParticleSettingsTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: ParticleSettingsTextureSlot
        """

    @classmethod
    def create(cls, index: int | None) -> ParticleSettingsTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: ParticleSettingsTextureSlot
        """

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ParticleSystems(bpy_prop_collection[ParticleSystem], bpy_struct):
    """Collection of particle systems"""

    active: ParticleSystem | None
    """ Active particle system being displayed

    :type: ParticleSystem | None
    """

    active_index: int | None
    """ Index of active particle system slot

    :type: int | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class PathCompareCollection(bpy_prop_collection[PathCompare], bpy_struct):
    """Collection of paths"""

    @classmethod
    def new(cls) -> PathCompare:
        """Add a new path

        :return:
        :rtype: PathCompare
        """

    @classmethod
    def remove(cls, pathcmp: PathCompare):
        """Remove path

        :param pathcmp:
        :type pathcmp: PathCompare
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class PointCaches(bpy_prop_collection[PointCacheItem], bpy_struct):
    """Collection of point caches"""

    active_index: int | None
    """ 

    :type: int | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class PoseBoneConstraints(bpy_prop_collection[Constraint], bpy_struct):
    """Collection of pose bone constraints"""

    active: Constraint | None
    """ Active PoseChannel constraint

    :type: Constraint | None
    """

    def new(self, type: bpy._typing.rna_enums.ConstraintTypeItems | None) -> Constraint:
        """Add a constraint to this object

        :param type: Constraint type to add
        :type type: bpy._typing.rna_enums.ConstraintTypeItems | None
        :return: New constraint
        :rtype: Constraint
        """

    def remove(self, constraint: Constraint):
        """Remove a constraint from this object

        :param constraint: Removed constraint
        :type constraint: Constraint
        """

    def move(self, from_index: int | None, to_index: int | None):
        """Move a constraint to a different position

        :param from_index: From Index, Index to move
        :type from_index: int | None
        :param to_index: To Index, Target index
        :type to_index: int | None
        """

    def copy(self, constraint: Constraint) -> Constraint:
        """Add a new constraint that is a copy of the given one

        :param constraint: Constraint to copy - may belong to a different object
        :type constraint: Constraint
        :return: New constraint
        :rtype: Constraint
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class RenderPasses(bpy_prop_collection[RenderPass], bpy_struct):
    """Collection of render passes"""

    def find_by_type(
        self, pass_type: bpy._typing.rna_enums.RenderPassTypeItems | None, view: str
    ) -> RenderPass:
        """Get the render pass for a given type and view

        :param pass_type: Pass
        :type pass_type: bpy._typing.rna_enums.RenderPassTypeItems | None
        :param view: View, Render view to get pass from
        :type view: str
        :return: The matching render pass
        :rtype: RenderPass
        """

    def find_by_name(self, name: str, view: str) -> RenderPass:
        """Get the render pass for a given name and view

        :param name: Pass
        :type name: str
        :param view: View, Render view to get pass from
        :type view: str
        :return: The matching render pass
        :rtype: RenderPass
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class RenderSlots(bpy_prop_collection[RenderSlot], bpy_struct):
    """Collection of render layers"""

    active: RenderSlot | None
    """ Active render slot of the image

    :type: RenderSlot | None
    """

    active_index: int | None
    """ Active render slot of the image

    :type: int | None
    """

    def new(self, name: str = "") -> RenderSlot:
        """Add a render slot to the image

        :param name: Name, New name for the render slot
        :type name: str
        :return: Newly created render layer
        :rtype: RenderSlot
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class RenderViews(bpy_prop_collection[SceneRenderView], bpy_struct):
    """Collection of render views"""

    active: SceneRenderView | None
    """ Active Render View

    :type: SceneRenderView | None
    """

    active_index: int | None
    """ Active index in render view array

    :type: int | None
    """

    def new(self, name: str) -> SceneRenderView:
        """Add a render view to scene

        :param name: New name for the marker (not unique)
        :type name: str
        :return: Newly created render view
        :rtype: SceneRenderView
        """

    def remove(self, view: SceneRenderView):
        """Remove a render view

        :param view: Render view to remove
        :type view: SceneRenderView
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class RetimingKeys(bpy_prop_collection[RetimingKey], bpy_struct):
    """Collection of RetimingKey"""

    def add(self, timeline_frame: int | None = 0) -> RetimingKey:
        """Add retiming key

        :param timeline_frame: Timeline Frame
        :type timeline_frame: int | None
        :return: New RetimingKey
        :rtype: RetimingKey
        """

    def reset(self):
        """Remove all retiming keys"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class SceneObjects(bpy_prop_collection[Object], bpy_struct):
    """All of the scene objects"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ScriptDirectoryCollection(bpy_prop_collection[ScriptDirectory], bpy_struct):
    @classmethod
    def new(cls) -> ScriptDirectory:
        """Add a new Python script directory

        :return:
        :rtype: ScriptDirectory
        """

    @classmethod
    def remove(cls, script_directory: ScriptDirectory):
        """Remove a Python script directory

        :param script_directory:
        :type script_directory: ScriptDirectory
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class SequenceElements(bpy_prop_collection[SequenceElement], bpy_struct):
    """Collection of SequenceElement"""

    def append(self, filename: str) -> SequenceElement:
        """Push an image from ImageSequence.directory

        :param filename: Filepath to image
        :type filename: str
        :return: New SequenceElement
        :rtype: SequenceElement
        """

    def pop(self, index: int | None):
        """Pop an image off the collection

        :param index: Index of image to remove
        :type index: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class SequenceModifiers(bpy_prop_collection[SequenceModifier], bpy_struct):
    """Collection of strip modifiers"""

    def new(
        self, name: str, type: bpy._typing.rna_enums.SequenceModifierTypeItems | None
    ) -> SequenceModifier:
        """Add a new modifier

        :param name: New name for the modifier
        :type name: str
        :param type: Modifier type to add
        :type type: bpy._typing.rna_enums.SequenceModifierTypeItems | None
        :return: Newly created modifier
        :rtype: SequenceModifier
        """

    def remove(self, modifier: SequenceModifier):
        """Remove an existing modifier from the sequence

        :param modifier: Modifier to remove
        :type modifier: SequenceModifier
        """

    def clear(self):
        """Remove all modifiers from the sequence"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class SequencesMeta(bpy_prop_collection[Sequence], bpy_struct):
    """Collection of Sequences"""

    def new_clip(
        self, name: str, clip: MovieClip, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new movie clip sequence

        :param name: Name for the new sequence
        :type name: str
        :param clip: Movie clip to add
        :type clip: MovieClip
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_mask(
        self, name: str, mask: Mask, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new mask sequence

        :param name: Name for the new sequence
        :type name: str
        :param mask: Mask to add
        :type mask: Mask
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_scene(
        self, name: str, scene: Scene, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new scene sequence

        :param name: Name for the new sequence
        :type name: str
        :param scene: Scene to add
        :type scene: Scene
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_image(
        self,
        name: str,
        filepath: str,
        channel: int | None,
        frame_start: int | None,
        fit_method: typing.Literal["FIT", "FILL", "STRETCH", "ORIGINAL"]
        | None = "ORIGINAL",
    ) -> Sequence:
        """Add a new image sequence

                :param name: Name for the new sequence
                :type name: str
                :param filepath: Filepath to image
                :type filepath: str
                :param channel: Channel, The channel for the new sequence
                :type channel: int | None
                :param frame_start: The start frame for the new sequence
                :type frame_start: int | None
                :param fit_method: Image Fit Method

        FIT
        Scale to Fit -- Scale image so fits in preview.

        FILL
        Scale to Fill -- Scale image so it fills preview completely.

        STRETCH
        Stretch to Fill -- Stretch image so it fills preview.

        ORIGINAL
        Use Original Size -- Don't scale the image.
                :type fit_method: typing.Literal['FIT','FILL','STRETCH','ORIGINAL'] | None
                :return: New Sequence
                :rtype: Sequence
        """

    def new_movie(
        self,
        name: str,
        filepath: str,
        channel: int | None,
        frame_start: int | None,
        fit_method: typing.Literal["FIT", "FILL", "STRETCH", "ORIGINAL"]
        | None = "ORIGINAL",
    ) -> Sequence:
        """Add a new movie sequence

                :param name: Name for the new sequence
                :type name: str
                :param filepath: Filepath to movie
                :type filepath: str
                :param channel: Channel, The channel for the new sequence
                :type channel: int | None
                :param frame_start: The start frame for the new sequence
                :type frame_start: int | None
                :param fit_method: Image Fit Method

        FIT
        Scale to Fit -- Scale image so fits in preview.

        FILL
        Scale to Fill -- Scale image so it fills preview completely.

        STRETCH
        Stretch to Fill -- Stretch image so it fills preview.

        ORIGINAL
        Use Original Size -- Don't scale the image.
                :type fit_method: typing.Literal['FIT','FILL','STRETCH','ORIGINAL'] | None
                :return: New Sequence
                :rtype: Sequence
        """

    def new_sound(
        self, name: str, filepath: str, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new sound sequence

        :param name: Name for the new sequence
        :type name: str
        :param filepath: Filepath to movie
        :type filepath: str
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_meta(
        self, name: str, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new meta sequence

        :param name: Name for the new sequence
        :type name: str
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_effect(
        self,
        name: str,
        type: typing.Literal[
            "CROSS",
            "ADD",
            "SUBTRACT",
            "ALPHA_OVER",
            "ALPHA_UNDER",
            "GAMMA_CROSS",
            "MULTIPLY",
            "OVER_DROP",
            "WIPE",
            "GLOW",
            "TRANSFORM",
            "COLOR",
            "SPEED",
            "MULTICAM",
            "ADJUSTMENT",
            "GAUSSIAN_BLUR",
            "TEXT",
            "COLORMIX",
        ]
        | None,
        channel: int | None,
        frame_start: int | None,
        frame_end: int | None = 0,
        seq1: Sequence | None = None,
        seq2: Sequence | None = None,
        seq3: Sequence | None = None,
    ) -> Sequence:
        """Add a new effect sequence

        :param name: Name for the new sequence
        :type name: str
        :param type: Type, type for the new sequence
        :type type: typing.Literal['CROSS','ADD','SUBTRACT','ALPHA_OVER','ALPHA_UNDER','GAMMA_CROSS','MULTIPLY','OVER_DROP','WIPE','GLOW','TRANSFORM','COLOR','SPEED','MULTICAM','ADJUSTMENT','GAUSSIAN_BLUR','TEXT','COLORMIX'] | None
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :param frame_end: The end frame for the new sequence
        :type frame_end: int | None
        :param seq1: Sequence 1 for effect
        :type seq1: Sequence | None
        :param seq2: Sequence 2 for effect
        :type seq2: Sequence | None
        :param seq3: Sequence 3 for effect
        :type seq3: Sequence | None
        :return: New Sequence
        :rtype: Sequence
        """

    def remove(self, sequence: Sequence):
        """Remove a Sequence

        :param sequence: Sequence to remove
        :type sequence: Sequence
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class SequencesTopLevel(bpy_prop_collection[Sequence], bpy_struct):
    """Collection of Sequences"""

    def new_clip(
        self, name: str, clip: MovieClip, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new movie clip sequence

        :param name: Name for the new sequence
        :type name: str
        :param clip: Movie clip to add
        :type clip: MovieClip
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_mask(
        self, name: str, mask: Mask, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new mask sequence

        :param name: Name for the new sequence
        :type name: str
        :param mask: Mask to add
        :type mask: Mask
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_scene(
        self, name: str, scene: Scene, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new scene sequence

        :param name: Name for the new sequence
        :type name: str
        :param scene: Scene to add
        :type scene: Scene
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_image(
        self,
        name: str,
        filepath: str,
        channel: int | None,
        frame_start: int | None,
        fit_method: typing.Literal["FIT", "FILL", "STRETCH", "ORIGINAL"]
        | None = "ORIGINAL",
    ) -> Sequence:
        """Add a new image sequence

                :param name: Name for the new sequence
                :type name: str
                :param filepath: Filepath to image
                :type filepath: str
                :param channel: Channel, The channel for the new sequence
                :type channel: int | None
                :param frame_start: The start frame for the new sequence
                :type frame_start: int | None
                :param fit_method: Image Fit Method

        FIT
        Scale to Fit -- Scale image so fits in preview.

        FILL
        Scale to Fill -- Scale image so it fills preview completely.

        STRETCH
        Stretch to Fill -- Stretch image so it fills preview.

        ORIGINAL
        Use Original Size -- Don't scale the image.
                :type fit_method: typing.Literal['FIT','FILL','STRETCH','ORIGINAL'] | None
                :return: New Sequence
                :rtype: Sequence
        """

    def new_movie(
        self,
        name: str,
        filepath: str,
        channel: int | None,
        frame_start: int | None,
        fit_method: typing.Literal["FIT", "FILL", "STRETCH", "ORIGINAL"]
        | None = "ORIGINAL",
    ) -> Sequence:
        """Add a new movie sequence

                :param name: Name for the new sequence
                :type name: str
                :param filepath: Filepath to movie
                :type filepath: str
                :param channel: Channel, The channel for the new sequence
                :type channel: int | None
                :param frame_start: The start frame for the new sequence
                :type frame_start: int | None
                :param fit_method: Image Fit Method

        FIT
        Scale to Fit -- Scale image so fits in preview.

        FILL
        Scale to Fill -- Scale image so it fills preview completely.

        STRETCH
        Stretch to Fill -- Stretch image so it fills preview.

        ORIGINAL
        Use Original Size -- Don't scale the image.
                :type fit_method: typing.Literal['FIT','FILL','STRETCH','ORIGINAL'] | None
                :return: New Sequence
                :rtype: Sequence
        """

    def new_sound(
        self, name: str, filepath: str, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new sound sequence

        :param name: Name for the new sequence
        :type name: str
        :param filepath: Filepath to movie
        :type filepath: str
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_meta(
        self, name: str, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new meta sequence

        :param name: Name for the new sequence
        :type name: str
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_effect(
        self,
        name: str,
        type: typing.Literal[
            "CROSS",
            "ADD",
            "SUBTRACT",
            "ALPHA_OVER",
            "ALPHA_UNDER",
            "GAMMA_CROSS",
            "MULTIPLY",
            "OVER_DROP",
            "WIPE",
            "GLOW",
            "TRANSFORM",
            "COLOR",
            "SPEED",
            "MULTICAM",
            "ADJUSTMENT",
            "GAUSSIAN_BLUR",
            "TEXT",
            "COLORMIX",
        ]
        | None,
        channel: int | None,
        frame_start: int | None,
        frame_end: int | None = 0,
        seq1: Sequence | None = None,
        seq2: Sequence | None = None,
        seq3: Sequence | None = None,
    ) -> Sequence:
        """Add a new effect sequence

        :param name: Name for the new sequence
        :type name: str
        :param type: Type, type for the new sequence
        :type type: typing.Literal['CROSS','ADD','SUBTRACT','ALPHA_OVER','ALPHA_UNDER','GAMMA_CROSS','MULTIPLY','OVER_DROP','WIPE','GLOW','TRANSFORM','COLOR','SPEED','MULTICAM','ADJUSTMENT','GAUSSIAN_BLUR','TEXT','COLORMIX'] | None
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :param frame_end: The end frame for the new sequence
        :type frame_end: int | None
        :param seq1: Sequence 1 for effect
        :type seq1: Sequence | None
        :param seq2: Sequence 2 for effect
        :type seq2: Sequence | None
        :param seq3: Sequence 3 for effect
        :type seq3: Sequence | None
        :return: New Sequence
        :rtype: Sequence
        """

    def remove(self, sequence: Sequence):
        """Remove a Sequence

        :param sequence: Sequence to remove
        :type sequence: Sequence
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class SpaceNodeEditorPath(bpy_prop_collection[NodeTreePath], bpy_struct):
    """Get the node tree path as a string"""

    to_string: str
    """ 

    :type: str
    """

    def clear(self):
        """Reset the node tree path"""

    def start(self, node_tree: NodeTree | None):
        """Set the root node tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        """

    def append(self, node_tree: NodeTree | None, node: Node | None = None):
        """Append a node group tree to the path

        :param node_tree: Node Tree, Node tree to append to the node editor path
        :type node_tree: NodeTree | None
        :param node: Node, Group node linking to this node tree
        :type node: Node | None
        """

    def pop(self):
        """Remove the last node tree from the path"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class SplineBezierPoints(bpy_prop_collection[BezierSplinePoint], bpy_struct):
    """Collection of spline Bezier points"""

    def add(self, count: int | None):
        """Add a number of points to this spline

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class SplinePoints(bpy_prop_collection[SplinePoint], bpy_struct):
    """Collection of spline points"""

    def add(self, count: int | None):
        """Add a number of points to this spline

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class StudioLights(bpy_prop_collection[StudioLight], bpy_struct):
    """Collection of studio lights"""

    def load(
        self, path: str, type: typing.Literal["STUDIO", "WORLD", "MATCAP"] | None
    ) -> StudioLight:
        """Load studiolight from file

        :param path: File Path, File path where the studio light file can be found
        :type path: str
        :param type: Type, The type for the new studio light
        :type type: typing.Literal['STUDIO','WORLD','MATCAP'] | None
        :return: Newly created StudioLight
        :rtype: StudioLight
        """

    def new(self, path: str) -> StudioLight:
        """Create studiolight from default lighting

        :param path: Path, Path to the file that will contain the lighting info (without extension)
        :type path: str
        :return: Newly created StudioLight
        :rtype: StudioLight
        """

    def remove(self, studio_light: StudioLight):
        """Remove a studio light

        :param studio_light: The studio light to remove
        :type studio_light: StudioLight
        """

    def refresh(self):
        """Refresh Studio Lights from disk"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class TimelineMarkers(bpy_prop_collection[TimelineMarker], bpy_struct):
    """Collection of timeline markers"""

    def new(self, name: str, frame: int | None = 1) -> TimelineMarker:
        """Add a keyframe to the curve

        :param name: New name for the marker (not unique)
        :type name: str
        :param frame: The frame for the new marker
        :type frame: int | None
        :return: Newly created timeline marker
        :rtype: TimelineMarker
        """

    def remove(self, marker: TimelineMarker):
        """Remove a timeline marker

        :param marker: Timeline marker to remove
        :type marker: TimelineMarker
        """

    def clear(self):
        """Remove all timeline markers"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class UDIMTiles(bpy_prop_collection[UDIMTile], bpy_struct):
    """Collection of UDIM tiles"""

    active: UDIMTile | None
    """ Active Image Tile

    :type: UDIMTile | None
    """

    active_index: int | None
    """ Active index in tiles array

    :type: int | None
    """

    def new(self, tile_number: int | None, label: str = "") -> UDIMTile:
        """Add a tile to the image

        :param tile_number: Number of the newly created tile
        :type tile_number: int | None
        :param label: Optional label for the tile
        :type label: str
        :return: Newly created image tile
        :rtype: UDIMTile
        """

    def get(self, tile_number: int | None) -> UDIMTile:
        """Get a tile based on its tile number

        :param tile_number: Number of the tile
        :type tile_number: int | None
        :return: The tile
        :rtype: UDIMTile
        """

    def remove(self, tile: UDIMTile):
        """Remove an image tile

        :param tile: Image tile to remove
        :type tile: UDIMTile
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class UVLoopLayers(bpy_prop_collection[MeshUVLoopLayer], bpy_struct):
    """Collection of UV map layers"""

    active: MeshUVLoopLayer | None
    """ Active UV Map layer

    :type: MeshUVLoopLayer | None
    """

    active_index: int | None
    """ Active UV map index

    :type: int | None
    """

    def new(self, name: str = "UVMap", do_init: bool | None = True) -> MeshUVLoopLayer:
        """Add a UV map layer to Mesh

        :param name: UV map name
        :type name: str
        :param do_init: Whether new layer's data should be initialized by copying current active one, or if none is active, with a default UVmap
        :type do_init: bool | None
        :return: The newly created layer
        :rtype: MeshUVLoopLayer
        """

    def remove(self, layer: MeshUVLoopLayer):
        """Remove a vertex color layer

        :param layer: The layer to remove
        :type layer: MeshUVLoopLayer
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class UserExtensionRepoCollection(bpy_prop_collection[UserExtensionRepo], bpy_struct):
    """Collection of user extension repositories"""

    @classmethod
    def new(
        cls,
        name: str = "",
        module: str = "",
        directory: str = "",
        remote_path: str = "",
    ) -> UserExtensionRepo:
        """Add a new repository

        :param name: Name
        :type name: str
        :param module: Module
        :type module: str
        :param directory: Directories
        :type directory: str
        :param remote_path: Remote Path
        :type remote_path: str
        :return: Newly added repository
        :rtype: UserExtensionRepo
        """

    @classmethod
    def remove(cls, repo: UserExtensionRepo):
        """Remove repos

        :param repo: Repository to remove
        :type repo: UserExtensionRepo
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class VertexGroups(bpy_prop_collection[VertexGroup], bpy_struct):
    """Collection of vertex groups"""

    active: VertexGroup | None
    """ Vertex groups of the object

    :type: VertexGroup | None
    """

    active_index: int | None
    """ Active index in vertex group array

    :type: int | None
    """

    def new(self, name: str = "Group") -> VertexGroup:
        """Add vertex group to object

        :param name: Vertex group name
        :type name: str
        :return: New vertex group
        :rtype: VertexGroup
        """

    def remove(self, group: VertexGroup):
        """Delete vertex group from object

        :param group: Vertex group to remove
        :type group: VertexGroup
        """

    def clear(self):
        """Delete all vertex groups from object"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ViewLayers(bpy_prop_collection[ViewLayer], bpy_struct):
    """Collection of render layers"""

    def new(self, name: str) -> ViewLayer:
        """Add a view layer to scene

        :param name: New name for the view layer (not unique)
        :type name: str
        :return: Newly created view layer
        :rtype: ViewLayer
        """

    def remove(self, layer: ViewLayer):
        """Remove a view layer

        :param layer: View layer to remove
        :type layer: ViewLayer
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class VolumeGrids(bpy_prop_collection[VolumeGrid], bpy_struct):
    """3D volume grids"""

    active_index: int | None
    """ Index of active volume grid

    :type: int | None
    """

    error_message: str
    """ If loading grids failed, error message with details

    :type: str
    """

    frame: int
    """ Frame number that volume grids will be loaded at, based on scene time and volume parameters

    :type: int
    """

    frame_filepath: str
    """ Volume file used for loading the volume at the current frame. Empty if the volume has not be loaded or the frame only exists in memory

    :type: str
    """

    is_loaded: bool
    """ List of grids and metadata are loaded in memory

    :type: bool
    """

    def load(self) -> bool:
        """Load list of grids and metadata from file

        :return: True if grid list was successfully loaded
        :rtype: bool
        """

    def unload(self):
        """Unload all grid and voxel data from memory"""

    def save(self, filepath: str) -> bool:
        """Save grids and metadata to file

        :param filepath: File path to save to
        :type filepath: str
        :return: True if grid list was successfully loaded
        :rtype: bool
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class XrActionMapBindings(bpy_prop_collection[XrActionMapBinding], bpy_struct):
    """Collection of XR action map bindings"""

    def new(self, name: str, replace_existing: bool | None) -> XrActionMapBinding:
        """new

        :param name: Name of the action map binding
        :type name: str
        :param replace_existing: Replace Existing, Replace any existing binding with the same name
        :type replace_existing: bool | None
        :return: Binding, Added action map binding
        :rtype: XrActionMapBinding
        """

    def new_from_binding(self, binding: XrActionMapBinding) -> XrActionMapBinding:
        """new_from_binding

        :param binding: Binding, Binding to use as a reference
        :type binding: XrActionMapBinding
        :return: Binding, Added action map binding
        :rtype: XrActionMapBinding
        """

    def remove(self, binding: XrActionMapBinding):
        """remove

        :param binding: Binding
        :type binding: XrActionMapBinding
        """

    def find(self, name: str) -> XrActionMapBinding:
        """find

        :param name: Name
        :type name: str
        :return: Binding, The action map binding with the given name
        :rtype: XrActionMapBinding
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class XrActionMapItems(bpy_prop_collection[XrActionMapItem], bpy_struct):
    """Collection of XR action map items"""

    def new(self, name: str, replace_existing: bool | None) -> XrActionMapItem:
        """new

        :param name: Name of the action map item
        :type name: str
        :param replace_existing: Replace Existing, Replace any existing item with the same name
        :type replace_existing: bool | None
        :return: Item, Added action map item
        :rtype: XrActionMapItem
        """

    def new_from_item(self, item: XrActionMapItem) -> XrActionMapItem:
        """new_from_item

        :param item: Item, Item to use as a reference
        :type item: XrActionMapItem
        :return: Item, Added action map item
        :rtype: XrActionMapItem
        """

    def remove(self, item: XrActionMapItem):
        """remove

        :param item: Item
        :type item: XrActionMapItem
        """

    def find(self, name: str) -> XrActionMapItem:
        """find

        :param name: Name
        :type name: str
        :return: Item, The action map item with the given name
        :rtype: XrActionMapItem
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class XrActionMaps(bpy_prop_collection[XrActionMap], bpy_struct):
    """Collection of XR action maps"""

    @classmethod
    def new(
        cls, xr_session_state: XrSessionState, name: str, replace_existing: bool | None
    ) -> XrActionMap:
        """new

        :param xr_session_state: XR Session State
        :type xr_session_state: XrSessionState
        :param name: Name
        :type name: str
        :param replace_existing: Replace Existing, Replace any existing actionmap with the same name
        :type replace_existing: bool | None
        :return: Action Map, Added action map
        :rtype: XrActionMap
        """

    @classmethod
    def new_from_actionmap(
        cls, xr_session_state: XrSessionState, actionmap: XrActionMap
    ) -> XrActionMap:
        """new_from_actionmap

        :param xr_session_state: XR Session State
        :type xr_session_state: XrSessionState
        :param actionmap: Action Map, Action map to use as a reference
        :type actionmap: XrActionMap
        :return: Action Map, Added action map
        :rtype: XrActionMap
        """

    @classmethod
    def remove(cls, xr_session_state: XrSessionState, actionmap: XrActionMap):
        """remove

        :param xr_session_state: XR Session State
        :type xr_session_state: XrSessionState
        :param actionmap: Action Map, Removed action map
        :type actionmap: XrActionMap
        """

    @classmethod
    def find(cls, xr_session_state: XrSessionState, name: str) -> XrActionMap:
        """find

        :param xr_session_state: XR Session State
        :type xr_session_state: XrSessionState
        :param name: Name
        :type name: str
        :return: Action Map, The action map with the given name
        :rtype: XrActionMap
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class XrComponentPaths(bpy_prop_collection[XrComponentPath], bpy_struct):
    """Collection of OpenXR component paths"""

    def new(self, path: str) -> XrComponentPath:
        """new

        :param path: Path, OpenXR component path
        :type path: str
        :return: Component Path, Added component path
        :rtype: XrComponentPath
        """

    def remove(self, component_path: XrComponentPath):
        """remove

        :param component_path: Component Path
        :type component_path: XrComponentPath
        """

    def find(self, path: str) -> XrComponentPath:
        """find

        :param path: Path, OpenXR component path
        :type path: str
        :return: Component Path, The component path with the given path
        :rtype: XrComponentPath
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class XrUserPaths(bpy_prop_collection[XrUserPath], bpy_struct):
    """Collection of OpenXR user paths"""

    def new(self, path: str) -> XrUserPath:
        """new

        :param path: Path, OpenXR user path
        :type path: str
        :return: User Path, Added user path
        :rtype: XrUserPath
        """

    def remove(self, user_path: XrUserPath):
        """remove

        :param user_path: User Path
        :type user_path: XrUserPath
        """

    def find(self, path: str) -> XrUserPath:
        """find

        :param path: Path, OpenXR user path
        :type path: str
        :return: User Path, The user path with the given path
        :rtype: XrUserPath
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class wmOwnerIDs(bpy_prop_collection[wmOwnerID], bpy_struct):
    def new(self, name: str) -> wmOwnerID:
        """Add ui tag

        :param name: New name for the tag
        :type name: str
        :return:
        :rtype: wmOwnerID
        """

    def remove(self, owner_id: wmOwnerID):
        """Remove ui tag

        :param owner_id: Tag to remove
        :type owner_id: wmOwnerID
        """

    def clear(self):
        """Remove all tags"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class wmTools(bpy_prop_collection[WorkSpaceTool], bpy_struct):
    def from_space_view3d_mode(
        self,
        mode: bpy._typing.rna_enums.ContextModeItems | None,
        create: bool | None = False,
    ) -> WorkSpaceTool:
        """

        :param mode:
        :type mode: bpy._typing.rna_enums.ContextModeItems | None
        :param create: Create
        :type create: bool | None
        :return:
        :rtype: WorkSpaceTool
        """

    def from_space_image_mode(
        self,
        mode: bpy._typing.rna_enums.SpaceImageModeAllItems | None,
        create: bool | None = False,
    ) -> WorkSpaceTool:
        """

        :param mode:
        :type mode: bpy._typing.rna_enums.SpaceImageModeAllItems | None
        :param create: Create
        :type create: bool | None
        :return:
        :rtype: WorkSpaceTool
        """

    def from_space_node(self, create: bool | None = False) -> WorkSpaceTool:
        """

        :param create: Create
        :type create: bool | None
        :return:
        :rtype: WorkSpaceTool
        """

    def from_space_sequencer(
        self,
        mode: bpy._typing.rna_enums.SpaceSequencerViewTypeItems | None,
        create: bool | None = False,
    ) -> WorkSpaceTool:
        """

        :param mode:
        :type mode: bpy._typing.rna_enums.SpaceSequencerViewTypeItems | None
        :param create: Create
        :type create: bool | None
        :return:
        :rtype: WorkSpaceTool
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AOV(bpy_struct):
    is_valid: bool
    """ Is the name of the AOV conflicting

    :type: bool
    """

    name: str
    """ Name of the AOV

    :type: str
    """

    type: typing.Literal["COLOR", "VALUE"]
    """ Data type of the AOV

    :type: typing.Literal['COLOR','VALUE']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ASSETBROWSER_UL_metadata_tags(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        _data,
        item,
        icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param _data:
        :param item:
        :param icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Action(ID, bpy_struct):
    """A collection of F-Curves for animation"""

    curve_frame_range: mathutils.Vector
    """ The combined frame range of all F-Curves within this action

    :type: mathutils.Vector
    """

    fcurves: ActionFCurves
    """ The individual F-Curves that make up the action

    :type: ActionFCurves
    """

    frame_end: float
    """ The end frame of the manually set intended playback range

    :type: float
    """

    frame_range: mathutils.Vector
    """ The intended playback frame range of this action, using the manually set range if available, or the combined frame range of all F-Curves within this action if not (assigning sets the manual frame range)

    :type: mathutils.Vector
    """

    frame_start: float
    """ The start frame of the manually set intended playback range

    :type: float
    """

    groups: ActionGroups
    """ Convenient groupings of F-Curves

    :type: ActionGroups
    """

    id_root: bpy._typing.rna_enums.IdTypeItems
    """ Type of ID block that action can be used on - DO NOT CHANGE UNLESS YOU KNOW WHAT YOU ARE DOING

    :type: bpy._typing.rna_enums.IdTypeItems
    """

    pose_markers: ActionPoseMarkers
    """ Markers specific to this action, for labeling poses

    :type: ActionPoseMarkers
    """

    use_cyclic: bool
    """ The action is intended to be used as a cycle looping over its manually set playback frame range (enabling this doesn't automatically make it loop)

    :type: bool
    """

    use_frame_range: bool
    """ Manually specify the intended playback frame range for the action (this range is used by some tools, but does not affect animation evaluation)

    :type: bool
    """

    def flip_with_pose(self, object: Object):
        """Flip the action around the X axis using a pose

        :param object: The reference armature object to use when flipping
        :type object: Object
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ActionConstraint(Constraint, bpy_struct):
    """Map an action to the transform axes of a bone"""

    action: Action | None
    """ The constraining action

    :type: Action | None
    """

    eval_time: float
    """ Interpolates between Action Start and End frames

    :type: float
    """

    frame_end: int
    """ Last frame of the Action to use

    :type: int
    """

    frame_start: int
    """ First frame of the Action to use

    :type: int
    """

    max: float
    """ Maximum value for target channel range

    :type: float
    """

    min: float
    """ Minimum value for target channel range

    :type: float
    """

    mix_mode: typing.Literal[
        "BEFORE_FULL", "BEFORE", "BEFORE_SPLIT", "AFTER_FULL", "AFTER", "AFTER_SPLIT"
    ]
    """ Specify how existing transformations and the action channels are combined

    :type: typing.Literal['BEFORE_FULL','BEFORE','BEFORE_SPLIT','AFTER_FULL','AFTER','AFTER_SPLIT']
    """

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object | None
    """ Target object

    :type: Object | None
    """

    transform_channel: typing.Literal[
        "LOCATION_X",
        "LOCATION_Y",
        "LOCATION_Z",
        "ROTATION_X",
        "ROTATION_Y",
        "ROTATION_Z",
        "SCALE_X",
        "SCALE_Y",
        "SCALE_Z",
    ]
    """ Transformation channel from the target that is used to key the Action

    :type: typing.Literal['LOCATION_X','LOCATION_Y','LOCATION_Z','ROTATION_X','ROTATION_Y','ROTATION_Z','SCALE_X','SCALE_Y','SCALE_Z']
    """

    use_bone_object_action: bool
    """ Bones only: apply the object's transformation channels of the action to the constrained bone, instead of bone's channels

    :type: bool
    """

    use_eval_time: bool
    """ Interpolate between Action Start and End frames, with the Evaluation Time slider instead of the Target object/bone

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ActionGroup(bpy_struct):
    """Groups of F-Curves"""

    channels: bpy_prop_collection[FCurve]
    """ F-Curves in this group

    :type: bpy_prop_collection[FCurve]
    """

    color_set: bpy._typing.rna_enums.ColorSetsItems
    """ Custom color set to use

    :type: bpy._typing.rna_enums.ColorSetsItems
    """

    colors: ThemeBoneColorSet
    """ Copy of the colors associated with the group's color set

    :type: ThemeBoneColorSet
    """

    is_custom_color_set: bool
    """ Color set is user-defined instead of a fixed theme color set

    :type: bool
    """

    lock: bool
    """ Action group is locked

    :type: bool
    """

    mute: bool
    """ Action group is muted

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    select: bool
    """ Action group is selected

    :type: bool
    """

    show_expanded: bool
    """ Action group is expanded except in graph editor

    :type: bool
    """

    show_expanded_graph: bool
    """ Action group is expanded in graph editor

    :type: bool
    """

    use_pin: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AddSequence(EffectSequence, Sequence, bpy_struct):
    """Add Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Addon(bpy_struct):
    """Python add-ons to be loaded automatically"""

    module: str
    """ Module name

    :type: str
    """

    preferences: AddonPreferences
    """ 

    :type: AddonPreferences
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AddonPreferences(bpy_struct):
    bl_idname: str
    """ 

    :type: str
    """

    layout: UILayout
    """ 

    :type: UILayout
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AdjustmentSequence(EffectSequence, Sequence, bpy_struct):
    """Sequence strip to perform filter adjustments to layers below"""

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    input_count: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AlphaOverSequence(EffectSequence, Sequence, bpy_struct):
    """Alpha Over Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AlphaUnderSequence(EffectSequence, Sequence, bpy_struct):
    """Alpha Under Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AnimData(bpy_struct):
    """Animation data for data-block"""

    action: Action | None
    """ Active Action for this data-block

    :type: Action | None
    """

    action_blend_type: typing.Literal[
        "REPLACE", "COMBINE", "ADD", "SUBTRACT", "MULTIPLY"
    ]
    """ Method used for combining Active Action's result with result of NLA stack

    :type: typing.Literal['REPLACE','COMBINE','ADD','SUBTRACT','MULTIPLY']
    """

    action_extrapolation: typing.Literal["NOTHING", "HOLD", "HOLD_FORWARD"]
    """ Action to take for gaps past the Active Action's range (when evaluating with NLA)

    :type: typing.Literal['NOTHING','HOLD','HOLD_FORWARD']
    """

    action_influence: float
    """ Amount the Active Action contributes to the result of the NLA stack

    :type: float
    """

    action_tweak_storage: Action | None
    """ Slot to temporarily hold the main action while in tweak mode

    :type: Action | None
    """

    drivers: AnimDataDrivers
    """ The Drivers/Expressions for this data-block

    :type: AnimDataDrivers
    """

    nla_tracks: NlaTracks
    """ NLA Tracks (i.e. Animation Layers)

    :type: NlaTracks
    """

    use_nla: bool
    """ NLA stack is evaluated when evaluating this block

    :type: bool
    """

    use_pin: bool
    """ 

    :type: bool
    """

    use_tweak_mode: bool
    """ Whether to enable or disable tweak mode in NLA

    :type: bool
    """

    def nla_tweak_strip_time_to_scene(
        self, frame: float | None, invert: bool | None = False
    ) -> float:
        """Convert a time value from the local time of the tweaked strip to scene time, exactly as done by built-in key editing tools. Returns the input time unchanged if not tweaking.

        :param frame: Input time
        :type frame: float | None
        :param invert: Invert, Convert scene time to action time
        :type invert: bool | None
        :return: Converted time
        :rtype: float
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AnimViz(bpy_struct):
    """Settings for the visualization of motion"""

    motion_path: AnimVizMotionPaths
    """ Motion Path settings for visualization

    :type: AnimVizMotionPaths
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AnimVizMotionPaths(bpy_struct):
    """Motion Path settings for animation visualization"""

    bake_location: bpy._typing.rna_enums.MotionpathBakeLocationItems
    """ When calculating Bone Paths, use Head or Tips

    :type: bpy._typing.rna_enums.MotionpathBakeLocationItems
    """

    frame_after: int
    """ Number of frames to show after the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_before: int
    """ Number of frames to show before the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_end: int
    """ End frame of range of paths to display/calculate (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_start: int
    """ Starting frame of range of paths to display/calculate (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_step: int
    """ Number of frames between paths shown (not for 'On Keyframes' Onion-skinning method)

    :type: int
    """

    has_motion_paths: bool
    """ Are there any bone paths that will need updating (read-only)

    :type: bool
    """

    range: bpy._typing.rna_enums.MotionpathRangeItems
    """ Type of range to calculate for Motion Paths

    :type: bpy._typing.rna_enums.MotionpathRangeItems
    """

    show_frame_numbers: bool
    """ Show frame numbers on Motion Paths

    :type: bool
    """

    show_keyframe_action_all: bool
    """ For bone motion paths, search whole Action for keyframes instead of in group with matching name only (is slower)

    :type: bool
    """

    show_keyframe_highlight: bool
    """ Emphasize position of keyframes on Motion Paths

    :type: bool
    """

    show_keyframe_numbers: bool
    """ Show frame numbers of Keyframes on Motion Paths

    :type: bool
    """

    type: bpy._typing.rna_enums.MotionpathDisplayTypeItems
    """ Type of range to show for Motion Paths

    :type: bpy._typing.rna_enums.MotionpathDisplayTypeItems
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AnyType(bpy_struct):
    """RNA type used for pointers to any possible data"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Area(bpy_struct):
    """Area in a subdivided screen, containing an editor"""

    height: int
    """ Area height

    :type: int
    """

    regions: bpy_prop_collection[Region]
    """ Regions this area is subdivided in

    :type: bpy_prop_collection[Region]
    """

    show_menus: bool
    """ Show menus in the header

    :type: bool
    """

    spaces: AreaSpaces
    """ Spaces contained in this area, the first being the active space (NOTE: Useful for example to restore a previously used 3D view space in a certain area to get the old view orientation)

    :type: AreaSpaces
    """

    type: bpy._typing.rna_enums.SpaceTypeItems
    """ Current editor type for this area

    :type: bpy._typing.rna_enums.SpaceTypeItems
    """

    ui_type: str
    """ Current editor type for this area

    :type: str
    """

    width: int
    """ Area width

    :type: int
    """

    x: int
    """ The window relative vertical location of the area

    :type: int
    """

    y: int
    """ The window relative horizontal location of the area

    :type: int
    """

    def tag_redraw(self):
        """tag_redraw"""

    def header_text_set(self, text: str | None):
        """Set the header status text

        :param text: Text, New string for the header, None clears the text
        :type text: str | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AreaLight(Light, ID, bpy_struct):
    """Directional area Light"""

    contact_shadow_bias: float
    """ Bias to avoid self shadowing

    :type: float
    """

    contact_shadow_distance: float
    """ World space distance in which to search for screen space occluder

    :type: float
    """

    contact_shadow_thickness: float
    """ Pixel thickness used to detect occlusion

    :type: float
    """

    energy: float
    """ Light energy emitted over the entire area of the light in all directions

    :type: float
    """

    shadow_buffer_bias: float
    """ Bias for reducing self shadowing

    :type: float
    """

    shadow_buffer_clip_start: float
    """ Shadow map clip start, below which objects will not generate shadows

    :type: float
    """

    shadow_color: mathutils.Color
    """ Color of shadows cast by the light

    :type: mathutils.Color
    """

    shadow_soft_size: float
    """ Light size for ray shadow sampling (Raytraced shadows)

    :type: float
    """

    shadow_softness_factor: float
    """ Scale light shape for smaller penumbra

    :type: float
    """

    shape: typing.Literal["SQUARE", "RECTANGLE", "DISK", "ELLIPSE"]
    """ Shape of the area Light

    :type: typing.Literal['SQUARE','RECTANGLE','DISK','ELLIPSE']
    """

    size: float
    """ Size of the area of the area light, X direction size for rectangle shapes

    :type: float
    """

    size_y: float
    """ Size of the area of the area light in the Y direction for rectangle shapes

    :type: float
    """

    spread: float
    """ How widely the emitted light fans out, as in the case of a gridded softbox

    :type: float
    """

    use_contact_shadow: bool
    """ Use screen space ray-tracing to have correct shadowing near occluder, or for small features that does not appear in shadow maps

    :type: bool
    """

    use_shadow: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Armature(ID, bpy_struct):
    """Armature data-block containing a hierarchy of bones, usually used for rigging characters"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    axes_position: float
    """ The position for the axes on the bone. Increasing the value moves it closer to the tip; decreasing moves it closer to the root

    :type: float
    """

    bones: ArmatureBones
    """ 

    :type: ArmatureBones
    """

    collections: BoneCollections
    """ 

    :type: BoneCollections
    """

    display_type: typing.Literal["OCTAHEDRAL", "STICK", "BBONE", "ENVELOPE", "WIRE"]
    """ 

    :type: typing.Literal['OCTAHEDRAL','STICK','BBONE','ENVELOPE','WIRE']
    """

    edit_bones: ArmatureEditBones
    """ 

    :type: ArmatureEditBones
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    pose_position: typing.Literal["POSE", "REST"]
    """ Show armature in binding pose or final posed state

    :type: typing.Literal['POSE','REST']
    """

    relation_line_position: typing.Literal["TAIL", "HEAD"]
    """ The start position of the relation lines from parent to child bones

    :type: typing.Literal['TAIL','HEAD']
    """

    show_axes: bool
    """ Display bone axes

    :type: bool
    """

    show_bone_colors: bool
    """ Display bone colors

    :type: bool
    """

    show_bone_custom_shapes: bool
    """ Display bones with their custom shapes

    :type: bool
    """

    show_names: bool
    """ Display bone names

    :type: bool
    """

    use_mirror_x: bool
    """ Apply changes to matching bone on opposite side of X-Axis

    :type: bool
    """

    def transform(
        self,
        matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None,
    ):
        """Transform armature bones by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ArmatureConstraint(Constraint, bpy_struct):
    """Applies transformations done by the Armature modifier"""

    targets: ArmatureConstraintTargets
    """ Target Bones

    :type: ArmatureConstraintTargets
    """

    use_bone_envelopes: bool
    """ Multiply weights by envelope for all bones, instead of acting like Vertex Group based blending. The specified weights are still used, and only the listed bones are considered

    :type: bool
    """

    use_current_location: bool
    """ Use the current bone location for envelopes and choosing B-Bone segments instead of rest position

    :type: bool
    """

    use_deform_preserve_volume: bool
    """ Deform rotation interpolation with quaternions

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ArmatureGpencilModifier(GpencilModifier, bpy_struct):
    """Change stroke using armature to deform modifier"""

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    object: Object | None
    """ Armature object to deform with

    :type: Object | None
    """

    use_bone_envelopes: bool
    """ Bind Bone envelopes to armature modifier

    :type: bool
    """

    use_deform_preserve_volume: bool
    """ Deform rotation interpolation with quaternions

    :type: bool
    """

    use_vertex_groups: bool
    """ Bind vertex groups to armature modifier

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ArmatureModifier(Modifier, bpy_struct):
    """Armature deformation modifier"""

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    object: Object | None
    """ Armature object to deform with

    :type: Object | None
    """

    use_bone_envelopes: bool
    """ Bind Bone envelopes to armature modifier

    :type: bool
    """

    use_deform_preserve_volume: bool
    """ Deform rotation interpolation with quaternions

    :type: bool
    """

    use_multi_modifier: bool
    """ Use same input as previous modifier, and mix results using overall vgroup

    :type: bool
    """

    use_vertex_groups: bool
    """ Bind vertex groups to armature modifier

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ArrayGpencilModifier(GpencilModifier, bpy_struct):
    """Create grid of duplicate instances"""

    constant_offset: mathutils.Vector
    """ Value for the distance between items

    :type: mathutils.Vector
    """

    count: int
    """ Number of items

    :type: int
    """

    invert_layer_pass: bool
    """ Inverse filter

    :type: bool
    """

    invert_layers: bool
    """ Inverse filter

    :type: bool
    """

    invert_material_pass: bool
    """ Inverse filter

    :type: bool
    """

    invert_materials: bool
    """ Inverse filter

    :type: bool
    """

    layer: str
    """ Layer name

    :type: str
    """

    layer_pass: int
    """ Layer pass index

    :type: int
    """

    material: Material | None
    """ Material used for filtering effect

    :type: Material | None
    """

    offset_object: Object | None
    """ Use the location and rotation of another object to determine the distance and rotational change between arrayed items

    :type: Object | None
    """

    pass_index: int
    """ Pass index

    :type: int
    """

    random_offset: mathutils.Vector
    """ Value for changes in location

    :type: mathutils.Vector
    """

    random_rotation: mathutils.Euler
    """ Value for changes in rotation

    :type: mathutils.Euler
    """

    random_scale: mathutils.Vector
    """ Value for changes in scale

    :type: mathutils.Vector
    """

    relative_offset: mathutils.Vector
    """ The size of the geometry will determine the distance between arrayed items

    :type: mathutils.Vector
    """

    replace_material: int
    """ Index of the material used for generated strokes (0 keep original material)

    :type: int
    """

    seed: int
    """ Random seed

    :type: int
    """

    use_constant_offset: bool
    """ Enable offset

    :type: bool
    """

    use_object_offset: bool
    """ Enable object offset

    :type: bool
    """

    use_relative_offset: bool
    """ Enable shift

    :type: bool
    """

    use_uniform_random_scale: bool
    """ Use the same random seed for each scale axis for a uniform scale

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ArrayModifier(Modifier, bpy_struct):
    """Array duplication modifier"""

    constant_offset_displace: mathutils.Vector
    """ Value for the distance between arrayed items

    :type: mathutils.Vector
    """

    count: int
    """ Number of duplicates to make

    :type: int
    """

    curve: Object | None
    """ Curve object to fit array length to

    :type: Object | None
    """

    end_cap: Object | None
    """ Mesh object to use as an end cap

    :type: Object | None
    """

    fit_length: float
    """ Length to fit array within

    :type: float
    """

    fit_type: typing.Literal["FIXED_COUNT", "FIT_LENGTH", "FIT_CURVE"]
    """ Array length calculation method

    :type: typing.Literal['FIXED_COUNT','FIT_LENGTH','FIT_CURVE']
    """

    merge_threshold: float
    """ Limit below which to merge vertices

    :type: float
    """

    offset_object: Object | None
    """ Use the location and rotation of another object to determine the distance and rotational change between arrayed items

    :type: Object | None
    """

    offset_u: float
    """ Amount to offset array UVs on the U axis

    :type: float
    """

    offset_v: float
    """ Amount to offset array UVs on the V axis

    :type: float
    """

    relative_offset_displace: mathutils.Vector
    """ The size of the geometry will determine the distance between arrayed items

    :type: mathutils.Vector
    """

    start_cap: Object | None
    """ Mesh object to use as a start cap

    :type: Object | None
    """

    use_constant_offset: bool
    """ Add a constant offset

    :type: bool
    """

    use_merge_vertices: bool
    """ Merge vertices in adjacent duplicates

    :type: bool
    """

    use_merge_vertices_cap: bool
    """ Merge vertices in first and last duplicates

    :type: bool
    """

    use_object_offset: bool
    """ Add another object's transformation to the total offset

    :type: bool
    """

    use_relative_offset: bool
    """ Add an offset relative to the object's bounding box

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AssetCatalogPath(bpy_struct):
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AssetHandle(PropertyGroup, bpy_struct):
    """Reference to some asset"""

    file_data: FileSelectEntry | None
    """ TEMPORARY, DO NOT USE - File data used to refer to the asset

    :type: FileSelectEntry | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AssetLibraryReference(bpy_struct):
    """Identifier to refer to the asset library"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AssetMetaData(bpy_struct):
    """Additional data stored for an asset data-block"""

    active_tag: int | None
    """ Index of the tag set for editing

    :type: int | None
    """

    author: str
    """ Name of the creator of the asset

    :type: str
    """

    catalog_id: str
    """ Identifier for the asset's catalog, used by Blender to look up the asset's catalog path. Must be a UUID according to RFC4122

    :type: str
    """

    catalog_simple_name: str
    """ Simple name of the asset's catalog, for debugging and data recovery purposes

    :type: str
    """

    copyright: str
    """ Copyright notice for this asset. An empty copyright notice does not necessarily indicate that this is copyright-free. Contact the author if any clarification is needed

    :type: str
    """

    description: str
    """ A description of the asset to be displayed for the user

    :type: str
    """

    license: str
    """ The type of license this asset is distributed under. An empty license name does not necessarily indicate that this is free of licensing terms. Contact the author if any clarification is needed

    :type: str
    """

    tags: AssetTags
    """ Custom tags (name tokens) for the asset, used for filtering and general asset management

    :type: AssetTags
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AssetRepresentation(bpy_struct):
    """Information about an entity that makes it possible for the asset system to deal with the entity as asset"""

    full_library_path: str
    """ Absolute path to the .blend file containing this asset

    :type: str
    """

    full_path: str
    """ Absolute path to the .blend file containing this asset extended with the path of the asset inside the file

    :type: str
    """

    id_type: bpy._typing.rna_enums.IdTypeItems
    """ The type of the data-block, if the asset represents one ('NONE' otherwise)

    :type: bpy._typing.rna_enums.IdTypeItems
    """

    local_id: ID
    """ The local data-block this asset represents; only valid if that is a data-block in this file

    :type: ID
    """

    metadata: AssetMetaData
    """ Additional information about the asset

    :type: AssetMetaData
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AssetShelf(bpy_struct):
    """Regions for quick access to assets"""

    asset_library_reference: str
    """ Choose the asset library to display assets from

    :type: str
    """

    bl_idname: str
    """ If this is set, the asset gets a custom ID, otherwise it takes the name of the class used to define the menu (for example, if the class name is "OBJECT_AST_hello", and bl_idname is not set by the script, then bl_idname = "OBJECT_AST_hello")

    :type: str
    """

    bl_options: set[typing.Literal["NO_ASSET_DRAG"]]
    """ Options for this asset shelf type

    :type: set[typing.Literal['NO_ASSET_DRAG']]
    """

    bl_space_type: bpy._typing.rna_enums.SpaceTypeItems
    """ The space where the asset shelf is going to be used in

    :type: bpy._typing.rna_enums.SpaceTypeItems
    """

    preview_size: int
    """ Size of the asset preview thumbnails in pixels

    :type: int
    """

    search_filter: str
    """ Filter assets by name

    :type: str
    """

    show_names: bool
    """ Show the asset name together with the preview. Otherwise only the preview will be visible

    :type: bool
    """

    @classmethod
    def poll(cls, context: Context) -> bool:
        """If this method returns a non-null output, the asset shelf will be visible

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """

    @classmethod
    def asset_poll(cls, asset: AssetRepresentation | None) -> bool:
        """Determine if an asset should be visible in the asset shelf. If this method returns a non-null output, the asset will be visible

        :param asset:
        :type asset: AssetRepresentation | None
        :return:
        :rtype: bool
        """

    @classmethod
    def draw_context_menu(
        cls,
        context: Context,
        asset: AssetRepresentation | None,
        layout: UILayout | None,
    ):
        """Draw UI elements into the context menu UI layout displayed on right click

        :param context:
        :type context: Context
        :param asset:
        :type asset: AssetRepresentation | None
        :param layout:
        :type layout: UILayout | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class AssetTag(bpy_struct):
    """User defined tag (name token)"""

    name: str
    """ The identifier that makes up this tag

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Attribute(bpy_struct):
    """Geometry attribute"""

    data_type: bpy._typing.rna_enums.AttributeTypeItems
    """ Type of data stored in attribute

    :type: bpy._typing.rna_enums.AttributeTypeItems
    """

    domain: bpy._typing.rna_enums.AttributeDomainItems
    """ Domain of the Attribute

    :type: bpy._typing.rna_enums.AttributeDomainItems
    """

    is_internal: bool
    """ The attribute is meant for internal use by Blender

    :type: bool
    """

    is_required: bool
    """ Whether the attribute can be removed or renamed

    :type: bool
    """

    name: str
    """ Name of the Attribute

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BakeSettings(bpy_struct):
    """Bake data for a Scene data-block"""

    cage_extrusion: float
    """ Inflate the active object by the specified distance for baking. This helps matching to points nearer to the outside of the selected object meshes

    :type: float
    """

    cage_object: Object | None
    """ Object to use as cage instead of calculating the cage from the active object with cage extrusion

    :type: Object | None
    """

    filepath: str
    """ Image filepath to use when saving externally

    :type: str
    """

    height: int
    """ Vertical dimension of the baking map

    :type: int
    """

    image_settings: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    margin: int
    """ Extends the baked result as a post process filter

    :type: int
    """

    margin_type: bpy._typing.rna_enums.BakeMarginTypeItems
    """ Algorithm to extend the baked result

    :type: bpy._typing.rna_enums.BakeMarginTypeItems
    """

    max_ray_distance: float
    """ The maximum ray distance for matching points between the active and selected objects. If zero, there is no limit

    :type: float
    """

    normal_b: bpy._typing.rna_enums.NormalSwizzleItems
    """ Axis to bake in blue channel

    :type: bpy._typing.rna_enums.NormalSwizzleItems
    """

    normal_g: bpy._typing.rna_enums.NormalSwizzleItems
    """ Axis to bake in green channel

    :type: bpy._typing.rna_enums.NormalSwizzleItems
    """

    normal_r: bpy._typing.rna_enums.NormalSwizzleItems
    """ Axis to bake in red channel

    :type: bpy._typing.rna_enums.NormalSwizzleItems
    """

    normal_space: bpy._typing.rna_enums.NormalSpaceItems
    """ Choose normal space for baking

    :type: bpy._typing.rna_enums.NormalSpaceItems
    """

    pass_filter: set[bpy._typing.rna_enums.BakePassFilterTypeItems] | typing.Any
    """ Passes to include in the active baking pass

    :type: set[bpy._typing.rna_enums.BakePassFilterTypeItems] | typing.Any
    """

    save_mode: bpy._typing.rna_enums.BakeSaveModeItems
    """ Where to save baked image textures

    :type: bpy._typing.rna_enums.BakeSaveModeItems
    """

    target: bpy._typing.rna_enums.BakeTargetItems
    """ Where to output the baked map

    :type: bpy._typing.rna_enums.BakeTargetItems
    """

    use_automatic_name: bool
    """ Automatically name the output file with the pass type (external only)

    :type: bool
    """

    use_cage: bool
    """ Cast rays to active object from a cage

    :type: bool
    """

    use_clear: bool
    """ Clear Images before baking (internal only)

    :type: bool
    """

    use_pass_color: bool
    """ Color the pass

    :type: bool
    """

    use_pass_diffuse: bool
    """ Add diffuse contribution

    :type: bool
    """

    use_pass_direct: bool
    """ Add direct lighting contribution

    :type: bool
    """

    use_pass_emit: bool
    """ Add emission contribution

    :type: bool
    """

    use_pass_glossy: bool
    """ Add glossy contribution

    :type: bool
    """

    use_pass_indirect: bool
    """ Add indirect lighting contribution

    :type: bool
    """

    use_pass_transmission: bool
    """ Add transmission contribution

    :type: bool
    """

    use_selected_to_active: bool
    """ Bake shading on the surface of selected objects to the active object

    :type: bool
    """

    use_split_materials: bool
    """ Split external images per material (external only)

    :type: bool
    """

    view_from: typing.Literal["ABOVE_SURFACE", "ACTIVE_CAMERA"]
    """ Source of reflection ray directions

    :type: typing.Literal['ABOVE_SURFACE','ACTIVE_CAMERA']
    """

    width: int
    """ Horizontal dimension of the baking map

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BevelModifier(Modifier, bpy_struct):
    """Bevel modifier to make edges and vertices more rounded"""

    affect: typing.Literal["VERTICES", "EDGES"]
    """ Affect edges or vertices

    :type: typing.Literal['VERTICES','EDGES']
    """

    angle_limit: float
    """ Angle above which to bevel edges

    :type: float
    """

    custom_profile: CurveProfile
    """ The path for the custom profile

    :type: CurveProfile
    """

    face_strength_mode: typing.Literal[
        "FSTR_NONE", "FSTR_NEW", "FSTR_AFFECTED", "FSTR_ALL"
    ]
    """ Whether to set face strength, and which faces to set it on

    :type: typing.Literal['FSTR_NONE','FSTR_NEW','FSTR_AFFECTED','FSTR_ALL']
    """

    harden_normals: bool
    """ Match normals of new faces to adjacent faces

    :type: bool
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    limit_method: typing.Literal["NONE", "ANGLE", "WEIGHT", "VGROUP"]
    """ 

    :type: typing.Literal['NONE','ANGLE','WEIGHT','VGROUP']
    """

    loop_slide: bool
    """ Prefer sliding along edges to having even widths

    :type: bool
    """

    mark_seam: bool
    """ Mark Seams along beveled edges

    :type: bool
    """

    mark_sharp: bool
    """ Mark beveled edges as sharp

    :type: bool
    """

    material: int
    """ Material index of generated faces, -1 for automatic

    :type: int
    """

    miter_inner: typing.Literal["MITER_SHARP", "MITER_ARC"]
    """ Pattern to use for inside of miters

    :type: typing.Literal['MITER_SHARP','MITER_ARC']
    """

    miter_outer: typing.Literal["MITER_SHARP", "MITER_PATCH", "MITER_ARC"]
    """ Pattern to use for outside of miters

    :type: typing.Literal['MITER_SHARP','MITER_PATCH','MITER_ARC']
    """

    offset_type: typing.Literal["OFFSET", "WIDTH", "DEPTH", "PERCENT", "ABSOLUTE"]
    """ What distance Width measures

    :type: typing.Literal['OFFSET','WIDTH','DEPTH','PERCENT','ABSOLUTE']
    """

    profile: float
    """ The profile shape (0.5 = round)

    :type: float
    """

    profile_type: typing.Literal["SUPERELLIPSE", "CUSTOM"]
    """ The type of shape used to rebuild a beveled section

    :type: typing.Literal['SUPERELLIPSE','CUSTOM']
    """

    segments: int
    """ Number of segments for round edges/verts

    :type: int
    """

    spread: float
    """ Spread distance for inner miter arcs

    :type: float
    """

    use_clamp_overlap: bool
    """ Clamp the width to avoid overlap

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

    vmesh_method: typing.Literal["ADJ", "CUTOFF"]
    """ The method to use to create the mesh at intersections

    :type: typing.Literal['ADJ','CUTOFF']
    """

    width: float
    """ Bevel amount

    :type: float
    """

    width_pct: float
    """ Bevel amount for percentage method

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BezierSplinePoint(bpy_struct):
    """Bezier curve point with two handles"""

    co: mathutils.Vector
    """ Coordinates of the control point

    :type: mathutils.Vector
    """

    handle_left: mathutils.Vector
    """ Coordinates of the first handle

    :type: mathutils.Vector
    """

    handle_left_type: typing.Literal["FREE", "VECTOR", "ALIGNED", "AUTO"]
    """ Handle types

    :type: typing.Literal['FREE','VECTOR','ALIGNED','AUTO']
    """

    handle_right: mathutils.Vector
    """ Coordinates of the second handle

    :type: mathutils.Vector
    """

    handle_right_type: typing.Literal["FREE", "VECTOR", "ALIGNED", "AUTO"]
    """ Handle types

    :type: typing.Literal['FREE','VECTOR','ALIGNED','AUTO']
    """

    hide: bool
    """ Visibility status

    :type: bool
    """

    radius: float
    """ Radius for beveling

    :type: float
    """

    select_control_point: bool
    """ Control point selection status

    :type: bool
    """

    select_left_handle: bool
    """ Handle 1 selection status

    :type: bool
    """

    select_right_handle: bool
    """ Handle 2 selection status

    :type: bool
    """

    tilt: float
    """ Tilt in 3D View

    :type: float
    """

    weight_softbody: float
    """ Softbody goal weight

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlendData(bpy_struct):
    """Main data structure representing a .blend file and all its data-blocks"""

    actions: BlendDataActions
    """ Action data-blocks

    :type: BlendDataActions
    """

    armatures: BlendDataArmatures
    """ Armature data-blocks

    :type: BlendDataArmatures
    """

    brushes: BlendDataBrushes
    """ Brush data-blocks

    :type: BlendDataBrushes
    """

    cache_files: BlendDataCacheFiles
    """ Cache Files data-blocks

    :type: BlendDataCacheFiles
    """

    cameras: BlendDataCameras
    """ Camera data-blocks

    :type: BlendDataCameras
    """

    collections: BlendDataCollections
    """ Collection data-blocks

    :type: BlendDataCollections
    """

    curves: BlendDataCurves
    """ Curve data-blocks

    :type: BlendDataCurves
    """

    filepath: str
    """ Path to the .blend file

    :type: str
    """

    fonts: BlendDataFonts
    """ Vector font data-blocks

    :type: BlendDataFonts
    """

    grease_pencils: BlendDataGreasePencils
    """ Grease Pencil data-blocks

    :type: BlendDataGreasePencils
    """

    hair_curves: BlendDataHairCurves
    """ Hair curve data-blocks

    :type: BlendDataHairCurves
    """

    images: BlendDataImages
    """ Image data-blocks

    :type: BlendDataImages
    """

    is_dirty: bool
    """ Have recent edits been saved to disk

    :type: bool
    """

    is_saved: bool
    """ Has the current session been saved to disk as a .blend file

    :type: bool
    """

    lattices: BlendDataLattices
    """ Lattice data-blocks

    :type: BlendDataLattices
    """

    libraries: BlendDataLibraries
    """ Library data-blocks

    :type: BlendDataLibraries
    """

    lightprobes: BlendDataProbes
    """ Light Probe data-blocks

    :type: BlendDataProbes
    """

    lights: BlendDataLights
    """ Light data-blocks

    :type: BlendDataLights
    """

    linestyles: BlendDataLineStyles
    """ Line Style data-blocks

    :type: BlendDataLineStyles
    """

    masks: BlendDataMasks
    """ Masks data-blocks

    :type: BlendDataMasks
    """

    materials: BlendDataMaterials
    """ Material data-blocks

    :type: BlendDataMaterials
    """

    meshes: BlendDataMeshes
    """ Mesh data-blocks

    :type: BlendDataMeshes
    """

    metaballs: BlendDataMetaBalls
    """ Metaball data-blocks

    :type: BlendDataMetaBalls
    """

    movieclips: BlendDataMovieClips
    """ Movie Clip data-blocks

    :type: BlendDataMovieClips
    """

    node_groups: BlendDataNodeTrees
    """ Node group data-blocks

    :type: BlendDataNodeTrees
    """

    objects: BlendDataObjects
    """ Object data-blocks

    :type: BlendDataObjects
    """

    paint_curves: BlendDataPaintCurves
    """ Paint Curves data-blocks

    :type: BlendDataPaintCurves
    """

    palettes: BlendDataPalettes
    """ Palette data-blocks

    :type: BlendDataPalettes
    """

    particles: BlendDataParticles
    """ Particle data-blocks

    :type: BlendDataParticles
    """

    pointclouds: BlendDataPointClouds
    """ Point cloud data-blocks

    :type: BlendDataPointClouds
    """

    scenes: BlendDataScenes
    """ Scene data-blocks

    :type: BlendDataScenes
    """

    screens: BlendDataScreens
    """ Screen data-blocks

    :type: BlendDataScreens
    """

    shape_keys: bpy_prop_collection[Key]
    """ Shape Key data-blocks

    :type: bpy_prop_collection[Key]
    """

    sounds: BlendDataSounds
    """ Sound data-blocks

    :type: BlendDataSounds
    """

    speakers: BlendDataSpeakers
    """ Speaker data-blocks

    :type: BlendDataSpeakers
    """

    texts: BlendDataTexts
    """ Text data-blocks

    :type: BlendDataTexts
    """

    textures: BlendDataTextures
    """ Texture data-blocks

    :type: BlendDataTextures
    """

    use_autopack: bool
    """ Automatically pack all external data into .blend file

    :type: bool
    """

    version: bpy_prop_array[int]
    """ File format version the .blend file was saved with

    :type: bpy_prop_array[int]
    """

    volumes: BlendDataVolumes
    """ Volume data-blocks

    :type: BlendDataVolumes
    """

    window_managers: BlendDataWindowManagers
    """ Window manager data-blocks

    :type: BlendDataWindowManagers
    """

    workspaces: BlendDataWorkSpaces
    """ Workspace data-blocks

    :type: BlendDataWorkSpaces
    """

    worlds: BlendDataWorlds
    """ World data-blocks

    :type: BlendDataWorlds
    """

    def batch_remove(self, ids):
        """Remove (delete) several IDs at once.WARNING: Considered experimental feature currently.Note that this function is quicker than individual calls to `remove()` (from `bpy.types.BlendData`
        ID collections), but less safe/versatile (it can break Blender, e.g. by removing all scenes...).

                :param ids: Iterables of IDs (types can be mixed).
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

    def orphans_purge(self):
        """Remove (delete) all IDs with no user.

        :return: The number of deleted IDs.
        """

    def temp_data(self, filepath: bytes | str | None = None) -> typing_extensions.Self:
        """A context manager that temporarily creates blender file data.

        :param filepath: The file path for the newly temporary data. When None, the path of the currently open file is used.
        :type filepath: bytes | str | None
        :return: Blend file data which is freed once the context exists.
        :rtype: typing_extensions.Self
        """

    def user_map(
        self,
        subset: collections.abc.Sequence | None,
        key_types: set[str] | None,
        value_types: set[str] | None,
    ) -> dict:
        """Returns a mapping of all ID data-blocks in current bpy.data to a set of all datablocks using them.For list of valid set members for key_types & value_types, see: `bpy.types.KeyingSetPath.id_type`.

        :param subset: When passed, only these data-blocks and their users will be included as keys/values in the map.
        :type subset: collections.abc.Sequence | None
        :param key_types: Filter the keys mapped by ID types.
        :type key_types: set[str] | None
        :param value_types: Filter the values in the set by ID types.
        :type value_types: set[str] | None
        :return: dictionary of `bpy.types.ID` instances, with sets of ID's as their values.
        :rtype: dict
        """

class BlendTexture(Texture, ID, bpy_struct):
    """Procedural color blending texture"""

    progression: typing.Literal[
        "LINEAR",
        "QUADRATIC",
        "EASING",
        "DIAGONAL",
        "SPHERICAL",
        "QUADRATIC_SPHERE",
        "RADIAL",
    ]
    """ Style of the color blending

    :type: typing.Literal['LINEAR','QUADRATIC','EASING','DIAGONAL','SPHERICAL','QUADRATIC_SPHERE','RADIAL']
    """

    use_flip_axis: typing.Literal["HORIZONTAL", "VERTICAL"]
    """ Flip the texture's X and Y axis

    :type: typing.Literal['HORIZONTAL','VERTICAL']
    """

    users_material: typing.Any
    """ Materials that use this texture(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture(readonly)"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BlenderRNA(bpy_struct):
    """Blender RNA structure definitions"""

    structs: bpy_prop_collection[Struct]
    """ 

    :type: bpy_prop_collection[Struct]
    """

    properties: bpy_prop_collection[Property]
    """ 

    :type: bpy_prop_collection[Property]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoidRule(bpy_struct):
    name: str
    """ Boid rule name

    :type: str
    """

    type: bpy._typing.rna_enums.BoidruleTypeItems
    """ 

    :type: bpy._typing.rna_enums.BoidruleTypeItems
    """

    use_in_air: bool
    """ Use rule when boid is flying

    :type: bool
    """

    use_on_land: bool
    """ Use rule when boid is on land

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoidRuleAverageSpeed(BoidRule, bpy_struct):
    level: float
    """ How much velocity's z-component is kept constant

    :type: float
    """

    speed: float
    """ Percentage of maximum speed

    :type: float
    """

    wander: float
    """ How fast velocity's direction is randomized

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoidRuleAvoid(BoidRule, bpy_struct):
    fear_factor: float
    """ Avoid object if danger from it is above this threshold

    :type: float
    """

    object: Object | None
    """ Object to avoid

    :type: Object | None
    """

    use_predict: bool
    """ Predict target movement

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoidRuleAvoidCollision(BoidRule, bpy_struct):
    look_ahead: float
    """ Time to look ahead in seconds

    :type: float
    """

    use_avoid: bool
    """ Avoid collision with other boids

    :type: bool
    """

    use_avoid_collision: bool
    """ Avoid collision with deflector objects

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoidRuleFight(BoidRule, bpy_struct):
    distance: float
    """ Attack boids at max this distance

    :type: float
    """

    flee_distance: float
    """ Flee to this distance

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoidRuleFollowLeader(BoidRule, bpy_struct):
    distance: float
    """ Distance behind leader to follow

    :type: float
    """

    object: Object | None
    """ Follow this object instead of a boid

    :type: Object | None
    """

    queue_count: int
    """ How many boids in a line

    :type: int
    """

    use_line: bool
    """ Follow leader in a line

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoidRuleGoal(BoidRule, bpy_struct):
    object: Object | None
    """ Goal object

    :type: Object | None
    """

    use_predict: bool
    """ Predict target movement

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoidSettings(bpy_struct):
    """Settings for boid physics"""

    accuracy: float
    """ Accuracy of attack

    :type: float
    """

    active_boid_state: BoidRule | None
    """ 

    :type: BoidRule | None
    """

    active_boid_state_index: int | None
    """ 

    :type: int | None
    """

    aggression: float
    """ Boid will fight this times stronger enemy

    :type: float
    """

    air_acc_max: float
    """ Maximum acceleration in air (relative to maximum speed)

    :type: float
    """

    air_ave_max: float
    """ Maximum angular velocity in air (relative to 180 degrees)

    :type: float
    """

    air_personal_space: float
    """ Radius of boids personal space in air (% of particle size)

    :type: float
    """

    air_speed_max: float
    """ Maximum speed in air

    :type: float
    """

    air_speed_min: float
    """ Minimum speed in air (relative to maximum speed)

    :type: float
    """

    bank: float
    """ Amount of rotation around velocity vector on turns

    :type: float
    """

    health: float
    """ Initial boid health when born

    :type: float
    """

    height: float
    """ Boid height relative to particle size

    :type: float
    """

    land_acc_max: float
    """ Maximum acceleration on land (relative to maximum speed)

    :type: float
    """

    land_ave_max: float
    """ Maximum angular velocity on land (relative to 180 degrees)

    :type: float
    """

    land_jump_speed: float
    """ Maximum speed for jumping

    :type: float
    """

    land_personal_space: float
    """ Radius of boids personal space on land (% of particle size)

    :type: float
    """

    land_smooth: float
    """ How smoothly the boids land

    :type: float
    """

    land_speed_max: float
    """ Maximum speed on land

    :type: float
    """

    land_stick_force: float
    """ How strong a force must be to start effecting a boid on land

    :type: float
    """

    pitch: float
    """ Amount of rotation around side vector

    :type: float
    """

    range: float
    """ Maximum distance from which a boid can attack

    :type: float
    """

    states: bpy_prop_collection[BoidState]
    """ 

    :type: bpy_prop_collection[BoidState]
    """

    strength: float
    """ Maximum caused damage on attack per second

    :type: float
    """

    use_climb: bool
    """ Allow boids to climb goal objects

    :type: bool
    """

    use_flight: bool
    """ Allow boids to move in air

    :type: bool
    """

    use_land: bool
    """ Allow boids to move on land

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoidState(bpy_struct):
    """Boid state for boid physics"""

    active_boid_rule: BoidRule | None
    """ 

    :type: BoidRule | None
    """

    active_boid_rule_index: int | None
    """ 

    :type: int | None
    """

    falloff: float
    """ 

    :type: float
    """

    name: str
    """ Boid state name

    :type: str
    """

    rule_fuzzy: float
    """ 

    :type: float
    """

    rules: bpy_prop_collection[BoidRule]
    """ 

    :type: bpy_prop_collection[BoidRule]
    """

    ruleset_type: typing.Literal["FUZZY", "RANDOM", "AVERAGE"]
    """ How the rules in the list are evaluated

    :type: typing.Literal['FUZZY','RANDOM','AVERAGE']
    """

    volume: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Bone(bpy_struct):
    """Bone in an Armature data-block"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveinz: float
    """ Z-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutz: float
    """ Z-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_custom_handle_end: typing_extensions.Self | None
    """ Bone that serves as the end handle for the B-Bone curve

    :type: typing_extensions.Self | None
    """

    bbone_custom_handle_start: typing_extensions.Self | None
    """ Bone that serves as the start handle for the B-Bone curve

    :type: typing_extensions.Self | None
    """

    bbone_easein: float
    """ Length of first Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_easeout: float
    """ Length of second Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_handle_type_end: typing.Literal["AUTO", "ABSOLUTE", "RELATIVE", "TANGENT"]
    """ Selects how the end handle of the B-Bone is computed

    :type: typing.Literal['AUTO','ABSOLUTE','RELATIVE','TANGENT']
    """

    bbone_handle_type_start: typing.Literal["AUTO", "ABSOLUTE", "RELATIVE", "TANGENT"]
    """ Selects how the start handle of the B-Bone is computed

    :type: typing.Literal['AUTO','ABSOLUTE','RELATIVE','TANGENT']
    """

    bbone_handle_use_ease_end: bool
    """ Multiply the B-Bone Ease Out channel by the local Y scale value of the end handle. This is done after the Scale Easing option and isn't affected by it

    :type: bool
    """

    bbone_handle_use_ease_start: bool
    """ Multiply the B-Bone Ease In channel by the local Y scale value of the start handle. This is done after the Scale Easing option and isn't affected by it

    :type: bool
    """

    bbone_handle_use_scale_end: bpy_prop_array[bool]
    """ Multiply B-Bone Scale Out channels by the local scale values of the end handle. This is done after the Scale Easing option and isn't affected by it

    :type: bpy_prop_array[bool]
    """

    bbone_handle_use_scale_start: bpy_prop_array[bool]
    """ Multiply B-Bone Scale In channels by the local scale values of the start handle. This is done after the Scale Easing option and isn't affected by it

    :type: bpy_prop_array[bool]
    """

    bbone_mapping_mode: typing.Literal["STRAIGHT", "CURVED"]
    """ Selects how the vertices are mapped to B-Bone segments based on their position

    :type: typing.Literal['STRAIGHT','CURVED']
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: mathutils.Vector
    """ Scale factors for the start of the B-Bone, adjusts thickness (for tapering effects)

    :type: mathutils.Vector
    """

    bbone_scaleout: mathutils.Vector
    """ Scale factors for the end of the B-Bone, adjusts thickness (for tapering effects)

    :type: mathutils.Vector
    """

    bbone_segments: int
    """ Number of subdivisions of bone (for B-Bones only)

    :type: int
    """

    bbone_x: float
    """ B-Bone X size

    :type: float
    """

    bbone_z: float
    """ B-Bone Z size

    :type: float
    """

    children: bpy_prop_collection[Bone]
    """ Bones which are children of this bone

    :type: bpy_prop_collection[Bone]
    """

    collections: BoneCollectionMemberships
    """ Bone Collections that contain this bone

    :type: BoneCollectionMemberships
    """

    color: BoneColor
    """ 

    :type: BoneColor
    """

    envelope_distance: float
    """ Bone deformation distance (for Envelope deform only)

    :type: float
    """

    envelope_weight: float
    """ Bone deformation weight (for Envelope deform only)

    :type: float
    """

    head: mathutils.Vector
    """ Location of head end of the bone relative to its parent

    :type: mathutils.Vector
    """

    head_local: mathutils.Vector
    """ Location of head end of the bone relative to armature

    :type: mathutils.Vector
    """

    head_radius: float
    """ Radius of head of bone (for Envelope deform only)

    :type: float
    """

    hide: bool
    """ Bone is not visible when it is not in Edit Mode (i.e. in Object or Pose Modes)

    :type: bool
    """

    hide_select: bool
    """ Bone is able to be selected

    :type: bool
    """

    inherit_scale: typing.Literal[
        "FULL", "FIX_SHEAR", "ALIGNED", "AVERAGE", "NONE", "NONE_LEGACY"
    ]
    """ Specifies how the bone inherits scaling from the parent bone

    :type: typing.Literal['FULL','FIX_SHEAR','ALIGNED','AVERAGE','NONE','NONE_LEGACY']
    """

    length: float
    """ Length of the bone

    :type: float
    """

    matrix: mathutils.Matrix
    """ 3×3 bone matrix

    :type: mathutils.Matrix
    """

    matrix_local: mathutils.Matrix
    """ 4×4 bone matrix relative to armature

    :type: mathutils.Matrix
    """

    name: str
    """ 

    :type: str
    """

    parent: typing_extensions.Self
    """ Parent bone (in same Armature)

    :type: typing_extensions.Self
    """

    select: bool
    """ 

    :type: bool
    """

    select_head: bool
    """ 

    :type: bool
    """

    select_tail: bool
    """ 

    :type: bool
    """

    show_wire: bool
    """ Bone is always displayed in wireframe regardless of viewport shading mode (useful for non-obstructive custom bone shapes)

    :type: bool
    """

    tail: mathutils.Vector
    """ Location of tail end of the bone relative to its parent

    :type: mathutils.Vector
    """

    tail_local: mathutils.Vector
    """ Location of tail end of the bone relative to armature

    :type: mathutils.Vector
    """

    tail_radius: float
    """ Radius of tail of bone (for Envelope deform only)

    :type: float
    """

    use_connect: bool
    """ When bone has a parent, bone's head is stuck to the parent's tail

    :type: bool
    """

    use_cyclic_offset: bool
    """ When bone doesn't have a parent, it receives cyclic offset effects (Deprecated)

    :type: bool
    """

    use_deform: bool
    """ Enable Bone to deform geometry

    :type: bool
    """

    use_endroll_as_inroll: bool
    """ Add Roll Out of the Start Handle bone to the Roll In value

    :type: bool
    """

    use_envelope_multiply: bool
    """ When deforming bone, multiply effects of Vertex Group weights with Envelope influence

    :type: bool
    """

    use_inherit_rotation: bool
    """ Bone inherits rotation or scale from parent bone

    :type: bool
    """

    use_local_location: bool
    """ Bone location is set in local space

    :type: bool
    """

    use_relative_parent: bool
    """ Object children will use relative transform, like deform

    :type: bool
    """

    use_scale_easing: bool
    """ Multiply the final easing values by the Scale In/Out Y factors

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.(readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.(readonly)"""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.(readonly)"""

    def evaluate_envelope(
        self, point: collections.abc.Sequence[float] | mathutils.Vector | None
    ) -> float:
        """Calculate bone envelope at given point

        :param point: Point, Position in 3d space to evaluate
        :type point: collections.abc.Sequence[float] | mathutils.Vector | None
        :return: Factor, Envelope factor
        :rtype: float
        """

    def convert_local_to_pose(
        self,
        matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None,
        matrix_local: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None,
        parent_matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None = (
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
        ),
        parent_matrix_local: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None = (
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
            (0.0, 0.0, 0.0, 0.0),
        ),
        invert: bool | None = False,
    ) -> mathutils.Matrix:
        """Transform a matrix from Local to Pose space (or back), taking into account options like Inherit Scale and Local Location. Unlike Object.convert_space, this uses custom rest and pose matrices provided by the caller. If the parent matrices are omitted, the bone is assumed to have no parent.This method enables conversions between Local and Pose space for bones in
        the middle of updating the armature without having to update dependencies
        after each change, by manually carrying updated matrices in a recursive walk.

                :param matrix: The matrix to transform
                :type matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
                :param matrix_local: The custom rest matrix of this bone (Bone.matrix_local)
                :type matrix_local: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
                :param parent_matrix: The custom pose matrix of the parent bone (PoseBone.matrix)
                :type parent_matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
                :param parent_matrix_local: The custom rest matrix of the parent bone (Bone.matrix_local)
                :type parent_matrix_local: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
                :param invert: Convert from Pose to Local space
                :type invert: bool | None
                :return: The transformed matrix
                :rtype: mathutils.Matrix
        """

    @classmethod
    def MatrixFromAxisRoll(
        cls,
        axis: collections.abc.Sequence[float] | mathutils.Vector,
        roll: float | None,
    ) -> mathutils.Matrix:
        """Convert the axis + roll representation to a matrix

        :param axis: The main axis of the bone (tail - head)
        :type axis: collections.abc.Sequence[float] | mathutils.Vector
        :param roll: The roll of the bone
        :type roll: float | None
        :return: The resulting orientation matrix
        :rtype: mathutils.Matrix
        """

    @classmethod
    def AxisRollFromMatrix(
        cls,
        matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix,
        axis: collections.abc.Iterable[float] | None = (0.0, 0.0, 0.0),
    ):
        """Convert a rotational matrix to the axis + roll representation. Note that the resulting value of the roll may not be as expected if the matrix has shear or negative determinant.

                :param matrix: The orientation matrix of the bone
                :type matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix
                :param axis: The optional override for the axis (finds closest approximation for the matrix)
                :type axis: collections.abc.Iterable[float] | None
                :return: result_axis, The main axis of the bone, `mathutils.Vector` of 3 items in [-inf, inf]

        result_roll, The roll of the bone, float in [-inf, inf]
        """

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoneCollection(bpy_struct):
    """Bone collection in an Armature data-block"""

    bones: bpy_prop_collection[Bone]
    """ Bones assigned to this bone collection. In armature edit mode this will always return an empty list of bones, as the bone collection memberships are only synchronised when exiting edit mode

    :type: bpy_prop_collection[Bone]
    """

    is_editable: bool
    """ This collection is owned by a local Armature, or was added via a library override in the current blend file

    :type: bool
    """

    is_local_override: bool
    """ This collection was added via a library override in the current blend file

    :type: bool
    """

    is_visible: bool
    """ Bones in this collection will be visible in pose/object mode

    :type: bool
    """

    name: str
    """ Unique within the Armature

    :type: str
    """

    def assign(self, bone: typing.Any | None) -> bool:
        """Assign the given bone to this collection

        :param bone: Bone to assign to this collection. This must be a Bone, PoseBone, or EditBone
        :type bone: typing.Any | None
        :return: Assigned, Whether the bone was actually assigned; will be false if the bone was already member of the collection
        :rtype: bool
        """

    def unassign(self, bone: typing.Any | None) -> bool:
        """Remove the given bone from this collection

        :param bone: Bone to remove from this collection. This must be a Bone, PoseBone, or EditBone
        :type bone: typing.Any | None
        :return: Unassigned, Whether the bone was actually removed; will be false if the bone was not a member of the collection to begin with
        :rtype: bool
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoneColor(bpy_struct):
    """Theme color or custom color of a bone"""

    custom: ThemeBoneColorSet
    """ The custom bone colors, used when palette is 'CUSTOM'

    :type: ThemeBoneColorSet
    """

    is_custom: bool
    """ A color palette is user-defined, instead of using a theme-defined one

    :type: bool
    """

    palette: typing.Literal[
        "DEFAULT",
        "THEME01",
        "THEME02",
        "THEME03",
        "THEME04",
        "THEME05",
        "THEME06",
        "THEME07",
        "THEME08",
        "THEME09",
        "THEME10",
        "THEME11",
        "THEME12",
        "THEME13",
        "THEME14",
        "THEME15",
        "THEME16",
        "THEME17",
        "THEME18",
        "THEME19",
        "THEME20",
        "CUSTOM",
    ]
    """ Color palette to use

    :type: typing.Literal['DEFAULT','THEME01','THEME02','THEME03','THEME04','THEME05','THEME06','THEME07','THEME08','THEME09','THEME10','THEME11','THEME12','THEME13','THEME14','THEME15','THEME16','THEME17','THEME18','THEME19','THEME20','CUSTOM']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoolAttribute(Attribute, bpy_struct):
    """Geometry attribute that stores booleans"""

    data: bpy_prop_collection[BoolAttributeValue]
    """ 

    :type: bpy_prop_collection[BoolAttributeValue]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoolAttributeValue(bpy_struct):
    """Bool value in geometry attribute"""

    value: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BoolProperty(Property, bpy_struct):
    """RNA boolean property definition"""

    array_dimensions: bpy_prop_array[int]
    """ Length of each dimension of the array

    :type: bpy_prop_array[int]
    """

    array_length: int
    """ Maximum length of the array, 0 means unlimited

    :type: int
    """

    default: bool
    """ Default value for this number

    :type: bool
    """

    default_array: bpy_prop_array[bool]
    """ Default value for this array

    :type: bpy_prop_array[bool]
    """

    is_array: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BooleanModifier(Modifier, bpy_struct):
    """Boolean operations modifier"""

    collection: Collection | None
    """ Use mesh objects in this collection for Boolean operation

    :type: Collection | None
    """

    debug_options: set[str]
    """ Debugging options, only when started with '-d'

    :type: set[str]
    """

    double_threshold: float
    """ Threshold for checking overlapping geometry

    :type: float
    """

    material_mode: typing.Literal["INDEX", "TRANSFER"]
    """ Method for setting materials on the new faces

    :type: typing.Literal['INDEX','TRANSFER']
    """

    object: Object | None
    """ Mesh object to use for Boolean operation

    :type: Object | None
    """

    operand_type: typing.Literal["OBJECT", "COLLECTION"]
    """ 

    :type: typing.Literal['OBJECT','COLLECTION']
    """

    operation: typing.Literal["INTERSECT", "UNION", "DIFFERENCE"]
    """ 

    :type: typing.Literal['INTERSECT','UNION','DIFFERENCE']
    """

    solver: typing.Literal["FAST", "EXACT"]
    """ Method for calculating booleans

    :type: typing.Literal['FAST','EXACT']
    """

    use_hole_tolerant: bool
    """ Better results when there are holes (slower)

    :type: bool
    """

    use_self: bool
    """ Allow self-intersection in operands

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BrightContrastModifier(SequenceModifier, bpy_struct):
    """Bright/contrast modifier data for sequence strip"""

    bright: float
    """ Adjust the luminosity of the colors

    :type: float
    """

    contrast: float
    """ Adjust the difference in luminosity between pixels

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Brush(ID, bpy_struct):
    """Brush data-block for storing brush settings for painting and sculpting"""

    area_radius_factor: float
    """ Ratio between the brush radius and the radius that is going to be used to sample the area center

    :type: float
    """

    auto_smooth_factor: float
    """ Amount of smoothing to automatically apply to each stroke

    :type: float
    """

    automasking_boundary_edges_propagation_steps: int
    """ Distance where boundary edge automasking is going to protect vertices from the fully masked edge

    :type: int
    """

    automasking_cavity_blur_steps: int
    """ The number of times the cavity mask is blurred

    :type: int
    """

    automasking_cavity_curve: CurveMapping
    """ Curve used for the sensitivity

    :type: CurveMapping
    """

    automasking_cavity_factor: float
    """ The contrast of the cavity mask

    :type: float
    """

    blend: typing.Literal[
        "MIX",
        "DARKEN",
        "MUL",
        "COLORBURN",
        "LINEARBURN",
        "LIGHTEN",
        "SCREEN",
        "COLORDODGE",
        "ADD",
        "OVERLAY",
        "SOFTLIGHT",
        "HARDLIGHT",
        "VIVIDLIGHT",
        "LINEARLIGHT",
        "PINLIGHT",
        "DIFFERENCE",
        "EXCLUSION",
        "SUB",
        "HUE",
        "SATURATION",
        "COLOR",
        "LUMINOSITY",
        "ERASE_ALPHA",
        "ADD_ALPHA",
    ]
    """ Brush blending mode

    :type: typing.Literal['MIX','DARKEN','MUL','COLORBURN','LINEARBURN','LIGHTEN','SCREEN','COLORDODGE','ADD','OVERLAY','SOFTLIGHT','HARDLIGHT','VIVIDLIGHT','LINEARLIGHT','PINLIGHT','DIFFERENCE','EXCLUSION','SUB','HUE','SATURATION','COLOR','LUMINOSITY','ERASE_ALPHA','ADD_ALPHA']
    """

    blur_kernel_radius: int
    """ Radius of kernel used for soften and sharpen in pixels

    :type: int
    """

    blur_mode: typing.Literal["BOX", "GAUSSIAN"]
    """ 

    :type: typing.Literal['BOX','GAUSSIAN']
    """

    boundary_deform_type: typing.Literal[
        "BEND", "EXPAND", "INFLATE", "GRAB", "TWIST", "SMOOTH"
    ]
    """ Deformation type that is used in the brush

    :type: typing.Literal['BEND','EXPAND','INFLATE','GRAB','TWIST','SMOOTH']
    """

    boundary_falloff_type: typing.Literal["CONSTANT", "RADIUS", "LOOP", "LOOP_INVERT"]
    """ How the brush falloff is applied across the boundary

    :type: typing.Literal['CONSTANT','RADIUS','LOOP','LOOP_INVERT']
    """

    boundary_offset: float
    """ Offset of the boundary origin in relation to the brush radius

    :type: float
    """

    brush_capabilities: BrushCapabilities
    """ Brush's capabilities

    :type: BrushCapabilities
    """

    clone_alpha: float
    """ Opacity of clone image display

    :type: float
    """

    clone_image: Image | None
    """ Image for clone tool

    :type: Image | None
    """

    clone_offset: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    cloth_constraint_softbody_strength: float
    """ How much the cloth preserves the original shape, acting as a soft body

    :type: float
    """

    cloth_damping: float
    """ How much the applied forces are propagated through the cloth

    :type: float
    """

    cloth_deform_type: typing.Literal[
        "DRAG",
        "PUSH",
        "PINCH_POINT",
        "PINCH_PERPENDICULAR",
        "INFLATE",
        "GRAB",
        "EXPAND",
        "SNAKE_HOOK",
    ]
    """ Deformation type that is used in the brush

    :type: typing.Literal['DRAG','PUSH','PINCH_POINT','PINCH_PERPENDICULAR','INFLATE','GRAB','EXPAND','SNAKE_HOOK']
    """

    cloth_force_falloff_type: typing.Literal["RADIAL", "PLANE"]
    """ Shape used in the brush to apply force to the cloth

    :type: typing.Literal['RADIAL','PLANE']
    """

    cloth_mass: float
    """ Mass of each simulation particle

    :type: float
    """

    cloth_sim_falloff: float
    """ Area to apply deformation falloff to the effects of the simulation

    :type: float
    """

    cloth_sim_limit: float
    """ Factor added relative to the size of the radius to limit the cloth simulation effects

    :type: float
    """

    cloth_simulation_area_type: typing.Literal["LOCAL", "GLOBAL", "DYNAMIC"]
    """ Part of the mesh that is going to be simulated when the stroke is active

    :type: typing.Literal['LOCAL','GLOBAL','DYNAMIC']
    """

    color: mathutils.Color
    """ 

    :type: mathutils.Color
    """

    color_type: typing.Literal["COLOR", "GRADIENT"]
    """ Use single color or gradient when painting

    :type: typing.Literal['COLOR','GRADIENT']
    """

    crease_pinch_factor: float
    """ How much the crease brush pinches

    :type: float
    """

    cursor_color_add: bpy_prop_array[float]
    """ Color of cursor when adding

    :type: bpy_prop_array[float]
    """

    cursor_color_subtract: bpy_prop_array[float]
    """ Color of cursor when subtracting

    :type: bpy_prop_array[float]
    """

    cursor_overlay_alpha: int
    """ 

    :type: int
    """

    curve: CurveMapping
    """ Editable falloff curve

    :type: CurveMapping
    """

    curve_preset: typing.Literal[
        "CUSTOM",
        "SMOOTH",
        "SMOOTHER",
        "SPHERE",
        "ROOT",
        "SHARP",
        "LIN",
        "POW4",
        "INVSQUARE",
        "CONSTANT",
    ]
    """ 

    :type: typing.Literal['CUSTOM','SMOOTH','SMOOTHER','SPHERE','ROOT','SHARP','LIN','POW4','INVSQUARE','CONSTANT']
    """

    curves_sculpt_settings: BrushCurvesSculptSettings
    """ 

    :type: BrushCurvesSculptSettings
    """

    curves_sculpt_tool: bpy._typing.rna_enums.BrushCurvesSculptToolItems
    """ 

    :type: bpy._typing.rna_enums.BrushCurvesSculptToolItems
    """

    dash_ratio: float
    """ Ratio of samples in a cycle that the brush is enabled

    :type: float
    """

    dash_samples: int
    """ Length of a dash cycle measured in stroke samples

    :type: int
    """

    deform_target: typing.Literal["GEOMETRY", "CLOTH_SIM"]
    """ How the deformation of the brush will affect the object

    :type: typing.Literal['GEOMETRY','CLOTH_SIM']
    """

    density: float
    """ Amount of random elements that are going to be affected by the brush

    :type: float
    """

    direction: typing.Literal["ADD", "SUBTRACT"]
    """ 

    :type: typing.Literal['ADD','SUBTRACT']
    """

    disconnected_distance_max: float
    """ Maximum distance to search for disconnected loose parts in the mesh

    :type: float
    """

    elastic_deform_type: typing.Literal[
        "GRAB", "GRAB_BISCALE", "GRAB_TRISCALE", "SCALE", "TWIST"
    ]
    """ Deformation type that is used in the brush

    :type: typing.Literal['GRAB','GRAB_BISCALE','GRAB_TRISCALE','SCALE','TWIST']
    """

    elastic_deform_volume_preservation: float
    """ Poisson ratio for elastic deformation. Higher values preserve volume more, but also lead to more bulging

    :type: float
    """

    falloff_angle: float
    """ Paint most on faces pointing towards the view according to this angle

    :type: float
    """

    falloff_shape: typing.Literal["SPHERE", "PROJECTED"]
    """ Use projected or spherical falloff

    :type: typing.Literal['SPHERE','PROJECTED']
    """

    fill_threshold: float
    """ Threshold above which filling is not propagated

    :type: float
    """

    flow: float
    """ Amount of paint that is applied per stroke sample

    :type: float
    """

    gpencil_sculpt_tool: bpy._typing.rna_enums.BrushGpencilSculptTypesItems
    """ 

    :type: bpy._typing.rna_enums.BrushGpencilSculptTypesItems
    """

    gpencil_settings: BrushGpencilSettings
    """ 

    :type: BrushGpencilSettings
    """

    gpencil_tool: bpy._typing.rna_enums.BrushGpencilTypesItems
    """ 

    :type: bpy._typing.rna_enums.BrushGpencilTypesItems
    """

    gpencil_vertex_tool: bpy._typing.rna_enums.BrushGpencilVertexTypesItems
    """ 

    :type: bpy._typing.rna_enums.BrushGpencilVertexTypesItems
    """

    gpencil_weight_tool: bpy._typing.rna_enums.BrushGpencilWeightTypesItems
    """ 

    :type: bpy._typing.rna_enums.BrushGpencilWeightTypesItems
    """

    grad_spacing: int
    """ Spacing before brush gradient goes full circle

    :type: int
    """

    gradient: ColorRamp
    """ 

    :type: ColorRamp
    """

    gradient_fill_mode: typing.Literal["LINEAR", "RADIAL"]
    """ 

    :type: typing.Literal['LINEAR','RADIAL']
    """

    gradient_stroke_mode: typing.Literal["PRESSURE", "SPACING_REPEAT", "SPACING_CLAMP"]
    """ 

    :type: typing.Literal['PRESSURE','SPACING_REPEAT','SPACING_CLAMP']
    """

    hardness: float
    """ How close the brush falloff starts from the edge of the brush

    :type: float
    """

    height: float
    """ Affectable height of brush (layer height for layer tool, i.e.)

    :type: float
    """

    icon_filepath: str
    """ File path to brush icon

    :type: str
    """

    image_paint_capabilities: BrushCapabilitiesImagePaint
    """ 

    :type: BrushCapabilitiesImagePaint
    """

    image_tool: bpy._typing.rna_enums.BrushImageToolItems
    """ 

    :type: bpy._typing.rna_enums.BrushImageToolItems
    """

    invert_density_pressure: bool
    """ Invert the modulation of pressure in density

    :type: bool
    """

    invert_flow_pressure: bool
    """ Invert the modulation of pressure in flow

    :type: bool
    """

    invert_hardness_pressure: bool
    """ Invert the modulation of pressure in hardness

    :type: bool
    """

    invert_to_scrape_fill: bool
    """ Use Scrape or Fill tool when inverting this brush instead of inverting its displacement direction

    :type: bool
    """

    invert_wet_mix_pressure: bool
    """ Invert the modulation of pressure in wet mix

    :type: bool
    """

    invert_wet_persistence_pressure: bool
    """ Invert the modulation of pressure in wet persistence

    :type: bool
    """

    jitter: float
    """ Jitter the position of the brush while painting

    :type: float
    """

    jitter_absolute: int
    """ Jitter the position of the brush in pixels while painting

    :type: int
    """

    jitter_unit: typing.Literal["VIEW", "BRUSH"]
    """ Jitter in screen space or relative to brush size

    :type: typing.Literal['VIEW','BRUSH']
    """

    mask_overlay_alpha: int
    """ 

    :type: int
    """

    mask_stencil_dimension: mathutils.Vector
    """ Dimensions of mask stencil in viewport

    :type: mathutils.Vector
    """

    mask_stencil_pos: mathutils.Vector
    """ Position of mask stencil in viewport

    :type: mathutils.Vector
    """

    mask_texture: Texture | None
    """ 

    :type: Texture | None
    """

    mask_texture_slot: BrushTextureSlot
    """ 

    :type: BrushTextureSlot
    """

    mask_tool: typing.Literal["DRAW", "SMOOTH"]
    """ 

    :type: typing.Literal['DRAW','SMOOTH']
    """

    multiplane_scrape_angle: float
    """ Angle between the planes of the crease

    :type: float
    """

    normal_radius_factor: float
    """ Ratio between the brush radius and the radius that is going to be used to sample the normal

    :type: float
    """

    normal_weight: float
    """ How much grab will pull vertices out of surface during a grab

    :type: float
    """

    paint_curve: PaintCurve | None
    """ Active paint curve

    :type: PaintCurve | None
    """

    plane_offset: float
    """ Adjust plane on which the brush acts towards or away from the object surface

    :type: float
    """

    plane_trim: float
    """ If a vertex is further away from offset plane than this, then it is not affected

    :type: float
    """

    pose_deform_type: typing.Literal[
        "ROTATE_TWIST", "SCALE_TRANSLATE", "SQUASH_STRETCH"
    ]
    """ Deformation type that is used in the brush

    :type: typing.Literal['ROTATE_TWIST','SCALE_TRANSLATE','SQUASH_STRETCH']
    """

    pose_ik_segments: int
    """ Number of segments of the inverse kinematics chain that will deform the mesh

    :type: int
    """

    pose_offset: float
    """ Offset of the pose origin in relation to the brush radius

    :type: float
    """

    pose_origin_type: typing.Literal["TOPOLOGY", "FACE_SETS", "FACE_SETS_FK"]
    """ Method to set the rotation origins for the segments of the brush

    :type: typing.Literal['TOPOLOGY','FACE_SETS','FACE_SETS_FK']
    """

    pose_smooth_iterations: int
    """ Smooth iterations applied after calculating the pose factor of each vertex

    :type: int
    """

    rake_factor: float
    """ How much grab will follow cursor rotation

    :type: float
    """

    rate: float
    """ Interval between paints for Airbrush

    :type: float
    """

    sculpt_capabilities: BrushCapabilitiesSculpt
    """ 

    :type: BrushCapabilitiesSculpt
    """

    sculpt_plane: typing.Literal["AREA", "VIEW", "X", "Y", "Z"]
    """ 

    :type: typing.Literal['AREA','VIEW','X','Y','Z']
    """

    sculpt_tool: bpy._typing.rna_enums.BrushSculptToolItems
    """ 

    :type: bpy._typing.rna_enums.BrushSculptToolItems
    """

    secondary_color: mathutils.Color
    """ 

    :type: mathutils.Color
    """

    sharp_threshold: float
    """ Threshold below which, no sharpening is done

    :type: float
    """

    show_multiplane_scrape_planes_preview: bool
    """ Preview the scrape planes in the cursor during the stroke

    :type: bool
    """

    size: int
    """ Radius of the brush in pixels

    :type: int
    """

    slide_deform_type: typing.Literal["DRAG", "PINCH", "EXPAND"]
    """ Deformation type that is used in the brush

    :type: typing.Literal['DRAG','PINCH','EXPAND']
    """

    smear_deform_type: typing.Literal["DRAG", "PINCH", "EXPAND"]
    """ Deformation type that is used in the brush

    :type: typing.Literal['DRAG','PINCH','EXPAND']
    """

    smooth_deform_type: typing.Literal["LAPLACIAN", "SURFACE"]
    """ Deformation type that is used in the brush

    :type: typing.Literal['LAPLACIAN','SURFACE']
    """

    smooth_stroke_factor: float
    """ Higher values give a smoother stroke

    :type: float
    """

    smooth_stroke_radius: int
    """ Minimum distance from last point before stroke continues

    :type: int
    """

    snake_hook_deform_type: typing.Literal["FALLOFF", "ELASTIC"]
    """ Deformation type that is used in the brush

    :type: typing.Literal['FALLOFF','ELASTIC']
    """

    spacing: int
    """ Spacing between brush daubs as a percentage of brush diameter

    :type: int
    """

    stencil_dimension: mathutils.Vector
    """ Dimensions of stencil in viewport

    :type: mathutils.Vector
    """

    stencil_pos: mathutils.Vector
    """ Position of stencil in viewport

    :type: mathutils.Vector
    """

    strength: float
    """ How powerful the effect of the brush is when applied

    :type: float
    """

    stroke_method: typing.Literal[
        "DOTS", "DRAG_DOT", "SPACE", "AIRBRUSH", "ANCHORED", "LINE", "CURVE"
    ]
    """ 

    :type: typing.Literal['DOTS','DRAG_DOT','SPACE','AIRBRUSH','ANCHORED','LINE','CURVE']
    """

    surface_smooth_current_vertex: float
    """ How much the position of each individual vertex influences the final result

    :type: float
    """

    surface_smooth_iterations: int
    """ Number of smoothing iterations per brush step

    :type: int
    """

    surface_smooth_shape_preservation: float
    """ How much of the original shape is preserved when smoothing

    :type: float
    """

    texture: Texture | None
    """ 

    :type: Texture | None
    """

    texture_overlay_alpha: int
    """ 

    :type: int
    """

    texture_sample_bias: float
    """ Value added to texture samples

    :type: float
    """

    texture_slot: BrushTextureSlot
    """ 

    :type: BrushTextureSlot
    """

    tilt_strength_factor: float
    """ How much the tilt of the pen will affect the brush

    :type: float
    """

    tip_roundness: float
    """ Roundness of the brush tip

    :type: float
    """

    tip_scale_x: float
    """ Scale of the brush tip in the X axis

    :type: float
    """

    topology_rake_factor: float
    """ Automatically align edges to the brush direction to generate cleaner topology and define sharp features. Best used on low-poly meshes as it has a performance impact

    :type: float
    """

    unprojected_radius: float
    """ Radius of brush in Blender units

    :type: float
    """

    use_accumulate: bool
    """ Accumulate stroke daubs on top of each other

    :type: bool
    """

    use_adaptive_space: bool
    """ Space daubs according to surface orientation instead of screen space

    :type: bool
    """

    use_airbrush: bool
    """ Keep applying paint effect while holding mouse (spray)

    :type: bool
    """

    use_alpha: bool
    """ When this is disabled, lock alpha while painting

    :type: bool
    """

    use_anchor: bool
    """ Keep the brush anchored to the initial location

    :type: bool
    """

    use_automasking_boundary_edges: bool
    """ Do not affect non manifold boundary edges

    :type: bool
    """

    use_automasking_boundary_face_sets: bool
    """ Do not affect vertices that belong to a Face Set boundary

    :type: bool
    """

    use_automasking_cavity: bool
    """ Do not affect vertices on peaks, based on the surface curvature

    :type: bool
    """

    use_automasking_cavity_inverted: bool
    """ Do not affect vertices within crevices, based on the surface curvature

    :type: bool
    """

    use_automasking_custom_cavity_curve: bool
    """ Use custom curve

    :type: bool
    """

    use_automasking_face_sets: bool
    """ Affect only vertices that share Face Sets with the active vertex

    :type: bool
    """

    use_automasking_start_normal: bool
    """ Affect only vertices with a similar normal to where the stroke starts

    :type: bool
    """

    use_automasking_topology: bool
    """ Affect only vertices connected to the active vertex under the brush

    :type: bool
    """

    use_automasking_view_normal: bool
    """ Affect only vertices with a normal that faces the viewer

    :type: bool
    """

    use_automasking_view_occlusion: bool
    """ Only affect vertices that are not occluded by other faces. (Slower performance)

    :type: bool
    """

    use_cloth_collision: bool
    """ Collide with objects during the simulation

    :type: bool
    """

    use_cloth_pin_simulation_boundary: bool
    """ Lock the position of the vertices in the simulation falloff area to avoid artifacts and create a softer transition with unaffected areas

    :type: bool
    """

    use_color_as_displacement: bool
    """ Handles each pixel color as individual vector for displacement. Works only with area plane mapping

    :type: bool
    """

    use_connected_only: bool
    """ Affect only topologically connected elements

    :type: bool
    """

    use_cursor_overlay: bool
    """ Show cursor in viewport

    :type: bool
    """

    use_cursor_overlay_override: bool
    """ Don't show overlay during a stroke

    :type: bool
    """

    use_curve: bool
    """ Define the stroke curve with a bezier curve. Dabs are separated according to spacing

    :type: bool
    """

    use_custom_icon: bool
    """ Set the brush icon from an image file

    :type: bool
    """

    use_density_pressure: bool
    """ Use pressure to modulate density

    :type: bool
    """

    use_edge_to_edge: bool
    """ Drag anchor brush from edge-to-edge

    :type: bool
    """

    use_flow_pressure: bool
    """ Use pressure to modulate flow

    :type: bool
    """

    use_frontface: bool
    """ Brush only affects vertices that face the viewer

    :type: bool
    """

    use_frontface_falloff: bool
    """ Blend brush influence by how much they face the front

    :type: bool
    """

    use_grab_active_vertex: bool
    """ Apply the maximum grab strength to the active vertex instead of the cursor location

    :type: bool
    """

    use_grab_silhouette: bool
    """ Grabs trying to automask the silhouette of the object

    :type: bool
    """

    use_hardness_pressure: bool
    """ Use pressure to modulate hardness

    :type: bool
    """

    use_inverse_smooth_pressure: bool
    """ Lighter pressure causes more smoothing to be applied

    :type: bool
    """

    use_line: bool
    """ Draw a line with dabs separated according to spacing

    :type: bool
    """

    use_locked_size: typing.Literal["VIEW", "SCENE"]
    """ Measure brush size relative to the view or the scene

    :type: typing.Literal['VIEW','SCENE']
    """

    use_multiplane_scrape_dynamic: bool
    """ The angle between the planes changes during the stroke to fit the surface under the cursor

    :type: bool
    """

    use_offset_pressure: bool
    """ Enable tablet pressure sensitivity for offset

    :type: bool
    """

    use_original_normal: bool
    """ When locked keep using normal of surface where stroke was initiated

    :type: bool
    """

    use_original_plane: bool
    """ When locked keep using the plane origin of surface where stroke was initiated

    :type: bool
    """

    use_paint_antialiasing: bool
    """ Smooths the edges of the strokes

    :type: bool
    """

    use_paint_grease_pencil: bool
    """ Use this brush in grease pencil drawing mode

    :type: bool
    """

    use_paint_image: bool
    """ Use this brush in texture paint mode

    :type: bool
    """

    use_paint_sculpt: bool
    """ Use this brush in sculpt mode

    :type: bool
    """

    use_paint_sculpt_curves: bool
    """ Use this brush in sculpt curves mode

    :type: bool
    """

    use_paint_uv_sculpt: bool
    """ Use this brush in UV sculpt mode

    :type: bool
    """

    use_paint_vertex: bool
    """ Use this brush in vertex paint mode

    :type: bool
    """

    use_paint_weight: bool
    """ Use this brush in weight paint mode

    :type: bool
    """

    use_persistent: bool
    """ Sculpt on a persistent layer of the mesh

    :type: bool
    """

    use_plane_trim: bool
    """ Limit the distance from the offset plane that a vertex can be affected

    :type: bool
    """

    use_pose_ik_anchored: bool
    """ Keep the position of the last segment in the IK chain fixed

    :type: bool
    """

    use_pose_lock_rotation: bool
    """ Do not rotate the segment when using the scale deform mode

    :type: bool
    """

    use_pressure_area_radius: bool
    """ Enable tablet pressure sensitivity for area radius

    :type: bool
    """

    use_pressure_jitter: bool
    """ Enable tablet pressure sensitivity for jitter

    :type: bool
    """

    use_pressure_masking: typing.Literal["NONE", "RAMP", "CUTOFF"]
    """ Pen pressure makes texture influence smaller

    :type: typing.Literal['NONE','RAMP','CUTOFF']
    """

    use_pressure_size: bool
    """ Enable tablet pressure sensitivity for size

    :type: bool
    """

    use_pressure_spacing: bool
    """ Enable tablet pressure sensitivity for spacing

    :type: bool
    """

    use_pressure_strength: bool
    """ Enable tablet pressure sensitivity for strength

    :type: bool
    """

    use_primary_overlay: bool
    """ Show texture in viewport

    :type: bool
    """

    use_primary_overlay_override: bool
    """ Don't show overlay during a stroke

    :type: bool
    """

    use_restore_mesh: bool
    """ Allow a single dot to be carefully positioned

    :type: bool
    """

    use_scene_spacing: typing.Literal["VIEW", "SCENE"]
    """ Calculate the brush spacing using view or scene distance

    :type: typing.Literal['VIEW','SCENE']
    """

    use_secondary_overlay: bool
    """ Show texture in viewport

    :type: bool
    """

    use_secondary_overlay_override: bool
    """ Don't show overlay during a stroke

    :type: bool
    """

    use_smooth_stroke: bool
    """ Brush lags behind mouse and follows a smoother path

    :type: bool
    """

    use_space: bool
    """ Limit brush application to the distance specified by spacing

    :type: bool
    """

    use_space_attenuation: bool
    """ Automatically adjust strength to give consistent results for different spacings

    :type: bool
    """

    use_vertex_grease_pencil: bool
    """ Use this brush in grease pencil vertex color mode

    :type: bool
    """

    use_wet_mix_pressure: bool
    """ Use pressure to modulate wet mix

    :type: bool
    """

    use_wet_persistence_pressure: bool
    """ Use pressure to modulate wet persistence

    :type: bool
    """

    uv_sculpt_tool: bpy._typing.rna_enums.BrushUvSculptToolItems
    """ 

    :type: bpy._typing.rna_enums.BrushUvSculptToolItems
    """

    vertex_paint_capabilities: BrushCapabilitiesVertexPaint
    """ 

    :type: BrushCapabilitiesVertexPaint
    """

    vertex_tool: bpy._typing.rna_enums.BrushVertexToolItems
    """ 

    :type: bpy._typing.rna_enums.BrushVertexToolItems
    """

    weight: float
    """ Vertex weight when brush is applied

    :type: float
    """

    weight_paint_capabilities: BrushCapabilitiesWeightPaint
    """ 

    :type: BrushCapabilitiesWeightPaint
    """

    weight_tool: bpy._typing.rna_enums.BrushWeightToolItems
    """ 

    :type: bpy._typing.rna_enums.BrushWeightToolItems
    """

    wet_mix: float
    """ Amount of paint that is picked from the surface into the brush color

    :type: float
    """

    wet_paint_radius_factor: float
    """ Ratio between the brush radius and the radius that is going to be used to sample the color to blend in wet paint

    :type: float
    """

    wet_persistence: float
    """ Amount of wet paint that stays in the brush after applying paint to the surface

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BrushCapabilities(bpy_struct):
    """Read-only indications of supported operations"""

    has_overlay: bool
    """ 

    :type: bool
    """

    has_random_texture_angle: bool
    """ 

    :type: bool
    """

    has_smooth_stroke: bool
    """ 

    :type: bool
    """

    has_spacing: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BrushCapabilitiesImagePaint(bpy_struct):
    """Read-only indications of supported operations"""

    has_accumulate: bool
    """ 

    :type: bool
    """

    has_color: bool
    """ 

    :type: bool
    """

    has_radius: bool
    """ 

    :type: bool
    """

    has_space_attenuation: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BrushCapabilitiesSculpt(bpy_struct):
    """Read-only indications of which brush operations are supported by the current sculpt tool"""

    has_accumulate: bool
    """ 

    :type: bool
    """

    has_auto_smooth: bool
    """ 

    :type: bool
    """

    has_color: bool
    """ 

    :type: bool
    """

    has_direction: bool
    """ 

    :type: bool
    """

    has_gravity: bool
    """ 

    :type: bool
    """

    has_height: bool
    """ 

    :type: bool
    """

    has_jitter: bool
    """ 

    :type: bool
    """

    has_normal_weight: bool
    """ 

    :type: bool
    """

    has_persistence: bool
    """ 

    :type: bool
    """

    has_pinch_factor: bool
    """ 

    :type: bool
    """

    has_plane_offset: bool
    """ 

    :type: bool
    """

    has_rake_factor: bool
    """ 

    :type: bool
    """

    has_random_texture_angle: bool
    """ 

    :type: bool
    """

    has_sculpt_plane: bool
    """ 

    :type: bool
    """

    has_secondary_color: bool
    """ 

    :type: bool
    """

    has_smooth_stroke: bool
    """ 

    :type: bool
    """

    has_space_attenuation: bool
    """ 

    :type: bool
    """

    has_strength_pressure: bool
    """ 

    :type: bool
    """

    has_tilt: bool
    """ 

    :type: bool
    """

    has_topology_rake: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BrushCapabilitiesVertexPaint(bpy_struct):
    """Read-only indications of supported operations"""

    has_color: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BrushCapabilitiesWeightPaint(bpy_struct):
    """Read-only indications of supported operations"""

    has_weight: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BrushCurvesSculptSettings(bpy_struct):
    add_amount: int
    """ Number of curves added by the Add brush

    :type: int
    """

    curve_length: float
    """ Length of newly added curves when it is not interpolated from other curves

    :type: float
    """

    curve_parameter_falloff: CurveMapping
    """ Falloff that is applied from the tip to the root of each curve

    :type: CurveMapping
    """

    density_add_attempts: int
    """ How many times the Density brush tries to add a new curve

    :type: int
    """

    density_mode: typing.Literal["AUTO", "ADD", "REMOVE"]
    """ Determines whether the brush adds or removes curves

    :type: typing.Literal['AUTO','ADD','REMOVE']
    """

    interpolate_length: bool
    """ Use length of the curves in close proximity

    :type: bool
    """

    interpolate_point_count: bool
    """ Use the number of points from the curves in close proximity

    :type: bool
    """

    interpolate_shape: bool
    """ Use shape of the curves in close proximity

    :type: bool
    """

    minimum_distance: float
    """ Goal distance between curve roots for the Density brush

    :type: float
    """

    minimum_length: float
    """ Avoid shrinking curves shorter than this length

    :type: float
    """

    points_per_curve: int
    """ Number of control points in a newly added curve

    :type: int
    """

    scale_uniform: bool
    """ Grow or shrink curves by changing their size uniformly instead of using trimming or extrapolation

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BrushGpencilSettings(bpy_struct):
    """Settings for grease pencil brush"""

    active_smooth_factor: float | None
    """ Amount of smoothing while drawing

    :type: float | None
    """

    angle: float
    """ Direction of the stroke at which brush gives maximal thickness (0° for horizontal)

    :type: float
    """

    angle_factor: float
    """ Reduce brush thickness by this factor when stroke is perpendicular to 'Angle' direction

    :type: float
    """

    aspect: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    brush_draw_mode: typing.Literal["ACTIVE", "MATERIAL", "VERTEXCOLOR"]
    """ Preselected mode when using this brush

    :type: typing.Literal['ACTIVE','MATERIAL','VERTEXCOLOR']
    """

    caps_type: typing.Literal["ROUND", "FLAT"]
    """ The shape of the start and end of the stroke

    :type: typing.Literal['ROUND','FLAT']
    """

    curve_jitter: CurveMapping
    """ Curve used for the jitter effect

    :type: CurveMapping
    """

    curve_random_hue: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_pressure: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_saturation: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_strength: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_uv: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_random_value: CurveMapping
    """ Curve used for modulating effect

    :type: CurveMapping
    """

    curve_sensitivity: CurveMapping
    """ Curve used for the sensitivity

    :type: CurveMapping
    """

    curve_strength: CurveMapping
    """ Curve used for the strength

    :type: CurveMapping
    """

    dilate: int
    """ Number of pixels to expand or contract fill area

    :type: int
    """

    direction: typing.Literal["ADD", "SUBTRACT"]
    """ 

    :type: typing.Literal['ADD','SUBTRACT']
    """

    eraser_mode: typing.Literal["SOFT", "HARD", "STROKE"]
    """ Eraser Mode

    :type: typing.Literal['SOFT','HARD','STROKE']
    """

    eraser_strength_factor: float
    """ Amount of erasing for strength

    :type: float
    """

    eraser_thickness_factor: float
    """ Amount of erasing for thickness

    :type: float
    """

    extend_stroke_factor: float
    """ Strokes end extension for closing gaps, use zero to disable

    :type: float
    """

    fill_direction: typing.Literal["NORMAL", "INVERT"]
    """ Direction of the fill

    :type: typing.Literal['NORMAL','INVERT']
    """

    fill_draw_mode: typing.Literal["BOTH", "STROKE", "CONTROL"]
    """ Mode to draw boundary limits

    :type: typing.Literal['BOTH','STROKE','CONTROL']
    """

    fill_extend_mode: typing.Literal["EXTEND", "RADIUS"]
    """ Types of stroke extensions used for closing gaps

    :type: typing.Literal['EXTEND','RADIUS']
    """

    fill_factor: float
    """ Factor for fill boundary accuracy, higher values are more accurate but slower

    :type: float
    """

    fill_layer_mode: typing.Literal[
        "VISIBLE", "ACTIVE", "ABOVE", "BELOW", "ALL_ABOVE", "ALL_BELOW"
    ]
    """ Layers used as boundaries

    :type: typing.Literal['VISIBLE','ACTIVE','ABOVE','BELOW','ALL_ABOVE','ALL_BELOW']
    """

    fill_simplify_level: int
    """ Number of simplify steps (large values reduce fill accuracy)

    :type: int
    """

    fill_threshold: float
    """ Threshold to consider color transparent for filling

    :type: float
    """

    gpencil_paint_icon: typing.Literal[
        "PENCIL",
        "PEN",
        "INK",
        "INKNOISE",
        "BLOCK",
        "MARKER",
        "AIRBRUSH",
        "CHISEL",
        "FILL",
        "SOFT",
        "HARD",
        "STROKE",
    ]
    """ 

    :type: typing.Literal['PENCIL','PEN','INK','INKNOISE','BLOCK','MARKER','AIRBRUSH','CHISEL','FILL','SOFT','HARD','STROKE']
    """

    gpencil_sculpt_icon: typing.Literal[
        "SMOOTH",
        "THICKNESS",
        "STRENGTH",
        "RANDOMIZE",
        "GRAB",
        "PUSH",
        "TWIST",
        "PINCH",
        "CLONE",
    ]
    """ 

    :type: typing.Literal['SMOOTH','THICKNESS','STRENGTH','RANDOMIZE','GRAB','PUSH','TWIST','PINCH','CLONE']
    """

    gpencil_vertex_icon: typing.Literal["DRAW", "BLUR", "AVERAGE", "SMEAR", "REPLACE"]
    """ 

    :type: typing.Literal['DRAW','BLUR','AVERAGE','SMEAR','REPLACE']
    """

    gpencil_weight_icon: typing.Literal["DRAW", "BLUR", "AVERAGE", "SMEAR"]
    """ 

    :type: typing.Literal['DRAW','BLUR','AVERAGE','SMEAR']
    """

    hardness: float
    """ Gradient from the center of Dot and Box strokes (set to 1 for a solid stroke)

    :type: float
    """

    input_samples: int
    """ Generate intermediate points for very fast mouse movements. Set to 0 to disable

    :type: int
    """

    material: Material | None
    """ Material used for strokes drawn using this brush

    :type: Material | None
    """

    material_alt: Material | None
    """ Material used for secondary uses for this brush

    :type: Material | None
    """

    outline_thickness_factor: float
    """ Thickness of the outline stroke relative to current brush thickness

    :type: float
    """

    pen_jitter: float
    """ Jitter factor for new strokes

    :type: float
    """

    pen_smooth_factor: float
    """ Amount of smoothing to apply after finish newly created strokes, to reduce jitter/noise

    :type: float
    """

    pen_smooth_steps: int
    """ Number of times to smooth newly created strokes

    :type: int
    """

    pen_strength: float
    """ Color strength for new strokes (affect alpha factor of color)

    :type: float
    """

    pen_subdivision_steps: int
    """ Number of times to subdivide newly created strokes, for less jagged strokes

    :type: int
    """

    pin_draw_mode: bool
    """ Pin the mode to the brush

    :type: bool
    """

    random_hue_factor: float
    """ Random factor to modify original hue

    :type: float
    """

    random_pressure: float
    """ Randomness factor for pressure in new strokes

    :type: float
    """

    random_saturation_factor: float
    """ Random factor to modify original saturation

    :type: float
    """

    random_strength: float
    """ Randomness factor strength in new strokes

    :type: float
    """

    random_value_factor: float
    """ Random factor to modify original value

    :type: float
    """

    show_fill: bool
    """ Show transparent lines to use as boundary for filling

    :type: bool
    """

    show_fill_boundary: bool
    """ Show help lines for filling to see boundaries

    :type: bool
    """

    show_fill_extend: bool
    """ Show help lines for stroke extension

    :type: bool
    """

    show_lasso: bool
    """ Do not display fill color while drawing the stroke

    :type: bool
    """

    simplify_factor: float
    """ Factor of Simplify using adaptive algorithm

    :type: float
    """

    use_active_layer_only: bool
    """ Only edit the active layer of the object

    :type: bool
    """

    use_collide_strokes: bool
    """ Check if extend lines collide with strokes

    :type: bool
    """

    use_default_eraser: bool
    """ Use this brush when enable eraser with fast switch key

    :type: bool
    """

    use_edit_position: bool
    """ The brush affects the position of the point

    :type: bool
    """

    use_edit_strength: bool
    """ The brush affects the color strength of the point

    :type: bool
    """

    use_edit_thickness: bool
    """ The brush affects the thickness of the point

    :type: bool
    """

    use_edit_uv: bool
    """ The brush affects the UV rotation of the point

    :type: bool
    """

    use_fill_limit: bool
    """ Fill only visible areas in viewport

    :type: bool
    """

    use_jitter_pressure: bool
    """ Use tablet pressure for jitter

    :type: bool
    """

    use_keep_caps_eraser: bool
    """ Keep the caps as they are and don't flatten them when erasing

    :type: bool
    """

    use_material_pin: bool
    """ Keep material assigned to brush

    :type: bool
    """

    use_occlude_eraser: bool
    """ Erase only strokes visible and not occluded

    :type: bool
    """

    use_pressure: bool
    """ Use tablet pressure

    :type: bool
    """

    use_random_press_hue: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_radius: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_sat: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_strength: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_uv: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_random_press_val: bool
    """ Use pressure to modulate randomness

    :type: bool
    """

    use_settings_outline: bool
    """ Convert stroke to perimeter

    :type: bool
    """

    use_settings_postprocess: bool
    """ Additional post processing options for new strokes

    :type: bool
    """

    use_settings_random: bool
    """ Random brush settings

    :type: bool
    """

    use_settings_stabilizer: bool
    """ Draw lines with a delay to allow smooth strokes. Press Shift key to override while drawing

    :type: bool
    """

    use_strength_pressure: bool
    """ Use tablet pressure for color strength

    :type: bool
    """

    use_stroke_random_hue: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_radius: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_sat: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_strength: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_uv: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_stroke_random_val: bool
    """ Use randomness at stroke level

    :type: bool
    """

    use_trim: bool
    """ Trim intersecting stroke ends

    :type: bool
    """

    uv_random: float
    """ Random factor for auto-generated UV rotation

    :type: float
    """

    vertex_color_factor: float
    """ Factor used to mix vertex color to get final color

    :type: float
    """

    vertex_mode: typing.Literal["STROKE", "FILL", "BOTH"]
    """ Defines how vertex color affect to the strokes

    :type: typing.Literal['STROKE','FILL','BOTH']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BrushTextureSlot(TextureSlot, bpy_struct):
    """Texture slot for textures in a Brush data-block"""

    angle: float
    """ Brush texture rotation

    :type: float
    """

    has_random_texture_angle: bool
    """ 

    :type: bool
    """

    has_texture_angle: bool
    """ 

    :type: bool
    """

    has_texture_angle_source: bool
    """ 

    :type: bool
    """

    map_mode: typing.Literal[
        "VIEW_PLANE", "AREA_PLANE", "TILED", "3D", "RANDOM", "STENCIL"
    ]
    """ 

    :type: typing.Literal['VIEW_PLANE','AREA_PLANE','TILED','3D','RANDOM','STENCIL']
    """

    mask_map_mode: typing.Literal["VIEW_PLANE", "TILED", "RANDOM", "STENCIL"]
    """ 

    :type: typing.Literal['VIEW_PLANE','TILED','RANDOM','STENCIL']
    """

    random_angle: float
    """ Brush texture random angle

    :type: float
    """

    use_rake: bool
    """ 

    :type: bool
    """

    use_random: bool
    """ 

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BuildGpencilModifier(GpencilModifier, bpy_struct):
    """Animate strokes appearing and disappearing"""

    concurrent_time_alignment: typing.Literal["START", "END"]
    """ How should strokes start to appear/disappear

    :type: typing.Literal['START','END']
    """

    fade_factor: float
    """ Defines how much of the stroke is fading in/out

    :type: float
    """

    fade_opacity_strength: float
    """ How much strength fading applies on top of stroke opacity

    :type: float
    """

    fade_thickness_strength: float
    """ How much strength fading applies on top of stroke thickness

    :type: float
    """

    frame_end: float
    """ End Frame (when Restrict Frame Range is enabled)

    :type: float
    """

    frame_start: float
    """ Start Frame (when Restrict Frame Range is enabled)

    :type: float
    """

    invert_layer_pass: bool
    """ Inverse filter

    :type: bool
    """

    invert_layers: bool
    """ Inverse filter

    :type: bool
    """

    layer: str
    """ Layer name

    :type: str
    """

    layer_pass: int
    """ Layer pass index

    :type: int
    """

    length: float
    """ Maximum number of frames that the build effect can run for (unless another GP keyframe occurs before this time has elapsed)

    :type: float
    """

    mode: typing.Literal["SEQUENTIAL", "CONCURRENT", "ADDITIVE"]
    """ How strokes are being built

    :type: typing.Literal['SEQUENTIAL','CONCURRENT','ADDITIVE']
    """

    object: Object | None
    """ Object used as build starting position

    :type: Object | None
    """

    percentage_factor: float
    """ Defines how much of the stroke is visible

    :type: float
    """

    speed_factor: float
    """ Multiply recorded drawing speed by a factor

    :type: float
    """

    speed_maxgap: float
    """ The maximum gap between strokes in seconds

    :type: float
    """

    start_delay: float
    """ Number of frames after each GP keyframe before the modifier has any effect

    :type: float
    """

    target_vertex_group: str
    """ Output Vertex group

    :type: str
    """

    time_mode: typing.Literal["DRAWSPEED", "FRAMES", "PERCENTAGE"]
    """ Use drawing speed, a number of frames, or a manual factor to build strokes

    :type: typing.Literal['DRAWSPEED','FRAMES','PERCENTAGE']
    """

    transition: typing.Literal["GROW", "SHRINK", "FADE"]
    """ How are strokes animated (i.e. are they appearing or disappearing)

    :type: typing.Literal['GROW','SHRINK','FADE']
    """

    use_fading: bool
    """ Fade out strokes instead of directly cutting off

    :type: bool
    """

    use_percentage: bool
    """ Use a percentage factor to determine the visible points

    :type: bool
    """

    use_restrict_frame_range: bool
    """ Only modify strokes during the specified frame range

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class BuildModifier(Modifier, bpy_struct):
    """Build effect modifier"""

    frame_duration: float
    """ Total time the build effect requires

    :type: float
    """

    frame_start: float
    """ Start frame of the effect

    :type: float
    """

    seed: int
    """ Seed for random if used

    :type: int
    """

    use_random_order: bool
    """ Randomize the faces or edges during build

    :type: bool
    """

    use_reverse: bool
    """ Deconstruct the mesh instead of building it

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ByteColorAttribute(Attribute, bpy_struct):
    """Geometry attribute that stores RGBA colors as positive integer values using 8-bits per channel"""

    data: bpy_prop_collection[ByteColorAttributeValue]
    """ 

    :type: bpy_prop_collection[ByteColorAttributeValue]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ByteColorAttributeValue(bpy_struct):
    """Color value in geometry attribute"""

    color: bpy_prop_array[float]
    """ RGBA color in scene linear color space

    :type: bpy_prop_array[float]
    """

    color_srgb: bpy_prop_array[float]
    """ RGBA color in sRGB color space

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ByteIntAttribute(Attribute, bpy_struct):
    """Geometry attribute that stores 8-bit integers"""

    data: bpy_prop_collection[ByteIntAttributeValue]
    """ 

    :type: bpy_prop_collection[ByteIntAttributeValue]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ByteIntAttributeValue(bpy_struct):
    """8-bit value in geometry attribute"""

    value: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CLIP_UL_tracking_objects(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        _data,
        item,
        _icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param _data:
        :param item:
        :param _icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CURVES_UL_attributes(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        _data,
        attribute,
        _icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param _data:
        :param attribute:
        :param _icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    def filter_items(self, _context, data, property):
        """

        :param _context:
        :param data:
        :param property:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CacheFile(ID, bpy_struct):
    active_index: int | None
    """ 

    :type: int | None
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    filepath: str
    """ Path to external displacements file

    :type: str
    """

    forward_axis: bpy._typing.rna_enums.ObjectAxisItems
    """ 

    :type: bpy._typing.rna_enums.ObjectAxisItems
    """

    frame: float
    """ The time to use for looking up the data in the cache file, or to determine which file to use in a file sequence

    :type: float
    """

    frame_offset: float
    """ Subtracted from the current frame to use for looking up the data in the cache file, or to determine which file to use in a file sequence

    :type: float
    """

    is_sequence: bool
    """ Whether the cache is separated in a series of files

    :type: bool
    """

    layers: CacheFileLayers
    """ Layers of the cache

    :type: CacheFileLayers
    """

    object_paths: CacheObjectPaths
    """ Paths of the objects inside the Alembic archive

    :type: CacheObjectPaths
    """

    override_frame: bool
    """ Whether to use a custom frame for looking up data in the cache file, instead of using the current scene frame

    :type: bool
    """

    prefetch_cache_size: int
    """ Memory usage limit in megabytes for the Cycles Procedural cache, if the data does not fit within the limit, rendering is aborted

    :type: int
    """

    scale: float
    """ Value by which to enlarge or shrink the object with respect to the world's origin (only applicable through a Transform Cache constraint)

    :type: float
    """

    up_axis: bpy._typing.rna_enums.ObjectAxisItems
    """ 

    :type: bpy._typing.rna_enums.ObjectAxisItems
    """

    use_prefetch: bool
    """ When enabled, the Cycles Procedural will preload animation data for faster updates

    :type: bool
    """

    use_render_procedural: bool
    """ Display boxes in the viewport as placeholders for the objects, Cycles will use a procedural to load the objects during viewport rendering in experimental mode, other render engines will also receive a placeholder and should take care of loading the Alembic data themselves if possible

    :type: bool
    """

    velocity_name: str
    """ Name of the Alembic attribute used for generating motion blur data

    :type: str
    """

    velocity_unit: bpy._typing.rna_enums.VelocityUnitItems
    """ Define how the velocity vectors are interpreted with regard to time, 'frame' means the delta time is 1 frame, 'second' means the delta time is 1 / FPS

    :type: bpy._typing.rna_enums.VelocityUnitItems
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CacheFileLayer(bpy_struct):
    """Layer of the cache, used to load or override data from the first the first layer"""

    filepath: str
    """ Path to the archive

    :type: str
    """

    hide_layer: bool
    """ Do not load data from this layer

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CacheObjectPath(bpy_struct):
    """Path of an object inside of an Alembic archive"""

    path: str
    """ Object path

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Camera(ID, bpy_struct):
    """Camera data-block for storing camera settings"""

    angle: float
    """ Camera lens field of view

    :type: float
    """

    angle_x: float
    """ Camera lens horizontal field of view

    :type: float
    """

    angle_y: float
    """ Camera lens vertical field of view

    :type: float
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    background_images: CameraBackgroundImages
    """ List of background images

    :type: CameraBackgroundImages
    """

    clip_end: float
    """ Camera far clipping distance

    :type: float
    """

    clip_start: float
    """ Camera near clipping distance

    :type: float
    """

    display_size: float
    """ Apparent size of the Camera object in the 3D View

    :type: float
    """

    dof: CameraDOFSettings
    """ 

    :type: CameraDOFSettings
    """

    fisheye_fov: float
    """ Field of view for the fisheye lens

    :type: float
    """

    fisheye_lens: float
    """ Lens focal length (mm)

    :type: float
    """

    fisheye_polynomial_k0: float
    """ Coefficient K0 of the lens polynomial

    :type: float
    """

    fisheye_polynomial_k1: float
    """ Coefficient K1 of the lens polynomial

    :type: float
    """

    fisheye_polynomial_k2: float
    """ Coefficient K2 of the lens polynomial

    :type: float
    """

    fisheye_polynomial_k3: float
    """ Coefficient K3 of the lens polynomial

    :type: float
    """

    fisheye_polynomial_k4: float
    """ Coefficient K4 of the lens polynomial

    :type: float
    """

    latitude_max: float
    """ Maximum latitude (vertical angle) for the equirectangular lens

    :type: float
    """

    latitude_min: float
    """ Minimum latitude (vertical angle) for the equirectangular lens

    :type: float
    """

    lens: float
    """ Perspective Camera focal length value in millimeters

    :type: float
    """

    lens_unit: typing.Literal["MILLIMETERS", "FOV"]
    """ Unit to edit lens in for the user interface

    :type: typing.Literal['MILLIMETERS','FOV']
    """

    longitude_max: float
    """ Maximum longitude (horizontal angle) for the equirectangular lens

    :type: float
    """

    longitude_min: float
    """ Minimum longitude (horizontal angle) for the equirectangular lens

    :type: float
    """

    ortho_scale: float
    """ Orthographic Camera scale (similar to zoom)

    :type: float
    """

    panorama_type: typing.Literal[
        "EQUIRECTANGULAR",
        "EQUIANGULAR_CUBEMAP_FACE",
        "MIRRORBALL",
        "FISHEYE_EQUIDISTANT",
        "FISHEYE_EQUISOLID",
        "FISHEYE_LENS_POLYNOMIAL",
    ]
    """ Distortion to use for the calculation

    :type: typing.Literal['EQUIRECTANGULAR','EQUIANGULAR_CUBEMAP_FACE','MIRRORBALL','FISHEYE_EQUIDISTANT','FISHEYE_EQUISOLID','FISHEYE_LENS_POLYNOMIAL']
    """

    passepartout_alpha: float
    """ Opacity (alpha) of the darkened overlay in Camera view

    :type: float
    """

    sensor_fit: typing.Literal["AUTO", "HORIZONTAL", "VERTICAL"]
    """ Method to fit image and field of view angle inside the sensor

    :type: typing.Literal['AUTO','HORIZONTAL','VERTICAL']
    """

    sensor_height: float
    """ Vertical size of the image sensor area in millimeters

    :type: float
    """

    sensor_width: float
    """ Horizontal size of the image sensor area in millimeters

    :type: float
    """

    shift_x: float
    """ Camera horizontal shift

    :type: float
    """

    shift_y: float
    """ Camera vertical shift

    :type: float
    """

    show_background_images: bool
    """ Display reference images behind objects in the 3D View

    :type: bool
    """

    show_composition_center: bool
    """ Display center composition guide inside the camera view

    :type: bool
    """

    show_composition_center_diagonal: bool
    """ Display diagonal center composition guide inside the camera view

    :type: bool
    """

    show_composition_golden: bool
    """ Display golden ratio composition guide inside the camera view

    :type: bool
    """

    show_composition_golden_tria_a: bool
    """ Display golden triangle A composition guide inside the camera view

    :type: bool
    """

    show_composition_golden_tria_b: bool
    """ Display golden triangle B composition guide inside the camera view

    :type: bool
    """

    show_composition_harmony_tri_a: bool
    """ Display harmony A composition guide inside the camera view

    :type: bool
    """

    show_composition_harmony_tri_b: bool
    """ Display harmony B composition guide inside the camera view

    :type: bool
    """

    show_composition_thirds: bool
    """ Display rule of thirds composition guide inside the camera view

    :type: bool
    """

    show_limits: bool
    """ Display the clipping range and focus point on the camera

    :type: bool
    """

    show_mist: bool
    """ Display a line from the Camera to indicate the mist area

    :type: bool
    """

    show_name: bool
    """ Show the active Camera's name in Camera view

    :type: bool
    """

    show_passepartout: bool
    """ Show a darkened overlay outside the image area in Camera view

    :type: bool
    """

    show_safe_areas: bool
    """ Show TV title safe and action safe areas in Camera view

    :type: bool
    """

    show_safe_center: bool
    """ Show safe areas to fit content in a different aspect ratio

    :type: bool
    """

    show_sensor: bool
    """ Show sensor size (film gate) in Camera view

    :type: bool
    """

    stereo: CameraStereoData
    """ 

    :type: CameraStereoData
    """

    type: typing.Literal["PERSP", "ORTHO", "PANO"]
    """ Camera types

    :type: typing.Literal['PERSP','ORTHO','PANO']
    """

    def view_frame(self, scene: Scene | None = None):
        """Return 4 points for the cameras frame (before object transformation)

                :param scene: Scene to use for aspect calculation, when omitted 1:1 aspect is used
                :type scene: Scene | None
                :return: result_1, Result, `mathutils.Vector` of 3 items in [-inf, inf]

        result_2, Result, `mathutils.Vector` of 3 items in [-inf, inf]

        result_3, Result, `mathutils.Vector` of 3 items in [-inf, inf]

        result_4, Result, `mathutils.Vector` of 3 items in [-inf, inf]
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CameraBackgroundImage(bpy_struct):
    """Image and settings for display in the 3D View background"""

    alpha: float
    """ Image opacity to blend the image against the background color

    :type: float
    """

    clip: MovieClip | None
    """ Movie clip displayed and edited in this space

    :type: MovieClip | None
    """

    clip_user: MovieClipUser
    """ Parameters defining which frame of the movie clip is displayed

    :type: MovieClipUser
    """

    display_depth: typing.Literal["BACK", "FRONT"]
    """ Display under or over everything

    :type: typing.Literal['BACK','FRONT']
    """

    frame_method: typing.Literal["STRETCH", "FIT", "CROP"]
    """ How the image fits in the camera frame

    :type: typing.Literal['STRETCH','FIT','CROP']
    """

    image: Image | None
    """ Image displayed and edited in this space

    :type: Image | None
    """

    image_user: ImageUser
    """ Parameters defining which layer, pass and frame of the image is displayed

    :type: ImageUser
    """

    is_override_data: bool
    """ In a local override camera, whether this background image comes from the linked reference camera, or is local to the override

    :type: bool
    """

    offset: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    rotation: float
    """ Rotation for the background image (ortho view only)

    :type: float
    """

    scale: float
    """ Scale the background image

    :type: float
    """

    show_background_image: bool
    """ Show this image as background

    :type: bool
    """

    show_expanded: bool
    """ Show the details in the user interface

    :type: bool
    """

    show_on_foreground: bool
    """ Show this image in front of objects in viewport

    :type: bool
    """

    source: typing.Literal["IMAGE", "MOVIE_CLIP"]
    """ Data source used for background

    :type: typing.Literal['IMAGE','MOVIE_CLIP']
    """

    use_camera_clip: bool
    """ Use movie clip from active scene camera

    :type: bool
    """

    use_flip_x: bool
    """ Flip the background image horizontally

    :type: bool
    """

    use_flip_y: bool
    """ Flip the background image vertically

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CameraDOFSettings(bpy_struct):
    """Depth of Field settings"""

    aperture_blades: int
    """ Number of blades in aperture for polygonal bokeh (at least 3)

    :type: int
    """

    aperture_fstop: float
    """ F-Stop ratio (lower numbers give more defocus, higher numbers give a sharper image)

    :type: float
    """

    aperture_ratio: float
    """ Distortion to simulate anamorphic lens bokeh

    :type: float
    """

    aperture_rotation: float
    """ Rotation of blades in aperture

    :type: float
    """

    focus_distance: float
    """ Distance to the focus point for depth of field

    :type: float
    """

    focus_object: Object | None
    """ Use this object to define the depth of field focal point

    :type: Object | None
    """

    focus_subtarget: str
    """ Use this armature bone to define the depth of field focal point

    :type: str
    """

    use_dof: bool
    """ Use Depth of Field

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CameraSolverConstraint(Constraint, bpy_struct):
    """Lock motion to the reconstructed camera movement"""

    clip: MovieClip | None
    """ Movie Clip to get tracking data from

    :type: MovieClip | None
    """

    use_active_clip: bool
    """ Use active clip defined in scene

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CameraStereoData(bpy_struct):
    """Stereoscopy settings for a Camera data-block"""

    convergence_distance: float
    """ The converge point for the stereo cameras (often the distance between a projector and the projection screen)

    :type: float
    """

    convergence_mode: typing.Literal["OFFAXIS", "PARALLEL", "TOE"]
    """ 

    :type: typing.Literal['OFFAXIS','PARALLEL','TOE']
    """

    interocular_distance: float
    """ Set the distance between the eyes - the stereo plane distance / 30 should be fine

    :type: float
    """

    pivot: typing.Literal["LEFT", "RIGHT", "CENTER"]
    """ 

    :type: typing.Literal['LEFT','RIGHT','CENTER']
    """

    pole_merge_angle_from: float
    """ Angle at which interocular distance starts to fade to 0

    :type: float
    """

    pole_merge_angle_to: float
    """ Angle at which interocular distance is 0

    :type: float
    """

    use_pole_merge: bool
    """ Fade interocular distance to 0 after the given cutoff angle

    :type: bool
    """

    use_spherical_stereo: bool
    """ Render every pixel rotating the camera around the middle of the interocular distance

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CastModifier(Modifier, bpy_struct):
    """Modifier to cast to other shapes"""

    cast_type: typing.Literal["SPHERE", "CYLINDER", "CUBOID"]
    """ Target object shape

    :type: typing.Literal['SPHERE','CYLINDER','CUBOID']
    """

    factor: float
    """ 

    :type: float
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    object: Object | None
    """ Control object: if available, its location determines the center of the effect

    :type: Object | None
    """

    radius: float
    """ Only deform vertices within this distance from the center of the effect (leave as 0 for infinite.)

    :type: float
    """

    size: float
    """ Size of projection shape (leave as 0 for auto)

    :type: float
    """

    use_radius_as_size: bool
    """ Use radius as size of projection shape (0 = auto)

    :type: bool
    """

    use_transform: bool
    """ Use object transform to control projection shape

    :type: bool
    """

    use_x: bool
    """ 

    :type: bool
    """

    use_y: bool
    """ 

    :type: bool
    """

    use_z: bool
    """ 

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ChildOfConstraint(Constraint, bpy_struct):
    """Create constraint-based parent-child relationship"""

    inverse_matrix: mathutils.Matrix
    """ Transformation matrix to apply before

    :type: mathutils.Matrix
    """

    set_inverse_pending: bool
    """ Set to true to request recalculation of the inverse matrix

    :type: bool
    """

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object | None
    """ Target object

    :type: Object | None
    """

    use_location_x: bool
    """ Use X Location of Parent

    :type: bool
    """

    use_location_y: bool
    """ Use Y Location of Parent

    :type: bool
    """

    use_location_z: bool
    """ Use Z Location of Parent

    :type: bool
    """

    use_rotation_x: bool
    """ Use X Rotation of Parent

    :type: bool
    """

    use_rotation_y: bool
    """ Use Y Rotation of Parent

    :type: bool
    """

    use_rotation_z: bool
    """ Use Z Rotation of Parent

    :type: bool
    """

    use_scale_x: bool
    """ Use X Scale of Parent

    :type: bool
    """

    use_scale_y: bool
    """ Use Y Scale of Parent

    :type: bool
    """

    use_scale_z: bool
    """ Use Z Scale of Parent

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ChildParticle(bpy_struct):
    """Child particle interpolated from simulated or edited particles"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ClampToConstraint(Constraint, bpy_struct):
    """Constrain an object's location to the nearest point along the target path"""

    main_axis: typing.Literal["CLAMPTO_AUTO", "CLAMPTO_X", "CLAMPTO_Y", "CLAMPTO_Z"]
    """ Main axis of movement

    :type: typing.Literal['CLAMPTO_AUTO','CLAMPTO_X','CLAMPTO_Y','CLAMPTO_Z']
    """

    target: Object | None
    """ Target Object (Curves only)

    :type: Object | None
    """

    use_cyclic: bool
    """ Treat curve as cyclic curve (no clamping to curve bounding box)

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ClothCollisionSettings(bpy_struct):
    """Cloth simulation settings for self collision and collision with other objects"""

    collection: Collection | None
    """ Limit colliders to this Collection

    :type: Collection | None
    """

    collision_quality: int
    """ How many collision iterations should be done. (higher is better quality but slower)

    :type: int
    """

    damping: float
    """ Amount of velocity lost on collision

    :type: float
    """

    distance_min: float
    """ Minimum distance between collision objects before collision response takes effect

    :type: float
    """

    friction: float
    """ Friction force if a collision happened (higher = less movement)

    :type: float
    """

    impulse_clamp: float
    """ Clamp collision impulses to avoid instability (0.0 to disable clamping)

    :type: float
    """

    self_distance_min: float
    """ Minimum distance between cloth faces before collision response takes effect

    :type: float
    """

    self_friction: float
    """ Friction with self contact

    :type: float
    """

    self_impulse_clamp: float
    """ Clamp collision impulses to avoid instability (0.0 to disable clamping)

    :type: float
    """

    use_collision: bool
    """ Enable collisions with other objects

    :type: bool
    """

    use_self_collision: bool
    """ Enable self collisions

    :type: bool
    """

    vertex_group_object_collisions: str
    """ Triangles with all vertices in this group are not used during object collisions

    :type: str
    """

    vertex_group_self_collisions: str
    """ Triangles with all vertices in this group are not used during self collisions

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ClothModifier(Modifier, bpy_struct):
    """Cloth simulation modifier"""

    collision_settings: ClothCollisionSettings
    """ 

    :type: ClothCollisionSettings
    """

    hair_grid_max: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    hair_grid_min: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    hair_grid_resolution: bpy_prop_array[int]
    """ 

    :type: bpy_prop_array[int]
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    settings: ClothSettings
    """ 

    :type: ClothSettings
    """

    solver_result: ClothSolverResult
    """ 

    :type: ClothSolverResult
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ClothSettings(bpy_struct):
    """Cloth simulation settings for an object"""

    air_damping: float
    """ Air has normally some thickness which slows falling things down

    :type: float
    """

    bending_damping: float
    """ Amount of damping in bending behavior

    :type: float
    """

    bending_model: typing.Literal["ANGULAR", "LINEAR"]
    """ Physical model for simulating bending forces

    :type: typing.Literal['ANGULAR','LINEAR']
    """

    bending_stiffness: float
    """ How much the material resists bending

    :type: float
    """

    bending_stiffness_max: float
    """ Maximum bending stiffness value

    :type: float
    """

    collider_friction: float
    """ 

    :type: float
    """

    compression_damping: float
    """ Amount of damping in compression behavior

    :type: float
    """

    compression_stiffness: float
    """ How much the material resists compression

    :type: float
    """

    compression_stiffness_max: float
    """ Maximum compression stiffness value

    :type: float
    """

    density_strength: float
    """ Influence of target density on the simulation

    :type: float
    """

    density_target: float
    """ Maximum density of hair

    :type: float
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    fluid_density: float
    """ Density (kg/l) of the fluid contained inside the object, used to create a hydrostatic pressure gradient simulating the weight of the internal fluid, or buoyancy from the surrounding fluid if negative

    :type: float
    """

    goal_default: float
    """ Default Goal (vertex target position) value, when no Vertex Group used

    :type: float
    """

    goal_friction: float
    """ Goal (vertex target position) friction

    :type: float
    """

    goal_max: float
    """ Goal maximum, vertex group weights are scaled to match this range

    :type: float
    """

    goal_min: float
    """ Goal minimum, vertex group weights are scaled to match this range

    :type: float
    """

    goal_spring: float
    """ Goal (vertex target position) spring stiffness

    :type: float
    """

    gravity: mathutils.Vector
    """ Gravity or external force vector

    :type: mathutils.Vector
    """

    internal_compression_stiffness: float
    """ How much the material resists compression

    :type: float
    """

    internal_compression_stiffness_max: float
    """ Maximum compression stiffness value

    :type: float
    """

    internal_friction: float
    """ 

    :type: float
    """

    internal_spring_max_diversion: float
    """ How much the rays used to connect the internal points can diverge from the vertex normal

    :type: float
    """

    internal_spring_max_length: float
    """ The maximum length an internal spring can have during creation. If the distance between internal points is greater than this, no internal spring will be created between these points. A length of zero means that there is no length limit

    :type: float
    """

    internal_spring_normal_check: bool
    """ Require the points the internal springs connect to have opposite normal directions

    :type: bool
    """

    internal_tension_stiffness: float
    """ How much the material resists stretching

    :type: float
    """

    internal_tension_stiffness_max: float
    """ Maximum tension stiffness value

    :type: float
    """

    mass: float
    """ The mass of each vertex on the cloth material

    :type: float
    """

    pin_stiffness: float
    """ Pin (vertex target position) spring stiffness

    :type: float
    """

    pressure_factor: float
    """ Ambient pressure (kPa) that balances out between the inside and outside of the object when it has the target volume

    :type: float
    """

    quality: int
    """ Quality of the simulation in steps per frame (higher is better quality but slower)

    :type: int
    """

    rest_shape_key: ShapeKey | None
    """ Shape key to use the rest spring lengths from

    :type: ShapeKey | None
    """

    sewing_force_max: float
    """ Maximum sewing force

    :type: float
    """

    shear_damping: float
    """ Amount of damping in shearing behavior

    :type: float
    """

    shear_stiffness: float
    """ How much the material resists shearing

    :type: float
    """

    shear_stiffness_max: float
    """ Maximum shear scaling value

    :type: float
    """

    shrink_max: float
    """ Max amount to shrink cloth by

    :type: float
    """

    shrink_min: float
    """ Factor by which to shrink cloth

    :type: float
    """

    target_volume: float
    """ The mesh volume where the inner/outer pressure will be the same. If set to zero the change in volume will not affect pressure

    :type: float
    """

    tension_damping: float
    """ Amount of damping in stretching behavior

    :type: float
    """

    tension_stiffness: float
    """ How much the material resists stretching

    :type: float
    """

    tension_stiffness_max: float
    """ Maximum tension stiffness value

    :type: float
    """

    time_scale: float
    """ Cloth speed is multiplied by this value

    :type: float
    """

    uniform_pressure_force: float
    """ The uniform pressure that is constantly applied to the mesh, in units of Pressure Scale. Can be negative

    :type: float
    """

    use_dynamic_mesh: bool
    """ Make simulation respect deformations in the base mesh

    :type: bool
    """

    use_internal_springs: bool
    """ Simulate an internal volume structure by creating springs connecting the opposite sides of the mesh

    :type: bool
    """

    use_pressure: bool
    """ Simulate pressure inside a closed cloth mesh

    :type: bool
    """

    use_pressure_volume: bool
    """ Use the Target Volume parameter as the initial volume, instead of calculating it from the mesh itself

    :type: bool
    """

    use_sewing_springs: bool
    """ Pulls loose edges together

    :type: bool
    """

    vertex_group_bending: str
    """ Vertex group for fine control over bending stiffness

    :type: str
    """

    vertex_group_intern: str
    """ Vertex group for fine control over the internal spring stiffness

    :type: str
    """

    vertex_group_mass: str
    """ Vertex Group for pinning of vertices

    :type: str
    """

    vertex_group_pressure: str
    """ Vertex Group for where to apply pressure. Zero weight means no pressure while a weight of one means full pressure. Faces with a vertex that has zero weight will be excluded from the volume calculation

    :type: str
    """

    vertex_group_shear_stiffness: str
    """ Vertex group for fine control over shear stiffness

    :type: str
    """

    vertex_group_shrink: str
    """ Vertex Group for shrinking cloth

    :type: str
    """

    vertex_group_structural_stiffness: str
    """ Vertex group for fine control over structural stiffness

    :type: str
    """

    voxel_cell_size: float
    """ Size of the voxel grid cells for interaction effects

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ClothSolverResult(bpy_struct):
    """Result of cloth solver iteration"""

    avg_error: float
    """ Average error during substeps

    :type: float
    """

    avg_iterations: float
    """ Average iterations during substeps

    :type: float
    """

    max_error: float
    """ Maximum error during substeps

    :type: float
    """

    max_iterations: int
    """ Maximum iterations during substeps

    :type: int
    """

    min_error: float
    """ Minimum error during substeps

    :type: float
    """

    min_iterations: int
    """ Minimum iterations during substeps

    :type: int
    """

    status: set[
        typing.Literal["SUCCESS", "NUMERICAL_ISSUE", "NO_CONVERGENCE", "INVALID_INPUT"]
    ]
    """ Status of the solver iteration

    :type: set[typing.Literal['SUCCESS','NUMERICAL_ISSUE','NO_CONVERGENCE','INVALID_INPUT']]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CloudsTexture(Texture, ID, bpy_struct):
    """Procedural noise texture"""

    cloud_type: typing.Literal["GRAYSCALE", "COLOR"]
    """ Determine whether Noise returns grayscale or RGB values

    :type: typing.Literal['GRAYSCALE','COLOR']
    """

    nabla: float
    """ Size of derivative offset used for calculating normal

    :type: float
    """

    noise_basis: typing.Literal[
        "BLENDER_ORIGINAL",
        "ORIGINAL_PERLIN",
        "IMPROVED_PERLIN",
        "VORONOI_F1",
        "VORONOI_F2",
        "VORONOI_F3",
        "VORONOI_F4",
        "VORONOI_F2_F1",
        "VORONOI_CRACKLE",
        "CELL_NOISE",
    ]
    """ Noise basis used for turbulence

    :type: typing.Literal['BLENDER_ORIGINAL','ORIGINAL_PERLIN','IMPROVED_PERLIN','VORONOI_F1','VORONOI_F2','VORONOI_F3','VORONOI_F4','VORONOI_F2_F1','VORONOI_CRACKLE','CELL_NOISE']
    """

    noise_depth: int
    """ Depth of the cloud calculation

    :type: int
    """

    noise_scale: float
    """ Scaling for noise input

    :type: float
    """

    noise_type: typing.Literal["SOFT_NOISE", "HARD_NOISE"]
    """ 

    :type: typing.Literal['SOFT_NOISE','HARD_NOISE']
    """

    users_material: typing.Any
    """ Materials that use this texture(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture(readonly)"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Collection(ID, bpy_struct):
    """Collection of Object data-blocks"""

    all_objects: bpy_prop_collection[Object]
    """ Objects that are in this collection and its child collections

    :type: bpy_prop_collection[Object]
    """

    children: CollectionChildren
    """ Collections that are immediate children of this collection

    :type: CollectionChildren
    """

    collection_children: bpy_prop_collection[CollectionChild]
    """ Children collections their parent-collection-specific settings

    :type: bpy_prop_collection[CollectionChild]
    """

    collection_objects: bpy_prop_collection[CollectionObject]
    """ Objects of the collection with their parent-collection-specific settings

    :type: bpy_prop_collection[CollectionObject]
    """

    color_tag: bpy._typing.rna_enums.CollectionColorItems
    """ Color tag for a collection

    :type: bpy._typing.rna_enums.CollectionColorItems
    """

    hide_render: bool
    """ Globally disable in renders

    :type: bool
    """

    hide_select: bool
    """ Disable selection in viewport

    :type: bool
    """

    hide_viewport: bool
    """ Globally disable in viewports

    :type: bool
    """

    instance_offset: mathutils.Vector
    """ Offset from the origin to use when instancing

    :type: mathutils.Vector
    """

    lineart_intersection_mask: bpy_prop_array[bool]
    """ Intersection generated by this collection will have this mask value

    :type: bpy_prop_array[bool]
    """

    lineart_intersection_priority: int
    """ The intersection line will be included into the object with the higher intersection priority value

    :type: int
    """

    lineart_usage: typing.Literal[
        "INCLUDE",
        "OCCLUSION_ONLY",
        "EXCLUDE",
        "INTERSECTION_ONLY",
        "NO_INTERSECTION",
        "FORCE_INTERSECTION",
    ]
    """ How to use this collection in line art

    :type: typing.Literal['INCLUDE','OCCLUSION_ONLY','EXCLUDE','INTERSECTION_ONLY','NO_INTERSECTION','FORCE_INTERSECTION']
    """

    lineart_use_intersection_mask: bool
    """ Use custom intersection mask for faces in this collection

    :type: bool
    """

    objects: CollectionObjects
    """ Objects that are directly in this collection

    :type: CollectionObjects
    """

    use_lineart_intersection_priority: bool
    """ Assign intersection priority value for this collection

    :type: bool
    """

    children_recursive: typing.Any
    """ A list of all children from this collection.(readonly)"""

    users_dupli_group: typing.Any
    """ The collection instance objects this collection is used in(readonly)"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CollectionChild(bpy_struct):
    """Child collection with its collection related settings"""

    light_linking: CollectionLightLinking
    """ Light linking settings of the collection object

    :type: CollectionLightLinking
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CollectionLightLinking(bpy_struct):
    """Light linking settings of objects and children collections of a collection"""

    link_state: typing.Literal["INCLUDE", "EXCLUDE"]
    """ Light or shadow receiving state of the object or collection

    :type: typing.Literal['INCLUDE','EXCLUDE']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CollectionObject(bpy_struct):
    """Object of a collection with its collection related settings"""

    light_linking: CollectionLightLinking
    """ Light linking settings of the collection

    :type: CollectionLightLinking
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CollectionProperty(Property, bpy_struct):
    """RNA collection property to define lists, arrays and mappings"""

    fixed_type: Struct
    """ Fixed pointer type, empty if variable type

    :type: Struct
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CollisionModifier(Modifier, bpy_struct):
    """Collision modifier defining modifier stack position used for collision"""

    settings: CollisionSettings
    """ 

    :type: CollisionSettings
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CollisionSettings(bpy_struct):
    """Collision settings for object in physics simulation"""

    absorption: float
    """ How much of effector force gets lost during collision with this object (in percent)

    :type: float
    """

    cloth_friction: float
    """ Friction for cloth collisions

    :type: float
    """

    damping: float
    """ Amount of damping during collision

    :type: float
    """

    damping_factor: float
    """ Amount of damping during particle collision

    :type: float
    """

    damping_random: float
    """ Random variation of damping

    :type: float
    """

    friction_factor: float
    """ Amount of friction during particle collision

    :type: float
    """

    friction_random: float
    """ Random variation of friction

    :type: float
    """

    permeability: float
    """ Chance that the particle will pass through the mesh

    :type: float
    """

    stickiness: float
    """ Amount of stickiness to surface collision

    :type: float
    """

    thickness_inner: float
    """ Inner face thickness (only used by softbodies)

    :type: float
    """

    thickness_outer: float
    """ Outer face thickness

    :type: float
    """

    use: bool
    """ Enable this object as a collider for physics systems

    :type: bool
    """

    use_culling: bool
    """ Cloth collision acts with respect to the collider normals (improves penetration recovery)

    :type: bool
    """

    use_normal: bool
    """ Cloth collision impulses act in the direction of the collider normals (more reliable in some cases)

    :type: bool
    """

    use_particle_kill: bool
    """ Kill collided particles

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorBalanceModifier(SequenceModifier, bpy_struct):
    """Color balance modifier for sequence strip"""

    color_balance: SequenceColorBalanceData
    """ 

    :type: SequenceColorBalanceData
    """

    color_multiply: float
    """ Multiply the intensity of each pixel

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorGpencilModifier(GpencilModifier, bpy_struct):
    """Change Hue/Saturation modifier"""

    curve: CurveMapping
    """ Custom curve to apply effect

    :type: CurveMapping
    """

    hue: float
    """ Color Hue

    :type: float
    """

    invert_layer_pass: bool
    """ Inverse filter

    :type: bool
    """

    invert_layers: bool
    """ Inverse filter

    :type: bool
    """

    invert_material_pass: bool
    """ Inverse filter

    :type: bool
    """

    invert_materials: bool
    """ Inverse filter

    :type: bool
    """

    layer: str
    """ Layer name

    :type: str
    """

    layer_pass: int
    """ Layer pass index

    :type: int
    """

    material: Material | None
    """ Material used for filtering effect

    :type: Material | None
    """

    modify_color: typing.Literal["BOTH", "STROKE", "FILL"]
    """ Set what colors of the stroke are affected

    :type: typing.Literal['BOTH','STROKE','FILL']
    """

    pass_index: int
    """ Pass index

    :type: int
    """

    saturation: float
    """ Color Saturation

    :type: float
    """

    use_custom_curve: bool
    """ Use a custom curve to define color effect along the strokes

    :type: bool
    """

    value: float
    """ Color Value

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorManagedDisplaySettings(bpy_struct):
    """Color management specific to display device"""

    display_device: typing.Literal["NONE"]
    """ Display device name

    :type: typing.Literal['NONE']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorManagedInputColorspaceSettings(bpy_struct):
    """Input color space settings"""

    is_data: bool
    """ Treat image as non-color data without color management, like normal or displacement maps

    :type: bool
    """

    name: bpy._typing.rna_enums.ColorSpaceConvertDefaultItems
    """ Color space in the image file, to convert to and from when saving and loading the image

    :type: bpy._typing.rna_enums.ColorSpaceConvertDefaultItems
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorManagedSequencerColorspaceSettings(bpy_struct):
    """Input color space settings"""

    name: bpy._typing.rna_enums.ColorSpaceConvertDefaultItems
    """ Color space that the sequencer operates in

    :type: bpy._typing.rna_enums.ColorSpaceConvertDefaultItems
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorManagedViewSettings(bpy_struct):
    """Color management settings used for displaying images on the display"""

    curve_mapping: CurveMapping
    """ Color curve mapping applied before display transform

    :type: CurveMapping
    """

    exposure: float
    """ Exposure (stops) applied before display transform

    :type: float
    """

    gamma: float
    """ Amount of gamma modification applied after display transform

    :type: float
    """

    look: typing.Literal["NONE"]
    """ Additional transform applied before view transform for artistic needs

    :type: typing.Literal['NONE']
    """

    use_curve_mapping: bool
    """ Use RGB curved for pre-display transformation

    :type: bool
    """

    use_hdr_view: bool
    """ Enable high dynamic range display in rendered viewport, uncapping display brightness. This requires a monitor with HDR support and a view transform designed for HDR. 'Filmic' and 'AgX' do not generate HDR colors

    :type: bool
    """

    view_transform: typing.Literal["NONE"]
    """ View used when converting image to a display space

    :type: typing.Literal['NONE']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorMapping(bpy_struct):
    """Color mapping settings"""

    blend_color: mathutils.Color
    """ Blend color to mix with texture output color

    :type: mathutils.Color
    """

    blend_factor: float
    """ 

    :type: float
    """

    blend_type: typing.Literal[
        "MIX",
        "DARKEN",
        "MULTIPLY",
        "LIGHTEN",
        "SCREEN",
        "ADD",
        "OVERLAY",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
        "DIFFERENCE",
        "SUBTRACT",
        "DIVIDE",
        "HUE",
        "SATURATION",
        "COLOR",
        "VALUE",
    ]
    """ Mode used to mix with texture output color

    :type: typing.Literal['MIX','DARKEN','MULTIPLY','LIGHTEN','SCREEN','ADD','OVERLAY','SOFT_LIGHT','LINEAR_LIGHT','DIFFERENCE','SUBTRACT','DIVIDE','HUE','SATURATION','COLOR','VALUE']
    """

    brightness: float
    """ Adjust the brightness of the texture

    :type: float
    """

    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    contrast: float
    """ Adjust the contrast of the texture

    :type: float
    """

    saturation: float
    """ Adjust the saturation of colors in the texture

    :type: float
    """

    use_color_ramp: bool
    """ Toggle color ramp operations

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorMixSequence(EffectSequence, Sequence, bpy_struct):
    """Color Mix Sequence"""

    blend_effect: typing.Literal[
        "DARKEN",
        "MULTIPLY",
        "BURN",
        "LINEAR_BURN",
        "LIGHTEN",
        "SCREEN",
        "DODGE",
        "ADD",
        "OVERLAY",
        "SOFT_LIGHT",
        "HARD_LIGHT",
        "VIVID_LIGHT",
        "LINEAR_LIGHT",
        "PIN_LIGHT",
        "DIFFERENCE",
        "EXCLUSION",
        "SUBTRACT",
        "HUE",
        "SATURATION",
        "COLOR",
        "VALUE",
    ]
    """ Method for controlling how the strip combines with other strips

    :type: typing.Literal['DARKEN','MULTIPLY','BURN','LINEAR_BURN','LIGHTEN','SCREEN','DODGE','ADD','OVERLAY','SOFT_LIGHT','HARD_LIGHT','VIVID_LIGHT','LINEAR_LIGHT','PIN_LIGHT','DIFFERENCE','EXCLUSION','SUBTRACT','HUE','SATURATION','COLOR','VALUE']
    """

    factor: float
    """ Percentage of how much the strip's colors affect other strips

    :type: float
    """

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorRamp(bpy_struct):
    """Color ramp mapping a scalar value to a color"""

    color_mode: typing.Literal["RGB", "HSV", "HSL"]
    """ Set color mode to use for interpolation

    :type: typing.Literal['RGB','HSV','HSL']
    """

    elements: ColorRampElements
    """ 

    :type: ColorRampElements
    """

    hue_interpolation: typing.Literal["NEAR", "FAR", "CW", "CCW"]
    """ Set color interpolation

    :type: typing.Literal['NEAR','FAR','CW','CCW']
    """

    interpolation: typing.Literal["EASE", "CARDINAL", "LINEAR", "B_SPLINE", "CONSTANT"]
    """ Set interpolation between color stops

    :type: typing.Literal['EASE','CARDINAL','LINEAR','B_SPLINE','CONSTANT']
    """

    def evaluate(self, position: float | None) -> bpy_prop_array[float]:
        """Evaluate Color Ramp

        :param position: Position, Evaluate Color Ramp at position
        :type position: float | None
        :return: Color, Color at given position
        :rtype: bpy_prop_array[float]
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorRampElement(bpy_struct):
    """Element defining a color at a position in the color ramp"""

    alpha: float
    """ Set alpha of selected color stop

    :type: float
    """

    color: bpy_prop_array[float]
    """ Set color of selected color stop

    :type: bpy_prop_array[float]
    """

    position: float
    """ Set position of selected color stop

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ColorSequence(EffectSequence, Sequence, bpy_struct):
    """Sequence strip creating an image filled with a single color"""

    color: mathutils.Color
    """ Effect Strip color

    :type: mathutils.Color
    """

    input_count: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNode(NodeInternal, Node, bpy_struct):
    def tag_need_exec(self):
        """Tag the node for compositor update"""

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeAlphaOver(CompositorNode, NodeInternal, Node, bpy_struct):
    premul: float
    """ Mix Factor

    :type: float
    """

    use_premultiply: bool
    """ 

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeAntiAliasing(CompositorNode, NodeInternal, Node, bpy_struct):
    contrast_limit: float
    """ How much to eliminate spurious edges to avoid artifacts (the larger value makes less active; the value 2.0, for example, means discard a detected edge if there is a neighboring edge that has 2.0 times bigger contrast than the current one)

    :type: float
    """

    corner_rounding: float
    """ How much sharp corners will be rounded

    :type: float
    """

    threshold: float
    """ Threshold to detect edges (smaller threshold makes more sensitive detection)

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeBilateralblur(CompositorNode, NodeInternal, Node, bpy_struct):
    iterations: int
    """ 

    :type: int
    """

    sigma_color: float
    """ 

    :type: float
    """

    sigma_space: float
    """ 

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeBlur(CompositorNode, NodeInternal, Node, bpy_struct):
    aspect_correction: typing.Literal["NONE", "Y", "X"]
    """ Type of aspect correction to use

    :type: typing.Literal['NONE','Y','X']
    """

    factor: float
    """ 

    :type: float
    """

    factor_x: float
    """ 

    :type: float
    """

    factor_y: float
    """ 

    :type: float
    """

    filter_type: typing.Literal[
        "FLAT", "TENT", "QUAD", "CUBIC", "GAUSS", "FAST_GAUSS", "CATROM", "MITCH"
    ]
    """ 

    :type: typing.Literal['FLAT','TENT','QUAD','CUBIC','GAUSS','FAST_GAUSS','CATROM','MITCH']
    """

    size_x: int
    """ 

    :type: int
    """

    size_y: int
    """ 

    :type: int
    """

    use_bokeh: bool
    """ Use circular filter (slower)

    :type: bool
    """

    use_extended_bounds: bool
    """ Extend bounds of the input image to fully fit blurred image

    :type: bool
    """

    use_gamma_correction: bool
    """ Apply filter on gamma corrected values

    :type: bool
    """

    use_relative: bool
    """ Use relative (percent) values to define blur radius

    :type: bool
    """

    use_variable_size: bool
    """ Support variable blur per pixel when using an image for size input

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeBokehBlur(CompositorNode, NodeInternal, Node, bpy_struct):
    blur_max: float
    """ Blur limit, maximum CoC radius

    :type: float
    """

    use_extended_bounds: bool
    """ Extend bounds of the input image to fully fit blurred image

    :type: bool
    """

    use_variable_size: bool
    """ Support variable blur per pixel when using an image for size input

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeBokehImage(CompositorNode, NodeInternal, Node, bpy_struct):
    angle: float
    """ Angle of the bokeh

    :type: float
    """

    catadioptric: float
    """ Level of catadioptric of the bokeh

    :type: float
    """

    flaps: int
    """ Number of flaps

    :type: int
    """

    rounding: float
    """ Level of rounding of the bokeh

    :type: float
    """

    shift: float
    """ Shift of the lens components

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeBoxMask(CompositorNode, NodeInternal, Node, bpy_struct):
    height: float
    """ Height of the box

    :type: float
    """

    mask_type: typing.Literal["ADD", "SUBTRACT", "MULTIPLY", "NOT"]
    """ 

    :type: typing.Literal['ADD','SUBTRACT','MULTIPLY','NOT']
    """

    rotation: float
    """ Rotation angle of the box

    :type: float
    """

    width: float
    """ Width of the box

    :type: float
    """

    x: float
    """ X position of the middle of the box

    :type: float
    """

    y: float
    """ Y position of the middle of the box

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeBrightContrast(CompositorNode, NodeInternal, Node, bpy_struct):
    use_premultiply: bool
    """ Keep output image premultiplied alpha

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeChannelMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    color_space: typing.Literal["RGB", "HSV", "YUV", "YCC"]
    """ 

    :type: typing.Literal['RGB','HSV','YUV','YCC']
    """

    limit_channel: typing.Literal["R", "G", "B"]
    """ Limit by this channel's value

    :type: typing.Literal['R','G','B']
    """

    limit_max: float
    """ Values higher than this setting are 100% opaque

    :type: float
    """

    limit_method: typing.Literal["SINGLE", "MAX"]
    """ Algorithm to use to limit channel

    :type: typing.Literal['SINGLE','MAX']
    """

    limit_min: float
    """ Values lower than this setting are 100% keyed

    :type: float
    """

    matte_channel: typing.Literal["R", "G", "B"]
    """ Channel used to determine matte

    :type: typing.Literal['R','G','B']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeChromaMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    gain: float
    """ Alpha falloff

    :type: float
    """

    lift: float
    """ Alpha lift

    :type: float
    """

    shadow_adjust: float
    """ Adjusts the brightness of any shadows captured

    :type: float
    """

    threshold: float
    """ Tolerance below which colors will be considered as exact matches

    :type: float
    """

    tolerance: float
    """ Tolerance for a color to be considered a keying color

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeColorBalance(CompositorNode, NodeInternal, Node, bpy_struct):
    correction_method: typing.Literal["LIFT_GAMMA_GAIN", "OFFSET_POWER_SLOPE"]
    """ 

    :type: typing.Literal['LIFT_GAMMA_GAIN','OFFSET_POWER_SLOPE']
    """

    gain: mathutils.Color
    """ Correction for highlights

    :type: mathutils.Color
    """

    gamma: mathutils.Color
    """ Correction for midtones

    :type: mathutils.Color
    """

    lift: mathutils.Color
    """ Correction for shadows

    :type: mathutils.Color
    """

    offset: mathutils.Color
    """ Correction for entire tonal range

    :type: mathutils.Color
    """

    offset_basis: float
    """ Support negative color by using this as the RGB basis

    :type: float
    """

    power: mathutils.Color
    """ Correction for midtones

    :type: mathutils.Color
    """

    slope: mathutils.Color
    """ Correction for highlights

    :type: mathutils.Color
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeColorCorrection(CompositorNode, NodeInternal, Node, bpy_struct):
    blue: bool
    """ Blue channel active

    :type: bool
    """

    green: bool
    """ Green channel active

    :type: bool
    """

    highlights_contrast: float
    """ Highlights contrast

    :type: float
    """

    highlights_gain: float
    """ Highlights gain

    :type: float
    """

    highlights_gamma: float
    """ Highlights gamma

    :type: float
    """

    highlights_lift: float
    """ Highlights lift

    :type: float
    """

    highlights_saturation: float
    """ Highlights saturation

    :type: float
    """

    master_contrast: float
    """ Master contrast

    :type: float
    """

    master_gain: float
    """ Master gain

    :type: float
    """

    master_gamma: float
    """ Master gamma

    :type: float
    """

    master_lift: float
    """ Master lift

    :type: float
    """

    master_saturation: float
    """ Master saturation

    :type: float
    """

    midtones_contrast: float
    """ Midtones contrast

    :type: float
    """

    midtones_end: float
    """ End of midtones

    :type: float
    """

    midtones_gain: float
    """ Midtones gain

    :type: float
    """

    midtones_gamma: float
    """ Midtones gamma

    :type: float
    """

    midtones_lift: float
    """ Midtones lift

    :type: float
    """

    midtones_saturation: float
    """ Midtones saturation

    :type: float
    """

    midtones_start: float
    """ Start of midtones

    :type: float
    """

    red: bool
    """ Red channel active

    :type: bool
    """

    shadows_contrast: float
    """ Shadows contrast

    :type: float
    """

    shadows_gain: float
    """ Shadows gain

    :type: float
    """

    shadows_gamma: float
    """ Shadows gamma

    :type: float
    """

    shadows_lift: float
    """ Shadows lift

    :type: float
    """

    shadows_saturation: float
    """ Shadows saturation

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeColorMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    color_hue: float
    """ Hue tolerance for colors to be considered a keying color

    :type: float
    """

    color_saturation: float
    """ Saturation tolerance for the color

    :type: float
    """

    color_value: float
    """ Value tolerance for the color

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeColorSpill(CompositorNode, NodeInternal, Node, bpy_struct):
    channel: typing.Literal["R", "G", "B"]
    """ 

    :type: typing.Literal['R','G','B']
    """

    limit_channel: typing.Literal["R", "G", "B"]
    """ 

    :type: typing.Literal['R','G','B']
    """

    limit_method: typing.Literal["SIMPLE", "AVERAGE"]
    """ 

    :type: typing.Literal['SIMPLE','AVERAGE']
    """

    ratio: float
    """ Scale limit by value

    :type: float
    """

    unspill_blue: float
    """ Blue spillmap scale

    :type: float
    """

    unspill_green: float
    """ Green spillmap scale

    :type: float
    """

    unspill_red: float
    """ Red spillmap scale

    :type: float
    """

    use_unspill: bool
    """ Compensate all channels (differently) by hand

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCombHSVA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCombRGBA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCombYCCA(CompositorNode, NodeInternal, Node, bpy_struct):
    mode: typing.Literal["ITUBT601", "ITUBT709", "JFIF"]
    """ 

    :type: typing.Literal['ITUBT601','ITUBT709','JFIF']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCombYUVA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCombineColor(CompositorNode, NodeInternal, Node, bpy_struct):
    mode: typing.Literal["RGB", "HSV", "HSL", "YCC", "YUV"]
    """ Mode of color processing

    :type: typing.Literal['RGB','HSV','HSL','YCC','YUV']
    """

    ycc_mode: typing.Literal["ITUBT601", "ITUBT709", "JFIF"]
    """ Color space used for YCbCrA processing

    :type: typing.Literal['ITUBT601','ITUBT709','JFIF']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCombineXYZ(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeComposite(CompositorNode, NodeInternal, Node, bpy_struct):
    use_alpha: bool
    """ Colors are treated alpha premultiplied, or colors output straight (alpha gets set to 1)

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeConvertColorSpace(CompositorNode, NodeInternal, Node, bpy_struct):
    from_color_space: bpy._typing.rna_enums.ColorSpaceConvertDefaultItems
    """ Color space of the input image

    :type: bpy._typing.rna_enums.ColorSpaceConvertDefaultItems
    """

    to_color_space: bpy._typing.rna_enums.ColorSpaceConvertDefaultItems
    """ Color space of the output image

    :type: bpy._typing.rna_enums.ColorSpaceConvertDefaultItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCornerPin(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCrop(CompositorNode, NodeInternal, Node, bpy_struct):
    max_x: int
    """ 

    :type: int
    """

    max_y: int
    """ 

    :type: int
    """

    min_x: int
    """ 

    :type: int
    """

    min_y: int
    """ 

    :type: int
    """

    rel_max_x: float
    """ 

    :type: float
    """

    rel_max_y: float
    """ 

    :type: float
    """

    rel_min_x: float
    """ 

    :type: float
    """

    rel_min_y: float
    """ 

    :type: float
    """

    relative: bool
    """ Use relative values to crop image

    :type: bool
    """

    use_crop_size: bool
    """ Whether to crop the size of the input image

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCryptomatte(CompositorNode, NodeInternal, Node, bpy_struct):
    add: mathutils.Color
    """ Add object or material to matte, by picking a color from the Pick output

    :type: mathutils.Color
    """

    matte_id: str
    """ List of object and material crypto IDs to include in matte

    :type: str
    """

    remove: mathutils.Color
    """ Remove object or material from matte, by picking a color from the Pick output

    :type: mathutils.Color
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCryptomatteV2(CompositorNode, NodeInternal, Node, bpy_struct):
    add: mathutils.Color
    """ Add object or material to matte, by picking a color from the Pick output

    :type: mathutils.Color
    """

    entries: bpy_prop_collection[CryptomatteEntry]
    """ 

    :type: bpy_prop_collection[CryptomatteEntry]
    """

    frame_duration: int
    """ Number of images of a movie to use

    :type: int
    """

    frame_offset: int
    """ Offset the number of the frame to use in the animation

    :type: int
    """

    frame_start: int
    """ Global starting frame of the movie/sequence, assuming first picture has a #1

    :type: int
    """

    has_layers: bool
    """ True if this image has any named layer

    :type: bool
    """

    has_views: bool
    """ True if this image has multiple views

    :type: bool
    """

    image: Image | None
    """ 

    :type: Image | None
    """

    layer: typing.Literal["PLACEHOLDER"]
    """ 

    :type: typing.Literal['PLACEHOLDER']
    """

    layer_name: typing.Literal["CryptoObject", "CryptoMaterial", "CryptoAsset"]
    """ What Cryptomatte layer is used

    :type: typing.Literal['CryptoObject','CryptoMaterial','CryptoAsset']
    """

    matte_id: str
    """ List of object and material crypto IDs to include in matte

    :type: str
    """

    remove: mathutils.Color
    """ Remove object or material from matte, by picking a color from the Pick output

    :type: mathutils.Color
    """

    scene: Scene | None
    """ 

    :type: Scene | None
    """

    source: typing.Literal["RENDER", "IMAGE"]
    """ Where the Cryptomatte passes are loaded from

    :type: typing.Literal['RENDER','IMAGE']
    """

    use_auto_refresh: bool
    """ Always refresh image on frame changes

    :type: bool
    """

    use_cyclic: bool
    """ Cycle the images in the movie

    :type: bool
    """

    view: typing.Literal["ALL"]
    """ 

    :type: typing.Literal['ALL']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCurveRGB(CompositorNode, NodeInternal, Node, bpy_struct):
    mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCurveVec(CompositorNode, NodeInternal, Node, bpy_struct):
    mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeCustomGroup(CompositorNode, NodeInternal, Node, bpy_struct):
    """Custom Compositor Group Node for Python nodes"""

    node_tree: CompositorNodeTree | None
    """ 

    :type: CompositorNodeTree | None
    """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeDBlur(CompositorNode, NodeInternal, Node, bpy_struct):
    angle: float
    """ 

    :type: float
    """

    center_x: float
    """ 

    :type: float
    """

    center_y: float
    """ 

    :type: float
    """

    distance: float
    """ 

    :type: float
    """

    iterations: int
    """ 

    :type: int
    """

    spin: float
    """ 

    :type: float
    """

    zoom: float
    """ 

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeDefocus(CompositorNode, NodeInternal, Node, bpy_struct):
    angle: float
    """ Bokeh shape rotation offset

    :type: float
    """

    blur_max: float
    """ Blur limit, maximum CoC radius

    :type: float
    """

    bokeh: typing.Literal[
        "OCTAGON", "HEPTAGON", "HEXAGON", "PENTAGON", "SQUARE", "TRIANGLE", "CIRCLE"
    ]
    """ 

    :type: typing.Literal['OCTAGON','HEPTAGON','HEXAGON','PENTAGON','SQUARE','TRIANGLE','CIRCLE']
    """

    f_stop: float
    """ Amount of focal blur, 128 (infinity) is perfect focus, half the value doubles the blur radius

    :type: float
    """

    scene: Scene | None
    """ Scene from which to select the active camera (render scene if undefined)

    :type: Scene | None
    """

    threshold: float
    """ CoC radius threshold, prevents background bleed on in-focus midground, 0 is disabled

    :type: float
    """

    use_gamma_correction: bool
    """ Enable gamma correction before and after main process

    :type: bool
    """

    use_preview: bool
    """ Enable low quality mode, useful for preview

    :type: bool
    """

    use_zbuffer: bool
    """ Disable when using an image as input instead of actual z-buffer (auto enabled if node not image based, eg. time node)

    :type: bool
    """

    z_scale: float
    """ Scale the Z input when not using a z-buffer, controls maximum blur designated by the color white or input value 1

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeDenoise(CompositorNode, NodeInternal, Node, bpy_struct):
    prefilter: typing.Literal["NONE", "FAST", "ACCURATE"]
    """ Denoising prefilter

    :type: typing.Literal['NONE','FAST','ACCURATE']
    """

    use_hdr: bool
    """ Process HDR images

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeDespeckle(CompositorNode, NodeInternal, Node, bpy_struct):
    threshold: float
    """ Threshold for detecting pixels to despeckle

    :type: float
    """

    threshold_neighbor: float
    """ Threshold for the number of neighbor pixels that must match

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeDiffMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    falloff: float
    """ Color distances below this additional threshold are partially keyed

    :type: float
    """

    tolerance: float
    """ Color distances below this threshold are keyed

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeDilateErode(CompositorNode, NodeInternal, Node, bpy_struct):
    distance: int
    """ Distance to grow/shrink (number of iterations)

    :type: int
    """

    edge: float
    """ Edge to inset

    :type: float
    """

    falloff: bpy._typing.rna_enums.ProportionalFalloffCurveOnlyItems
    """ Falloff type the feather

    :type: bpy._typing.rna_enums.ProportionalFalloffCurveOnlyItems
    """

    mode: typing.Literal["STEP", "THRESHOLD", "DISTANCE", "FEATHER"]
    """ Growing/shrinking mode

    :type: typing.Literal['STEP','THRESHOLD','DISTANCE','FEATHER']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeDisplace(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeDistanceMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    channel: typing.Literal["RGB", "YCC"]
    """ 

    :type: typing.Literal['RGB','YCC']
    """

    falloff: float
    """ Color distances below this additional threshold are partially keyed

    :type: float
    """

    tolerance: float
    """ Color distances below this threshold are keyed

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeDoubleEdgeMask(CompositorNode, NodeInternal, Node, bpy_struct):
    edge_mode: typing.Literal["BLEED_OUT", "KEEP_IN"]
    """ 

    :type: typing.Literal['BLEED_OUT','KEEP_IN']
    """

    inner_mode: typing.Literal["ALL", "ADJACENT_ONLY"]
    """ 

    :type: typing.Literal['ALL','ADJACENT_ONLY']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeEllipseMask(CompositorNode, NodeInternal, Node, bpy_struct):
    height: float
    """ Height of the ellipse

    :type: float
    """

    mask_type: typing.Literal["ADD", "SUBTRACT", "MULTIPLY", "NOT"]
    """ 

    :type: typing.Literal['ADD','SUBTRACT','MULTIPLY','NOT']
    """

    rotation: float
    """ Rotation angle of the ellipse

    :type: float
    """

    width: float
    """ Width of the ellipse

    :type: float
    """

    x: float
    """ X position of the middle of the ellipse

    :type: float
    """

    y: float
    """ Y position of the middle of the ellipse

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeExposure(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeFilter(CompositorNode, NodeInternal, Node, bpy_struct):
    filter_type: bpy._typing.rna_enums.NodeFilterItems
    """ 

    :type: bpy._typing.rna_enums.NodeFilterItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeFlip(CompositorNode, NodeInternal, Node, bpy_struct):
    axis: typing.Literal["X", "Y", "XY"]
    """ 

    :type: typing.Literal['X','Y','XY']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeGamma(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeGlare(CompositorNode, NodeInternal, Node, bpy_struct):
    angle_offset: float
    """ Streak angle offset

    :type: float
    """

    color_modulation: float
    """ Amount of Color Modulation, modulates colors of streaks and ghosts for a spectral dispersion effect

    :type: float
    """

    fade: float
    """ Streak fade-out factor

    :type: float
    """

    glare_type: typing.Literal["GHOSTS", "STREAKS", "FOG_GLOW", "SIMPLE_STAR"]
    """ 

    :type: typing.Literal['GHOSTS','STREAKS','FOG_GLOW','SIMPLE_STAR']
    """

    iterations: int
    """ 

    :type: int
    """

    mix: float
    """ -1 is original image only, 0 is exact 50/50 mix, 1 is processed image only

    :type: float
    """

    quality: typing.Literal["HIGH", "MEDIUM", "LOW"]
    """ If not set to high quality, the effect will be applied to a low-res copy of the source image

    :type: typing.Literal['HIGH','MEDIUM','LOW']
    """

    size: int
    """ Glow/glare size (not actual size; relative to initial size of bright area of pixels)

    :type: int
    """

    streaks: int
    """ Total number of streaks

    :type: int
    """

    threshold: float
    """ The glare filter will only be applied to pixels brighter than this value

    :type: float
    """

    use_rotate_45: bool
    """ Simple star filter: add 45 degree rotation offset

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeGroup(CompositorNode, NodeInternal, Node, bpy_struct):
    node_tree: CompositorNodeTree | None
    """ 

    :type: CompositorNodeTree | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeHueCorrect(CompositorNode, NodeInternal, Node, bpy_struct):
    mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeHueSat(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeIDMask(CompositorNode, NodeInternal, Node, bpy_struct):
    index: int
    """ Pass index number to convert to alpha

    :type: int
    """

    use_antialiasing: bool
    """ Apply an anti-aliasing filter to the mask

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeImage(CompositorNode, NodeInternal, Node, bpy_struct):
    frame_duration: int
    """ Number of images of a movie to use

    :type: int
    """

    frame_offset: int
    """ Offset the number of the frame to use in the animation

    :type: int
    """

    frame_start: int
    """ Global starting frame of the movie/sequence, assuming first picture has a #1

    :type: int
    """

    has_layers: bool
    """ True if this image has any named layer

    :type: bool
    """

    has_views: bool
    """ True if this image has multiple views

    :type: bool
    """

    image: Image | None
    """ 

    :type: Image | None
    """

    layer: typing.Literal["PLACEHOLDER"]
    """ 

    :type: typing.Literal['PLACEHOLDER']
    """

    use_auto_refresh: bool
    """ Always refresh image on frame changes

    :type: bool
    """

    use_cyclic: bool
    """ Cycle the images in the movie

    :type: bool
    """

    use_straight_alpha_output: bool
    """ Put node output buffer to straight alpha instead of premultiplied

    :type: bool
    """

    view: typing.Literal["ALL"]
    """ 

    :type: typing.Literal['ALL']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeInpaint(CompositorNode, NodeInternal, Node, bpy_struct):
    distance: int
    """ Distance to inpaint (number of iterations)

    :type: int
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeInvert(CompositorNode, NodeInternal, Node, bpy_struct):
    invert_alpha: bool
    """ 

    :type: bool
    """

    invert_rgb: bool
    """ 

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeKeying(CompositorNode, NodeInternal, Node, bpy_struct):
    blur_post: int
    """ Matte blur size which applies after clipping and dilate/eroding

    :type: int
    """

    blur_pre: int
    """ Chroma pre-blur size which applies before running keyer

    :type: int
    """

    clip_black: float
    """ Value of non-scaled matte pixel which considers as fully background pixel

    :type: float
    """

    clip_white: float
    """ Value of non-scaled matte pixel which considers as fully foreground pixel

    :type: float
    """

    despill_balance: float
    """ Balance between non-key colors used to detect amount of key color to be removed

    :type: float
    """

    despill_factor: float
    """ Factor of despilling screen color from image

    :type: float
    """

    dilate_distance: int
    """ Distance to grow/shrink the matte

    :type: int
    """

    edge_kernel_radius: int
    """ Radius of kernel used to detect whether pixel belongs to edge

    :type: int
    """

    edge_kernel_tolerance: float
    """ Tolerance to pixels inside kernel which are treating as belonging to the same plane

    :type: float
    """

    feather_distance: int
    """ Distance to grow/shrink the feather

    :type: int
    """

    feather_falloff: bpy._typing.rna_enums.ProportionalFalloffCurveOnlyItems
    """ Falloff type the feather

    :type: bpy._typing.rna_enums.ProportionalFalloffCurveOnlyItems
    """

    screen_balance: float
    """ Balance between two non-primary channels primary channel is comparing against

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeKeyingScreen(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    tracking_object: str
    """ 

    :type: str
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeKuwahara(CompositorNode, NodeInternal, Node, bpy_struct):
    eccentricity: float
    """ Controls how directional the filter is. 0 means the filter is completely omnidirectional while 2 means it is maximally directed along the edges of the image

    :type: float
    """

    sharpness: float
    """ Controls the sharpness of the filter. 0 means completely smooth while 1 means completely sharp

    :type: float
    """

    size: int
    """ Size of filter. Larger values give stronger stylized effect

    :type: int
    """

    uniformity: int
    """ Controls the uniformity of the direction of the filter. Higher values produces more uniform directions

    :type: int
    """

    variation: typing.Literal["CLASSIC", "ANISOTROPIC"]
    """ Variation of Kuwahara filter to use

    :type: typing.Literal['CLASSIC','ANISOTROPIC']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeLensdist(CompositorNode, NodeInternal, Node, bpy_struct):
    use_fit: bool
    """ For positive distortion factor only: scale image such that black areas are not visible

    :type: bool
    """

    use_jitter: bool
    """ Enable/disable jittering (faster, but also noisier)

    :type: bool
    """

    use_projector: bool
    """ Enable/disable projector mode (the effect is applied in horizontal direction only)

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeLevels(CompositorNode, NodeInternal, Node, bpy_struct):
    channel: typing.Literal["COMBINED_RGB", "RED", "GREEN", "BLUE", "LUMINANCE"]
    """ 

    :type: typing.Literal['COMBINED_RGB','RED','GREEN','BLUE','LUMINANCE']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeLumaMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    limit_max: float
    """ Values higher than this setting are 100% opaque

    :type: float
    """

    limit_min: float
    """ Values lower than this setting are 100% keyed

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeMapRange(CompositorNode, NodeInternal, Node, bpy_struct):
    use_clamp: bool
    """ Clamp the result of the node to the target range

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeMapUV(CompositorNode, NodeInternal, Node, bpy_struct):
    alpha: int
    """ 

    :type: int
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeMapValue(CompositorNode, NodeInternal, Node, bpy_struct):
    max: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    min: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    offset: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    size: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    use_max: bool
    """ 

    :type: bool
    """

    use_min: bool
    """ 

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeMask(CompositorNode, NodeInternal, Node, bpy_struct):
    mask: Mask | None
    """ 

    :type: Mask | None
    """

    motion_blur_samples: int
    """ Number of motion blur samples

    :type: int
    """

    motion_blur_shutter: float
    """ Exposure for motion blur as a factor of FPS

    :type: float
    """

    size_source: typing.Literal["SCENE", "FIXED", "FIXED_SCENE"]
    """ Where to get the mask size from for aspect/size information

    :type: typing.Literal['SCENE','FIXED','FIXED_SCENE']
    """

    size_x: int
    """ 

    :type: int
    """

    size_y: int
    """ 

    :type: int
    """

    use_feather: bool
    """ Use feather information from the mask

    :type: bool
    """

    use_motion_blur: bool
    """ Use multi-sampled motion blur of the mask

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeMath(CompositorNode, NodeInternal, Node, bpy_struct):
    operation: bpy._typing.rna_enums.NodeMathItems
    """ 

    :type: bpy._typing.rna_enums.NodeMathItems
    """

    use_clamp: bool
    """ Clamp result of the node to 0.0 to 1.0 range

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeMixRGB(CompositorNode, NodeInternal, Node, bpy_struct):
    blend_type: bpy._typing.rna_enums.RampBlendItems
    """ 

    :type: bpy._typing.rna_enums.RampBlendItems
    """

    use_alpha: bool
    """ Include alpha of second input in this operation

    :type: bool
    """

    use_clamp: bool
    """ Clamp result of the node to 0.0 to 1.0 range

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeMovieClip(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeMovieDistortion(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    distortion_type: typing.Literal["UNDISTORT", "DISTORT"]
    """ Distortion to use to filter image

    :type: typing.Literal['UNDISTORT','DISTORT']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeNormal(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeNormalize(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeOutputFile(CompositorNode, NodeInternal, Node, bpy_struct):
    active_input_index: int | None
    """ Active input index in details view list

    :type: int | None
    """

    base_path: str
    """ Base output path for the image

    :type: str
    """

    file_slots: CompositorNodeOutputFileFileSlots
    """ 

    :type: CompositorNodeOutputFileFileSlots
    """

    format: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    layer_slots: CompositorNodeOutputFileLayerSlots
    """ 

    :type: CompositorNodeOutputFileLayerSlots
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodePixelate(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodePlaneTrackDeform(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    motion_blur_samples: int
    """ Number of motion blur samples

    :type: int
    """

    motion_blur_shutter: float
    """ Exposure for motion blur as a factor of FPS

    :type: float
    """

    plane_track_name: str
    """ 

    :type: str
    """

    tracking_object: str
    """ 

    :type: str
    """

    use_motion_blur: bool
    """ Use multi-sampled motion blur of the mask

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodePosterize(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodePremulKey(CompositorNode, NodeInternal, Node, bpy_struct):
    mapping: typing.Literal["STRAIGHT_TO_PREMUL", "PREMUL_TO_STRAIGHT"]
    """ Conversion between premultiplied alpha and key alpha

    :type: typing.Literal['STRAIGHT_TO_PREMUL','PREMUL_TO_STRAIGHT']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeRGB(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeRGBToBW(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeRLayers(CompositorNode, NodeInternal, Node, bpy_struct):
    layer: typing.Literal["PLACEHOLDER"]
    """ 

    :type: typing.Literal['PLACEHOLDER']
    """

    scene: Scene | None
    """ 

    :type: Scene | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeRotate(CompositorNode, NodeInternal, Node, bpy_struct):
    filter_type: typing.Literal["NEAREST", "BILINEAR", "BICUBIC"]
    """ Method to use to filter rotation

    :type: typing.Literal['NEAREST','BILINEAR','BICUBIC']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeScale(CompositorNode, NodeInternal, Node, bpy_struct):
    frame_method: typing.Literal["STRETCH", "FIT", "CROP"]
    """ How the image fits in the camera frame

    :type: typing.Literal['STRETCH','FIT','CROP']
    """

    offset_x: float
    """ Offset image horizontally (factor of image size)

    :type: float
    """

    offset_y: float
    """ Offset image vertically (factor of image size)

    :type: float
    """

    space: typing.Literal["RELATIVE", "ABSOLUTE", "SCENE_SIZE", "RENDER_SIZE"]
    """ Coordinate space to scale relative to

    :type: typing.Literal['RELATIVE','ABSOLUTE','SCENE_SIZE','RENDER_SIZE']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSceneTime(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSepHSVA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSepRGBA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSepYCCA(CompositorNode, NodeInternal, Node, bpy_struct):
    mode: typing.Literal["ITUBT601", "ITUBT709", "JFIF"]
    """ 

    :type: typing.Literal['ITUBT601','ITUBT709','JFIF']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSepYUVA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSeparateColor(CompositorNode, NodeInternal, Node, bpy_struct):
    mode: typing.Literal["RGB", "HSV", "HSL", "YCC", "YUV"]
    """ Mode of color processing

    :type: typing.Literal['RGB','HSV','HSL','YCC','YUV']
    """

    ycc_mode: typing.Literal["ITUBT601", "ITUBT709", "JFIF"]
    """ Color space used for YCbCrA processing

    :type: typing.Literal['ITUBT601','ITUBT709','JFIF']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSeparateXYZ(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSetAlpha(CompositorNode, NodeInternal, Node, bpy_struct):
    mode: typing.Literal["APPLY", "REPLACE_ALPHA"]
    """ 

    :type: typing.Literal['APPLY','REPLACE_ALPHA']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSplitViewer(CompositorNode, NodeInternal, Node, bpy_struct):
    axis: bpy._typing.rna_enums.AxisXyItems
    """ 

    :type: bpy._typing.rna_enums.AxisXyItems
    """

    factor: int
    """ 

    :type: int
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeStabilize(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    filter_type: typing.Literal["NEAREST", "BILINEAR", "BICUBIC"]
    """ Method to use to filter stabilization

    :type: typing.Literal['NEAREST','BILINEAR','BICUBIC']
    """

    invert: bool
    """ Invert stabilization to re-introduce motion to the frame

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSunBeams(CompositorNode, NodeInternal, Node, bpy_struct):
    ray_length: float
    """ Length of rays as a factor of the image size

    :type: float
    """

    source: bpy_prop_array[float]
    """ Source point of rays as a factor of the image width and height

    :type: bpy_prop_array[float]
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSwitch(CompositorNode, NodeInternal, Node, bpy_struct):
    check: bool
    """ Off: first socket, On: second socket

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeSwitchView(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeTexture(CompositorNode, NodeInternal, Node, bpy_struct):
    node_output: int
    """ For node-based textures, which output node to use

    :type: int
    """

    texture: Texture | None
    """ 

    :type: Texture | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeTime(CompositorNode, NodeInternal, Node, bpy_struct):
    curve: CurveMapping
    """ 

    :type: CurveMapping
    """

    frame_end: int
    """ 

    :type: int
    """

    frame_start: int
    """ 

    :type: int
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeTonemap(CompositorNode, NodeInternal, Node, bpy_struct):
    adaptation: float
    """ If 0, global; if 1, based on pixel intensity

    :type: float
    """

    contrast: float
    """ Set to 0 to use estimate from input image

    :type: float
    """

    correction: float
    """ If 0, same for all channels; if 1, each independent

    :type: float
    """

    gamma: float
    """ If not used, set to 1

    :type: float
    """

    intensity: float
    """ If less than zero, darkens image; otherwise, makes it brighter

    :type: float
    """

    key: float
    """ The value the average luminance is mapped to

    :type: float
    """

    offset: float
    """ Normally always 1, but can be used as an extra control to alter the brightness curve

    :type: float
    """

    tonemap_type: typing.Literal["RD_PHOTORECEPTOR", "RH_SIMPLE"]
    """ 

    :type: typing.Literal['RD_PHOTORECEPTOR','RH_SIMPLE']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeTrackPos(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    frame_relative: int
    """ Frame to be used for relative position

    :type: int
    """

    position: typing.Literal[
        "ABSOLUTE", "RELATIVE_START", "RELATIVE_FRAME", "ABSOLUTE_FRAME"
    ]
    """ Which marker position to use for output

    :type: typing.Literal['ABSOLUTE','RELATIVE_START','RELATIVE_FRAME','ABSOLUTE_FRAME']
    """

    track_name: str
    """ 

    :type: str
    """

    tracking_object: str
    """ 

    :type: str
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeTransform(CompositorNode, NodeInternal, Node, bpy_struct):
    filter_type: typing.Literal["NEAREST", "BILINEAR", "BICUBIC"]
    """ Method to use to filter transform

    :type: typing.Literal['NEAREST','BILINEAR','BICUBIC']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeTranslate(CompositorNode, NodeInternal, Node, bpy_struct):
    use_relative: bool
    """ Use relative (fraction of input image size) values to define translation

    :type: bool
    """

    wrap_axis: typing.Literal["NONE", "XAXIS", "YAXIS", "BOTH"]
    """ Wrap image on a specific axis

    :type: typing.Literal['NONE','XAXIS','YAXIS','BOTH']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeTree(NodeTree, ID, bpy_struct):
    """Node tree consisting of linked nodes used for compositing"""

    chunk_size: typing.Literal["32", "64", "128", "256", "512", "1024"]
    """ Max size of a tile (smaller values gives better distribution of multiple threads, but more overhead)

    :type: typing.Literal['32','64','128','256','512','1024']
    """

    edit_quality: typing.Literal["HIGH", "MEDIUM", "LOW"]
    """ Quality when editing

    :type: typing.Literal['HIGH','MEDIUM','LOW']
    """

    execution_mode: typing.Literal["TILED", "FULL_FRAME", "REALTIME"]
    """ Set how compositing is executed

    :type: typing.Literal['TILED','FULL_FRAME','REALTIME']
    """

    render_quality: typing.Literal["HIGH", "MEDIUM", "LOW"]
    """ Quality when rendering

    :type: typing.Literal['HIGH','MEDIUM','LOW']
    """

    use_groupnode_buffer: bool
    """ Enable buffering of group nodes

    :type: bool
    """

    use_opencl: bool
    """ Enable GPU calculations

    :type: bool
    """

    use_two_pass: bool
    """ Use two pass execution during editing: first calculate fast nodes, second pass calculate all nodes

    :type: bool
    """

    use_viewer_border: bool
    """ Use boundaries for viewer nodes and composite backdrop

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeValToRGB(CompositorNode, NodeInternal, Node, bpy_struct):
    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeValue(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeVecBlur(CompositorNode, NodeInternal, Node, bpy_struct):
    factor: float
    """ Scaling factor for motion vectors (actually, 'shutter speed', in frames)

    :type: float
    """

    samples: int
    """ 

    :type: int
    """

    speed_max: int
    """ Maximum speed, or zero for none

    :type: int
    """

    speed_min: int
    """ Minimum speed for a pixel to be blurred (used to separate background from foreground)

    :type: int
    """

    use_curved: bool
    """ Interpolate between frames in a Bezier curve, rather than linearly

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeViewer(CompositorNode, NodeInternal, Node, bpy_struct):
    center_x: float
    """ 

    :type: float
    """

    center_y: float
    """ 

    :type: float
    """

    tile_order: typing.Literal["CENTEROUT", "RANDOM", "BOTTOMUP", "RULE_OF_THIRDS"]
    """ Tile order

    :type: typing.Literal['CENTEROUT','RANDOM','BOTTOMUP','RULE_OF_THIRDS']
    """

    use_alpha: bool
    """ Colors are treated alpha premultiplied, or colors output straight (alpha gets set to 1)

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CompositorNodeZcombine(CompositorNode, NodeInternal, Node, bpy_struct):
    use_alpha: bool
    """ Take alpha channel into account when doing the Z operation

    :type: bool
    """

    use_antialias_z: bool
    """ Anti-alias the z-buffer to try to avoid artifacts, mostly useful for Blender renders

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ConsoleLine(bpy_struct):
    """Input line for the interactive console"""

    body: str
    """ Text in the line

    :type: str
    """

    current_character: int
    """ 

    :type: int
    """

    type: typing.Literal["OUTPUT", "INPUT", "INFO", "ERROR"]
    """ Console line type when used in scrollback

    :type: typing.Literal['OUTPUT','INPUT','INFO','ERROR']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Constraint(bpy_struct):
    """Constraint modifying the transformation of objects and bones"""

    active: bool | None
    """ Constraint is the one being edited

    :type: bool | None
    """

    enabled: bool
    """ Use the results of this constraint

    :type: bool
    """

    error_location: float
    """ Amount of residual error in Blender space unit for constraints that work on position

    :type: float
    """

    error_rotation: float
    """ Amount of residual error in radians for constraints that work on orientation

    :type: float
    """

    influence: float
    """ Amount of influence constraint will have on the final solution

    :type: float
    """

    is_override_data: bool
    """ In a local override object, whether this constraint comes from the linked reference object, or is local to the override

    :type: bool
    """

    is_valid: bool
    """ Constraint has valid settings and can be evaluated

    :type: bool
    """

    mute: bool
    """ Enable/Disable Constraint

    :type: bool
    """

    name: str
    """ Constraint name

    :type: str
    """

    owner_space: typing.Literal["WORLD", "CUSTOM", "POSE", "LOCAL_WITH_PARENT", "LOCAL"]
    """ Space that owner is evaluated in

    :type: typing.Literal['WORLD','CUSTOM','POSE','LOCAL_WITH_PARENT','LOCAL']
    """

    show_expanded: bool
    """ Constraint's panel is expanded in UI

    :type: bool
    """

    space_object: Object | None
    """ Object for Custom Space

    :type: Object | None
    """

    space_subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target_space: typing.Literal[
        "WORLD", "CUSTOM", "POSE", "LOCAL_WITH_PARENT", "LOCAL", "LOCAL_OWNER_ORIENT"
    ]
    """ Space that target is evaluated in

    :type: typing.Literal['WORLD','CUSTOM','POSE','LOCAL_WITH_PARENT','LOCAL','LOCAL_OWNER_ORIENT']
    """

    type: bpy._typing.rna_enums.ConstraintTypeItems
    """ 

    :type: bpy._typing.rna_enums.ConstraintTypeItems
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ConstraintTarget(bpy_struct):
    """Target object for multi-target constraints"""

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object | None
    """ Target object

    :type: Object | None
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ConstraintTargetBone(bpy_struct):
    """Target bone for multi-target constraints"""

    subtarget: str
    """ Target armature bone

    :type: str
    """

    target: Object | None
    """ Target armature

    :type: Object | None
    """

    weight: float
    """ Blending weight of this bone

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Context(bpy_struct):
    """Current windowmanager and data context"""

    area: Area
    """ 

    :type: Area
    """

    asset: AssetRepresentation
    """ 

    :type: AssetRepresentation
    """

    blend_data: BlendData
    """ 

    :type: BlendData
    """

    collection: Collection
    """ 

    :type: Collection
    """

    engine: str
    """ 

    :type: str
    """

    gizmo_group: GizmoGroup
    """ 

    :type: GizmoGroup
    """

    layer_collection: LayerCollection
    """ 

    :type: LayerCollection
    """

    mode: bpy._typing.rna_enums.ContextModeItems
    """ 

    :type: bpy._typing.rna_enums.ContextModeItems
    """

    preferences: Preferences
    """ 

    :type: Preferences
    """

    region: Region
    """ 

    :type: Region
    """

    region_data: RegionView3D
    """ 

    :type: RegionView3D
    """

    scene: Scene
    """ 

    :type: Scene
    """

    screen: Screen
    """ 

    :type: Screen
    """

    space_data: Space
    """ The current space, may be None in background-mode, when the cursor is outside the window or when using menu-search

    :type: Space
    """

    tool_settings: ToolSettings
    """ 

    :type: ToolSettings
    """

    view_layer: ViewLayer
    """ 

    :type: ViewLayer
    """

    window: Window
    """ 

    :type: Window
    """

    window_manager: WindowManager
    """ 

    :type: WindowManager
    """

    workspace: WorkSpace
    """ 

    :type: WorkSpace
    """

    texture_slot: TextureSlot | None
    """ 

    :type: TextureSlot | None
    """

    world: World | None
    """ 

    :type: World | None
    """

    object: Object | None
    """ 

    :type: Object | None
    """

    mesh: Mesh | None
    """ 

    :type: Mesh | None
    """

    armature: Armature | None
    """ 

    :type: Armature | None
    """

    lattice: Lattice | None
    """ 

    :type: Lattice | None
    """

    curve: Curve | None
    """ 

    :type: Curve | None
    """

    meta_ball: MetaBall | None
    """ 

    :type: MetaBall | None
    """

    light: Light | None
    """ 

    :type: Light | None
    """

    speaker: Speaker | None
    """ 

    :type: Speaker | None
    """

    lightprobe: LightProbe | None
    """ 

    :type: LightProbe | None
    """

    camera: Camera | None
    """ 

    :type: Camera | None
    """

    material: Material | None
    """ 

    :type: Material | None
    """

    material_slot: MaterialSlot | None
    """ 

    :type: MaterialSlot | None
    """

    texture: Texture | None
    """ 

    :type: Texture | None
    """

    texture_user: ID | None
    """ 

    :type: ID | None
    """

    texture_user_property: Property | None
    """ 

    :type: Property | None
    """

    bone: Bone | None
    """ 

    :type: Bone | None
    """

    edit_bone: EditBone | None
    """ 

    :type: EditBone | None
    """

    pose_bone: PoseBone | None
    """ 

    :type: PoseBone | None
    """

    particle_system: ParticleSystem | None
    """ 

    :type: ParticleSystem | None
    """

    particle_system_editable: ParticleSystem | None
    """ 

    :type: ParticleSystem | None
    """

    particle_settings: ParticleSettings | None
    """ 

    :type: ParticleSettings | None
    """

    cloth: ClothModifier | None
    """ 

    :type: ClothModifier | None
    """

    soft_body: SoftBodyModifier | None
    """ 

    :type: SoftBodyModifier | None
    """

    fluid: typing.Any
    collision: CollisionModifier | None
    """ 

    :type: CollisionModifier | None
    """

    brush: Brush | None
    """ 

    :type: Brush | None
    """

    dynamic_paint: DynamicPaintModifier | None
    """ 

    :type: DynamicPaintModifier | None
    """

    line_style: FreestyleLineStyle | None
    """ 

    :type: FreestyleLineStyle | None
    """

    gpencil: GreasePencil | None
    """ 

    :type: GreasePencil | None
    """

    grease_pencil: typing.Any
    curves: typing.Any
    volume: Volume | None
    """ 

    :type: Volume | None
    """

    edit_movieclip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    edit_mask: Mask | None
    """ 

    :type: Mask | None
    """

    active_file: FileSelectEntry | None
    """ 

    :type: FileSelectEntry | None
    """

    selected_files: list[FileSelectEntry]
    """ 

    :type: list[FileSelectEntry]
    """

    asset_library_reference: AssetLibraryReference | None
    """ 

    :type: AssetLibraryReference | None
    """

    selected_assets: list[AssetRepresentation]
    """ 

    :type: list[AssetRepresentation]
    """

    id: ID | None
    """ 

    :type: ID | None
    """

    selected_ids: list[ID]
    """ 

    :type: list[ID]
    """

    edit_image: Image | None
    """ 

    :type: Image | None
    """

    selected_nodes: list[Node]
    """ 

    :type: list[Node]
    """

    active_node: Node | None
    """ 

    :type: Node | None
    """

    visible_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selectable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_objects: list[Object]
    """ 

    :type: list[Object]
    """

    editable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_editable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    objects_in_mode: list[Object]
    """ 

    :type: list[Object]
    """

    objects_in_mode_unique_data: list[Object]
    """ 

    :type: list[Object]
    """

    visible_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    editable_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    selected_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    selected_editable_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    visible_pose_bones: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    selected_pose_bones: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    selected_pose_bones_from_active_object: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    active_bone: EditBone | None
    """ 

    :type: EditBone | None
    """

    active_pose_bone: PoseBone | None
    """ 

    :type: PoseBone | None
    """

    active_object: Object | None
    """ 

    :type: Object | None
    """

    edit_object: Object | None
    """ 

    :type: Object | None
    """

    sculpt_object: Object | None
    """ 

    :type: Object | None
    """

    vertex_paint_object: Object | None
    """ 

    :type: Object | None
    """

    weight_paint_object: Object | None
    """ 

    :type: Object | None
    """

    image_paint_object: Object | None
    """ 

    :type: Object | None
    """

    particle_edit_object: Object | None
    """ 

    :type: Object | None
    """

    pose_object: Object | None
    """ 

    :type: Object | None
    """

    active_sequence_strip: Sequence | None
    """ 

    :type: Sequence | None
    """

    sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    selected_sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    selected_editable_sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    active_nla_track: NlaTrack | None
    """ 

    :type: NlaTrack | None
    """

    active_nla_strip: NlaStrip | None
    """ 

    :type: NlaStrip | None
    """

    selected_nla_strips: list[NlaStrip]
    """ 

    :type: list[NlaStrip]
    """

    selected_movieclip_tracks: list[MovieTrackingTrack]
    """ 

    :type: list[MovieTrackingTrack]
    """

    gpencil_data: GreasePencil | None
    """ 

    :type: GreasePencil | None
    """

    gpencil_data_owner: ID | None
    """ 

    :type: ID | None
    """

    annotation_data: GreasePencil | None
    """ 

    :type: GreasePencil | None
    """

    annotation_data_owner: ID | None
    """ 

    :type: ID | None
    """

    visible_gpencil_layers: list[GPencilLayer]
    """ 

    :type: list[GPencilLayer]
    """

    editable_gpencil_layers: list[GPencilLayer]
    """ 

    :type: list[GPencilLayer]
    """

    editable_gpencil_strokes: list[GPencilStroke]
    """ 

    :type: list[GPencilStroke]
    """

    active_gpencil_layer: list[GPencilLayer] | None
    """ 

    :type: list[GPencilLayer] | None
    """

    active_gpencil_frame: typing.Any
    active_annotation_layer: GPencilLayer | None
    """ 

    :type: GPencilLayer | None
    """

    active_operator: Operator | None
    """ 

    :type: Operator | None
    """

    active_action: Action | None
    """ 

    :type: Action | None
    """

    selected_visible_actions: list[Action]
    """ 

    :type: list[Action]
    """

    selected_editable_actions: list[Action]
    """ 

    :type: list[Action]
    """

    visible_fcurves: list[FCurve]
    """ 

    :type: list[FCurve]
    """

    editable_fcurves: list[FCurve]
    """ 

    :type: list[FCurve]
    """

    selected_visible_fcurves: list[FCurve]
    """ 

    :type: list[FCurve]
    """

    selected_editable_fcurves: list[FCurve]
    """ 

    :type: list[FCurve]
    """

    active_editable_fcurve: FCurve | None
    """ 

    :type: FCurve | None
    """

    selected_editable_keyframes: list[Keyframe]
    """ 

    :type: list[Keyframe]
    """

    ui_list: UIList | None
    """ 

    :type: UIList | None
    """

    property: tuple[ID]
    """ Get the property associated with a hovered button.
Returns a tuple of the datablock, data path to the property, and array index.

    :type: tuple[ID]
    """

    edit_text: Text | None
    """ 

    :type: Text | None
    """

    def evaluated_depsgraph_get(self) -> Depsgraph:
        """Get the dependency graph for the current scene and view layer, to access to data-blocks with animation and modifiers applied. If any data-blocks have been edited, the dependency graph will be updated. This invalidates all references to evaluated data-blocks from the dependency graph.

        :return: Evaluated dependency graph
        :rtype: Depsgraph
        """

    def copy(self): ...
    def path_resolve(self, path: str | None, coerce: bool | None = True):
        """Returns the property from the path, raise an exception when not found.

        :param path: patch which this property resolves.
        :type path: str | None
        :param coerce: optional argument, when True, the property will be converted into its Python representation.
        :type coerce: bool | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

    def temp_override(
        self,
        window: Window | None,
        area: Area | None,
        region: Region | None,
        **keywords,
    ):
        """Context manager to temporarily override members in the context.Overriding the context can be used to temporarily activate another window / area & region,
        as well as other members such as the active_object or bone.Notes:Overriding the context can be useful to set the context after loading files
        (which would otherwise by None). For example:This example shows how it's possible to add an object to the scene in another window.

                :param window: Window override or None.
                :type window: Window | None
                :param area: Area override or None.
                :type area: Area | None
                :param region: Region override or None.
                :type region: Region | None
                :param keywords: Additional keywords override context members.
                :return: The context manager .
        """

class CopyLocationConstraint(Constraint, bpy_struct):
    """Copy the location of the target"""

    head_tail: float
    """ Target along length of bone: Head is 0, Tail is 1

    :type: float
    """

    invert_x: bool
    """ Invert the X location

    :type: bool
    """

    invert_y: bool
    """ Invert the Y location

    :type: bool
    """

    invert_z: bool
    """ Invert the Z location

    :type: bool
    """

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object | None
    """ Target object

    :type: Object | None
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    use_offset: bool
    """ Add original location into copied location

    :type: bool
    """

    use_x: bool
    """ Copy the target's X location

    :type: bool
    """

    use_y: bool
    """ Copy the target's Y location

    :type: bool
    """

    use_z: bool
    """ Copy the target's Z location

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CopyRotationConstraint(Constraint, bpy_struct):
    """Copy the rotation of the target"""

    euler_order: typing.Literal["AUTO", "XYZ", "XZY", "YXZ", "YZX", "ZXY", "ZYX"]
    """ Explicitly specify the euler rotation order

    :type: typing.Literal['AUTO','XYZ','XZY','YXZ','YZX','ZXY','ZYX']
    """

    invert_x: bool
    """ Invert the X rotation

    :type: bool
    """

    invert_y: bool
    """ Invert the Y rotation

    :type: bool
    """

    invert_z: bool
    """ Invert the Z rotation

    :type: bool
    """

    mix_mode: typing.Literal["REPLACE", "ADD", "BEFORE", "AFTER", "OFFSET"]
    """ Specify how the copied and existing rotations are combined

    :type: typing.Literal['REPLACE','ADD','BEFORE','AFTER','OFFSET']
    """

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object | None
    """ Target object

    :type: Object | None
    """

    use_offset: bool
    """ DEPRECATED: Add original rotation into copied rotation

    :type: bool
    """

    use_x: bool
    """ Copy the target's X rotation

    :type: bool
    """

    use_y: bool
    """ Copy the target's Y rotation

    :type: bool
    """

    use_z: bool
    """ Copy the target's Z rotation

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CopyScaleConstraint(Constraint, bpy_struct):
    """Copy the scale of the target"""

    power: float
    """ Raise the target's scale to the specified power

    :type: float
    """

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object | None
    """ Target object

    :type: Object | None
    """

    use_add: bool
    """ Use addition instead of multiplication to combine scale (2.7 compatibility)

    :type: bool
    """

    use_make_uniform: bool
    """ Redistribute the copied change in volume equally between the three axes of the owner

    :type: bool
    """

    use_offset: bool
    """ Combine original scale with copied scale

    :type: bool
    """

    use_x: bool
    """ Copy the target's X scale

    :type: bool
    """

    use_y: bool
    """ Copy the target's Y scale

    :type: bool
    """

    use_z: bool
    """ Copy the target's Z scale

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CopyTransformsConstraint(Constraint, bpy_struct):
    """Copy all the transforms of the target"""

    head_tail: float
    """ Target along length of bone: Head is 0, Tail is 1

    :type: float
    """

    mix_mode: typing.Literal[
        "REPLACE",
        "BEFORE_FULL",
        "BEFORE",
        "BEFORE_SPLIT",
        "AFTER_FULL",
        "AFTER",
        "AFTER_SPLIT",
    ]
    """ Specify how the copied and existing transformations are combined

    :type: typing.Literal['REPLACE','BEFORE_FULL','BEFORE','BEFORE_SPLIT','AFTER_FULL','AFTER','AFTER_SPLIT']
    """

    remove_target_shear: bool
    """ Remove shear from the target transformation before combining

    :type: bool
    """

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object | None
    """ Target object

    :type: Object | None
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CorrectiveSmoothModifier(Modifier, bpy_struct):
    """Correct distortion caused by deformation"""

    factor: float
    """ Smooth effect factor

    :type: float
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    is_bind: bool
    """ 

    :type: bool
    """

    iterations: int
    """ 

    :type: int
    """

    rest_source: typing.Literal["ORCO", "BIND"]
    """ Select the source of rest positions

    :type: typing.Literal['ORCO','BIND']
    """

    scale: float
    """ Compensate for scale applied by other modifiers

    :type: float
    """

    smooth_type: typing.Literal["SIMPLE", "LENGTH_WEIGHTED"]
    """ Method used for smoothing

    :type: typing.Literal['SIMPLE','LENGTH_WEIGHTED']
    """

    use_only_smooth: bool
    """ Apply smoothing without reconstructing the surface

    :type: bool
    """

    use_pin_boundary: bool
    """ Excludes boundary vertices from being smoothed

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CrossSequence(EffectSequence, Sequence, bpy_struct):
    """Cross Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CryptomatteEntry(bpy_struct):
    encoded_hash: float
    """ 

    :type: float
    """

    name: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Curve(ID, bpy_struct):
    """Curve data-block storing curves, splines and NURBS"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    bevel_depth: float
    """ Radius of the bevel geometry, not including extrusion

    :type: float
    """

    bevel_factor_end: float
    """ Define where along the spline the curve geometry ends (0 for the beginning, 1 for the end)

    :type: float
    """

    bevel_factor_mapping_end: typing.Literal["RESOLUTION", "SEGMENTS", "SPLINE"]
    """ Determine how the geometry end factor is mapped to a spline

    :type: typing.Literal['RESOLUTION','SEGMENTS','SPLINE']
    """

    bevel_factor_mapping_start: typing.Literal["RESOLUTION", "SEGMENTS", "SPLINE"]
    """ Determine how the geometry start factor is mapped to a spline

    :type: typing.Literal['RESOLUTION','SEGMENTS','SPLINE']
    """

    bevel_factor_start: float
    """ Define where along the spline the curve geometry starts (0 for the beginning, 1 for the end)

    :type: float
    """

    bevel_mode: typing.Literal["ROUND", "OBJECT", "PROFILE"]
    """ Determine how to build the curve's bevel geometry

    :type: typing.Literal['ROUND','OBJECT','PROFILE']
    """

    bevel_object: Object | None
    """ The name of the Curve object that defines the bevel shape

    :type: Object | None
    """

    bevel_profile: CurveProfile
    """ The path for the curve's custom profile

    :type: CurveProfile
    """

    bevel_resolution: int
    """ The number of segments in each quarter-circle of the bevel

    :type: int
    """

    cycles: typing.Any
    """ Cycles mesh settings

    :type: typing.Any
    """

    dimensions: typing.Literal["2D", "3D"]
    """ Select 2D or 3D curve type

    :type: typing.Literal['2D','3D']
    """

    eval_time: float
    """ Parametric position along the length of the curve that Objects 'following' it should be at (position is evaluated by dividing by the 'Path Length' value)

    :type: float
    """

    extrude: float
    """ Length of the depth added in the local Z direction along the curve, perpendicular to its normals

    :type: float
    """

    fill_mode: typing.Literal["FULL", "BACK", "FRONT", "HALF"]
    """ Mode of filling curve

    :type: typing.Literal['FULL','BACK','FRONT','HALF']
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    materials: IDMaterials
    """ 

    :type: IDMaterials
    """

    offset: float
    """ Distance to move the curve parallel to its normals

    :type: float
    """

    path_duration: int
    """ The number of frames that are needed to traverse the path, defining the maximum value for the 'Evaluation Time' setting

    :type: int
    """

    render_resolution_u: int
    """ Surface resolution in U direction used while rendering (zero uses preview resolution)

    :type: int
    """

    render_resolution_v: int
    """ Surface resolution in V direction used while rendering (zero uses preview resolution)

    :type: int
    """

    resolution_u: int
    """ Number of computed points in the U direction between every pair of control points

    :type: int
    """

    resolution_v: int
    """ The number of computed points in the V direction between every pair of control points

    :type: int
    """

    shape_keys: Key
    """ 

    :type: Key
    """

    splines: CurveSplines
    """ Collection of splines in this curve data object

    :type: CurveSplines
    """

    taper_object: Object | None
    """ Curve object name that defines the taper (width)

    :type: Object | None
    """

    taper_radius_mode: typing.Literal["OVERRIDE", "MULTIPLY", "ADD"]
    """ Determine how the effective radius of the spline point is computed when a taper object is specified

    :type: typing.Literal['OVERRIDE','MULTIPLY','ADD']
    """

    texspace_location: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    texspace_size: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    twist_mode: typing.Literal["Z_UP", "MINIMUM", "TANGENT"]
    """ The type of tilt calculation for 3D Curves

    :type: typing.Literal['Z_UP','MINIMUM','TANGENT']
    """

    twist_smooth: float
    """ Smoothing iteration for tangents

    :type: float
    """

    use_auto_texspace: bool
    """ Adjust active object's texture space automatically when transforming object

    :type: bool
    """

    use_deform_bounds: bool
    """ Option for curve-deform: Use the mesh bounds to clamp the deformation

    :type: bool
    """

    use_fill_caps: bool
    """ Fill caps for beveled curves

    :type: bool
    """

    use_map_taper: bool
    """ Map effect of the taper object to the beveled part of the curve

    :type: bool
    """

    use_path: bool
    """ Enable the curve to become a translation path

    :type: bool
    """

    use_path_clamp: bool
    """ Clamp the curve path children so they can't travel past the start/end point of the curve

    :type: bool
    """

    use_path_follow: bool
    """ Make curve path children rotate along the path

    :type: bool
    """

    use_radius: bool
    """ Option for paths and curve-deform: apply the curve radius to objects following it and to deformed objects

    :type: bool
    """

    use_stretch: bool
    """ Option for curve-deform: make deformed child stretch along entire path

    :type: bool
    """

    def transform(
        self,
        matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None,
        shape_keys: bool | None = False,
    ):
        """Transform curve by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
        :param shape_keys: Transform Shape Keys
        :type shape_keys: bool | None
        """

    def validate_material_indices(self) -> bool:
        """Validate material indices of splines or letters, return True when the curve has had invalid indices corrected (to default 0)

        :return: Result
        :rtype: bool
        """

    def update_gpu_tag(self):
        """update_gpu_tag"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveMap(bpy_struct):
    """Curve in a curve mapping"""

    points: CurveMapPoints
    """ 

    :type: CurveMapPoints
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveMapPoint(bpy_struct):
    """Point of a curve used for a curve mapping"""

    handle_type: typing.Literal["AUTO", "AUTO_CLAMPED", "VECTOR"]
    """ Curve interpolation at this point: Bezier or vector

    :type: typing.Literal['AUTO','AUTO_CLAMPED','VECTOR']
    """

    location: mathutils.Vector
    """ X/Y coordinates of the curve point

    :type: mathutils.Vector
    """

    select: bool
    """ Selection state of the curve point

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveMapping(bpy_struct):
    """Curve mapping to map color, vector and scalar values to other values using a user defined curve"""

    black_level: mathutils.Color
    """ For RGB curves, the color that black is mapped to

    :type: mathutils.Color
    """

    clip_max_x: float
    """ 

    :type: float
    """

    clip_max_y: float
    """ 

    :type: float
    """

    clip_min_x: float
    """ 

    :type: float
    """

    clip_min_y: float
    """ 

    :type: float
    """

    curves: bpy_prop_collection[CurveMap]
    """ 

    :type: bpy_prop_collection[CurveMap]
    """

    extend: typing.Literal["HORIZONTAL", "EXTRAPOLATED"]
    """ Extrapolate the curve or extend it horizontally

    :type: typing.Literal['HORIZONTAL','EXTRAPOLATED']
    """

    tone: typing.Literal["STANDARD", "FILMLIKE"]
    """ Tone of the curve

    :type: typing.Literal['STANDARD','FILMLIKE']
    """

    use_clip: bool
    """ Force the curve view to fit a defined boundary

    :type: bool
    """

    white_level: mathutils.Color
    """ For RGB curves, the color that white is mapped to

    :type: mathutils.Color
    """

    def update(self):
        """Update curve mapping after making changes"""

    def reset_view(self):
        """Reset the curve mapping grid to its clipping size"""

    def initialize(self):
        """Initialize curve"""

    def evaluate(self, curve: CurveMap, position: float | None) -> float:
        """Evaluate curve at given location

        :param curve: curve, Curve to evaluate
        :type curve: CurveMap
        :param position: Position, Position to evaluate curve at
        :type position: float | None
        :return: Value, Value of curve at given location
        :rtype: float
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveModifier(Modifier, bpy_struct):
    """Curve deformation modifier"""

    deform_axis: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ The axis that the curve deforms along

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    object: Object | None
    """ Curve object to deform with

    :type: Object | None
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurvePaintSettings(bpy_struct):
    corner_angle: float
    """ Angles above this are considered corners

    :type: float
    """

    curve_type: typing.Literal["POLY", "BEZIER"]
    """ Type of curve to use for new strokes

    :type: typing.Literal['POLY','BEZIER']
    """

    depth_mode: typing.Literal["CURSOR", "SURFACE"]
    """ Method of projecting depth

    :type: typing.Literal['CURSOR','SURFACE']
    """

    error_threshold: int
    """ Allow deviation for a smoother, less precise line

    :type: int
    """

    fit_method: bpy._typing.rna_enums.CurveFitMethodItems
    """ Curve fitting method

    :type: bpy._typing.rna_enums.CurveFitMethodItems
    """

    radius_max: float
    """ Radius to use when the maximum pressure is applied (or when a tablet isn't used)

    :type: float
    """

    radius_min: float
    """ Minimum radius when the minimum pressure is applied (also the minimum when tapering)

    :type: float
    """

    radius_taper_end: float
    """ Taper factor for the radius of each point along the curve

    :type: float
    """

    radius_taper_start: float
    """ Taper factor for the radius of each point along the curve

    :type: float
    """

    surface_offset: float
    """ Offset the stroke from the surface

    :type: float
    """

    surface_plane: typing.Literal["NORMAL_VIEW", "NORMAL_SURFACE", "VIEW"]
    """ Plane for projected stroke

    :type: typing.Literal['NORMAL_VIEW','NORMAL_SURFACE','VIEW']
    """

    use_corners_detect: bool
    """ Detect corners and use non-aligned handles

    :type: bool
    """

    use_offset_absolute: bool
    """ Apply a fixed offset (don't scale by the radius)

    :type: bool
    """

    use_pressure_radius: bool
    """ Map tablet pressure to curve radius

    :type: bool
    """

    use_stroke_endpoints: bool
    """ Use the start of the stroke for the depth

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurvePoint(bpy_struct):
    """Curve control point"""

    index: int
    """ Index of this point

    :type: int
    """

    position: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    radius: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveProfile(bpy_struct):
    """Profile Path editor used to build a profile path"""

    points: CurveProfilePoints
    """ Profile control points

    :type: CurveProfilePoints
    """

    preset: typing.Literal["LINE", "SUPPORTS", "CORNICE", "CROWN", "STEPS"]
    """ 

    :type: typing.Literal['LINE','SUPPORTS','CORNICE','CROWN','STEPS']
    """

    segments: bpy_prop_collection[CurveProfilePoint]
    """ Segments sampled from control points

    :type: bpy_prop_collection[CurveProfilePoint]
    """

    use_clip: bool
    """ Force the path view to fit a defined boundary

    :type: bool
    """

    use_sample_even_lengths: bool
    """ Sample edges with even lengths

    :type: bool
    """

    use_sample_straight_edges: bool
    """ Sample edges with vector handles

    :type: bool
    """

    def update(self):
        """Refresh internal data, remove doubles and clip points"""

    def reset_view(self):
        """Reset the curve profile grid to its clipping size"""

    def initialize(self, totsegments: int):
        """Set the number of display segments and fill tables

        :param totsegments: The number of segment values to initialize the segments table with
        :type totsegments: int
        """

    def evaluate(self, length_portion: float | None) -> mathutils.Vector:
        """Evaluate the at the given portion of the path length

        :param length_portion: Length Portion, Portion of the path length to travel before evaluation
        :type length_portion: float | None
        :return: Location, The location at the given portion of the profile
        :rtype: mathutils.Vector
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveProfilePoint(bpy_struct):
    """Point of a path used to define a profile"""

    handle_type_1: typing.Literal["AUTO", "VECTOR", "FREE", "ALIGN"]
    """ Path interpolation at this point

    :type: typing.Literal['AUTO','VECTOR','FREE','ALIGN']
    """

    handle_type_2: typing.Literal["AUTO", "VECTOR", "FREE", "ALIGN"]
    """ Path interpolation at this point

    :type: typing.Literal['AUTO','VECTOR','FREE','ALIGN']
    """

    location: mathutils.Vector
    """ X/Y coordinates of the path point

    :type: mathutils.Vector
    """

    select: bool
    """ Selection state of the path point

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurveSlice(bpy_struct):
    """A single curve from a curves data-block"""

    first_point_index: int
    """ The index of this curve's first control point

    :type: int
    """

    index: int
    """ Index of this curve

    :type: int
    """

    points: bpy_prop_collection[CurvePoint]
    """ Control points of the curve

    :type: bpy_prop_collection[CurvePoint]
    """

    points_length: int
    """ Number of control points in the curve

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Curves(ID, bpy_struct):
    """Hair data-block for hair curves"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    attributes: AttributeGroup
    """ Geometry attributes

    :type: AttributeGroup
    """

    color_attributes: AttributeGroup
    """ Geometry color attributes

    :type: AttributeGroup
    """

    curve_offset_data: bpy_prop_collection[IntAttributeValue]
    """ 

    :type: bpy_prop_collection[IntAttributeValue]
    """

    curves: bpy_prop_collection[CurveSlice]
    """ All curves in the data-block

    :type: bpy_prop_collection[CurveSlice]
    """

    materials: IDMaterials
    """ 

    :type: IDMaterials
    """

    normals: bpy_prop_collection[FloatVectorValueReadOnly]
    """ The curve normal value at each of the curve's control points

    :type: bpy_prop_collection[FloatVectorValueReadOnly]
    """

    points: bpy_prop_collection[CurvePoint]
    """ Control points of all curves

    :type: bpy_prop_collection[CurvePoint]
    """

    position_data: bpy_prop_collection[FloatVectorAttributeValue]
    """ 

    :type: bpy_prop_collection[FloatVectorAttributeValue]
    """

    selection_domain: bpy._typing.rna_enums.AttributeCurvesDomainItems
    """ 

    :type: bpy._typing.rna_enums.AttributeCurvesDomainItems
    """

    surface: Object | None
    """ Mesh object that the curves can be attached to

    :type: Object | None
    """

    surface_uv_map: str
    """ The name of the attribute on the surface mesh used to define the attachment of each curve

    :type: str
    """

    use_mirror_x: bool
    """ Enable symmetry in the X axis

    :type: bool
    """

    use_mirror_y: bool
    """ Enable symmetry in the Y axis

    :type: bool
    """

    use_mirror_z: bool
    """ Enable symmetry in the Z axis

    :type: bool
    """

    use_sculpt_collision: bool
    """ Enable collision with the surface while sculpting

    :type: bool
    """

    def add_curves(self, sizes: collections.abc.Iterable[int] | None):
        """add_curves

        :param sizes: Sizes, The number of points in each curve
        :type sizes: collections.abc.Iterable[int] | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurvesModifier(SequenceModifier, bpy_struct):
    """RGB curves modifier for sequence strip"""

    curve_mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class CurvesSculpt(Paint, bpy_struct):
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DATA_UL_bone_collections(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        armature,
        bcoll,
        _icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param armature:
        :param bcoll:
        :param _icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DampedTrackConstraint(Constraint, bpy_struct):
    """Point toward target by taking the shortest rotation path"""

    head_tail: float
    """ Target along length of bone: Head is 0, Tail is 1

    :type: float
    """

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object | None
    """ Target object

    :type: Object | None
    """

    track_axis: typing.Literal[
        "TRACK_X",
        "TRACK_Y",
        "TRACK_Z",
        "TRACK_NEGATIVE_X",
        "TRACK_NEGATIVE_Y",
        "TRACK_NEGATIVE_Z",
    ]
    """ Axis that points to the target object

    :type: typing.Literal['TRACK_X','TRACK_Y','TRACK_Z','TRACK_NEGATIVE_X','TRACK_NEGATIVE_Y','TRACK_NEGATIVE_Z']
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DashGpencilModifierData(GpencilModifier, bpy_struct):
    """Create dot-dash effect for strokes"""

    dash_offset: int
    """ Offset into each stroke before the beginning of the dashed segment generation

    :type: int
    """

    invert_layer_pass: bool
    """ Inverse filter

    :type: bool
    """

    invert_layers: bool
    """ Inverse filter

    :type: bool
    """

    invert_material_pass: bool
    """ Inverse filter

    :type: bool
    """

    invert_materials: bool
    """ Inverse filter

    :type: bool
    """

    layer: str
    """ Layer name

    :type: str
    """

    layer_pass: int
    """ Layer pass index

    :type: int
    """

    material: Material | None
    """ Material used for filtering effect

    :type: Material | None
    """

    pass_index: int
    """ Pass index

    :type: int
    """

    segment_active_index: int
    """ Active index in the segment list

    :type: int
    """

    segments: bpy_prop_collection[DashGpencilModifierSegment]
    """ 

    :type: bpy_prop_collection[DashGpencilModifierSegment]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DashGpencilModifierSegment(bpy_struct):
    """Configuration for a single dash segment"""

    dash: int
    """ The number of consecutive points from the original stroke to include in this segment

    :type: int
    """

    gap: int
    """ The number of points skipped after this segment

    :type: int
    """

    material_index: int
    """ Use this index on generated segment. -1 means using the existing material

    :type: int
    """

    name: str
    """ Name of the dash segment

    :type: str
    """

    opacity: float
    """ The factor to apply to the original point's opacity for the new points

    :type: float
    """

    radius: float
    """ The factor to apply to the original point's radius for the new points

    :type: float
    """

    use_cyclic: bool
    """ Enable cyclic on individual stroke dashes

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DataTransferModifier(Modifier, bpy_struct):
    """Modifier transferring some data from a source mesh"""

    data_types_edges: set[
        typing.Literal[
            "SHARP_EDGE", "SEAM", "CREASE", "BEVEL_WEIGHT_EDGE", "FREESTYLE_EDGE"
        ]
    ]
    """ Which edge data layers to transfer

    :type: set[typing.Literal['SHARP_EDGE','SEAM','CREASE','BEVEL_WEIGHT_EDGE','FREESTYLE_EDGE']]
    """

    data_types_loops: set[typing.Literal["CUSTOM_NORMAL", "COLOR_CORNER", "UV"]]
    """ Which face corner data layers to transfer

    :type: set[typing.Literal['CUSTOM_NORMAL','COLOR_CORNER','UV']]
    """

    data_types_polys: set[typing.Literal["SMOOTH", "FREESTYLE_FACE"]]
    """ Which face data layers to transfer

    :type: set[typing.Literal['SMOOTH','FREESTYLE_FACE']]
    """

    data_types_verts: set[
        typing.Literal["VGROUP_WEIGHTS", "BEVEL_WEIGHT_VERT", "COLOR_VERTEX"]
    ]
    """ Which vertex data layers to transfer

    :type: set[typing.Literal['VGROUP_WEIGHTS','BEVEL_WEIGHT_VERT','COLOR_VERTEX']]
    """

    edge_mapping: bpy._typing.rna_enums.DtMethodEdgeItems
    """ Method used to map source edges to destination ones

    :type: bpy._typing.rna_enums.DtMethodEdgeItems
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    islands_precision: float
    """ Factor controlling precision of islands handling (typically, 0.1 should be enough, higher values can make things really slow)

    :type: float
    """

    layers_uv_select_dst: bpy._typing.rna_enums.DtLayersSelectDstItems
    """ How to match source and destination layers

    :type: bpy._typing.rna_enums.DtLayersSelectDstItems
    """

    layers_uv_select_src: bpy._typing.rna_enums.DtLayersSelectSrcItems
    """ Which layers to transfer, in case of multi-layers types

    :type: bpy._typing.rna_enums.DtLayersSelectSrcItems
    """

    layers_vcol_loop_select_dst: bpy._typing.rna_enums.DtLayersSelectDstItems
    """ How to match source and destination layers

    :type: bpy._typing.rna_enums.DtLayersSelectDstItems
    """

    layers_vcol_loop_select_src: bpy._typing.rna_enums.DtLayersSelectSrcItems
    """ Which layers to transfer, in case of multi-layers types

    :type: bpy._typing.rna_enums.DtLayersSelectSrcItems
    """

    layers_vcol_vert_select_dst: bpy._typing.rna_enums.DtLayersSelectDstItems
    """ How to match source and destination layers

    :type: bpy._typing.rna_enums.DtLayersSelectDstItems
    """

    layers_vcol_vert_select_src: bpy._typing.rna_enums.DtLayersSelectSrcItems
    """ Which layers to transfer, in case of multi-layers types

    :type: bpy._typing.rna_enums.DtLayersSelectSrcItems
    """

    layers_vgroup_select_dst: bpy._typing.rna_enums.DtLayersSelectDstItems
    """ How to match source and destination layers

    :type: bpy._typing.rna_enums.DtLayersSelectDstItems
    """

    layers_vgroup_select_src: bpy._typing.rna_enums.DtLayersSelectSrcItems
    """ Which layers to transfer, in case of multi-layers types

    :type: bpy._typing.rna_enums.DtLayersSelectSrcItems
    """

    loop_mapping: bpy._typing.rna_enums.DtMethodLoopItems
    """ Method used to map source faces' corners to destination ones

    :type: bpy._typing.rna_enums.DtMethodLoopItems
    """

    max_distance: float
    """ Maximum allowed distance between source and destination element, for non-topology mappings

    :type: float
    """

    mix_factor: float
    """ Factor to use when applying data to destination (exact behavior depends on mix mode, multiplied with weights from vertex group when defined)

    :type: float
    """

    mix_mode: bpy._typing.rna_enums.DtMixModeItems
    """ How to affect destination elements with source values

    :type: bpy._typing.rna_enums.DtMixModeItems
    """

    object: Object | None
    """ Object to transfer data from

    :type: Object | None
    """

    poly_mapping: bpy._typing.rna_enums.DtMethodPolyItems
    """ Method used to map source faces to destination ones

    :type: bpy._typing.rna_enums.DtMethodPolyItems
    """

    ray_radius: float
    """ 'Width' of rays (especially useful when raycasting against vertices or edges)

    :type: float
    """

    use_edge_data: bool
    """ Enable edge data transfer

    :type: bool
    """

    use_loop_data: bool
    """ Enable face corner data transfer

    :type: bool
    """

    use_max_distance: bool
    """ Source elements must be closer than given distance from destination one

    :type: bool
    """

    use_object_transform: bool
    """ Evaluate source and destination meshes in global space

    :type: bool
    """

    use_poly_data: bool
    """ Enable face data transfer

    :type: bool
    """

    use_vert_data: bool
    """ Enable vertex data transfer

    :type: bool
    """

    vert_mapping: bpy._typing.rna_enums.DtMethodVertexItems
    """ Method used to map source vertices to destination ones

    :type: bpy._typing.rna_enums.DtMethodVertexItems
    """

    vertex_group: str
    """ Vertex group name for selecting the affected areas

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DecimateModifier(Modifier, bpy_struct):
    """Decimation modifier"""

    angle_limit: float
    """ Only dissolve angles below this (planar only)

    :type: float
    """

    decimate_type: typing.Literal["COLLAPSE", "UNSUBDIV", "DISSOLVE"]
    """ 

    :type: typing.Literal['COLLAPSE','UNSUBDIV','DISSOLVE']
    """

    delimit: set[bpy._typing.rna_enums.MeshDelimitModeItems]
    """ Limit merging geometry

    :type: set[bpy._typing.rna_enums.MeshDelimitModeItems]
    """

    face_count: int
    """ The current number of faces in the decimated mesh

    :type: int
    """

    invert_vertex_group: bool
    """ Invert vertex group influence (collapse only)

    :type: bool
    """

    iterations: int
    """ Number of times reduce the geometry (unsubdivide only)

    :type: int
    """

    ratio: float
    """ Ratio of triangles to reduce to (collapse only)

    :type: float
    """

    symmetry_axis: bpy._typing.rna_enums.AxisXyzItems
    """ Axis of symmetry

    :type: bpy._typing.rna_enums.AxisXyzItems
    """

    use_collapse_triangulate: bool
    """ Keep triangulated faces resulting from decimation (collapse only)

    :type: bool
    """

    use_dissolve_boundaries: bool
    """ Dissolve all vertices in between face boundaries (planar only)

    :type: bool
    """

    use_symmetry: bool
    """ Maintain symmetry on an axis

    :type: bool
    """

    vertex_group: str
    """ Vertex group name (collapse only)

    :type: str
    """

    vertex_group_factor: float
    """ Vertex group strength

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Depsgraph(bpy_struct):
    ids: bpy_prop_collection[ID]
    """ All evaluated data-blocks

    :type: bpy_prop_collection[ID]
    """

    mode: typing.Literal["VIEWPORT", "RENDER"]
    """ Evaluation mode

    :type: typing.Literal['VIEWPORT','RENDER']
    """

    object_instances: bpy_prop_collection[DepsgraphObjectInstance]
    """ All object instances to display or render (Warning: Only use this as an iterator, never as a sequence, and do not keep any references to its items)

    :type: bpy_prop_collection[DepsgraphObjectInstance]
    """

    objects: bpy_prop_collection[Object]
    """ Evaluated objects in the dependency graph

    :type: bpy_prop_collection[Object]
    """

    scene: Scene
    """ Original scene dependency graph is built for

    :type: Scene
    """

    scene_eval: Scene
    """ Scene at its evaluated state

    :type: Scene
    """

    updates: bpy_prop_collection[DepsgraphUpdate]
    """ Updates to data-blocks

    :type: bpy_prop_collection[DepsgraphUpdate]
    """

    view_layer: ViewLayer
    """ Original view layer dependency graph is built for

    :type: ViewLayer
    """

    view_layer_eval: ViewLayer
    """ View layer at its evaluated state

    :type: ViewLayer
    """

    def debug_relations_graphviz(self, filepath: str):
        """debug_relations_graphviz

        :param filepath: File Name, Output path for the graphviz debug file
        :type filepath: str
        """

    def debug_stats_gnuplot(self, filepath: str, output_filepath: str):
        """debug_stats_gnuplot

        :param filepath: File Name, Output path for the gnuplot debug file
        :type filepath: str
        :param output_filepath: Output File Name, File name where gnuplot script will save the result
        :type output_filepath: str
        """

    def debug_tag_update(self):
        """debug_tag_update"""

    def debug_stats(self) -> str:
        """Report the number of elements in the Dependency Graph

        :return: result
        :rtype: str
        """

    def update(self):
        """Re-evaluate any modified data-blocks, for example for animation or modifiers. This invalidates all references to evaluated data-blocks from this dependency graph."""

    def id_eval_get[_GenericType1](self, id: _GenericType1 | None) -> _GenericType1:
        """id_eval_get

        :param id: Original ID to get evaluated complementary part for
        :type id: _GenericType1 | None
        :return: Evaluated ID for the given original one
        :rtype: _GenericType1
        """

    def id_type_updated(
        self, id_type: bpy._typing.rna_enums.IdTypeItems | None
    ) -> bool:
        """id_type_updated

        :param id_type: ID Type
        :type id_type: bpy._typing.rna_enums.IdTypeItems | None
        :return: Updated, True if any datablock with this type was added, updated or removed
        :rtype: bool
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DepsgraphObjectInstance(bpy_struct):
    """Extended information about dependency graph object iterator (Warning: All data here is 'evaluated' one, not original .blend IDs)"""

    instance_object: Object
    """ Evaluated object which is being instanced by this iterator

    :type: Object
    """

    is_instance: bool
    """ Denotes if the object is generated by another object

    :type: bool
    """

    matrix_world: mathutils.Matrix
    """ Generated transform matrix in world space

    :type: mathutils.Matrix
    """

    object: Object
    """ Evaluated object the iterator points to

    :type: Object
    """

    orco: mathutils.Vector
    """ Generated coordinates in parent object space

    :type: mathutils.Vector
    """

    parent: Object
    """ If the object is an instance, the parent object that generated it

    :type: Object
    """

    particle_system: ParticleSystem
    """ Evaluated particle system that this object was instanced from

    :type: ParticleSystem
    """

    persistent_id: bpy_prop_array[int]
    """ Persistent identifier for inter-frame matching of objects with motion blur

    :type: bpy_prop_array[int]
    """

    random_id: int
    """ Random id for this instance, typically for randomized shading

    :type: int
    """

    show_particles: bool
    """ Particles part of the object should be visible in the render

    :type: bool
    """

    show_self: bool
    """ The object geometry itself should be visible in the render

    :type: bool
    """

    uv: bpy_prop_array[float]
    """ UV coordinates in parent object space

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DepsgraphUpdate(bpy_struct):
    """Information about ID that was updated"""

    id: ID
    """ Updated data-block

    :type: ID
    """

    is_updated_geometry: bool
    """ Object geometry is updated

    :type: bool
    """

    is_updated_shading: bool
    """ Object shading is updated

    :type: bool
    """

    is_updated_transform: bool
    """ Object transformation is updated

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DisplaceModifier(Modifier, bpy_struct):
    """Displacement modifier"""

    direction: typing.Literal["X", "Y", "Z", "NORMAL", "CUSTOM_NORMAL", "RGB_TO_XYZ"]
    """ 

    :type: typing.Literal['X','Y','Z','NORMAL','CUSTOM_NORMAL','RGB_TO_XYZ']
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    mid_level: float
    """ Material value that gives no displacement

    :type: float
    """

    space: typing.Literal["LOCAL", "GLOBAL"]
    """ 

    :type: typing.Literal['LOCAL','GLOBAL']
    """

    strength: float
    """ Amount to displace geometry

    :type: float
    """

    texture: Texture | None
    """ 

    :type: Texture | None
    """

    texture_coords: typing.Literal["LOCAL", "GLOBAL", "OBJECT", "UV"]
    """ 

    :type: typing.Literal['LOCAL','GLOBAL','OBJECT','UV']
    """

    texture_coords_bone: str
    """ Bone to set the texture coordinates

    :type: str
    """

    texture_coords_object: Object | None
    """ Object to set the texture coordinates

    :type: Object | None
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DisplaySafeAreas(bpy_struct):
    """Safe areas used in 3D view and the sequencer"""

    action: mathutils.Vector
    """ Safe area for general elements

    :type: mathutils.Vector
    """

    action_center: mathutils.Vector
    """ Safe area for general elements in a different aspect ratio

    :type: mathutils.Vector
    """

    title: mathutils.Vector
    """ Safe area for text and graphics

    :type: mathutils.Vector
    """

    title_center: mathutils.Vector
    """ Safe area for text and graphics in a different aspect ratio

    :type: mathutils.Vector
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DistortedNoiseTexture(Texture, ID, bpy_struct):
    """Procedural distorted noise texture"""

    distortion: float
    """ Amount of distortion

    :type: float
    """

    nabla: float
    """ Size of derivative offset used for calculating normal

    :type: float
    """

    noise_basis: typing.Literal[
        "BLENDER_ORIGINAL",
        "ORIGINAL_PERLIN",
        "IMPROVED_PERLIN",
        "VORONOI_F1",
        "VORONOI_F2",
        "VORONOI_F3",
        "VORONOI_F4",
        "VORONOI_F2_F1",
        "VORONOI_CRACKLE",
        "CELL_NOISE",
    ]
    """ Noise basis used for turbulence

    :type: typing.Literal['BLENDER_ORIGINAL','ORIGINAL_PERLIN','IMPROVED_PERLIN','VORONOI_F1','VORONOI_F2','VORONOI_F3','VORONOI_F4','VORONOI_F2_F1','VORONOI_CRACKLE','CELL_NOISE']
    """

    noise_distortion: typing.Literal[
        "BLENDER_ORIGINAL",
        "ORIGINAL_PERLIN",
        "IMPROVED_PERLIN",
        "VORONOI_F1",
        "VORONOI_F2",
        "VORONOI_F3",
        "VORONOI_F4",
        "VORONOI_F2_F1",
        "VORONOI_CRACKLE",
        "CELL_NOISE",
    ]
    """ Noise basis for the distortion

    :type: typing.Literal['BLENDER_ORIGINAL','ORIGINAL_PERLIN','IMPROVED_PERLIN','VORONOI_F1','VORONOI_F2','VORONOI_F3','VORONOI_F4','VORONOI_F2_F1','VORONOI_CRACKLE','CELL_NOISE']
    """

    noise_scale: float
    """ Scaling for noise input

    :type: float
    """

    users_material: typing.Any
    """ Materials that use this texture(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture(readonly)"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DopeSheet(bpy_struct):
    """Settings for filtering the channels shown in animation editors"""

    filter_collection: Collection | None
    """ Collection that included object should be a member of

    :type: Collection | None
    """

    filter_fcurve_name: str
    """ F-Curve live filtering string

    :type: str
    """

    filter_text: str
    """ Live filtering string

    :type: str
    """

    show_armatures: bool
    """ Include visualization of armature related animation data

    :type: bool
    """

    show_cache_files: bool
    """ Include visualization of cache file related animation data

    :type: bool
    """

    show_cameras: bool
    """ Include visualization of camera related animation data

    :type: bool
    """

    show_curves: bool
    """ Include visualization of curve related animation data

    :type: bool
    """

    show_datablock_filters: bool
    """ Show options for whether channels related to certain types of data are included

    :type: bool
    """

    show_expanded_summary: bool
    """ Collapse summary when shown, so all other channels get hidden (Dope Sheet editors only)

    :type: bool
    """

    show_gpencil: bool
    """ Include visualization of Grease Pencil related animation data and frames

    :type: bool
    """

    show_hair_curves: bool
    """ Include visualization of hair related animation data

    :type: bool
    """

    show_hidden: bool
    """ Include channels from objects/bone that are not visible

    :type: bool
    """

    show_lattices: bool
    """ Include visualization of lattice related animation data

    :type: bool
    """

    show_lights: bool
    """ Include visualization of light related animation data

    :type: bool
    """

    show_linestyles: bool
    """ Include visualization of Line Style related Animation data

    :type: bool
    """

    show_materials: bool
    """ Include visualization of material related animation data

    :type: bool
    """

    show_meshes: bool
    """ Include visualization of mesh related animation data

    :type: bool
    """

    show_metaballs: bool
    """ Include visualization of metaball related animation data

    :type: bool
    """

    show_missing_nla: bool
    """ Include animation data-blocks with no NLA data (NLA editor only)

    :type: bool
    """

    show_modifiers: bool
    """ Include visualization of animation data related to data-blocks linked to modifiers

    :type: bool
    """

    show_movieclips: bool
    """ Include visualization of movie clip related animation data

    :type: bool
    """

    show_nodes: bool
    """ Include visualization of node related animation data

    :type: bool
    """

    show_only_errors: bool
    """ Only include F-Curves and drivers that are disabled or have errors

    :type: bool
    """

    show_only_selected: bool
    """ Only include channels relating to selected objects and data

    :type: bool
    """

    show_particles: bool
    """ Include visualization of particle related animation data

    :type: bool
    """

    show_pointclouds: bool
    """ Include visualization of point cloud related animation data

    :type: bool
    """

    show_scenes: bool
    """ Include visualization of scene related animation data

    :type: bool
    """

    show_shapekeys: bool
    """ Include visualization of shape key related animation data

    :type: bool
    """

    show_speakers: bool
    """ Include visualization of speaker related animation data

    :type: bool
    """

    show_summary: bool
    """ Display an additional 'summary' line (Dope Sheet editors only)

    :type: bool
    """

    show_textures: bool
    """ Include visualization of texture related animation data

    :type: bool
    """

    show_transforms: bool
    """ Include visualization of object-level animation data (mostly transforms)

    :type: bool
    """

    show_volumes: bool
    """ Include visualization of volume related animation data

    :type: bool
    """

    show_worlds: bool
    """ Include visualization of world related animation data

    :type: bool
    """

    source: ID
    """ ID-Block representing source data, usually ID_SCE (i.e. Scene)

    :type: ID
    """

    use_datablock_sort: bool
    """ Alphabetically sorts data-blocks - mainly objects in the scene (disable to increase viewport speed)

    :type: bool
    """

    use_filter_invert: bool
    """ Invert filter search

    :type: bool
    """

    use_multi_word_filter: bool
    """ Perform fuzzy/multi-word matching.
Warning: May be slow

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Driver(bpy_struct):
    """Driver for the value of a setting based on an external value"""

    expression: str
    """ Expression to use for Scripted Expression

    :type: str
    """

    is_simple_expression: bool
    """ The scripted expression can be evaluated without using the full Python interpreter

    :type: bool
    """

    is_valid: bool
    """ Driver could not be evaluated in past, so should be skipped

    :type: bool
    """

    type: typing.Literal["AVERAGE", "SUM", "SCRIPTED", "MIN", "MAX"]
    """ Driver type

    :type: typing.Literal['AVERAGE','SUM','SCRIPTED','MIN','MAX']
    """

    use_self: bool
    """ Include a 'self' variable in the name-space, so drivers can easily reference the data being modified (object, bone, etc...)

    :type: bool
    """

    variables: ChannelDriverVariables
    """ Properties acting as inputs for this driver

    :type: ChannelDriverVariables
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DriverTarget(bpy_struct):
    """Source of input values for driver variables"""

    bone_target: str
    """ Name of PoseBone to use as target

    :type: str
    """

    context_property: typing.Literal["ACTIVE_SCENE", "ACTIVE_VIEW_LAYER"]
    """ Type of a context-dependent data-block to access property from

    :type: typing.Literal['ACTIVE_SCENE','ACTIVE_VIEW_LAYER']
    """

    data_path: str
    """ RNA Path (from ID-block) to property used

    :type: str
    """

    id: ID | None
    """ ID-block that the specific property used can be found from (id_type property must be set first)

    :type: ID | None
    """

    id_type: bpy._typing.rna_enums.IdTypeItems
    """ Type of ID-block that can be used

    :type: bpy._typing.rna_enums.IdTypeItems
    """

    rotation_mode: bpy._typing.rna_enums.DriverTargetRotationModeItems
    """ Mode for calculating rotation channel values

    :type: bpy._typing.rna_enums.DriverTargetRotationModeItems
    """

    transform_space: typing.Literal["WORLD_SPACE", "TRANSFORM_SPACE", "LOCAL_SPACE"]
    """ Space in which transforms are used

    :type: typing.Literal['WORLD_SPACE','TRANSFORM_SPACE','LOCAL_SPACE']
    """

    transform_type: typing.Literal[
        "LOC_X",
        "LOC_Y",
        "LOC_Z",
        "ROT_X",
        "ROT_Y",
        "ROT_Z",
        "ROT_W",
        "SCALE_X",
        "SCALE_Y",
        "SCALE_Z",
        "SCALE_AVG",
    ]
    """ Driver variable type

    :type: typing.Literal['LOC_X','LOC_Y','LOC_Z','ROT_X','ROT_Y','ROT_Z','ROT_W','SCALE_X','SCALE_Y','SCALE_Z','SCALE_AVG']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DriverVariable(bpy_struct):
    """Variable from some source/target for driver relationship"""

    is_name_valid: bool
    """ Is this a valid name for a driver variable

    :type: bool
    """

    name: str
    """ Name to use in scripted expressions/functions (no spaces or dots are allowed, and must start with a letter)

    :type: str
    """

    targets: bpy_prop_collection[DriverTarget]
    """ Sources of input data for evaluating this variable

    :type: bpy_prop_collection[DriverTarget]
    """

    type: typing.Literal[
        "SINGLE_PROP", "TRANSFORMS", "ROTATION_DIFF", "LOC_DIFF", "CONTEXT_PROP"
    ]
    """ Driver variable type

    :type: typing.Literal['SINGLE_PROP','TRANSFORMS','ROTATION_DIFF','LOC_DIFF','CONTEXT_PROP']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DynamicPaintBrushSettings(bpy_struct):
    """Brush settings"""

    invert_proximity: bool
    """ Proximity falloff is applied inside the volume

    :type: bool
    """

    paint_alpha: float
    """ Paint alpha

    :type: float
    """

    paint_color: mathutils.Color
    """ Color of the paint

    :type: mathutils.Color
    """

    paint_distance: float
    """ Maximum distance from brush to mesh surface to affect paint

    :type: float
    """

    paint_ramp: ColorRamp
    """ Color ramp used to define proximity falloff

    :type: ColorRamp
    """

    paint_source: typing.Literal[
        "PARTICLE_SYSTEM", "POINT", "DISTANCE", "VOLUME_DISTANCE", "VOLUME"
    ]
    """ 

    :type: typing.Literal['PARTICLE_SYSTEM','POINT','DISTANCE','VOLUME_DISTANCE','VOLUME']
    """

    paint_wetness: float
    """ Paint wetness, visible in wetmap (some effects only affect wet paint)

    :type: float
    """

    particle_system: ParticleSystem | None
    """ The particle system to paint with

    :type: ParticleSystem | None
    """

    proximity_falloff: typing.Literal["SMOOTH", "CONSTANT", "RAMP"]
    """ Proximity falloff type

    :type: typing.Literal['SMOOTH','CONSTANT','RAMP']
    """

    ray_direction: typing.Literal["CANVAS", "BRUSH", "Z_AXIS"]
    """ Ray direction to use for projection (if brush object is located in that direction it's painted)

    :type: typing.Literal['CANVAS','BRUSH','Z_AXIS']
    """

    smooth_radius: float
    """ Smooth falloff added after solid radius

    :type: float
    """

    smudge_strength: float
    """ Smudge effect strength

    :type: float
    """

    solid_radius: float
    """ Radius that will be painted solid

    :type: float
    """

    use_absolute_alpha: bool
    """ Only increase alpha value if paint alpha is higher than existing

    :type: bool
    """

    use_negative_volume: bool
    """ Negate influence inside the volume

    :type: bool
    """

    use_paint_erase: bool
    """ Erase / remove paint instead of adding it

    :type: bool
    """

    use_particle_radius: bool
    """ Use radius from particle settings

    :type: bool
    """

    use_proximity_project: bool
    """ Brush is projected to canvas from defined direction within brush proximity

    :type: bool
    """

    use_proximity_ramp_alpha: bool
    """ Only read color ramp alpha

    :type: bool
    """

    use_smudge: bool
    """ Make this brush to smudge existing paint as it moves

    :type: bool
    """

    use_velocity_alpha: bool
    """ Multiply brush influence by velocity color ramp alpha

    :type: bool
    """

    use_velocity_color: bool
    """ Replace brush color by velocity color ramp

    :type: bool
    """

    use_velocity_depth: bool
    """ Multiply brush intersection depth (displace, waves) by velocity ramp alpha

    :type: bool
    """

    velocity_max: float
    """ Velocity considered as maximum influence (Blender units per frame)

    :type: float
    """

    velocity_ramp: ColorRamp
    """ Color ramp used to define brush velocity effect

    :type: ColorRamp
    """

    wave_clamp: float
    """ Maximum level of surface intersection used to influence waves (use 0.0 to disable)

    :type: float
    """

    wave_factor: float
    """ Multiplier for wave influence of this brush

    :type: float
    """

    wave_type: typing.Literal["CHANGE", "DEPTH", "FORCE", "REFLECT"]
    """ 

    :type: typing.Literal['CHANGE','DEPTH','FORCE','REFLECT']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DynamicPaintCanvasSettings(bpy_struct):
    """Dynamic Paint canvas settings"""

    canvas_surfaces: DynamicPaintSurfaces
    """ Paint surface list

    :type: DynamicPaintSurfaces
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DynamicPaintModifier(Modifier, bpy_struct):
    """Dynamic Paint modifier"""

    brush_settings: DynamicPaintBrushSettings
    """ 

    :type: DynamicPaintBrushSettings
    """

    canvas_settings: DynamicPaintCanvasSettings
    """ 

    :type: DynamicPaintCanvasSettings
    """

    ui_type: bpy._typing.rna_enums.PropDynamicpaintTypeItems
    """ 

    :type: bpy._typing.rna_enums.PropDynamicpaintTypeItems
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class DynamicPaintSurface(bpy_struct):
    """A canvas surface layer"""

    brush_collection: Collection | None
    """ Only use brush objects from this collection

    :type: Collection | None
    """

    brush_influence_scale: float
    """ Adjust influence brush objects have on this surface

    :type: float
    """

    brush_radius_scale: float
    """ Adjust radius of proximity brushes or particles for this surface

    :type: float
    """

    color_dry_threshold: float
    """ The wetness level when colors start to shift to the background

    :type: float
    """

    color_spread_speed: float
    """ How fast colors get mixed within wet paint

    :type: float
    """

    depth_clamp: float
    """ Maximum level of depth intersection in object space (use 0.0 to disable)

    :type: float
    """

    displace_factor: float
    """ Strength of displace when applied to the mesh

    :type: float
    """

    displace_type: typing.Literal["DISPLACE", "DEPTH"]
    """ 

    :type: typing.Literal['DISPLACE','DEPTH']
    """

    dissolve_speed: int
    """ Approximately in how many frames should dissolve happen

    :type: int
    """

    drip_acceleration: float
    """ How much surface acceleration affects dripping

    :type: float
    """

    drip_velocity: float
    """ How much surface velocity affects dripping

    :type: float
    """

    dry_speed: int
    """ Approximately in how many frames should drying happen

    :type: int
    """

    effect_ui: typing.Literal["SPREAD", "DRIP", "SHRINK"]
    """ 

    :type: typing.Literal['SPREAD','DRIP','SHRINK']
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    frame_end: int
    """ Simulation end frame

    :type: int
    """

    frame_start: int
    """ Simulation start frame

    :type: int
    """

    frame_substeps: int
    """ Do extra frames between scene frames to ensure smooth motion

    :type: int
    """

    image_fileformat: typing.Literal["PNG", "OPENEXR"]
    """ 

    :type: typing.Literal['PNG','OPENEXR']
    """

    image_output_path: str
    """ Directory to save the textures

    :type: str
    """

    image_resolution: int
    """ Output image resolution

    :type: int
    """

    init_color: bpy_prop_array[float]
    """ Initial color of the surface

    :type: bpy_prop_array[float]
    """

    init_color_type: typing.Literal["NONE", "COLOR", "TEXTURE", "VERTEX_COLOR"]
    """ 

    :type: typing.Literal['NONE','COLOR','TEXTURE','VERTEX_COLOR']
    """

    init_layername: str
    """ 

    :type: str
    """

    init_texture: Texture | None
    """ 

    :type: Texture | None
    """

    is_active: bool
    """ Toggle whether surface is processed or ignored

    :type: bool
    """

    is_cache_user: bool
    """ 

    :type: bool
    """

    name: str
    """ Surface name

    :type: str
    """

    output_name_a: str
    """ Name used to save output from this surface

    :type: str
    """

    output_name_b: str
    """ Name used to save output from this surface

    :type: str
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    shrink_speed: float
    """ How fast shrink effect moves on the canvas surface

    :type: float
    """

    spread_speed: float
    """ How fast spread effect moves on the canvas surface

    :type: float
    """

    surface_format: typing.Literal["VERTEX", "IMAGE"]
    """ Surface Format

    :type: typing.Literal['VERTEX','IMAGE']
    """

    surface_type: typing.Literal["PAINT"]
    """ Surface Type

    :type: typing.Literal['PAINT']
    """

    use_antialiasing: bool
    """ Use 5× multisampling to smooth paint edges

    :type: bool
    """

    use_dissolve: bool
    """ Enable to make surface changes disappear over time

    :type: bool
    """

    use_dissolve_log: bool
    """ Use logarithmic dissolve (makes high values to fade faster than low values)

    :type: bool
    """

    use_drip: bool
    """ Process drip effect (drip wet paint to gravity direction)

    :type: bool
    """

    use_dry_log: bool
    """ Use logarithmic drying (makes high values to dry faster than low values)

    :type: bool
    """

    use_drying: bool
    """ Enable to make surface wetness dry over time

    :type: bool
    """

    use_incremental_displace: bool
    """ New displace is added cumulatively on top of existing

    :type: bool
    """

    use_output_a: bool
    """ Save this output layer

    :type: bool
    """

    use_output_b: bool
    """ Save this output layer

    :type: bool
    """

    use_premultiply: bool
    """ Multiply color by alpha (recommended for Blender input)

    :type: bool
    """

    use_shrink: bool
    """ Process shrink effect (shrink paint areas)

    :type: bool
    """

    use_spread: bool
    """ Process spread effect (spread wet paint around surface)

    :type: bool
    """

    use_wave_open_border: bool
    """ Pass waves through mesh edges

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    wave_damping: float
    """ Wave damping factor

    :type: float
    """

    wave_smoothness: float
    """ Limit maximum steepness of wave slope between simulation points (use higher values for smoother waves at expense of reduced detail)

    :type: float
    """

    wave_speed: float
    """ Wave propagation speed

    :type: float
    """

    wave_spring: float
    """ Spring force that pulls water level back to zero

    :type: float
    """

    wave_timescale: float
    """ Wave time scaling factor

    :type: float
    """

    def output_exists(self, object: Object, index: int | None) -> bool:
        """Checks if surface output layer of given name exists

        :param object:
        :type object: Object
        :param index: Index
        :type index: int | None
        :return:
        :rtype: bool
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class EQCurveMappingData(bpy_struct):
    """EQCurveMappingData"""

    curve_mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class EdgeSplitModifier(Modifier, bpy_struct):
    """Edge splitting modifier to create sharp edges"""

    split_angle: float
    """ Angle above which to split edges

    :type: float
    """

    use_edge_angle: bool
    """ Split edges with high angle between faces

    :type: bool
    """

    use_edge_sharp: bool
    """ Split edges that are marked as sharp

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class EditBone(bpy_struct):
    """Edit mode bone in an armature data-block"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveinz: float
    """ Z-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutz: float
    """ Z-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_custom_handle_end: typing_extensions.Self | None
    """ Bone that serves as the end handle for the B-Bone curve

    :type: typing_extensions.Self | None
    """

    bbone_custom_handle_start: typing_extensions.Self | None
    """ Bone that serves as the start handle for the B-Bone curve

    :type: typing_extensions.Self | None
    """

    bbone_easein: float
    """ Length of first Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_easeout: float
    """ Length of second Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_handle_type_end: typing.Literal["AUTO", "ABSOLUTE", "RELATIVE", "TANGENT"]
    """ Selects how the end handle of the B-Bone is computed

    :type: typing.Literal['AUTO','ABSOLUTE','RELATIVE','TANGENT']
    """

    bbone_handle_type_start: typing.Literal["AUTO", "ABSOLUTE", "RELATIVE", "TANGENT"]
    """ Selects how the start handle of the B-Bone is computed

    :type: typing.Literal['AUTO','ABSOLUTE','RELATIVE','TANGENT']
    """

    bbone_handle_use_ease_end: bool
    """ Multiply the B-Bone Ease Out channel by the local Y scale value of the end handle. This is done after the Scale Easing option and isn't affected by it

    :type: bool
    """

    bbone_handle_use_ease_start: bool
    """ Multiply the B-Bone Ease In channel by the local Y scale value of the start handle. This is done after the Scale Easing option and isn't affected by it

    :type: bool
    """

    bbone_handle_use_scale_end: bpy_prop_array[bool]
    """ Multiply B-Bone Scale Out channels by the local scale values of the end handle. This is done after the Scale Easing option and isn't affected by it

    :type: bpy_prop_array[bool]
    """

    bbone_handle_use_scale_start: bpy_prop_array[bool]
    """ Multiply B-Bone Scale In channels by the local scale values of the start handle. This is done after the Scale Easing option and isn't affected by it

    :type: bpy_prop_array[bool]
    """

    bbone_mapping_mode: typing.Literal["STRAIGHT", "CURVED"]
    """ Selects how the vertices are mapped to B-Bone segments based on their position

    :type: typing.Literal['STRAIGHT','CURVED']
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: mathutils.Vector
    """ Scale factors for the start of the B-Bone, adjusts thickness (for tapering effects)

    :type: mathutils.Vector
    """

    bbone_scaleout: mathutils.Vector
    """ Scale factors for the end of the B-Bone, adjusts thickness (for tapering effects)

    :type: mathutils.Vector
    """

    bbone_segments: int
    """ Number of subdivisions of bone (for B-Bones only)

    :type: int
    """

    bbone_x: float
    """ B-Bone X size

    :type: float
    """

    bbone_z: float
    """ B-Bone Z size

    :type: float
    """

    collections: bpy_prop_collection[BoneCollection]
    """ Bone Collections that contain this bone

    :type: bpy_prop_collection[BoneCollection]
    """

    color: BoneColor
    """ 

    :type: BoneColor
    """

    envelope_distance: float
    """ Bone deformation distance (for Envelope deform only)

    :type: float
    """

    envelope_weight: float
    """ Bone deformation weight (for Envelope deform only)

    :type: float
    """

    head: mathutils.Vector
    """ Location of head end of the bone

    :type: mathutils.Vector
    """

    head_radius: float
    """ Radius of head of bone (for Envelope deform only)

    :type: float
    """

    hide: bool
    """ Bone is not visible when in Edit Mode

    :type: bool
    """

    hide_select: bool
    """ Bone is able to be selected

    :type: bool
    """

    inherit_scale: typing.Literal[
        "FULL", "FIX_SHEAR", "ALIGNED", "AVERAGE", "NONE", "NONE_LEGACY"
    ]
    """ Specifies how the bone inherits scaling from the parent bone

    :type: typing.Literal['FULL','FIX_SHEAR','ALIGNED','AVERAGE','NONE','NONE_LEGACY']
    """

    length: float
    """ Length of the bone. Changing moves the tail end

    :type: float
    """

    lock: bool
    """ Bone is not able to be transformed when in Edit Mode

    :type: bool
    """

    matrix: mathutils.Matrix
    """ Matrix combining location and rotation of the bone (head position, direction and roll), in armature space (does not include/support bone's length/size)

    :type: mathutils.Matrix
    """

    name: str
    """ 

    :type: str
    """

    parent: typing_extensions.Self | None
    """ Parent edit bone (in same Armature)

    :type: typing_extensions.Self | None
    """

    roll: float
    """ Bone rotation around head-tail axis

    :type: float
    """

    select: bool
    """ 

    :type: bool
    """

    select_head: bool
    """ 

    :type: bool
    """

    select_tail: bool
    """ 

    :type: bool
    """

    show_wire: bool
    """ Bone is always displayed in wireframe regardless of viewport shading mode (useful for non-obstructive custom bone shapes)

    :type: bool
    """

    tail: mathutils.Vector
    """ Location of tail end of the bone

    :type: mathutils.Vector
    """

    tail_radius: float
    """ Radius of tail of bone (for Envelope deform only)

    :type: float
    """

    use_connect: bool
    """ When bone has a parent, bone's head is stuck to the parent's tail

    :type: bool
    """

    use_cyclic_offset: bool
    """ When bone doesn't have a parent, it receives cyclic offset effects (Deprecated)

    :type: bool
    """

    use_deform: bool
    """ Enable Bone to deform geometry

    :type: bool
    """

    use_endroll_as_inroll: bool
    """ Add Roll Out of the Start Handle bone to the Roll In value

    :type: bool
    """

    use_envelope_multiply: bool
    """ When deforming bone, multiply effects of Vertex Group weights with Envelope influence

    :type: bool
    """

    use_inherit_rotation: bool
    """ Bone inherits rotation or scale from parent bone

    :type: bool
    """

    use_local_location: bool
    """ Bone location is set in local space

    :type: bool
    """

    use_relative_parent: bool
    """ Object children will use relative transform, like deform

    :type: bool
    """

    use_scale_easing: bool
    """ Multiply the final easing values by the Scale In/Out Y factors

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.(readonly)"""

    children: typing.Any
    """ A list of all the bones children.(readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.(readonly)"""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.(readonly)"""

    def align_roll(
        self, vector: collections.abc.Sequence[float] | mathutils.Vector | None
    ):
        """Align the bone to a local-space roll so the Z axis points in the direction of the vector given

        :param vector: Vector
        :type vector: collections.abc.Sequence[float] | mathutils.Vector | None
        """

    def align_orientation(self, other):
        """Align this bone to another by moving its tail and settings its roll
        the length of the other bone is not used.

                :param other:
        """

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """

    def transform(
        self,
        matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None,
        *,
        scale: bool | None = True,
        roll: bool | None = True,
    ):
        """Transform the the bones head, tail, roll and envelope
        (when the matrix has a scale component).

                :param matrix: 3x3 or 4x4 transformation matrix.
                :type matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
                :param scale: Scale the bone envelope by the matrix.
                :type scale: bool | None
                :param roll: Correct the roll to point in the same relative
        direction to the head and tail.
                :type roll: bool | None
        """

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class EffectSequence(Sequence, bpy_struct):
    """Sequence strip applying an effect on the images created by other strips"""

    alpha_mode: typing.Literal["STRAIGHT", "PREMUL"]
    """ Representation of alpha information in the RGBA pixels

    :type: typing.Literal['STRAIGHT','PREMUL']
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ Adjust the intensity of the input's color

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    multiply_alpha: bool
    """ Multiply alpha along with color channels

    :type: bool
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_deinterlace: bool
    """ Remove fields from video movies

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or time-code index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class EffectorWeights(bpy_struct):
    """Effector weights for physics simulation"""

    all: float
    """ All effector's weight

    :type: float
    """

    apply_to_hair_growing: bool
    """ Use force fields when growing hair

    :type: bool
    """

    boid: float
    """ Boid effector weight

    :type: float
    """

    charge: float
    """ Charge effector weight

    :type: float
    """

    collection: Collection | None
    """ Limit effectors to this collection

    :type: Collection | None
    """

    curve_guide: float
    """ Curve guide effector weight

    :type: float
    """

    drag: float
    """ Drag effector weight

    :type: float
    """

    force: float
    """ Force effector weight

    :type: float
    """

    gravity: float
    """ Global gravity weight

    :type: float
    """

    harmonic: float
    """ Harmonic effector weight

    :type: float
    """

    lennardjones: float
    """ Lennard-Jones effector weight

    :type: float
    """

    magnetic: float
    """ Magnetic effector weight

    :type: float
    """

    smokeflow: float
    """ Fluid Flow effector weight

    :type: float
    """

    texture: float
    """ Texture effector weight

    :type: float
    """

    turbulence: float
    """ Turbulence effector weight

    :type: float
    """

    vortex: float
    """ Vortex effector weight

    :type: float
    """

    wind: float
    """ Wind effector weight

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class EnumProperty(Property, bpy_struct):
    """RNA enumeration property definition, to choose from a number of predefined options"""

    default: typing.Literal["DUMMY"]
    """ Default value for this enum

    :type: typing.Literal['DUMMY']
    """

    default_flag: set[str]
    """ Default value for this enum

    :type: set[str]
    """

    enum_items: bpy_prop_collection[EnumPropertyItem]
    """ Possible values for the property

    :type: bpy_prop_collection[EnumPropertyItem]
    """

    enum_items_static: bpy_prop_collection[EnumPropertyItem]
    """ Possible values for the property (never calls optional dynamic generation of those)

    :type: bpy_prop_collection[EnumPropertyItem]
    """

    enum_items_static_ui: bpy_prop_collection[EnumPropertyItem]
    """ Possible values for the property (never calls optional dynamic generation of those). Includes UI elements (separators and section headings)

    :type: bpy_prop_collection[EnumPropertyItem]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class EnumPropertyItem(bpy_struct):
    """Definition of a choice in an RNA enum property"""

    description: str
    """ Description of the item's purpose

    :type: str
    """

    icon: bpy._typing.rna_enums.IconItems
    """ Icon of the item

    :type: bpy._typing.rna_enums.IconItems
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    name: str
    """ Human readable name

    :type: str
    """

    value: int
    """ Value of the item

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class EnvelopeGpencilModifier(GpencilModifier, bpy_struct):
    """Envelope stroke effect modifier"""

    invert_layer_pass: bool
    """ Inverse filter

    :type: bool
    """

    invert_layers: bool
    """ Inverse filter

    :type: bool
    """

    invert_material_pass: bool
    """ Inverse filter

    :type: bool
    """

    invert_materials: bool
    """ Inverse filter

    :type: bool
    """

    invert_vertex: bool
    """ Inverse filter

    :type: bool
    """

    layer: str
    """ Layer name

    :type: str
    """

    layer_pass: int
    """ Layer pass index

    :type: int
    """

    mat_nr: int
    """ The material to use for the new strokes

    :type: int
    """

    material: Material | None
    """ Material used for filtering effect

    :type: Material | None
    """

    mode: typing.Literal["DEFORM", "SEGMENTS", "FILLS"]
    """ Algorithm to use for generating the envelope

    :type: typing.Literal['DEFORM','SEGMENTS','FILLS']
    """

    pass_index: int
    """ Pass index

    :type: int
    """

    skip: int
    """ The number of generated segments to skip to reduce complexity

    :type: int
    """

    spread: int
    """ The number of points to skip to create straight segments

    :type: int
    """

    strength: float
    """ Multiplier for the strength of the new strokes

    :type: float
    """

    thickness: float
    """ Multiplier for the thickness of the new strokes

    :type: float
    """

    vertex_group: str
    """ Vertex group name for modulating the deform

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Event(bpy_struct):
    """Window Manager Event"""

    alt: bool
    """ True when the Alt/Option key is held

    :type: bool
    """

    ascii: str
    """ Single ASCII character for this event

    :type: str
    """

    ctrl: bool
    """ True when the Ctrl key is held

    :type: bool
    """

    direction: bpy._typing.rna_enums.EventDirectionItems
    """ The direction (only applies to drag events)

    :type: bpy._typing.rna_enums.EventDirectionItems
    """

    is_consecutive: bool
    """ Part of a track-pad or NDOF motion, interrupted by cursor motion, button or key press events

    :type: bool
    """

    is_mouse_absolute: bool
    """ The last motion event was an absolute input

    :type: bool
    """

    is_repeat: bool
    """ The event is generated by holding a key down

    :type: bool
    """

    is_tablet: bool
    """ The event has tablet data

    :type: bool
    """

    mouse_prev_press_x: int
    """ The window relative horizontal location of the last press event

    :type: int
    """

    mouse_prev_press_y: int
    """ The window relative vertical location of the last press event

    :type: int
    """

    mouse_prev_x: int
    """ The window relative horizontal location of the mouse

    :type: int
    """

    mouse_prev_y: int
    """ The window relative vertical location of the mouse

    :type: int
    """

    mouse_region_x: int
    """ The region relative horizontal location of the mouse

    :type: int
    """

    mouse_region_y: int
    """ The region relative vertical location of the mouse

    :type: int
    """

    mouse_x: int
    """ The window relative horizontal location of the mouse

    :type: int
    """

    mouse_y: int
    """ The window relative vertical location of the mouse

    :type: int
    """

    oskey: bool
    """ True when the Cmd key is held

    :type: bool
    """

    pressure: float
    """ The pressure of the tablet or 1.0 if no tablet present

    :type: float
    """

    shift: bool
    """ True when the Shift key is held

    :type: bool
    """

    tilt: mathutils.Vector
    """ The pressure of the tablet or zeroes if no tablet present

    :type: mathutils.Vector
    """

    type: bpy._typing.rna_enums.EventTypeItems
    """ 

    :type: bpy._typing.rna_enums.EventTypeItems
    """

    type_prev: bpy._typing.rna_enums.EventTypeItems
    """ 

    :type: bpy._typing.rna_enums.EventTypeItems
    """

    unicode: str
    """ Single unicode character for this event

    :type: str
    """

    value: bpy._typing.rna_enums.EventValueItems
    """ The type of event, only applies to some

    :type: bpy._typing.rna_enums.EventValueItems
    """

    value_prev: bpy._typing.rna_enums.EventValueItems
    """ The type of event, only applies to some

    :type: bpy._typing.rna_enums.EventValueItems
    """

    xr: XrEventData
    """ XR event data

    :type: XrEventData
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class ExplodeModifier(Modifier, bpy_struct):
    """Explosion effect modifier based on a particle system"""

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    particle_uv: str
    """ UV map to change with particle age

    :type: str
    """

    protect: float
    """ Clean vertex group edges

    :type: float
    """

    show_alive: bool
    """ Show mesh when particles are alive

    :type: bool
    """

    show_dead: bool
    """ Show mesh when particles are dead

    :type: bool
    """

    show_unborn: bool
    """ Show mesh when particles are unborn

    :type: bool
    """

    use_edge_cut: bool
    """ Cut face edges for nicer shrapnel

    :type: bool
    """

    use_size: bool
    """ Use particle size for the shrapnel

    :type: bool
    """

    vertex_group: str
    """ 

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FCurve(bpy_struct):
    """F-Curve defining values of a period of time"""

    array_index: int
    """ Index to the specific property affected by F-Curve if applicable

    :type: int
    """

    auto_smoothing: bpy._typing.rna_enums.FcurveAutoSmoothingItems
    """ Algorithm used to compute automatic handles

    :type: bpy._typing.rna_enums.FcurveAutoSmoothingItems
    """

    color: mathutils.Color
    """ Color of the F-Curve in the Graph Editor

    :type: mathutils.Color
    """

    color_mode: typing.Literal["AUTO_RAINBOW", "AUTO_RGB", "AUTO_YRGB", "CUSTOM"]
    """ Method used to determine color of F-Curve in Graph Editor

    :type: typing.Literal['AUTO_RAINBOW','AUTO_RGB','AUTO_YRGB','CUSTOM']
    """

    data_path: str
    """ RNA Path to property affected by F-Curve

    :type: str
    """

    driver: Driver
    """ Channel Driver (only set for Driver F-Curves)

    :type: Driver
    """

    extrapolation: typing.Literal["CONSTANT", "LINEAR"]
    """ Method used for evaluating value of F-Curve outside first and last keyframes

    :type: typing.Literal['CONSTANT','LINEAR']
    """

    group: ActionGroup | None
    """ Action Group that this F-Curve belongs to

    :type: ActionGroup | None
    """

    hide: bool
    """ F-Curve and its keyframes are hidden in the Graph Editor graphs

    :type: bool
    """

    is_empty: bool
    """ True if the curve contributes no animation due to lack of keyframes or useful modifiers, and should be deleted

    :type: bool
    """

    is_valid: bool
    """ False when F-Curve could not be evaluated in past, so should be skipped when evaluating

    :type: bool
    """

    keyframe_points: FCurveKeyframePoints
    """ User-editable keyframes

    :type: FCurveKeyframePoints
    """

    lock: bool
    """ F-Curve's settings cannot be edited

    :type: bool
    """

    modifiers: FCurveModifiers
    """ Modifiers affecting the shape of the F-Curve

    :type: FCurveModifiers
    """

    mute: bool
    """ Disable F-Curve evaluation

    :type: bool
    """

    sampled_points: bpy_prop_collection[FCurveSample]
    """ Sampled animation data

    :type: bpy_prop_collection[FCurveSample]
    """

    select: bool
    """ F-Curve is selected for editing

    :type: bool
    """

    def evaluate(self, frame: float | None) -> float:
        """Evaluate F-Curve

        :param frame: Frame, Evaluate F-Curve at given frame
        :type frame: float | None
        :return: Value, Value of F-Curve specific frame
        :rtype: float
        """

    def update(self):
        """Ensure keyframes are sorted in chronological order and handles are set correctly"""

    def range(self) -> mathutils.Vector:
        """Get the time extents for F-Curve

        :return: Range, Min/Max values
        :rtype: mathutils.Vector
        """

    def update_autoflags(self, data: typing.Any):
        """Update FCurve flags set automatically from affected property (currently, integer/discrete flags set when the property is not a float)

        :param data: Data, Data containing the property controlled by given FCurve
        :type data: typing.Any
        """

    def convert_to_samples(self, start: int | None, end: int | None):
        """Convert current FCurve from keyframes to sample points, if necessary

        :param start: Start Frame
        :type start: int | None
        :param end: End Frame
        :type end: int | None
        """

    def convert_to_keyframes(self, start: int | None, end: int | None):
        """Convert current FCurve from sample points to keyframes (linear interpolation), if necessary

        :param start: Start Frame
        :type start: int | None
        :param end: End Frame
        :type end: int | None
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FCurveSample(bpy_struct):
    """Sample point for F-Curve"""

    co: mathutils.Vector
    """ Point coordinates

    :type: mathutils.Vector
    """

    select: bool
    """ Selection status

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FFmpegSettings(bpy_struct):
    """FFmpeg related settings for the scene"""

    audio_bitrate: int
    """ Audio bitrate (kb/s)

    :type: int
    """

    audio_channels: typing.Literal[
        "MONO", "STEREO", "SURROUND4", "SURROUND51", "SURROUND71"
    ]
    """ Audio channel count

    :type: typing.Literal['MONO','STEREO','SURROUND4','SURROUND51','SURROUND71']
    """

    audio_codec: typing.Literal[
        "NONE", "AAC", "AC3", "FLAC", "MP2", "MP3", "OPUS", "PCM", "VORBIS"
    ]
    """ FFmpeg audio codec to use

    :type: typing.Literal['NONE','AAC','AC3','FLAC','MP2','MP3','OPUS','PCM','VORBIS']
    """

    audio_mixrate: int
    """ Audio sample rate (samples/s)

    :type: int
    """

    audio_volume: float
    """ Audio volume

    :type: float
    """

    buffersize: int
    """ Rate control: buffer size (kb)

    :type: int
    """

    codec: typing.Literal[
        "NONE",
        "DNXHD",
        "DV",
        "FFV1",
        "FLASH",
        "H264",
        "HUFFYUV",
        "MPEG1",
        "MPEG2",
        "MPEG4",
        "PNG",
        "QTRLE",
        "THEORA",
        "WEBM",
        "AV1",
    ]
    """ FFmpeg codec to use for video output

    :type: typing.Literal['NONE','DNXHD','DV','FFV1','FLASH','H264','HUFFYUV','MPEG1','MPEG2','MPEG4','PNG','QTRLE','THEORA','WEBM','AV1']
    """

    constant_rate_factor: typing.Literal[
        "NONE",
        "LOSSLESS",
        "PERC_LOSSLESS",
        "HIGH",
        "MEDIUM",
        "LOW",
        "VERYLOW",
        "LOWEST",
    ]
    """ Constant Rate Factor (CRF); tradeoff between video quality and file size

    :type: typing.Literal['NONE','LOSSLESS','PERC_LOSSLESS','HIGH','MEDIUM','LOW','VERYLOW','LOWEST']
    """

    ffmpeg_preset: typing.Literal["BEST", "GOOD", "REALTIME"]
    """ Tradeoff between encoding speed and compression ratio

    :type: typing.Literal['BEST','GOOD','REALTIME']
    """

    format: typing.Literal[
        "MPEG1",
        "MPEG2",
        "MPEG4",
        "AVI",
        "QUICKTIME",
        "DV",
        "OGG",
        "MKV",
        "FLASH",
        "WEBM",
    ]
    """ Output file container

    :type: typing.Literal['MPEG1','MPEG2','MPEG4','AVI','QUICKTIME','DV','OGG','MKV','FLASH','WEBM']
    """

    gopsize: int
    """ Distance between key frames, also known as GOP size; influences file size and seekability

    :type: int
    """

    max_b_frames: int
    """ Maximum number of B-frames between non-B-frames; influences file size and seekability

    :type: int
    """

    maxrate: int
    """ Rate control: max rate (kbit/s)

    :type: int
    """

    minrate: int
    """ Rate control: min rate (kbit/s)

    :type: int
    """

    muxrate: int
    """ Mux rate (bits/second)

    :type: int
    """

    packetsize: int
    """ Mux packet size (byte)

    :type: int
    """

    use_autosplit: bool
    """ Autosplit output at 2GB boundary

    :type: bool
    """

    use_lossless_output: bool
    """ Use lossless output for video streams

    :type: bool
    """

    use_max_b_frames: bool
    """ Set a maximum number of B-frames

    :type: bool
    """

    video_bitrate: int
    """ Video bitrate (kbit/s)

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FILEBROWSER_UL_dir(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        _data,
        item,
        icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param _data:
        :param item:
        :param icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifier(bpy_struct):
    """Modifier for values of F-Curve"""

    active: bool | None
    """ F-Curve modifier will show settings in the editor

    :type: bool | None
    """

    blend_in: float
    """ Number of frames from start frame for influence to take effect

    :type: float
    """

    blend_out: float
    """ Number of frames from end frame for influence to fade out

    :type: float
    """

    frame_end: float
    """ Frame that modifier's influence ends (if Restrict Frame Range is in use)

    :type: float
    """

    frame_start: float
    """ Frame that modifier's influence starts (if Restrict Frame Range is in use)

    :type: float
    """

    influence: float
    """ Amount of influence F-Curve Modifier will have when not fading in/out

    :type: float
    """

    is_valid: bool
    """ F-Curve Modifier has invalid settings and will not be evaluated

    :type: bool
    """

    mute: bool
    """ Enable F-Curve modifier evaluation

    :type: bool
    """

    name: str
    """ F-Curve Modifier name

    :type: str
    """

    show_expanded: bool
    """ F-Curve Modifier's panel is expanded in UI

    :type: bool
    """

    type: bpy._typing.rna_enums.FmodifierTypeItems
    """ F-Curve Modifier Type

    :type: bpy._typing.rna_enums.FmodifierTypeItems
    """

    use_influence: bool
    """ F-Curve Modifier's effects will be tempered by a default factor

    :type: bool
    """

    use_restricted_range: bool
    """ F-Curve Modifier is only applied for the specified frame range to help mask off effects in order to chain them

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierCycles(FModifier, bpy_struct):
    """Repeat the values of the modified F-Curve"""

    cycles_after: int
    """ Maximum number of cycles to allow after last keyframe (0 = infinite)

    :type: int
    """

    cycles_before: int
    """ Maximum number of cycles to allow before first keyframe (0 = infinite)

    :type: int
    """

    mode_after: typing.Literal["NONE", "REPEAT", "REPEAT_OFFSET", "MIRROR"]
    """ Cycling mode to use after last keyframe

    :type: typing.Literal['NONE','REPEAT','REPEAT_OFFSET','MIRROR']
    """

    mode_before: typing.Literal["NONE", "REPEAT", "REPEAT_OFFSET", "MIRROR"]
    """ Cycling mode to use before first keyframe

    :type: typing.Literal['NONE','REPEAT','REPEAT_OFFSET','MIRROR']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierEnvelope(FModifier, bpy_struct):
    """Scale the values of the modified F-Curve"""

    control_points: FModifierEnvelopeControlPoints
    """ Control points defining the shape of the envelope

    :type: FModifierEnvelopeControlPoints
    """

    default_max: float
    """ Upper distance from Reference Value for 1:1 default influence

    :type: float
    """

    default_min: float
    """ Lower distance from Reference Value for 1:1 default influence

    :type: float
    """

    reference_value: float
    """ Value that envelope's influence is centered around / based on

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierEnvelopeControlPoint(bpy_struct):
    """Control point for envelope F-Modifier"""

    frame: float
    """ Frame this control-point occurs on

    :type: float
    """

    max: float
    """ Upper bound of envelope at this control-point

    :type: float
    """

    min: float
    """ Lower bound of envelope at this control-point

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierFunctionGenerator(FModifier, bpy_struct):
    """Generate values using a built-in function"""

    amplitude: float
    """ Scale factor determining the maximum/minimum values

    :type: float
    """

    function_type: typing.Literal["SIN", "COS", "TAN", "SQRT", "LN", "SINC"]
    """ Type of built-in function to use

    :type: typing.Literal['SIN','COS','TAN','SQRT','LN','SINC']
    """

    phase_multiplier: float
    """ Scale factor determining the 'speed' of the function

    :type: float
    """

    phase_offset: float
    """ Constant factor to offset time by for function

    :type: float
    """

    use_additive: bool
    """ Values generated by this modifier are applied on top of the existing values instead of overwriting them

    :type: bool
    """

    value_offset: float
    """ Constant factor to offset values by

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierGenerator(FModifier, bpy_struct):
    """Deterministically generate values for the modified F-Curve"""

    coefficients: bpy_prop_array[float]
    """ Coefficients for 'x' (starting from lowest power of x^0)

    :type: bpy_prop_array[float]
    """

    mode: typing.Literal["POLYNOMIAL", "POLYNOMIAL_FACTORISED"]
    """ Type of generator to use

    :type: typing.Literal['POLYNOMIAL','POLYNOMIAL_FACTORISED']
    """

    poly_order: int
    """ The highest power of 'x' for this polynomial (number of coefficients - 1)

    :type: int
    """

    use_additive: bool
    """ Values generated by this modifier are applied on top of the existing values instead of overwriting them

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierLimits(FModifier, bpy_struct):
    """Limit the time/value ranges of the modified F-Curve"""

    max_x: float
    """ Highest X value to allow

    :type: float
    """

    max_y: float
    """ Highest Y value to allow

    :type: float
    """

    min_x: float
    """ Lowest X value to allow

    :type: float
    """

    min_y: float
    """ Lowest Y value to allow

    :type: float
    """

    use_max_x: bool
    """ Use the maximum X value

    :type: bool
    """

    use_max_y: bool
    """ Use the maximum Y value

    :type: bool
    """

    use_min_x: bool
    """ Use the minimum X value

    :type: bool
    """

    use_min_y: bool
    """ Use the minimum Y value

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierNoise(FModifier, bpy_struct):
    """Give randomness to the modified F-Curve"""

    blend_type: typing.Literal["REPLACE", "ADD", "SUBTRACT", "MULTIPLY"]
    """ Method of modifying the existing F-Curve

    :type: typing.Literal['REPLACE','ADD','SUBTRACT','MULTIPLY']
    """

    depth: int
    """ Amount of fine level detail present in the noise

    :type: int
    """

    offset: float
    """ Time offset for the noise effect

    :type: float
    """

    phase: float
    """ A random seed for the noise effect

    :type: float
    """

    scale: float
    """ Scaling (in time) of the noise

    :type: float
    """

    strength: float
    """ Amplitude of the noise - the amount that it modifies the underlying curve

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierPython(FModifier, bpy_struct):
    """Perform user-defined operation on the modified F-Curve"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FModifierStepped(FModifier, bpy_struct):
    """Hold each interpolated value from the F-Curve for several frames without changing the timing"""

    frame_end: float
    """ Frame that modifier's influence ends (if applicable)

    :type: float
    """

    frame_offset: float
    """ Reference number of frames before frames get held (use to get hold for '1-3' vs '5-7' holding patterns)

    :type: float
    """

    frame_start: float
    """ Frame that modifier's influence starts (if applicable)

    :type: float
    """

    frame_step: float
    """ Number of frames to hold each value

    :type: float
    """

    use_frame_end: bool
    """ Restrict modifier to only act before its 'end' frame

    :type: bool
    """

    use_frame_start: bool
    """ Restrict modifier to only act after its 'start' frame

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FieldSettings(bpy_struct):
    """Field settings for an object in physics simulation"""

    apply_to_location: bool
    """ Affect particle's location

    :type: bool
    """

    apply_to_rotation: bool
    """ Affect particle's dynamic rotation

    :type: bool
    """

    distance_max: float
    """ Maximum distance for the field to work

    :type: float
    """

    distance_min: float
    """ Minimum distance for the field's falloff

    :type: float
    """

    falloff_power: float
    """ How quickly strength falls off with distance from the force field

    :type: float
    """

    falloff_type: typing.Literal["CONE", "SPHERE", "TUBE"]
    """ 

    :type: typing.Literal['CONE','SPHERE','TUBE']
    """

    flow: float
    """ Convert effector force into air flow velocity

    :type: float
    """

    guide_clump_amount: float
    """ Amount of clumping

    :type: float
    """

    guide_clump_shape: float
    """ Shape of clumping

    :type: float
    """

    guide_free: float
    """ Guide-free time from particle life's end

    :type: float
    """

    guide_kink_amplitude: float
    """ The amplitude of the offset

    :type: float
    """

    guide_kink_axis: bpy._typing.rna_enums.AxisXyzItems
    """ Which axis to use for offset

    :type: bpy._typing.rna_enums.AxisXyzItems
    """

    guide_kink_frequency: float
    """ The frequency of the offset (1/total length)

    :type: float
    """

    guide_kink_shape: float
    """ Adjust the offset to the beginning/end

    :type: float
    """

    guide_kink_type: typing.Literal[
        "NONE", "BRAID", "CURL", "RADIAL", "ROLL", "ROTATION", "WAVE"
    ]
    """ Type of periodic offset on the curve

    :type: typing.Literal['NONE','BRAID','CURL','RADIAL','ROLL','ROTATION','WAVE']
    """

    guide_minimum: float
    """ The distance from which particles are affected fully

    :type: float
    """

    harmonic_damping: float
    """ Damping of the harmonic force

    :type: float
    """

    inflow: float
    """ Inwards component of the vortex force

    :type: float
    """

    linear_drag: float
    """ Drag component proportional to velocity

    :type: float
    """

    noise: float
    """ Amount of noise for the force strength

    :type: float
    """

    quadratic_drag: float
    """ Drag component proportional to the square of velocity

    :type: float
    """

    radial_falloff: float
    """ Radial falloff power (real gravitational falloff = 2)

    :type: float
    """

    radial_max: float
    """ Maximum radial distance for the field to work

    :type: float
    """

    radial_min: float
    """ Minimum radial distance for the field's falloff

    :type: float
    """

    rest_length: float
    """ Rest length of the harmonic force

    :type: float
    """

    seed: int
    """ Seed of the noise

    :type: int
    """

    shape: typing.Literal["POINT", "LINE", "PLANE", "SURFACE", "POINTS"]
    """ Which direction is used to calculate the effector force

    :type: typing.Literal['POINT','LINE','PLANE','SURFACE','POINTS']
    """

    size: float
    """ Size of the turbulence

    :type: float
    """

    source_object: Object | None
    """ Select domain object of the smoke simulation

    :type: Object | None
    """

    strength: float
    """ Strength of force field

    :type: float
    """

    texture: Texture | None
    """ Texture to use as force

    :type: Texture | None
    """

    texture_mode: typing.Literal["CURL", "GRADIENT", "RGB"]
    """ How the texture effect is calculated (RGB and Curl need a RGB texture, else Gradient will be used instead)

    :type: typing.Literal['CURL','GRADIENT','RGB']
    """

    texture_nabla: float
    """ Defines size of derivative offset used for calculating gradient and curl

    :type: float
    """

    type: typing.Literal[
        "NONE",
        "BOID",
        "CHARGE",
        "GUIDE",
        "DRAG",
        "FLUID_FLOW",
        "FORCE",
        "HARMONIC",
        "LENNARDJ",
        "MAGNET",
        "TEXTURE",
        "TURBULENCE",
        "VORTEX",
        "WIND",
    ]
    """ Type of field

    :type: typing.Literal['NONE','BOID','CHARGE','GUIDE','DRAG','FLUID_FLOW','FORCE','HARMONIC','LENNARDJ','MAGNET','TEXTURE','TURBULENCE','VORTEX','WIND']
    """

    use_2d_force: bool
    """ Apply force only in 2D

    :type: bool
    """

    use_absorption: bool
    """ Force gets absorbed by collision objects

    :type: bool
    """

    use_global_coords: bool
    """ Use effector/global coordinates for turbulence

    :type: bool
    """

    use_gravity_falloff: bool
    """ Multiply force by 1/distance²

    :type: bool
    """

    use_guide_path_add: bool
    """ Based on distance/falloff it adds a portion of the entire path

    :type: bool
    """

    use_guide_path_weight: bool
    """ Use curve weights to influence the particle influence along the curve

    :type: bool
    """

    use_max_distance: bool
    """ Use a maximum distance for the field to work

    :type: bool
    """

    use_min_distance: bool
    """ Use a minimum distance for the field's falloff

    :type: bool
    """

    use_multiple_springs: bool
    """ Every point is affected by multiple springs

    :type: bool
    """

    use_object_coords: bool
    """ Use object/global coordinates for texture

    :type: bool
    """

    use_radial_max: bool
    """ Use a maximum radial distance for the field to work

    :type: bool
    """

    use_radial_min: bool
    """ Use a minimum radial distance for the field's falloff

    :type: bool
    """

    use_root_coords: bool
    """ Texture coordinates from root particle locations

    :type: bool
    """

    use_smoke_density: bool
    """ Adjust force strength based on smoke density

    :type: bool
    """

    wind_factor: float
    """ How much the force is reduced when acting parallel to a surface, e.g. cloth

    :type: float
    """

    z_direction: typing.Literal["POSITIVE", "NEGATIVE", "BOTH"]
    """ Effect in full or only positive/negative Z direction

    :type: typing.Literal['POSITIVE','NEGATIVE','BOTH']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FileAssetSelectIDFilter(bpy_struct):
    """Which asset types to show/hide, when browsing an asset library"""

    experimental_filter_armature: bool
    """ Show Armature data-blocks

    :type: bool
    """

    experimental_filter_brush: bool
    """ Show Brushes data-blocks

    :type: bool
    """

    experimental_filter_cachefile: bool
    """ Show Cache File data-blocks

    :type: bool
    """

    experimental_filter_camera: bool
    """ Show Camera data-blocks

    :type: bool
    """

    experimental_filter_curve: bool
    """ Show Curve data-blocks

    :type: bool
    """

    experimental_filter_curves: bool
    """ Show/hide Curves data-blocks

    :type: bool
    """

    experimental_filter_font: bool
    """ Show Font data-blocks

    :type: bool
    """

    experimental_filter_grease_pencil: bool
    """ Show Grease pencil data-blocks

    :type: bool
    """

    experimental_filter_image: bool
    """ Show Image data-blocks

    :type: bool
    """

    experimental_filter_lattice: bool
    """ Show Lattice data-blocks

    :type: bool
    """

    experimental_filter_light: bool
    """ Show Light data-blocks

    :type: bool
    """

    experimental_filter_light_probe: bool
    """ Show Light Probe data-blocks

    :type: bool
    """

    experimental_filter_linestyle: bool
    """ Show Freestyle's Line Style data-blocks

    :type: bool
    """

    experimental_filter_mask: bool
    """ Show Mask data-blocks

    :type: bool
    """

    experimental_filter_mesh: bool
    """ Show Mesh data-blocks

    :type: bool
    """

    experimental_filter_metaball: bool
    """ Show Metaball data-blocks

    :type: bool
    """

    experimental_filter_movie_clip: bool
    """ Show Movie Clip data-blocks

    :type: bool
    """

    experimental_filter_paint_curve: bool
    """ Show Paint Curve data-blocks

    :type: bool
    """

    experimental_filter_palette: bool
    """ Show Palette data-blocks

    :type: bool
    """

    experimental_filter_particle_settings: bool
    """ Show Particle Settings data-blocks

    :type: bool
    """

    experimental_filter_pointcloud: bool
    """ Show/hide Point Cloud data-blocks

    :type: bool
    """

    experimental_filter_scene: bool
    """ Show Scene data-blocks

    :type: bool
    """

    experimental_filter_sound: bool
    """ Show Sound data-blocks

    :type: bool
    """

    experimental_filter_speaker: bool
    """ Show Speaker data-blocks

    :type: bool
    """

    experimental_filter_text: bool
    """ Show Text data-blocks

    :type: bool
    """

    experimental_filter_texture: bool
    """ Show Texture data-blocks

    :type: bool
    """

    experimental_filter_volume: bool
    """ Show/hide Volume data-blocks

    :type: bool
    """

    experimental_filter_work_space: bool
    """ Show workspace data-blocks

    :type: bool
    """

    filter_action: bool
    """ Show Action data-blocks

    :type: bool
    """

    filter_group: bool
    """ Show Collection data-blocks

    :type: bool
    """

    filter_material: bool
    """ Show Material data-blocks

    :type: bool
    """

    filter_node_tree: bool
    """ Show Node Tree data-blocks

    :type: bool
    """

    filter_object: bool
    """ Show Object data-blocks

    :type: bool
    """

    filter_world: bool
    """ Show World data-blocks

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FileAssetSelectParams(FileSelectParams, bpy_struct):
    """Settings for the file selection in Asset Browser mode"""

    asset_library_reference: str
    """ 

    :type: str
    """

    catalog_id: str
    """ The UUID of the catalog shown in the browser

    :type: str
    """

    filter_asset_id: FileAssetSelectIDFilter
    """ Which asset types to show/hide, when browsing an asset library

    :type: FileAssetSelectIDFilter
    """

    import_method: typing.Literal["FOLLOW_PREFS", "LINK", "APPEND", "APPEND_REUSE"]
    """ Determine how the asset will be imported

    :type: typing.Literal['FOLLOW_PREFS','LINK','APPEND','APPEND_REUSE']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FileBrowserFSMenuEntry(bpy_struct):
    """File Select Parameters"""

    icon: int
    """ 

    :type: int
    """

    is_valid: bool
    """ Whether this path is currently reachable

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    path: str
    """ 

    :type: str
    """

    use_save: bool
    """ Whether this path is saved in bookmarks, or generated from OS

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FileSelectEntry(bpy_struct):
    """A file viewable in the File Browser"""

    asset_data: AssetMetaData
    """ Asset data, valid if the file represents an asset

    :type: AssetMetaData
    """

    name: str
    """ 

    :type: str
    """

    preview_icon_id: int
    """ Unique integer identifying the preview of this file as an icon (zero means invalid)

    :type: int
    """

    relative_path: str
    """ Path relative to the directory currently displayed in the File Browser (includes the file name)

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FileSelectIDFilter(bpy_struct):
    """Which ID types to show/hide, when browsing a library"""

    category_animation: bool
    """ Show animation data

    :type: bool
    """

    category_environment: bool
    """ Show worlds, lights, cameras and speakers

    :type: bool
    """

    category_geometry: bool
    """ Show meshes, curves, lattice, armatures and metaballs data

    :type: bool
    """

    category_image: bool
    """ Show images, movie clips, sounds and masks

    :type: bool
    """

    category_misc: bool
    """ Show other data types

    :type: bool
    """

    category_object: bool
    """ Show objects and collections

    :type: bool
    """

    category_scene: bool
    """ Show scenes

    :type: bool
    """

    category_shading: bool
    """ Show materials, node-trees, textures and Freestyle's line-styles

    :type: bool
    """

    filter_action: bool
    """ Show Action data-blocks

    :type: bool
    """

    filter_armature: bool
    """ Show Armature data-blocks

    :type: bool
    """

    filter_brush: bool
    """ Show Brushes data-blocks

    :type: bool
    """

    filter_cachefile: bool
    """ Show Cache File data-blocks

    :type: bool
    """

    filter_camera: bool
    """ Show Camera data-blocks

    :type: bool
    """

    filter_curve: bool
    """ Show Curve data-blocks

    :type: bool
    """

    filter_curves: bool
    """ Show/hide Curves data-blocks

    :type: bool
    """

    filter_font: bool
    """ Show Font data-blocks

    :type: bool
    """

    filter_grease_pencil: bool
    """ Show Grease pencil data-blocks

    :type: bool
    """

    filter_group: bool
    """ Show Collection data-blocks

    :type: bool
    """

    filter_image: bool
    """ Show Image data-blocks

    :type: bool
    """

    filter_lattice: bool
    """ Show Lattice data-blocks

    :type: bool
    """

    filter_light: bool
    """ Show Light data-blocks

    :type: bool
    """

    filter_light_probe: bool
    """ Show Light Probe data-blocks

    :type: bool
    """

    filter_linestyle: bool
    """ Show Freestyle's Line Style data-blocks

    :type: bool
    """

    filter_mask: bool
    """ Show Mask data-blocks

    :type: bool
    """

    filter_material: bool
    """ Show Material data-blocks

    :type: bool
    """

    filter_mesh: bool
    """ Show Mesh data-blocks

    :type: bool
    """

    filter_metaball: bool
    """ Show Metaball data-blocks

    :type: bool
    """

    filter_movie_clip: bool
    """ Show Movie Clip data-blocks

    :type: bool
    """

    filter_node_tree: bool
    """ Show Node Tree data-blocks

    :type: bool
    """

    filter_object: bool
    """ Show Object data-blocks

    :type: bool
    """

    filter_paint_curve: bool
    """ Show Paint Curve data-blocks

    :type: bool
    """

    filter_palette: bool
    """ Show Palette data-blocks

    :type: bool
    """

    filter_particle_settings: bool
    """ Show Particle Settings data-blocks

    :type: bool
    """

    filter_pointcloud: bool
    """ Show/hide Point Cloud data-blocks

    :type: bool
    """

    filter_scene: bool
    """ Show Scene data-blocks

    :type: bool
    """

    filter_sound: bool
    """ Show Sound data-blocks

    :type: bool
    """

    filter_speaker: bool
    """ Show Speaker data-blocks

    :type: bool
    """

    filter_text: bool
    """ Show Text data-blocks

    :type: bool
    """

    filter_texture: bool
    """ Show Texture data-blocks

    :type: bool
    """

    filter_volume: bool
    """ Show/hide Volume data-blocks

    :type: bool
    """

    filter_work_space: bool
    """ Show workspace data-blocks

    :type: bool
    """

    filter_world: bool
    """ Show World data-blocks

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FileSelectParams(bpy_struct):
    """File Select Parameters"""

    directory: str
    """ Directory displayed in the file browser

    :type: str
    """

    display_size: int
    """ Change the size of thumbnails

    :type: int
    """

    display_size_discrete: typing.Literal["TINY", "SMALL", "NORMAL", "BIG", "LARGE"]
    """ Change the size of thumbnails in discrete steps

    :type: typing.Literal['TINY','SMALL','NORMAL','BIG','LARGE']
    """

    display_type: typing.Literal["LIST_VERTICAL", "LIST_HORIZONTAL", "THUMBNAIL"]
    """ Display mode for the file list

    :type: typing.Literal['LIST_VERTICAL','LIST_HORIZONTAL','THUMBNAIL']
    """

    filename: str
    """ Active file in the file browser

    :type: str
    """

    filter_glob: str
    """ UNIX shell-like filename patterns matching, supports wildcards ('*') and list of patterns separated by ';'

    :type: str
    """

    filter_id: FileSelectIDFilter
    """ Which ID types to show/hide, when browsing a library

    :type: FileSelectIDFilter
    """

    filter_search: str
    """ Filter by name or tag, supports '*' wildcard

    :type: str
    """

    recursion_level: typing.Literal["NONE", "BLEND", "ALL_1", "ALL_2", "ALL_3"]
    """ Numbers of dirtree levels to show simultaneously

    :type: typing.Literal['NONE','BLEND','ALL_1','ALL_2','ALL_3']
    """

    show_details_datetime: bool
    """ Show a column listing the date and time of modification for each file

    :type: bool
    """

    show_details_size: bool
    """ Show a column listing the size of each file

    :type: bool
    """

    show_hidden: bool
    """ Show hidden dot files

    :type: bool
    """

    sort_method: bpy._typing.rna_enums.FileselectParamsSortItems
    """ 

    :type: bpy._typing.rna_enums.FileselectParamsSortItems
    """

    title: str
    """ Title for the file browser

    :type: str
    """

    use_filter: bool
    """ Enable filtering of files

    :type: bool
    """

    use_filter_asset_only: bool
    """ Hide .blend files items that are not data-blocks with asset metadata

    :type: bool
    """

    use_filter_backup: bool
    """ Show .blend1, .blend2, etc. files

    :type: bool
    """

    use_filter_blender: bool
    """ Show .blend files

    :type: bool
    """

    use_filter_blendid: bool
    """ Show .blend files items (objects, materials, etc.)

    :type: bool
    """

    use_filter_folder: bool
    """ Show folders

    :type: bool
    """

    use_filter_font: bool
    """ Show font files

    :type: bool
    """

    use_filter_image: bool
    """ Show image files

    :type: bool
    """

    use_filter_movie: bool
    """ Show movie files

    :type: bool
    """

    use_filter_script: bool
    """ Show script files

    :type: bool
    """

    use_filter_sound: bool
    """ Show sound files

    :type: bool
    """

    use_filter_text: bool
    """ Show text files

    :type: bool
    """

    use_filter_volume: bool
    """ Show 3D volume files

    :type: bool
    """

    use_library_browsing: bool
    """ Whether we may browse Blender files' content or not

    :type: bool
    """

    use_sort_invert: bool
    """ Sort items descending, from highest value to lowest

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Float2Attribute(Attribute, bpy_struct):
    """Geometry attribute that stores floating-point 2D vectors"""

    data: bpy_prop_collection[Float2AttributeValue]
    """ 

    :type: bpy_prop_collection[Float2AttributeValue]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Float2AttributeValue(bpy_struct):
    """2D Vector value in geometry attribute"""

    vector: mathutils.Vector
    """ 2D vector

    :type: mathutils.Vector
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FloatAttribute(Attribute, bpy_struct):
    """Geometry attribute that stores floating-point values"""

    data: bpy_prop_collection[FloatAttributeValue]
    """ 

    :type: bpy_prop_collection[FloatAttributeValue]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FloatAttributeValue(bpy_struct):
    """Floating-point value in geometry attribute"""

    value: float
    """ 

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FloatColorAttribute(Attribute, bpy_struct):
    """Geometry attribute that stores RGBA colors as floating-point values using 32-bits per channel"""

    data: bpy_prop_collection[FloatColorAttributeValue]
    """ 

    :type: bpy_prop_collection[FloatColorAttributeValue]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FloatColorAttributeValue(bpy_struct):
    """Color value in geometry attribute"""

    color: bpy_prop_array[float]
    """ RGBA color in scene linear color space

    :type: bpy_prop_array[float]
    """

    color_srgb: bpy_prop_array[float]
    """ RGBA color in sRGB color space

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FloatProperty(Property, bpy_struct):
    """RNA floating-point number (single precision) property definition"""

    array_dimensions: bpy_prop_array[int]
    """ Length of each dimension of the array

    :type: bpy_prop_array[int]
    """

    array_length: int
    """ Maximum length of the array, 0 means unlimited

    :type: int
    """

    default: float
    """ Default value for this number

    :type: float
    """

    default_array: bpy_prop_array[float]
    """ Default value for this array

    :type: bpy_prop_array[float]
    """

    hard_max: float
    """ Maximum value used by buttons

    :type: float
    """

    hard_min: float
    """ Minimum value used by buttons

    :type: float
    """

    is_array: bool
    """ 

    :type: bool
    """

    precision: int
    """ Number of digits after the dot used by buttons. Fraction is automatically hidden for exact integer values of fields with unit 'NONE' or 'TIME' (frame count) and step divisible by 100

    :type: int
    """

    soft_max: float
    """ Maximum value used by buttons

    :type: float
    """

    soft_min: float
    """ Minimum value used by buttons

    :type: float
    """

    step: float
    """ Step size used by number buttons, for floats 1/100th of the step size

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FloatVectorAttribute(Attribute, bpy_struct):
    """Geometry attribute that stores floating-point 3D vectors"""

    data: bpy_prop_collection[FloatVectorAttributeValue]
    """ 

    :type: bpy_prop_collection[FloatVectorAttributeValue]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FloatVectorAttributeValue(bpy_struct):
    """Vector value in geometry attribute"""

    vector: mathutils.Vector
    """ 3D vector

    :type: mathutils.Vector
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FloatVectorValueReadOnly(bpy_struct):
    vector: mathutils.Vector
    """ 3D vector

    :type: mathutils.Vector
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FloorConstraint(Constraint, bpy_struct):
    """Use the target object for location limitation"""

    floor_location: typing.Literal[
        "FLOOR_X",
        "FLOOR_Y",
        "FLOOR_Z",
        "FLOOR_NEGATIVE_X",
        "FLOOR_NEGATIVE_Y",
        "FLOOR_NEGATIVE_Z",
    ]
    """ Location of target that object will not pass through

    :type: typing.Literal['FLOOR_X','FLOOR_Y','FLOOR_Z','FLOOR_NEGATIVE_X','FLOOR_NEGATIVE_Y','FLOOR_NEGATIVE_Z']
    """

    offset: float
    """ Offset of floor from object origin

    :type: float
    """

    subtarget: str
    """ Armature bone, mesh or lattice vertex group, ...

    :type: str
    """

    target: Object | None
    """ Target object

    :type: Object | None
    """

    use_rotation: bool
    """ Use the target's rotation to determine floor

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FluidDomainSettings(bpy_struct):
    """Fluid domain settings"""

    adapt_margin: int
    """ Margin added around fluid to minimize boundary interference

    :type: int
    """

    adapt_threshold: float
    """ Minimum amount of fluid a cell can contain before it is considered empty

    :type: float
    """

    additional_res: int
    """ Maximum number of additional cells

    :type: int
    """

    alpha: float
    """ Buoyant force based on smoke density (higher value results in faster rising smoke)

    :type: float
    """

    beta: float
    """ Buoyant force based on smoke heat (higher value results in faster rising smoke)

    :type: float
    """

    burning_rate: float
    """ Speed of the burning reaction (higher value results in smaller flames)

    :type: float
    """

    cache_data_format: typing.Literal["NONE"]
    """ Select the file format to be used for caching volumetric data

    :type: typing.Literal['NONE']
    """

    cache_directory: str
    """ Directory that contains fluid cache files

    :type: str
    """

    cache_frame_end: int
    """ Frame on which the simulation stops. This is the last frame that will be baked

    :type: int
    """

    cache_frame_offset: int
    """ Frame offset that is used when loading the simulation from the cache. It is not considered when baking the simulation, only when loading it

    :type: int
    """

    cache_frame_pause_data: int
    """ 

    :type: int
    """

    cache_frame_pause_guide: int
    """ 

    :type: int
    """

    cache_frame_pause_mesh: int
    """ 

    :type: int
    """

    cache_frame_pause_noise: int
    """ 

    :type: int
    """

    cache_frame_pause_particles: int
    """ 

    :type: int
    """

    cache_frame_start: int
    """ Frame on which the simulation starts. This is the first frame that will be baked

    :type: int
    """

    cache_mesh_format: typing.Literal["NONE"]
    """ Select the file format to be used for caching surface data

    :type: typing.Literal['NONE']
    """

    cache_noise_format: typing.Literal["NONE"]
    """ Select the file format to be used for caching noise data

    :type: typing.Literal['NONE']
    """

    cache_particle_format: typing.Literal["NONE"]
    """ Select the file format to be used for caching particle data

    :type: typing.Literal['NONE']
    """

    cache_resumable: bool
    """ Additional data will be saved so that the bake jobs can be resumed after pausing. Because more data will be written to disk it is recommended to avoid enabling this option when baking at high resolutions

    :type: bool
    """

    cache_type: typing.Literal["REPLAY", "MODULAR", "ALL"]
    """ Change the cache type of the simulation

    :type: typing.Literal['REPLAY','MODULAR','ALL']
    """

    cell_size: mathutils.Vector
    """ Cell Size

    :type: mathutils.Vector
    """

    cfl_condition: float
    """ Maximal velocity per cell (greater CFL numbers will minimize the number of simulation steps and the computation time.)

    :type: float
    """

    clipping: float
    """ Value under which voxels are considered empty space to optimize rendering

    :type: float
    """

    color_grid: bpy_prop_array[float]
    """ Smoke color grid

    :type: bpy_prop_array[float]
    """

    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    color_ramp_field: typing.Literal["NONE"]
    """ Simulation field to color map

    :type: typing.Literal['NONE']
    """

    color_ramp_field_scale: float
    """ Multiplier for scaling the selected field to color map

    :type: float
    """

    delete_in_obstacle: bool
    """ Delete fluid inside obstacles

    :type: bool
    """

    density_grid: bpy_prop_array[float]
    """ Smoke density grid

    :type: bpy_prop_array[float]
    """

    display_interpolation: typing.Literal["LINEAR", "CUBIC", "CLOSEST"]
    """ Interpolation method to use for smoke/fire volumes in solid mode

    :type: typing.Literal['LINEAR','CUBIC','CLOSEST']
    """

    display_thickness: float
    """ Thickness of smoke display in the viewport

    :type: float
    """

    dissolve_speed: int
    """ Determine how quickly the smoke dissolves (lower value makes smoke disappear faster)

    :type: int
    """

    domain_resolution: bpy_prop_array[int]
    """ Smoke Grid Resolution

    :type: bpy_prop_array[int]
    """

    domain_type: typing.Literal["GAS", "LIQUID"]
    """ Change domain type of the simulation

    :type: typing.Literal['GAS','LIQUID']
    """

    effector_group: Collection | None
    """ Limit effectors to this collection

    :type: Collection | None
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    export_manta_script: bool
    """ Generate and export Mantaflow script from current domain settings during bake. This is only needed if you plan to analyze the cache (e.g. view grids, velocity vectors, particles) in Mantaflow directly (outside of Blender) after baking the simulation

    :type: bool
    """

    flame_grid: bpy_prop_array[float]
    """ Smoke flame grid

    :type: bpy_prop_array[float]
    """

    flame_ignition: float
    """ Minimum temperature of the flames (higher value results in faster rising flames)

    :type: float
    """

    flame_max_temp: float
    """ Maximum temperature of the flames (higher value results in faster rising flames)

    :type: float
    """

    flame_smoke: float
    """ Amount of smoke created by burning fuel

    :type: float
    """

    flame_smoke_color: mathutils.Color
    """ Color of smoke emitted from burning fuel

    :type: mathutils.Color
    """

    flame_vorticity: float
    """ Additional vorticity for the flames

    :type: float
    """

    flip_ratio: float
    """ PIC/FLIP Ratio. A value of 1.0 will result in a completely FLIP based simulation. Use a lower value for simulations which should produce smaller splashes

    :type: float
    """

    fluid_group: Collection | None
    """ Limit fluid objects to this collection

    :type: Collection | None
    """

    force_collection: Collection | None
    """ Limit forces to this collection

    :type: Collection | None
    """

    fractions_distance: float
    """ Determines how far apart fluid and obstacle are (higher values will result in fluid being further away from obstacles, smaller values will let fluid move towards the inside of obstacles)

    :type: float
    """

    fractions_threshold: float
    """ Determines how much fluid is allowed in an obstacle cell (higher values will tag a boundary cell as an obstacle easier and reduce the boundary smoothening effect)

    :type: float
    """

    gravity: mathutils.Vector
    """ Gravity in X, Y and Z direction

    :type: mathutils.Vector
    """

    gridlines_cell_filter: typing.Literal[
        "NONE", "FLUID", "OBSTACLE", "EMPTY", "INFLOW", "OUTFLOW"
    ]
    """ Cell type to be highlighted

    :type: typing.Literal['NONE','FLUID','OBSTACLE','EMPTY','INFLOW','OUTFLOW']
    """

    gridlines_color_field: typing.Literal["NONE", "FLAGS", "RANGE"]
    """ Simulation field to color map onto gridlines

    :type: typing.Literal['NONE','FLAGS','RANGE']
    """

    gridlines_lower_bound: float
    """ Lower bound of the highlighting range

    :type: float
    """

    gridlines_range_color: bpy_prop_array[float]
    """ Color used to highlight the range

    :type: bpy_prop_array[float]
    """

    gridlines_upper_bound: float
    """ Upper bound of the highlighting range

    :type: float
    """

    guide_alpha: float
    """ Guiding weight (higher value results in greater lag)

    :type: float
    """

    guide_beta: int
    """ Guiding size (higher value results in larger vortices)

    :type: int
    """

    guide_parent: Object | None
    """ Use velocities from this object for the guiding effect (object needs to have fluid modifier and be of type domain))

    :type: Object | None
    """

    guide_source: typing.Literal["DOMAIN", "EFFECTOR"]
    """ Choose where to get guiding velocities from

    :type: typing.Literal['DOMAIN','EFFECTOR']
    """

    guide_vel_factor: float
    """ Guiding velocity factor (higher value results in greater guiding velocities)

    :type: float
    """

    has_cache_baked_any: bool
    """ 

    :type: bool
    """

    has_cache_baked_data: bool
    """ 

    :type: bool
    """

    has_cache_baked_guide: bool
    """ 

    :type: bool
    """

    has_cache_baked_mesh: bool
    """ 

    :type: bool
    """

    has_cache_baked_noise: bool
    """ 

    :type: bool
    """

    has_cache_baked_particles: bool
    """ 

    :type: bool
    """

    heat_grid: bpy_prop_array[float]
    """ Smoke heat grid

    :type: bpy_prop_array[float]
    """

    highres_sampling: typing.Literal["FULLSAMPLE", "LINEAR", "NEAREST"]
    """ Method for sampling the high resolution flow

    :type: typing.Literal['FULLSAMPLE','LINEAR','NEAREST']
    """

    is_cache_baking_any: bool
    """ 

    :type: bool
    """

    is_cache_baking_data: bool
    """ 

    :type: bool
    """

    is_cache_baking_guide: bool
    """ 

    :type: bool
    """

    is_cache_baking_mesh: bool
    """ 

    :type: bool
    """

    is_cache_baking_noise: bool
    """ 

    :type: bool
    """

    is_cache_baking_particles: bool
    """ 

    :type: bool
    """

    mesh_concave_lower: float
    """ Lower mesh concavity bound (high values tend to smoothen and fill out concave regions)

    :type: float
    """

    mesh_concave_upper: float
    """ Upper mesh concavity bound (high values tend to smoothen and fill out concave regions)

    :type: float
    """

    mesh_generator: typing.Literal["IMPROVED", "UNION"]
    """ Which particle level set generator to use

    :type: typing.Literal['IMPROVED','UNION']
    """

    mesh_particle_radius: float
    """ Particle radius factor (higher value results in larger (meshed) particles). Needs to be adjusted after changing the mesh scale

    :type: float
    """

    mesh_scale: int
    """ The mesh simulation is scaled up by this factor (compared to the base resolution of the domain). For best meshing, it is recommended to adjust the mesh particle radius alongside this value

    :type: int
    """

    mesh_smoothen_neg: int
    """ Negative mesh smoothening

    :type: int
    """

    mesh_smoothen_pos: int
    """ Positive mesh smoothening

    :type: int
    """

    noise_pos_scale: float
    """ Scale of noise (higher value results in larger vortices)

    :type: float
    """

    noise_scale: int
    """ The noise simulation is scaled up by this factor (compared to the base resolution of the domain)

    :type: int
    """

    noise_strength: float
    """ Strength of noise

    :type: float
    """

    noise_time_anim: float
    """ Animation time of noise

    :type: float
    """

    openvdb_cache_compress_type: typing.Literal["ZIP", "BLOSC", "NONE"]
    """ Compression method to be used

    :type: typing.Literal['ZIP','BLOSC','NONE']
    """

    openvdb_data_depth: typing.Literal["NONE"]
    """ Bit depth for fluid particles and grids (lower bit values reduce file size)

    :type: typing.Literal['NONE']
    """

    particle_band_width: float
    """ Particle (narrow) band width (higher value results in thicker band and more particles)

    :type: float
    """

    particle_max: int
    """ Maximum number of particles per cell (ensures that each cell has at most this amount of particles)

    :type: int
    """

    particle_min: int
    """ Minimum number of particles per cell (ensures that each cell has at least this amount of particles)

    :type: int
    """

    particle_number: int
    """ Particle number factor (higher value results in more particles)

    :type: int
    """

    particle_radius: float
    """ Particle radius factor. Increase this value if the simulation appears to leak volume, decrease it if the simulation seems to gain volume

    :type: float
    """

    particle_randomness: float
    """ Randomness factor for particle sampling

    :type: float
    """

    particle_scale: int
    """ The particle simulation is scaled up by this factor (compared to the base resolution of the domain)

    :type: int
    """

    resolution_max: int
    """ Resolution used for the fluid domain. Value corresponds to the longest domain side (resolution for other domain sides is calculated automatically)

    :type: int
    """

    show_gridlines: bool
    """ Show gridlines

    :type: bool
    """

    show_velocity: bool
    """ Visualize vector fields

    :type: bool
    """

    simulation_method: typing.Literal["FLIP", "APIC"]
    """ Change the underlying simulation method

    :type: typing.Literal['FLIP','APIC']
    """

    slice_axis: typing.Literal["AUTO", "X", "Y", "Z"]
    """ 

    :type: typing.Literal['AUTO','X','Y','Z']
    """

    slice_depth: float
    """ Position of the slice

    :type: float
    """

    slice_per_voxel: float
    """ How many slices per voxel should be generated

    :type: float
    """

    sndparticle_boundary: typing.Literal["DELETE", "PUSHOUT"]
    """ How particles that left the domain are treated

    :type: typing.Literal['DELETE','PUSHOUT']
    """

    sndparticle_bubble_buoyancy: float
    """ Amount of buoyancy force that rises bubbles (high value results in bubble movement mainly upwards)

    :type: float
    """

    sndparticle_bubble_drag: float
    """ Amount of drag force that moves bubbles along with the fluid (high value results in bubble movement mainly along with the fluid)

    :type: float
    """

    sndparticle_combined_export: typing.Literal[
        "OFF", "SPRAY_FOAM", "SPRAY_BUBBLES", "FOAM_BUBBLES", "SPRAY_FOAM_BUBBLES"
    ]
    """ Determines which particle systems are created from secondary particles

    :type: typing.Literal['OFF','SPRAY_FOAM','SPRAY_BUBBLES','FOAM_BUBBLES','SPRAY_FOAM_BUBBLES']
    """

    sndparticle_life_max: float
    """ Highest possible particle lifetime

    :type: float
    """

    sndparticle_life_min: float
    """ Lowest possible particle lifetime

    :type: float
    """

    sndparticle_potential_max_energy: float
    """ Upper clamping threshold that indicates the fluid speed where cells no longer emit more particles (higher value results in generally less particles)

    :type: float
    """

    sndparticle_potential_max_trappedair: float
    """ Upper clamping threshold for marking fluid cells where air is trapped (higher value results in less marked cells)

    :type: float
    """

    sndparticle_potential_max_wavecrest: float
    """ Upper clamping threshold for marking fluid cells as wave crests (higher value results in less marked cells)

    :type: float
    """

    sndparticle_potential_min_energy: float
    """ Lower clamping threshold that indicates the fluid speed where cells start to emit particles (lower values result in generally more particles)

    :type: float
    """

    sndparticle_potential_min_trappedair: float
    """ Lower clamping threshold for marking fluid cells where air is trapped (lower value results in more marked cells)

    :type: float
    """

    sndparticle_potential_min_wavecrest: float
    """ Lower clamping threshold for marking fluid cells as wave crests (lower value results in more marked cells)

    :type: float
    """

    sndparticle_potential_radius: int
    """ Radius to compute potential for each cell (higher values are slower but create smoother potential grids)

    :type: int
    """

    sndparticle_sampling_trappedair: int
    """ Maximum number of particles generated per trapped air cell per frame

    :type: int
    """

    sndparticle_sampling_wavecrest: int
    """ Maximum number of particles generated per wave crest cell per frame

    :type: int
    """

    sndparticle_update_radius: int
    """ Radius to compute position update for each particle (higher values are slower but particles move less chaotic)

    :type: int
    """

    start_point: mathutils.Vector
    """ Start point

    :type: mathutils.Vector
    """

    surface_tension: float
    """ Surface tension of liquid (higher value results in greater hydrophobic behavior)

    :type: float
    """

    sys_particle_maximum: int
    """ Maximum number of fluid particles that are allowed in this simulation

    :type: int
    """

    temperature_grid: bpy_prop_array[float]
    """ Smoke temperature grid, range 0 to 1 represents 0 to 1000K

    :type: bpy_prop_array[float]
    """

    time_scale: float
    """ Adjust simulation speed

    :type: float
    """

    timesteps_max: int
    """ Maximum number of simulation steps to perform for one frame

    :type: int
    """

    timesteps_min: int
    """ Minimum number of simulation steps to perform for one frame

    :type: int
    """

    use_adaptive_domain: bool
    """ Adapt simulation resolution and size to fluid

    :type: bool
    """

    use_adaptive_timesteps: bool
    """ 

    :type: bool
    """

    use_bubble_particles: bool
    """ Create bubble particle system

    :type: bool
    """

    use_collision_border_back: bool
    """ Enable collisions with back domain border

    :type: bool
    """

    use_collision_border_bottom: bool
    """ Enable collisions with bottom domain border

    :type: bool
    """

    use_collision_border_front: bool
    """ Enable collisions with front domain border

    :type: bool
    """

    use_collision_border_left: bool
    """ Enable collisions with left domain border

    :type: bool
    """

    use_collision_border_right: bool
    """ Enable collisions with right domain border

    :type: bool
    """

    use_collision_border_top: bool
    """ Enable collisions with top domain border

    :type: bool
    """

    use_color_ramp: bool
    """ Render a simulation field while mapping its voxels values to the colors of a ramp or using a predefined color code

    :type: bool
    """

    use_diffusion: bool
    """ Enable fluid diffusion settings (e.g. viscosity, surface tension)

    :type: bool
    """

    use_dissolve_smoke: bool
    """ Let smoke disappear over time

    :type: bool
    """

    use_dissolve_smoke_log: bool
    """ Dissolve smoke in a logarithmic fashion. Dissolves quickly at first, but lingers longer

    :type: bool
    """

    use_flip_particles: bool
    """ Create liquid particle system

    :type: bool
    """

    use_foam_particles: bool
    """ Create foam particle system

    :type: bool
    """

    use_fractions: bool
    """ Fractional obstacles improve and smoothen the fluid-obstacle boundary

    :type: bool
    """

    use_guide: bool
    """ Enable fluid guiding

    :type: bool
    """

    use_mesh: bool
    """ Enable fluid mesh (using amplification)

    :type: bool
    """

    use_noise: bool
    """ Enable fluid noise (using amplification)

    :type: bool
    """

    use_slice: bool
    """ Perform a single slice of the domain object

    :type: bool
    """

    use_speed_vectors: bool
    """ Caches velocities of mesh vertices. These will be used (automatically) when rendering with motion blur enabled

    :type: bool
    """

    use_spray_particles: bool
    """ Create spray particle system

    :type: bool
    """

    use_tracer_particles: bool
    """ Create tracer particle system

    :type: bool
    """

    use_viscosity: bool
    """ Enable fluid viscosity settings

    :type: bool
    """

    vector_display_type: typing.Literal["NEEDLE", "STREAMLINE", "MAC"]
    """ 

    :type: typing.Literal['NEEDLE','STREAMLINE','MAC']
    """

    vector_field: typing.Literal["FLUID_VELOCITY", "GUIDE_VELOCITY", "FORCE"]
    """ Vector field to be represented by the display vectors

    :type: typing.Literal['FLUID_VELOCITY','GUIDE_VELOCITY','FORCE']
    """

    vector_scale: float
    """ Multiplier for scaling the vectors

    :type: float
    """

    vector_scale_with_magnitude: bool
    """ Scale vectors with their magnitudes

    :type: bool
    """

    vector_show_mac_x: bool
    """ Show X-component of MAC Grid

    :type: bool
    """

    vector_show_mac_y: bool
    """ Show Y-component of MAC Grid

    :type: bool
    """

    vector_show_mac_z: bool
    """ Show Z-component of MAC Grid

    :type: bool
    """

    velocity_grid: bpy_prop_array[float]
    """ Smoke velocity grid

    :type: bpy_prop_array[float]
    """

    velocity_scale: float
    """ Factor to control the amount of motion blur

    :type: float
    """

    viscosity_base: float
    """ Viscosity setting: value that is multiplied by 10 to the power of (exponent*-1)

    :type: float
    """

    viscosity_exponent: int
    """ Negative exponent for the viscosity value (to simplify entering small values e.g. 5*10^-6)

    :type: int
    """

    viscosity_value: float
    """ Viscosity of liquid (higher values result in more viscous fluids, a value of 0 will still apply some viscosity)

    :type: float
    """

    vorticity: float
    """ Amount of turbulence and rotation in smoke

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FluidEffectorSettings(bpy_struct):
    """Smoke collision settings"""

    effector_type: typing.Literal["COLLISION", "GUIDE"]
    """ Change type of effector in the simulation

    :type: typing.Literal['COLLISION','GUIDE']
    """

    guide_mode: typing.Literal["MAXIMUM", "MINIMUM", "OVERRIDE", "AVERAGED"]
    """ How to create guiding velocities

    :type: typing.Literal['MAXIMUM','MINIMUM','OVERRIDE','AVERAGED']
    """

    subframes: int
    """ Number of additional samples to take between frames to improve quality of fast moving effector objects

    :type: int
    """

    surface_distance: float
    """ Additional distance around mesh surface to consider as effector

    :type: float
    """

    use_effector: bool
    """ Control when to apply the effector

    :type: bool
    """

    use_plane_init: bool
    """ Treat this object as a planar, unclosed mesh

    :type: bool
    """

    velocity_factor: float
    """ Multiplier of obstacle velocity

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FluidFlowSettings(bpy_struct):
    """Fluid flow settings"""

    density: float
    """ 

    :type: float
    """

    density_vertex_group: str
    """ Name of vertex group which determines surface emission rate

    :type: str
    """

    flow_behavior: typing.Literal["INFLOW", "OUTFLOW", "GEOMETRY"]
    """ Change flow behavior in the simulation

    :type: typing.Literal['INFLOW','OUTFLOW','GEOMETRY']
    """

    flow_source: typing.Literal["NONE"]
    """ Change how fluid is emitted

    :type: typing.Literal['NONE']
    """

    flow_type: typing.Literal["SMOKE", "BOTH", "FIRE", "LIQUID"]
    """ Change type of fluid in the simulation

    :type: typing.Literal['SMOKE','BOTH','FIRE','LIQUID']
    """

    fuel_amount: float
    """ 

    :type: float
    """

    noise_texture: Texture | None
    """ Texture that controls emission strength

    :type: Texture | None
    """

    particle_size: float
    """ Particle size in simulation cells

    :type: float
    """

    particle_system: ParticleSystem | None
    """ Particle systems emitted from the object

    :type: ParticleSystem | None
    """

    smoke_color: mathutils.Color
    """ Color of smoke

    :type: mathutils.Color
    """

    subframes: int
    """ Number of additional samples to take between frames to improve quality of fast moving flows

    :type: int
    """

    surface_distance: float
    """ Controls fluid emission from the mesh surface (higher value results in emission further away from the mesh surface

    :type: float
    """

    temperature: float
    """ Temperature difference to ambient temperature

    :type: float
    """

    texture_map_type: typing.Literal["AUTO", "UV"]
    """ Texture mapping type

    :type: typing.Literal['AUTO','UV']
    """

    texture_offset: float
    """ Z-offset of texture mapping

    :type: float
    """

    texture_size: float
    """ Size of texture mapping

    :type: float
    """

    use_absolute: bool
    """ Only allow given density value in emitter area and will not add up

    :type: bool
    """

    use_inflow: bool
    """ Control when to apply fluid flow

    :type: bool
    """

    use_initial_velocity: bool
    """ Fluid has some initial velocity when it is emitted

    :type: bool
    """

    use_particle_size: bool
    """ Set particle size in simulation cells or use nearest cell

    :type: bool
    """

    use_plane_init: bool
    """ Treat this object as a planar and unclosed mesh. Fluid will only be emitted from the mesh surface and based on the surface emission value

    :type: bool
    """

    use_texture: bool
    """ Use a texture to control emission strength

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    velocity_coord: mathutils.Vector
    """ Additional initial velocity in X, Y and Z direction (added to source velocity)

    :type: mathutils.Vector
    """

    velocity_factor: float
    """ Multiplier of source velocity passed to fluid (source velocity is non-zero only if object is moving)

    :type: float
    """

    velocity_normal: float
    """ Amount of normal directional velocity

    :type: float
    """

    velocity_random: float
    """ Amount of random velocity

    :type: float
    """

    volume_density: float
    """ Controls fluid emission from within the mesh (higher value results in greater emissions from inside the mesh)

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FluidModifier(Modifier, bpy_struct):
    """Fluid simulation modifier"""

    domain_settings: FluidDomainSettings
    """ 

    :type: FluidDomainSettings
    """

    effector_settings: FluidEffectorSettings
    """ 

    :type: FluidEffectorSettings
    """

    flow_settings: FluidFlowSettings
    """ 

    :type: FluidFlowSettings
    """

    fluid_type: typing.Literal["NONE", "DOMAIN", "FLOW", "EFFECTOR"]
    """ 

    :type: typing.Literal['NONE','DOMAIN','FLOW','EFFECTOR']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FollowPathConstraint(Constraint, bpy_struct):
    """Lock motion to the target path"""

    forward_axis: typing.Literal[
        "FORWARD_X",
        "FORWARD_Y",
        "FORWARD_Z",
        "TRACK_NEGATIVE_X",
        "TRACK_NEGATIVE_Y",
        "TRACK_NEGATIVE_Z",
    ]
    """ Axis that points forward along the path

    :type: typing.Literal['FORWARD_X','FORWARD_Y','FORWARD_Z','TRACK_NEGATIVE_X','TRACK_NEGATIVE_Y','TRACK_NEGATIVE_Z']
    """

    offset: float
    """ Offset from the position corresponding to the time frame

    :type: float
    """

    offset_factor: float
    """ Percentage value defining target position along length of curve

    :type: float
    """

    target: Object | None
    """ Target Curve object

    :type: Object | None
    """

    up_axis: typing.Literal["UP_X", "UP_Y", "UP_Z"]
    """ Axis that points upward

    :type: typing.Literal['UP_X','UP_Y','UP_Z']
    """

    use_curve_follow: bool
    """ Object will follow the heading and banking of the curve

    :type: bool
    """

    use_curve_radius: bool
    """ Object is scaled by the curve radius

    :type: bool
    """

    use_fixed_location: bool
    """ Object will stay locked to a single point somewhere along the length of the curve regardless of time

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FollowTrackConstraint(Constraint, bpy_struct):
    """Lock motion to the target motion track"""

    camera: Object | None
    """ Camera to which motion is parented (if empty active scene camera is used)

    :type: Object | None
    """

    clip: MovieClip | None
    """ Movie Clip to get tracking data from

    :type: MovieClip | None
    """

    depth_object: Object | None
    """ Object used to define depth in camera space by projecting onto surface of this object

    :type: Object | None
    """

    frame_method: typing.Literal["STRETCH", "FIT", "CROP"]
    """ How the footage fits in the camera frame

    :type: typing.Literal['STRETCH','FIT','CROP']
    """

    object: str
    """ Movie tracking object to follow (if empty, camera object is used)

    :type: str
    """

    track: str
    """ Movie tracking track to follow

    :type: str
    """

    use_3d_position: bool
    """ Use 3D position of track to parent to

    :type: bool
    """

    use_active_clip: bool
    """ Use active clip defined in scene

    :type: bool
    """

    use_undistorted_position: bool
    """ Parent to undistorted position of 2D track

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FreestyleLineSet(bpy_struct):
    """Line set for associating lines and style parameters"""

    collection: Collection | None
    """ A collection of objects based on which feature edges are selected

    :type: Collection | None
    """

    collection_negation: typing.Literal["INCLUSIVE", "EXCLUSIVE"]
    """ Specify either inclusion or exclusion of feature edges belonging to a collection of objects

    :type: typing.Literal['INCLUSIVE','EXCLUSIVE']
    """

    edge_type_combination: typing.Literal["OR", "AND"]
    """ Specify a logical combination of selection conditions on feature edge types

    :type: typing.Literal['OR','AND']
    """

    edge_type_negation: typing.Literal["INCLUSIVE", "EXCLUSIVE"]
    """ Specify either inclusion or exclusion of feature edges selected by edge types

    :type: typing.Literal['INCLUSIVE','EXCLUSIVE']
    """

    exclude_border: bool
    """ Exclude border edges

    :type: bool
    """

    exclude_contour: bool
    """ Exclude contours

    :type: bool
    """

    exclude_crease: bool
    """ Exclude crease edges

    :type: bool
    """

    exclude_edge_mark: bool
    """ Exclude edge marks

    :type: bool
    """

    exclude_external_contour: bool
    """ Exclude external contours

    :type: bool
    """

    exclude_material_boundary: bool
    """ Exclude edges at material boundaries

    :type: bool
    """

    exclude_ridge_valley: bool
    """ Exclude ridges and valleys

    :type: bool
    """

    exclude_silhouette: bool
    """ Exclude silhouette edges

    :type: bool
    """

    exclude_suggestive_contour: bool
    """ Exclude suggestive contours

    :type: bool
    """

    face_mark_condition: typing.Literal["ONE", "BOTH"]
    """ Specify a feature edge selection condition based on face marks

    :type: typing.Literal['ONE','BOTH']
    """

    face_mark_negation: typing.Literal["INCLUSIVE", "EXCLUSIVE"]
    """ Specify either inclusion or exclusion of feature edges selected by face marks

    :type: typing.Literal['INCLUSIVE','EXCLUSIVE']
    """

    linestyle: FreestyleLineStyle
    """ Line style settings

    :type: FreestyleLineStyle
    """

    name: str
    """ Line set name

    :type: str
    """

    qi_end: int
    """ Last QI value of the QI range

    :type: int
    """

    qi_start: int
    """ First QI value of the QI range

    :type: int
    """

    select_border: bool
    """ Select border edges (open mesh edges)

    :type: bool
    """

    select_by_collection: bool
    """ Select feature edges based on a collection of objects

    :type: bool
    """

    select_by_edge_types: bool
    """ Select feature edges based on edge types

    :type: bool
    """

    select_by_face_marks: bool
    """ Select feature edges by face marks

    :type: bool
    """

    select_by_image_border: bool
    """ Select feature edges by image border (less memory consumption)

    :type: bool
    """

    select_by_visibility: bool
    """ Select feature edges based on visibility

    :type: bool
    """

    select_contour: bool
    """ Select contours (outer silhouettes of each object)

    :type: bool
    """

    select_crease: bool
    """ Select crease edges (those between two faces making an angle smaller than the Crease Angle)

    :type: bool
    """

    select_edge_mark: bool
    """ Select edge marks (edges annotated by Freestyle edge marks)

    :type: bool
    """

    select_external_contour: bool
    """ Select external contours (outer silhouettes of occluding and occluded objects)

    :type: bool
    """

    select_material_boundary: bool
    """ Select edges at material boundaries

    :type: bool
    """

    select_ridge_valley: bool
    """ Select ridges and valleys (boundary lines between convex and concave areas of surface)

    :type: bool
    """

    select_silhouette: bool
    """ Select silhouettes (edges at the boundary of visible and hidden faces)

    :type: bool
    """

    select_suggestive_contour: bool
    """ Select suggestive contours (almost silhouette/contour edges)

    :type: bool
    """

    show_render: bool
    """ Enable or disable this line set during stroke rendering

    :type: bool
    """

    visibility: typing.Literal["VISIBLE", "HIDDEN", "RANGE"]
    """ Determine how to use visibility for feature edge selection

    :type: typing.Literal['VISIBLE','HIDDEN','RANGE']
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FreestyleLineStyle(ID, bpy_struct):
    """Freestyle line style, reusable by multiple line sets"""

    active_texture: Texture | None
    """ Active texture slot being displayed

    :type: Texture | None
    """

    active_texture_index: int | None
    """ Index of active texture slot

    :type: int | None
    """

    alpha: float
    """ Base alpha transparency, possibly modified by alpha transparency modifiers

    :type: float
    """

    alpha_modifiers: LineStyleAlphaModifiers
    """ List of alpha transparency modifiers

    :type: LineStyleAlphaModifiers
    """

    angle_max: float
    """ Maximum 2D angle for splitting chains

    :type: float
    """

    angle_min: float
    """ Minimum 2D angle for splitting chains

    :type: float
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    caps: typing.Literal["BUTT", "ROUND", "SQUARE"]
    """ Select the shape of both ends of strokes

    :type: typing.Literal['BUTT','ROUND','SQUARE']
    """

    chain_count: int
    """ Chain count for the selection of first N chains

    :type: int
    """

    chaining: typing.Literal["PLAIN", "SKETCHY"]
    """ Select the way how feature edges are jointed to form chains

    :type: typing.Literal['PLAIN','SKETCHY']
    """

    color: mathutils.Color
    """ Base line color, possibly modified by line color modifiers

    :type: mathutils.Color
    """

    color_modifiers: LineStyleColorModifiers
    """ List of line color modifiers

    :type: LineStyleColorModifiers
    """

    dash1: int
    """ Length of the 1st dash for dashed lines

    :type: int
    """

    dash2: int
    """ Length of the 2nd dash for dashed lines

    :type: int
    """

    dash3: int
    """ Length of the 3rd dash for dashed lines

    :type: int
    """

    gap1: int
    """ Length of the 1st gap for dashed lines

    :type: int
    """

    gap2: int
    """ Length of the 2nd gap for dashed lines

    :type: int
    """

    gap3: int
    """ Length of the 3rd gap for dashed lines

    :type: int
    """

    geometry_modifiers: LineStyleGeometryModifiers
    """ List of stroke geometry modifiers

    :type: LineStyleGeometryModifiers
    """

    integration_type: typing.Literal["MEAN", "MIN", "MAX", "FIRST", "LAST"]
    """ Select the way how the sort key is computed for each chain

    :type: typing.Literal['MEAN','MIN','MAX','FIRST','LAST']
    """

    length_max: float
    """ Maximum curvilinear 2D length for the selection of chains

    :type: float
    """

    length_min: float
    """ Minimum curvilinear 2D length for the selection of chains

    :type: float
    """

    material_boundary: bool
    """ If true, chains of feature edges are split at material boundaries

    :type: bool
    """

    node_tree: ShaderNodeTree | None
    """ Node tree for node-based shaders

    :type: ShaderNodeTree | None
    """

    panel: typing.Literal[
        "STROKES", "COLOR", "ALPHA", "THICKNESS", "GEOMETRY", "TEXTURE"
    ]
    """ Select the property panel to be shown

    :type: typing.Literal['STROKES','COLOR','ALPHA','THICKNESS','GEOMETRY','TEXTURE']
    """

    rounds: int
    """ Number of rounds in a sketchy multiple touch

    :type: int
    """

    sort_key: typing.Literal[
        "DISTANCE_FROM_CAMERA", "2D_LENGTH", "PROJECTED_X", "PROJECTED_Y"
    ]
    """ Select the sort key to determine the stacking order of chains

    :type: typing.Literal['DISTANCE_FROM_CAMERA','2D_LENGTH','PROJECTED_X','PROJECTED_Y']
    """

    sort_order: typing.Literal["DEFAULT", "REVERSE"]
    """ Select the sort order

    :type: typing.Literal['DEFAULT','REVERSE']
    """

    split_dash1: int
    """ Length of the 1st dash for splitting

    :type: int
    """

    split_dash2: int
    """ Length of the 2nd dash for splitting

    :type: int
    """

    split_dash3: int
    """ Length of the 3rd dash for splitting

    :type: int
    """

    split_gap1: int
    """ Length of the 1st gap for splitting

    :type: int
    """

    split_gap2: int
    """ Length of the 2nd gap for splitting

    :type: int
    """

    split_gap3: int
    """ Length of the 3rd gap for splitting

    :type: int
    """

    split_length: float
    """ Curvilinear 2D length for chain splitting

    :type: float
    """

    texture_slots: LineStyleTextureSlots
    """ Texture slots defining the mapping and influence of textures

    :type: LineStyleTextureSlots
    """

    texture_spacing: float
    """ Spacing for textures along stroke length

    :type: float
    """

    thickness: float
    """ Base line thickness, possibly modified by line thickness modifiers

    :type: float
    """

    thickness_modifiers: LineStyleThicknessModifiers
    """ List of line thickness modifiers

    :type: LineStyleThicknessModifiers
    """

    thickness_position: typing.Literal["CENTER", "INSIDE", "OUTSIDE", "RELATIVE"]
    """ Thickness position of silhouettes and border edges (applicable when plain chaining is used with the Same Object option)

    :type: typing.Literal['CENTER','INSIDE','OUTSIDE','RELATIVE']
    """

    thickness_ratio: float
    """ A number between 0 (inside) and 1 (outside) specifying the relative position of stroke thickness

    :type: float
    """

    use_angle_max: bool
    """ Split chains at points with angles larger than the maximum 2D angle

    :type: bool
    """

    use_angle_min: bool
    """ Split chains at points with angles smaller than the minimum 2D angle

    :type: bool
    """

    use_chain_count: bool
    """ Enable the selection of first N chains

    :type: bool
    """

    use_chaining: bool
    """ Enable chaining of feature edges

    :type: bool
    """

    use_dashed_line: bool
    """ Enable or disable dashed line

    :type: bool
    """

    use_length_max: bool
    """ Enable the selection of chains by a maximum 2D length

    :type: bool
    """

    use_length_min: bool
    """ Enable the selection of chains by a minimum 2D length

    :type: bool
    """

    use_nodes: bool
    """ Use shader nodes for the line style

    :type: bool
    """

    use_same_object: bool
    """ If true, only feature edges of the same object are joined

    :type: bool
    """

    use_sorting: bool
    """ Arrange the stacking order of strokes

    :type: bool
    """

    use_split_length: bool
    """ Enable chain splitting by curvilinear 2D length

    :type: bool
    """

    use_split_pattern: bool
    """ Enable chain splitting by dashed line patterns

    :type: bool
    """

    use_texture: bool
    """ Enable or disable textured strokes

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FreestyleModuleSettings(bpy_struct):
    """Style module configuration for specifying a style module"""

    script: Text | None
    """ Python script to define a style module

    :type: Text | None
    """

    use: bool
    """ Enable or disable this style module during stroke rendering

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FreestyleSettings(bpy_struct):
    """Freestyle settings for a ViewLayer data-block"""

    as_render_pass: bool
    """ Renders Freestyle output to a separate pass instead of overlaying it on the Combined pass

    :type: bool
    """

    crease_angle: float
    """ Angular threshold for detecting crease edges

    :type: float
    """

    kr_derivative_epsilon: float
    """ Kr derivative epsilon for computing suggestive contours

    :type: float
    """

    linesets: Linesets
    """ 

    :type: Linesets
    """

    mode: typing.Literal["SCRIPT", "EDITOR"]
    """ Select the Freestyle control mode

    :type: typing.Literal['SCRIPT','EDITOR']
    """

    modules: FreestyleModules
    """ A list of style modules (to be applied from top to bottom)

    :type: FreestyleModules
    """

    sphere_radius: float
    """ Sphere radius for computing curvatures

    :type: float
    """

    use_culling: bool
    """ If enabled, out-of-view edges are ignored

    :type: bool
    """

    use_material_boundaries: bool
    """ Enable material boundaries

    :type: bool
    """

    use_ridges_and_valleys: bool
    """ Enable ridges and valleys

    :type: bool
    """

    use_smoothness: bool
    """ Take face smoothness into account in view map calculation

    :type: bool
    """

    use_suggestive_contours: bool
    """ Enable suggestive contours

    :type: bool
    """

    use_view_map_cache: bool
    """ Keep the computed view map and avoid recalculating it if mesh geometry is unchanged

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class Function(bpy_struct):
    """RNA function definition"""

    description: str
    """ Description of the Function's purpose

    :type: str
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    is_registered: bool
    """ Function is registered as callback as part of type registration

    :type: bool
    """

    is_registered_optional: bool
    """ Function is optionally registered as callback part of type registration

    :type: bool
    """

    parameters: bpy_prop_collection[Property]
    """ Parameters for the function

    :type: bpy_prop_collection[Property]
    """

    use_self: bool
    """ Function does not pass itself as an argument (becomes a static method in Python)

    :type: bool
    """

    use_self_type: bool
    """ Function passes itself type as an argument (becomes a class method in Python if use_self is false)

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNode(NodeInternal, Node, bpy_struct):
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeAlignEulerToVector(FunctionNode, NodeInternal, Node, bpy_struct):
    axis: typing.Literal["X", "Y", "Z"]
    """ Axis to align to the vector

    :type: typing.Literal['X','Y','Z']
    """

    pivot_axis: typing.Literal["AUTO", "X", "Y", "Z"]
    """ Axis to rotate around

    :type: typing.Literal['AUTO','X','Y','Z']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeAxisAngleToRotation(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeBooleanMath(FunctionNode, NodeInternal, Node, bpy_struct):
    operation: bpy._typing.rna_enums.NodeBooleanMathItems
    """ 

    :type: bpy._typing.rna_enums.NodeBooleanMathItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeCombineColor(FunctionNode, NodeInternal, Node, bpy_struct):
    mode: bpy._typing.rna_enums.NodeCombsepColorItems
    """ Mode of color processing

    :type: bpy._typing.rna_enums.NodeCombsepColorItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeCompare(FunctionNode, NodeInternal, Node, bpy_struct):
    data_type: bpy._typing.rna_enums.NodeSocketDataTypeItems
    """ 

    :type: bpy._typing.rna_enums.NodeSocketDataTypeItems
    """

    mode: typing.Literal["ELEMENT", "LENGTH", "AVERAGE", "DOT_PRODUCT", "DIRECTION"]
    """ 

    :type: typing.Literal['ELEMENT','LENGTH','AVERAGE','DOT_PRODUCT','DIRECTION']
    """

    operation: bpy._typing.rna_enums.NodeCompareOperationItems
    """ 

    :type: bpy._typing.rna_enums.NodeCompareOperationItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeEulerToRotation(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeFloatToInt(FunctionNode, NodeInternal, Node, bpy_struct):
    rounding_mode: bpy._typing.rna_enums.NodeFloatToIntItems
    """ Method used to convert the float to an integer

    :type: bpy._typing.rna_enums.NodeFloatToIntItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeInputBool(FunctionNode, NodeInternal, Node, bpy_struct):
    boolean: bool
    """ Input value used for unconnected socket

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeInputColor(FunctionNode, NodeInternal, Node, bpy_struct):
    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeInputInt(FunctionNode, NodeInternal, Node, bpy_struct):
    integer: int
    """ Input value used for unconnected socket

    :type: int
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeInputSpecialCharacters(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeInputString(FunctionNode, NodeInternal, Node, bpy_struct):
    string: str
    """ 

    :type: str
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeInputVector(FunctionNode, NodeInternal, Node, bpy_struct):
    vector: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeInvertRotation(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeQuaternionToRotation(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeRandomValue(FunctionNode, NodeInternal, Node, bpy_struct):
    data_type: bpy._typing.rna_enums.AttributeTypeItems
    """ Type of data stored in attribute

    :type: bpy._typing.rna_enums.AttributeTypeItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeReplaceString(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeRotateEuler(FunctionNode, NodeInternal, Node, bpy_struct):
    space: typing.Literal["OBJECT", "LOCAL"]
    """ Base orientation for rotation

    :type: typing.Literal['OBJECT','LOCAL']
    """

    type: typing.Literal["AXIS_ANGLE", "EULER"]
    """ Method used to describe the rotation

    :type: typing.Literal['AXIS_ANGLE','EULER']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeRotateVector(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeRotationToAxisAngle(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeRotationToEuler(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeRotationToQuaternion(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeSeparateColor(FunctionNode, NodeInternal, Node, bpy_struct):
    mode: bpy._typing.rna_enums.NodeCombsepColorItems
    """ Mode of color processing

    :type: bpy._typing.rna_enums.NodeCombsepColorItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeSliceString(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeStringLength(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class FunctionNodeValueToString(FunctionNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPENCIL_UL_annotation_layer(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        _data,
        item,
        icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param _data:
        :param item:
        :param icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPENCIL_UL_layer(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        _data,
        item,
        icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param _data:
        :param item:
        :param icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPENCIL_UL_masks(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        _data,
        item,
        icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param _data:
        :param item:
        :param icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPENCIL_UL_matslots(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        _data,
        item,
        icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param _data:
        :param item:
        :param icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPENCIL_UL_vgroups(UIList, bpy_struct):
    def draw_item(
        self,
        _context,
        layout,
        _data,
        item,
        icon,
        _active_data,
        _active_propname,
        _index,
    ):
        """

        :param _context:
        :param layout:
        :param _data:
        :param item:
        :param icon:
        :param _active_data:
        :param _active_propname:
        :param _index:
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilEditCurve(bpy_struct):
    """Edition Curve"""

    curve_points: bpy_prop_collection[GPencilEditCurvePoint]
    """ Curve data points

    :type: bpy_prop_collection[GPencilEditCurvePoint]
    """

    select: bool
    """ Curve is selected for viewport editing

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilEditCurvePoint(bpy_struct):
    """Bezier curve point with two handles"""

    co: mathutils.Vector
    """ Coordinates of the control point

    :type: mathutils.Vector
    """

    handle_left: mathutils.Vector
    """ Coordinates of the first handle

    :type: mathutils.Vector
    """

    handle_right: mathutils.Vector
    """ Coordinates of the second handle

    :type: mathutils.Vector
    """

    hide: bool
    """ Visibility status

    :type: bool
    """

    point_index: int
    """ Index of the corresponding grease pencil stroke point

    :type: int
    """

    pressure: float
    """ Pressure of the grease pencil stroke point

    :type: float
    """

    select_control_point: bool
    """ Control point selection status

    :type: bool
    """

    select_left_handle: bool
    """ Handle 1 selection status

    :type: bool
    """

    select_right_handle: bool
    """ Handle 2 selection status

    :type: bool
    """

    strength: float
    """ Color intensity (alpha factor) of the grease pencil stroke point

    :type: float
    """

    uv_factor: float
    """ Internal UV factor

    :type: float
    """

    uv_rotation: float
    """ Internal UV factor for dot mode

    :type: float
    """

    vertex_color: bpy_prop_array[float]
    """ Vertex color of the grease pencil stroke point

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilFrame(bpy_struct):
    """Collection of related sketches on a particular frame"""

    frame_number: int
    """ The frame on which this sketch appears

    :type: int
    """

    is_edited: bool
    """ Frame is being edited (painted on)

    :type: bool
    """

    keyframe_type: typing.Literal[
        "KEYFRAME", "BREAKDOWN", "MOVING_HOLD", "EXTREME", "JITTER"
    ]
    """ Type of keyframe

    :type: typing.Literal['KEYFRAME','BREAKDOWN','MOVING_HOLD','EXTREME','JITTER']
    """

    select: bool
    """ Frame is selected for editing in the Dope Sheet

    :type: bool
    """

    strokes: GPencilStrokes
    """ Freehand curves defining the sketch on this frame

    :type: GPencilStrokes
    """

    def clear(self):
        """Remove all the grease pencil frame data"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilInterpolateSettings(bpy_struct):
    """Settings for Grease Pencil interpolation tools"""

    interpolation_curve: CurveMapping
    """ Custom curve to control 'sequence' interpolation between Grease Pencil frames

    :type: CurveMapping
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilLayer(bpy_struct):
    """Collection of related sketches"""

    active_frame: GPencilFrame | None
    """ Frame currently being displayed for this layer

    :type: GPencilFrame | None
    """

    annotation_hide: bool
    """ Set annotation Visibility

    :type: bool
    """

    annotation_onion_after_color: mathutils.Color
    """ Base color for ghosts after the active frame

    :type: mathutils.Color
    """

    annotation_onion_after_range: int
    """ Maximum number of frames to show after current frame

    :type: int
    """

    annotation_onion_before_color: mathutils.Color
    """ Base color for ghosts before the active frame

    :type: mathutils.Color
    """

    annotation_onion_before_range: int
    """ Maximum number of frames to show before current frame

    :type: int
    """

    annotation_opacity: float
    """ Annotation Layer Opacity

    :type: float
    """

    blend_mode: typing.Literal[
        "REGULAR", "HARDLIGHT", "ADD", "SUBTRACT", "MULTIPLY", "DIVIDE"
    ]
    """ Blend mode

    :type: typing.Literal['REGULAR','HARDLIGHT','ADD','SUBTRACT','MULTIPLY','DIVIDE']
    """

    channel_color: mathutils.Color
    """ Custom color for animation channel in Dopesheet

    :type: mathutils.Color
    """

    color: mathutils.Color
    """ Color for all strokes in this layer

    :type: mathutils.Color
    """

    frames: GPencilFrames
    """ Sketches for this layer on different frames

    :type: GPencilFrames
    """

    hide: bool
    """ Set layer Visibility

    :type: bool
    """

    info: str
    """ Layer name

    :type: str
    """

    is_parented: bool
    """ True when the layer parent object is set

    :type: bool
    """

    is_ruler: bool
    """ This is a special ruler layer

    :type: bool
    """

    line_change: int
    """ Thickness change to apply to current strokes (in pixels)

    :type: int
    """

    location: mathutils.Vector
    """ Values for change location

    :type: mathutils.Vector
    """

    lock: bool
    """ Protect layer from further editing and/or frame changes

    :type: bool
    """

    lock_frame: bool
    """ Lock current frame displayed by layer

    :type: bool
    """

    lock_material: bool
    """ Avoids editing locked materials in the layer

    :type: bool
    """

    mask_layers: GreasePencilMaskLayers
    """ List of Masking Layers

    :type: GreasePencilMaskLayers
    """

    matrix_inverse: mathutils.Matrix
    """ Parent inverse transformation matrix

    :type: mathutils.Matrix
    """

    matrix_inverse_layer: mathutils.Matrix
    """ Local Layer transformation inverse matrix

    :type: mathutils.Matrix
    """

    matrix_layer: mathutils.Matrix
    """ Local Layer transformation matrix

    :type: mathutils.Matrix
    """

    opacity: float
    """ Layer Opacity

    :type: float
    """

    parent: Object | None
    """ Parent object

    :type: Object | None
    """

    parent_bone: str
    """ Name of parent bone in case of a bone parenting relation

    :type: str
    """

    parent_type: typing.Literal["OBJECT", "ARMATURE", "BONE"]
    """ Type of parent relation

    :type: typing.Literal['OBJECT','ARMATURE','BONE']
    """

    pass_index: int
    """ Index number for the "Layer Index" pass

    :type: int
    """

    rotation: mathutils.Euler
    """ Values for changes in rotation

    :type: mathutils.Euler
    """

    scale: mathutils.Vector
    """ Values for changes in scale

    :type: mathutils.Vector
    """

    select: bool
    """ Layer is selected for editing in the Dope Sheet

    :type: bool
    """

    show_in_front: bool
    """ Make the layer display in front of objects

    :type: bool
    """

    show_points: bool
    """ Show the points which make up the strokes (for debugging purposes)

    :type: bool
    """

    thickness: int
    """ Thickness of annotation strokes

    :type: int
    """

    tint_color: mathutils.Color
    """ Color for tinting stroke colors

    :type: mathutils.Color
    """

    tint_factor: float
    """ Factor of tinting color

    :type: float
    """

    use_annotation_onion_skinning: bool
    """ Display annotation onion skins before and after the current frame

    :type: bool
    """

    use_lights: bool
    """ Enable the use of lights on stroke and fill materials

    :type: bool
    """

    use_mask_layer: bool
    """ The visibility of drawings on this layer is affected by the layers in its masks list

    :type: bool
    """

    use_onion_skinning: bool
    """ Display onion skins before and after the current frame

    :type: bool
    """

    use_solo_mode: bool
    """ In Draw Mode only display layers with keyframe in current frame

    :type: bool
    """

    use_viewlayer_masks: bool
    """ Include the mask layers when rendering the view-layer

    :type: bool
    """

    vertex_paint_opacity: float
    """ Vertex Paint mix factor

    :type: float
    """

    viewlayer_render: str
    """ Only include Layer in this View Layer render output (leave blank to include always)

    :type: str
    """

    def clear(self):
        """Remove all the grease pencil layer data"""

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilLayerMask(bpy_struct):
    """List of Mask Layers"""

    hide: bool
    """ Set mask Visibility

    :type: bool
    """

    invert: bool
    """ Invert mask

    :type: bool
    """

    name: str
    """ Mask layer name

    :type: str
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilSculptGuide(bpy_struct):
    """Guides for drawing"""

    angle: float
    """ Direction of lines

    :type: float
    """

    angle_snap: float
    """ Angle snapping

    :type: float
    """

    location: bpy_prop_array[float]
    """ Custom reference point for guides

    :type: bpy_prop_array[float]
    """

    reference_object: Object | None
    """ Object used for reference point

    :type: Object | None
    """

    reference_point: typing.Literal["CURSOR", "CUSTOM", "OBJECT"]
    """ Type of speed guide

    :type: typing.Literal['CURSOR','CUSTOM','OBJECT']
    """

    spacing: float
    """ Guide spacing

    :type: float
    """

    type: typing.Literal["CIRCULAR", "RADIAL", "PARALLEL", "GRID", "ISO"]
    """ Type of speed guide

    :type: typing.Literal['CIRCULAR','RADIAL','PARALLEL','GRID','ISO']
    """

    use_guide: bool
    """ Enable speed guides

    :type: bool
    """

    use_snapping: bool
    """ Enable snapping to guides angle or spacing options

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilSculptSettings(bpy_struct):
    """General properties for Grease Pencil stroke sculpting tools"""

    guide: GPencilSculptGuide
    """ 

    :type: GPencilSculptGuide
    """

    intersection_threshold: float
    """ Threshold for stroke intersections

    :type: float
    """

    lock_axis: typing.Literal["VIEW", "AXIS_Y", "AXIS_X", "AXIS_Z", "CURSOR"]
    """ 

    :type: typing.Literal['VIEW','AXIS_Y','AXIS_X','AXIS_Z','CURSOR']
    """

    multiframe_falloff_curve: CurveMapping
    """ Custom curve to control falloff of brush effect by Grease Pencil frames

    :type: CurveMapping
    """

    thickness_primitive_curve: CurveMapping
    """ Custom curve to control primitive thickness

    :type: CurveMapping
    """

    use_automasking_layer_active: bool
    """ Affect only the Active Layer

    :type: bool
    """

    use_automasking_layer_stroke: bool
    """ Affect only strokes below the cursor

    :type: bool
    """

    use_automasking_material_active: bool
    """ Affect only the Active Material

    :type: bool
    """

    use_automasking_material_stroke: bool
    """ Affect only strokes below the cursor

    :type: bool
    """

    use_automasking_stroke: bool
    """ Affect only strokes below the cursor

    :type: bool
    """

    use_multiframe_falloff: bool
    """ Use falloff effect when edit in multiframe mode to compute brush effect by frame

    :type: bool
    """

    use_scale_thickness: bool
    """ Scale the stroke thickness when transforming strokes

    :type: bool
    """

    use_thickness_curve: bool
    """ Use curve to define primitive stroke thickness

    :type: bool
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilStroke(bpy_struct):
    """Freehand curve defining part of a sketch"""

    aspect: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    bound_box_max: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    bound_box_min: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    display_mode: typing.Literal["SCREEN", "3DSPACE", "2DSPACE", "2DIMAGE"]
    """ Coordinate space that stroke is in

    :type: typing.Literal['SCREEN','3DSPACE','2DSPACE','2DIMAGE']
    """

    edit_curve: GPencilEditCurve
    """ Temporary data for Edit Curve

    :type: GPencilEditCurve
    """

    end_cap_mode: typing.Literal["ROUND", "FLAT"]
    """ Stroke end extreme cap style

    :type: typing.Literal['ROUND','FLAT']
    """

    hardness: float
    """ Amount of gradient along section of stroke

    :type: float
    """

    has_edit_curve: bool
    """ Stroke has Curve data to edit shape

    :type: bool
    """

    is_nofill_stroke: bool
    """ Special stroke to use as boundary for filling areas

    :type: bool
    """

    line_width: int
    """ Thickness of stroke (in pixels)

    :type: int
    """

    material_index: int
    """ Material slot index of this stroke

    :type: int
    """

    points: GPencilStrokePoints
    """ Stroke data points

    :type: GPencilStrokePoints
    """

    select: bool
    """ Stroke is selected for viewport editing

    :type: bool
    """

    select_index: int
    """ Index of selection used for interpolation

    :type: int
    """

    start_cap_mode: typing.Literal["ROUND", "FLAT"]
    """ Stroke start extreme cap style

    :type: typing.Literal['ROUND','FLAT']
    """

    time_start: float
    """ Initial time of the stroke

    :type: float
    """

    triangles: bpy_prop_collection[GPencilTriangle]
    """ Triangulation data for HQ fill

    :type: bpy_prop_collection[GPencilTriangle]
    """

    use_cyclic: bool
    """ Enable cyclic drawing, closing the stroke

    :type: bool
    """

    uv_rotation: float
    """ Rotation of the UV

    :type: float
    """

    uv_scale: float
    """ Scale of the UV

    :type: float
    """

    uv_translation: mathutils.Vector
    """ Translation of default UV position

    :type: mathutils.Vector
    """

    vertex_color_fill: bpy_prop_array[float]
    """ Color used to mix with fill color to get final color

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilStrokePoint(bpy_struct):
    """Data point for freehand stroke curve"""

    co: mathutils.Vector
    """ 

    :type: mathutils.Vector
    """

    pressure: float
    """ Pressure of tablet at point when drawing it

    :type: float
    """

    select: bool
    """ Point is selected for viewport editing

    :type: bool
    """

    strength: float
    """ Color intensity (alpha factor)

    :type: float
    """

    time: float
    """ Time relative to stroke start

    :type: float
    """

    uv_factor: float
    """ Internal UV factor

    :type: float
    """

    uv_fill: mathutils.Vector
    """ Internal UV factor for filling

    :type: mathutils.Vector
    """

    uv_rotation: float
    """ Internal UV factor for dot mode

    :type: float
    """

    vertex_color: bpy_prop_array[float]
    """ Color used to mix with point color to get final color

    :type: bpy_prop_array[float]
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GPencilTriangle(bpy_struct):
    """Triangulation data for Grease Pencil fills"""

    v1: int
    """ First triangle vertex index

    :type: int
    """

    v2: int
    """ Second triangle vertex index

    :type: int
    """

    v3: int
    """ Third triangle vertex index

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GammaCrossSequence(EffectSequence, Sequence, bpy_struct):
    """Gamma Cross Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GaussianBlurSequence(EffectSequence, Sequence, bpy_struct):
    """Sequence strip creating a gaussian blur"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

    size_x: float
    """ Size of the blur along X axis

    :type: float
    """

    size_y: float
    """ Size of the blur along Y axis

    :type: float
    """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GeometryNode(NodeInternal, Node, bpy_struct):
    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GeometryNodeAccumulateField(GeometryNode, NodeInternal, Node, bpy_struct):
    """Add the values of an evaluated field together and output the running total for each element"""

    data_type: bpy._typing.rna_enums.AttributeTypeItems
    """ Type of data stored in attribute

    :type: bpy._typing.rna_enums.AttributeTypeItems
    """

    domain: bpy._typing.rna_enums.AttributeDomainItems
    """ 

    :type: bpy._typing.rna_enums.AttributeDomainItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GeometryNodeAttributeDomainSize(GeometryNode, NodeInternal, Node, bpy_struct):
    """Retrieve the number of elements in a geometry for each attribute domain"""

    component: bpy._typing.rna_enums.GeometryComponentTypeItems
    """ 

    :type: bpy._typing.rna_enums.GeometryComponentTypeItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GeometryNodeAttributeStatistic(GeometryNode, NodeInternal, Node, bpy_struct):
    """Calculate statistics about a data set from a field evaluated on a geometry"""

    data_type: bpy._typing.rna_enums.AttributeTypeItems
    """ The data type the attribute is converted to before calculating the results

    :type: bpy._typing.rna_enums.AttributeTypeItems
    """

    domain: bpy._typing.rna_enums.AttributeDomainItems
    """ Which domain to read the data from

    :type: bpy._typing.rna_enums.AttributeDomainItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GeometryNodeBlurAttribute(GeometryNode, NodeInternal, Node, bpy_struct):
    """Mix attribute values of neighboring elements"""

    data_type: bpy._typing.rna_enums.AttributeTypeItems
    """ 

    :type: bpy._typing.rna_enums.AttributeTypeItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GeometryNodeBoundBox(GeometryNode, NodeInternal, Node, bpy_struct):
    """Calculate the limits of a geometry's positions and generate a box mesh with those dimensions"""

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GeometryNodeCaptureAttribute(GeometryNode, NodeInternal, Node, bpy_struct):
    """Store the result of a field on a geometry and output the data as a node socket. Allows remembering or interpolating data as the geometry changes, such as positions before deformation"""

    data_type: bpy._typing.rna_enums.AttributeTypeItems
    """ Type of data stored in attribute

    :type: bpy._typing.rna_enums.AttributeTypeItems
    """

    domain: bpy._typing.rna_enums.AttributeDomainItems
    """ Which domain to store the data in

    :type: bpy._typing.rna_enums.AttributeDomainItems
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GeometryNodeCollectionInfo(GeometryNode, NodeInternal, Node, bpy_struct):
    """Retrieve geometry instances from a collection"""

    transform_space: typing.Literal["ORIGINAL", "RELATIVE"]
    """ The transformation of the instances output. Does not affect the internal geometry

    :type: typing.Literal['ORIGINAL','RELATIVE']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def bl_rna_get_subclass(cls, id: str | None, default=None) -> Struct:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The RNA type or default when not found.
        :rtype: Struct
        """

    @classmethod
    def bl_rna_get_subclass_py(cls, id: str | None, default=None) -> typing.Any:
        """

        :param id: The RNA type identifier.
        :type id: str | None
        :param default:
        :return: The class or default when not found.
        :rtype: typing.Any
        """

class GeometryNodeConvexHull(GeometryNode, NodeInternal, Node, bpy_struct):
    """Create a mesh that encloses all points in the input geometry with the smallest number of points"""

    @classmethod
    de