"""
IBM Watsonx AI Provider for Banko Assistant

This module provides integration with IBM Watsonx AI services to power the 
Banko Assistant's conversational capabilities. It includes functions for:
- Expense data search and retrieval
- RAG (Retrieval Augmented Generation) responses
- Financial data analysis

Dependencies:
- requests: For HTTP API calls to Watsonx
- numpy: For vector operations
- json: For data serialization
- sentence_transformers: For embedding generation
- sqlalchemy: For database operations

Author: Banko AI Team
Date: 2025
"""

import os
import requests
import json
import numpy as np
from typing import List, Optional, Dict, Any
from ..ai_providers.base import AIProvider, RAGResponse, SearchResult


class WatsonxProvider(AIProvider):
    """IBM Watsonx AI Provider implementation."""
    
    def __init__(self, config: Dict[str, Any] = None, cache_manager=None):
        """Initialize Watsonx provider with configuration."""
        if config is None:
            config = {}
        
        # Store config for base class compatibility
        self.config = config
        self.cache_manager = cache_manager
        
        self.api_key = config.get('api_key') or os.getenv('WATSONX_API_KEY')
        self.project_id = config.get('project_id') or os.getenv('WATSONX_PROJECT_ID')
        self.current_model = config.get('model', config.get('model_id')) or os.getenv('WATSONX_MODEL_ID', 'openai/gpt-oss-120b')
        self.api_url = "https://us-south.ml.cloud.ibm.com/ml/v1/text/chat?version=2023-05-29"
        
        # Make API key and project ID optional for demo purposes
        if not self.api_key:
            print("⚠️ WATSONX_API_KEY not found - running in demo mode")
        if not self.project_id:
            print("⚠️ WATSONX_PROJECT_ID not found - running in demo mode")
    
    def _validate_config(self) -> None:
        """Validate Watsonx configuration."""
        # Configuration is optional for demo mode
        pass
    
    def get_default_model(self) -> str:
        """Get the default model for Watsonx."""
        return 'openai/gpt-oss-120b'
    
    def generate_embedding(self, text: str) -> List[float]:
        """Generate embedding vector for the given text."""
        try:
            from sentence_transformers import SentenceTransformer
            model = SentenceTransformer('all-MiniLM-L6-v2')
            embedding = model.encode([text])[0]
            return embedding.tolist()
        except Exception as e:
            print(f"Error generating embedding: {e}")
            return []
    
    def search_expenses(
        self, 
        query: str, 
        user_id: Optional[str] = None,
        limit: int = 10,
        threshold: float = 0.7
    ) -> List[SearchResult]:
        """Search for expenses using vector similarity."""
        # This method should delegate to the vector search engine
        # For now, return empty list as the search is handled by the web app
        return []
    
    def get_available_models(self) -> List[str]:
        """Get list of available Watsonx models."""
        return [
            'openai/gpt-oss-120b',
            'meta-llama/llama-2-70b-chat',
            'meta-llama/llama-2-13b-chat',
            'meta-llama/llama-2-7b-chat',
            'ibm/granite-13b-chat-v2',
            'ibm/granite-13b-instruct-v2'
        ]
    
    def set_model(self, model_id: str) -> bool:
        """Set the current model."""
        if model_id in self.get_available_models():
            self.current_model = model_id
            return True
        return False
    
    def test_connection(self) -> bool:
        """Test connection to Watsonx API."""
        if not self.api_key or not self.project_id:
            return False
        
        try:
            test_messages = [{"role": "user", "content": "Hello, this is a test message."}]
            response = self._call_watsonx_api(test_messages)
            return True
        except Exception as e:
            return False
    
    def _get_access_token(self):
        """Get IBM Cloud access token from API key (copied from original)."""
        token_url = "https://iam.cloud.ibm.com/identity/token"
        headers = {
            "Content-Type": "application/x-www-form-urlencoded",
            "Accept": "application/json"
        }
        data = {
            "grant_type": "urn:ibm:params:oauth:grant-type:apikey",
            "apikey": self.api_key
        }
        
        try:
            response = requests.post(token_url, headers=headers, data=data, timeout=30)
            if response.status_code != 200:
                raise Exception(f"Failed to get access token (status {response.status_code}): {response.text}")
            token_data = response.json()
            access_token = token_data.get('access_token')
            if access_token:
                return access_token
            else:
                raise Exception("No access token in response")
        except Exception as e:
            raise Exception(f"Token request failed: {str(e)}")
    
    def _call_watsonx_api(self, messages):
        """Make a direct API call to IBM Watsonx chat endpoint (copied from original)."""
        access_token = self._get_access_token()
        headers = {
            "Accept": "application/json",
            "Content-Type": "application/json",
            "Authorization": f"Bearer {access_token}"
        }
        body = {
            "project_id": self.project_id,
            "model_id": self.current_model,
            "messages": messages,
            "frequency_penalty": 0,
            "max_tokens": 2000,
            "presence_penalty": 0,
            "temperature": 0.7,
            "top_p": 1
        }
        
        try:
            response = requests.post(
                self.api_url,
                headers=headers,
                json=body,
                timeout=30
            )
            if response.status_code != 200:
                raise Exception(f"Watsonx API error (status {response.status_code}): {response.text}")
            data = response.json()
            if 'choices' in data and len(data['choices']) > 0:
                return data['choices'][0]['message']['content']
            elif 'generated_text' in data:
                return data['generated_text']
            else:
                print(f"Unexpected Watsonx response format: {data}")
                return "I apologize, but I'm having trouble generating a response right now."
        except Exception as e:
            raise Exception(f"Watsonx API call failed: {str(e)}")
    
    def _get_financial_insights(self, search_results: List[SearchResult]) -> dict:
        """Generate comprehensive financial insights from expense data (copied from original)."""
        if not search_results:
            return {}
        
        total_amount = sum(float(result.amount) for result in search_results)
        categories = {}
        merchants = {}
        payment_methods = {}
        
        for result in search_results:
            # Category analysis
            category = result.metadata.get('shopping_type', 'Unknown')
            categories[category] = categories.get(category, 0) + float(result.amount)
            
            # Merchant analysis
            merchant = result.merchant
            merchants[merchant] = merchants.get(merchant, 0) + float(result.amount)
            
            # Payment method analysis
            payment = result.metadata.get('payment_method', 'Unknown')
            payment_methods[payment] = payment_methods.get(payment, 0) + float(result.amount)
        
        # Find top categories and merchants
        top_category = max(categories.items(), key=lambda x: x[1]) if categories else None
        top_merchant = max(merchants.items(), key=lambda x: x[1]) if merchants else None
        
        return {
            'total_amount': total_amount,
            'num_transactions': len(search_results),
            'avg_transaction': total_amount / len(search_results) if search_results else 0,
            'categories': categories,
            'top_category': top_category,
            'top_merchant': top_merchant,
            'payment_methods': payment_methods
        }
    
    def _generate_budget_recommendations(self, insights: dict, prompt: str) -> str:
        """Generate personalized budget recommendations based on spending patterns (copied from original)."""
        if not insights:
            return ""
        
        recommendations = []
        
        # High spending category recommendations
        if insights.get('top_category'):
            category, amount = insights['top_category']
            recommendations.append(f"Your highest spending category is **{category}** at **${amount:.2f}**. Consider setting a monthly budget limit for this category.")
        
        # Average transaction analysis
        avg_amount = insights.get('avg_transaction', 0)
        if avg_amount > 100:
            recommendations.append(f"Your average transaction is **${avg_amount:.2f}**. Consider reviewing larger purchases to identify potential savings.")
        
        # Merchant frequency analysis
        if insights.get('top_merchant'):
            merchant, amount = insights['top_merchant']
            recommendations.append(f"You frequently shop at **{merchant}** (${amount:.2f} total). Look for loyalty programs or discounts at this merchant.")
        
        # General budgeting tips
        if insights.get('total_amount', 0) > 500:
            recommendations.append("💡 **Budget Tip**: Consider the 50/30/20 rule: 50% for needs, 30% for wants, 20% for savings and debt repayment.")
        
        return "\n".join(recommendations) if recommendations else ""
    
    def rag_response(
        self,
        query: str,
        context: List[SearchResult],
        language: str = "en"
    ) -> str:
        """Generate a RAG response using Watsonx API (copied from original working implementation)."""
        try:
            if not self.api_key or not self.project_id:
                # Return structured demo response if no API credentials
                if not context:
                    return f"""## Financial Analysis for: "{query}"

### 📋 Transaction Details
No expense records found for this query.

### 📊 Financial Summary
No data available for analysis.

### 🤖 AI-Powered Insights
I couldn't find any relevant expense records for your query. Please try:
- Different keywords (e.g., "groceries", "restaurants", "transportation")
- Broader categories (e.g., "food", "shopping", "bills")
- Time periods (e.g., "last month", "this week")

**Note**: I need API credentials to generate more detailed AI-powered insights."""

                # Generate financial insights from search results
                insights = self._get_financial_insights(context)
                budget_recommendations = self._generate_budget_recommendations(insights, query)
                
                # Create table text from search results
                table_text = ""
                if context:
                    table_text = "\n".join([
                        f"• **{result.metadata.get('shopping_type', 'Unknown')}** at {result.merchant}: ${result.amount} ({result.metadata.get('payment_method', 'Unknown')}) - {result.description}"
                        for result in context
                    ])
                
                # Create context text with financial summary
                context_text = f"""**📊 Financial Summary:**
• Total Amount: ${insights.get('total_amount', 0):.2f}
• Number of Transactions: {insights.get('num_transactions', 0)}
• Average Transaction: ${insights.get('avg_transaction', 0):.2f}
• Top Category: {insights.get('top_category', ('Unknown', 0))[0] if insights.get('top_category') else 'Unknown'}
• Most frequent category: {insights.get('top_category', ('Unknown', 0))[0] if insights.get('top_category') else 'Unknown'}

**Recommendations:**
{budget_recommendations if budget_recommendations else '• Consider reviewing your spending patterns regularly' + chr(10) + '• Set up budget alerts for high-value categories'}

**Note**: I can see {len(context)} relevant expense records, but I need API credentials to generate more detailed AI-powered insights."""
            else:
                # Make actual Watsonx API call with enhanced prompt (copied from original)
                try:
                    # Generate financial insights from search results
                    insights = self._get_financial_insights(context)
                    budget_recommendations = self._generate_budget_recommendations(insights, query)
                    
                    # Prepare the search results context with enhanced analysis (copied from original)
                    search_results_text = ""
                    if context:
                        search_results_text = "\n".join(
                            f"• **{result.metadata.get('shopping_type', 'Unknown')}** at {result.merchant}: ${result.amount} ({result.metadata.get('payment_method', 'Unknown')}) - {result.description}"
                            for result in context
                        )
                        
                        # Add financial summary (copied from original)
                        if insights:
                            search_results_text += f"\n\n**📊 Financial Summary:**\n"
                            search_results_text += f"• Total Amount: **${insights['total_amount']:.2f}**\n"
                            search_results_text += f"• Number of Transactions: **{insights['num_transactions']}**\n"
                            search_results_text += f"• Average Transaction: **${insights['avg_transaction']:.2f}**\n"
                            if insights.get('top_category'):
                                cat, amt = insights['top_category']
                                search_results_text += f"• Top Category: **{cat}** (${amt:.2f})\n"
                    else:
                        search_results_text = "No specific expense records found for this query."
                    
                    # Create optimized prompt (copied from original)
                    enhanced_prompt = f"""You are Banko, a financial assistant. Answer based on this expense data:

Q: {query}

Data:
{search_results_text}

{budget_recommendations if budget_recommendations else ''}

Provide helpful insights with numbers, markdown formatting, and actionable advice."""
                    
                    # Prepare messages for chat format (copied from original)
                    messages = [
                        {
                            "role": "user",
                            "content": enhanced_prompt
                        }
                    ]
                    
                    # Call Watsonx API (copied from original implementation)
                    response_text = self._call_watsonx_api(messages)
                    
                except Exception as e:
                    # Fallback to structured response if API call fails
                    error_msg = str(e)
                    response_text = "## Financial Analysis for: \"" + query + "\"\n\n"
                    response_text += "### 📋 Transaction Details\n"
                    response_text += search_results_text if 'search_results_text' in locals() else 'No data available'
                    response_text += "\n\n### 📊 Financial Summary\n"
                    response_text += f"{insights.get('total_amount', 0):.2f} total across {insights.get('num_transactions', 0)} transactions"
                    response_text += "\n\n### 🤖 AI-Powered Insights\n"
                    response_text += f"Based on your expense data, I found {len(context)} relevant records. Here's a comprehensive analysis:\n\n"
                    response_text += "**Spending Analysis:**\n"
                    response_text += f"- Total Amount: ${insights.get('total_amount', 0):.2f}\n"
                    response_text += f"- Transaction Count: {insights.get('num_transactions', 0)}\n"
                    response_text += f"- Average Transaction: ${insights.get('avg_transaction', 0):.2f}\n"
                    top_category = insights.get('top_category', ('Unknown', 0))
                    response_text += "- Top Category: " + (top_category[0] if top_category else 'Unknown') + " ($" + f"{top_category[1]:.2f}" + " if top_category else 0)\n\n"
                    response_text += "**Smart Recommendations:**\n"
                    response_text += budget_recommendations if budget_recommendations else '• Monitor your spending patterns regularly\n• Consider setting up budget alerts\n• Review high-value transactions for optimization opportunities'
                    response_text += "\n\n**Next Steps:**\n"
                    response_text += "• Track your spending trends over time\n"
                    response_text += "• Set realistic budget goals for each category\n"
                    response_text += "• Review and optimize your payment methods\n\n"
                    response_text += "**Note**: API call failed, showing structured analysis above."
            
            return response_text
            
        except Exception as e:
            return f"Sorry, I'm experiencing technical difficulties. Error: {str(e)}"
    
    def generate_rag_response(
        self,
        query: str,
        context: List[SearchResult],
        user_id: Optional[str] = None,
        language: str = "en"
    ) -> RAGResponse:
        """Generate a RAG response using Watsonx API (copied from original working implementation)."""
        try:
            print(f"\n🤖 WATSONX RAG (with caching):")
            print(f"1. Query: '{query[:60]}...'")
            
            # Check for cached response first
            if self.cache_manager:
                # Convert SearchResult objects to dict format for cache lookup
                search_results_dict = []
                for result in context:
                    search_results_dict.append({
                        'expense_id': result.expense_id,
                        'user_id': result.user_id,
                        'description': result.description,
                        'merchant': result.merchant,
                        'expense_amount': result.amount,
                        'expense_date': result.date,
                        'similarity_score': result.similarity_score,
                        'shopping_type': result.metadata.get('shopping_type'),
                        'payment_method': result.metadata.get('payment_method'),
                        'recurring': result.metadata.get('recurring'),
                        'tags': result.metadata.get('tags')
                    })
                
                cached_response = self.cache_manager.get_cached_response(
                    query, search_results_dict, "watsonx"
                )
                if cached_response:
                    print(f"2. ✅ Response cache HIT! Returning cached response")
                    return RAGResponse(
                        response=cached_response,
                        sources=context,
                        metadata={
                            'provider': 'watsonx',
                            'model': self.current_model,
                            'user_id': user_id,
                            'language': language,
                            'cached': True
                        }
                    )
                print(f"2. ❌ Response cache MISS, generating fresh response")
            else:
                print(f"2. No cache manager available, generating fresh response")
            if not self.api_key or not self.project_id:
                # Return structured demo response if no API credentials
                if not context:
                    ai_response = f"""## Financial Analysis for: "{query}"

### 📋 Transaction Details
No expense records found for this query.

### 📊 Financial Summary
No data available for analysis.

### 🤖 AI-Powered Insights
I couldn't find any relevant expense records for your query. Please try:
- Different keywords (e.g., "groceries", "restaurants", "transportation")
- Broader categories (e.g., "food", "shopping", "bills")
- Time periods (e.g., "last month", "this week")

**Note**: I need API credentials to generate more detailed AI-powered insights."""

                # Generate financial insights from search results
                insights = self._get_financial_insights(context)
                budget_recommendations = self._generate_budget_recommendations(insights, query)
                
                # Create table text from search results
                table_text = ""
                if context:
                    table_text = "\n".join([
                        f"• **{result.metadata.get('shopping_type', 'Unknown')}** at {result.merchant}: ${result.amount} ({result.metadata.get('payment_method', 'Unknown')}) - {result.description}"
                        for result in context
                    ])
                
                # Create context text with financial summary
                context_text = f"""**📊 Financial Summary:**
• Total Amount: ${insights.get('total_amount', 0):.2f}
• Number of Transactions: {insights.get('num_transactions', 0)}
• Average Transaction: ${insights.get('avg_transaction', 0):.2f}
• Top Category: {insights.get('top_category', ('Unknown', 0))[0] if insights.get('top_category') else 'Unknown'}
• Most frequent category: {insights.get('top_category', ('Unknown', 0))[0] if insights.get('top_category') else 'Unknown'}

**Recommendations:**
{budget_recommendations if budget_recommendations else '• Consider reviewing your spending patterns regularly' + chr(10) + '• Set up budget alerts for high-value categories'}

**Note**: I can see {len(context)} relevant expense records, but I need API credentials to generate more detailed AI-powered insights."""
            else:
                # Make actual Watsonx API call with enhanced prompt (copied from original)
                try:
                    # Generate financial insights from search results
                    insights = self._get_financial_insights(context)
                    budget_recommendations = self._generate_budget_recommendations(insights, query)
                    
                    # Prepare the search results context with enhanced analysis (copied from original)
                    search_results_text = ""
                    if context:
                        search_results_text = "\n".join(
                            f"• **{result.metadata.get('shopping_type', 'Unknown')}** at {result.merchant}: ${result.amount} ({result.metadata.get('payment_method', 'Unknown')}) - {result.description}"
                            for result in context
                        )
                        
                        # Add financial summary (copied from original)
                        if insights:
                            search_results_text += f"\n\n**📊 Financial Summary:**\n"
                            search_results_text += f"• Total Amount: **${insights['total_amount']:.2f}**\n"
                            search_results_text += f"• Number of Transactions: **{insights['num_transactions']}**\n"
                            search_results_text += f"• Average Transaction: **${insights['avg_transaction']:.2f}**\n"
                            if insights.get('top_category'):
                                cat, amt = insights['top_category']
                                search_results_text += f"• Top Category: **{cat}** (${amt:.2f})\n"
                    else:
                        search_results_text = "No specific expense records found for this query."
                    
                    # Create optimized prompt (copied from original)
                    enhanced_prompt = f"""You are Banko, a financial assistant. Answer based on this expense data:

Q: {query}

Data:
{search_results_text}

{budget_recommendations if budget_recommendations else ''}

Provide helpful insights with numbers, markdown formatting, and actionable advice."""
                    
                    # Prepare messages for chat format (copied from original)
                    messages = [
                        {
                            "role": "user",
                            "content": enhanced_prompt
                        }
                    ]
                    
                    # Call Watsonx API (copied from original implementation)
                    ai_response = self._call_watsonx_api(messages)
                    
                except Exception as e:
                    # Fallback to structured response if API call fails
                    error_msg = str(e)
                    ai_response = "## Financial Analysis for: \"" + query + "\"\n\n"
                    ai_response += "### 📋 Transaction Details\n"
                    ai_response += search_results_text if 'search_results_text' in locals() else 'No data available'
                    ai_response += "\n\n### 📊 Financial Summary\n"
                    ai_response += f"{insights.get('total_amount', 0):.2f} total across {insights.get('num_transactions', 0)} transactions"
                    ai_response += "\n\n### 🤖 AI-Powered Insights\n"
                    ai_response += f"Based on your expense data, I found {len(context)} relevant records. Here's a comprehensive analysis:\n\n"
                    ai_response += "**Spending Analysis:**\n"
                    ai_response += f"- Total Amount: ${insights.get('total_amount', 0):.2f}\n"
                    ai_response += f"- Transaction Count: {insights.get('num_transactions', 0)}\n"
                    ai_response += f"- Average Transaction: ${insights.get('avg_transaction', 0):.2f}\n"
                    top_category = insights.get('top_category', ('Unknown', 0))
                    ai_response += "- Top Category: " + (top_category[0] if top_category else 'Unknown') + " ($" + f"{top_category[1]:.2f}" + " if top_category else 0)\n\n"
                    ai_response += "**Smart Recommendations:**\n"
                    ai_response += budget_recommendations if budget_recommendations else '• Monitor your spending patterns regularly\n• Consider setting up budget alerts\n• Review high-value transactions for optimization opportunities'
                    ai_response += "\n\n**Next Steps:**\n"
                    ai_response += "• Track your spending trends over time\n"
                    ai_response += "• Set realistic budget goals for each category\n"
                    ai_response += "• Review and optimize your payment methods\n\n"
                    ai_response += "**Note**: API call failed, showing structured analysis above."
            
            # Cache the response for future similar queries
            if self.cache_manager and ai_response:
                # Convert SearchResult objects to dict format for caching
                search_results_dict = []
                for result in context:
                    search_results_dict.append({
                        'expense_id': result.expense_id,
                        'user_id': result.user_id,
                        'description': result.description,
                        'merchant': result.merchant,
                        'expense_amount': result.amount,
                        'expense_date': result.date,
                        'similarity_score': result.similarity_score,
                        'shopping_type': result.metadata.get('shopping_type'),
                        'payment_method': result.metadata.get('payment_method'),
                        'recurring': result.metadata.get('recurring'),
                        'tags': result.metadata.get('tags')
                    })
                
                # Estimate token usage (rough approximation)
                prompt_tokens = len(query.split()) * 1.3  # ~1.3 tokens per word
                response_tokens = len(ai_response.split()) * 1.3
                
                self.cache_manager.cache_response(
                    query, ai_response, search_results_dict, "watsonx",
                    int(prompt_tokens), int(response_tokens)
                )
                print(f"3. ✅ Cached response (est. {int(prompt_tokens + response_tokens)} tokens)")
            
            return RAGResponse(
                response=ai_response,
                sources=context,
                metadata={
                    'provider': 'watsonx',
                    'model': self.current_model,
                    'user_id': user_id,
                    'language': language
                }
            )
            
        except Exception as e:
            return RAGResponse(
                response=f"Sorry, I'm experiencing technical difficulties. Error: {str(e)}",
                sources=context,
                metadata={
                    'provider': 'watsonx',
                    'model': self.current_model,
                    'user_id': user_id,
                    'language': language,
                    'error': str(e)
                }
            )