"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubActionsIdentityProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
/**
 * Github Actions as OpenID Connect Identity Provider for AWS IAM.
 * There can be only one (per AWS Account).
 *
 * Use `fromAccount` to retrieve a reference to existing Github OIDC provider.
 *
 * @see https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
 */
class GithubActionsIdentityProvider extends iam.OpenIdConnectProvider {
    /**
     * Define a new Github OpenID Connect Identity PRovider for AWS IAM.
     * There can be only one (per AWS Account).
     *
     * @param scope CDK Stack or Construct to which the provider is assigned to
     * @param id CDK Construct ID given to the construct
     *
     * @example
     * new GithubActionsIdentityProvider(scope, "GithubProvider");
     */
    constructor(scope, id) {
        super(scope, id, {
            url: `https://${GithubActionsIdentityProvider.issuer}`,
            clientIds: ['sts.amazonaws.com'],
        });
    }
    /**
     * Retrieve a reference to existing Github OIDC provider in your AWS account.
     * An AWS account can only have single Github OIDC provider configured into it,
     * so internally the reference is made by constructing the ARN from AWS
     * Account ID & Github issuer URL.
     *
     * @param scope CDK Stack or Construct to which the provider is assigned to
     * @param id CDK Construct ID given to the construct
     * @returns a CDK Construct representing the Github OIDC provider
     *
     * @example
     * GithubActionsIdentityProvider.fromAccount(scope, "GithubProvider");
     */
    static fromAccount(scope, id) {
        const accountId = cdk.Stack.of(scope).account;
        const providerArn = `arn:aws:iam::${accountId}:oidc-provider/${GithubActionsIdentityProvider.issuer}`;
        return iam.OpenIdConnectProvider.fromOpenIdConnectProviderArn(scope, id, providerArn);
    }
}
exports.GithubActionsIdentityProvider = GithubActionsIdentityProvider;
_a = JSII_RTTI_SYMBOL_1;
GithubActionsIdentityProvider[_a] = { fqn: "aws-cdk-github-oidc.GithubActionsIdentityProvider", version: "2.4.1" };
GithubActionsIdentityProvider.issuer = 'token.actions.githubusercontent.com';
//# sourceMappingURL=data:application/json;base64,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