"""
Conditional Generator model definition.
"""

import tensorflow as tf

from ..layers import ResidualLayer, GenActivation


class Generator(tf.keras.Model):
    """Conditional Generator model.

    This model uses intermediary :class:`ctgan.layers.ResidualLayer`
    and mix of activation functions in the last layer of the network
    (:class:`ctgan.layers.GenActivation`), according to the data provided by
    ``transformer_info`` (generated by
    :class:`ctgan.data_modules.DataTransformer`).

    For further details of the network architecture, please check section 4.4
    of :cite:`xu2019modeling`.

    Parameters
    ----------
    input_dim: int
        Dimension of the input data.

    gen_dims: list[int]
        List of the number of neurons per layer.

    data_dim: int
        Output dimension of the Generator, i.e., the dimension of
        the data it is trying to replicate.

    transformer_info: list[tf.Tensor]
        List of tensors containing information regarding the activation
        functions of each data column.

    tau: float
        Gumbel-Softmax non-negative scalar temperature
        :cite:`maddison2016concrete`, :cite:`jang2016categorical`.

    See Also
    --------
    ctgan.data_modules.DataTransformer : Transforms the input dataset by
        applying mode-specific normalization and OneHot encoding.

    ctgan.layers.ResidualLayer : Custom layer designed to use in the Generator
        neural network.

    ctgan.layers.GenActivation : Custom layer designed to apply a mix of
        activation functions to specific data columns.

    References
    ----------
    .. bibliography:: ../bibtex/refs.bib
    """
    # pylint: disable=too-few-public-methods

    def __init__(self, input_dim, gen_dims, data_dim, transformer_info, tau):
        super(Generator, self).__init__()

        self._input_dim = input_dim
        self._model = list()
        dim = input_dim
        for layer_dim in list(gen_dims):
            self._model += [ResidualLayer(dim, layer_dim)]
            dim += layer_dim

        self._model += [GenActivation(dim, data_dim, transformer_info, tau)]

    def call(self, inputs, **kwargs):
        """Feedforward pass on the network layers.

        Parameters
        ----------
        inputs: tf.Tensor
            Input data tensor.

        kwargs: dict
            Dictionary with functions options. Mainly used to set
            ``training=True`` when training the network layers.

        Returns
        -------
        tf.Tensor
            The output tensor resultant from the feedforward pass.
        """
        outputs = inputs
        for layer in self._model:
            outputs = layer(outputs, **kwargs)
        return outputs
