#!/usr/bin/env python3

import logging
import os
import pathlib
import typing
import warnings

import click
import humanize
import questionary

from ideas import __version__, commands, config, constants, containers, exceptions
from ideas.constants import EXIT_STATUS
from ideas.tools import commands as tools
from ideas.tools.executor import ToolConfig
from ideas.utils import custom_click_types, display, sentry
from ideas.utils.common_arguments import base_cli_arguments, require_tenant_id
from ideas.utils.os_utils import is_windows, open_file_with_default_app

# This logger is only enabled if `DEBUG` is True; it's meant to be used for
# debugging only. Use `display` methods for user-facing messages.
logger = logging.getLogger(__name__)


@click.group()
@click.version_option(__version__)
@click.pass_context
@click.option(
    "--debug",
    default=False,
    is_flag=True,
    envvar="IDEAS_CLI_DEBUG",
    help="Show debug info, for troubleshooting purposes.",
)
@click.option(
    "--error-log-filepath",
    type=click.Path(dir_okay=False),
    envvar="IDEAS_CLI_ERROR_LOG_FILEPATH",
    required=False,
    help="Where to log error messages, for support. If not set, log will be placed in a temporary file",
)
@base_cli_arguments
def cli(ctx, debug: bool, error_log_filepath: typing.Optional[str]):
    """
    A command-line interface to the IDEAS platform.
    """
    # A click group to put all commands under. Ensures the context is set up correctly, and sets the
    # debugging mode.

    display.setup_logging(debug, error_log_filepath)


@cli.command()
@click.option(
    "--dev",
    "include_dev",
    default=False,
    is_flag=True,
    help="Include development environments (for testing only)",
    hidden=True,
)
@click.option(
    "--environments-url",
    default=None,
    envvar="IDEAS_CLI_ENVIRONMENTS_URL",
    # Don't expose this to the users in the --help, they don't need it
    hidden=True,
)
def environments(include_dev: bool, environments_url: str | None):
    """
    Display supported environments.
    """
    display.pretty_print(commands.get_environments(include_dev, environments_url))


@cli.command()
@click.option(
    "-n",
    "--non-interactive",
    is_flag=True,
    help="Open configuration in preferred editor",
)
@click.option(
    "-p",
    "--profile",
    type=custom_click_types.ProfileOption(),
    help="Which configuration profile to edit",
    required=False,
)
@click.option(
    "--dev",
    "include_dev",
    default=False,
    is_flag=True,
    help="Include development environments (for testing only)",
    hidden=True,
)
@click.option(
    "--environments-url",
    default=None,
    envvar="IDEAS_CLI_ENVIRONMENTS_URL",
    # Don't expose this to the users in the --help, they don't need it
    hidden=True,
)
@click.option(
    "--django-url",
    default=None,
    envvar="IDEAS_CLI_DJANGO_URL",
    # Don't expose this to the users in the --help, they don't need it
    hidden=True,
)
@click.pass_context
def configure(
    ctx,
    non_interactive: bool,
    profile: str,
    include_dev: bool,
    environments_url: str | None,
    django_url: str | None,
):
    """
    Configure the CLI, including authentication and tenant selection.
    """
    config.edit(
        # This option used to be under configure command, like `ideas configure --profile` but there
        # is already a top-level flag like `ideas --profile configure`, so this just keeps parity
        # with whatever the user is used to
        profile or ctx.obj["profile"],
        interactive=not non_interactive,
        include_dev=include_dev,
        environments_url=environments_url,
        django_url=django_url,
        config_file_path=ctx.obj["config_file"],
    )


@cli.command()
@click.pass_context
@click.option(
    "--filter",
    "filters",
    type=custom_click_types.KeyValueOption(),
    required=False,
    help="Optional and repeatable filters. For example, `--filter id=1 --filter key=inscopix`",
    multiple=True,
)
@click.option(
    "--search",
    required=False,
    type=custom_click_types.SearchOption(),
    default=tuple(),
    help="Will search for partial matches across some fields for the supplied string.",
)
def tenants(ctx, filters, search):
    """
    Return a list of all tenants.
    """
    display.pretty_print(
        list(
            commands.get_tenants(
                filters + search,
                session=ctx.obj["session"],
            )
        )
    )


ProjectAvailableFilter: custom_click_types.KeyValueOptionValue = ("status", "1")


@cli.command()
@click.pass_context
@click.option(
    "--filter",
    "filters",
    type=custom_click_types.KeyValueOption(),
    required=False,
    help="Optional and repeatable filters. For example, `--filter key=project-key --filter status=1`",
    multiple=True,
)
@click.option(
    "--search",
    required=False,
    type=custom_click_types.SearchOption(),
    default=tuple(),
    help="Will search for partial matches across some fields for the supplied string.",
)
@click.option(
    "--show-all",
    default=False,
    is_flag=True,
    help="By default, we filter on available projects only; this flag will return all projects instead",
)
def projects(ctx, filters, search, show_all: bool):
    """
    Return a list of all available projects.
    """
    status_filtered_by_user = "status" in [filter_val[0] for filter_val in filters]

    if not (show_all or status_filtered_by_user):
        filters = filters + (ProjectAvailableFilter,)

    display.pretty_print(
        list(
            commands.get_projects(
                filters + search,
                session=ctx.obj["session"],
            )
        )
    )


@cli.command()
@click.pass_context
@require_tenant_id
@click.option(
    "--filter",
    "filters",
    type=custom_click_types.KeyValueOption(),
    required=False,
    help="Optional and repeatable filters. For example, `--filter id=<UUID> --filter status=2`",
    multiple=True,
)
@click.option(
    "--search",
    required=False,
    type=custom_click_types.SearchOption(),
    default=tuple(),
    help="Will search for partial matches across some fields for the supplied string.",
)
def files(ctx, tenant_id, filters, search):
    """
    Return a list of files.
    """
    display.pretty_print(
        list(
            commands.get_files(
                filters + search,
                session=ctx.obj["session"],
            )
        )
    )


@cli.command()
@click.pass_context
@click.option("--project-id", required=True)
@require_tenant_id
@click.option(
    "-f",
    "--upload-progress-filepath",
    envvar="IDEAS_CLI_UPLOAD_PROGRESS_FILEPATH",
    required=False,
)
@click.option(
    "--metadata",
    "metadata",
    type=custom_click_types.KeyValueOption(),
    required=False,
    help="Optional and repeatable metadata. For example, `--metadata ideas.idas.filepath=/path/to/my/file.isxd`",
    multiple=True,
)
@click.option(
    "--resume-file-id",
    "file_id",
    type=str,
    help="Will attempt to resume the upload using the given file ID",
)
@click.option(
    "-n",
    "--upload-threads",
    type=click.IntRange(min=1, max=32),
    help="How many parallel threads to upload with. Increases memory usage and upload performance. Defaults to 5 x number of CPU cores.",
    default=None,
    envvar="IDEAS_CLI_PARALLEL_UPLOADS",
)
@click.argument(
    "files",
    nargs=-1,
    type=click.Path(exists=True, dir_okay=False, resolve_path=True),
)
def upload(
    ctx,
    files: list[pathlib.Path],
    metadata: tuple[custom_click_types.KeyValueOptionValue],
    project_id: str,
    tenant_id: int,
    upload_progress_filepath: typing.Optional[pathlib.Path],
    upload_threads: int,
    file_id: typing.Optional[str],
):
    """
    Upload a file to a specified project.
    """
    max_label_length = max([len(os.path.basename(file)) for file in files])
    uploaded_files = []
    for file in files:
        uploaded_files.append(
            commands.multipart_upload_source(
                filepath=file,
                project=project_id,
                upload_progress_filepath=upload_progress_filepath,
                metadata=metadata,
                file_id=file_id,
                max_label_length=max_label_length,
                upload_threads=upload_threads,
                session=ctx.obj["session"],
            )
        )

    display.pretty_print(uploaded_files)


@cli.command()
@click.pass_context
@require_tenant_id
@click.option("--file-id", required=True)
@click.option(
    "-f",
    "--download-progress-filepath",
    required=False,
    envvar="IDEAS_CLI_DOWNLOAD_PROGRESS_FILEPATH",
)
@click.option(
    "-d",
    "--download-dir",
    required=False,
    type=click.Path(
        exists=True,
        file_okay=False,
        dir_okay=True,
        path_type=str,
    ),
    help="If specified, download the file to this directory. Otherwise, downloads files to current directory.",
)
def download(
    ctx,
    file_id: str,
    tenant_id: int,
    download_dir: str,
    download_progress_filepath: typing.Optional[pathlib.Path],
):
    """
    Download a file to local computer
    """
    display.show_success(
        commands.download_file(
            file_id=file_id,
            download_dir=download_dir,
            download_progress_filepath=download_progress_filepath,
            session=ctx.obj["session"],
        )
    )


@cli.group("tools")
def tools_group():
    """Commands for local tool execution."""
    pass


@tools_group.command("configure")
@click.pass_context
@click.argument(
    "key",
    type=custom_click_types.ToolKeyOption(),
)
@click.option(
    "--file",
    type=click.Path(
        exists=False,
        file_okay=True,
        dir_okay=False,
        path_type=str,
    ),
    default=None,
    help="The file containing the entry point of analysis code to run for the tool. For python execution, this is the python module containing the entry point function to execute.",
)
@click.option(
    "--function",
    type=str,
    default=None,
    help="The entry point function within the specified file to run for the tool.",
)
@click.option(
    "--container",
    type=custom_click_types.LocalContainerImageOption(),
    default=None,
    help="The container to run the local code within.",
)
@click.option(
    "-d",
    "--code-dir",
    type=click.Path(
        exists=True,
        file_okay=False,
        dir_okay=True,
        path_type=str,
    ),
    default=os.getcwd,
    help="The local code directory to run within, by default the current directory.",
)
@click.option(
    "-n",
    "--non-interactive",
    "non_interactive",
    is_flag=True,
    default=False,
    help="Runs command in non interactive mode (no prompts)",
)
def tools_configure(
    ctx,
    key: str,
    file: typing.Optional[str],
    function: typing.Optional[str],
    container: typing.Optional[str],
    code_dir: typing.Optional[str],
    non_interactive: bool,
):
    """
    Configure tool for local execution.
    """

    def handle_error(*args):
        display.abort(*args, title="Failed to configure tool")

    try:
        if not file or not function or not container:
            try:
                tool_config = ToolConfig(key=key, code_dir=code_dir)
                tool_config.load()
            except Exception:
                logger.exception("Failed to load existing tool config")
            else:
                # try to merge cli inputs with existing configuration
                file = file if file else tool_config.code_file
                function = function if function else tool_config.function
                container = container if container else tool_config.container

            # if any param is not defined, and in non interactive mode, raise an error
            if non_interactive:
                if not file or not function or not container:
                    display.abort(
                        "Specify [bold magenta]--file --function --container[/bold magenta] options to configure new tool in non-interactive mode.",
                        title="Failed to configure tool",
                    )
            else:
                # ask the user to provide a path, use questionary for nice tab completion of paths
                file = questionary.path(
                    "Enter the path of the python module you want to execute for this tool:",
                    qmark=">",
                    default=file if file else "",
                ).ask()

                # based on the selected file, load all functions in module and present options for selection
                from ideas.tools.introspection.parsers import introspect_code

                _, ast = introspect_code(file)
                functions = list(ast.keys())
                function = questionary.select(
                    "Select the function in this module you want to execute for this tool:",
                    qmark=">",
                    choices=functions,
                    default=function if function in functions else None,
                ).ask()

                download_container = False
                if not container:
                    download_container = questionary.confirm(
                        "Do you need to download a container from IDEAS?"
                    ).ask()
                    if download_container:
                        containers_list = list(
                            containers.get_containers(
                                filters=(),
                                session=ctx.obj["session"],
                            ),
                        )
                        container_names = [c["full_name"] for c in containers_list]
                        container = questionary.select(
                            "Select the container to download from IDEAS:",
                            qmark=">",
                            choices=container_names,
                        ).ask()

                        containers.download_container(
                            full_name=container, session=ctx.obj["session"]
                        )

                # get all container image tags and present options for selection
                if not download_container:
                    from ideas.utils.docker_utils import get_docker_client

                    client = get_docker_client()
                    container_names = [
                        tag for image in client.images.list() for tag in image.tags
                    ]
                    container = questionary.select(
                        "Select the container you want to use as the tool execution environment:",
                        qmark=">",
                        choices=container_names,
                        default=container,
                    ).ask()

        tools.configure(
            key=key,
            file=file,
            function=function,
            container=container,
            code_dir=code_dir,
        )

        tool_config_path = os.path.join(code_dir, f".ideas/{key}")
        default_inputs_path = os.path.join(tool_config_path, "inputs.json")
        display.show_success(
            f"Configuration files written to:\n[bold magenta]{os.path.relpath(tool_config_path)}[/bold magenta]",
            "\nNext run your tool and verify outputs:",
            display.bash(f"ideas tools run {key}"),
            f"\nDefine inputs to run your tool with here:\n[bold magenta]{os.path.relpath(default_inputs_path)}[/bold magenta]",
            "\nOr specify a different inputs file:",
            display.bash(f"ideas tools run {key} --inputs /path/to/inputs.json"),
            title="Successfully configured tool!",
        )
    except exceptions.ToolConfigFileNotFoundError:
        handle_error(
            f"Could not find file:\n[bold magenta]{file}[/bold magenta]\n\nCheck path exists:",
            display.bash(f"ls {file}"),
        )
    except exceptions.ToolConfigFunctionNotFoundError:
        handle_error(
            f"Could not find function:\n[bold magenta]{function}[/bold magenta]",
            f"\nCheck function exists in module:\n[bold magenta]{file}[/bold magenta]",
        )
    except exceptions.ToolConfigContainerNotFoundError:
        handle_error(
            f"Could not find container:\n[bold magenta]{container}[/bold magenta]",
            "\nCheck container exists locally:",
            display.bash(f"docker image inspect {container}"),
            "\nThe container either needs to be downloaded:",
            display.bash(f"ideas containers download {container}"),
            "\nOr if it's a custom container, ensure it has been built locally and the container name is spelled correctly.",
        )
    except exceptions.DockerUnavailableError:
        handle_error(*exceptions.DockerUnavailableError.get_instructions())
    except Exception:
        handle_error()


@tools_group.command("run")
@click.pass_context
@click.argument(
    "key",
    type=custom_click_types.ToolKeyOption(),
)
@click.option(
    "--inputs",
    "inputs_file",
    required=False,
    default=None,
    type=click.Path(
        exists=False,
        file_okay=True,
        dir_okay=False,
        path_type=str,
    ),
    help="",
)
@click.option(
    "-c",
    "--clean",
    default=False,
    is_flag=True,
    help="Cleans previous output folders generated for tool",
)
@click.option(
    "--code-dir",
    "code_dir",
    type=click.Path(
        exists=True,
        file_okay=False,
        dir_okay=True,
        path_type=str,
    ),
    default=None,
    help="The local code directory to run within, by default the current directory.",
)
@click.option(
    "-n",
    "--non-interactive",
    "non_interactive",
    is_flag=True,
    default=False,
    help="Runs command in non interactive mode (no prompts)",
)
@click.option(
    "-s",
    "--skip-introspect",
    is_flag=True,
    default=False,
    help="Skips re-introspecting function to ensure all arguments are up to date.",
    hidden=True,
)
@click.option(
    "-g",
    "--gpus",
    "gpus",
    type=str,
    required=False,
    help="Enable gpu access to docker container. Value can either be `all` indicating to use all gpus, or a number indicating the amount of gpus required. Uses nvidia gpu docker runtime.",
    default=None,
)
def tools_run(
    ctx,
    key: str,
    inputs_file: typing.Optional[str],
    clean: bool,
    code_dir: typing.Optional[str],
    non_interactive: bool,
    skip_introspect: bool,
    gpus: typing.Optional[str],
):
    """
    Run tool locally
    """

    def handle_error(*args):
        display.abort(*args, title="Failed to run tool")

    try:
        if is_windows():
            default_inputs_file = f".ideas\\{key}\\inputs.json"
        else:
            default_inputs_file = f".ideas/{key}/inputs.json"
        if not inputs_file and not non_interactive:
            import glob

            inputs_files = glob.glob(
                "**/*inputs*.json", recursive=True, include_hidden=True
            )
            inputs_file = questionary.select(
                "Select an inputs file to run your tool with:",
                qmark=">",
                choices=inputs_files,
                default=default_inputs_file,
            ).ask()

        output_dir = tools.run(
            key=key,
            inputs_file=inputs_file,
            clean=clean,
            code_dir=code_dir,
            skip_introspect=skip_introspect,
            gpus=gpus,
        )
        display.show_success(
            f"Outputs available at:\n[bold magenta]{os.path.relpath(output_dir)}[/bold magenta]",
            "\nOnce you are satisfied with your tool outputs, create a code bundle to upload to IDEAS:",
            display.bash("ideas tools bundle"),
            title="Successfully ran tool!",
        )
    except exceptions.ToolMisconfiguredError:
        handle_error(
            f"Tool is not configured correctly for key [bold magenta]{key}[/bold magenta] in current folder.",
            "\nEnsure the tool key is spelled correctly, and you have configured the tool first before running locally:",
            display.bash(f"ideas tools configure {key}"),
        )
    except exceptions.ToolInputsNotFoundError:
        handle_error(
            f"Tool inputs not found at [bold magenta]{inputs_file}[/bold magenta] in current folder.",
            "\nEnsure inputs file exists for the tool:",
            display.bash(f"ls {inputs_file}"),
            "\nOr regenerate the default inputs file:",
            display.bash(f"ideas tools configure {key}"),
        )
    except exceptions.ToolInputsFormatError as error:
        if not inputs_file:
            inputs_file = f".ideas/{key}/inputs.json"
        handle_error(
            f"Tool inputs not formatted correctly: [bold magenta]{error.message}[/bold magenta]",
            f"\nEnsure inputs are formatted correctly at [bold magenta]{inputs_file}[/bold magenta]",
            "\nYou may need to reconfigure your tool:",
            display.bash(f"ideas tools configure {key}"),
        )
    except exceptions.ToolInputsFileNotFoundError as error:
        if not inputs_file:
            inputs_file = f".ideas/{key}/inputs.json"
        handle_error(
            f"Tool input file not found: [bold magenta]{error.file}[/bold magenta]\n\nEnsure path exists:",
            display.bash(f"ls {error.file}"),
        )
    except exceptions.ToolExecutionError:
        handle_error(
            "An error occurred running the analysis code. Check inputs and logs."
        )
    except exceptions.ToolConfigContainerNotFoundError as error:
        handle_error(
            f"Could not find container:\n[bold magenta]{error.container}[/bold magenta]",
            "\nCheck container exists locally:",
            display.bash(f"docker image inspect {error.container}"),
            "\nThe container either needs to be downloaded from IDEAS container registry:",
            display.bash(f"ideas containers download {error.container}"),
            "\nOr, if it's a custom container, ensure it has been built locally and the container name is spelled correctly.",
        )
    except exceptions.ContainerPythonNotFound:
        handle_error(
            "The command [bold magenta]python[/bold magenta] is not available in the configured container.",
            "\nIf this is a custom container please ensure [bold magenta]python[/bold magenta] is available.",
            "\nAn easy way to do this is to add the following command to the end of your Dockerfile.",
            display.bash("RUN ln -sf /path/to/python /usr/local/bin/python"),
            "\nEnsure you rebuild the container after modifying the Dockerfile.",
        )
    except exceptions.NvidiaRuntimeMissingError:
        handle_error(*exceptions.NvidiaRuntimeMissingError.get_instructions())
    except exceptions.DockerUnavailableError:
        handle_error(*exceptions.DockerUnavailableError.get_instructions())
    except Exception:
        handle_error()


@tools_group.command("bundle")
@click.pass_context
@require_tenant_id
@click.option(
    "-d",
    "--code-dir",
    type=click.Path(
        exists=True,
        file_okay=False,
        dir_okay=True,
        path_type=pathlib.Path,
    ),
    default=pathlib.Path.cwd,
    help="Directory that contains the code to bundle. Defaults to current working directory",
)
@click.option(
    "--skip-container-check",
    "skip_container_check",
    default=False,
    is_flag=True,
    help="Cleans previous output folders generated for tool",
)
@click.option(
    "--skip-validation",
    "skip_validation",
    default=False,
    is_flag=True,
    help="Ignores tool spec validation, for sharing a bundle with a malformed tool spec with support",
)
@click.option(
    "-o",
    "--output-file",
    required=False,
    type=click.Path(dir_okay=False),
)
@click.option(
    "-e",
    "--exclude",
    "excludes",
    type=custom_click_types.ListOption(),
    multiple=True,
    help="Files or directories within code folder to exclude from including in code bundle. Only relative file patterns are supported",
)
@click.option(
    "-t",
    "--tool",
    "included_tools",
    type=custom_click_types.ListOption(),
    multiple=True,
    help="Specify which tools you want to bundle, specifically. If omitted, all tools in the specified code directory are bundled.",
)
def tools_bundle(
    ctx,
    code_dir: pathlib.Path,
    tenant_id: int,
    skip_container_check: bool,
    skip_validation: bool,
    excludes: typing.Tuple[str] = (),
    included_tools: typing.Tuple[str] = (),
    output_file: str | None = None,
):
    """
    Generate code bundle for uploading to IDEAS.
    """

    def handle_error(*args):
        display.abort(*args, title="Failed to create bundle")

    try:
        bundle_file = tools.bundle(
            code_dir=code_dir,
            tenant_id=tenant_id,
            skip_container_check=skip_container_check,
            skip_validation=skip_validation,
            output_file=output_file,
            excludes=excludes,
            included_tools=included_tools,
            session=ctx.obj["session"],
        )

        # check bundle file size and warn if too large
        file_size = os.path.getsize(bundle_file)
        if file_size > constants.MAX_BUNDLE_FILE_SIZE:
            display.show_warning(
                f"Your bundle file size is very large (~{humanize.naturalsize(file_size)}).\nConsider excluding non-code files when creating the bundle to file size for upload: ideas tools bundle --exclude"
            )
        display.show_success(
            f"Code bundle saved to:\n[bold magenta]{os.path.relpath(bundle_file)}[/bold magenta]",
            "\nNext, go to IDEAS, open the draft for this tool, and upload this bundle.",
            title="Successfully created bundle!",
        )
    except exceptions.ToolMisconfiguredError as error:
        handle_error(
            f"Tool [bold magenta]{error.key}[/bold magenta] is not configured correctly.",
            "\nFirst reconfigure the tool:",
            display.bash(f"ideas tools configure {error.key}"),
        )
    except exceptions.ToolNotFoundError:
        handle_error(
            "No tools configured in this folder.",
            "\nFirst configure your tool:",
            display.bash("ideas tools configure"),
        )
    except exceptions.ContainerNotPublishedError as error:
        handle_error(
            f"Cannot generate code bundle due to unpublished local image:\n[bold magenta]{error.container}[/bold magenta]",
            "\nFirst publish the container to IDEAS:",
            display.bash(f"ideas containers publish {error.container}"),
        )
    except exceptions.ToolConfigContainerNotFoundError as error:
        handle_error(
            f"Could not find container:\n[bold magenta]{error.container}[/bold magenta]",
            "\nCheck container exists locally:",
            display.bash(f"docker image inspect {error.container}"),
            "\nThe container either needs to be downloaded from IDEAS container registry:",
            display.bash(f"ideas containers download {error.container}"),
            "\nOr, if it's a custom container, ensure it has been built locally and the container name is spelled correctly.",
        )
    except exceptions.DockerUnavailableError:
        handle_error(*exceptions.DockerUnavailableError.get_instructions())
    except Exception:
        handle_error()


@tools_group.command("extract")
@click.pass_context
@click.argument(
    "cbundle",
    type=click.Path(
        exists=True,
        path_type=pathlib.Path,
    ),
)
@click.option(
    "-o",
    "--output-tar",
    type=click.Path(
        exists=False,
        file_okay=True,
        dir_okay=False,
        path_type=pathlib.Path,
    ),
    help="Where to dump the code bundle tar file. If not specified, only the metadata will be extracted.",
)
def tools_extract(ctx, cbundle: pathlib.Path, output_tar: pathlib.Path | None = None):
    """
    Extract code and metadata from a given bundle
    """
    try:
        tools.extract(cbundle, output_tar)
    except exceptions.ToolException as e:
        raise click.ClickException(
            f"Unable to extract specified code bundle: {e}",
        ) from e


@tools_group.command("download")
@click.pass_context
@click.argument(
    "tool_version_id",
    type=click.INT,
)
@click.option(
    "-o",
    "--output-file",
    required=False,
    type=click.Path(dir_okay=False, path_type=pathlib.Path),
)
def tools_download(ctx, tool_version_id: int, output_file: pathlib.Path | None):
    """
    Download a bundle from a given tool version on IDEAS.
    """
    try:
        tools.download(tool_version_id, output_file)
    except exceptions.ToolException as e:
        raise click.ClickException(
            f"Unable to extract specified code bundle: {e}",
        ) from e


@tools_group.command("migrate", hidden=True)
@click.pass_context
@click.option(
    "-d",
    "--code-dir",
    required=True,
    type=click.Path(
        exists=True,
        file_okay=False,
        dir_okay=True,
        path_type=pathlib.Path,
    ),
    default=pathlib.Path.cwd,
    help="Directory that contains the code to bundle. Defaults to current working directory",
)
@click.option(
    "-s",
    "--src",
    required=True,
    type=click.Path(
        exists=True,
        file_okay=False,
        dir_okay=True,
        path_type=pathlib.Path,
    ),
)
def tools_migrate(ctx, code_dir: pathlib.Path, src: pathlib.Path):
    """
    [internal] Migrate tool specs from v2 toolbox to v3 schema
    """
    try:
        tools.migrate(code_dir, src)
    except exceptions.ToolException as e:
        raise click.ClickException(
            f"Unable to migrate tool specs: {e}",
        ) from e


@cli.group("containers")
def containers_group():
    """Commands for management of containers for tool execution."""
    pass


@containers_group.command("list")
@click.pass_context
@click.option(
    "--filter",
    "filters",
    type=custom_click_types.KeyValueOption(),
    required=False,
    help="Optional and repeatable filters. For example, `--filter key=project-key --filter status=1`",
    multiple=True,
)
@click.option(
    "--search",
    required=False,
    type=custom_click_types.SearchOption(),
    default=tuple(),
    help="Will search for partial matches across some fields for the supplied string.",
)
def containers_list(ctx, filters, search):
    """
    List all containers in IDEAS registry
    """
    import datetime as dt

    from dateutil import parser
    from rich.table import Table

    table = Table(show_lines=True)
    table.add_column("Container image")
    table.add_column("Description")
    table.add_column("Tags")
    table.add_column("Size")
    table.add_column("Checksum")
    table.add_column("Updated")

    now = dt.datetime.now(dt.timezone.utc)

    for container in containers.get_containers(
        filters=filters + search,
        session=ctx.obj["session"],
    ):
        checksum = container["checksum"]
        if checksum:
            checksum = checksum[:12]

        table.add_row(
            container["full_name"],
            container["description"],
            ", ".join(container["tags"] or []),
            humanize.naturalsize(container["size"]) if container["size"] else "",
            checksum,
            humanize.naturaltime(now - parser.parse(container["date_updated"])),
        )
    display.console.print(table)


@containers_group.command("publish")
@click.pass_context
@require_tenant_id
@click.argument(
    "name",
    type=custom_click_types.LocalContainerImageOption(),
)
@click.option(
    "--description",
    "description",
    type=str,
    required=False,
    help="Short description of the image",
)
@click.option(
    "--label",
    "label",
    type=str,
    required=False,
    help="Label of the image to assign on IDEAS. Must be unique for the image namespace. If not specified, defaults to the docker image id.",
)
@click.option(
    "--namespace",
    "namespace",
    type=str,
    required=False,
    help="Namespace of the image to publish to in the IDEAS registry. If not specified, defaults to the namespace of the local image name.",
)
@click.option(
    "--tag",
    "tags",
    type=custom_click_types.ListOption(),
    required=False,
    help="Optional repeatable tags to assign to the image in IDEAS",
    multiple=True,
)
@click.option(
    "--readme",
    "readme",
    type=click.Path(
        exists=True,
        file_okay=True,
        dir_okay=False,
        path_type=pathlib.Path,
    ),
    required=False,
    help="Long readme of contents and functionality of image in markdown format.",
)
def containers_publish(
    ctx,
    name: str,
    tenant_id: int,
    description: str | None,
    label: str | None,
    namespace: str | None,
    tags: typing.Tuple[str],
    readme: pathlib.Path | None,
):
    """
    Publish a local container to IDEAS container registry
    """

    def handle_error(*args):
        display.abort(*args, title="Failed to publish container")

    try:
        response = containers.publish_container(
            name,
            description=description,
            label=label,
            namespace=namespace,
            tags=tags,
            readme=readme,
            tenant_id=tenant_id,
            session=ctx.obj["session"],
        )
        display.show_success(
            f"Your local container is now available on IDEAS, under the name:\n[bold cyan]{response['full_name']}[/bold cyan]",
            "\nNext create your code bundle",
            display.bash("ideas tools bundle"),
            title="Successfully published container!",
        )
    except exceptions.ContainerUnsupportedArchException as error:
        supported_arch = constants.SUPPORTED_ARCHITECTURES[0]
        handle_error(
            f"Container has unsupported architecture: [bold red]{error.arch}[/bold red].",
            f"IDEAS currently only supports containers with [bold green]{supported_arch}[/bold green] architectures.",
            "\nFor custom containers, ensure you specify the architecture when building the docker container image:",
            display.bash(f"docker build --platform {supported_arch} ..."),
        )
    except exceptions.ContainerNotFoundException:
        handle_error(
            "Container not found locally.",
            "\nSee the list of available containers: ",
            display.bash("docker images"),
            "\nEnsure the container name is formatted as [bold green]REPOSITORY:TAG[/bold green]",
        )
    except exceptions.ContainerPublishExistsException as error:
        display.show_warning(str(error))
    except exceptions.ContainerPublishPostException as error:
        display.show_warning(str(error), *error.suggestions)
    except exceptions.ContainerPublishPostException as error:
        display.show_error(
            f"Container with image id has already been published to IDEAS:\n{error}."
            "\nSpecify a different name to publish to with --namespace",
        )
    except exceptions.UserNotTenantMemberError:
        handle_error(
            f"You are not a member of tenant with id [bold magenta]{tenant_id}[/bold magenta] in IDEAS.",
            "\nEnsure value of option --tenant-id is correct. See list of your tenants:",
            display.bash("ideas tenants"),
        )


@containers_group.command("download")
@click.pass_context
@click.argument(
    "name",
    type=str,
)
def containers_download(ctx, name: str):
    """
    Publish a local container to IDEAS container registry
    """

    def handle_error(*args):
        display.abort(*args, title="Failed to download container")

    try:
        containers.download_container(name, session=ctx.obj["session"])
        display.show_success(
            f"Downloaded container [bold magenta]{name}[/bold magenta]",
            "\nNext configure your tool to use this container:",
            display.bash(f"ideas tools configure <key> --container {name}"),
            title="Successfully downloaded container!",
        )
    except exceptions.ContainerNotFoundException:
        handle_error(
            f"Container with name [bold magenta]{name}[/bold magenta] not found.",
            "\nSee the list of available containers: ",
            display.bash("ideas containers list"),
        )
    except exceptions.ContainerDownloadAccessException:
        handle_error(
            f"You do not have access to the container {name}."
            "\nSee the list of available containers: ",
            display.bash("ideas containers list"),
        )
    except Exception:
        handle_error()


@cli.command()
@click.option(
    "-n",
    "--non-interactive",
    "non_interactive",
    is_flag=True,
    default=False,
    help="Runs command in non interactive mode (no prompts)",
)
def log(non_interactive):
    """Open IDEAS log file"""
    if non_interactive:

        def read_log():
            with open(display.get_error_log_path()) as f:
                for line in f:
                    yield line

        click.echo_via_pager(read_log())
    else:
        open_file_with_default_app(display.get_error_log_path())


def main():
    sentry.setup_sentry()
    warnings.simplefilter("always", DeprecationWarning)

    try:
        cli(obj={})
    except exceptions.UnauthorizedError:
        display.abort(
            "Unable to authenticate, please check your login credentials before retrying:\n",
            display.bash("ideas configure"),
            log="Invalid credentials",
            exit_code=EXIT_STATUS.INVALID_CREDENTIALS,
        )
    except exceptions.InvalidProfileError:
        display.abort(
            "Specified configuration profile not found, please check value of --profile",
            display.bash("ideas configure"),
            log="Invalid configuration profile",
            exit_code=EXIT_STATUS.UNSPECIFIED_ERROR,
        )
    except exceptions.QuotaExceededError as e:
        display.abort(
            str(e),
            title="Quota exceeded",
            log="Quota exceeded error",
            exit_code=EXIT_STATUS.QUOTA_EXCEEDED,
        )
    except exceptions.ApiException as e:
        display.pretty_print(e.json)
        display.abort(
            log="Unhandled API error", exit_code=EXIT_STATUS.UNSPECIFIED_ERROR
        )
    except (exceptions.UnhandledError, Exception):
        display.abort(exit_code=EXIT_STATUS.UNHANDLED_ERROR)


if __name__ == "__main__":
    main()
