import configparser
import logging
import os
import pathlib
import platform
import typing as t

import click
import questionary

from ideas import commands
from ideas.exceptions import InvalidProfileError, UnauthorizedError
from ideas.utils import api_types, display

APP_NAME = "ideas-cli"
DEFAULT_PROFILE = "auth"
SAMPLE_CONFIG = f"""[{DEFAULT_PROFILE}]
"""

logger = logging.getLogger(__name__)


def get_default_config_path() -> pathlib.Path:
    return pathlib.Path(click.get_app_dir(APP_NAME)) / "config.ini"


def get_credentials_cache_path() -> pathlib.Path:
    return pathlib.Path(click.get_app_dir(APP_NAME)) / ".cache"


def read_cached_token(
    username: str, user_pool_id: str
) -> t.Tuple[t.Optional[str], t.Optional[str]]:
    try:
        with open(get_credentials_cache_path(), "r") as f:
            (
                stored_user_pool_id,
                stored_username,
                access_token,
                refresh_token,
            ) = f.read().split("|")

        if username != stored_username or user_pool_id != stored_user_pool_id:
            # Prevent using token stored for another account or environment
            return (None, None)

        return (access_token, refresh_token)
    except (FileNotFoundError, ValueError):
        # File is malformed or doesn't exist yet; will be regenerated on exit
        # Can pass this to a Cognito client and it will initialize itself for us
        return (None, None)


def write_cached_token(
    username: str,
    user_pool_id: str,
    access_token: t.Optional[str],
    refresh_token: t.Optional[str],
) -> None:
    """
    Write the access and refresh tokens to a cache file, taking care to not expose it to anyone but
    the current user.

    """
    # If the tokens aren't populated, abort early and clear the cached token. This is expected if
    # the auth flow failed.
    if access_token is None or refresh_token is None:
        clear_cached_token()
        return

    # Make sure the parent exists with restrictive permissions before creating
    # cache file.
    path = get_credentials_cache_path()
    path.parent.mkdir(mode=0o700, parents=True, exist_ok=True)

    fd = os.open(
        path=path,
        flags=(
            os.O_WRONLY  # access mode: write only
            | os.O_CREAT  # create if not exists
            | os.O_TRUNC  # truncate the file to zero
        ),
        mode=0o600,
    )
    with open(fd, "w") as f:
        f.write("|".join([user_pool_id, username, access_token, refresh_token]))


def clear_cached_token() -> None:
    path = get_credentials_cache_path()

    try:
        os.remove(path)
    except FileNotFoundError:
        pass


def read_from_config_file(
    config_file: pathlib.Path, profile: str = DEFAULT_PROFILE
) -> dict:
    """
    Read parameters from the configuration file, under the [auth] block by default, but optionally
    respecting the `IDEAS_CLI_PROFILE` environment variable to select another block.
    """
    config = configparser.ConfigParser(interpolation=None)
    config.read(config_file)

    config_profile = profile
    if config_profile is not DEFAULT_PROFILE:
        profile_overridden = True
    else:
        profile_overridden = False

    try:
        options = dict(config[config_profile])
    except KeyError:
        if profile_overridden:
            # Only log a warning if the user is trying to override the profile name; else, they
            # may not even have a config file nor want one.
            raise InvalidProfileError(
                f"Unable to find specified profile '{config_profile}' in {config_file}, please "
                "check --profile option."
            )
        options = {}

    return options


def configure_cli(ctx, param, config_file_path: str) -> pathlib.Path:
    """
    Take the configured parameters and apply them to the click context's default mapping. This
    allows the user to still overwrite them with individual command-line arguments.
    """
    # Callback is eager so fires before type conversion
    config_file = pathlib.Path(config_file_path)

    profile = ctx.params.get("profile", DEFAULT_PROFILE)
    try:
        ctx.default_map = read_from_config_file(config_file, profile)
    except InvalidProfileError:
        # Profile is missing, but that's okay at this point. User might be configuring it for the
        # first time. We will raise when we try to read the profile in the Session object, if not.
        pass
    return config_file


# TODO move to os_utils once merged
def _make_file_access_for_user_only_windows(config_file):
    """
    Remove all access entries for config file, and add single access entry for current user.
    Meant to mimic 0o600 settings on Unix systems, but different since
    Windows permissions are more complicated.
    This still doesn't make access to the file limited only to the current user,
    since access can be inherited from parent folders.
    However since this folder is in the user's directory, only the user and admins
    should have access by default anyways.

    Windows permissions are generally evaluated in this order:

    * If a user matches explicit deny access, they're denied permission.
    * If a user matches explicit allow access, they're granted permission.
    * If a user matches inherited deny access, they're denied permission.
    * If a user matches inherited allow access, they're granted permission.
    * If there are no matches for a user, they're denied permission by default.

    Based on this example code: https://timgolden.me.uk/python/win32_how_do_i/add-security-to-a-file.html
    """
    import getpass

    import ntsecuritycon as con
    import win32security

    sd = win32security.GetFileSecurity(
        str(config_file), win32security.DACL_SECURITY_INFORMATION
    )
    dacl = sd.GetSecurityDescriptorDacl()  # instead of dacl = win32security.ACL()
    ace_count = dacl.GetAceCount()

    for _ in range(0, ace_count):
        dacl.DeleteAce(0)

    user_name = getpass.getuser()
    userx, _, _ = win32security.LookupAccountName("", user_name)
    dacl.AddAccessAllowedAce(win32security.ACL_REVISION, con.FILE_ALL_ACCESS, userx)
    sd.SetSecurityDescriptorDacl(1, dacl, 0)  # may not be necessary
    win32security.SetFileSecurity(
        str(config_file), win32security.DACL_SECURITY_INFORMATION, sd
    )


def edit(
    profile: str,
    interactive: bool,
    include_dev: bool,
    environments_url: str | None,
    django_url: str | None,
    config_file_path: pathlib.Path | None,
) -> None:
    """
    Edits the configuration for the CLI, interactively by default but with user's preferred editor
    if interactive is false.
    """
    config_file: pathlib.Path = config_file_path or get_default_config_path()

    # Populate sample configuration if file doesn't exist - mostly as a starting point for
    # non-interactive editing.
    if not config_file.exists():
        # Make parent directory (if it doesn't already exist) and open an editor with the template
        # pre-populated, and restrict access to the owner only
        config_file.parent.mkdir(mode=0o700, parents=True, exist_ok=True)
        config_file.touch(mode=0o600, exist_ok=True)
        config_file.write_text(SAMPLE_CONFIG)

        if platform.system() == "Windows":
            _make_file_access_for_user_only_windows(config_file)

    if interactive:
        config_wizard(profile, config_file, include_dev, environments_url, django_url)
    else:
        click.edit(filename=str(config_file))


def config_wizard(
    profile: str,
    config_file: pathlib.Path,
    include_dev: bool,
    environments_url: str | None,
    django_url: str | None,
):
    """
    🧙

    Prompts user to select things like region/environment, email, password, and tenant.

    If user only belongs to a single tenant, skips the question.
    """
    from ideas.commands import get_environments
    from ideas.session import Session

    config = configparser.ConfigParser(interpolation=None)
    config.read(config_file)

    if not config.has_section(profile):
        config.add_section(profile)

    environments = get_environments(
        include_dev=False, environments_url=environments_url
    )

    try:
        current_environment = config[profile]["env"]
    except KeyError:
        # Nothing configured for this profile
        current_environment = None

    if include_dev or (
        current_environment is not None and current_environment not in environments
    ):
        # Environment could already be configured with --dev but not passed this time
        environments = get_environments(
            include_dev=True, environments_url=environments_url
        )
    if current_environment not in environments:
        # Unset, it's some unsupported environment
        current_environment = None

    # We use unsafe_ask to allow aborting the command
    environment = questionary.select(
        "Which region are you in?",
        choices=environments,
        default=current_environment,
    ).unsafe_ask()
    config[profile]["env"] = environment

    configure_auth = True
    if "username" in config[profile]:
        # Ask user if they want to reconfigure auth if they have something set already
        configure_auth = questionary.confirm(
            "Do you want to change your IDEAS credentials?"
        ).unsafe_ask()
    if configure_auth or not config[profile]["username"]:
        # If user answered yes, or they don't have a username set, prompt for password
        email = questionary.text("Email:").unsafe_ask()
        password = questionary.password("Password:").unsafe_ask()

        config[profile]["username"] = email
        config[profile]["password"] = password

    # Write profile so far so Session doesn't complain
    with config_file.open("w") as f:
        config.write(f)

    # Test, retrieve tenants
    with display.console.status("Retrieving list of tenants"):
        try:
            session = Session(
                env=environment,
                # Session takes null values (like returned by questionary if user skips question) to
                # mean read from config, but it's confusing to use credentials the user didn't specify
                username=config[profile]["username"],
                password=config[profile]["password"],
                use_token=False,
                environments_url=environments_url,
                django_url=django_url,
                profile=profile,
                config_file_path=config_file,
            )
        except Exception:
            raise UnauthorizedError()

        tenants = list(commands.get_tenants(tuple(), session=session))

    if len(tenants) == 0:
        display.show_error("No tenants found, please contact support.")
        # Still want to write config, but user can re-configure to select a tenant once this has
        # been resolved
        tenant_id = None
    elif len(tenants) == 1:
        # Auto-select this tenant
        tenant_id = str(tenants[0]["id"])
    elif len(tenants) > 1:

        def to_choice(tenant: api_types.Tenant) -> questionary.Choice:
            return questionary.Choice(
                title=tenant["name"],
                value=tenant["id"],
            )

        tenant_choices = list(map(to_choice, tenants))
        try:
            current_tenant_id = int(config[profile]["tenant_id"])
        except KeyError:
            default = None
        else:
            # Preserve user's current choice as default, providing it's still an available tenant
            default: questionary.Choice | None = next(
                (t for t in tenant_choices if t.value == current_tenant_id), None
            )

        selected_tenant = questionary.select(
            "Which tenant would you like to use?",
            choices=tenant_choices,
            default=default,
        ).unsafe_ask()
        tenant_id = selected_tenant

    if tenant_id:
        config[profile]["tenant_id"] = str(tenant_id)

    with config_file.open("w") as f:
        config.write(f)

    # Replace token in case auth changed
    session.write_cached_token()


def get_config_profiles(config_file_path: str | None = None) -> list[str]:
    """
    Returns a list of profiles configured for the specified config file. If not config file is
    specified, uses the default config path.
    """
    if config_file_path is not None:
        config_file: pathlib.Path = (
            pathlib.Path(config_file_path).expanduser().resolve()
        )
    else:
        config_file = get_default_config_path()

    if not config_file.exists():
        return []

    config = configparser.ConfigParser(interpolation=None)
    config.read(config_file)
    return config.sections()
