import enum
import logging
import pathlib
import typing

import docker
import docker.errors
from rich.progress import (
    BarColumn,
    DownloadColumn,
    Progress,
    TextColumn,
    TransferSpeedColumn,
)

from ideas import constants, exceptions
from ideas.commands import validate_tenant
from ideas.session import Session, get_default_session
from ideas.utils import api_types, display, http
from ideas.utils.docker_utils import get_docker_client

logger = logging.getLogger()


class ContainerImageStatus(enum.IntEnum):
    PUBLISHED = 1
    DEPRECATED = 2
    UPLOADING = 3


def get_containers(
    filters, session: typing.Optional[Session] = None
) -> typing.Iterator[api_types.Container]:
    """Get containers published in IDEAS"""
    session = session or get_default_session()

    yield from http.handle_pagination(
        http.get,
        f"{session.base_url}/api/v0/tes/image/",
        session.headers,
        session.auth,
        filters,
    )


def get_repo_digests_for_registry(
    name: str, session: Session, repository: typing.Optional[str] = None
):
    """
    Get the repo digest of a local image.
    A repo digest is a unique hash of an image that has been published to a docker registry.
    If the local image has been published to IDEAS registry, the repo digest for that registry will be returned.
    An image can have many repo digests, all repo digests which begin with the session registry url will be returned.
    Otherwise, if the local image has never been published, or published to a different registry, None will be returned.
    """
    client = get_docker_client()
    image = client.images.get(name)
    repo_digests = image.attrs.get("RepoDigests", [])
    matched_repo_digests = []

    registry_url = session.registry_url
    if repository:
        registry_url += f"/{repository}"
    for repo_digest in repo_digests:
        if repo_digest.startswith(registry_url):
            matched_repo_digests.append(repo_digest.rsplit("@sha256:", maxsplit=1)[-1])
    logger.info(
        f"Found repo digests from image matching registry url: {registry_url}, {repo_digests}"
    )
    return matched_repo_digests


def get_container_from_repo_digest(
    repo_digests: typing.List[str],
    tenant_id: int,
    session: typing.Optional[Session] = None,
):
    """
    Get IDEAS container info based on repo digests for IDEAS registry inspected from local docker image
    Return the first container in IDEAS that has a checksum which matches one of the repo digests.
    """
    for repo_digest in repo_digests:
        result = list(
            get_containers(
                filters=(
                    ("checksum", repo_digest),
                    ("tenant", tenant_id),
                ),
                session=session,
            )
        )
        # NOTE: For end users publishing/downloading through the CLI, the expectation is that there
        # will be only one container image in a tenant repo with a particular checksum to enable
        # proper mapping of local to remote images.
        # When you publish, the cli checks whether there are existing images in the repo with the
        # same checksum, which should prevent duplicates in the db.
        # HOWEVER, there is nothing in the backend that places this constraint on the model
        # So edge cases are possible where multiple images with the same checksum in a tenant repo are found.
        # For now, use the first match. If the checksum (i.e., repo digest) is the same,
        # then the images should be the same. May need to revist in the future if we want to enable
        # the duplicate checksums in a tenant repo.
        if len(result) >= 1:
            if len(result) > 1:
                logger.warning(
                    "More than one container image found with matching checksum, defaulting to first container image in IDEAS"
                )
            return result[0]
    return None


def split_full_name(full_name: str):
    repository = full_name.split(":", maxsplit=1)
    if len(repository) == 1:
        repository, tag = repository[0], "latest"
    else:
        repository, tag = repository

    return repository, tag


def publish_container(
    full_name: str,
    description: str,
    tenant_id: int,
    label: typing.Optional[str] = None,
    namespace: typing.Optional[str] = None,
    tags: typing.Tuple[str] = (),
    readme: typing.Optional[pathlib.Path] = None,
    session: typing.Optional[Session] = None,
):
    logger.debug(f"Publishing container: {full_name}")

    session = session or get_default_session()
    tenant_id = tenant_id or session.tenant_id
    validate_tenant(tenant_id=tenant_id, session=session)

    try:
        client = get_docker_client()
        image = client.images.get(full_name)
    except docker.errors.ImageNotFound:
        raise exceptions.ContainerNotFoundException(
            f"Failed to find local container {full_name}"
        )
    except docker.errors.DockerException:
        raise exceptions.DockerExecutionError(
            f"Failed to find local container {full_name} due to unexpected error with docker."
        )

    _, image_id = image.attrs["Id"].split(":", maxsplit=1)
    size = image.attrs["Size"]
    arch = image.attrs["Architecture"]

    if arch not in constants.SUPPORTED_ARCHITECTURES:
        raise exceptions.ContainerUnsupportedArchException(
            f"Container arch {arch} not supported in IDEAS", arch=arch
        )

    # full name should be parsable since docker image is retrieved by full name earlier
    name, _ = split_full_name(full_name=full_name)

    # If user-friendly label provided, use that as the tag in IDEAS
    # otherwise default to the first 12 characters of the image id
    if label:
        tag = label
    else:
        tag = image_id[:12]

    # If user-friendly namespace provided, use that as the namespace in IDEAS
    # otherwise default to namespace of local image
    if namespace:
        name = namespace

    data = {
        "name": name,
        "label": tag,
        "description": description,
        "tags": tags,
        "size": size,
        "tenant": tenant_id,
        # Temporarily use image ID to allow re-pushing; this will be replaced with repo digest
        # later on the backend
        "checksum": image_id,
    }

    if readme:
        try:
            with open(readme, "r") as f:
                data["readme"] = f.read()
        except FileNotFoundError:
            raise exceptions.ContainerException(
                f"Readme file for container not found: {readme}"
            )

    # examine image checksum/repo digest and determine if it exists in the be
    logger.debug("Checking existence of container image_id in IDEAS container registry")
    ideas_container = None

    repo_digests = get_repo_digests_for_registry(name=full_name, session=session)
    # Use image ID to find image, in case it's in UPLOADING state (we don't know the repo digest
    # until the image is published)
    repo_digests.append(image_id)
    ideas_container = get_container_from_repo_digest(
        repo_digests=repo_digests, tenant_id=tenant_id, session=session
    )

    if ideas_container:
        if ideas_container["status"] != ContainerImageStatus.UPLOADING.value:
            raise exceptions.ContainerPublishExistsException(
                f"This container has already been published and exists on IDEAS as {ideas_container['full_name']}."
            )
        elif ideas_container["name"] != name or ideas_container["label"] != tag:
            # We found an UPLOADING image matching this image, but the user is changing the name
            # or the label. Give them instructions on how to resume the publish with the matching
            # name/label instead.
            raise exceptions.ContainerPublishExistsException(
                f"This container was partially published to IDEAS as {ideas_container['full_name']}.",
                suggestions=[
                    "\nTo finish publishing, run:\n",
                    display.bash(
                        f"ideas containers publish {full_name} --namespace {ideas_container['name']} --label {ideas_container['label']}"
                    ),
                ],
            )
        logger.debug("Container exists in IDEAS registry, skipping creation")
    else:
        logger.debug(
            "Container doesn't exist in IDEAS registry, proceeding with creation"
        )
        ideas_container = http.post(
            f"{session.base_url}/api/v0/tes/image/",
            session.headers,
            session.auth,
            data,
        )

    logger.debug(
        f"Pushing container to IDEAS container registry with name: {ideas_container['full_name']}"
    )

    # get repository name from BE response
    repository = ideas_container["repository"]
    token = session.get_access_token()
    registry_repository_name = f"{session.registry_url}/{repository}"
    try:
        client.login(username=tag, password=token, registry=session.registry_url)
        image = client.images.get(full_name)
        image.tag(registry_repository_name, tag=tag)

        progress = Progress(
            TextColumn("{task.fields[layer_id]}: {task.fields[status]}"),
            BarColumn(bar_width=40),
            DownloadColumn(),
            TransferSpeedColumn(),
            transient=True,
        )
        progress.start()
        tasks = {}

        try:
            for chunk in client.images.push(
                registry_repository_name, tag=tag, stream=True, decode=True
            ):
                layer_id = chunk.get("id")
                status = chunk.get("status", "")
                detail = chunk.get("progressDetail", {})
                current_bytes = detail.get("current", 0)
                total_bytes = detail.get("total", 0)

                if not layer_id:
                    # Not something we can progress on
                    continue

                if layer_id not in tasks:
                    # Initialize this layer in the progress bar tasks
                    tasks[layer_id] = progress.add_task(
                        "layer",
                        layer_id=layer_id,
                        total=total_bytes,
                        status="Waiting",
                    )

                if status in ("Preparing", "Waiting", "Pushed"):
                    # Update label and total size for layers waiting for push
                    try:
                        progress.update(
                            tasks[layer_id],
                            status=status,
                            total=total_bytes,
                        )
                    except KeyError:
                        logger.debug(
                            "Failed to update progress bar with total size on container publish"
                        )
                elif status == "Layer already exists":
                    # Remove progress bar for this layer; we can't know the layer size so it's just
                    # confusing to the user
                    progress.remove_task(tasks[layer_id])
                elif total_bytes:
                    # Pushing, we know the layer total size
                    try:
                        progress.update(
                            tasks[layer_id],
                            completed=current_bytes,
                            total=total_bytes,
                            status=status,
                        )
                    except KeyError:
                        logger.debug(
                            "Failed to update progress bar with total size on container publish"
                        )

        finally:
            progress.stop()
    except docker.errors.DockerException:
        raise exceptions.DockerExecutionError(
            "Failed to push local container to IDEAS container registry due to unexpected error with docker."
        )

    return ideas_container


def download_container(full_name: str, session: typing.Optional[Session] = None):
    session = session or get_default_session()
    logger.debug(f"Downloading image {full_name} from IDEAS registry")
    repository, tag = split_full_name(full_name=full_name)

    result = list(
        get_containers(
            filters=(("repository", repository), ("label", tag)),
            session=session,
        )
    )

    if len(result) != 1:
        raise exceptions.ContainerNotFoundException(
            f"Failed to find container with name {full_name} in IDEAS container registry"
        )

    result = result[0]
    registry_repository_name = f"{session.registry_url}/{repository}"
    token = session.get_access_token()

    try:
        client = get_docker_client()
        client.login(username=tag, password=token, registry=session.registry_url)

        progress = Progress(
            TextColumn("{task.fields[layer_id]}: {task.fields[status]}"),
            BarColumn(bar_width=40),
            DownloadColumn(),
            TransferSpeedColumn(),
            transient=True,
        )
        progress.start()
        tasks = {}

        try:
            for chunk in client.api.pull(
                registry_repository_name,
                tag=tag,
                platform=constants.CONTAINER_ARCHITECTURE,
                stream=True,
                decode=True,
            ):
                layer_id = chunk.get("id")
                status = chunk.get("status", "")
                detail = chunk.get("progressDetail", {})
                current_bytes = detail.get("current", 0)
                total_bytes = detail.get("total", 0)

                if not layer_id or status.startswith("Pulling from"):
                    # Not something we can progress on, or top-level image pulling status, we can skip
                    continue

                if layer_id not in tasks:
                    # Initialize this layer in the progress bar tasks
                    tasks[layer_id] = progress.add_task(
                        "layer",
                        layer_id=layer_id,
                        total=total_bytes,
                        status="Waiting",
                    )

                if status in (
                    "Pulling fs layer",
                    "Waiting",
                    "Verifying Checksum",
                    "Download complete",
                    "Pull complete",
                ):
                    # Update label and total size for layers waiting for push
                    try:
                        progress.update(
                            tasks[layer_id],
                            status=status,
                        )
                    except KeyError:
                        logger.debug(
                            "Failed to update progress bar with label and total size on container download"
                        )
                elif status in ["Already exists"]:
                    progress.remove_task(tasks[layer_id])
                elif total_bytes:
                    # Pushing, we know the layer total size
                    try:
                        progress.update(
                            tasks[layer_id],
                            completed=current_bytes,
                            total=total_bytes,
                            # status="Downloading",
                            status=status,
                        )
                    except KeyError:
                        logger.debug(
                            "Failed to update progress bar with layer total size on container download"
                        )
        finally:
            progress.stop()

        image = client.images.get(f"{registry_repository_name}:{tag}")
        image.tag(repository, tag=tag)
    except docker.errors.ImageNotFound:
        raise exceptions.ContainerDownloadAccessException(
            f"Failed to find {full_name} in IDEAS container registry. Ensure you have download access to the container."
        )
    except docker.errors.DockerException:
        raise exceptions.DockerExecutionError(
            "Failed to download containe from IDEAS container registry due to unexpected error with docker."
        )
