import functools
import logging
import pathlib
import typing
from dataclasses import dataclass

import botocore.exceptions
import questionary
from pycognito.exceptions import SoftwareTokenMFAChallengeException
from pycognito.utils import Cognito, RequestsSrpAuth, TokenType
from pyotp import TOTP

from ideas import config
from ideas.environments import (
    DEFAULT_ENVIRONMENTS_URL,
    Environment,
    get_environments,
)
from ideas.exceptions import UnauthorizedError

logger = logging.getLogger(__name__)


def _auth_with_mfa(cognito, password, mfa_secret):
    try:
        cognito.authenticate(password=password)
    except SoftwareTokenMFAChallengeException:
        if not mfa_secret:
            code = questionary.text("Enter the 6-digit TOTP code:").unsafe_ask()
        else:
            code = TOTP(mfa_secret).now()
        cognito.respond_to_software_token_mfa_challenge(code)


def get_cognito_auth(
    username,
    password,
    mfa_secret,
    environment: Environment,
    use_token: bool = True,
) -> RequestsSrpAuth:
    if use_token:
        access_token, refresh_token = config.read_cached_token(
            username, environment["USER_POOL_ID"]
        )
    else:
        access_token = None
        refresh_token = None

    if not username or not password:
        raise UnauthorizedError()

    cognito = Cognito(
        user_pool_id=environment["USER_POOL_ID"],
        client_id=environment["CLIENT_ID"],
        user_pool_region=environment["REGION"],
        username=username,
        access_token=access_token,
        refresh_token=refresh_token,
    )
    # Will refresh token if it has expired; if we get an exception here it is likely because the
    # refresh token has expired, so we need to re-authenticate.
    try:
        cognito.check_token()
    except AttributeError:
        # Raised by `Cognito` when there's no access token to begin with, so leave the auth to
        # PyCognito in this case
        _auth_with_mfa(cognito, password, mfa_secret)
    except botocore.exceptions.ClientError as e:
        # We can't get access to the client to catch specific exceptions for client errors, so
        # we have to inspect the error code as follows:
        if e.response["Error"]["Message"] == "Refresh Token has expired":
            logger.debug("Re-authenticating due to refresh token expiration.")
            _auth_with_mfa(cognito, password, mfa_secret)
        else:
            # Unknown error, pass the exception up
            raise

    return RequestsSrpAuth(
        username=username,
        password=password,
        user_pool_id=environment["USER_POOL_ID"],
        client_id=environment["CLIENT_ID"],
        user_pool_region=environment["REGION"],
        # Populate Cognito client just as RequestsSrpAuth does it, but pass the cached token
        # RequestsSrpAuth will check it for expiry
        cognito=cognito,
    ), cognito


@dataclass
class Session:
    """
    Defines a session that can be used for all commands, handling authentication and token caching.
    """

    auth: RequestsSrpAuth
    base_url: str
    headers: dict
    tenant_id: int
    username: str

    def __init__(
        self,
        env=None,
        username=None,
        password=None,
        django_url=None,
        registry_url=None,
        environments_url=None,
        tenant_id=None,
        mfa_secret=None,
        config_file_path: typing.Optional[pathlib.Path] = None,
        use_token: bool = True,
        profile: str = config.DEFAULT_PROFILE,
    ):
        if config_file_path is None:
            config_file_path = config.get_default_config_path()

        self.options = config.read_from_config_file(config_file_path, profile)

        # Retrieve options that users can override from CLI arguments, falling back to configuration
        django_url = django_url or self.options.get("django_url")
        registry_url = registry_url or self.options.get("registry_url")
        environments_url = (
            environments_url
            or self.options.get("environments_url")
            or DEFAULT_ENVIRONMENTS_URL
        )

        self.username = username or self.options.get("username")
        password = password or self.options.get("password")
        mfa_secret = mfa_secret or self.options.get("mfa_secret", None)
        env = env or self.options.get("env", "")

        try:
            self.environment = get_environments(environments_url, include_dev=True)[env]
        except KeyError:
            raise UnauthorizedError()

        self.auth, self.cognito = get_cognito_auth(
            self.username,
            password,
            mfa_secret,
            self.environment,
            use_token=use_token,
        )
        self.base_url = django_url or self.environment["DJANGO_URL"]
        self.registry_url = registry_url or self.environment["REGISTRY_URL"]
        self.tenant_id = tenant_id or self.options.get("tenant_id")
        self.headers = {"x-tenant-id": self.tenant_id}

    def write_cached_token(self):
        config.write_cached_token(
            self.username,
            self.environment["USER_POOL_ID"],
            self.auth.cognito_client.access_token,
            self.auth.cognito_client.refresh_token,
        )

    def get_access_token(self):
        """
        Retrieve access token from RequestsSrpAuth object
        Based on this code from the RequestsSrpAuth class:
        https://github.com/NabuCasa/pycognito/blob/master/pycognito/utils.py#L86
        """
        # Checks if token is expired and fetches a new token if available
        self.cognito.check_token(renew=True)
        token = getattr(self.cognito, TokenType.ACCESS_TOKEN)
        return token

    def __del__(self):
        """
        TODO write the cached cognito tokens.
        """
        pass


@functools.cache
def get_default_session():
    """
    Return a singleton `Session` object to be used where an explicit `Session` isn't passed, so that
    caching the auth works.

    TODO deprecate this as it doesn't support overriding things like --profile, etc. I don't think
    that it's actually used outside of tests.
    """
    return Session()


class LazySession:
    """
    Like Session, but lazily evaluated the first time the session is accessed.
    """

    def __init__(self, ctx, *args, **kwargs):
        self._ctx = ctx
        self._args = args
        self._kwargs = kwargs
        self._session = None

    def _create_session(self):
        if self._session is None:
            self._session = Session(
                *self._args,
                **self._kwargs,
            )

            # Register the cache callback
            self._ctx.call_on_close(self._session.write_cached_token)
        return self._session

    def __getattr__(self, name):
        return getattr(self._create_session(), name)
