import json
import logging
import os
import pathlib
import sys
import typing
from datetime import datetime

import requests
from ideas_schemas import schemas
from ideas_schemas.exceptions import SchemaValidationError

from ideas import constants, exceptions
from ideas.session import Session, get_default_session
from ideas.tools.bundle import (
    BUNDLE_METADATA_SCHEMA_VERSION,
    CodeBundle,
    extract_code_bundle,
)
from ideas.tools.executor import Tool
from ideas.utils import api_types, display, http

logger = logging.getLogger(__name__)


def configure(
    key: str,
    file: str,
    function: typing.Optional[str],
    container: str,
    code_dir: typing.Optional[str] = None,
) -> None:
    """
    Configure tool for local execution.

    :param key: The key of the tool in IDEAS. Configured in the cloud when a tool is created.
    :param file: The file containing the entry point of analysis code to run for the tool.
        For python execution, this is the python module containing the entry point function to execute.
    :param function: The entry point function within the specified file to run for the tool.
    :param container: The container to run the local code within.
    :param code_dir: The code directory to configure tools within. By default, the current directory is used.

    :raises ToolMisconfiguredError: If the tool is misconfigured.
    :raises ToolIntrospectionError: If the code file could not be introspected.
    :raises ToolConfigFileNotFoundError: If the code file could not be found in the user's local codebase.
    :raises ToolConfigFunctionNotFoundError: If the function could not be found in the code file.
    :raises ToolConfigContainerNotFoundError: If the container could not be found in the local docker images.
    :raises ToolSchemaValidationError: If the tool spec failed to validate using the v3 tool spec IDEAS schema.
    :raises ToolInputsFormatError: If the tool inputs are formatted incorrectly.
    :raises DockerExecutionError: If there was an unexpected error with docker.
    """
    logger.debug(
        f"Generating tool config:\n\tkey: {key}\n\tfile: {file}\n\tfunction: {function}\n\tcontainer: {container}"
    )
    tool = Tool(key=key, code_dir=code_dir)
    try:
        tool.load()
    except (
        exceptions.ToolMisconfiguredError,
        exceptions.ToolConfigContainerNotFoundError,
    ):
        pass
    tool.configure(file=file, function=function, container=container)


def run(
    key: str,
    inputs_file: typing.Optional[str] = None,
    clean: bool = False,
    code_dir: typing.Optional[str] = None,
    skip_introspect: bool = False,
    gpus: typing.Optional[str] = None,
) -> None:
    """
    Run tool locally by executing local analysis code in container image and process output files.

    :param key: The key of the tool in IDEAS. Configured in the cloud when a tool is created.
    :param inputs_file: The file containing inputs to run the tool with, formatted in JSON.
        By default, .ideas/<key>/inputs.json is used.
    :param clean: Remove previous tool output directories.
    :param code_dir: The code directory to configure tools within. By default, the current directory is used.
    :param skip_introspect: Skips re-introspecting function to ensure all arguments are up to date.
    :param gpus: Enable gpu access to docker container. Value can either be `all` indicating to use all gpus, or a number indicating the amount of gpus required. Uses nvidia gpu docker runtime.

    :raises ToolMisconfiguredError: If the tool is not configured correctly.
    :raises ToolConfigFileNotFoundError: If the code file could not
        be found in the user's local codebase.
    :raises ToolConfigFunctionNotFoundError: If the function could not
        be found in the code file.
    :raises ToolConfigContainerNotFoundError: If the container could not
        be found in the local docker images.
    :raises ToolInputsFormatError: If the tool inputs are formatted incorrectly.
    :raises ToolExecutionError: If there was an error running the analysis code in the docker container.
    :raises ToolException: If there was an unexpected tool error.
    :raises DockerExecutionError: If there was an unexpected error with docker.
    """
    logger.debug(f"Running tool {key}")

    tool = Tool(key=key, code_dir=code_dir)
    tool_output_dir = tool.run(
        inputs_file=inputs_file,
        clean=clean,
        skip_introspect=skip_introspect,
        gpus=gpus,
    )
    return tool_output_dir


def bundle(
    code_dir: pathlib.Path,
    tenant_id: int,
    skip_container_check: bool = False,
    skip_validation: bool = False,
    output_file: str | None = None,
    excludes: typing.Tuple[str] = (),
    included_tools: typing.Tuple[str] = (),
    session: typing.Optional[Session] = None,
) -> str:
    """
    Bundle the metadata and source code for IDEAS to consume.

    :param code_dir: The code directory to configure tools within.
    :param tenant_id: The tenant, or organization, that owns this tool in IDEAS.
    :param skip_container_check: Skip checking whether the containers configured
        for tools are published to IDEAS.
    :param skip_validation: Skip validation of the bundle metadata.
    :param output_file: The name of the output bundle file.
        If not specified, the bundle name is auto-generated,
        based on name of code directory, and the current date time.
    :param excludes: Files or directories to exclude from the bundle.
    :param included_tools: Optional list of tool keys to include in the bundlle.
    :param session: Session to use for requests to IDEAS.

    :returns: The name of the output bundle file.
    :raises ToolMisconfiguredError: If a tool in the codebase is not configured correctly.
    :raises ToolNotFoundError: If no tools were configured in the codebase.
    :raises ToolBundleBuildError: If the bundle failed to build.
    :raises ContainerNotPublishedError: If a container configured for a tool has not been published to IDEAS.
    """
    session = session or get_default_session()

    code_dir_name = os.path.basename(code_dir)
    timestamp = datetime.now().strftime("%Y-%m-%dT%H.%M.%S")
    if output_file is None:
        output_file = code_dir / f"{code_dir_name}_{timestamp}_bundle.cbundle"

    bundle = CodeBundle(
        code_dir,
        metadata_dir=code_dir / constants.IDEAS_DIR_NAME,
        included_tools=included_tools,
    )
    with display.console.status("Writing bundle..."):
        bundle.write(
            output_file,
            skip_container_check=skip_container_check,
            skip_validation=skip_validation,
            excludes=excludes,
            tenant_id=tenant_id,
            session=session,
        )
    logger.debug(f"Wrote bundle to {output_file}")
    return output_file


def extract(
    cbundle: pathlib.Path, output_tar_file: typing.Optional[pathlib.Path]
) -> None:
    """
    Extract metadata (and optionally, the source code tar file) for a specified code bundle.

    :param cbundle: The path to the bundle file to extract.
    :param output_tar_file: Optional tar output file to write the source code from the bundle.

    :raises ToolBundleExtractError: If the bundle failed to extract.
    """
    metadata = extract_code_bundle(cbundle, output_tar_file)
    display.pretty_print(metadata)
    if output_tar_file is not None:
        sys.stderr.write(f"Wrote tar bundle to {output_tar_file}\n")


def download(
    tool_version_id: int,
    output_file: pathlib.Path | None = None,
    session: typing.Optional[Session] = None,
) -> None:
    """
    [internal use only] Download bundle for a tool version from IDEAS.

    :param tool_version_id: The id of the tool version.
    :param output_file: The path to save the downloaded bundle.
        If not specified, the bundle name is auto-generated,
        based on name of code directory, and the current date time.
    :param session: Session to use for requests to IDEAS.

    :raises ToolException: If there was an unexpected tool error.
    """
    session = session or get_default_session()

    tool_version = typing.cast(
        api_types.ToolVersion,
        http.get(
            f"{session.base_url}/api/v0/trs/tool/version/{tool_version_id}/",
            session.headers,
            session.auth,
        ),
    )
    key = tool_version["tool"]["key"]
    tool_spec = tool_version["draft_tool_spec"] or tool_version["tool_spec"]

    if output_file is None:
        timestamp = datetime.now().strftime("%Y-%m-%dT%H.%M.%S")
        output_file = pathlib.Path(f"{key}_{timestamp}_bundle.cbundle")

    metadata = {
        "schema_version": BUNDLE_METADATA_SCHEMA_VERSION,
        key: {
            "tool_spec": tool_spec,
        },
    }
    try:
        schemas.validate_v3_bundle_metadata(metadata)
    except SchemaValidationError as e:
        logger.error(
            "Unable to validate bundle metadata: %s",
            e.message,
            extra={"metadata": metadata},
        )
        raise exceptions.ToolException(
            f"Unable to validate bundle metadata: {e.message}"
        )

    # Retrieve code file
    code_file_id = tool_version["code_file"]

    if code_file_id is None:
        raise exceptions.ToolException(
            f"Tool version {tool_version_id} does not have a current code bundle associated with it, cannot download."
        )

    # Stream code file into bundle
    response = typing.cast(
        api_types.FileDownload,
        http.get(
            f"{session.base_url}/api/{http.IDEAS_API_VERSION}/drs/code_file/{code_file_id}/download_url/",
            session.headers,
            session.auth,
        ),
    )
    url = response["Url"]
    response = requests.get(url, stream=True)
    response.raise_for_status()

    with CodeBundle._write(output_file, metadata) as f:
        for chunk in response.iter_content(chunk_size=8192):
            f.write(chunk)

    logging.debug(f"Wrote bundle to {output_file}")


def migrate(code_dir, src):
    """
    Migrate tool specs from v2 toolbox to v3 schema.
    Propagates any customizations from the v2 tool specs to the v3 tool specs.

    :code_dir: The code directory for the new v3 tools
    :src: The src directory of the v2 toolbox
    """
    # Read v2 toolbox info
    with open(src / "info" / "toolbox_info.json", "r") as f:
        toolbox_info = json.load(f)

    # Process each tool spec and migrate
    for tool_spec in toolbox_info["tools"]:
        key = tool_spec["key"]
        tool_config_dir = code_dir / ".ideas" / key
        if tool_config_dir.exists():
            with open(tool_config_dir / "tool_spec.json", "r") as f:
                new_tool_spec = json.load(f)

            new_tool_spec["name"] = tool_spec["name"]
            new_tool_spec["description"] = tool_spec["help"]
            new_tool_spec["visibility"] = "public"

            # Read current version and increment major version
            major_version = int(tool_spec["version"].split(".")[0])
            major_version += 1
            new_tool_spec["version"] = f"{major_version}.0.0"

            new_tool_spec["resources"]["cpu"] = tool_spec["resources"]["cpu"]
            new_tool_spec["resources"]["gpu"] = tool_spec["resources"]["gpu"]
            new_tool_spec["resources"]["memory"] = tool_spec["resources"]["memory"]

            for param in tool_spec["params"]:
                for new_param in new_tool_spec["params"]:
                    if new_param["key"] != param["key"]:
                        continue

                    new_param["name"] = param["name"]
                    new_param["description"] = param["help"]
                    new_param["required"] = param["required"]

                    new_param["type"]["param_type"] = param["type"]["param_type"]
                    optional_keys = [
                        "min",
                        "max",
                        "choices",
                        "display_options",
                        "file_formats",
                    ]
                    for optional_key in optional_keys:
                        if optional_key in param["type"]:
                            new_param["type"][optional_key] = param["type"][
                                optional_key
                            ]

                    if "default" in param:
                        new_param["type"]["default"] = param["default"]

                    if "display" in param["type"]:
                        new_param["group"] = param["type"]["display"]["group"]

            new_results = []
            for result in tool_spec["results"][0]["files"]:
                new_results.append(
                    {
                        "key": result["result_key"],
                        "name": result["result_name"],
                        "description": result["help"],
                        "file_patterns": [f"{result['result_key']}**"],
                    }
                )
            new_tool_spec["results"] = new_results

            try:
                schemas.validate_v3_tool_spec(new_tool_spec)
            except SchemaValidationError as e:
                raise exceptions.ToolSchemaValidationError(
                    f"Failed to validate v3 tool spec\n{e.message}"
                )

            with open(tool_config_dir / "tool_spec.json", "w") as f:
                json.dump(new_tool_spec, f, indent=4)
