"""Module with framework for generating output data in IDEAS tools"""

import json
import logging
from contextlib import contextmanager
from dataclasses import dataclass, field
from pathlib import Path
from typing import Dict, List, Optional, TypeVar, Union

from ideas import constants, exceptions

logger = logging.getLogger()

# On most Linux max filename len is 255 bytes, leave some buffer for prefix, suffix, and ext
MAX_FILENAME_LEN = 200


def input_paths_to_output_prefix(
    *input_paths: List[List[str | Path | None] | str | Path | None],
    max_name_len=15,
    sep="_",
):
    """ "
    Generate an output prefix from a set of input paths

    :param input_paths: Set of input paths, can be list of strings, paths, or lists of strings or paths
    :param max_name_len: Maximum number of characters to use from each input path
    :param sep: Seperator to use between input paths
    """
    output_prefix = ""
    for _input_paths in input_paths:
        if not isinstance(_input_paths, list):
            _input_paths = [_input_paths]

        for _input_path in _input_paths:
            if _input_path is None:
                continue
            if isinstance(_input_path, str):
                _input_path = Path(_input_path)
            input_name = _input_path.stem.replace(" ", sep)

            if output_prefix:
                output_prefix += "_"
            output_prefix += input_name[0 : min(len(input_name), max_name_len)]

    if len(output_prefix) > (MAX_FILENAME_LEN - 1):
        output_prefix = output_prefix[0 : (MAX_FILENAME_LEN - 1)]
    output_prefix += sep

    return output_prefix


def _load_and_remove_output_metadata():
    """[internal] Load and remove v2 output metadata to register"""
    path = Path("output_metadata.json")
    with open(path, "r") as f:
        metadata = json.load(f)
    path.unlink()
    return metadata


def _resolve_output_file_path(
    output_file: Path,
    output_dir: Path,
    subdir: Optional[str] = None,
    name: Optional[str] = None,
    prefix: Optional[str] = None,
    suffix: Optional[str] = None,
):
    """Helper function for resolving output file paths relative to an output directory"""
    # See if output file exists
    if not output_file.exists():
        # See if output file exists relative to output dir
        if output_file.is_absolute():
            raise exceptions.OutputDataFileDoesNotExist(
                f"Failed to find output file while writing output data: {output_file}"
            )

        # check if relative file exists in output dir
        if (output_dir / output_file).exists():
            output_file = output_dir / output_file
        else:
            # check if relative file exists in subdir
            if subdir and (output_dir / subdir / output_file).exists():
                output_file = output_dir / subdir / output_file
            else:
                raise exceptions.OutputDataFileDoesNotExist(
                    f"Failed to find output file while writing output data: {output_file}"
                )

    original_output_dir = output_dir
    if subdir:  # append subdir if specified
        output_dir = output_dir / subdir
    if not output_dir.exists():
        output_dir.mkdir()

    # rename output file as neccessary
    if output_dir not in output_file.parents:
        output_file = output_file.rename(output_dir / output_file.name)
    if name:
        output_file = output_file.rename(output_dir / name)
    if prefix:
        output_file = output_file.rename(output_dir / (prefix + output_file.name))
    if suffix:
        output_file = output_file.rename(
            output_dir / (output_file.stem + suffix + output_file.suffix)
        )

    return output_file.relative_to(original_output_dir)


"""Tag is a categorical label that describes an output file."""
Tag = TypeVar("Tag", bound=str)


@dataclass
class Metadata:
    """
    Piece of key, value data that describes an output file.

    :param key: Unique identifier for the metadata across output files
    :type key: str
    :param name: Friendly name for the metadata to display in IDEAS
    :type name: str
    :param value: The value of the metadata
    :type value: Union[str, bool, int, float, List[str]]
    """

    key: str
    name: str
    value: Union[str, bool, int, float, List[str]]

    def to_dict(self):
        """Serialize to dict to save in output data file"""
        return {"key": self.key, "name": self.name, "value": self.value}

    @classmethod
    def from_dict(cls, instance):
        """Deserialize from dict in output data file"""
        return cls(key=instance["key"], name=instance["name"], value=instance["value"])


@dataclass
class Preview:
    """
    A file which describes an output file.
    Preview files can be figures (i.e., images), movies (i.e., videos), json, or even an html webpage (static assets only).
    Previews are meant to be small in size, giving a brief glance, or impression at the results store in particular output file.

    :param file_path: The path to the preview file
    :type file_path: Path
    :param caption: A short description of the preview.
    :type caption: str
    """

    file_path: Path
    caption: str

    def to_dict(self):
        """Serialize to dict to save in output data file"""
        return {"file": str(self.file_path), "caption": self.caption}

    @classmethod
    def from_dict(cls, instance):
        """Deserialize from dict in output data file"""
        return cls(file_path=Path(instance["file"]), caption=instance["caption"])


@dataclass
class OutputFile:
    """
    Results generated by a tool, saved to a file, and saved in IDEAS.

    :param file_path: The path to the output file
    :type file_path: Path
    :param output_dir: Output directory where output files are generated.
    :type output_dir: Path
    :param subdir: A subdirectory in the output directory where output files should be moved.
    Helpful for organizing result files into subdirectories for output columns
    :type subdir: Optional[str | Path]
    :param prefix: Prefix to optionally prepend to the output filename, and associated preview filenames.
    :type prefix: Optional[str]
    :param prefix: Suffix to optionally append to the output filename, and associated preview filenames.
    :type prefix: Optional[str]
    :param previews: List of previews to show in IDEAS for the output file
    :type previews: List[Preview]
    :param metadata: List of metadata to show in IDEAS for the output file
    :type metadata: List[Metadata]
    :param tags: List of tags to show in IDEAS for the output file
    :type tags: List[Tag]
    :param raise_missing_file: Raise an error if an output file is missing
    :type raise_missing_file: bool
    """

    file_path: Path
    output_dir: Path
    subdir: Optional[str | Path] = None
    prefix: Optional[str] = None
    suffix: Optional[str] = None
    previews: List[Preview] = field(default_factory=list)
    metadata: List[Metadata] = field(default_factory=list)
    tags: List[Tag] = field(default_factory=list)
    raise_missing_file: bool = True

    def to_dict(self):
        """Serialize to dict to save in output data file"""
        return {
            "file": str(self.file_path),
            "previews": [p.to_dict() for p in self.previews],
            "metadata": [m.to_dict() for m in self.metadata],
            "tags": self.tags,
        }

    @classmethod
    def from_dict(cls, instance: Dict, output_dir: str | Path):
        """Deserialize from dict in output data file"""
        return cls(
            file_path=Path(instance["file"]),
            previews=[Preview.from_dict(p) for p in instance["previews"]],
            metadata=[Metadata.from_dict(m) for m in instance["metadata"]],
            tags=[Tag(t) for t in instance["tags"]],
            output_dir=output_dir,
        )

    def register_preview(
        self,
        file_path: str | Path,
        caption: str = "",
        output_dir: Optional[str | Path] = None,
        subdir: Optional[str | Path] = None,
        name: Optional[str] = None,
        prefix: Optional[str] = None,
        suffix: Optional[str] = None,
    ):
        """
        Register a preview with the output file

        :param file_path: The path to the preview file
        :type file_path: str | Path
        :param caption: A short description of the preview.
        :type caption: str
        :param output_dir: Output directory where output files are generated.
        If not specified, default to output directory for output data.
        :type output_dir: Optional[str | Path]
        :param subdir: A subdirectory in the output directory where output files should be moved.
        Helpful for organizing result files into subdirectories for output columns
        :type subdir: Optional[str | Path]
        :param prefix: Prefix to optionally prepend to the output filename, and associated preview filenames.
        :type prefix: Optional[str]
        :param prefix: Suffix to optionally append to the output filename, and associated preview filenames.
        :type prefix: Optional[str]

        :return: The output file (for command chaining)
        :rtype: OutputFile
        :raises OutputDataFileOutsideOutputDir: If the preview file is outside the output directory.
        :raises OutputDataFileDoesNotExist: If the preview file does not exist.
        """
        if output_dir is None:
            output_dir = self.output_dir

        if not isinstance(output_dir, Path):
            output_dir = Path(output_dir)

        if subdir is None:
            subdir = self.subdir
        if prefix is None:
            prefix = self.prefix
        if suffix is None:
            suffix = self.suffix

        if not isinstance(file_path, Path):
            file_path = Path(file_path)

        # check if this preview file has already been registered to output data
        preview_file = None
        try:
            resolved_file_path = _resolve_output_file_path(
                file_path,
                output_dir,
                subdir=subdir,
                name=name,
                prefix=prefix,
                suffix=suffix,
            )
        except exceptions.OutputDataFileDoesNotExist as error:
            if self.raise_missing_file:
                raise error
            else:
                logging.warning(error)
                return self

        for _preview_file in self.previews:
            if resolved_file_path == _preview_file.file_path:
                preview_file = _preview_file

        if not preview_file:
            self.previews.append(
                Preview(
                    file_path=resolved_file_path,
                    caption=caption,
                )
            )
        return self

    def register_metadata(self, key: str, value: str, name: Optional[str] = None):
        """
        Register metadata with the output file

        :param key: Unique identifier for the metadata across output files
        :type key: str
        :param name: Friendly name for the metadata to display in IDEAS
        :type name: str
        :param value: The value of the metadata
        :type value: Union[str, bool, int, float, List[str]]

        :return: The output file (for command chaining)
        :rtype: OutputFile
        """
        if not name:
            # give a pretty name for metadata key
            name = key.replace("-", " ").replace("_", " ").title()

        self.metadata.append(Metadata(key=key, name=name, value=value))
        return self

    def register_metadata_dict(
        self, **kwargs: Dict[str, Union[str, bool, int, float, List[str]]]
    ):
        """
        Register metadata dictionary with the output file

        :param kwargs: Dictionary with key-value pairs of metadata
        :type kwargs: str
        :param value: The value of the metadata
        :type value: Dict[str, Union[str, bool, int, float, List[str]]]

        :return: The output file (for command chaining)
        :rtype: OutputFile
        """
        for key, value in kwargs.items():
            self.register_metadata(key=key, value=value)
        return self

    def register_tags(self, *tags: List[str]):
        """
        Register tags with the output file

        :param tags: Tags to register
        :type tags: List[str]

        :return: The output file (for command chaining)
        :rtype: OutputFile
        """
        for tag in tags:
            self.tags.append(tag)
        return self


class OutputData:
    """
    Collection of output files generated by an IDEAS tool.
    Each output file has associated previews, metadata, and tags to display in IDEAS.

    :param output_files: List of output files registered by a tool.
    :type output_files: List[OutputFile]
    :param output_dir: Output directory where output files are generated.
    :type output_dir: Path
    :param append: Flag indicating whether to append new output files
    to existing output data already registered by the tool.
    :param raise_missing_file: Raise an error if an output file is missing
    :type raise_missing_file: bool
    """

    output_files: List[OutputFile]
    output_dir: Path
    append: bool
    raise_missing_file: bool

    def __init__(
        self,
        output_dir: Optional[str | Path] = None,
        append=True,
        raise_missing_file=True,
    ):
        """Initialize output data

        :param output_dir: Output directory where output files are generated.
        If None, then the current working directory is used.
        :type output_dir: Optional[str | Path]
        :param append: Flag indicating whether to append new output files
        to existing output data already registered by the tool.
        :param raise_missing_file: Raise an error if an output file is missing
        :type raise_missing_file: bool
        """
        if not output_dir:
            output_dir = Path.cwd()
        if not isinstance(output_dir, Path):
            output_dir = Path(output_dir)
        self.output_dir = output_dir

        self.file_pathname = output_dir / constants.OUTPUT_DATA_JSON_FILENAME
        self.output_files = []
        self.append = append
        self.raise_missing_file = raise_missing_file

    def open(self):
        """Open output data for writing."""
        if self.append and self.file_pathname.exists():
            with open(self.file_pathname, "r") as file:
                output_data = json.load(file)
                # todo: validate schema here
                for output_file in output_data["output_files"]:
                    self.output_files.append(
                        OutputFile.from_dict(output_file, self.output_dir)
                    )

    def close(self):
        """Close output data for writing."""
        output_data = {
            "schema_version": constants.OUTPUT_DATA_SCHEMA_VERSION,
            "output_files": [f.to_dict() for f in self.output_files],
        }
        with open(self.file_pathname, "w") as file:
            json.dump(output_data, file, indent=4)

    def __enter__(self):
        """Enter for context manager."""
        self.open()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        """Exit for context manager."""
        self.close()

    def register_file(
        self,
        file_path: str | Path,
        previews: Optional[List[Preview]] = None,
        metadata: Optional[List[Metadata]] = None,
        tags: Optional[List[Tag]] = None,
        output_dir: Optional[str | Path] = None,
        subdir: Optional[str | Path] = None,
        name: Optional[str] = None,
        prefix: Optional[str] = None,
        suffix: Optional[str] = None,
    ):
        """
        Register an output file as output data.

        :param file_path: The path to the output file
        :type file_path: str | Path
        :param output_dir: Output directory where output files are generated.
        :type output_dir: Path
        :param previews: List of previews to show in IDEAS for the output file
        :type previews: Optional[List[Preview]]
        :param metadata: List of metadata to show in IDEAS for the output file
        :type metadata: Optional[List[Metadata]]
        :param tags: List of tags to show in IDEAS for the output file
        :type tags: Optional[List[Tag]]
        :param output_dir: Output directory where output files are generated.
        If not specified, default to current working directory.
        :type output_dir: Optional[str | Path]
        :param subdir: A subdirectory in the output directory where output files should be moved.
        Helpful for organizing result files into subdirectories for output columns
        :type subdir: Optional[str | Path]
        :param prefix: Prefix to optionally prepend to the output filename, and associated preview filenames.
        :type prefix: Optional[str]
        :param prefix: Suffix to optionally append to the output filename, and associated preview filenames.
        :type prefix: Optional[str]

        :return: The output file (for command chaining)
        :rtype: OutputFile
        :raises OutputDataFileOutsideOutputDir: If an output file is outside the output directory.
        :raises OutputDataFileDoesNotExist: If an output file does not exist.
        """
        if output_dir is None:
            output_dir = self.output_dir
        if not isinstance(output_dir, Path):
            output_dir = Path(output_dir)

        if not isinstance(file_path, Path):
            file_path = Path(file_path)

        # check if this output file has already been registered to output data
        output_file = None
        try:
            resolved_file_path = _resolve_output_file_path(
                file_path,
                output_dir,
                subdir=subdir,
                name=name,
                prefix=prefix,
                suffix=suffix,
            )
        except exceptions.OutputDataFileDoesNotExist as error:
            if self.raise_missing_file:
                raise error
            else:
                logging.exception("Failed to find output file")
                return output_file

        for _output_file in self.output_files:
            if resolved_file_path == _output_file.file_path:
                output_file = _output_file

        if not output_file:
            output_file = OutputFile(
                file_path=resolved_file_path,
                previews=previews if previews else [],
                metadata=metadata if metadata else [],
                tags=tags if tags else [],
                output_dir=self.output_dir,
                subdir=subdir,
                prefix=prefix,
                suffix=suffix,
                raise_missing_file=self.raise_missing_file,
            )
            self.output_files.append(output_file)
        return output_file


@contextmanager
def register(
    output_dir: Optional[str | Path] = None,
    append: bool = True,
    raise_missing_file: bool = True,
):
    """
    Opens a context manager to register output data for an IDEAS tool.

    Example usage:
    ```
    from ideas.tools import outputs

    with outputs.register() as output_data:
        output_data.register_file(
            "file.txt",
        ).register_preview(
            "figure1.svg",
            caption="A very special figure!"
        ).register_metadata_dict(
            key1="cool"
            key2="beans"
        )
    ```

    :param output_dir: Output directory where output files are generated.
    If None, then the current working directory is used.
    :type output_dir: Optional[str | Path]
    :param append: Flag indicating whether to append new output files
    to existing output data already registered by the tool.
    :param raise_missing_file: Raise an error if an output file is missing, otherwise a warning is logged.
    :type raise_missing_file: bool

    :raises OutputDataFileOutsideOutputDir: If an output file is outside the output directory.
    :raises OutputDataFileDoesNotExist: If an output file does not exist.
    """
    with OutputData(
        output_dir, append=append, raise_missing_file=raise_missing_file
    ) as output_data:
        try:
            yield output_data
        finally:
            pass
