"""
A place to define decorators that can prevent repeated definitions of common options/arguments to
commands, for example, for authentication or choosing a tenant.
"""

import functools
import logging
import pathlib

import click
import pycognito.exceptions
from pycognito.utils import Cognito, RequestsSrpAuth

from ideas import config
from ideas.environments import Environment
from ideas.session import LazySession
from ideas.utils import custom_click_types

logger = logging.getLogger(__name__)


def get_cognito_auth(username, password, environment: Environment) -> RequestsSrpAuth:
    access_token, refresh_token = config.read_cached_token(
        username, environment["USER_POOL_ID"]
    )

    # Populate Cognito client just as RequestsSrpAuth does it, but check it ourselves for expiry
    # before passing it to RequestsSrpAuth, so that we can force a refresh if the refresh token is
    # still valid
    cognito = Cognito(
        user_pool_id=environment["USER_POOL_ID"],
        client_id=environment["CLIENT_ID"],
        user_pool_region=environment["REGION"],
        username=username,
        access_token=access_token,
        refresh_token=refresh_token,
    )

    if access_token and refresh_token:
        """
        Verify and possibly refresh the cached tokens. If we get an error while checking the token,
        we re-authenticate with the user credentials to get new tokens. If that fails, we wipe the
        cached tokens and pass the error up, as it's likely due to invalid credentials.
        """
        try:
            # Will refresh token if it has expired; if we get an exception here it is likely because
            # the refresh token has expired, so we need to re-authenticate.
            cognito.check_token()
        except pycognito.exceptions.TokenVerificationException:
            # Force a re-auth, the cached tokens are no longer valid; if credentials are valid then
            # cognito will store the new valid tokens and we will write them on exit
            cognito.authenticate(password=password)
        except Exception:
            # We don't know what has failed, but the cached tokens are certainly invalid at this
            # point, so wipe them and log a detailed traceback
            logger.exception("Failed to verify or refresh cached token")
            config.clear_cached_token()
            raise

    return RequestsSrpAuth(
        username=username,
        password=password,
        user_pool_id=environment["USER_POOL_ID"],
        client_id=environment["CLIENT_ID"],
        user_pool_region=environment["REGION"],
        cognito=cognito,
    )


def base_cli_arguments(f):
    """
    A decorator that defines several common arguments for CLI usage. Builds a Session object from
    those arguments, but only instantiates it when it's used (so that we don't prompt for
    credentials if the command doesn't require them).

    Builds the context to include authentication and the url to use for any
    API requests, but you must request the context from click, for example:

        @cli.command()
        @click.pass_context
        @common_cli_arguments
        def show_url(ctx):
            print(ctx.obj["url"])

    """

    @click.option(
        "-e",
        "--env",
        required=False,
        help="Which environment to use. Use `environments` to list supported environments",
    )
    @click.option(
        "-u",
        "--username",
        envvar="IDEAS_CLI_USERNAME",
        required=False,
        help="IDEAS username",
        hidden=True,
    )
    @click.option(
        "-p",
        "--password",
        envvar="IDEAS_CLI_PASSWORD",
        required=False,
        help="IDEAS password",
        hidden=True,
    )
    @click.option(
        "--django-url",
        default=None,
        envvar="IDEAS_CLI_DJANGO_URL",
        # Don't expose this to the users in the --help, they don't need it
        hidden=True,
    )
    @click.option(
        "--registry-url",
        default=None,
        envvar="IDEAS_CLI_REGISTRY_URL",
        # Don't expose this to the users in the --help, they don't need it
        hidden=True,
    )
    @click.option(
        "--environments-url",
        default=None,
        envvar="IDEAS_CLI_ENVIRONMENTS_URL",
        # Don't expose this to the users in the --help, they don't need it
        hidden=True,
    )
    @click.option(
        "-c",
        "--config",
        "config_file",
        type=click.Path(dir_okay=False),
        default=config.get_default_config_path,
        callback=config.configure_cli,
        is_eager=False,
        help="Read options from the specified configuration file",
        show_default=True,
    )
    @click.option(
        "-p",
        "--profile",
        envvar="IDEAS_CLI_PROFILE",
        default=config.DEFAULT_PROFILE,
        help="Which profile to use",
        type=custom_click_types.ProfileOption(),
    )
    @click.pass_context
    @functools.wraps(f)
    def wrapped_with_auth_options(
        ctx,
        *args,
        env: str,
        username: str,
        password: str,
        django_url: str,
        registry_url: str,
        environments_url: str,
        profile: str,
        config_file: pathlib.Path,
        **kwargs,
    ):
        ctx.ensure_object(dict)
        session = LazySession(
            ctx,
            env,
            username,
            password,
            django_url,
            registry_url,
            environments_url,
            # # This one is special as some operations don't require the tenant-id, so
            # # pass it through the kwargs to the function as well.
            tenant_id=kwargs.get("tenant_id"),
            profile=profile,
            config_file_path=config_file,
        )
        ctx.obj["session"] = session
        ctx.obj["profile"] = profile
        ctx.obj["config_file"] = config_file

        return f(*args, **kwargs)

    return wrapped_with_auth_options


def require_tenant_id(f):
    """
    A decorator that includes the --tenant-id flag as a required argument.
    """

    @click.option(
        "-t",
        "--tenant-id",
        required=False,
        type=custom_click_types.TenantIdOption(),
        help="The tenant ID to use, use `ideas tenants` to view a list",
        envvar="IDEAS_CLI_TENANT_ID",
    )
    @click.pass_context
    @functools.wraps(f)
    def wrapped_with_tenant_id(ctx, *args, tenant_id: str, **kwargs):
        return f(*args, tenant_id=tenant_id, **kwargs)

    return wrapped_with_tenant_id
