import typing
from datetime import datetime, timezone
from pathlib import Path

import click
import humanize
from click.shell_completion import CompletionItem

from ideas.config import get_config_profiles
from ideas.utils.docker_utils import get_docker_client

KeyValueOptionValue: typing.TypeAlias = tuple[str, str]


class KeyValueOption(click.ParamType):
    """
    Process arguments expected to be in the form `foo=bar`, raising an error
    if they do not conform to this specification.
    """

    name = "key-value option"

    def convert(
        self,
        value: str,
        param: typing.Optional[click.core.Option],
        ctx: click.core.Context,
    ) -> KeyValueOptionValue:
        try:
            key, value = value.split("=", maxsplit=1)
            return (key, value)
        except ValueError:
            if param is not None:
                param_msg = f" for {param.name}"
            else:
                param_msg = ""

            self.fail(
                f"{value} is not a valid value{param_msg}. Values must be separated as key=value",
                param,
                ctx,
            )


class SearchOption(click.ParamType):
    """
    Returns a single `KeyValueOptionValue` suitable to mix into the filters to
    search objects in IDEAS by partial match.
    """

    name = "search option"

    def convert(
        self,
        value: str,
        param: typing.Optional[click.core.Option],
        ctx: click.core.Context,
    ) -> KeyValueOptionValue:
        if not value:
            return tuple()
        return (("search", str(value)),)


class TenantIdOption(click.ParamType):
    """
    Acts like an `int` but returns a str that can be used in a header.
    """

    name = "tenant-id"

    def convert(self, value, param, ctx):
        try:
            int(value)
            return str(value)
        except (TypeError, ValueError):
            self.fail(f"{value} is not a valid tenant-id", param, ctx)


class ListOption(click.ParamType):
    """
    Process arguments expected to be in the form `foo,bar`, raising an error
    if they do not conform to this specification.
    """

    name = "list option"

    def convert(
        self,
        value: str,
        param: typing.Optional[click.core.Option],
        ctx: click.core.Context,
    ) -> KeyValueOptionValue:
        try:
            str(value)
            return str(value)
        except ValueError:
            self.fail(f"{value} is not a valid list option", param, ctx)


class LocalContainerImageOption(click.ParamType):
    """
    Lists all local supported docker images for shell completion.

    For example:

        platform/neuro-stats:0.0.3  (2 days ago; 3.6 GB)
        platform/neuro-stats:latest  (2 days ago; 3.6 GB)
    """

    name = "containerimage"

    def shell_complete(self, ctx, param, incomplete):
        client = get_docker_client()
        images = client.api.images()
        now = datetime.now(timezone.utc)

        completions = []
        for img in images:
            tags = img.get("RepoTags") or ["<none>:<none>"]
            for tag in tags:
                if (
                    "/" in tag
                    and tag.split("/")[0].count(".") > 0
                    or ":" in tag.split("/")[0]
                ):
                    # skip remote repositories with host
                    continue
                repository, _ = tag.split(":", maxsplit=1)
                size = img.get("Size", 0)
                created = datetime.fromtimestamp(img["Created"], tz=timezone.utc)

                completions.append((created, size, repository, tag))

        for created, size, repository, tag in sorted(completions, key=lambda x: x[0]):
            size = humanize.naturalsize(size)
            updated = humanize.naturaltime(now - created)
            if tag.startswith(incomplete):
                yield CompletionItem(tag, help=f"{updated}; {size}")


class ProfileOption(click.ParamType):
    """
    Lists all profile names from the current config file.
    """

    name = "profile"

    def shell_complete(self, ctx, param, incomplete):
        """
        Lists all configured profiles for the current set config file.
        """
        for profile in get_config_profiles(ctx.params.get("config_file")):
            if profile.startswith(incomplete):
                yield CompletionItem(profile)


class ToolKeyOption(click.ParamType):
    """
    Lists all tool keys available in the .ideas folder within the cwd.
    """

    name = "toolkey"

    def shell_complete(self, ctx, param, incomplete):
        """
        Lists all tool keys available in the .ideas folder within the cwd.
        """
        ideas_dir = Path.cwd() / ".ideas"
        if ideas_dir.exists():
            for item in [d for d in ideas_dir.iterdir() if d.is_dir()]:
                if item.name.startswith(incomplete):
                    yield CompletionItem(item.name)
