"""
A set of helpers to interact with the IDEAS HTTP API.
"""

import collections
import functools
import json
import os
import typing

import botocore
import requests
import requests.adapters
from requests.packages.urllib3.util.retry import Retry

from ideas import __version__, exceptions
from ideas.utils.api_types import PaginatedResponse
from ideas.utils.custom_click_types import KeyValueOptionValue

IDEAS_API_VERSION = os.getenv("IDEAS_CLI_API_VERSION", "v1")
APICallParamSpec = typing.ParamSpec("APICallParamSpec")


def get_retry_strategy() -> Retry:
    return Retry(
        # Any HTTP response codes outside of this list will not be retried
        status_forcelist=[
            429,  # too many requests
            500,  # internal server error
            502,  # bad gateway
            503,  # service unavailable
            504,  # gateway timeout
        ],
        # We only upload through a PUT request, and being explicit prevents mistakes later
        allowed_methods=["PUT"],
        # Backoff with exponential decay starting at one second
        backoff_factor=1,
        # Up to a total of four retries
        total=4,
    )


MULTIPART_UPLOAD_RETRY_STRATEGY = get_retry_strategy()


def get_http_session(strategy=MULTIPART_UPLOAD_RETRY_STRATEGY) -> requests.Session:
    """
    Creates and returns a requests.Session object that has been configured with
    a retry strategy that retries failed (but retryable) PUT requests with an
    exponential backoff.

    Also nice because it uses connection pooling to save overhead on repeated
    requests to the same host.

    When used, this session will retry failed requests up to four times, with
    the following delays between attempts:

        attempt: delay
        --------------
        1: 1s
        2: 2s
        3: 4s
        4: 8s
    """
    adapter = requests.adapters.HTTPAdapter(max_retries=strategy)
    session = requests.Session()
    session.mount("https://", adapter)
    session.mount("http://", adapter)

    # Set the user-agent
    session.headers.update({"User-Agent": f"ideas-cli/{__version__}"})
    return session


http_session = get_http_session()


def process_filters(filters: tuple[KeyValueOptionValue]) -> dict:
    """
    Process the filters produced by `custom_click_types.KeyValueOption`,
    preparing a dictionary to be passed to requests as params.

    In particular, turns multiple values for the same key into a list of values.
    """
    params = collections.defaultdict(list)
    for key, value in filters:
        params[key].append(value)

    return params


def handle_pagination(method, url, headers, auth, filters) -> typing.Iterator[dict]:
    """
    Handles pagination transparently to the user by requesting all the pages
    that the API returns until they are exhausted.

    Expects pagination responses to include the following structure:

        {
            "count": 300,
            "next": "https://api/?page=2",
            "previous": None,
            "results": [
                ...
            ]
        }
    """
    response = typing.cast(
        PaginatedResponse[dict],
        method(
            url,
            headers,
            auth,
            params=process_filters(filters),
        ),
    )

    # First consume the initial page
    for thing in response["results"]:
        yield thing

    while response["next"] is not None:
        # As long as the API keep returning next pages, return them as well
        # Note that you don't need to pass params here as the API already
        # includes them in the next URL.
        response = typing.cast(
            PaginatedResponse[dict], method(response["next"], headers, auth)
        )
        for thing in response["results"]:
            yield thing


def handle_api_response(
    f: typing.Callable[APICallParamSpec, requests.Response],
) -> typing.Callable[APICallParamSpec, typing.Union[PaginatedResponse, str]]:
    """
    A decorator that can be used to handle API errors and automatically raise
    them through click so that the user is notified in a readable way.

    The decorator expects to be given a callable that returns a requests.Response
    object, and then it handles errors and processing the response. It also tries to
    return the JSON of the response but falls back to text if that fails to parse.

    For example:

        @handle_api_response
        def get(url, headers):
            return requests.get(url, headers=headers)

    :raises exceptions.ApiException: on any API request that fails with an HTTP
        status code, with the message being the JSON response from the API.
    :raises exceptions.UnhandledError: on any unexpected errors from the API or
        Cognito, where the JSON cannot be processed.
    :raises exceptions.UnauthorizedError: on an `NotAuthorizedException` from
        botocore, or the API returns a 401/403 HTTP status code.
    """

    @functools.wraps(f)
    def wrapper(
        *args: APICallParamSpec.args, **kwargs: APICallParamSpec.kwargs
    ) -> typing.Union[PaginatedResponse, str]:
        # TODO error handling for things like connection timeouts, DNS failures
        # https://inscopix.atlassian.net/browse/ID-1930
        try:
            response = f(*args, **kwargs)
        except botocore.exceptions.ClientError as e:
            # We can't get access to the client to catch specific exceptions for client errors, so
            # we have to inspect the error code as follows:
            if e.response["Error"]["Code"] == "NotAuthorizedException":
                raise exceptions.UnauthorizedError("Authorization failed")

            raise exceptions.UnhandledError("Unhandled botocore error") from e

        try:
            # This will raise for any non-success HTTP status codes returned
            # by the API
            response.raise_for_status()
        except requests.exceptions.HTTPError as error:
            if error.response.status_code in (401, 403):
                raise exceptions.UnauthorizedError("Authorization failed")

            if error.response.status_code == 409:
                try:
                    details = "\n".join(
                        [e["detail"] for e in error.response.json()["errors"]]
                    )
                except Exception:
                    details = json.dumps(error.response.json(), indent=2)
                raise exceptions.QuotaExceededError(
                    f"Quota exceeded: {details}",
                    json=error.response.json(),
                    status_code=409,
                )

            # Optimistically try to return the JSON response from the API, but
            # if that fails, fall back to the response text
            try:
                e = exceptions.ApiException(
                    "API returned an error",
                    status_code=error.response.status_code,
                    json=error.response.json(),
                )
                e.add_note(str(e.json))
                raise e from error
            except requests.exceptions.JSONDecodeError:
                e = exceptions.UnhandledError(error.response.text)
                e.add_note(str(error.response.text))
                raise e from error

        try:
            # Not all API endpoints return a JSON response (e.g., a file
            # download) but we optimistically try to parse the response as JSON
            # and fall back to text otherwise
            return response.json()
        except requests.exceptions.JSONDecodeError:
            return response.text

    return wrapper


@handle_api_response
def get(url_get, headers, auth, params=None):
    return http_session.get(url_get, params=params, headers=headers, auth=auth)


@handle_api_response
def patch(url_post, headers, auth, data):
    return http_session.patch(url_post, headers=headers, auth=auth, json=data)


@handle_api_response
def post(url_post, headers, auth, data):
    return http_session.post(url_post, headers=headers, auth=auth, json=data)


@handle_api_response
def put(url: str, data: bytes) -> requests.Response:
    return http_session.put(url, data)
