import math
import os

MIN_THREADS = 4
MAX_THREADS = 16

MAX_NUM_PARTS = 10_000
PART_SIZE_MB = int(os.environ.get("IDEAS_CLI_PART_SIZE", 50))
DEFAULT_PART_SIZE = PART_SIZE_MB * 1024 * 1024


def thread_count(num_cores=os.cpu_count()):
    """
    Return a sliding scale of threads to CPU cores, strictly capped between MIN_THREADS and
    MAX_THREADS, to prevent too much memory usage, or too low of upload throughput.
    """
    return max(MIN_THREADS, min(MAX_THREADS, num_cores * 2))


def get_part_size(file_size: int, parts: int | None = None) -> int:
    """
    Try and return the specified default part size, but stay below the AWS limit of 10k parts per
    multipart upload.

    If parts is specified, calculate the part size based on what AWS has already predetermined to be
    the part count.

    Source: https://docs.aws.amazon.com/AmazonS3/latest/userguide/qfacts.html
    """
    if parts is None:
        num_parts = MAX_NUM_PARTS
        part_size = DEFAULT_PART_SIZE
    else:
        # Part size has been predetermined by AWS: will be the largest factor of 1 MiB
        num_parts = parts
        part_size = 1_048_576
    factor = math.ceil(file_size / (part_size * num_parts))

    return factor * part_size
