import json
import os
from importlib import resources
from typing import Any

import jsonschema.exceptions
from jsonschema import Draft202012Validator
from referencing import Registry, Resource
from referencing.exceptions import NoSuchResource
from referencing.jsonschema import DRAFT202012
from ideas_schemas.exceptions import SchemaValidationError


def _retrieve_from_filesystem(uri: str):
    """Helper function to retrieve reference schemas"""
    try:
        contents = get(f"v3/{uri}")
    except FileNotFoundError:
        raise NoSuchResource(f"Failed to retrieve reference schema: {uri}")
    return Resource(contents=contents, specification=DRAFT202012)


# Create singelton registry for schema with special function to retrieve ref schemas
_registry = Registry(retrieve=_retrieve_from_filesystem)


def _get_validator(schema):
    return Draft202012Validator(
        schema,
        registry=_registry,
    )


def _validate(instance, schema):
    validator = _get_validator(schema=schema)
    try:
        validator.validate(instance)
    except jsonschema.exceptions.ValidationError as e:
        raise SchemaValidationError(message=e.message, schema=instance) from e


def get(name: str, module: str = "tools"):
    """Get a schema from the module folder.

    Reads the json file and returns a dictionary with the contents.

    :param name: The name of the schema in the schema folder (excluding .json)
    :param module: schema folder. defaults to 'tools'
    """

    module_dir = os.path.join(os.path.dirname(__file__), "..")
    schemas_dir = os.path.join(module_dir, module)

    schemas_dir = resources.files(f"ideas_schemas.{module}")

    if not name.endswith(".json"):
        name = f"{name}.json"
    schema_file = schemas_dir.joinpath(name)
    if not schema_file.exists():
        raise FileNotFoundError(
            f"Failed to find {name} schema in schema dir ({schemas_dir})"
        )

    with schema_file.open("r") as f:
        schema = json.loads(f.read())

    return schema


def validate_v3_schema(instance: Any, schema_name: str):
    """Validate instance of V3 spec"""

    schema = get(name=f"v3/{schema_name}")

    _validate(instance=instance, schema=schema)


def validate_v3_tool_spec(instance: Any):
    """Validate instance of V3 tool spec"""

    validate_v3_schema(instance, schema_name="tool_spec_schema")


def validate_v3_bundle_metadata(instance: Any):
    """Validate instance of V3 tool spec"""

    validate_v3_schema(instance, schema_name="bundle_metadata_schema")


def validate_v3_output_data(instance: Any):
    """Validate instance of V3 output_data"""

    validate_v3_schema(instance, schema_name="output_data_schema")


def validate_v3_params(instance):
    """Validate instance of V3 params from the tool spec"""
    schema = get("v3/tool_spec_schema")
    param_schema = schema["properties"]["params"]
    _validate(instance=instance, schema=param_schema)


def validate_v3_results(instance):
    """Validate instance of V3 results from the tool spec"""
    schema = get("v3/tool_spec_schema")
    param_schema = schema["properties"]["results"]
    _validate(instance=instance, schema=param_schema)


def validate_v3_container_images_spec(instance):
    """Validate instance of V3 results from the tool spec"""
    schema = get("v3/container_image_spec_schema", module="container_images")
    _validate(instance=instance, schema=schema)
