# type: ignore
import os
import huey
import redis
import decouple
from karrio.server.settings import base as settings


# Karrio Server Background jobs interval config
DEFAULT_SCHEDULER_RUN_INTERVAL = 3600  # value is seconds. so 3600 seconds = 1 Hour
DEFAULT_TRACKERS_UPDATE_INTERVAL = decouple.config(
    "TRACKING_PULSE", default=7200, cast=int
)  # value is seconds. so 10800 seconds = 3 Hours

WORKER_IMMEDIATE_MODE = decouple.config(
    "WORKER_IMMEDIATE_MODE", default=False, cast=bool
)

REDIS_HOST = decouple.config("REDIS_HOST", default=None)
REDIS_PORT = decouple.config("REDIS_PORT", default=None)
REDIS_PASSWORD = decouple.config("REDIS_PASSWORD", default=None)
REDIS_USERNAME = decouple.config("REDIS_USERNAME", default="default")


# Use redis if available
if REDIS_HOST is not None:
    pool = redis.ConnectionPool(
        host=REDIS_HOST,
        port=REDIS_PORT or "6379",
        max_connections=20,
        **({"password": REDIS_PASSWORD} if REDIS_PASSWORD else {}),
        **({"username": REDIS_USERNAME} if REDIS_USERNAME else {}),
    )
    HUEY = huey.RedisHuey(
        "default",
        connection_pool=pool,
        **({"immediate": WORKER_IMMEDIATE_MODE} if WORKER_IMMEDIATE_MODE else {}),
    )

else:
    WORKER_DB_DIR = decouple.config("WORKER_DB_DIR", default=settings.WORK_DIR)
    WORKER_DB_FILE_NAME = os.path.join(WORKER_DB_DIR, "tasks.sqlite3")

    settings.DATABASES["workers"] = {
        "NAME": WORKER_DB_FILE_NAME,
        "ENGINE": "django.db.backends.sqlite3",
    }

    HUEY = huey.SqliteHuey(
        name="default",
        filename=WORKER_DB_FILE_NAME,
        **({"immediate": WORKER_IMMEDIATE_MODE} if WORKER_IMMEDIATE_MODE else {}),
    )


# Apply OpenTelemetry instrumentation to Huey if enabled
OTEL_ENABLED = decouple.config("OTEL_ENABLED", default=False, cast=bool)
OTEL_EXPORTER_OTLP_ENDPOINT = decouple.config("OTEL_EXPORTER_OTLP_ENDPOINT", default=None)

if OTEL_ENABLED and OTEL_EXPORTER_OTLP_ENDPOINT:
    try:
        # Import and apply instrumentation to the Huey instance
        from karrio.server.lib.otel_huey import HueyInstrumentor
        instrumentor = HueyInstrumentor()
        instrumentor.instrument(HUEY)
    except Exception as e:
        import logging
        logger = logging.getLogger(__name__)
        logger.warning(f"Failed to instrument Huey in worker settings: {e}")
