"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createBackupConfig = exports.createProcessingConfig = exports.createEncryptionConfig = exports.createBufferingHints = exports.createLoggingOptions = void 0;
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const constructs_1 = require("constructs");
function createLoggingOptions(scope, props) {
    if (props.logging === false && props.logGroup) {
        throw new Error('logging cannot be set to false when logGroup is provided');
    }
    if (props.logging !== false || props.logGroup) {
        const logGroup = props.logGroup ?? constructs_1.Node.of(scope).tryFindChild('LogGroup') ?? new logs.LogGroup(scope, 'LogGroup');
        const logGroupGrant = logGroup.grantWrite(props.role);
        return {
            loggingOptions: {
                enabled: true,
                logGroupName: logGroup.logGroupName,
                logStreamName: logGroup.addStream(props.streamId).logStreamName,
            },
            dependables: [logGroupGrant],
        };
    }
    return undefined;
}
exports.createLoggingOptions = createLoggingOptions;
function createBufferingHints(interval, size) {
    if (!interval && !size) {
        return undefined;
    }
    const intervalInSeconds = interval?.toSeconds() ?? 300;
    if (intervalInSeconds < 60 || intervalInSeconds > 900) {
        throw new Error(`Buffering interval must be between 60 and 900 seconds. Buffering interval provided was ${intervalInSeconds} seconds.`);
    }
    const sizeInMBs = size?.toMebibytes() ?? 5;
    if (sizeInMBs < 1 || sizeInMBs > 128) {
        throw new Error(`Buffering size must be between 1 and 128 MiBs. Buffering size provided was ${sizeInMBs} MiBs.`);
    }
    return { intervalInSeconds, sizeInMBs };
}
exports.createBufferingHints = createBufferingHints;
function createEncryptionConfig(role, encryptionKey) {
    encryptionKey?.grantEncryptDecrypt(role);
    return encryptionKey
        ? { kmsEncryptionConfig: { awskmsKeyArn: encryptionKey.keyArn } }
        : undefined;
}
exports.createEncryptionConfig = createEncryptionConfig;
function createProcessingConfig(scope, role, dataProcessor) {
    return dataProcessor
        ? {
            enabled: true,
            processors: [renderDataProcessor(dataProcessor, scope, role)],
        }
        : undefined;
}
exports.createProcessingConfig = createProcessingConfig;
function renderDataProcessor(processor, scope, role) {
    const processorConfig = processor.bind(scope, { role });
    const parameters = [{ parameterName: 'RoleArn', parameterValue: role.roleArn }];
    parameters.push(processorConfig.processorIdentifier);
    if (processor.props.bufferInterval) {
        parameters.push({ parameterName: 'BufferIntervalInSeconds', parameterValue: processor.props.bufferInterval.toSeconds().toString() });
    }
    if (processor.props.bufferSize) {
        parameters.push({ parameterName: 'BufferSizeInMBs', parameterValue: processor.props.bufferSize.toMebibytes().toString() });
    }
    if (processor.props.retries) {
        parameters.push({ parameterName: 'NumberOfRetries', parameterValue: processor.props.retries.toString() });
    }
    return {
        type: processorConfig.processorType,
        parameters,
    };
}
function createBackupConfig(scope, role, props) {
    if (!props || (props.mode === undefined && !props.bucket)) {
        return undefined;
    }
    const bucket = props.bucket ?? new s3.Bucket(scope, 'BackupBucket');
    const bucketGrant = bucket.grantReadWrite(role);
    const { loggingOptions, dependables: loggingDependables } = createLoggingOptions(scope, {
        logging: props.logging,
        logGroup: props.logGroup,
        role,
        streamId: 'S3Backup',
    }) ?? {};
    return {
        backupConfig: {
            bucketArn: bucket.bucketArn,
            roleArn: role.roleArn,
            prefix: props.dataOutputPrefix,
            errorOutputPrefix: props.errorOutputPrefix,
            bufferingHints: createBufferingHints(props.bufferingInterval, props.bufferingSize),
            compressionFormat: props.compression?.value,
            encryptionConfiguration: createEncryptionConfig(role, props.encryptionKey),
            cloudWatchLoggingOptions: loggingOptions,
        },
        dependables: [bucketGrant, ...(loggingDependables ?? [])],
    };
}
exports.createBackupConfig = createBackupConfig;
//# sourceMappingURL=data:application/json;base64,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