# AUTOGENERATED! DO NOT EDIT! File to edit: ../../2-8-calc-ground-state.ipynb.

# %% auto 0
__all__ = ['calc_ground_state']

# %% ../../2-8-calc-ground-state.ipynb 1
import torch
import numpy as np
from einops import einsum
from typing import List, Tuple
from ..utils.checking import check_quantum_gate
from scipy.sparse.linalg import LinearOperator, eigsh
from copy import deepcopy

# %% ../../2-8-calc-ground-state.ipynb 2
def calc_ground_state(
    hamiltonian: torch.Tensor | List[torch.Tensor],
    interact_positions: List[List[int]] | torch.Tensor,
    num_qubits: int,
    smallest_k: int = 1,
) -> Tuple[torch.Tensor, torch.Tensor]:
    """
    Calculate the ground state of a quantum system using the linear operator method.

    Args:
        hamiltonian: The Hamiltonian of the quantum system.
        interact_positions: The positions of the interactions.
        num_qubits: The number of qubits in the system.
        smallest_k: The number of smallest eigenvalues to calculate.

    Returns:
        ground_state: The ground state of the quantum system.
        ground_energy: The energy of the ground state.
    """
    assert smallest_k >= 1
    assert num_qubits >= 2
    if isinstance(hamiltonian, torch.Tensor):
        assert all(x == 2 for x in hamiltonian.shape), (
            "hamiltonian must be a tensor with all dimensions of size 2"
        )
        gate_apply_qubit_num = check_quantum_gate(hamiltonian)
        assert gate_apply_qubit_num <= num_qubits
        hamiltonian = [hamiltonian]
    else:
        assert len(hamiltonian) == len(interact_positions)
        gate_apply_qubit_num = check_quantum_gate(hamiltonian[0])
        assert all(x == 2 for x in hamiltonian[0].shape), (
            "hamiltonian must be a tensor with all dimensions of size 2"
        )
        for i in range(1, len(hamiltonian)):
            assert gate_apply_qubit_num == check_quantum_gate(hamiltonian[i]), (
                "all hamiltonian must have the same number of qubits for now"
            )
            assert all(x == 2 for x in hamiltonian[i].shape), (
                "all hamiltonian must be a tensor with all dimensions of size 2"
            )

    assert isinstance(interact_positions, (List, torch.Tensor))
    if isinstance(interact_positions, List):
        interact_positions = torch.tensor(interact_positions)
    assert interact_positions.max() < num_qubits and interact_positions.min() >= 0
    assert interact_positions.ndim == 2
    interaction_num, apply_position_num = interact_positions.shape
    assert apply_position_num == gate_apply_qubit_num
    for i in range(interaction_num):
        assert len(interact_positions[i]) == len(interact_positions[i].unique())

    gate_bra_dim_names = [f"b{i}" for i in range(gate_apply_qubit_num)]  # bra/left dimensions
    gate_ket_dim_names = [f"k{i}" for i in range(gate_apply_qubit_num)]  # ket/right dimensions
    qubit_dim_names = [f"q{i}" for i in range(num_qubits)]
    gate_braket_dims_einexp = " ".join(gate_bra_dim_names + gate_ket_dim_names)

    ein_exps = []
    for positions in interact_positions:
        qubit_input_dims = deepcopy(qubit_dim_names)
        qubit_output_dims = deepcopy(qubit_dim_names)
        for gate_braket_idx, qubit_idx in enumerate(positions.tolist()):
            # qubit dim get contracted with gate left/bra dims, yielding ket dims
            qubit_input_dims[qubit_idx] = gate_bra_dim_names[gate_braket_idx]
            qubit_output_dims[qubit_idx] = gate_ket_dim_names[gate_braket_idx]
        ein_exp = "{qubit_input_dims}, {gate_braket_dims} -> {qubit_output_dims}".format(
            qubit_input_dims=" ".join(qubit_input_dims),
            gate_braket_dims=gate_braket_dims_einexp,
            qubit_output_dims=" ".join(qubit_output_dims),
        )
        ein_exps.append(ein_exp)

    hamiltonian_np = torch.stack(hamiltonian).numpy()

    def matvec(state: np.ndarray):
        state = state.reshape(*([2] * num_qubits))
        new_state = 0
        if hamiltonian_np.shape[0] == 1:
            for ein_exp in ein_exps:
                new_state += einsum(state, hamiltonian_np[0], ein_exp)
        else:
            for ein_exp, h in zip(ein_exps, hamiltonian_np):
                new_state += einsum(state, h, ein_exp)
        return new_state.flatten()

    linear_fn = LinearOperator(shape=(2**num_qubits, 2**num_qubits), matvec=matvec)
    smallest_eigvalue, eigenvec = eigsh(linear_fn, k=smallest_k, which="SA")
    ground_energy = torch.from_numpy(smallest_eigvalue)
    ground_state = torch.from_numpy(eigenvec).squeeze()
    return ground_state, ground_energy
