"""核心是波恩概率诠释，见 [2.1](2-1.ipynb)"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../4-5.ipynb.

# %% auto 0
__all__ = ['EPS', 'calc_left_to_right_step', 'calc_right_to_left_step', 'calc_nll', 'calc_gradient', 'eval_nll', 'train_gmps', 'labels_to_binary', 'prepend_labels', 'generate_sample_with_gmps', 'gmps_classify', 'eval_nll_selected_features', 'gmps_classify_with_selected_features']

# %% ../../4-5.ipynb 2
import torch
from ..mps.modules import MPS, MPSType
from einops import einsum
from tqdm.auto import tqdm
from typing import Tuple, List, Dict, Any
from torch.utils.data import DataLoader, TensorDataset

# %% ../../4-5.ipynb 4
EPS = 1e-14


def calc_left_to_right_step(
    current_tensor: torch.Tensor,
    current_env_vector_left: torch.Tensor,
    current_sample: torch.Tensor,
) -> Tuple[torch.Tensor, torch.Tensor]:
    """
    Calculate one step from left to right of the sweep algorithm
    """
    next_env_vector_left = einsum(
        current_env_vector_left,
        current_sample,
        current_tensor,
        "batch left, batch physical, left physical right -> batch right",
    )
    current_norm_factor = next_env_vector_left.norm(dim=1, keepdim=True)
    return next_env_vector_left / (current_norm_factor + EPS), current_norm_factor.squeeze(-1)


def calc_right_to_left_step(
    current_tensor: torch.Tensor,
    current_env_vector_right: torch.Tensor,
    current_sample: torch.Tensor,
) -> Tuple[torch.Tensor, torch.Tensor]:
    """
    Calculate one step from right to left of the sweep algorithm
    """
    next_env_vector_right = einsum(
        current_env_vector_right,
        current_sample,
        current_tensor,
        "batch right, batch physical, left physical right -> batch left",
    )
    current_norm_factor = next_env_vector_right.norm(dim=1, keepdim=True)
    return next_env_vector_right / (current_norm_factor + EPS), current_norm_factor.squeeze(-1)


def calc_nll(norm_factors: torch.Tensor) -> torch.Tensor:
    """
    Calculate the negative log likelihood from the norm factors in a batch
    """
    nll = -2 * torch.log(norm_factors.abs() + EPS).sum(dim=1)  # (batch)
    return nll


@torch.compile(dynamic=True)
def calc_gradient(
    env_left_vector: torch.Tensor,
    env_right_vector: torch.Tensor,
    current_sample: torch.Tensor,
    current_tensor: torch.Tensor,
    enable_tsgo: bool,
) -> torch.Tensor:
    """
    Calculate the gradient w.r.t. the current tensor
    """
    raw_grad = einsum(
        env_left_vector,
        current_sample,
        env_right_vector,
        "batch left, batch physical, batch right -> batch left physical right",
    )
    norm = einsum(
        current_tensor,
        raw_grad,
        "left physical right, batch left physical right -> batch",
    )
    norm += torch.sign(norm) * EPS  # add a small number to avoid division by zero
    grad_part = (raw_grad / norm.view(-1, 1, 1, 1)).mean(dim=0)
    grad = 2 * (current_tensor - grad_part)
    grad_shape = grad.shape
    assert grad_shape == current_tensor.shape
    if enable_tsgo:
        grad = grad.reshape(-1)
        current_tensor = current_tensor.reshape(-1)
        projection = torch.dot(grad, current_tensor) * current_tensor
        grad = grad - projection
        grad = grad.reshape(grad_shape)

    grad /= grad.norm()
    return grad


def eval_nll(
    *,
    samples: torch.Tensor,
    mps: MPS,
    device: torch.device,
    return_avg: bool = True,
) -> torch.Tensor:
    """
    Evaluate the negative log likelihood of the MPS, given the feature-mapped samples.

    Args:
        samples: torch.Tensor, the feature-mapped samples.
        mps: MPS, the MPS to evaluate the negative log likelihood of.
        device: torch.device, the device to evaluate the negative log likelihood on.
        return_avg: bool, whether to return the average negative log likelihood.
    Returns:
        torch.Tensor, the negative log likelihood of the MPS.
    """
    assert samples.ndim == 3  # (dataset_size, feature_num, feature_dim)
    assert mps.center is not None
    dataset_size, feature_num, _ = samples.shape
    assert feature_num == mps.length
    batch_size = dataset_size  # since we do the init NLL evaluation in one go
    # set default device to device
    prev_device = torch.get_default_device()
    torch.set_default_device(device)
    mps_local_tensors = mps.local_tensors
    # init env vectors and norm factors
    left_virtual_dim = mps_local_tensors[0].shape[0]
    env_vector_left = torch.ones(batch_size, left_virtual_dim)
    right_virtual_dim = mps_local_tensors[-1].shape[-1]
    env_vector_right = torch.ones(batch_size, right_virtual_dim)
    norm_factors = [None] * feature_num

    def samples_at(idx):
        return samples[:, idx, :]  # (batch, feature_dim)

    for idx in range(mps.center):
        next_env_vector_left, current_norm_factor = calc_left_to_right_step(
            mps_local_tensors[idx],
            env_vector_left,
            samples_at(idx),
        )
        # set the norm factor for current position to be the norm of the next env vector, since the next env vector is to be normalized
        norm_factors[idx] = current_norm_factor
        env_vector_left = next_env_vector_left

    # prepare env vectors from right to left
    for idx in range(mps.length - 1, mps.center, -1):
        next_env_vector_right, current_norm_factor = calc_right_to_left_step(
            mps_local_tensors[idx],
            env_vector_right,
            samples_at(idx),
        )
        norm_factors[idx] = current_norm_factor
        env_vector_right = next_env_vector_right

    # update the norm factor at the center
    norm_factors[mps.center] = einsum(
        mps_local_tensors[mps.center],
        env_vector_left,
        samples_at(mps.center),
        env_vector_right,
        "left physical right, batch left, batch physical, batch right -> batch",
    )

    norm_factors = torch.stack(norm_factors, dim=1)  # (batch, feature_num)

    if return_avg:
        nll = calc_nll(norm_factors).mean()
    else:
        nll = calc_nll(norm_factors)  # (batch)
    # restore the default device
    torch.set_default_device(prev_device)
    return nll


def train_gmps(
    *,
    samples: torch.Tensor,
    batch_size: int,
    mps: MPS,
    sweep_times: int,
    lr: float,
    device: torch.device,
    enable_tsgo: bool,
    progress_bar_kwargs: Dict[str, Any] = {},
) -> Tuple[torch.Tensor, MPS]:
    """
    Train a MPS model with the GMPS algorithm.

    Args:
        samples: torch.Tensor, the feature-mapped samples.
        batch_size: int, the batch size.
        mps: MPS, the MPS to train.
        sweep_times: int, the number of sweeps/training epochs.
        lr: float, the learning rate.
        device: torch.device, the device to train on.
        enable_tsgo: bool, whether to enable the TSGO algorithm.
        progress_bar_kwargs: Dict[str, Any], the keyword arguments for the progress bar.
    Returns:
        Tuple[torch.Tensor, MPS], the training losses and the trained MPS.
    """
    dataset_size = samples.shape[0]
    assert dataset_size % batch_size == 0
    assert mps.mps_type == MPSType.Open
    # prepare mps, normalize first to avoid numerical instability
    mps.center_orthogonalization_(0, mode="qr", normalize=True, check_nan=True)
    init_nll = eval_nll(samples=samples, mps=mps, device=device)

    # set default device to device
    prev_device = torch.get_default_device()
    torch.set_default_device(device)
    # prepare dataloader
    dataloader = DataLoader(TensorDataset(samples), batch_size=batch_size, shuffle=True)

    mps_local_tensors = mps._mps  # CAREFUL for inplace operation
    feature_num = mps.length
    left_virtual_dim = mps_local_tensors[0].shape[0]
    right_virtual_dim = mps_local_tensors[-1].shape[-1]

    nll_losses = [init_nll]

    progress_bar = tqdm(range(sweep_times), **progress_bar_kwargs)
    disable_batch_progress_bar = (dataset_size // batch_size) == 1
    for i in progress_bar:
        epoch_nll_losses = []
        for batch_data in tqdm(dataloader, leave=False, disable=disable_batch_progress_bar):
            batch_data = batch_data[0]
            batch_size = batch_data.shape[0]
            # prepare aux variables
            env_vectors_left: List[torch.Tensor | None] = [None] * mps.length
            env_vectors_left[0] = torch.ones(batch_size, left_virtual_dim)
            env_vectors_right: List[torch.Tensor | None] = [None] * mps.length
            env_vectors_right[-1] = torch.ones(batch_size, right_virtual_dim)
            norm_factors = torch.ones(batch_size, feature_num)

            def data_at(idx):
                return batch_data[:, idx, :]  # (batch, feature_dim)

            # prepare env vectors from right to left
            # leave out left-to-right because the center of mps always starts at 0
            for idx in range(mps.length - 1, mps.center, -1):
                next_env_vector_right, current_norm_factor = calc_right_to_left_step(
                    mps_local_tensors[idx],
                    env_vectors_right[idx],
                    data_at(idx),
                )
                norm_factors[:, idx] = current_norm_factor
                env_vectors_right[idx - 1] = next_env_vector_right

            # update the norm factor at the center
            norm_factors[:, mps.center] = einsum(
                mps_local_tensors[mps.center],
                env_vectors_left[mps.center],
                data_at(mps.center),
                env_vectors_right[mps.center],
                "left physical right, batch left, batch physical, batch right -> batch",
            )

            # gradient calculation and optimization, from left to right
            for idx in range(mps.length):
                assert idx == mps.center
                grad = calc_gradient(
                    env_vectors_left[idx],
                    env_vectors_right[idx],
                    data_at(idx),
                    mps_local_tensors[idx],
                    enable_tsgo,
                )
                # update the tensor with gradient, inplace operation, will change mps._mps
                mps_local_tensors[idx] -= lr * grad

                # prepare for the next iteration
                if idx < mps.length - 1:
                    # move the center to the right
                    # the local tensors (mps._mps) at idx and idx + 1 will be changed
                    mps.center_orthogonalization_(idx + 1, mode="qr", normalize=True)
                    # so we need to update aux variables, only env_vectors_left affected
                    new_next_env_vector_left, new_norm_factor = calc_left_to_right_step(
                        mps_local_tensors[idx],
                        env_vectors_left[idx],
                        data_at(idx),
                    )
                    env_vectors_left[idx + 1] = new_next_env_vector_left
                    norm_factors[:, idx] = new_norm_factor
                else:
                    # we need normalization here to make mps as a unit norm state so to preserve the probability interpretation
                    mps.center_normalize_()

            for idx in range(mps.length - 1, -1, -1):
                assert idx == mps.center
                grad = calc_gradient(
                    env_vectors_left[idx],
                    env_vectors_right[idx],
                    data_at(idx),
                    mps_local_tensors[idx],
                    enable_tsgo,
                )
                mps_local_tensors[idx] -= lr * grad
                # prepare for the next iteration
                if idx > 0:
                    # move the center to the left
                    # the local tensors (mps._mps) at idx and idx - 1 will be changed
                    mps.center_orthogonalization_(idx - 1, mode="qr", normalize=True)
                    # so we need to update aux variables, only env_vectors_right affected
                    new_next_env_vector_right, new_norm_factor = calc_right_to_left_step(
                        mps_local_tensors[idx],
                        env_vectors_right[idx],
                        data_at(idx),
                    )
                    env_vectors_right[idx - 1] = new_next_env_vector_right
                    norm_factors[:, idx] = new_norm_factor
                else:
                    # we need normalization here to make mps as a unit norm state so to preserve the probability interpretation
                    mps.center_normalize_()

            assert mps.center == 0
            # update the norm factor at the center
            norm_factors[:, mps.center] = einsum(
                mps_local_tensors[mps.center],
                env_vectors_left[mps.center],
                data_at(mps.center),
                env_vectors_right[mps.center],
                "left physical right, batch left, batch physical, batch right -> batch",
            )
            batch_nll_loss = calc_nll(norm_factors)
            epoch_nll_losses.append(batch_nll_loss)

        epoch_nll_loss = torch.cat(epoch_nll_losses).mean()
        nll_losses.append(epoch_nll_loss)
        progress_bar.set_description(f"Iter {i} NLL: {epoch_nll_loss:.4f}")

    # restore the default device
    torch.set_default_device(prev_device)
    return torch.stack(nll_losses), mps

# %% ../../4-5.ipynb 25
def labels_to_binary(labels: torch.Tensor, num_bits: int) -> torch.Tensor:
    """
    Converts a tensor of labels to a binary representation.

    Args:
        labels: A 1D tensor of integer labels.
        num_bits: The number of bits for the binary representation.

    Returns:
        A 2D float tensor where each row is the binary representation
        of the corresponding label.
    """
    assert labels.dim() == 1, "labels must be a 1D tensor"
    assert labels.dtype == torch.long, "labels must be a long tensor"
    assert labels.min() >= 0, "labels must be non-negative"
    assert labels.max() < 2**num_bits, "labels must be less than 2 ** num_bits"

    # Create a tensor of bit positions to extract (e.g., [3, 2, 1, 0] for 4 bits)
    powers = torch.arange(num_bits - 1, -1, -1, device=labels.device)

    # Expand labels to a column vector and broadcast the bitwise right shift
    # across the bit positions. Then, use bitwise AND with 1 to get the bit value.
    binary_labels = (labels.unsqueeze(1) >> powers) & 1

    return binary_labels.to(torch.float32)


def prepend_labels(raw_images: torch.Tensor, labels: torch.Tensor) -> torch.Tensor:
    """
    Prepend the label "pixels" to the raw images.
    """
    assert raw_images.ndim == 2  # (batch, 28 * 28)
    assert raw_images.shape[0] == labels.shape[0]
    assert raw_images.shape[1] == 28 * 28
    bin_labels = labels_to_binary(labels, num_bits=4)  # (batch, 4)
    return torch.cat([bin_labels, raw_images], dim=1)

# %% ../../4-6.ipynb 5
import numpy as np
from ..feature_mapping import cossin_feature_map
from copy import deepcopy
from typing import Literal, Dict, Any


def generate_sample_with_gmps(
    mps: MPS,
    *,
    sample: torch.Tensor | None,
    sample_num: int,
    gen_indices: List[int] | None,
    feature_mapping: Literal["cossin"],
    feature_mapping_kwargs: Dict[str, Any] = {},
    gen_order: Literal["ascending", "descending"] = "ascending",
):
    """
    Generate a sample with GMPS.

    Args:
        mps: MPS to generate sample from.
        sample: Sample to partially sample from. If None, sample and generate from all qubits.
        sample_num: Number of samples to average over.
        gen_indices: Indices of the qubits to generate.
        feature_mapping: Feature mapping to use.
        feature_mapping_kwargs: Keyword arguments for the feature mapping.
    """
    assert sample_num > 0, "sample_num must be positive"
    assert gen_order in ["ascending", "descending"], "gen_order must be 'ascending' or 'descending'"
    assert feature_mapping in ["cossin"], "feature_mapping must be 'cossin'"
    length = mps.length
    generate_all = False
    if sample is None or gen_indices is None:
        if gen_order == "ascending":
            gen_indices = list(range(length))
        else:
            gen_indices = list(range(length - 1, -1, -1))
        sample = None
        generate_all = True

    if generate_all:
        sample = torch.zeros(length, device=mps.device, dtype=mps.dtype)
    else:
        assert sample.shape in [(length,), (1, length)], (
            f"sample cannot be batched, got one with shape {sample.shape}"
        )
        sample = sample.squeeze()  # (length,)
        project_indices = list(range(length))
        for gi in gen_indices:
            project_indices.remove(gi)
        sample_for_projection = sample[project_indices]
        if feature_mapping == "cossin":
            features_for_projection = cossin_feature_map(
                sample_for_projection, **feature_mapping_kwargs
            )
        else:
            raise ValueError(f"Feature mapping {feature_mapping} not supported")
        features_for_projection = features_for_projection.squeeze(
            0
        )  # get rid of the batch dimension
        mps = mps.project_multi_qubits(project_indices, features_for_projection)

    # Remap gen_indices in case of a new mps after projection
    # the two argsorts effectively give the ranking of each in gen_indices, which is a new item in gen_indices_new
    # in the case that sample is None, gen_indices_new is the same as gen_indices
    #
    # the algorithm is the same as below:
    # ```
    # indices = np.argsort(np.argsort(gen_indices))
    # ranks = [None for _ in len(gen_indices)]
    # for i in len(gen_indices):
    #     original_index = indices[i]
    #     rank = i
    #     ranks[original_index] = rank
    # gen_indices_new = ranks
    # ```
    #
    # This remapping is needed since project_indices will be projected and removed from the new MPS
    gen_indices = np.array(gen_indices)
    gen_indices_new = np.argsort(
        np.argsort(gen_indices)
    )  # TODO: refactor the function - this line is not useful when generate_all is True

    # defensive coding
    if generate_all:
        assert np.all(gen_indices_new == gen_indices), (
            "gen_indices_new should be the same as gen_indices"
        )

    samples = []
    for _ in range(sample_num):
        sample_i = sample.clone()
        mps_i = deepcopy(mps)
        gen_indices_i = gen_indices_new.copy()
        pos = 0
        while len(gen_indices_i) > 0:
            gen_idx = gen_indices_i[0]
            mps_i.center_orthogonalization_(gen_idx, mode="qr", normalize=True)
            rdm = mps_i.one_body_reduced_density_matrix(idx=gen_idx, do_tracing=True)
            assert rdm.shape == (2, 2)
            lm = rdm.diag()
            prob_1 = lm[1]
            p1 = prob_1.cpu().item()
            assert 0.0 <= p1 <= 1.0, f"probability should be between 0 and 1, got {p1}"
            state = torch.bernoulli(prob_1).to(torch.long).cpu().item()
            gen_idx_on_sample = gen_indices[pos]
            sample_i[gen_idx_on_sample] = state
            new_mps_i = mps_i.project_one_qubit(gen_idx, state)
            # the center of mps_i is gen_idx and the qubit at gen_idx has been projected
            new_mps_i._center = max(0, mps_i.center - 1)
            mps_i = new_mps_i
            # update gen_indices_i since mps_i has been projected
            gen_indices_i[gen_indices_i > gen_idx] -= 1
            gen_indices_i = gen_indices_i[1:]
            pos += 1

        samples.append(sample_i)

    samples = torch.stack(samples)
    generated_sample = samples.mean(dim=0)
    return generated_sample

# %% ../../4-7.ipynb 9
from tqdm.auto import tqdm


def gmps_classify(
    gmpss: List[MPS], data: torch.Tensor, progress_bar_kwargs: dict = {}
) -> torch.Tensor:
    """
    Use a group of MPS to classify the data.

    Args:
        gmpss: List[MPS], the group of MPS to classify the data.
        data: torch.Tensor, the feature-mapped data to classify.
        progress_bar_kwargs: dict, the keyword arguments for the progress bar.
    Returns:
        torch.Tensor, the predictions of the data.
    """
    num_gmps = len(gmpss)
    assert num_gmps > 0, "No GMPSs provided"
    assert data.ndim == 3, "Data must be a 3D tensor of shape (batch, feature_num, feature_dim)"
    feature_num = data.shape[1]
    assert feature_num == gmpss[0].length, "Feature number mismatch"

    nll_of_gmps = []
    for gmps in tqdm(gmpss, **progress_bar_kwargs):
        nll = eval_nll(samples=data, mps=gmps, device=gmps.device, return_avg=False)  # (batch)
        nll_of_gmps.append(nll)

    nll_of_gmps = torch.stack(nll_of_gmps, dim=1)  # (batch, num_gmps)
    predictions = torch.argmin(nll_of_gmps, dim=1)  # (batch)
    return predictions

# %% ../../4-9.ipynb 28
from typing import Literal


@torch.compile(dynamic=True)
def _left_to_right_step(
    local_tensor_i: torch.Tensor, env_vectors_left: torch.Tensor, sample_i: torch.Tensor
) -> torch.Tensor:
    new_local_tensor = einsum(
        local_tensor_i,
        sample_i,
        "left physical right, batch physical -> batch left right",
    )
    return einsum(
        new_local_tensor.conj(),
        env_vectors_left,
        new_local_tensor,
        "batch left_conj right_conj, batch left_conj left, batch left right -> batch right_conj right",
    )


def _map_left_to_right(
    local_tensor_i: torch.Tensor, env_vectors_left: torch.Tensor, sample_i: torch.Tensor
) -> torch.Tensor:
    # local_tensor_i: (left_virtual_dim, physical_dim, right_virtual_dim)
    # env_vectors_left: (left_virtual_dim)
    # sample_i: (physical_dim)
    new_local_tensor = einsum(
        local_tensor_i,
        sample_i,
        "left physical right, physical -> left right",
    )
    return einsum(
        new_local_tensor.conj(),
        env_vectors_left,
        new_local_tensor,
        "left_conj right_conj, left_conj left, left right -> right_conj right",
    )


_left_to_right_step_vmapped = torch.compile(
    torch.vmap(_map_left_to_right, in_dims=(None, 0, 0)), dynamic=True
)


@torch.compile(dynamic=True)
def _right_to_left_step(
    local_tensor_i: torch.Tensor, env_vectors_right: torch.Tensor, sample_i: torch.Tensor
) -> torch.Tensor:
    new_local_tensor = einsum(
        local_tensor_i,
        sample_i,
        "left physical right, batch physical -> batch left right",
    )
    return einsum(
        new_local_tensor.conj(),
        env_vectors_right,
        new_local_tensor,
        "batch left_conj right_conj, batch right_conj right, batch left right -> batch left_conj left",
    )


def _map_right_to_left(
    local_tensor_i: torch.Tensor, env_vectors_right: torch.Tensor, sample_i: torch.Tensor
) -> torch.Tensor:
    # local_tensor_i: (left_virtual_dim, physical_dim, right_virtual_dim)
    # env_vectors_right: (right_virtual_dim)
    # sample_i: (physical_dim)
    new_local_tensor = einsum(
        local_tensor_i,
        sample_i,
        "left physical right, physical -> left right",
    )
    return einsum(
        new_local_tensor.conj(),
        env_vectors_right,
        new_local_tensor,
        "left_conj right_conj, right_conj right, left right -> left_conj left",
    )


_right_to_left_step_vmapped = torch.compile(
    torch.vmap(_map_right_to_left, in_dims=(None, 0, 0)), dynamic=True
)


def eval_nll_selected_features(
    *,
    samples: torch.Tensor,
    mps: MPS,
    indices: List[int] | torch.Tensor,
    device: torch.device,
    return_avg: bool = True,
    compute_method: Literal["compiled_einsum", "vmap"] = "vmap",
    progress_bar_kwargs: dict = {},
) -> torch.Tensor:
    """
    Evaluate the negative log likelihood of the MPS, given the feature-mapped samples.

    Args:
        samples: torch.Tensor, the feature-mapped samples.
        mps: MPS, the MPS to evaluate the negative log likelihood of.
        indices: the positions of features to be evaluated at.
        device: torch.device, the device to evaluate the negative log likelihood on.
        return_avg: bool, whether to return the average negative log likelihood.
        compute_method: Literal["compiled_einsum", "vmap"], underlying implementation of the heavylifting steps. "vmap" is usually faster but with more memory consumption.
    Returns:
        torch.Tensor, the negative log likelihood of the MPS.
    """
    assert samples.ndim == 3  # (dataset_size, feature_num, feature_dim)
    assert mps.center is not None
    dataset_size, feature_num, _ = samples.shape
    assert feature_num == mps.length
    assert isinstance(indices, (List, torch.Tensor))
    if isinstance(indices, torch.Tensor):
        assert indices.ndim == 1
        assert indices.dtype == torch.long
        assert indices.min() >= 0
        assert indices.max() < feature_num
        indices = indices.tolist()
    list_length = len(indices)
    indices = set(indices)
    assert len(indices) == list_length, "indices must be unique"
    # set default device to device
    prev_device = torch.get_default_device()
    torch.set_default_device(device)
    mps_local_tensors = mps.local_tensors
    batch_size = dataset_size  # since we do the init NLL evaluation in one go
    env_vectors_left = torch.ones(batch_size, 1, 1, dtype=samples.dtype)
    env_vectors_right = torch.ones(batch_size, 1, 1, dtype=samples.dtype)
    norm_factors = torch.ones(batch_size, feature_num)

    def samples_at(idx):
        return samples[:, idx, :]  # (batch, feature_dim)

    left_to_right_step = (
        _left_to_right_step if compute_method == "compiled_einsum" else _left_to_right_step_vmapped
    )
    right_to_left_step = (
        _right_to_left_step if compute_method == "compiled_einsum" else _right_to_left_step_vmapped
    )

    for idx in tqdm(range(mps.center), **progress_bar_kwargs):
        local_tensor_i = mps_local_tensors[
            idx
        ]  # (left_virtual_dim, physical_dim, right_virtual_dim)
        if idx in indices:
            env_vectors_left = left_to_right_step(local_tensor_i, env_vectors_left, samples_at(idx))
        else:
            env_vectors_left = einsum(
                local_tensor_i.conj(),
                env_vectors_left,
                local_tensor_i,
                "left_conj physical right_conj, batch left_conj left, left physical right -> batch right_conj right",
            )

        norm = env_vectors_left.norm(dim=[1, 2])  # (batch)
        norm_factors[:, idx] = norm
        env_vectors_left = env_vectors_left / (norm.reshape(batch_size, 1, 1) + EPS)

    for idx in tqdm(range(feature_num - 1, mps.center, -1), **progress_bar_kwargs):
        local_tensor_i = mps_local_tensors[
            idx
        ]  # (left_virtual_dim, physical_dim, right_virtual_dim)
        if idx in indices:
            env_vectors_right = right_to_left_step(
                local_tensor_i, env_vectors_right, samples_at(idx)
            )
        else:
            env_vectors_right = einsum(
                local_tensor_i.conj(),
                env_vectors_right,
                local_tensor_i,
                "left_conj physical right_conj, batch right_conj right, left physical right -> batch left_conj left",
            )

        norm = env_vectors_right.norm(dim=[1, 2])  # (batch)
        norm_factors[:, idx] = norm
        env_vectors_right = env_vectors_right / (norm.reshape(batch_size, 1, 1) + EPS)

    center_tensor = mps_local_tensors[
        mps.center
    ]  # (left_virtual_dim, physical_dim, right_virtual_dim)
    if mps.center in indices:
        new_center_tensor = einsum(
            center_tensor,
            samples_at(mps.center),
            "left physical right, batch physical -> batch left right",
        )
        norm = einsum(
            env_vectors_left,
            new_center_tensor.conj(),
            new_center_tensor,
            env_vectors_right,
            "batch left_conj left, batch left_conj right_conj, batch left right, batch right_conj right -> batch",
        ).abs()
    else:
        norm = einsum(
            center_tensor.conj(),
            center_tensor,
            env_vectors_left,
            env_vectors_right,
            "left_conj physical right_conj, left physical right, batch left_conj left, batch right_conj right -> batch",
        ).abs()
    norm_factors[:, mps.center] = norm
    if return_avg:
        nll = calc_nll(norm_factors).mean()
    else:
        nll = calc_nll(norm_factors)  # (batch)
    # restore the default device
    torch.set_default_device(prev_device)
    return nll

# %% ../../4-9.ipynb 34
import torch.multiprocessing as mp
from typing import Dict, Any
from ..utils.checking import is_notebook


def _process(
    rank: int,
    gmps_local_tensors: List[torch.Tensor],
    gmps_center: int,
    data: torch.Tensor,
    indices: List[int],
    prediction_out: torch.Tensor,
):
    mps = MPS(mps_tensors=gmps_local_tensors)
    mps._center = gmps_center
    nll = eval_nll_selected_features(
        samples=data,
        mps=mps,
        indices=indices,
        device=mps.device,
        return_avg=False,
        progress_bar_kwargs={"disable": True},
    )  # (batch)
    prediction_out[:, rank] = nll


def _gmps_classify_with_selected_features_multiprocessing(
    gmpss: List[MPS],
    data: torch.Tensor,
    indices: List[int] | torch.Tensor,
    progress_bar_kwargs: Dict[str, Any] = {},
) -> torch.Tensor:
    """
    Use a group of MPS to classify the data.

    Args:
        gmpss: List[MPS], the group of MPS to classify the data.
        data: torch.Tensor, the feature-mapped data to classify.
        indices: List[int] | torch.Tensor, the indices of the features to use for classification.
        progress_bar_kwargs: dict, the keyword arguments for the progress bar.
    Returns:
        torch.Tensor, the predictions of the data.
    """
    num_gmps = len(gmpss)

    if isinstance(indices, torch.Tensor):
        indices = indices.tolist()

    nll_out = torch.zeros((data.shape[0], num_gmps), device=data.device)
    nll_out.share_memory_()
    data.share_memory_()

    local_tensors = []
    for mps in gmpss:
        tensors = mps.local_tensors
        for t in tensors:
            t.share_memory_()
        local_tensors.append(tensors)

    centers = []
    for mps in gmpss:
        centers.append(mps.center)

    processes = [
        mp.Process(
            target=_process, args=(rank, local_tensors[rank], centers[rank], data, indices, nll_out)
        )
        for rank in range(num_gmps)
    ]

    for p in processes:
        p.start()
    for p in tqdm(processes, **progress_bar_kwargs):
        p.join()

    predictions = torch.argmin(nll_out, dim=1)  # (batch)
    return predictions


def gmps_classify_with_selected_features(
    gmpss: List[MPS],
    data: torch.Tensor,
    indices: List[int] | torch.Tensor,
    progress_bar_kwargs: dict = {},
    use_multiprocessing: bool = False,
) -> torch.Tensor:
    """
    Use a group of MPS to classify the data.

    Args:
        gmpss: List[MPS], the group of MPS to classify the data.
        data: torch.Tensor, the feature-mapped data to classify.
        indices: List[int] | torch.Tensor, the indices of the features to use for classification.
        progress_bar_kwargs: dict, the keyword arguments for the progress bar.
        use_multiprocessing: bool, whether to use multiprocessing. Multi-processing is times faster but with times more memory usage.
    Returns:
        torch.Tensor, the predictions of the data.
    """
    num_gmps = len(gmpss)
    assert num_gmps > 0, "No GMPSs provided"
    assert data.ndim == 3, "Data must be a 3D tensor of shape (batch, feature_num, feature_dim)"
    feature_num = data.shape[1]
    assert feature_num == gmpss[0].length, "Feature number mismatch"

    if isinstance(indices, torch.Tensor):
        indices = indices.tolist()

    indices_set = set(indices)
    assert len(indices_set) == len(indices), "Indices must be unique"

    if len(indices) == feature_num:
        return gmps_classify(gmpss, data, progress_bar_kwargs)

    assert not (use_multiprocessing and is_notebook()), "Cannot use multiprocessing in notebook"
    if use_multiprocessing:
        return _gmps_classify_with_selected_features_multiprocessing(
            gmpss, data, indices, progress_bar_kwargs
        )

    nll_of_gmps = []
    for gmps in tqdm(gmpss, **progress_bar_kwargs):
        nll = eval_nll_selected_features(
            samples=data,
            mps=gmps,
            indices=indices,
            device=gmps.device,
            return_avg=False,
            progress_bar_kwargs={"leave": False},
        )  # (batch)
        nll_of_gmps.append(nll)

    nll_of_gmps = torch.stack(nll_of_gmps, dim=1)  # (batch, num_gmps)
    predictions = torch.argmin(nll_of_gmps, dim=1)  # (batch)
    return predictions