# AUTOGENERATED! DO NOT EDIT! File to edit: ../../4-8-lazy-classifier.ipynb.

# %% auto 0
__all__ = ['lazy_classify']

# %% ../../4-8-lazy-classifier.ipynb 1
import torch
from typing import Literal

# %% ../../4-8.ipynb 18
from typing import Dict, Any
from tensor_network.algorithms.quantum_kernels import (
    metric_neg_log_cos_sin,
    metric_neg_chebyshev,
    metric_neg_cossin_chebyshev,
)


def lazy_classify(
    *,
    samples: torch.Tensor,
    reference_samples: torch.Tensor,
    reference_labels: torch.Tensor,
    kernel: Literal["euclidean", "nll_cossin", "r_nll_cossin", "chebyshev", "cossin_chebyshev"],
    kernel_kwargs: Dict[str, Any] = {},
) -> torch.Tensor:
    """
    Use reference samples to classify samples. A lazy classifier with no trainable parameters.

    Args:
        samples: samples to classify, shape (N samples, feature num)
        reference_samples: reference samples, shape (N reference samples, feature num)
        reference_labels: labels of reference samples, shape (N reference samples,)
        kernel: kernel to use
        kernel_kwargs: keyword arguments for the kernel

    Returns:
        predicted labels of shape (N samples,)
    """
    assert kernel in ["euclidean", "nll_cossin", "r_nll_cossin", "chebyshev", "cossin_chebyshev"]
    assert reference_labels.ndim == 1
    assert reference_samples.ndim == 2  # (N samples, feature num)
    assert reference_samples.min() >= 0.0, f"reference_samples.min(): {reference_samples.min()}"
    assert reference_samples.max() <= 1.0, f"reference_samples.max(): {reference_samples.max()}"
    ref_sample_num = reference_samples.shape[0]
    assert reference_labels.shape[0] == ref_sample_num

    assert samples.ndim == 2  # (N samples, feature num)
    assert samples.min() >= 0.0, f"samples.min(): {samples.min()}"
    assert samples.max() <= 1.0, f"samples.max(): {samples.max()}"

    classes = reference_labels.unique(sorted=True)

    prob = []
    for c in classes:
        ref_samples_c = reference_samples[reference_labels == c]
        if kernel == "euclidean":
            dist = torch.cdist(samples, ref_samples_c, p=2)  # (N samples, N reference samples)
            dist = dist.mean(dim=1)  # (N samples,)
        elif kernel == "nll_cossin":
            dist = metric_neg_log_cos_sin(
                samples, ref_samples_c, **kernel_kwargs
            )  # (N samples, N reference samples)
            dist = dist.mean(dim=1)
        elif kernel == "r_nll_cossin":
            dist = metric_neg_log_cos_sin(
                samples, ref_samples_c, **kernel_kwargs
            )  # (N samples, N reference samples)
            assert "beta" in kernel_kwargs, "beta is required for r_nll_cossin"
            dist = (kernel_kwargs["beta"] ** dist).mean(dim=1)  # (N samples,)
        elif kernel == "chebyshev":
            dist = metric_neg_chebyshev(samples, ref_samples_c)  # (N samples,)
        elif kernel == "cossin_chebyshev":
            dist = metric_neg_cossin_chebyshev(
                samples, ref_samples_c, **kernel_kwargs
            )  # (N samples,)
        else:
            raise Exception("Unreachable")
        prob.append(dist)

    prob = torch.stack(prob, dim=1)  # (N samples, M classes)
    pred = prob.argmin(dim=1)  # (N samples,)
    pred_classes = classes[pred]  # (N samples,)
    return pred_classes