"""References:"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../4-8.ipynb.

# %% auto 0
__all__ = ['metric_matrix_neg_log_cos_sin', 'metric_neg_log_cos_sin', 'metric_neg_chebyshev', 'metric_neg_cossin_chebyshev']

# %% ../../4-8.ipynb 2
import torch
from torch.utils.data import DataLoader, TensorDataset
from sklearn.manifold import TSNE
import matplotlib.pyplot as plt
from ..utils.data import load_mnist_images
from typing import Literal

# %% ../../4-8.ipynb 6
def _check_samples(samples: torch.Tensor):
    assert samples.ndim == 2  # (N samples, feature num)
    assert samples.min() >= 0.0, f"samples.min(): {samples.min()}"
    assert samples.max() <= 1.0, f"samples.max(): {samples.max()}"


def metric_matrix_neg_log_cos_sin(
    samples: torch.Tensor,
    theta: float = torch.pi / 2 - 1e-7,
    calculation_method: Literal["deduplicate", "no_deduplicate"] = "deduplicate",
) -> torch.Tensor:
    """
    Calculate the distance matrix between samples.

    Args:
        samples: samples of shape (N samples, feature num)
        theta: cossin mapping theta
        calculation_method: "deduplicate" or "no_deduplicate", "deduplicate" use symmetry to avoid duplicate calculation

    Returns:
        distance matrix of shape (N samples, N samples)
    """
    _check_samples(samples)
    assert torch.pi / 2 >= theta >= 0.0
    assert calculation_method in ["deduplicate", "no_deduplicate"]
    sample_num = samples.shape[0]
    if calculation_method == "deduplicate":
        metric = torch.zeros(sample_num, sample_num, device=samples.device, dtype=samples.dtype)
        for n in range(sample_num - 1):
            sample_n = samples[n].reshape(1, -1)  # (1, feature num)
            others = samples[n + 1 :]  # (sample_num - n - 1, feature num)
            diff = sample_n - others  # (sample_num - n - 1, feature num)
            distances = -torch.log(torch.cos(diff * theta)).mean(dim=1)  # (sample_num - n - 1)
            assert not torch.isnan(distances).any(), "if there's nan, try to reduce theta"
            metric[n, n + 1 :] = distances
            metric[n + 1 :, n] = distances
    else:
        samples = samples.unsqueeze(0)  # (1, sample_num, feature num)
        others = samples.transpose(0, 1)  # (sample_num, 1, feature num)
        diff = samples - others  # (sample_num, sample_num, feature num)
        metric = -torch.log(torch.cos(diff * theta)).mean(dim=2)  # (sample_num, sample_num)

        for n in range(sample_num):
            metric[n, n] = 0.0

        assert not torch.isnan(metric).any(), "if there's nan, try to reduce theta"

    return metric

# %% ../../4-8.ipynb 16
def metric_neg_log_cos_sin(
    samples: torch.Tensor,
    reference_samples: torch.Tensor,
    theta: float = torch.pi / 2 - 1e-7,
    batch_size: int | None = None,
) -> torch.Tensor:
    """
    Calculate the distance matrix between samples and reference samples with NLL and cossin mapping.

    Args:
        samples: samples of shape (N samples, feature num)
        reference_samples: reference samples of shape (N reference samples, feature num)
        theta: cossin mapping theta
        batch_size: batch size in computation. If None, batch_size = N // 2.

    Returns:
        distance matrix of shape (N samples, N reference samples)
    """
    _check_samples(samples)
    _check_samples(reference_samples)
    assert torch.pi / 2 >= theta >= 0.0
    sample_num = samples.shape[0]
    if batch_size is None:
        batch_size = sample_num // 2 if sample_num >= 2 else 1
    dataloader = DataLoader(TensorDataset(samples), batch_size=batch_size, shuffle=False)
    ref_sample_num = reference_samples.shape[0]
    ref_samples = reference_samples.unsqueeze(0)  # (1, ref_sample_num, feature num)

    metric = []

    # To reduce peak memory usage, we do the calculation in a loop
    for batch_i in dataloader:
        batch_samples = batch_i[0].unsqueeze(1)  # (batch_size, 1, feature num)
        diff = batch_samples - ref_samples  # (batch_size, ref_sample_num, feature num)
        result = -torch.log(torch.cos(diff * theta)).mean(dim=2)  # (batch_size, ref_sample_num)
        metric.append(result)

    metric = torch.cat(metric, dim=0)

    assert not torch.isnan(metric).any(), "if there's nan, try to reduce theta"
    assert metric.shape == (sample_num, ref_sample_num)
    return metric


def metric_neg_chebyshev(
    samples: torch.Tensor, reference_samples: torch.Tensor, batch_size: int | None = None
) -> torch.Tensor:
    """
    Calculate the distance matrix between samples and reference samples with negative Chebyshev distance.

    Args:
        samples: samples of shape (N samples, feature num)
        reference_samples: reference samples of shape (N reference samples, feature num)
        batch_size: batch size in computation. If None, batch_size = N // 2.

    Returns:
        distance matrix of shape (N samples, N reference samples)
    """
    _check_samples(samples)
    _check_samples(reference_samples)
    sample_num = samples.shape[0]
    if batch_size is None:
        batch_size = sample_num // 2 if sample_num >= 2 else 1
    dataloader = DataLoader(TensorDataset(samples), batch_size=batch_size, shuffle=False)
    ref_samples = reference_samples.unsqueeze(0)  # (1, ref_sample_num, feature num)

    metric = []
    # To reduce peak memory usage, we do the calculation in a loop
    for batch_i in dataloader:
        batch_samples = batch_i[0].unsqueeze(1)  # (batch_size, 1, feature num)
        diff = batch_samples - ref_samples  # (batch_size, ref_sample_num, feature num)
        result = diff.norm(dim=-1).min(dim=-1)[0]  # (batch_size)
        metric.append(result)

    metric = torch.cat(metric, dim=0)
    assert metric.shape == (sample_num,)
    return metric


def metric_neg_cossin_chebyshev(
    samples: torch.Tensor,
    reference_samples: torch.Tensor,
    theta: float = torch.pi / 2 - 1e-7,
    batch_size: int | None = None,
) -> torch.Tensor:
    """
    Calculate the distance matrix between samples and reference samples with negative Chebyshev distance and cossin mapping.

    Args:
        samples: samples of shape (N samples, feature num)
        reference_samples: reference samples of shape (N reference samples, feature num)
        theta: cossin mapping theta
        batch_size: batch size in computation. If None, batch_size = N // 2.

    Returns:
        distance matrix of shape (N samples, N reference samples)
    """
    _check_samples(samples)
    _check_samples(reference_samples)
    sample_num = samples.shape[0]
    if batch_size is None:
        batch_size = sample_num // 2 if sample_num >= 2 else 1
    dataloader = DataLoader(TensorDataset(samples), batch_size=batch_size, shuffle=False)
    ref_samples = reference_samples.unsqueeze(0)  # (1, ref_sample_num, feature num)

    metric = []
    # To reduce peak memory usage, we do the calculation in a loop
    for batch_i in dataloader:
        batch_samples = batch_i[0].unsqueeze(1)  # (batch_size, 1, feature num)
        diff = batch_samples - ref_samples  # (batch_size, ref_sample_num, feature num)
        result = -torch.log(torch.cos(diff * theta)).mean(dim=2)  # (batch_size, ref_sample_num)
        result = result.min(dim=-1)[0]  # (batch_size,)
        assert not torch.isnan(result).any(), "if there's nan, try to reduce theta"
        metric.append(result)

    metric = torch.cat(metric, dim=0)
    assert metric.shape == (sample_num,)
    return metric
