"""References:"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../../4-4-mlx.ipynb.

# %% auto 0
__all__ = ['ResMPSSimple']

# %% ../../../4-4-mlx.ipynb 2
import mlx.core as mx
from mlx import nn
from einops import einsum
from einops.array_api import repeat

# %% ../../../4-4-mlx.ipynb 6
class ResMPSSimple(nn.Module):
    def __init__(
        self,
        *,
        num_features: int,
        feature_dim: int,
        num_classes: int,
        virtual_dim: int,
        eps_norm: float = 1e-4,
    ):
        super().__init__()
        assert num_features > 0, "num_features must be positive"
        assert num_classes > 0, "num_classes must be positive"
        assert feature_dim > 0, "feature_dim must be positive"
        assert virtual_dim > 0, "virtual_dim must be positive"
        self.num_features = num_features
        self.num_classes = num_classes
        self.virtual_dim = virtual_dim
        self.feature_dim = feature_dim
        self.class_idx = num_features // 2
        local_tensors = []
        dtype = mx.float32
        if num_features == 1:
            local_tensors.append(
                self._make_local_tensor(
                    virtual_dim,
                    feature_dim,
                    virtual_dim,
                    num_classes,
                    dtype=dtype,
                    eps_norm=eps_norm,
                )
            )
        else:
            for i in range(num_features):
                if i == self.class_idx:
                    tensor_shape = (virtual_dim, feature_dim, virtual_dim, num_classes)
                else:
                    tensor_shape = (virtual_dim, feature_dim, virtual_dim)
                local_tensors.append(
                    self._make_local_tensor(*tensor_shape, dtype=dtype, eps_norm=eps_norm)
                )
        self.local_tensors = local_tensors
        contract_vector = mx.ones([virtual_dim], dtype=dtype)
        contract_vector /= mx.linalg.norm(contract_vector)
        self.contract_vector = contract_vector

    @staticmethod
    def _make_local_tensor(*shape: int, dtype: mx.Dtype, eps_norm: float) -> mx.array:
        tensor = mx.random.normal(shape, dtype=dtype)
        tensor /= mx.linalg.norm(tensor)
        tensor = tensor * eps_norm
        return tensor

    @mx.compile
    def calc(self, features: mx.array) -> mx.array:
        batch_size, num_features, feature_dim = features.shape
        latent_left = repeat(
            mx.stop_gradient(self.contract_vector),
            "virtual_left -> batch virtual_left",
            batch=batch_size,
        )
        for feature_idx in range(self.class_idx):
            latent = einsum(
                self.local_tensors[feature_idx],
                latent_left,
                features[:, feature_idx, :],
                "left feature right, batch left, batch feature -> batch right",
            )
            latent_left = latent + latent_left  # residual

        latent_right = repeat(
            mx.stop_gradient(self.contract_vector),
            "virtual_right -> batch virtual_right",
            batch=batch_size,
        )
        for feature_idx in range(num_features - 1, self.class_idx, -1):
            latent = einsum(
                self.local_tensors[feature_idx],
                latent_right,
                features[:, feature_idx, :],
                "left feature right, batch right, batch feature -> batch left",
            )
            latent_right = latent + latent_right  # residual

        activation = einsum(
            self.local_tensors[self.class_idx],
            latent_left,
            latent_right,
            features[:, self.class_idx, :],
            "left feature right classes, batch left, batch right, batch feature -> batch classes",
        )
        return activation

    @mx.compile
    @staticmethod
    def calc(
        local_tensors: list[mx.array],
        contract_vector: mx.array,
        features: mx.array,
        class_idx: int,
    ) -> mx.array:
        batch_size, num_features, _ = features.shape
        latent_left = repeat(
            mx.stop_gradient(contract_vector),
            "virtual_left -> batch virtual_left",
            batch=batch_size,
        )
        for feature_idx in range(class_idx):
            latent = einsum(
                local_tensors[feature_idx],
                latent_left,
                features[:, feature_idx, :],
                "left feature right, batch left, batch feature -> batch right",
            )
            latent_left = latent + latent_left  # residual

        latent_right = repeat(
            mx.stop_gradient(contract_vector),
            "virtual_right -> batch virtual_right",
            batch=batch_size,
        )
        for feature_idx in range(num_features - 1, class_idx, -1):
            latent = einsum(
                local_tensors[feature_idx],
                latent_right,
                features[:, feature_idx, :],
                "left feature right, batch right, batch feature -> batch left",
            )
            latent_right = latent + latent_right  # residual

        activation = einsum(
            local_tensors[class_idx],
            latent_left,
            latent_right,
            features[:, class_idx, :],
            "left feature right classes, batch left, batch right, batch feature -> batch classes",
        )
        return activation

    def __call__(self, features: mx.array) -> mx.array:
        # features shape: (batch_size, num_features, feature_dim)
        _, num_features, feature_dim = features.shape
        assert num_features == self.num_features, (
            f"num_features must be equal to {self.num_features}"
        )
        assert feature_dim == self.feature_dim, f"feature_dim must be equal to {self.feature_dim}"
        return self.calc(
            self.local_tensors,
            self.contract_vector,
            features,
            self.class_idx,
        )
