# AUTOGENERATED! DO NOT EDIT! File to edit: ../../3-7.ipynb.

# %% auto 0
__all__ = ['FCADQCHybridClassifier']

# %% ../../3-7.ipynb 2
import torch
from torch import nn
from .adqc import ADQCNet, probabilities_adqc_classifier
from typing import Literal
from ..feature_mapping import cossin_feature_map, feature_map_to_qubit_state
from torch.nn import functional as F

# %% ../../3-7.ipynb 3
class FCADQCHybridClassifier(nn.Module):
    """
    A hybrid classifier that combines a fully connected layer with an ADQC network.
    """

    def __init__(
        self,
        feature_num: int,
        dim_mid: int,
        class_num: int,
        num_qubits: int,
        num_adqc_layers: int,
        adqc_gate_pattern: Literal["brick", "stair"],
        identity_init: bool = False,
        double_precision: bool = False,
        feature_map: Literal["cossin"] = "cossin",
    ):
        """
        A hybrid classifier that combines a fully connected layer with an ADQC network.

        Args:
            feature_num (int): The number of features in the input data.
            dim_mid (int): The dimension of the middle layer of the fully connected network.
            class_num (int): The number of classes in the output.
            num_qubits (int): The number of qubits in the ADQC network.
            num_adqc_layers (int): The number of layers in the ADQC network.
            adqc_gate_pattern (str): The pattern of the ADQC network.
            identity_init (bool): Whether to use identity initialization for the ADQC network.
            double_precision (bool): Whether to use double precision for the ADQC network.
            feature_map (str): The feature map to use for the input data.
        """
        super().__init__()
        self.adqc_net = ADQCNet(
            num_qubits=num_qubits,
            num_layers=num_adqc_layers,
            gate_pattern=adqc_gate_pattern,
            identity_init=identity_init,
            double_precision=double_precision,
        )
        self.fc_layer = nn.Sequential(
            nn.Linear(feature_num, dim_mid),
            nn.ReLU(),
            nn.Linear(dim_mid, num_qubits),
            nn.Tanh(),
        )
        self.feature_num = feature_num
        self.class_num = class_num
        if feature_map == "cossin":
            self.feature_map = cossin_feature_map
            self.feature_map_compiled = torch.compile(cossin_feature_map, dynamic=True)
            self.feature_map_kwargs = {"theta": 0.5}
        else:
            raise Exception(f"The only implemented feature map is cossin, but got {feature_map}")

    def forward(self, data: torch.Tensor) -> torch.Tensor:
        assert data.ndim == 2 and data.shape[1] == self.feature_num, (
            f"The input data must be a 2D tensor with the shape of (batch_size, {self.feature_num}), but got {data.shape}"
        )
        is_mps = data.device.type == "mps"
        feature_map = self.feature_map if is_mps else self.feature_map_compiled
        x = self.fc_layer(data)  # (batch_size, num_qubits)
        x = feature_map(x, **self.feature_map_kwargs)  # (batch_size, num_qubits, 2)
        state = feature_map_to_qubit_state(x)  # (batch_size, *qubit_dims)
        state = self.adqc_net(state)  # (batch_size, *qubit_dims)
        classes_probabilities = probabilities_adqc_classifier(
            state, self.class_num, fast_mode=not is_mps
        )  # (batch_size, num_classes)
        return classes_probabilities
