"""映射函数形式上可写为：$f(x_t,h_{t-1}|W) \rightarrow y_t,h_t$"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../3-6.ipynb.

# %% auto 0
__all__ = ['gpu', 'cpu', 'ADQCRNN', 'series_sin_cos', 'prepare_series_samples']

# %% ../../3-6.ipynb 2
import torch

# useful devices
gpu = torch.device("mps")
cpu = torch.device("cpu")

# %% ../../3-6.ipynb 5
from typing import Literal
from .adqc import ADQCNet
from ..utils.tensors import zeros_state
from einops import einsum, rearrange
from torch import nn
from ..feature_mapping import cossin_feature_map, feature_map_to_qubit_state


class ADQCRNN(nn.Module):
    """
    ADQC-based QuantumRecurrent Neural Network

    The ADQCRNN processes sequential data by maintaining a quantum state across time steps.
    At each step, it encodes the input features into quantum states, combines them with
    the auxiliary qubits, and applies the ADQC network to update the state.
    """

    def __init__(
        self,
        *,
        num_aux_qubits: int,
        num_feature_qubits: int,
        num_layers: int,
        gate_pattern: Literal["brick", "stair"],
        identity_init: bool = False,
        double_precision: bool = False,
        feature_map: Literal["cossin"] = "cossin",
    ):
        """
        ADQC-based QuantumRecurrent Neural Network

        Args:
            num_aux_qubits: Number of auxiliary qubits in the ADQC network
            num_feature_qubits: Number of feature qubits (should match the feature dimension of input data)
            num_layers: Number of layers in the ADQC network
            gate_pattern: Pattern of gate connections, either "brick" or "stair"
            identity_init: Whether to initialize gates close to identity matrices
            double_precision: Whether to use double precision (complex128) or single precision (complex64)
            feature_map: Type of feature map to use for encoding classical data into quantum states
                         Currently only "cossin" is implemented

        """
        super().__init__()
        assert num_aux_qubits > 0
        assert num_feature_qubits > 0
        assert num_layers > 0
        assert gate_pattern in ["brick", "stair"]
        self.num_aux_qubits = num_aux_qubits
        self.num_feature_qubits = num_feature_qubits
        self.net = ADQCNet(
            num_qubits=num_aux_qubits + num_feature_qubits,
            num_layers=num_layers,
            gate_pattern=gate_pattern,
            identity_init=identity_init,
            double_precision=double_precision,
        )
        self.complex_dtype = torch.complex128 if double_precision else torch.complex64
        if feature_map == "cossin":
            self.feature_map = cossin_feature_map
            self.feature_map_compiled = torch.compile(cossin_feature_map, dynamic=True)
        else:
            raise Exception(f"The only implemented feature map is cossin, but got {feature_map}")

    def forward(self, data_batch: torch.Tensor) -> torch.Tensor:
        # data_batch shape: (batch_size, sample_length, feature_dim)
        assert len(data_batch.shape) == 3, (
            f"data_batch must be a 3D tensor of shape (batch_size, sample_length, feature_dim), but got {len(data_batch.shape)}"
        )
        batch_size, sample_length, feature_dim = data_batch.shape
        device = data_batch.device
        is_mps = device.type == "mps"
        assert feature_dim == self.num_feature_qubits, (
            f"feature_dim must be equal to the number of feature qubits, but got {feature_dim=} and {self.num_feature_qubits=}"
        )

        aux_qubit_states = zeros_state(num_qubits=self.num_aux_qubits, dtype=self.complex_dtype).to(
            device
        )  # (*aux_qubit_dims)
        aux_qubit_states = torch.stack(
            [aux_qubit_states] * batch_size, dim=0
        )  # (batch_size, *aux_qubit_dims)
        aux_qubit_states_shape = aux_qubit_states.shape
        batch_dim_name = "batch"
        aux_qubit_names = [f"a{i}" for i in range(self.num_aux_qubits)]
        feature_qubit_names = [f"f{i}" for i in range(self.num_feature_qubits)]
        qubit_cross_product_expression = "{aux_dims}, {feature_dims} -> {state_dims}".format(
            aux_dims=" ".join([batch_dim_name] + aux_qubit_names),
            feature_dims=" ".join([batch_dim_name] + feature_qubit_names),
            state_dims=" ".join([batch_dim_name] + aux_qubit_names + feature_qubit_names),
        )
        projected_feature_state = [0] * self.num_feature_qubits

        for t in range(sample_length):
            if is_mps:
                features = self.feature_map(data_batch[:, t, :])  # (batch_size, feature_dim, 2)
            else:
                features = self.feature_map_compiled(
                    data_batch[:, t, :]
                )  # (batch_size, feature_dim, 2)

            feature_qubit_states = feature_map_to_qubit_state(
                features
            )  # (batch_size, *feature_qubit_dims)

            states = einsum(
                aux_qubit_states, feature_qubit_states, qubit_cross_product_expression
            )  # (batch_size, *aux_qubit_dims, *feature_qubit_dims)
            states = self.net(states)  # (batch_size, *aux_qubit_dims, *feature_qubit_dims)
            states = rearrange(
                states,
                "batch {aux_dims} {feature_dims} -> {feature_dims} batch {aux_dims} ".format(
                    aux_dims=" ".join(aux_qubit_names),
                    feature_dims=" ".join(feature_qubit_names),
                ),
            )  # (*feature_qubit_dims, batch_size, *aux_qubit_dims)
            # projection
            projected_aux_qubit_states = states[
                *projected_feature_state
            ]  # (batch_size, *aux_qubit_dims)
            projected_aux_qubit_states = projected_aux_qubit_states.reshape(
                batch_size, -1
            )  # (batch_size, 2**num_aux_qubits)
            if is_mps:
                # see https://github.com/pytorch/pytorch/issues/146691
                norms = (
                    (projected_aux_qubit_states * projected_aux_qubit_states.conj())
                    .real.sum(dim=1, keepdim=True)
                    .sqrt()
                )  # (batch_size, 1)
            else:
                norms = projected_aux_qubit_states.norm(dim=1, keepdim=True)  # (batch_size, 1)
            projected_aux_qubit_states = (
                projected_aux_qubit_states / norms
            )  # (batch_size, 2**num_aux_qubits)
            # finished one step and set the aux_qubit_states for the next step
            aux_qubit_states = projected_aux_qubit_states.reshape(
                aux_qubit_states_shape
            )  # (batch_size, *aux_qubit_dims)

        # the norm of the projected_aux_qubit_states is the probability of the projected feature state
        prob_of_projected_feature_state = norms.squeeze(1)  # (batch_size,)
        return prob_of_projected_feature_state

# %% ../../3-6.ipynb 10
def series_sin_cos(
    length: int, coeff_sin: torch.Tensor, coeff_cos: torch.Tensor, k_step: float = 0.02
) -> torch.Tensor:
    """
    Calculate a series of cos and sin waves combined.
    Args:
        length (int): The length of the series.
        coeff_sin (torch.Tensor): The coefficients of the sin waves.
        coeff_cos (torch.Tensor): The coefficients of the cos waves.
        k_step (float): The step size of the time steps.
    Returns:
        torch.Tensor: The series of cos and sin waves combined.
    """
    assert coeff_sin.ndim == 1 and coeff_cos.ndim == 1
    assert coeff_sin.numel() == coeff_cos.numel()
    assert coeff_sin.device == coeff_cos.device
    order = coeff_sin.numel()
    device = coeff_sin.device
    time_steps = torch.arange(length, device=device)  # (length,)
    time_steps = time_steps.unsqueeze(0)  # (1, length)
    coeff_sin = coeff_sin.unsqueeze(1)  # (order, 1)
    coeff_cos = coeff_cos.unsqueeze(1)  # (order, 1)
    orders = torch.arange(order, device=device).unsqueeze(1)  # (order, 1)
    y_sin = torch.sin(time_steps * (orders * k_step)) * coeff_sin  # (order, length)
    y_sin = y_sin.sum(dim=0)  # (length,)
    y_cos = torch.cos(time_steps * (orders * k_step)) * coeff_cos  # (order, length)
    y_cos = y_cos.sum(dim=0)  # (length,)
    series = y_sin + y_cos  # (length,)
    return series

# %% ../../3-6.ipynb 12
def prepare_series_samples(
    series: torch.Tensor, sample_length: int, step_size: int
) -> torch.Tensor:
    """
    Prepare samples from a series.
    Args:
        series (torch.Tensor): The series to prepare samples from.
        sample_length (int): The length of the samples.
        step_size (int): The step size of the samples.
    Returns:
        torch.Tensor: The samples.
    """
    assert sample_length >= step_size >= 1
    length = series.shape[0]
    assert length >= sample_length
    samples = []
    for start in range(0, length - sample_length, step_size):
        sample = series[start : start + sample_length]
        samples.append(sample)

    return torch.stack(samples, dim=0)
