# AUTOGENERATED! DO NOT EDIT! File to edit: ../../2-5.ipynb.

# %% auto 0
__all__ = ['apply_gate', 'kron', 'pauli_operator', 'rotate', 'apply_gate_batched_with_vmap', 'apply_gate_batched', 'apply_gate_nonbatched', 'gate_outer_product', 'spin_operator', 'identity_gate_tensor', 'get_control_gate_tensor', 'rand_unitary', 'rand_gate_tensor']

# %% ../../2-5.ipynb 3
from tensor_network.utils.checking import (
    iterable_have_common,
    check_quantum_gate,
    check_state_tensor,
)
from ..utils.mapping import inverse_permutation, unify_tensor_dtypes
import torch
from typing import List
from einops import einsum

# %% ../../2-5.ipynb 4
def apply_gate(
    *,
    quantum_state: torch.Tensor,
    gate: torch.Tensor,
    target_qubit: int | List[int],
    control_qubit: int | List[int] | None = None,
) -> torch.Tensor:
    """
    Apply a quantum gate to a quantum state tensor. It can also be used to implement controlled gates by specifying control qubits.

    Args:
        quantum_state (torch.Tensor): The quantum state tensor.
        gate (torch.Tensor): The quantum gate tensor.
        target_qubit (int | List[int]): The target qubit(s) to apply the gate to.
        control_qubit (int | List[int] | None): The control qubit(s) for the gate. If None, no control qubits are used.
    Returns:
        torch.Tensor: The new quantum state tensor after applying the gate.
    """
    check_state_tensor(quantum_state)

    # check types
    assert isinstance(target_qubit, (int, list)), "target qubit must be int or list"
    assert control_qubit is None or isinstance(control_qubit, (int, list)), (
        "control_qubit must be int or list"
    )

    # unify types
    if isinstance(target_qubit, int):
        target_qubit = [target_qubit]
    if control_qubit is None:
        control_qubit = []
    elif isinstance(control_qubit, int):
        control_qubit = [control_qubit]
    assert not iterable_have_common(target_qubit, control_qubit), (
        "target qubit and control qubit must not overlap"
    )

    num_qubits = quantum_state.ndim
    num_target_qubit = len(target_qubit)
    num_control_qubit = len(control_qubit)
    assert num_qubits >= num_target_qubit + num_control_qubit, (
        "number of qubits must be greater than or equal to the number of target qubits and control qubits"
    )
    check_quantum_gate(gate, num_target_qubit)

    quantum_state, gate = unify_tensor_dtypes(quantum_state, gate)

    # check indices
    for qidx in target_qubit:
        assert 0 <= qidx < num_qubits, f"target qubit index {qidx} out of range"
    for qidx in control_qubit:
        assert 0 <= qidx < num_qubits, f"control qubit index {qidx} out of range"

    if gate.ndim == 2:
        # if in matrix form, reshape to tensor form
        new_shape = [2] * (num_target_qubit * 2)
        gate = gate.reshape(new_shape)

    other_qubits = list(range(num_qubits))
    for qubit_idx in target_qubit:
        other_qubits.remove(qubit_idx)
    for qubit_idx in control_qubit:
        other_qubits.remove(qubit_idx)

    num_other_qubits = len(other_qubits)
    permutation = target_qubit + other_qubits + control_qubit
    state = torch.permute(quantum_state, permutation)
    state_shape = state.shape  # (*target_qubit_shapes, *other_qubit_shapes, *control_qubit_shapes)
    # Flatten the state tensor, so that the shape is (target_qubit_shapes, other_qubit_shapes, -1)
    new_shape = [2] * (num_target_qubit + num_other_qubits) + [-1]
    state = state.reshape(new_shape)
    # only when control qubits are 11111... the gate is applied
    unaffected_state = state[
        ..., :-1
    ]  # (*target_qubit_shapes, *other_qubit_shapes, flattened_dim-1)
    state_to_apply_gate = state[..., -1]  # (*target_qubit_shapes, *other_qubit_shapes)
    # apply gate
    target_qubit_names = [f"t{i}" for i in target_qubit]
    other_qubit_names = [f"o{i}" for i in other_qubits]
    gate_output_qubit_names = [f"g{i}" for i in target_qubit]
    einsum_str = "{gate_dims}, {state_dims} -> {output_dims}".format(
        gate_dims=" ".join(gate_output_qubit_names + target_qubit_names),
        state_dims=" ".join(target_qubit_names + other_qubit_names),
        output_dims=" ".join(gate_output_qubit_names + other_qubit_names),
    )
    new_state = einsum(gate, state_to_apply_gate, einsum_str)
    new_state = new_state.unsqueeze(-1)

    final_state = torch.cat(
        [unaffected_state, new_state], dim=-1
    )  # (*target_qubit_shapes, *other_qubit_shapes, flattened_dim)
    final_state = final_state.reshape(
        state_shape
    )  # (*target_qubit_shapes, *other_qubit_shapes, *control_qubit_shapes)
    inversed_permutation = inverse_permutation(permutation)
    final_state = final_state.permute(inversed_permutation)
    return final_state

# %% ../../2-8.ipynb 6
def kron(*matrices: torch.Tensor) -> torch.Tensor:
    """
    Calculate the Kronecker product of a list of matrices.

    Args:
        *matrices: torch.Tensor, the matrices to be multiplied.
    Returns:
        torch.Tensor, the Kronecker product of the matrices.
    """
    assert len(matrices) >= 2, "At least two matrices are needed for Kronecker product"
    mat = torch.kron(matrices[0], matrices[1])
    for m in matrices[2:]:
        mat = torch.kron(mat, m)
    return mat

# %% ../../3-1.ipynb 3
from typing import Literal
from ..utils.mapping import map_float_to_complex
from functools import cache


@cache
def pauli_operator(
    *,
    pauli: Literal["X", "Y", "Z", "ID"],
    double_precision: bool = False,
    force_complex: bool = False,
) -> torch.Tensor:
    """
    Returns the Pauli operator as a tensor.
    Args:
        pauli (str): The Pauli operator to return. Must be one of 'X', 'Y', 'Z', or 'ID'.
        double_precision (bool): If True, use double precision for the tensor.
        force_complex (bool): If True, force the tensor to be complex.
    Returns:
        torch.Tensor: The Pauli operator as a tensor.
    """
    assert pauli in ["X", "Y", "Z", "ID"], f"Invalid Pauli operator: {pauli}"
    if double_precision:
        dtype_complex = torch.complex128
        if force_complex:
            dtype_default = dtype_complex
        else:
            dtype_default = torch.float64
    else:
        dtype_complex = torch.complex64
        if force_complex:
            dtype_default = dtype_complex
        else:
            dtype_default = torch.float32

    if pauli == "X":
        return torch.tensor([[0, 1], [1, 0]], dtype=dtype_default)
    elif pauli == "Y":
        return torch.tensor([[0, -1j], [1j, 0]], dtype=dtype_complex)
    elif pauli == "Z":
        return torch.tensor([[1, 0], [0, -1]], dtype=dtype_default)
    elif pauli == "ID":
        return torch.eye(2, dtype=dtype_default)
    else:
        raise Exception("Unreachable code")


def _float_convert_to_tensor(
    value: float | torch.Tensor, device: torch.device, dtype: torch.dtype
) -> torch.Tensor:
    if isinstance(value, torch.Tensor):
        return value
    elif isinstance(value, float):
        return torch.tensor(value, dtype=dtype, device=device)
    else:
        raise TypeError(f"Expected float or torch.Tensor, got {type(value)}")


def rotate(
    *,
    params_vec: torch.Tensor | None = None,
    ita: torch.Tensor | float | None = None,
    beta: torch.Tensor | float | None = None,
    delta: torch.Tensor | float | None = None,
    gamma: torch.Tensor | float | None = None,
    dtype: torch.dtype | None = None,
    device: torch.device | None = None,
) -> torch.Tensor:
    """
    Returns the rotation gate as a tensor.
    Args:
        params_vec (torch.Tensor): A 4-element vector containing the parameters [ita, beta, delta, gamma].
        ita (torch.Tensor | float | None): The first parameter of the rotation gate.
        beta (torch.Tensor | float | None): The second parameter of the rotation gate.
        delta (torch.Tensor | float | None): The third parameter of the rotation gate.
        gamma (torch.Tensor | float | None): The fourth parameter of the rotation gate.
        dtype (torch.dtype | None): The data type of the tensor.
        device (torch.device | None): The device to create the tensor on.
    Returns:
        torch.Tensor: The rotation gate as a tensor.
    """

    assert params_vec is not None or (
        ita is not None and beta is not None and delta is not None and gamma is not None
    ), "Either params_vec or ita, beta, delta, and gamma must be provided"
    if params_vec is not None:
        assert isinstance(params_vec, torch.Tensor), "params must be a torch.Tensor"
        assert params_vec.shape == (4,), "params must be a 4-element vector"
        dtype = params_vec.dtype if dtype is None else dtype
        device = params_vec.device if device is None else device
        assert dtype in [torch.float32, torch.float64], "params must be float32 or float64"
        beta, delta, ita, gamma = params_vec[0], params_vec[1], params_vec[2], params_vec[3]
    else:
        ita = _float_convert_to_tensor(ita, device=device, dtype=dtype)
        beta = _float_convert_to_tensor(beta, device=device, dtype=dtype)
        delta = _float_convert_to_tensor(delta, device=device, dtype=dtype)
        gamma = _float_convert_to_tensor(gamma, device=device, dtype=dtype)
        assert ita.shape == beta.shape == delta.shape == gamma.shape == (), (
            "ita, beta, delta, and gamma must be scalars"
        )
        assert ita.dtype == beta.dtype == delta.dtype == gamma.dtype, (
            "ita, beta, delta, and gamma must have the same dtype"
        )
        assert ita.device == beta.device == delta.device == gamma.device, (
            "ita, beta, delta, and gamma must have the same device"
        )
        dtype = ita.dtype if dtype is None else dtype
        device = ita.device if device is None else device
        assert dtype in [torch.float32, torch.float64], (
            "ita, beta, delta, and gamma must be float32 or float64"
        )

    gate_dtype = map_float_to_complex(dtype=dtype)
    # calculate the matrix for the beta terms
    beta_coefficient_matrix = torch.tensor(
        [[-0.5, -0.5], [0.5, 0.5]], device=device, dtype=gate_dtype
    )
    beta_matrix = beta_coefficient_matrix * beta
    # calculate the matrix for the delta terms
    delta_coefficient_matrix = beta_coefficient_matrix.T
    delta_matrix = delta_coefficient_matrix * delta
    # calculate the matrix for the gamma terms
    gamma_2 = gamma / 2
    gamma_coefficient_matrix_cosine = torch.eye(2, device=device, dtype=gate_dtype)
    gamma_coefficient_matrix_sine = torch.tensor([[0, 1], [1, 0]], device=device, dtype=gate_dtype)
    gamma_matrix = gamma_coefficient_matrix_cosine * torch.cos(
        gamma_2
    ) + gamma_coefficient_matrix_sine * torch.sin(gamma_2)
    # set the coefficient matrix in front of e
    coefficient_matrix = torch.tensor([[1, -1], [1, 1]], device=device, dtype=gate_dtype)
    gate = coefficient_matrix * torch.exp(1j * (ita + beta_matrix + delta_matrix)) * gamma_matrix
    return gate

# %% ../../3-5.ipynb 3
def apply_gate_batched(
    *,
    quantum_states: torch.Tensor,
    gate: torch.Tensor,
    target_qubit: int | List[int],
    control_qubit: int | List[int] | None = None,
) -> torch.Tensor:
    """
    Apply a quantum gate to a quantum state tensor. It can also be used to implement controlled gates by specifying control qubits.

    Args:
        quantum_states (torch.Tensor): The batched quantum state tensor with batch dimension at 0
        gate (torch.Tensor): The quantum gate tensor, not batched.
        target_qubit (int | List[int]): The target qubit(s) to apply the gate to.
        control_qubit (int | List[int] | None): The control qubit(s) for the gate. If None, no control qubits are used.
    Returns:
        torch.Tensor: The new quantum state tensor after applying the gate.
    """
    check_state_tensor(quantum_states[0])

    # check types
    assert isinstance(target_qubit, (int, list)), "target qubit must be int or list"
    assert control_qubit is None or isinstance(control_qubit, (int, list)), (
        "control_qubit must be int or list"
    )

    # unify types
    if isinstance(target_qubit, int):
        target_qubit = [target_qubit]
    if control_qubit is None:
        control_qubit = []
    elif isinstance(control_qubit, int):
        control_qubit = [control_qubit]
    assert not iterable_have_common(target_qubit, control_qubit), (
        "target qubit and control qubit must not overlap"
    )

    batch_num = quantum_states.shape[0]
    num_qubits = quantum_states.ndim - 1  # because of batch dimension
    num_target_qubit = len(target_qubit)
    num_control_qubit = len(control_qubit)
    assert num_qubits >= num_target_qubit + num_control_qubit, (
        "number of qubits must be greater than or equal to the number of target qubits and control qubits"
    )
    check_quantum_gate(gate, num_target_qubit)

    quantum_states, gate = unify_tensor_dtypes(quantum_states, gate)
    # permute the batch dim to the last dim
    quantum_states = torch.movedim(quantum_states, 0, -1)  # (*qubit_shapes, batch_dim)
    batch_dim_idx = quantum_states.ndim - 1

    # check indices
    for qidx in target_qubit:
        assert 0 <= qidx < num_qubits, f"target qubit index {qidx} out of range"
    for qidx in control_qubit:
        assert 0 <= qidx < num_qubits, f"control qubit index {qidx} out of range"

    if gate.ndim == 2:
        # if in matrix form, reshape to tensor form
        new_shape = [2] * (num_target_qubit * 2)
        gate = gate.reshape(new_shape)

    other_qubits = list(range(num_qubits))
    for qubit_idx in target_qubit:
        other_qubits.remove(qubit_idx)
    for qubit_idx in control_qubit:
        other_qubits.remove(qubit_idx)

    num_other_qubits = len(other_qubits)
    permute_qubit_dims = target_qubit + other_qubits + control_qubit + [batch_dim_idx]
    state = torch.permute(quantum_states, permute_qubit_dims)
    state_shape = (
        state.shape
    )  # (*target_qubit_shapes, *other_qubit_shapes, *control_qubit_shapes, batch_dim)
    # Flatten the state tensor, so that the shape is (target_qubit_shapes, other_qubit_shapes, -1, batch_dim)
    new_shape = [2] * (num_target_qubit + num_other_qubits) + [-1, batch_num]
    state = state.reshape(new_shape)
    # only when control qubits are 11111... the gate is applied
    unaffected_state = state[
        ..., :-1, :
    ]  # (*target_qubit_shapes, *other_qubit_shapes, flattened_dim-1, batch_dim)
    state_to_apply_gate = state[
        ..., -1, :
    ]  # (*target_qubit_shapes, *other_qubit_shapes, batch_dim)
    # apply gate
    batch_dim_name = "batch"
    target_qubit_names = [f"t{i}" for i in target_qubit]
    other_qubit_names = [f"o{i}" for i in other_qubits]
    gate_output_qubit_names = [f"g{i}" for i in target_qubit]
    einsum_str = "{gate_dims}, {state_dims} -> {output_dims}".format(
        gate_dims=" ".join(gate_output_qubit_names + target_qubit_names),
        state_dims=" ".join(target_qubit_names + other_qubit_names + [batch_dim_name]),
        output_dims=" ".join(gate_output_qubit_names + other_qubit_names + [batch_dim_name]),
    )
    new_state = einsum(gate, state_to_apply_gate, einsum_str)
    new_state = new_state.unsqueeze(-2)  # (*target_qubit_shapes, *other_qubit_shapes, 1, batch_dim)

    final_state = torch.cat(
        [unaffected_state, new_state], dim=-2
    )  # (*target_qubit_shapes, *other_qubit_shapes, flattened_dim)
    final_state = final_state.reshape(
        state_shape
    )  # (*target_qubit_shapes, *other_qubit_shapes, *control_qubit_shapes, batch_dim)

    # inverse the previous permutations
    inverse_permute_qubit_dims = inverse_permutation(permute_qubit_dims)
    final_state = final_state.permute(inverse_permute_qubit_dims)  # (*qubit_shapes, batch_dim)
    final_state = torch.movedim(final_state, -1, 0)  # (batch_dim, *qubit_shapes)
    return final_state


def apply_gate_nonbatched(
    *,
    quantum_state: torch.Tensor,
    gate: torch.Tensor,
    target_qubit: int | List[int],
    control_qubit: int | List[int] | None = None,
) -> torch.Tensor:
    """
    Apply a quantum gate to a quantum state tensor. It can also be used to implement controlled gates by specifying control qubits.

    Same as `apply_gate`, but use `apply_gate_batched` under the hood.

    Args:
        quantum_state (torch.Tensor): The quantum state tensor.
        gate (torch.Tensor): The quantum gate tensor, not batched.
        target_qubit (int | List[int]): The target qubit(s) to apply the gate to.
        control_qubit (int | List[int] | None): The control qubit(s) for the gate. If None, no control qubits are used.
    Returns:
        torch.Tensor: The new quantum state tensor after applying the gate.
    """
    quantum_state = quantum_state.unsqueeze(0)  # add batch dimension
    return apply_gate_batched(
        quantum_states=quantum_state,
        gate=gate,
        target_qubit=target_qubit,
        control_qubit=control_qubit,
    ).squeeze(0)


# need this to wrap apply_gate because of vmap's specification on kwargs
def __apply_gate_for_vmap(
    state: torch.Tensor,
    gate: torch.Tensor,
    *,
    target_qubit: int | List[int],
    control_qubit: int | List[int] | None = None,
):
    return apply_gate(
        quantum_state=state, gate=gate, target_qubit=target_qubit, control_qubit=control_qubit
    )


apply_gate_batched_with_vmap = torch.vmap(__apply_gate_for_vmap, in_dims=(0, None))

# %% ../../3-8.ipynb 5
from einops import rearrange
from ..utils.checking import check_quantum_gate


def gate_outer_product(*gates: torch.Tensor, matrix_form: bool = False) -> torch.Tensor:
    """
    Outer product of multiple quantum gates. This only deals with the case where the target qubits of gates are disjoint.
    Args:
        *gates: The quantum gates to be cross-multiplied.
        matrix_form: Whether the quantum gates are in matrix form.
    Returns:
        The cross product of the quantum gates.
    """
    gates_num = len(gates)
    assert gates_num >= 2, "at least 2 gates"
    num_qubits = [check_quantum_gate(gate) for gate in gates]
    gate_tensors = []
    for i, gate in enumerate(gates):
        if gate.ndim == 2 and gate.shape[0] == gate.shape[1]:  # if gate is in matrix form
            gate_tensors.append(gate.reshape([2] * (num_qubits[i] * 2)))
        else:
            gate_tensors.append(gate)
    gate_dim_names = []
    for i in range(gates_num):
        num_qubits_of_gate_i = num_qubits[i]
        gate_dim_names.append(
            {
                "left_dim_names": [f"g{i}l{j}" for j in range(num_qubits_of_gate_i)],
                "right_dim_names": [f"g{i}r{j}" for j in range(num_qubits_of_gate_i)],
            }
        )

    input_subexpressions = []
    for gate_dim_info in gate_dim_names:
        input_subexp = " ".join(gate_dim_info["left_dim_names"] + gate_dim_info["right_dim_names"])
        input_subexpressions.append(input_subexp)
    input_expression = ",".join(input_subexpressions)

    output_subexpressions = []
    for gate_dim_info in gate_dim_names:
        output_subexpressions.extend(gate_dim_info["left_dim_names"])
    for gate_dim_info in gate_dim_names:
        output_subexpressions.extend(gate_dim_info["right_dim_names"])
    output_expression = " ".join(output_subexpressions)

    einsum_expression = f"{input_expression} -> {output_expression}"
    gate_product = einsum(*gate_tensors, einsum_expression)

    if matrix_form:
        output_left_dim_num = len(output_subexpressions) // 2
        left_exp = " ".join(output_subexpressions[:output_left_dim_num])
        right_exp = " ".join(output_subexpressions[output_left_dim_num:])
        gate_product = rearrange(gate_product, f"{output_expression} -> ({left_exp}) ({right_exp})")
    return gate_product

# %% ../../3-8.ipynb 9
def spin_operator(direction: Literal["X", "Y", "Z", "ID"]):
    """
    Return the spin operator for the given direction.
    Args:
        direction: The direction of the spin operator. Can be "X", "Y", "Z", or "ID".
    Returns:
        The spin operator for the given direction.
    """
    assert direction in ["X", "Y", "Z", "ID"], "direction must be one of X, Y, Z, ID"
    if direction == "ID":
        return pauli_operator(pauli="ID")
    return pauli_operator(pauli=direction) / 2

# %% ../../tensor_gate_extra.ipynb 1
from itertools import product


def identity_gate_tensor(
    num_qubits: int,
    matrix_form: bool,
    *,
    dtype: torch.dtype | None = None,
    device: torch.device | None = None,
    _fast_mode: bool = True,
) -> torch.Tensor:
    """
    Create a tensor representation of the identity gate.

    Args:
        num_qubits: int, the number of qubits.
        matrix_form: bool, whether to return the matrix form of the identity gate.
        dtype: torch.dtype | None, the dtype of the identity gate tensor.
        device: torch.device | None, the device of the identity gate tensor.
    Returns:
        torch.Tensor, the identity gate tensor.
    """
    if matrix_form:
        return torch.eye(2**num_qubits, dtype=dtype, device=device)
    elif _fast_mode:
        matrix = torch.eye(2**num_qubits, dtype=dtype, device=device)
        return matrix.view([2] * (num_qubits * 2))
    else:
        tensor = torch.zeros([2] * (num_qubits * 2), dtype=dtype, device=device)
        for i in product([0, 1], repeat=num_qubits):
            indices = i + i
            tensor[indices] = 1
        return tensor

# %% ../../tensor_gate_extra.ipynb 3
from ..utils.checking import check_quantum_gate


def get_control_gate_tensor(
    num_control_qubits: int,
    applied_gate: torch.Tensor,
    matrix_form: bool,
    *,
    dtype: torch.dtype,
    device: torch.device,
    _fast_mode: bool = True,
) -> torch.Tensor:
    """
    Create a tensor representation of a controlled gate.

    Args:
        num_control_qubits: int, the number of control qubits.
        applied_gate: torch.Tensor, the gate to be applied.
        matrix_form: bool, whether to return the matrix form of the controlled gate.
        dtype: torch.dtype, the dtype of the controlled gate tensor.
        device: torch.device, the device of the controlled gate tensor.
    Returns:
        torch.Tensor, the controlled gate tensor.
    """
    num_applied_qubits = check_quantum_gate(applied_gate)
    num_total_qubits = num_control_qubits + num_applied_qubits
    if matrix_form or _fast_mode:
        if applied_gate.ndim > 2:
            applied_gate = applied_gate.view(2**num_applied_qubits, 2**num_applied_qubits)
        matrix = identity_gate_tensor(
            num_total_qubits, True, dtype=dtype, device=device, _fast_mode=_fast_mode
        )
        slice_start = (2**num_control_qubits - 1) << num_applied_qubits
        matrix[slice_start:, slice_start:] = applied_gate
        if matrix_form:
            return matrix
        else:
            return matrix.view([2] * (num_total_qubits * 2))
    else:
        tensor = identity_gate_tensor(
            num_total_qubits, False, dtype=dtype, device=device, _fast_mode=False
        )
        applied_gate = applied_gate.view([2] * (num_applied_qubits * 2))
        index = [1] * num_control_qubits + [slice(None, None, None)] * num_applied_qubits
        index = index + index
        tensor[index] = applied_gate
        return tensor

# %% ../../tensor_gate_extra.ipynb 4
def rand_unitary(
    dim: int, *, dtype: torch.dtype | None = None, device: torch.device | None = None
) -> torch.Tensor:
    """
    Generate a random unitary matrix of given dimension.

    Args:
        dim (int): Dimension of the matrix.
    Returns:
        torch.Tensor: A random unitary matrix of the specified dimension.
    """
    mat = torch.randn(dim, dim, dtype=dtype, device=device)
    Q, _R = torch.linalg.qr(mat)
    return Q


def rand_gate_tensor(
    num_qubits: int,
    matrix_form: bool,
    *,
    dtype: torch.dtype | None = None,
    device: torch.device | None = None,
) -> torch.Tensor:
    """
    Generate a random gate tensor of given dimension.

    Args:
        num_qubits (int): Number of qubits.
        matrix_form (bool): Whether to return the matrix form of the gate.
    Returns:
        torch.Tensor: A random gate tensor of the specified qubit number.
    """
    mat_dim = 2**num_qubits
    mat = rand_unitary(mat_dim, dtype=dtype, device=device)
    if matrix_form:
        return mat
    else:
        return mat.view([2] * (num_qubits * 2))