# AUTOGENERATED! DO NOT EDIT! File to edit: ../../0-utils-checking.ipynb.

# %% auto 0
__all__ = ['iterable_have_common', 'check_state_tensor', 'check_quantum_gate', 'is_notebook']

# %% ../../0-utils-checking.ipynb 0
from typing import List, Tuple
import torch

# %% ../../0-utils-checking.ipynb 2
def iterable_have_common(a: List[int] | Tuple[int], b: List[int] | Tuple[int]) -> bool:
    """
    Check if two iterables have any common elements.

    Args:
        a: First iterable.
        b: Second iterable.

    Returns:
        True if there are common elements, False otherwise.
    """
    assert isinstance(a, (list, tuple)), "First argument must be a list or tuple."
    assert isinstance(b, (list, tuple)), "Second argument must be a list or tuple."
    return len(set(a) & set(b)) > 0


def check_state_tensor(tensor: torch.Tensor):
    """
    Check if the tensor is a valid quantum state tensor.
    Args:
        tensor: The tensor to check.
    Raises:
        AssertionError: If the tensor is not a valid quantum state tensor.
    """
    assert isinstance(tensor, torch.Tensor), "quantum_state must be a torch.Tensor"
    assert tensor.dtype in [torch.float32, torch.float64, torch.complex64, torch.complex128], (
        "quantum_state must be a float or complex tensor"
    )
    assert all(x == 2 for x in tensor.shape), (
        "quantum_state must be a tensor with all dimensions of size 2"
    )
    assert tensor.ndim > 0, "quantum_state must be a tensor with at least one dimension"


def check_quantum_gate(
    tensor: torch.Tensor, num_qubits: int | None = None, assert_tensor_form: bool = False
) -> int:
    """
    Check if the tensor is a valid quantum gate tensor.
    Args:
        tensor: The tensor to check.
        num_qubits: The number of qubits.
    Returns:
        The number of qubits that the gate acts on.
    Raises:
        AssertionError: If the tensor is not a valid quantum gate tensor.
    """
    assert isinstance(tensor, torch.Tensor), "quantum_gate must be a torch.Tensor"
    assert tensor.dtype in [torch.float32, torch.float64, torch.complex64, torch.complex128], (
        "quantum_gate must be a float or complex tensor"
    )
    assert tensor.ndim >= 2, "quantum_gate must be a tensor with at least two dimensions"
    assert tensor.ndim % 2 == 0, "quantum_gate must have an even number of dimensions"

    if tensor.ndim == 2:
        # in matrix form
        num_qubits = int(tensor.shape[0].bit_length() - 1) if num_qubits is None else num_qubits
        assert tensor.shape[0] == tensor.shape[1] == 2**num_qubits, (
            f"gate must be a square matrix with dimensions 2^num_qubits, got {tensor.shape}"
        )
        assert not (assert_tensor_form and num_qubits > 1), "Quantum gate should be in tensor form"
    else:
        assert all(d == 2 for d in tensor.shape), "gate tensor must have all dimensions of size 2"
        num_qubits = tensor.ndim // 2 if num_qubits is None else num_qubits
        assert tensor.ndim == 2 * num_qubits, (
            f"gate tensor must have 2 * num_qubits dimensions, got {tensor.ndim}"
        )
    return num_qubits

# %% ../../0-utils-checking.ipynb 3
def is_notebook() -> bool:
    """
    Check if the code is running in a Jupyter/IPython notebook.
    Returns:
        bool, True if running in a Jupyter/IPython notebook, False otherwise.
    """
    try:
        from IPython import get_ipython

        shell = get_ipython().__class__.__name__
        if shell == "ZMQInteractiveShell":  # Jupyter/IPython notebook
            return True
        elif shell == "TerminalInteractiveShell":  # IPython terminal
            return False
        else:
            return False
    except NameError:
        return False  # Not in IPython
    except Exception:
        return False
