# AUTOGENERATED! DO NOT EDIT! File to edit: ../../0-utils-mapping.ipynb.

# %% auto 0
__all__ = ['inverse_permutation', 'unify_tensor_dtypes', 'map_float_to_complex', 'view_gate_tensor_as_matrix',
           'view_gate_matrix_as_tensor']

# %% ../../0-utils-mapping.ipynb 0
from typing import List, Tuple
import torch
from .checking import check_quantum_gate

# %% ../../0-utils-mapping.ipynb 2
def inverse_permutation(permutation: List[int]) -> List[int]:
    """
    Inverse a permutation.
    Args:
        permutation: The permutation to inverse.
    Returns:
        The inverse permutation.
    """
    permutation = torch.tensor(permutation, dtype=torch.long)
    inv = torch.empty_like(permutation)
    inv[permutation] = torch.arange(permutation.size(0))
    return inv.tolist()


def unify_tensor_dtypes(t1: torch.Tensor, t2: torch.Tensor) -> Tuple[torch.Tensor, torch.Tensor]:
    """
    Unify the dtypes of two tensors to the most appropriate type.
    Args:
        t1: First tensor.
        t2: Second tensor.
    Returns:
        Tuple of tensors with unified dtypes.
    Raises:
        AssertionError: if the dtypes of the tensors are not valid.
    """
    assert t1.dtype in [torch.float32, torch.float64, torch.complex64, torch.complex128], (
        "quantum_state must be a float or complex tensor"
    )
    assert t2.dtype in [torch.float32, torch.float64, torch.complex64, torch.complex128], (
        "quantum_state must be a float or complex tensor"
    )
    if t1.dtype == t2.dtype:
        return t1, t2
    convert_dtypes = [
        (torch.float32, torch.complex64, torch.complex64),
        (torch.float64, torch.complex64, torch.complex128),
        (torch.float32, torch.complex128, torch.complex128),
        (torch.float64, torch.complex128, torch.complex128),
    ]
    for d1, d2, td in convert_dtypes:
        if (t1.dtype == d1 and t2.dtype == d2) or (t1.dtype == d2 and t2.dtype == d1):
            return t1.to(td), t2.to(td)
    raise_dtypes = [
        (torch.float32, torch.float64),
        (torch.complex64, torch.complex128),
    ]
    for d1, d2 in raise_dtypes:
        if (t1.dtype == d1 and t2.dtype == d2) or (t1.dtype == d2 and t2.dtype == d1):
            return t1.to(d2), t2.to(d2)

    raise Exception("Unreachable code in unify_tensor_dtypes")


def map_float_to_complex(
    *, tensor: torch.Tensor | None = None, dtype: torch.dtype | None = None
) -> torch.Tensor | torch.dtype:
    """
    Map a float tensor or dtype to a complex tensor or dtype.
    Args:
        tensor: The input tensor.
        dtype: The input dtype.
    Returns:
        The complex tensor or dtype.
    Raises:
        AssertionError: If neither tensor nor dtype is provided.
    """
    assert tensor is not None or dtype is not None, "Either tensor or dtype must be provided"
    original_dtype = tensor.dtype if tensor is not None else dtype
    assert original_dtype in [torch.float32, torch.float64], "dtype must be float32 or float64"
    to_dtype = torch.complex64 if original_dtype == torch.float32 else torch.complex128
    if tensor is not None:
        return tensor.to(to_dtype)
    return to_dtype


def view_gate_tensor_as_matrix(
    tensor: torch.Tensor, *, num_qubit: int | None = None
) -> torch.Tensor:
    """
    Convert a tensor representing a quantum gate into a matrix form.
    The tensor should have an even number of dimensions, each of size 2.

    Args:
        tensor (torch.Tensor): The tensor representing the quantum gate.
        num_qubit (int | None): The number of qubits the gate is acting on. If None, it is inferred from the tensor shape.
    Returns:
        torch.Tensor: The matrix form of the quantum gate tensor.
    """
    assert tensor.ndim % 2 == 0, "Tensor must have an even number of dimensions"
    assert all(d == 2 for d in tensor.shape), "Tensor dimensions must be 2"
    qubit_count = tensor.ndim // 2 if num_qubit is None else num_qubit
    check_quantum_gate(tensor, qubit_count)
    return tensor.view(2**qubit_count, 2**qubit_count)


def view_gate_matrix_as_tensor(
    tensor: torch.Tensor, *, num_qubit: int | None = None
) -> torch.Tensor:
    """
    Convert a matrix representing a quantum gate into a tensor form.
    The matrix should have dimensions (2^n, 2^n) for some n.

    Args:
        tensor (torch.Tensor): The matrix representing the quantum gate.
        num_qubit (int | None): The number of qubits the gate is acting on. If None, it is inferred from the matrix shape.
    Returns:
        torch.Tensor: The tensor form of the quantum gate matrix.
    """
    assert tensor.ndim == 2, "Matrix must have 2 dimensions"
    assert tensor.shape[0] == tensor.shape[1], "Matrix must be square"
    qubit_count = int(tensor.shape[0].bit_length() - 1) if num_qubit is None else num_qubit
    assert tensor.shape[0] == 2**qubit_count, (
        f"Matrix size must be (2^q, 2^q) for q qubits, but got {tensor.shape}"
    )
    check_quantum_gate(tensor, qubit_count)
    return tensor.view(*([2] * (qubit_count * 2)))
